"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendRequestWorker = exports.searchNamedPipeServerForFile = exports.connect = exports.updatePipeTable = exports.readPipeTable = void 0;
const os = require("os");
const net = require("net");
const path = require("path");
const fs = require("fs");
var typescript_1 = require("@volar/typescript");
const { version } = require('../package.json');
const pipeTableFile = path.join(os.tmpdir(), `vue-tsp-table-${version}.json`);
function readPipeTable() {
    if (!fs.existsSync(pipeTableFile)) {
        return [];
    }
    try {
        const servers = JSON.parse(fs.readFileSync(pipeTableFile, 'utf8'));
        return servers;
    }
    catch {
        fs.unlinkSync(pipeTableFile);
        return [];
    }
}
exports.readPipeTable = readPipeTable;
function updatePipeTable(servers) {
    if (servers.length === 0) {
        fs.unlinkSync(pipeTableFile);
    }
    else {
        fs.writeFileSync(pipeTableFile, JSON.stringify(servers, undefined, 2));
    }
}
exports.updatePipeTable = updatePipeTable;
function connect(path) {
    return new Promise(resolve => {
        const client = net.connect(path);
        client.setTimeout(1000);
        client.on('connect', () => {
            resolve(client);
        });
        client.on('error', () => {
            return resolve(undefined);
        });
        client.on('timeout', () => {
            return resolve(undefined);
        });
    });
}
exports.connect = connect;
async function searchNamedPipeServerForFile(fileName) {
    const servers = readPipeTable();
    const configuredServers = servers
        .filter(item => item.serverKind === 1);
    const inferredServers = servers
        .filter(item => item.serverKind === 0)
        .sort((a, b) => b.currentDirectory.length - a.currentDirectory.length);
    for (const server of configuredServers.sort((a, b) => sortTSConfigs(fileName, a.currentDirectory, b.currentDirectory))) {
        const client = await connect(server.path);
        if (client) {
            const projectInfo = await sendRequestWorker({ type: 'projectInfoForFile', args: [fileName] }, client);
            if (projectInfo) {
                return {
                    server,
                    projectInfo,
                };
            }
        }
    }
    for (const server of inferredServers) {
        if (!path.relative(server.currentDirectory, fileName).startsWith('..')) {
            const client = await connect(server.path);
            if (client) {
                return {
                    server,
                    projectInfo: undefined,
                };
            }
        }
    }
}
exports.searchNamedPipeServerForFile = searchNamedPipeServerForFile;
function sortTSConfigs(file, a, b) {
    const inA = isFileInDir(file, path.dirname(a));
    const inB = isFileInDir(file, path.dirname(b));
    if (inA !== inB) {
        const aWeight = inA ? 1 : 0;
        const bWeight = inB ? 1 : 0;
        return bWeight - aWeight;
    }
    const aLength = a.split('/').length;
    const bLength = b.split('/').length;
    return bLength - aLength;
}
function isFileInDir(fileName, dir) {
    const relative = path.relative(dir, fileName);
    return !!relative && !relative.startsWith('..') && !path.isAbsolute(relative);
}
function sendRequestWorker(request, client) {
    return new Promise(resolve => {
        let dataChunks = [];
        client.setTimeout(5000);
        client.on('data', chunk => {
            dataChunks.push(chunk);
        });
        client.on('end', () => {
            if (!dataChunks.length) {
                console.warn('[Vue Named Pipe Client] No response from server for request:', request.type);
                resolve(undefined);
                return;
            }
            const data = Buffer.concat(dataChunks);
            const text = data.toString();
            let json = null;
            try {
                json = JSON.parse(text);
            }
            catch (e) {
                console.error('[Vue Named Pipe Client] Failed to parse response:', text);
                resolve(undefined);
                return;
            }
            resolve(json);
        });
        client.on('error', err => {
            console.error('[Vue Named Pipe Client] Error:', err.message);
            resolve(undefined);
        });
        client.on('timeout', () => {
            console.error('[Vue Named Pipe Client] Timeout');
            resolve(undefined);
        });
        client.write(JSON.stringify(request));
    });
}
exports.sendRequestWorker = sendRequestWorker;
//# sourceMappingURL=utils.js.map