"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMappingOffset = exports.toGeneratedOffsets = exports.toGeneratedOffset = exports.toSourceOffsets = exports.toSourceOffset = exports.transformTextSpan = exports.transformTextChange = exports.transformSpan = exports.transformDocumentSpan = exports.transformFileTextChanges = exports.fillSourceFileText = exports.transformDiagnostic = exports.transformCallHierarchyItem = void 0;
const language_core_1 = require("@volar/language-core");
const utils_1 = require("./utils");
const transformedDiagnostics = new WeakMap();
const transformedSourceFile = new WeakSet();
function transformCallHierarchyItem(language, item, filter) {
    const span = transformSpan(language, item.file, item.span, filter);
    const selectionSpan = transformSpan(language, item.file, item.selectionSpan, filter);
    return {
        ...item,
        file: span?.fileName ?? item.file,
        span: span?.textSpan ?? { start: 0, length: 0 },
        selectionSpan: selectionSpan?.textSpan ?? { start: 0, length: 0 },
    };
}
exports.transformCallHierarchyItem = transformCallHierarchyItem;
function transformDiagnostic(language, diagnostic, program, isTsc) {
    if (!transformedDiagnostics.has(diagnostic)) {
        transformedDiagnostics.set(diagnostic, undefined);
        const { relatedInformation } = diagnostic;
        if (relatedInformation) {
            diagnostic.relatedInformation = relatedInformation
                .map(d => transformDiagnostic(language, d, program, isTsc))
                .filter(utils_1.notEmpty);
        }
        if (diagnostic.file !== undefined
            && diagnostic.start !== undefined
            && diagnostic.length !== undefined) {
            const [serviceScript] = (0, utils_1.getServiceScript)(language, diagnostic.file.fileName);
            if (serviceScript) {
                const [sourceSpanFileName, sourceSpan] = transformTextSpan(undefined, language, serviceScript, {
                    start: diagnostic.start,
                    length: diagnostic.length
                }, language_core_1.shouldReportDiagnostics) ?? [];
                const actualDiagnosticFile = sourceSpanFileName
                    ? diagnostic.file.fileName === sourceSpanFileName
                        ? diagnostic.file
                        : program?.getSourceFile(sourceSpanFileName)
                    : undefined;
                if (sourceSpan && actualDiagnosticFile) {
                    if (isTsc) {
                        fillSourceFileText(language, diagnostic.file);
                    }
                    transformedDiagnostics.set(diagnostic, {
                        ...diagnostic,
                        file: actualDiagnosticFile,
                        start: sourceSpan.start,
                        length: sourceSpan.length,
                    });
                }
            }
            else {
                transformedDiagnostics.set(diagnostic, diagnostic);
            }
        }
        else {
            transformedDiagnostics.set(diagnostic, diagnostic);
        }
    }
    return transformedDiagnostics.get(diagnostic);
}
exports.transformDiagnostic = transformDiagnostic;
// fix https://github.com/vuejs/language-tools/issues/4099 without `incremental`
function fillSourceFileText(language, sourceFile) {
    if (transformedSourceFile.has(sourceFile)) {
        return;
    }
    transformedSourceFile.add(sourceFile);
    const [serviceScript] = (0, utils_1.getServiceScript)(language, sourceFile.fileName);
    if (serviceScript && !serviceScript.preventLeadingOffset) {
        const sourceScript = language.scripts.fromVirtualCode(serviceScript.code);
        sourceFile.text = sourceScript.snapshot.getText(0, sourceScript.snapshot.getLength())
            + sourceFile.text.substring(sourceScript.snapshot.getLength());
    }
}
exports.fillSourceFileText = fillSourceFileText;
function transformFileTextChanges(language, changes, filter) {
    const changesPerFile = {};
    const newFiles = new Set();
    for (const fileChanges of changes) {
        const [_, source] = (0, utils_1.getServiceScript)(language, fileChanges.fileName);
        if (source) {
            fileChanges.textChanges.forEach(c => {
                const { fileName, textSpan } = transformSpan(language, fileChanges.fileName, c.span, filter) ?? {};
                if (fileName && textSpan) {
                    (changesPerFile[fileName] ?? (changesPerFile[fileName] = [])).push({ ...c, span: textSpan });
                }
            });
        }
        else {
            const list = (changesPerFile[fileChanges.fileName] ?? (changesPerFile[fileChanges.fileName] = []));
            fileChanges.textChanges.forEach(c => {
                list.push(c);
            });
            if (fileChanges.isNewFile) {
                newFiles.add(fileChanges.fileName);
            }
        }
    }
    const result = [];
    for (const fileName in changesPerFile) {
        result.push({
            fileName,
            isNewFile: newFiles.has(fileName),
            textChanges: changesPerFile[fileName]
        });
    }
    return result;
}
exports.transformFileTextChanges = transformFileTextChanges;
function transformDocumentSpan(language, documentSpan, filter, shouldFallback) {
    let textSpan = transformSpan(language, documentSpan.fileName, documentSpan.textSpan, filter);
    if (!textSpan && shouldFallback) {
        textSpan = {
            fileName: documentSpan.fileName,
            textSpan: { start: 0, length: 0 },
        };
    }
    if (!textSpan) {
        return;
    }
    const contextSpan = transformSpan(language, documentSpan.fileName, documentSpan.contextSpan, filter);
    const originalTextSpan = transformSpan(language, documentSpan.originalFileName, documentSpan.originalTextSpan, filter);
    const originalContextSpan = transformSpan(language, documentSpan.originalFileName, documentSpan.originalContextSpan, filter);
    return {
        ...documentSpan,
        fileName: textSpan.fileName,
        textSpan: textSpan.textSpan,
        contextSpan: contextSpan?.textSpan,
        originalFileName: originalTextSpan?.fileName,
        originalTextSpan: originalTextSpan?.textSpan,
        originalContextSpan: originalContextSpan?.textSpan,
    };
}
exports.transformDocumentSpan = transformDocumentSpan;
function transformSpan(language, fileName, textSpan, filter) {
    if (!fileName || !textSpan) {
        return;
    }
    const [serviceScript] = (0, utils_1.getServiceScript)(language, fileName);
    if (serviceScript) {
        const [sourceSpanFileName, sourceSpan] = transformTextSpan(undefined, language, serviceScript, textSpan, filter) ?? [];
        if (sourceSpan && sourceSpanFileName) {
            return {
                fileName: sourceSpanFileName,
                textSpan: sourceSpan,
            };
        }
    }
    else {
        return {
            fileName,
            textSpan,
        };
    }
}
exports.transformSpan = transformSpan;
function transformTextChange(sourceScript, language, serviceScript, textChange, filter) {
    const [sourceSpanFileName, sourceSpan] = transformTextSpan(sourceScript, language, serviceScript, textChange.span, filter) ?? [];
    if (sourceSpan && sourceSpanFileName) {
        return [sourceSpanFileName, {
                newText: textChange.newText,
                span: sourceSpan,
            }];
    }
    return undefined;
}
exports.transformTextChange = transformTextChange;
function transformTextSpan(sourceScript, language, serviceScript, textSpan, filter) {
    const start = textSpan.start;
    const end = textSpan.start + textSpan.length;
    for (const sourceStart of toSourceOffsets(sourceScript, language, serviceScript, start, filter)) {
        for (const sourceEnd of toSourceOffsets(sourceScript, language, serviceScript, end, filter)) {
            if (sourceStart[0] === sourceEnd[0]
                && sourceEnd[1] >= sourceStart[1]) {
                return [sourceStart[0], {
                        start: sourceStart[1],
                        length: sourceEnd[1] - sourceStart[1],
                    }];
            }
        }
    }
}
exports.transformTextSpan = transformTextSpan;
function toSourceOffset(sourceScript, language, serviceScript, position, filter) {
    for (const source of toSourceOffsets(sourceScript, language, serviceScript, position, filter)) {
        return source;
    }
}
exports.toSourceOffset = toSourceOffset;
function* toSourceOffsets(sourceScript, language, serviceScript, position, filter) {
    if (sourceScript) {
        const map = language.maps.get(serviceScript.code, sourceScript);
        for (const [sourceOffset, mapping] of map.getSourceOffsets(position - getMappingOffset(language, serviceScript))) {
            if (filter(mapping.data)) {
                yield [sourceScript.id, sourceOffset];
            }
        }
    }
    else {
        for (const [fileName, _snapshot, map] of language.maps.forEach(serviceScript.code)) {
            for (const [sourceOffset, mapping] of map.getSourceOffsets(position - getMappingOffset(language, serviceScript))) {
                if (filter(mapping.data)) {
                    yield [fileName, sourceOffset];
                }
            }
        }
    }
}
exports.toSourceOffsets = toSourceOffsets;
function toGeneratedOffset(language, serviceScript, sourceScript, position, filter) {
    for (const [generateOffset] of toGeneratedOffsets(language, serviceScript, sourceScript, position, filter)) {
        return generateOffset;
    }
}
exports.toGeneratedOffset = toGeneratedOffset;
function* toGeneratedOffsets(language, serviceScript, sourceScript, position, filter) {
    const map = language.maps.get(serviceScript.code, sourceScript);
    for (const [generateOffset, mapping] of map.getGeneratedOffsets(position)) {
        if (filter(mapping.data)) {
            yield [generateOffset + getMappingOffset(language, serviceScript), mapping];
        }
    }
}
exports.toGeneratedOffsets = toGeneratedOffsets;
function getMappingOffset(language, serviceScript) {
    if (serviceScript.preventLeadingOffset) {
        return 0;
    }
    const sourceScript = language.scripts.fromVirtualCode(serviceScript.code);
    return sourceScript.snapshot.getLength();
}
exports.getMappingOffset = getMappingOffset;
//# sourceMappingURL=transform.js.map