"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createLanguageService = void 0;
const language_core_1 = require("@volar/language-core");
const vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
const vscode_uri_1 = require("vscode-uri");
const documents_1 = require("./documents");
const autoInsert = require("./features/provideAutoInsertSnippet");
const callHierarchy = require("./features/provideCallHierarchyItems");
const codeActions = require("./features/provideCodeActions");
const codeLens = require("./features/provideCodeLenses");
const colorPresentations = require("./features/provideColorPresentations");
const completions = require("./features/provideCompletionItems");
const definition = require("./features/provideDefinition");
const diagnostics = require("./features/provideDiagnostics");
const documentColors = require("./features/provideDocumentColors");
const documentDrop = require("./features/provideDocumentDropEdits");
const format = require("./features/provideDocumentFormattingEdits");
const documentHighlight = require("./features/provideDocumentHighlights");
const documentLink = require("./features/provideDocumentLinks");
const semanticTokens = require("./features/provideDocumentSemanticTokens");
const documentSymbols = require("./features/provideDocumentSymbols");
const fileReferences = require("./features/provideFileReferences");
const fileRename = require("./features/provideFileRenameEdits");
const foldingRanges = require("./features/provideFoldingRanges");
const hover = require("./features/provideHover");
const inlayHints = require("./features/provideInlayHints");
const linkedEditing = require("./features/provideLinkedEditingRanges");
const references = require("./features/provideReferences");
const rename = require("./features/provideRenameEdits");
const renamePrepare = require("./features/provideRenameRange");
const selectionRanges = require("./features/provideSelectionRanges");
const signatureHelp = require("./features/provideSignatureHelp");
const workspaceSymbol = require("./features/provideWorkspaceSymbols");
const codeActionResolve = require("./features/resolveCodeAction");
const codeLensResolve = require("./features/resolveCodeLens");
const completionResolve = require("./features/resolveCompletionItem");
const documentLinkResolve = require("./features/resolveDocumentLink");
const inlayHintResolve = require("./features/resolveInlayHint");
const uriMap_1 = require("./utils/uriMap");
function createLanguageService(language, plugins, env) {
    const documentVersions = (0, uriMap_1.createUriMap)();
    const map2DocMap = new WeakMap();
    const mirrorMap2DocMirrorMap = new WeakMap();
    const snapshot2Doc = new WeakMap();
    const embeddedContentScheme = 'volar-embedded-content';
    const context = {
        language,
        documents: {
            get(uri, languageId, snapshot) {
                if (!snapshot2Doc.has(snapshot)) {
                    snapshot2Doc.set(snapshot, (0, uriMap_1.createUriMap)());
                }
                const map = snapshot2Doc.get(snapshot);
                if (!map.has(uri)) {
                    const version = documentVersions.get(uri) ?? 0;
                    documentVersions.set(uri, version + 1);
                    map.set(uri, vscode_languageserver_textdocument_1.TextDocument.create(uri.toString(), languageId, version, snapshot.getText(0, snapshot.getLength())));
                }
                return map.get(uri);
            },
            *getMaps(virtualCode) {
                for (const [uri, snapshot, map] of context.language.maps.forEach(virtualCode)) {
                    if (!map2DocMap.has(map)) {
                        const embeddedUri = context.encodeEmbeddedDocumentUri(uri, virtualCode.id);
                        map2DocMap.set(map, new documents_1.SourceMapWithDocuments(this.get(uri, context.language.scripts.get(uri).languageId, snapshot), this.get(embeddedUri, virtualCode.languageId, virtualCode.snapshot), map, virtualCode));
                    }
                    yield map2DocMap.get(map);
                }
            },
            getLinkedCodeMap(virtualCode, documentUri) {
                const map = context.language.linkedCodeMaps.get(virtualCode);
                if (map) {
                    if (!mirrorMap2DocMirrorMap.has(map)) {
                        const embeddedUri = context.encodeEmbeddedDocumentUri(documentUri, virtualCode.id);
                        mirrorMap2DocMirrorMap.set(map, new documents_1.LinkedCodeMapWithDocument(this.get(embeddedUri, virtualCode.languageId, virtualCode.snapshot), map, virtualCode));
                    }
                    return mirrorMap2DocMirrorMap.get(map);
                }
            },
        },
        env,
        inject: (key, ...args) => {
            for (const plugin of context.plugins) {
                if (context.disabledServicePlugins.has(plugin[1])) {
                    continue;
                }
                const provide = plugin[1].provide?.[key];
                if (provide) {
                    return provide(...args);
                }
            }
        },
        plugins: [],
        commands: {
            rename: {
                create(uri, position) {
                    return {
                        title: '',
                        command: 'editor.action.rename',
                        arguments: [
                            uri,
                            position,
                        ],
                    };
                },
                is(command) {
                    return command.command === 'editor.action.rename';
                },
            },
            showReferences: {
                create(uri, position, locations) {
                    return {
                        title: locations.length === 1 ? '1 reference' : `${locations.length} references`,
                        command: 'editor.action.showReferences',
                        arguments: [
                            uri,
                            position,
                            locations,
                        ],
                    };
                },
                is(command) {
                    return command.command === 'editor.action.showReferences';
                },
            },
            setSelection: {
                create(position) {
                    return {
                        title: '',
                        command: 'setSelection',
                        arguments: [{
                                selection: {
                                    selectionStartLineNumber: position.line + 1,
                                    positionLineNumber: position.line + 1,
                                    selectionStartColumn: position.character + 1,
                                    positionColumn: position.character + 1,
                                },
                            }],
                    };
                },
                is(command) {
                    return command.command === 'setSelection';
                },
            },
        },
        disabledEmbeddedDocumentUris: (0, uriMap_1.createUriMap)(),
        disabledServicePlugins: new WeakSet(),
        decodeEmbeddedDocumentUri(maybeEmbeddedContentUri) {
            if (maybeEmbeddedContentUri.scheme === embeddedContentScheme) {
                const embeddedCodeId = decodeURIComponent(maybeEmbeddedContentUri.authority);
                const documentUri = decodeURIComponent(maybeEmbeddedContentUri.path.substring(1));
                return [
                    vscode_uri_1.URI.parse(documentUri),
                    embeddedCodeId,
                ];
            }
        },
        encodeEmbeddedDocumentUri(documentUri, embeddedContentId) {
            return vscode_uri_1.URI.from({
                scheme: embeddedContentScheme,
                authority: encodeURIComponent(embeddedContentId),
                path: '/' + encodeURIComponent(documentUri.toString()),
            });
        },
    };
    const api = {
        getSemanticTokenLegend: () => {
            const tokenModifiers = plugins.map(plugin => plugin.capabilities.semanticTokensProvider?.legend?.tokenModifiers ?? []).flat();
            const tokenTypes = plugins.map(plugin => plugin.capabilities.semanticTokensProvider?.legend?.tokenTypes ?? []).flat();
            return {
                tokenModifiers: [...new Set(tokenModifiers)],
                tokenTypes: [...new Set(tokenTypes)],
            };
        },
        getTriggerCharacters: () => plugins.map(plugin => plugin.capabilities.completionProvider?.triggerCharacters ?? []).flat(),
        getAutoFormatTriggerCharacters: () => plugins.map(plugin => plugin.capabilities.documentOnTypeFormattingProvider?.triggerCharacters ?? []).flat(),
        getSignatureHelpTriggerCharacters: () => plugins.map(plugin => plugin.capabilities.signatureHelpProvider?.triggerCharacters ?? []).flat(),
        getSignatureHelpRetriggerCharacters: () => plugins.map(plugin => plugin.capabilities.signatureHelpProvider?.retriggerCharacters ?? []).flat(),
        format: format.register(context),
        getFoldingRanges: foldingRanges.register(context),
        getSelectionRanges: selectionRanges.register(context),
        findLinkedEditingRanges: linkedEditing.register(context),
        findDocumentSymbols: documentSymbols.register(context),
        findDocumentColors: documentColors.register(context),
        getColorPresentations: colorPresentations.register(context),
        doValidation: diagnostics.register(context),
        findReferences: references.register(context),
        findFileReferences: fileReferences.register(context),
        findDefinition: definition.register(context, 'provideDefinition', language_core_1.isDefinitionEnabled),
        findTypeDefinition: definition.register(context, 'provideTypeDefinition', language_core_1.isTypeDefinitionEnabled),
        findImplementations: definition.register(context, 'provideImplementation', language_core_1.isImplementationEnabled),
        prepareRename: renamePrepare.register(context),
        doRename: rename.register(context),
        getEditsForFileRename: fileRename.register(context),
        getSemanticTokens: semanticTokens.register(context),
        doHover: hover.register(context),
        doComplete: completions.register(context),
        doCodeActions: codeActions.register(context),
        doCodeActionResolve: codeActionResolve.register(context),
        doCompletionResolve: completionResolve.register(context),
        getSignatureHelp: signatureHelp.register(context),
        doCodeLens: codeLens.register(context),
        doCodeLensResolve: codeLensResolve.register(context),
        findDocumentHighlights: documentHighlight.register(context),
        findDocumentLinks: documentLink.register(context),
        doDocumentLinkResolve: documentLinkResolve.register(context),
        findWorkspaceSymbols: workspaceSymbol.register(context),
        doAutoInsert: autoInsert.register(context),
        doDocumentDrop: documentDrop.register(context),
        getInlayHints: inlayHints.register(context),
        doInlayHintResolve: inlayHintResolve.register(context),
        callHierarchy: callHierarchy.register(context),
        dispose: () => context.plugins.forEach(plugin => plugin[1].dispose?.()),
        context,
    };
    for (const plugin of plugins) {
        context.plugins.push([plugin, plugin.create(context, api)]);
    }
    return api;
}
exports.createLanguageService = createLanguageService;
//# sourceMappingURL=languageService.js.map