"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.register = void 0;
const language_core_1 = require("@volar/language-core");
const vscode_uri_1 = require("vscode-uri");
const cancellation_1 = require("../utils/cancellation");
const featureWorkers_1 = require("../utils/featureWorkers");
const transform_1 = require("../utils/transform");
function register(context) {
    let lastResult;
    return async (uri, position, completionContext = { triggerKind: 1, }, token = cancellation_1.NoneCancellationToken) => {
        const sourceScript = context.language.scripts.get(uri);
        if (!sourceScript) {
            return {
                isIncomplete: false,
                items: [],
            };
        }
        if (completionContext?.triggerKind === 3
            && lastResult?.uri.toString() === uri.toString()) {
            for (const cacheData of lastResult.results) {
                if (!cacheData.list?.isIncomplete) {
                    continue;
                }
                const pluginIndex = context.plugins.findIndex(plugin => plugin[1] === cacheData.plugin);
                if (cacheData.embeddedDocumentUri) {
                    const decoded = context.decodeEmbeddedDocumentUri(cacheData.embeddedDocumentUri);
                    const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                    const virtualCode = decoded && sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                    if (!virtualCode) {
                        continue;
                    }
                    for (const map of context.documents.getMaps(virtualCode)) {
                        for (const mapped of map.getGeneratedPositions(position, data => (0, language_core_1.isCompletionEnabled)(data))) {
                            if (!cacheData.plugin.provideCompletionItems) {
                                continue;
                            }
                            cacheData.list = await cacheData.plugin.provideCompletionItems(map.embeddedDocument, mapped, completionContext, token);
                            if (!cacheData.list) {
                                continue;
                            }
                            for (const item of cacheData.list.items) {
                                item.data = {
                                    uri: uri.toString(),
                                    original: {
                                        additionalTextEdits: item.additionalTextEdits,
                                        textEdit: item.textEdit,
                                        data: item.data,
                                    },
                                    pluginIndex: pluginIndex,
                                    embeddedDocumentUri: map.embeddedDocument.uri,
                                };
                            }
                            cacheData.list = (0, transform_1.transformCompletionList)(cacheData.list, range => map.getSourceRange(range), map.embeddedDocument, context);
                        }
                    }
                }
                else {
                    if (!cacheData.plugin.provideCompletionItems) {
                        continue;
                    }
                    const document = context.documents.get(uri, sourceScript.languageId, sourceScript.snapshot);
                    cacheData.list = await cacheData.plugin.provideCompletionItems(document, position, completionContext, token);
                    if (!cacheData.list) {
                        continue;
                    }
                    for (const item of cacheData.list.items) {
                        item.data = {
                            uri: uri.toString(),
                            original: {
                                additionalTextEdits: item.additionalTextEdits,
                                textEdit: item.textEdit,
                                data: item.data,
                            },
                            pluginIndex: pluginIndex,
                            embeddedDocumentUri: undefined,
                        };
                    }
                }
            }
        }
        else {
            lastResult = {
                uri,
                results: [],
            };
            // monky fix https://github.com/johnsoncodehk/volar/issues/1358
            let isFirstMapping = true;
            let mainCompletionUri;
            const sortedPlugins = [...context.plugins]
                .filter(plugin => !context.disabledServicePlugins.has(plugin[1]))
                .sort((a, b) => sortServices(a[1], b[1]));
            const worker = async (document, position, map, codeInfo) => {
                for (const plugin of sortedPlugins) {
                    if (token.isCancellationRequested) {
                        break;
                    }
                    if (!plugin[1].provideCompletionItems) {
                        continue;
                    }
                    if (plugin[1].isAdditionalCompletion && !isFirstMapping) {
                        continue;
                    }
                    if (completionContext?.triggerCharacter && !plugin[0].capabilities.completionProvider?.triggerCharacters?.includes(completionContext.triggerCharacter)) {
                        continue;
                    }
                    const isAdditional = (codeInfo && typeof codeInfo.completion === 'object' && codeInfo.completion.isAdditional) || plugin[1].isAdditionalCompletion;
                    if (mainCompletionUri && (!isAdditional || mainCompletionUri !== document.uri)) {
                        continue;
                    }
                    // avoid duplicate items with .vue and .vue.html
                    if (plugin[1].isAdditionalCompletion && lastResult?.results.some(data => data.plugin === plugin[1])) {
                        continue;
                    }
                    let completionList = await plugin[1].provideCompletionItems(document, position, completionContext, token);
                    if (!completionList || !completionList.items.length) {
                        continue;
                    }
                    if (typeof codeInfo?.completion === 'object' && codeInfo.completion.onlyImport) {
                        completionList.items = completionList.items.filter(item => !!item.labelDetails);
                    }
                    if (!isAdditional) {
                        mainCompletionUri = document.uri;
                    }
                    const pluginIndex = context.plugins.indexOf(plugin);
                    for (const item of completionList.items) {
                        item.data = {
                            uri: uri.toString(),
                            original: {
                                additionalTextEdits: item.additionalTextEdits,
                                textEdit: item.textEdit,
                                data: item.data,
                            },
                            pluginIndex,
                            embeddedDocumentUri: map ? document.uri : undefined,
                        };
                    }
                    if (map) {
                        completionList = (0, transform_1.transformCompletionList)(completionList, range => map.getSourceRange(range, language_core_1.isCompletionEnabled), document, context);
                    }
                    lastResult?.results.push({
                        embeddedDocumentUri: map ? vscode_uri_1.URI.parse(document.uri) : undefined,
                        plugin: plugin[1],
                        list: completionList,
                    });
                }
                isFirstMapping = false;
            };
            if (sourceScript.generated) {
                for (const map of (0, featureWorkers_1.forEachEmbeddedDocument)(context, sourceScript.id, sourceScript.generated.root)) {
                    let _data;
                    for (const mappedPosition of map.getGeneratedPositions(position, data => {
                        _data = data;
                        return (0, language_core_1.isCompletionEnabled)(data);
                    })) {
                        await worker(map.embeddedDocument, mappedPosition, map, _data);
                    }
                }
            }
            else {
                const document = context.documents.get(uri, sourceScript.languageId, sourceScript.snapshot);
                await worker(document, position);
            }
        }
        return combineCompletionList(lastResult.results.map(cacheData => cacheData.list));
        function sortServices(a, b) {
            return (b.isAdditionalCompletion ? -1 : 1) - (a.isAdditionalCompletion ? -1 : 1);
        }
        function combineCompletionList(lists) {
            return {
                isIncomplete: lists.some(list => list?.isIncomplete),
                itemDefaults: lists.find(list => list?.itemDefaults)?.itemDefaults,
                items: lists.map(list => list?.items ?? []).flat(),
            };
        }
    };
}
exports.register = register;
//# sourceMappingURL=provideCompletionItems.js.map