"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createTypeScriptLS = void 0;
const language_service_1 = require("@volar/language-service");
const typescript_1 = require("@volar/typescript");
const path = require("path-browserify");
const vscode = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const fsFileSnapshots = (0, language_service_1.createUriMap)();
async function createTypeScriptLS(ts, tsLocalized, tsconfig, server, serviceEnv, workspaceFolder, getLanguagePlugins, { asUri, asFileName, }) {
    let parsedCommandLine;
    let projectVersion = 0;
    const sys = (0, typescript_1.createSys)(ts.sys, serviceEnv, workspaceFolder, {
        asFileName,
        asUri,
    });
    const projectHost = {
        getCurrentDirectory() {
            return asFileName(workspaceFolder);
        },
        getProjectVersion() {
            return projectVersion.toString();
        },
        getScriptFileNames() {
            return rootFiles;
        },
        getScriptSnapshot(fileName) {
            const uri = asUri(fileName);
            const documentKey = server.getSyncedDocumentKey(uri) ?? uri.toString();
            const document = server.documents.get(documentKey);
            askedFiles.set(uri, true);
            if (document) {
                return document.getSnapshot();
            }
        },
        getCompilationSettings() {
            return parsedCommandLine.options;
        },
        getLocalizedDiagnosticMessages: tsLocalized ? () => tsLocalized : undefined,
        getProjectReferences() {
            return parsedCommandLine.projectReferences;
        },
    };
    const languagePlugins = await getLanguagePlugins(serviceEnv, {
        configFileName: typeof tsconfig === 'string' ? tsconfig : undefined,
        projectHost,
        sys,
        asFileName,
        asUri,
    });
    const askedFiles = (0, language_service_1.createUriMap)();
    const docOpenWatcher = server.documents.onDidOpen(({ document }) => updateFsCacheFromSyncedDocument(document));
    const docSaveWatcher = server.documents.onDidSave(({ document }) => updateFsCacheFromSyncedDocument(document));
    const docChangeWatcher = server.documents.onDidChangeContent(() => projectVersion++);
    const fileWatch = serviceEnv.onDidChangeWatchedFiles?.(params => onWorkspaceFilesChanged(params.changes));
    let rootFiles = await getRootFiles(languagePlugins);
    const language = (0, language_service_1.createLanguage)([
        { getLanguageId: uri => server.documents.get(server.getSyncedDocumentKey(uri) ?? uri.toString())?.languageId },
        ...languagePlugins,
        { getLanguageId: uri => (0, typescript_1.resolveFileLanguageId)(uri.path) },
    ], (0, language_service_1.createUriMap)(sys.useCaseSensitiveFileNames), uri => {
        askedFiles.set(uri, true);
        const documentUri = server.getSyncedDocumentKey(uri);
        const syncedDocument = documentUri ? server.documents.get(documentUri) : undefined;
        let snapshot;
        if (syncedDocument) {
            snapshot = syncedDocument.getSnapshot();
        }
        else {
            // fs files
            const cache = fsFileSnapshots.get(uri);
            const fileName = asFileName(uri);
            const modifiedTime = sys.getModifiedTime?.(fileName)?.valueOf();
            if (!cache || cache[0] !== modifiedTime) {
                if (sys.fileExists(fileName)) {
                    const text = sys.readFile(fileName);
                    const snapshot = text !== undefined ? ts.ScriptSnapshot.fromString(text) : undefined;
                    fsFileSnapshots.set(uri, [modifiedTime, snapshot]);
                }
                else {
                    fsFileSnapshots.set(uri, [modifiedTime, undefined]);
                }
            }
            snapshot = fsFileSnapshots.get(uri)?.[1];
        }
        if (snapshot) {
            language.scripts.set(uri, snapshot);
        }
        else {
            language.scripts.delete(uri);
        }
    });
    language.typescript = {
        configFileName: typeof tsconfig === 'string' ? tsconfig : undefined,
        sys,
        asScriptId: asUri,
        asFileName: asFileName,
        ...(0, typescript_1.createLanguageServiceHost)(ts, sys, language, asUri, projectHost),
    };
    const languageService = (0, language_service_1.createLanguageService)(language, server.languageServicePlugins, serviceEnv);
    return {
        askedFiles,
        languageService,
        tryAddFile(fileName) {
            if (!rootFiles.includes(fileName)) {
                rootFiles.push(fileName);
                projectVersion++;
            }
        },
        dispose: () => {
            sys.dispose();
            languageService?.dispose();
            fileWatch?.dispose();
            docOpenWatcher.dispose();
            docSaveWatcher.dispose();
            docChangeWatcher.dispose();
        },
        getParsedCommandLine: () => parsedCommandLine,
    };
    function updateFsCacheFromSyncedDocument(document) {
        const uri = vscode_uri_1.URI.parse(document.uri);
        const fileName = asFileName(uri);
        if (fsFileSnapshots.has(uri) || sys.fileExists(fileName)) {
            const modifiedTime = sys.getModifiedTime?.(fileName);
            fsFileSnapshots.set(uri, [modifiedTime?.valueOf(), document.getSnapshot()]);
        }
    }
    async function getRootFiles(languagePlugins) {
        parsedCommandLine = await createParsedCommandLine(ts, sys, asFileName(workspaceFolder), tsconfig, languagePlugins.map(plugin => plugin.typescript?.extraFileExtensions ?? []).flat());
        return parsedCommandLine.fileNames;
    }
    async function onWorkspaceFilesChanged(changes) {
        const createsAndDeletes = changes.filter(change => change.type !== vscode.FileChangeType.Changed);
        if (createsAndDeletes.length) {
            rootFiles = await getRootFiles(languagePlugins);
        }
        projectVersion++;
    }
}
exports.createTypeScriptLS = createTypeScriptLS;
async function createParsedCommandLine(ts, sys, workspacePath, tsconfig, extraFileExtensions) {
    let content = {
        errors: [],
        fileNames: [],
        options: {},
    };
    let sysVersion;
    let newSysVersion = await sys.sync();
    while (sysVersion !== newSysVersion) {
        sysVersion = newSysVersion;
        try {
            if (typeof tsconfig === 'string') {
                const config = ts.readJsonConfigFile(tsconfig, sys.readFile);
                content = ts.parseJsonSourceFileConfigFileContent(config, sys, path.dirname(tsconfig), {}, tsconfig, undefined, extraFileExtensions);
            }
            else {
                content = ts.parseJsonConfigFileContent({ files: [] }, sys, workspacePath, tsconfig, workspacePath + '/jsconfig.json', undefined, extraFileExtensions);
            }
            // fix https://github.com/johnsoncodehk/volar/issues/1786
            // https://github.com/microsoft/TypeScript/issues/30457
            // patching ts server broke with outDir + rootDir + composite/incremental
            content.options.outDir = undefined;
            content.fileNames = content.fileNames.map(fileName => fileName.replace(/\\/g, '/'));
        }
        catch {
            // will be failed if web fs host first result not ready
        }
        newSysVersion = await sys.sync();
    }
    if (content) {
        return content;
    }
    return content;
}
//# sourceMappingURL=typescriptProjectLs.js.map