"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getWorkspaceFolder = exports.isFileInDir = exports.sortTSConfigs = exports.createUriConverter = exports.createTypeScriptProject = void 0;
const language_service_1 = require("@volar/language-service");
const path = require("path-browserify");
const vscode = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const inferredCompilerOptions_1 = require("./inferredCompilerOptions");
const simpleProject_1 = require("./simpleProject");
const typescriptProjectLs_1 = require("./typescriptProjectLs");
const rootTsConfigNames = ['tsconfig.json', 'jsconfig.json'];
function createTypeScriptProject(ts, tsLocalized, getLanguagePlugins) {
    let server;
    const { asFileName, asUri } = createUriConverter();
    const configProjects = (0, language_service_1.createUriMap)();
    const inferredProjects = (0, language_service_1.createUriMap)();
    const rootTsConfigs = new Set();
    const searchedDirs = new Set();
    const projects = {
        setup(_server) {
            server = _server;
            server.onDidChangeWatchedFiles(({ changes }) => {
                const tsConfigChanges = changes.filter(change => rootTsConfigNames.includes(change.uri.substring(change.uri.lastIndexOf('/') + 1)));
                for (const change of tsConfigChanges) {
                    const changeUri = vscode_uri_1.URI.parse(change.uri);
                    const changeFileName = asFileName(changeUri);
                    if (change.type === vscode.FileChangeType.Created) {
                        rootTsConfigs.add(changeFileName);
                    }
                    else if ((change.type === vscode.FileChangeType.Changed || change.type === vscode.FileChangeType.Deleted) && configProjects.has(changeUri)) {
                        if (change.type === vscode.FileChangeType.Deleted) {
                            rootTsConfigs.delete(changeFileName);
                        }
                        const project = configProjects.get(changeUri);
                        configProjects.delete(changeUri);
                        project?.then(project => project.dispose());
                    }
                }
                if (tsConfigChanges.length) {
                    server.clearPushDiagnostics();
                }
                server.refresh(projects);
            });
        },
        async getLanguageService(uri) {
            const tsconfig = await findMatchTSConfig(server, uri);
            if (tsconfig) {
                const project = await getOrCreateConfiguredProject(server, tsconfig);
                return project.languageService;
            }
            const workspaceFolder = getWorkspaceFolder(uri, server.workspaceFolders);
            const project = await getOrCreateInferredProject(server, uri, workspaceFolder);
            return project.languageService;
        },
        async getExistingLanguageServices() {
            const projects = await Promise.all([
                ...configProjects.values() ?? [],
                ...inferredProjects.values() ?? [],
            ]);
            return projects.map(project => project.languageService);
        },
        reload() {
            for (const project of [
                ...configProjects.values() ?? [],
                ...inferredProjects.values() ?? [],
            ]) {
                project.then(p => p.dispose());
            }
            configProjects.clear();
            inferredProjects.clear();
        },
    };
    return projects;
    async function findMatchTSConfig(server, uri) {
        const fileName = asFileName(uri);
        let dir = path.dirname(fileName);
        while (true) {
            if (searchedDirs.has(dir)) {
                break;
            }
            searchedDirs.add(dir);
            for (const tsConfigName of rootTsConfigNames) {
                const tsconfigPath = path.join(dir, tsConfigName);
                if ((await server.fs.stat?.(asUri(tsconfigPath)))?.type === language_service_1.FileType.File) {
                    rootTsConfigs.add(tsconfigPath);
                }
            }
            dir = path.dirname(dir);
        }
        await prepareClosestootParsedCommandLine();
        return await findDirectIncludeTsconfig() ?? await findIndirectReferenceTsconfig();
        async function prepareClosestootParsedCommandLine() {
            let matches = [];
            for (const rootTsConfig of rootTsConfigs) {
                if (isFileInDir(fileName, path.dirname(rootTsConfig))) {
                    matches.push(rootTsConfig);
                }
            }
            matches = matches.sort((a, b) => sortTSConfigs(fileName, a, b));
            if (matches.length) {
                await getParsedCommandLine(matches[0]);
            }
        }
        function findIndirectReferenceTsconfig() {
            return findTSConfig(async (tsconfig) => {
                const tsconfigUri = asUri(tsconfig);
                const project = await configProjects.get(tsconfigUri);
                return project?.askedFiles.has(uri) ?? false;
            });
        }
        function findDirectIncludeTsconfig() {
            return findTSConfig(async (tsconfig) => {
                const map = (0, language_service_1.createUriMap)();
                const parsedCommandLine = await getParsedCommandLine(tsconfig);
                for (const fileName of parsedCommandLine?.fileNames ?? []) {
                    const uri = asUri(fileName);
                    map.set(uri, true);
                }
                return map.has(uri);
            });
        }
        async function findTSConfig(match) {
            const checked = new Set();
            for (const rootTsConfig of [...rootTsConfigs].sort((a, b) => sortTSConfigs(fileName, a, b))) {
                const tsconfigUri = asUri(rootTsConfig);
                const project = await configProjects.get(tsconfigUri);
                if (project) {
                    let chains = await getReferencesChains(project.getParsedCommandLine(), rootTsConfig, []);
                    // This is to be consistent with tsserver behavior
                    chains = chains.reverse();
                    for (const chain of chains) {
                        for (let i = chain.length - 1; i >= 0; i--) {
                            const tsconfig = chain[i];
                            if (checked.has(tsconfig)) {
                                continue;
                            }
                            checked.add(tsconfig);
                            if (await match(tsconfig)) {
                                return tsconfig;
                            }
                        }
                    }
                }
            }
        }
        async function getReferencesChains(parsedCommandLine, tsConfig, before) {
            if (parsedCommandLine.projectReferences?.length) {
                const newChains = [];
                for (const projectReference of parsedCommandLine.projectReferences) {
                    let tsConfigPath = projectReference.path.replace(/\\/g, '/');
                    // fix https://github.com/johnsoncodehk/volar/issues/712
                    if ((await server.fs.stat?.(asUri(tsConfigPath)))?.type === language_service_1.FileType.File) {
                        const newTsConfigPath = path.join(tsConfigPath, 'tsconfig.json');
                        const newJsConfigPath = path.join(tsConfigPath, 'jsconfig.json');
                        if ((await server.fs.stat?.(asUri(newTsConfigPath)))?.type === language_service_1.FileType.File) {
                            tsConfigPath = newTsConfigPath;
                        }
                        else if ((await server.fs.stat?.(asUri(newJsConfigPath)))?.type === language_service_1.FileType.File) {
                            tsConfigPath = newJsConfigPath;
                        }
                    }
                    const beforeIndex = before.indexOf(tsConfigPath); // cycle
                    if (beforeIndex >= 0) {
                        newChains.push(before.slice(0, Math.max(beforeIndex, 1)));
                    }
                    else {
                        const referenceParsedCommandLine = await getParsedCommandLine(tsConfigPath);
                        if (referenceParsedCommandLine) {
                            for (const chain of await getReferencesChains(referenceParsedCommandLine, tsConfigPath, [...before, tsConfig])) {
                                newChains.push(chain);
                            }
                        }
                    }
                }
                return newChains;
            }
            else {
                return [[...before, tsConfig]];
            }
        }
        async function getParsedCommandLine(tsConfig) {
            const project = await getOrCreateConfiguredProject(server, tsConfig);
            return project?.getParsedCommandLine();
        }
    }
    function getOrCreateConfiguredProject(server, tsconfig) {
        tsconfig = tsconfig.replace(/\\/g, '/');
        const tsconfigUri = asUri(tsconfig);
        let projectPromise = configProjects.get(tsconfigUri);
        if (!projectPromise) {
            const workspaceFolder = getWorkspaceFolder(tsconfigUri, server.workspaceFolders);
            const serviceEnv = (0, simpleProject_1.createLanguageServiceEnvironment)(server, [workspaceFolder]);
            projectPromise = (0, typescriptProjectLs_1.createTypeScriptLS)(ts, tsLocalized, tsconfig, server, serviceEnv, workspaceFolder, getLanguagePlugins, { asUri, asFileName });
            configProjects.set(tsconfigUri, projectPromise);
        }
        return projectPromise;
    }
    async function getOrCreateInferredProject(server, uri, workspaceFolder) {
        if (!inferredProjects.has(workspaceFolder)) {
            inferredProjects.set(workspaceFolder, (async () => {
                const inferOptions = await (0, inferredCompilerOptions_1.getInferredCompilerOptions)(server);
                const serviceEnv = (0, simpleProject_1.createLanguageServiceEnvironment)(server, [workspaceFolder]);
                return (0, typescriptProjectLs_1.createTypeScriptLS)(ts, tsLocalized, inferOptions, server, serviceEnv, workspaceFolder, getLanguagePlugins, { asUri, asFileName });
            })());
        }
        const project = await inferredProjects.get(workspaceFolder);
        project.tryAddFile(asFileName(uri));
        return project;
    }
}
exports.createTypeScriptProject = createTypeScriptProject;
function createUriConverter() {
    const encodeds = new Map();
    return {
        asFileName,
        asUri,
    };
    function asFileName(parsed) {
        if (parsed.scheme === 'file') {
            return parsed.fsPath.replace(/\\/g, '/');
        }
        const encoded = encodeURIComponent(`${parsed.scheme}://${parsed.authority}`);
        encodeds.set(encoded, parsed);
        return `/${encoded}${parsed.path}`;
    }
    function asUri(fileName) {
        for (const [encoded, uri] of encodeds) {
            const prefix = `/${encoded}`;
            if (fileName.startsWith(prefix)) {
                return vscode_uri_1.URI.from({
                    scheme: uri.scheme,
                    authority: uri.authority,
                    path: fileName.substring(prefix.length),
                });
            }
        }
        return vscode_uri_1.URI.file(fileName);
    }
}
exports.createUriConverter = createUriConverter;
function sortTSConfigs(file, a, b) {
    const inA = isFileInDir(file, path.dirname(a));
    const inB = isFileInDir(file, path.dirname(b));
    if (inA !== inB) {
        const aWeight = inA ? 1 : 0;
        const bWeight = inB ? 1 : 0;
        return bWeight - aWeight;
    }
    const aLength = a.split('/').length;
    const bLength = b.split('/').length;
    if (aLength === bLength) {
        const aWeight = path.basename(a) === 'tsconfig.json' ? 1 : 0;
        const bWeight = path.basename(b) === 'tsconfig.json' ? 1 : 0;
        return bWeight - aWeight;
    }
    return bLength - aLength;
}
exports.sortTSConfigs = sortTSConfigs;
function isFileInDir(fileName, dir) {
    const relative = path.relative(dir, fileName);
    return !!relative && !relative.startsWith('..') && !path.isAbsolute(relative);
}
exports.isFileInDir = isFileInDir;
function getWorkspaceFolder(uri, workspaceFolders) {
    while (true) {
        if (workspaceFolders.has(uri)) {
            return uri;
        }
        const next = uri.with({ path: uri.path.substring(0, uri.path.lastIndexOf('/')) });
        if (next.path === uri.path) {
            break;
        }
        uri = next;
    }
    for (const folder of workspaceFolders.keys()) {
        return folder;
    }
    return uri.with({ path: '/' });
}
exports.getWorkspaceFolder = getWorkspaceFolder;
//# sourceMappingURL=typescriptProject.js.map