"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLanguagePlugins = exports.server = exports.connection = void 0;
const node_1 = require("@volar/language-server/node");
const language_core_1 = require("@vue/language-core");
const language_service_1 = require("@vue/language-service");
const tsPluginClient = require("@vue/typescript-plugin/lib/client");
const utils_1 = require("@vue/typescript-plugin/lib/utils");
const vscode_uri_1 = require("vscode-uri");
const hybridModeProject_1 = require("./lib/hybridModeProject");
const protocol_1 = require("./lib/protocol");
let tsdk;
let hybridMode;
let getTsPluginClient;
const envToVueOptions = new WeakMap();
const watchedExtensions = new Set();
exports.connection = (0, node_1.createConnection)();
exports.server = (0, node_1.createServer)(exports.connection);
const getLanguagePlugins = async ({ serviceEnv, configFileName, projectHost, sys, asFileName }) => {
    const commandLine = await parseCommandLine();
    const vueOptions = commandLine?.vueOptions ?? (0, language_core_1.resolveVueCompilerOptions)({});
    const vueLanguagePlugin = (0, language_core_1.createVueLanguagePlugin)(tsdk.typescript, asFileName, () => projectHost?.getProjectVersion?.() ?? '', fileName => {
        const fileMap = new language_core_1.FileMap(sys?.useCaseSensitiveFileNames ?? false);
        for (const vueFileName of projectHost?.getScriptFileNames() ?? []) {
            fileMap.set(vueFileName, undefined);
        }
        return fileMap.has(fileName);
    }, commandLine?.options ?? {}, vueOptions);
    if (!hybridMode) {
        const extensions = [
            'js', 'cjs', 'mjs', 'ts', 'cts', 'mts', 'jsx', 'tsx', 'json',
            ...vueOptions.extensions.map(ext => ext.slice(1)),
            ...vueOptions.vitePressExtensions.map(ext => ext.slice(1)),
            ...vueOptions.petiteVueExtensions.map(ext => ext.slice(1)),
        ];
        const newExtensions = extensions.filter(ext => !watchedExtensions.has(ext));
        if (newExtensions.length) {
            for (const ext of newExtensions) {
                watchedExtensions.add(ext);
            }
            exports.server.watchFiles(['**/*.{' + newExtensions.join(',') + '}']);
        }
    }
    envToVueOptions.set(serviceEnv, vueOptions);
    return [vueLanguagePlugin];
    async function parseCommandLine() {
        let commandLine;
        let sysVersion;
        if (sys) {
            let newSysVersion = await sys.sync();
            while (sysVersion !== newSysVersion) {
                sysVersion = newSysVersion;
                if (configFileName) {
                    commandLine = (0, language_core_1.createParsedCommandLine)(tsdk.typescript, sys, configFileName);
                }
                newSysVersion = await sys.sync();
            }
        }
        return commandLine;
    }
};
exports.getLanguagePlugins = getLanguagePlugins;
exports.connection.listen();
exports.connection.onInitialize(params => {
    const options = params.initializationOptions;
    hybridMode = options.vue?.hybridMode ?? true;
    tsdk = (0, node_1.loadTsdkByPath)(options.typescript.tsdk, params.locale);
    if (hybridMode) {
        getTsPluginClient = () => tsPluginClient;
    }
    else {
        getTsPluginClient = (0, language_service_1.createDefaultGetTsPluginClient)(tsdk.typescript);
    }
    const result = exports.server.initialize(params, hybridMode
        ? (0, hybridModeProject_1.createHybridModeProject)(tsdk.typescript.sys, exports.getLanguagePlugins)
        : (0, node_1.createTypeScriptProject)(tsdk.typescript, tsdk.diagnosticMessages, (env, ctx) => (0, exports.getLanguagePlugins)({
            serviceEnv: env,
            configFileName: ctx.configFileName,
            projectHost: ctx.projectHost,
            sys: ctx.sys,
            asFileName: ctx.asFileName,
        })), (0, language_service_1.getVueLanguageServicePlugins)(tsdk.typescript, env => envToVueOptions.get(env), getTsPluginClient, hybridMode), {
        pullModelDiagnostics: hybridMode,
    });
    if (hybridMode) {
        // provided by tsserver + @vue/typescript-plugin
        result.capabilities.semanticTokensProvider = undefined;
    }
    return result;
});
exports.connection.onInitialized(() => {
    exports.server.initialized();
});
exports.connection.onShutdown(() => {
    exports.server.shutdown();
});
exports.connection.onRequest(protocol_1.ParseSFCRequest.type, params => {
    return (0, language_core_1.parse)(params);
});
exports.connection.onRequest(protocol_1.DetectNameCasingRequest.type, async (params) => {
    const uri = vscode_uri_1.URI.parse(params.textDocument.uri);
    const languageService = await getService(uri);
    if (languageService) {
        return await (0, language_service_1.detect)(languageService.context, uri);
    }
});
exports.connection.onRequest(protocol_1.GetConvertTagCasingEditsRequest.type, async (params) => {
    const uri = vscode_uri_1.URI.parse(params.textDocument.uri);
    const languageService = await getService(uri);
    if (languageService) {
        return await (0, language_service_1.convertTagName)(languageService.context, uri, params.casing, getTsPluginClient(languageService.context));
    }
});
exports.connection.onRequest(protocol_1.GetConvertAttrCasingEditsRequest.type, async (params) => {
    const uri = vscode_uri_1.URI.parse(params.textDocument.uri);
    const languageService = await getService(uri);
    if (languageService) {
        return await (0, language_service_1.convertAttrName)(languageService.context, uri, params.casing, getTsPluginClient(languageService.context));
    }
});
exports.connection.onRequest(protocol_1.GetConnectedNamedPipeServerRequest.type, async (fileName) => {
    const server = (await (0, utils_1.searchNamedPipeServerForFile)(fileName))?.server;
    if (server) {
        return server;
    }
});
async function getService(uri) {
    return (await exports.server.project.getLanguageService(uri));
}
//# sourceMappingURL=node.js.map