"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.notEmpty = exports.createHybridModeProject = void 0;
const simpleProject_1 = require("@volar/language-server/lib/project/simpleProject");
const language_core_1 = require("@vue/language-core");
const language_service_1 = require("@vue/language-service");
const utils_1 = require("@vue/typescript-plugin/lib/utils");
const vscode_uri_1 = require("vscode-uri");
function createHybridModeProject(sys, getLanguagePlugins) {
    let initialized = false;
    let simpleLs;
    let serviceEnv;
    let server;
    const tsconfigProjects = (0, language_service_1.createUriMap)(sys.useCaseSensitiveFileNames);
    return {
        setup(_server) {
            server = _server;
        },
        async getLanguageService(uri) {
            if (!initialized) {
                initialized = true;
                initialize(server);
            }
            const fileName = asFileName(uri);
            const projectInfo = (await (0, utils_1.searchNamedPipeServerForFile)(fileName))?.projectInfo;
            if (projectInfo?.kind === 1) {
                const tsconfig = projectInfo.name;
                const tsconfigUri = vscode_uri_1.URI.file(tsconfig);
                if (!tsconfigProjects.has(tsconfigUri)) {
                    tsconfigProjects.set(tsconfigUri, (async () => {
                        serviceEnv ??= (0, simpleProject_1.createLanguageServiceEnvironment)(server, [...server.workspaceFolders.keys()]);
                        const languagePlugins = await getLanguagePlugins({
                            serviceEnv,
                            configFileName: tsconfig,
                            sys: {
                                ...sys,
                                version: 0,
                                async sync() {
                                    return 0;
                                },
                                dispose() { },
                            },
                            asFileName,
                        });
                        return createLs(server, serviceEnv, languagePlugins);
                    })());
                }
                return await tsconfigProjects.get(tsconfigUri);
            }
            else {
                simpleLs ??= (async () => {
                    serviceEnv ??= (0, simpleProject_1.createLanguageServiceEnvironment)(server, [...server.workspaceFolders.keys()]);
                    const languagePlugins = await getLanguagePlugins({ serviceEnv, asFileName });
                    return createLs(server, serviceEnv, languagePlugins);
                })();
                return await simpleLs;
            }
        },
        async getExistingLanguageServices() {
            return Promise.all([
                ...tsconfigProjects.values(),
                simpleLs,
            ].filter(notEmpty));
        },
        reload() {
            for (const ls of [
                ...tsconfigProjects.values(),
                simpleLs,
            ]) {
                ls?.then(ls => ls.dispose());
            }
            tsconfigProjects.clear();
            simpleLs = undefined;
        },
    };
    function asFileName(uri) {
        return uri.fsPath.replace(/\\/g, '/');
    }
    function initialize(server) {
        server.onDidChangeWatchedFiles(({ changes }) => {
            for (const change of changes) {
                const changeUri = vscode_uri_1.URI.parse(change.uri);
                if (tsconfigProjects.has(changeUri)) {
                    tsconfigProjects.get(changeUri)?.then(project => project.dispose());
                    tsconfigProjects.delete(changeUri);
                    server.clearPushDiagnostics();
                }
            }
        });
    }
    function createLs(server, serviceEnv, languagePlugins) {
        const language = (0, language_core_1.createLanguage)([
            { getLanguageId: uri => server.documents.get(server.getSyncedDocumentKey(uri) ?? uri.toString())?.languageId },
            ...languagePlugins,
        ], (0, language_service_1.createUriMap)(), uri => {
            const documentKey = server.getSyncedDocumentKey(uri);
            const document = documentKey ? server.documents.get(documentKey) : undefined;
            if (document) {
                language.scripts.set(uri, document.getSnapshot(), document.languageId);
            }
            else {
                language.scripts.delete(uri);
            }
        });
        return (0, language_service_1.createLanguageService)(language, server.languageServicePlugins, serviceEnv);
    }
}
exports.createHybridModeProject = createHybridModeProject;
function notEmpty(value) {
    return value !== null && value !== undefined;
}
exports.notEmpty = notEmpty;
//# sourceMappingURL=hybridModeProject.js.map