package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.38.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.38.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`/@`!````0"(```````!``````````$"Q````
M`````````$``.``-`$``&@`9``8````$````0`````````!``````````$``
M````````V`(```````#8`@````````@``````````P````0````8`P``````
M`!@#````````&`,````````<`````````!P``````````0`````````!````
M!````````````````````````````````````/`9````````\!D`````````
M$`````````$````%`````"``````````(``````````@````````P3P`````
M``#!/``````````0`````````0````0`````8`````````!@`````````&``
M``````#(1P```````,A'`````````!`````````!````!@```-"K````````
MT+L```````#0NP```````%@$````````@`0`````````$`````````(````&
M````X*L```````#@NP```````."[``````````(``````````@````````@`
M````````!`````0````X`P```````#@#````````.`,```````!0````````
M`%``````````"``````````$````!````(@#````````B`,```````"(`P``
M`````$0`````````1``````````$`````````%/E=&0$````.`,````````X
M`P```````#@#````````4`````````!0``````````@`````````4.5T9`0`
M``#`H````````,"@````````P*````````#L`````````.P`````````!```
M``````!1Y71D!@``````````````````````````````````````````````
M```````````0`````````%+E=&0$````T*L```````#0NP```````-"[````
M````,`0````````P!`````````$`````````+VQI8C8T+VQD+6QI;G5X+7@X
M-BTV-"YS;RXR```````$````0`````4```!'3E4``@``P`0````#````````
M``*``,`$`````0`````````!``'`!`````D``````````@`!P`0````!````
M``````0````4`````P```$=.50`0]"Q5^EY%07'=:(4@\1?.-\Y4]@0````0
M`````0```$=.50``````!`````0``````````````"L```!#````!`````@`
M```@$!``!4`$A"@"``2`A@"$@HD``@D@!`B(@!!!N@`""$,`````````````
M`$0``````````````$4`````````1@```$<`````````2`````````!*````
M``````````!+````3`````````!-````3@```%``````````40```%(```!3
M``````````````````````````````!4````50```%8```!7````6```````
M````````6@````````!;``````````````")!W#3:QJ>4*/@^TF93\.;XT?&
M`6:OS&9#1=7LN^.2?+,!BV]OQ=\?%8PES"[%F[9K"4/6(?_D#ONA;&##Y3\L
M#3J7"\_>1RW9<5@<?ZF<F^&G`@I"^M>':W^:?#-BV^W49-1\K4OCP```````
M``````````````````````````````":`@``$@``````````````````````
M```!`@``$@````````````````````````",````$0``````````````````
M``````"!`@``$@`````````````````````````^`0``$@``````````````
M``````````#<`0``$@`````````````````````````]`@``$@``````````
M```````````````!````(`````````````````````````"L`0``$@``````
M``````````````````!G````$@````````````````````````"Y`0``$@``
M``````````````````````"4````$@````````````````````````!Z`@``
M$@````````````````````````#,`@``$@````````````````````````#Q
M`0``$@````````````````````````#2````$@``````````````````````
M``#B````$@`````````````````````````P`0``$@``````````````````
M``````!D`0``$@````````````````````````"*````$0``````````````
M``````````!J`P``$@`````````````````````````0````(```````````
M``````````````"R`@``$@````````````````````````"?`@``$@``````
M``````````````````"]````$@`````````````````````````L````(```
M``````````````````````"=````$@````````````````````````"K````
M$@````````````````````````"=`0``$@````````````````````````"D
M`@``$@````````````````````````!8````(@``````````````````````
M``#H````%@````````````````````````#[````$@``````````````````
M``````!0`@``$@`````````````````````````6`0``$@``````````````
M``````````"]`@``$@````````````````````````".`0``$@``````````
M``````````````#J`0``$@````````````````````````!%`0``$@``````
M```````````````````G`0``$@````````````````````````!4`0``$@``
M``````````````````````!R`0``$@````````````````````````",`@``
M$@````````````````````````#+`0``$@````````````````````````!Y
M`0``$@`````````````````````````B`@``$@``````````````````````
M``!Y````$@````````````````````````"'`0``$@``````````````````
M``````"``0``$@````````````````````````"<````$@``````````````
M``````````"^`0``$@`````````````````````````-`@``$@``````````
M``````````````!N`@``$@````````````````````````"Q`0``$@``````
M``````````````````#9````$@````````````````````````#$````$@``
M``````````````````````"6`0``$0````````````````````````!A`@``
M$0````````````````````````"D````$@````````````````````````!<
M`0``$@`````````````````````````+`0``$@``````````````````````
M```N`@``$@````````````````````````!&````$@``````````````````
M``````!K`0``$@````````````````````````"S````$@``````````````
M``````````#%`0``$@````````````````````````#[`P``$0`6`"#`````
M````"`````````!.`P``$@`-`$!:````````50(````````-!```$@`-`/!$
M````````!04````````%!```$@`-`'!*````````P`````````#[`@``$@`-
M`'!#````````*`````````!"`P``$@`-`*!#````````0@$```````#5`P``
M$``6`"C```````````````````##`P``$``7`%#````````````````````>
M`P``$@`-``!*````````:@````````!S`P``$@`-`&!7````````VP``````
M``"Z`P``$@`-`$!8````````2@````````#J`@``$@`-`#!!````````"@``
M``````".`P``$``6``#```````````````````#N`P``$@`-`$!!````````
M)0(```````#<`P``$@`-`-!,````````X`````````"F`P``$@`-`"!`````
M````*@````````"0`P``(``6``#```````````````````#(`P``$@`-`+!-
M````````10<```````"N`P``$``7`"C```````````````````!8`P``$@`-
M`*!,````````(P`````````L`P``$@`-`%!`````````V0````````";`P``
M$@`-`)!8````````IP$```````!3````$@`-`"`@````````(`(```````"4
M`P``$@`-`$`B````````)@`````````(`P``$@`-``!5````````4P(`````
M``!_`P``$0`/``!@````````!```````````7U]G;6]N7W-T87)T7U\`7TE4
M35]D97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE
M5&%B;&4`7U]L:6)C7W-T87)T7VUA:6X`7U]C>&%?9FEN86QI>F4`7U]R96=I
M<W1E<E]A=&9O<FL`7U]S=&%C:U]C:&M?9F%I;`!?7V5N=FER;VX`<W1R;F-M
M<`!L<W1A=#8T`&%C8V5S<P!O<&5N9&ER`')E861D:7(V-`!S=')L96X`7U]S
M<')I;G1F7V-H:P!U;FQI;FL`8VQO<V5D:7(`<FUD:7(`4$Q?8W5R<F5N=%]C
M;VYT97AT`&)O;W1?1'EN84QO861E<@!097)L7VYE=UA3`%!E<FQ?;F5W6%-?
M9FQA9W,`<F5A9&QI;FL`4&5R;%]S=E]S971P=@!M96UC:'(`4&5R;%]S=E]S
M971P=FX`;65M;6]V90!R96%L;&]C`&=E='!I9`!M96UC<'D`<W1R8VAR`'-T
M<F1U<`!S=')T;VL`<W1R8VUP`'-T<G)C:'(`<W1D97)R`%]?=F9P<FEN=&9?
M8VAK`&5X:70`;'-E96LV-`!R96%D`&=E='5I9`!M:V1I<@!?7V5R<FYO7VQO
M8V%T:6]N`%]?9G!R:6YT9E]C:&L`;W!E;C8T`%!E<FQ?9W9?9F5T8VAP=@!0
M97)L7V=E=%]S=@!097)L7V-R;V%K7VYO8V]N=&5X=`!097)L7VUG7W-E=`!0
M97)L7W-Y<U]I;FET,P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]A=&9O<FM?
M;&]C:P!03%]D;U]U;F1U;7``4&5R;%]S869E<WES;6%L;&]C`'!E<FQ?<&%R
M<V4`<&5R;%]D97-T<G5C=`!P97)L7V9R964`4&5R;%]S>7-?=&5R;0!P97)L
M7V%L;&]C`'!E<FQ?8V]N<W1R=6-T`'!E<FQ?<G5N`&QI8G!E<FPN<V\`;&EB
M8RYS;RXV`'!A<E]C=7)R96YT7V5X96,`<&%R7V)A<V5N86UE`%A37TEN=&5R
M;F%L<U]005)?0D]/5`!P87)?96YV7V-L96%N`'!A<E]C=7)R96YT7V5X96-?
M<')O8P!P87)?9&ER;F%M90!S:&%?9FEN86P`9V5T7W5S97)N86UE7V9R;VU?
M9V5T<'=U:60`<&%R7V-L96%N=7``7TE/7W-T9&EN7W5S960`7U]D871A7W-T
M87)T`'-H85]U<&1A=&4`>'-?:6YI=`!?7V)S<U]S=&%R=`!S:&%?:6YI=`!?
M96YD`'!A<E]M:W1M<&1I<@!?961A=&$`<&%R7W-E='5P7VQI8G!A=&@`<&%R
M7V9I;F1P<F]G`&UY7W!A<E]P;`!P87)?9&EE`'!A<E]I;FET7V5N=@!'3$E"
M0U]!0DE?1%1?4D5,4@!'3$E"0U\R+C$T`$=,24)#7S(N,S0`1TQ)0D-?,BXT
M`$=,24)#7S(N,S,`1TQ)0D-?,BXS+C(`1TQ)0D-?,BXS+C0`1TQ)0D-?,BXR
M+C4`+W5S<B]L:6(O<&5R;#4O-2XS."]C;W)E7W!E<FPO0T]210````$``0`"
M``$``@`#``$``0`"``0``@`"``(``0`!``(``@`!``(``@`"``$``0`"``(`
M`0`%``(``P`!``(``0`!``$``0`!``(``@`!``(``@`"``$``@`"``$`!@`"
M``(`!0`"``$``0`"``(``P`"``$``@`"``$``0`'``@``@`"``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!
M``@`X`(``!``````````0@[]````"0`:!```$````)21E@8```@`+`0``!``
M``"TD98&```'`#<$```0````%&EI#0``!@!"!```$````+.1E@8```4`3`0`
M`!````!R&6D)```$`%<$```0````=!EI"0```P!C!```$````'4::0D```(`
M;P0```````#XO0````````8````!````````````````O@````````8````"
M```````````````(O@````````8````#```````````````0O@````````8`
M```$```````````````8O@````````8````%```````````````@O@``````
M``8````&```````````````HO@````````8````'```````````````PO@``
M``````8````(```````````````XO@````````8````)``````````````!`
MO@````````8````*``````````````!(O@````````8````+````````````
M``!0O@````````8````,``````````````!8O@````````8````-````````
M``````!@O@````````8````.``````````````!HO@````````8````/````
M``````````!PO@````````8````0``````````````!XO@````````8````1
M``````````````"`O@````````8````2``````````````"(O@````````8`
M```3``````````````"0O@````````8````5``````````````"8O@``````
M``8````6``````````````"@O@````````8````7``````````````"HO@``
M``````8````8``````````````"PO@````````8````9``````````````"X
MO@````````8````:``````````````#`O@````````8````;````````````
M``#(O@````````8````<``````````````#0O@````````8````=````````
M``````#8O@````````8````>``````````````#@O@````````8````?````
M``````````#HO@```````!(````@``````````````#PO@````````8````A
M``````````````#XO@````````8````B````````````````OP````````8`
M```C```````````````(OP````````8````D```````````````0OP``````
M``8````E```````````````8OP````````8````F```````````````@OP``
M``````8````G```````````````HOP````````8````H```````````````P
MOP````````8````I```````````````XOP````````8````J````````````
M``!`OP````````8````K``````````````!(OP````````8````L````````
M``````!0OP````````8````M``````````````!8OP````````8````N````
M``````````!@OP````````8````O``````````````!HOP````````8````P
M``````````````!POP````````8````Q``````````````!XOP````````8`
M```R``````````````"`OP````````8````S``````````````"(OP``````
M``8````T``````````````"0OP````````8````U``````````````"8OP``
M``````8````V``````````````"@OP````````8````W``````````````"H
MOP````````8````X``````````````"POP````````8````Y````````````
M``"XOP````````8````Z``````````````#`OP````````8````[````````
M``````#(OP````````8````\``````````````#0OP````````8````]````
M``````````#8OP````````8````^``````````````#@OP````````8````_
M``````````````#HOP````````8```!```````````````#POP````````8`
M``!!``````````````#XOP````````8```!"``````````````#0NP``````
M``,`````````",`````````)````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#S#Q[Z2(/L"$B+!2&>``!(A<!T`O_02(/$",,``````/,/'OI54TB#[#B)
M?"0<2(U\)!Q(B70D$$B-="002(E4)`A(C50D"&1(BP0E*````$B)1"0H,<#_
M%7N?``!(BS7$G0``2(L]C9X``$B)\F?H+#P``$B+!3V?``"`.``/A"(!``!(
MBP5=G@``9$B+`("(.@0```.+1"0<C7@$C5@#2&/_2,'G`_\5XYX``$B+3"00
M1(M$)!Q(B<9(B06'GP``2(L!2(D&2(T%&4$``$B)1@A(C05'GP``2(L`2(E&
M$$B-!0-!``!(B48808/X`0^.,`$``$B#P0A!@\`#N`0```!F#Q]$``!(BSE(
M8]"#P`%(@\$(2(D\UD0YP'7J2)A%,<!(B?%(BST+GP``2,<$Q@````")VDB-
M-5(;``#_%<R<``"%P`^$C````$B+/>6>``!(C6PD)/\5XIT``$B)[DB-/3L_
M``")PS'`Z#\8``!(A<!T=$B+/;N>``#_%76<``#_%4^=``!(BT0D*&1(*P0E
M*`````^%G0```$B#Q#B)V%M=P_\5])P``$B)!86>``!(B<=(A<!T=/\51YT`
M`$B+!2"=``!D2(L0QH(N"0```.FS_O__2(L]69X``/\5>YP``.EB____2(L]
M3YX``$B%_W0+9^AD-0``Z77___](C3U0/@``2(GNZ*`7``!(B04IG@``2(G'
M2(7`#X13____Z]&X!````.GO_O__OP$```#_%?Z;``#_%2"=``#S#Q[Z,>U)
MB=%>2(GB2(/D\%!413'`,<E(C3W!_?___Q5[G0``]&8N#Q^$``````!(C3VQ
MG0``2(T%JIT``$@Y^'052(L%#IP``$B%P'0)_^`/'X``````PP\?@`````!(
MC3V!G0``2(TU>IT``$@I_DB)\$C![C](P?@#2`'&2-'^=!1(BP7MFP``2(7`
M=`C_X&8/'T0``,,/'X``````\P\>^H`]/9T```!U,U5(@SWJFP```$B)Y70-
M2(L]!IT``/\5V)L``.AC____Q@44G0```5W#9BX/'X0``````,-F9BX/'X0`
M``````\?0`#S#Q[ZZ6?___\/'X``````05=(B?Y,C4<X059!54%455-(@>R8
M`@``9$B+!"4H````2(F$)(@"```QP$B)XDR-C"2`````2(G7#Q^``````$&+
M`$V+$$B#QQ!)@\`(#\B)P$R)T4B)1_!,B=!(P>D@2,'H*$G!ZC@E`/\``$P)
MT$F)RDC!X0A)P>(8@>$``/\`18G23`G02`G(2(E'^$PYSW6M9@]O3"009@]O
M7"0@2(U$)&A(@<)H`@``9@]O5"0P9@]O+8=\```/'X``````\P]O`&8/;^)F
M#V]0V$B#P!!F#^](B&8/[\)F#^_!9@]OR&8/<]`?9@]S\0%F#^O!9@]ORV8/
M;]QF#]O%#RE`"$@YT'6[3(L.3(M>$$B+?B!(BUX83(G(3(G*2(MN"$B+3"0(
M2,'J&TC!X`5("=`#!"1(C8P+F7F"6HV4.)EY@EI,B=A(,=A((>A(,=B-/`)(
MB>I(B>A(P>@"2,'B'DF)^D@)PDR)V$G!ZAM!B=!,,<!,(<A,,=A(`<A(B?E(
MP>$%3`G13(M4)!!$C20(3(G(3(G)2,'I`DC!X!Y/C903F7F"6DV)Y4@)R$G!
M[1M(,<*)P4@A^DPQPDP!TDV)XDG!X@5-">I,BVPD&$$!TDB)^DC![P)(P>(>
M18G23XV$*)EY@EI("?I-B=5(,=")UTPAX$G![1M(,<A,`<!-B=!)P>`%30GH
M3(ML)"!!`<!,B>!)P>P"2,'@'D6)P$J-C"F9>8):3`G@38G%2#'"2<'M&T&)
MQ$PATD@Q^D@!RDR)P4C!X05,">E,BVPD*`'13(G22<'J`DC!XAZ)R4J-O"^9
M>8):3`G228G-2#'02<'M&T&)TDPAP$PQX$@!^$B)STC!YP5,">],BVPD,`''
M3(G`2<'H`DC!X!Z)_T^-I"R9>8):3`G`28G]2#'"2<'M&T&)P$@ARDPQTDP!
MXDF)_$G!Y`5-">Q,BVPD.$$!U$B)RDC!Z0)(P>(>18GD3XV4*IEY@EI("<I-
MB>5(,=!)P>T;B=%((?A,,<!,`=!-B>))P>(%30GJ3(ML)$!!`<)(B?A(P>\"
M2,'@'D6)TD^-A"B9>8):2`GX38G52#'"2<'M&XG'3"'B2#'*3`'"38G02<'@
M!4T)Z$R+;"1(00'03(GB2<'L`DC!XAY%B<!*C8PIF7F"6DP)XDV)Q4@QT$G!
M[1M!B=1,(=!(,?A(`<A,B<%(P>$%3`GI3(ML)%`!P4R)T$G!Z@)(P>`>B<E*
MC;POF7F"6DP)T$F)S4@QPDG![1M!B<),(<),,>)(`?I(B<](P><%3`GO3(ML
M)%@!UTR)PDG!Z`)(P>(>B?]/C:0LF7F"6DP)PDF)_4@QT$G![1M!B=!((<A,
M,=!,`>!)B?Q)P>0%30GL3(ML)&!!`<1(B<A(P>`>2,'I`D6)Y$^-E"J9>8):
M2`G(38GE2#'"2<'M&XG!2"'Z3#'"3`'238GB2<'B!4T)ZDR+;"1H00'22(GZ
M2,'O`DC!XAY%B=)/C80HF7F"6D@)^DV)U4@QT$G![1N)UTPAX$@QR$P!P$V)
MT$G!X`5-">A,BVPD<$$!P$R)X$G![`)(P>`>18G`2HV,*9EY@EI,">!-B<5(
M,<))P>T;08G$3"'22#'Z2`'*3(G!2,'A!4P)Z4R+;"1X`=%,B=)(P>(>2<'J
M`HG)2HV\+YEY@EI,"=))B<U(,=!)P>T;08G23"'`3#'@2`'X2(G/2,'G!4P)
M[TR+K"2``````<=,B<!)P>@"2,'@'HG_3XVD+)EY@EI,"<!)B?U(,<))P>T;
M08G`2"'*3#'23`'B28G\2<'D!4T)[$:-+")(B<I(P>D"3(ND)(@```!(P>(>
M2`G*3XV4(IEY@EI-B>Q(,=!)P>P;B=%((?A,,<!,`=!-B>I)P>(%30GB3(ND
M))````!!`<)(B?A(P>`>2,'O`D6)TD^-A""9>8):2`GX38G42#'"2<'L&XG'
M3"'J2#'*3`'"38G02<'@!4T)X$6)P$P!PDV)Z$G![0))P>`>08G430GH3(NL
M))@```!,,<!,(=!*C8PIF7F"6DV)Y4@Q^$G![1M(`<A,B>%(P>$%3`GIB<E(
M`<A,B=%)P>H"2,'A'D&)Q4P)T4V)PD6)P$DQRD0QTDR+E"2@````2HV\%Z'K
MV6Y-B>I(`==,B>I)P>H;2,'B!4P)THG22`'73(GB2,'B'DG![`)!B?I,`X0D
MJ````$P)XDF)S(G)2`.,)+````!),=1$,>!-C:0`H>O9;DR)T$V)T$G!Z!M(
MP>`%3`G`B<!)`<1,B>A)P>T"2,'@'D6)X$P)Z$F)U8G22`.4)+@```!),<5$
M,>],C:PYH>O9;DR)P4R)QTC![QM(P>$%2`GYB<E)`<U,B=%)P>H"2,'A'D2)
M[TP)T4F)PHG`23'*13'B3HVD$J'KV6Y(B?I)B?I)P>H;2,'B!4P)THG220'4
M3(G"2<'H`DC!XAY%B>),"<))B<A(`X0DP````(G)23'02`.,),@```!%,>A.
MC:P`H>O9;DR)T$V)T$G!Z!M(P>`%3`G`B<!)`<5(B?A(P>\"2,'@'D6)Z$@)
M^$B)UXG22`.4)-````!(,<=$,>=,C:0YH>O9;DR)P4R)QTC![QM(P>$%2`GY
MB<E)`<Q,B=%)P>H"2,'A'D2)YTP)T4F)PHG`2`.$)-@```!),<I%,>I.C:P2
MH>O9;DB)^DF)^DG!ZAM(P>(%3`G2B=))`=5,B<))P>@"2,'B'D6)ZDP)PDF)
MR(G)23'02`.,).````!%,>!.C:0`H>O9;DR)T$V)T$G!Z!M(P>`%3`G`B<!)
M`<1(B?A(P>\"2,'@'D6)X$@)^$B)UXG22`.4).@```!(,<=$,>],C:PYH>O9
M;DR)P4R)QTC![QM(P>$%2`GYB<E)`<U,B=%)P>H"2,'A'D2)[TP)T4F)PHG`
M2`.$)/````!),<I%,>).C:02H>O9;DB)^DF)^DG!ZAM(P>(%3`G2B=))`=1,
MB<))P>@"2,'B'D6)XDP)PDF)R(G)23'013'H3HVL`*'KV6Y,B=!-B=!(P>`%
M2<'H&T@#C"3X````3`G`B<!)`<5(B?A(P>\"2,'@'D6)Z$@)^$B)UXG22`.4
M)``!``!(,<=$,>=,C:0YH>O9;DR)P4R)QTC![QM(P>$%2`GYB<E)`<Q,B=%)
MP>H"2,'A'D2)YTP)T4F)PHG`2`.$)`@!``!),<I%,>I.C:P2H>O9;DB)^DF)
M^DG!ZAM(P>(%3`G2B=))`=5,B<))P>@"2,'B'D6)ZDP)PDF)R(G)23'013'@
M3HVD`*'KV6Y,B=!-B=!)P>@;2,'@!4P)P$@#C"00`0``B<!)`<1(B?A(P>\"
M2,'@'D6)X$@)^$B)UXG22`.4)!@!``!(,<=$,>],C:PYH>O9;DR)P4R)QTC!
M[QM(P>$%2`GYB<E)`<U,B=%)P>H"2,'A'D2)[TP)T4F)PHG`2`.$)"`!``!)
M,<I%,>).C:02H>O9;DB)^DF)^DG!ZAM(P>(%3`G2B=))`=1,B<))P>@"2,'B
M'D6)XDP)PDF)R(G)23'013'H3HVL`*'KV6Y,B=!-B=!)P>@;2,'@!4P)P(G`
M20'%2(GX2,'@'DC![P)%B>A(`XPD*`$``$@)^$B)UXG22`.4)#`!``!(,<=$
M,>=-B<1(C8PYH>O9;DR)QTG![!M(P><%3`GG38G42<'J`DG!Y!Z)_TT)U$F)
MPD@!^8G`33'BB<](`X0D.`$``$4QZDF)_4Z-E!*AZ]EN2(GZ2<'M&TC!X@5,
M">I!`=),B<))P>@"2,'B'D6)TDP)PDV)X$6)Y$DQT$F)UD&)U40QP4V)T$B-
MC`BAZ]EN3(G02<'H&TC!X`5,"<!$C00!2(GY2,'O`KC<O!N/2,'A'D@)^4D)
MSHG/2"'Z32'620G62(N4)$`!``!(`<),`>)-B<1,`?)-B<9)P>0%2<'N&TT)
M]$$!U$R)TDG!Z@)(P>(>18GD3`G22`G128G608G222'^3"'!3`GQ3(NT)$@!
M``!)`<9-`?5-B>9,`>E-B>5)P>X;2<'E!4T)]4F)UD$!S4R)P4G!Z`)(P>$>
M18GM3`G!08G(20G.32'F3"'"3`GR3(NT)%`!``!)`<9,`?=-B>Y(`?I,B>])
MP>X;2,'G!4P)]P'73(GB2<'L`DC!XAZ)_TP)XD@)T4F)UD&)U$TAQDPAZ4P)
M\4R+M"18`0``20'&30'R28G^3`'128GZ2<'N&TG!X@5-"?))B=9!`<I,B>E)
MP>T"2,'A'D6)TDP)Z4&)S4D)SDDA_DPAZDP)\DR+M"1@`0``20'&30'P38G6
M3`'"38G02<'@!4G![AM-"?!!`=!(B?I(P>\"2,'B'D6)P$@)^D@)T4F)UHG7
M32'N3"'13`GQ3(NT)&@!``!)`<9-`?1-B<9,`>%-B<1)P>X;2<'D!4T)]$F)
MUD$!S$R)T4G!Z@)(P>$>18GD3`G108G*20G.32'&3"'23`GR3(NT)'`!``!)
M`<9-`?5-B>9,`>I-B>5)P>X;2<'E!4T)]4$!U4R)PDC!XAY)P>@"18GM3`G"
M2`G128G608G032'63"'A3`GQ3(NT)'@!``!)`<9,`?=-B>Y(`?E,B>])P>X;
M2,'G!4P)]TF)U@'/3(GA2<'L`DC!X1Z)_TP)X4&)S$D)SDTA[DPAXDP)\DR+
MM"2``0``20'&30'R28G^3`'228GZ2<'N&TG!X@5-"?)!`=),B>I)P>T"2,'B
M'D6)TDP)ZD@)T4F)UD&)U4@A^4TAYDP)\4R+M"2(`0``20'&30'P38G63`'!
M38G02<'N&TG!X`5-"?!)B=9!`<A(B?E(P>\"2,'A'D6)P$@)^8G/20G.32'6
M2"'Z3`GR3(NT))`!``!)`<9-`?1-B<9,`>)-B<1)P>X;2<'D!4T)]$$!U$R)
MTDG!Z@)(P>(>18GD3`G22`G128G608G222'^3"'!3`GQ3(NT))@!``!)`<9-
M`?5-B>9,`>E-B>5)P>X;2<'E!4T)]4F)UD$!S4R)P4G!Z`)(P>$>18GM3`G!
M08G(20G.32'F3"'"3`GR3(NT)*`!``!)`<9,`?=-B>Y(`?I,B>])P>X;2,'G
M!4P)]P'73(GB2<'L`DC!XAZ)_TP)XD@)T4F)UD&)U$TAQDPAZ4P)\4R+M"2H
M`0``20'&30'R28G^3`'128GZ2<'N&TG!X@5-"?))B=9!`<I,B>E)P>T"2,'A
M'D6)TDP)Z4&)S4D)SDDA_DPAZDP)\DR+M"2P`0``20'&30'P38G63`'"38G0
M2<'N&TG!X`5-"?!!`=!(B?I(P>\"2,'B'D6)P$@)^D@)T4F)UHG732'N3"'1
M3`GQ3(NT)+@!``!)`<9-`?1-B<9,`>%-B<1)P>X;2<'D!4T)]$F)UD$!S$R)
MT4G!Z@)(P>$>18GD3`G120G.08G*32'&3"'23`GR3(NT),`!``!)`<9-`?5-
MB>9,`>I-B>5)P>X;2<'E!4T)]4$!U4R)PDG!Z`)(P>(>18GM3`G"2`G128G0
M08G732'03"'A3`G!3(N$),@!``!)`<!,`<=-B>A(`?E,B>])P>@;2,'G!4P)
MQT2-!#E,B>=)P>P"2(G12,'G'DP)YT&)_$@)^4PAZ4PAXD@)RDB+C"30`0``
M2`'!3`'138G"2`.$)-@!``!(`<I,B<%)P>H;2,'A!4P)T8G)2`'*3(GI2<'M
M`DC!X1Y!B=),">E("<])B<U!B<Y,(<=-(>5)"?U*C3PX3(G03(N\)/@!``!,
M`>]-B=5(P>`%2<'M&TP)Z(G`2`''3(G`2<'H`DC!X!Y!B?U)"<"XUL%BRDPQ
MP3'12(N4).`!``!(`<),`>)-B>Q(`<I,B>E)P>P;2,'A!4P)X4V)U(G)2<'D
M'D@!RDG!Z@))`<=-">)-B<2)T46)P$TQU$0QYTR+I"3H`0``20'$30'T28G.
M3`'G28G,2<'N&TG!Y`5-"?1-B>Y)P>T"2<'F'D6)Y$T)]4R+M"3P`0``3`'G
M08G\20'&30'P38G633'N1#'R38GF20'03(GB2<'N&TC!X@5,"?*)TDD!T$B)
MRDC!Z0)(P>(>18G&2`G*1(G138GJ23'23`'Y3(N\)``"``!$,==-B?)(`<],
MB?%(P>$%2<'J&TD!QTP)T8G)2`'/3(GA2<'L`DC!X1Y!B?I)"<Q$B>E)B=6)
MTDTQY4P!^4R+O"00`@``13'H38G520'(3(G12<'M&TD!QTC!X05,">F)R4D!
MR$R)\4G![@)(P>$>18G%3`GQ3(NT)`@"``!)`<9,`?)-B>9),<Y$,?=-B>Y(
M`==,B>I)P>X;2,'B!4P)\HG22`'73(G22<'J`DC!XAY!B?Y)"=)$B>))B<R)
MR4P!^DTQU$R+O"0@`@``13'@38GT20'03(GR2<'L&TD!QTC!X@5,">*)TDD!
MT$R)ZDG![0)(P>(>18G$3`GJ3(NL)!@"``!)`<5,`>E-B=5),=5$,>]-B>5(
M`<],B>%)P>T;2,'A!4P)Z8G)2`'/3(GQ2<'N`DC!X1Y!B?U)"<Y$B=%)B=*)
MTDTQ\DP!^4R+O"0P`@``13'038GJ20'(3(GI2<'J&TC!X05,"=&)R4D!R$R)
MX4C!X1Y)P>P"18G"20''3`GA3(ND)"@"``!)`<1,`>)-B?1),<Q$,>=-B=1(
M`==,B=))P>P;2,'B!4P)XHG22`'73(GJ2<'M`DC!XAY!B?Q)"=5$B?))B<Z)
MR4TQ[DP!^DR+O"1``@``13'P38GF20'03(GB2<'N&TC!X@5,"?*)TDD!T$R)
MTDG!Z@)(P>(>18G&3`G23(N4)#@"``!)`<),`=%-B>I),=)$,==-B?)(`<],
MB?%(P>$%2<'J&TD!QTP)T8G)2`'/3(GA2<'L`DC!X1Y!B?I)"<Q$B>E)B=6)
MTDTQY4P!^4R+O"1@`@``13'H38G520'(3(G12<'M&TC!X05,">F)R4D!R$R)
M\4G![@)(P>$>18G%3`GQ3(NT)$@"``!)`<9,`?)-B>9%B>1),<Y$,?=-B>Y(
M`==,B>I)P>X;2,'B!4P)\HG22`'73(G22<'J`DC!XAY!B?Y)"=)(BY0D4`(`
M`$@!PDD!U$B)RHG)3#'220''1#'"38GP20'43(GR2<'H&TC!X@5,"<*)TDD!
MU$R)ZDG![0)(P>(>18G@3`GJ3(NL)%@"``!)`<5,`>E-B=5),=5$,>]-B<5(
M`<],B<%)P>T;2,'A!4P)Z8G)2`'/3(GQ2<'N`DC!X1Y!B?U)"<Y$B=%)B=*)
MTDTQ\DP!^4R+O"1P`@``13'B38GL3`'138GJ2<'L&TG!X@5-">)-B<1%B=))
MP>0>3`'12<'H`DD!QTT)X$R+I"1H`@``08G*20'$2`.$)'@"``!,`>)-B?1,
M`<A-,<1$,>=-B=1(`==,B=))P>P;2,'B!4P)XDV)[$G![0))P>0>B=)-">5%
MB?1-B<9(`==-,>Z)^DT!_$6)P$0Q\4F)UDP!P$V)Z$P!X4F)U$G![AM)P>0%
M30GT1`'A38G42<'J`DG!Y!Z)R4T)XDTQT$0QQTF)R$@!^$B)STC!YP5)P>@;
M`>E!`=I,"<=$`VX@2(E."`'X3(E6&$B)!DB)T$C!Z@)(P>`>3(EN($@)T$0!
MV$B)1A!(BX0DB`(``&1(*P0E*````'422('$F`(``%M=05Q!74%>05_#_Q6U
MA0``#Q]$``!!5T%6055!5%532(/L"$B%_P^$L0```$B+!3J$``!,BS!-A?8/
MA)X````/M@=)B?])B?6$P'0$/#UU544QY$F+'DB%VW0Q36/D3(GU#Q]``$R)
MXDR)_DB)W_\5080``(7`=0="@#PC/71&2(M="$B#Q0A(A=MUV4B#Q`A(B=A;
M74%<05U!7D%?PP\?@`````!)B?Q!#[9$)`%)@\0!A,!T!#P]=>Y%*?SKE&8/
M'T0``$PI]4J-7",!2,']`T&);0#KM68/'T0``#';ZZL/'T``4TB)^TB![*``
M``!D2(L$)2@```!(B80DF````#'`2(GF_Q71A```A<!U-8M4)!B!X@#P``"!
M^@!```!T"('Z`*```'4=O@(```!(B=__%>V$``"%P`^4P`^VP.L%#Q\`,<!(
MBY0DF````&1(*Q0E*````'4)2('$H````%O#_Q5<A```9F8N#Q^$``````"0
M055!5%532('LJ````&1(BP0E*````$B)A"28````,<!(A?]T"(`_`$B)^W4K
M2(N$))@```!D2"L$)2@````/A3,!``!(@<2H````6UU!7$%=PV8/'T0``/\5
M6H,``$B)Q4B%P'3'2(GO_Q5QA```2(7`#X2X````@'@3+DR-8!-U#D&`?"0!
M`'3<9@\?1```@'@3+@^$M@```$B)W_\5_8(``$R)YTF)Q?\5\8(``$&-?`4"
M2&/__Q6+@@``38GA28G8O@$```!)B<5(B<<QP$C'PO____](C0T6)```_Q6T
M@P``3(GO2(GF_Q7`@@``3(GO@_C_=!"+1"08)0#P```]`$```'1H_Q52@@``
M3(GO_Q6!@@``2(GO_Q7`@P``2(7`#X5/____#Q^``````$B)[_\55X,``$B)
MW_\5)H(``.GF_O__9@\?A```````08!\)`$N#X4^____08!\)`(`#X3^_O__
MZ2W___\/'P#HB_[__^N7_Q73@@``#Q\`\P\>^E-(BQU,@@``2(T-S6,``$B-
M-6`C``!(BQ4_@@``9$B+._\5%8,``$R-!0(E``!D2(L[13')2(T5,!@``$R)
MP4B--4PC``!;_R4?@@``9F8N#Q^$```````/'T``055!5%532('L`!```$B#
M#"0`2(/L&+K_#P``9$B+!"4H````2(F$)`@0```QP$F)Y$B)^TB)]4B-/0TC
M``!,B>;_%?&!``"%P'X2B<&#Z`%(8]"`/!0`=4>%P'4[2(L%E($``$B)ZDB)
MWF1(BSC_%1R!``!(BX0D"!```&1(*P0E*````'5.2('$&!```%M=05Q!7<,/
M'P")P68/'T0``$QCZ;XO````3(GG3(GJ_Q5T@```2(7`=*1(BP4X@0``3(GI
M3(GB2(G>9$B+./\578$``.N?_Q65@0``#Q]$``!!5T%6055!5%5(B?532(G[
M2(/L*&1(BP0E*````$B)1"08,<`QP(`^/0^4P$@!Q4B)[_\5J(```$B-="04
M2(G?28G$,<#HEOO__TB%P`^$#0$``$F)Q4B)Q_\5@8```$PYX`^#V````$AC
M3"043(LUQ'\``$R-/,T`````#[8#A,`/A*$!```\/0^$F0$``$B)V&8/'T0`
M``^V4`%(@\`!A-)T!8#Z/77O2"G82)A-BRY*C7P@`O\5S'\``$T!_4F)10!(
MA<!T44F+!DB-2P%*BQ0X#[8#B`(\/70AA,!U#.L;9@\?1```/#UT$0^V`4B#
MP@%(@\$!B`*$P'7KQ@(],<!F#Q]$```/MDP%`(A,`@%(@\`!A,EU[TB+1"08
M9$@K!"4H````#X4/`0``2(/$*%M=05Q!74%>05_##Q]``#'`9@\?1```#[94
M!0!!B%0%`$B#P`&$TG7NZ[T/'T``3(LUT7X``$V+!DF#.``/A+L```!,B<!%
M,>UF#Q]$``!(@\`(1(GJ08/%`4B#.`!U[XUR`TAC]DC!Y@.+!;^```"%P'5;
M2(GW3(E$)`C'!:F````!````_Q7+?@``2(G'2(7`#X10____26/-2(MT)`A,
MC3S-`````$R)^O\5?G\``$F)!DB)QT&-10%$B6PD%$B82,<$QP````#I?/[_
M_TR)Q_\5[W\``$F)!DB)QTB%P`^$`?___TECS4R-/,T`````Z\,/'P`QP.E_
M_O__OA````!%,>WI7_____\53G\``&9F+@\?A```````#Q\`\P\>^DB+!;U^
M``!(BQ6^?@``2(T--V```$B--<H?``!D2(LX_R6&?P``9@\?1```\P\>^E53
M2('L`!```$B##"0`2('L`!```$B##"0`2(/L*&1(BP0E*````$B)A"08(```
M,<!(C9PD$!```$B)Y?\5\GT``$B-#:T?``"Z`1```$B)WT&)P$R-#9<?``"^
M`0```#'`_Q7L?@``N@`0``!(B>Y(B=__%5M^```QR87`>#"->`%(B<-(8___
M%79]``!(B<%(A<!T&4ACVTB)[DB)QTB)VO\5['X``,8$&`!(B<%(BX0D&"``
M`&1(*P0E*````'4-2('$*"```$B)R%M=P_\5-WX```\?@`````#S#Q[ZZ1?_
M__^09@\?1```\P\>^D%7059!54%455-(@>P`$```2(,,)`!(@>RX````9$B+
M!"4H````2(F$)*@0```QP$F)_$B)\TB-/=<>``!(C70D#.@E^/__OB\```!,
MB>=(B<7_%9Q]``!(A<!T/S'`3(GF2(T]M!X``.@>_/__3(G@2(N4)*@0``!D
M2"L4)2@````/A9,!``!(@<2X$```6UU!7$%=05Y!7\-FD$B)WTR-+80>``!,
MC36#'P``_Q59?0``3(GN2(G'_Q5M?0``2(G#2(7`=):`.P`/A.<````/'T``
M2(7M=!1(B=Y(B>__%3]]``"%P`^$J@```$B)W_\5=GP``$B-5`/_28G'2#G3
M<A7K)`\?@`````#&`@!(@^H!2#G3=`6`.B]T[TB)W_\51GP``$F)QTR)Y_\5
M.GP``$F-1`<!2#W_#P``#X<:____3(V\)*````!,B?%)B=@QP$B#[`BZ`!``
M`+X!````3(G_051,C0V_'0``_Q7Q?```3(G_2(UT)"#_%?M[``!:687`=1"+
M1"0H)0#P```]`(```'153(GN,?__%8I\``!(B<-(A<`/A*_^__^`.P`/A1W_
M__](A>UT!H!]`"YT&D&_`0```$B-'4,=``#I5?___P\?A```````@'T!`'2U
MZ]X/'X0``````+X!````3(G__Q6"?```A<!UF4R)_DB-/14=``#H?_K__TR)
M__\5]GL``.E6_O___Q7[>P``9F8N#Q^$``````#S#Q[Z4[XO````2(G[_Q6-
M>P``2(U0`4B%P$@/1=I(B=];_R6X>P``#Q^$``````#S#Q[Z55-(@>P`$```
M2(,,)`!(@^P89$B+!"4H````2(F$)`@0```QP$B%_P^$S@```(`_`$B)^P^$
MP@```/\5S'H``$B-4`%(@?H`$```#X?C````2(GE2(GO@_H(<@Z)T4B)WL'I
M`_-(I4B)\S')]L($=`F+"XD/N00```#VP@)T#D0/MP0+9D2)!`](@\$"@^(!
M=`</MA0+B!0/2(U$!?](.<5R%>L8#Q^``````,8``$B#Z`%(.>AT!8`X+W3O
MOB\```!(B>__%:%Z``!(A<!T+$@YQ7)72(GO_Q7.>@``2(N4)`@0``!D2"L4
M)2@```!U1TB!Q!@0``!;7<.02(N$)`@0``!D2"L$)2@```!U*4B!Q!@0``!(
MC3V5&P``6UW_)8=Z```/'X``````Q@``ZZ0/'P`QP.NF_Q5^>@``9F8N#Q^$
M```````/'P#S#Q[Z05=!5D%505152(TM;AL``%-(@^P89$B+!"4H````2(E$
M)`@QP$B-7"0$9I`QP$B)WDB)[^B#]/__2(7`=#Y(BP7/>```2&-4)`1(BP!(
MC0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(2(D02(72=?#KMF8/'T0`
M`$B-+00;``!F#Q^$```````QP$B)WDB)[^@C]/__2(7`=#Y(BP5O>```2&-4
M)`1(BP!(C0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(2(D02(72=?#K
MMF8/'T0``$B-+:P:``!F#Q^$```````QP$B)WDB)[^C#\___2(7`=#Y(BP4/
M>```2&-4)`1(BP!(C0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(2(D0
M2(72=?#KMF8/'T0``$B-+54:``!F#Q^$```````QP$B)WDB)[^AC\___2(7`
M=#Y(BP6O=P``2&-4)`1(BP!(C0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(
M@\`(2(D02(72=?#KMF8/'T0``$B-+?P9``!F#Q^$```````QP$B)WDB)[^@#
M\___2(7`=#Y(BP5/=P``2&-4)`1(BP!(C0302(M0"$B)$$B%TG3/#Q^`````
M`$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$R-):P9``!F#Q^$```````QP$B)
MWDR)Y^BC\O__2(7`=#Y(BP7O=@``2&-4)`1(BP!(C0302(M0"$B)$$B%TG3/
M#Q^``````$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$R-)0$9``!F#Q^$````
M```QP$B)WDR)Y^A#\O__2(7`=#Y(BP6/=@``2&-4)`1(BP!(C0302(M0"$B)
M$$B%TG3/#Q^``````$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$R--?@8``!F
M#Q^$```````QP$B)WDR)]^CC\?__2(7`=#Y(BP4O=@``2&-4)`1(BP!(C030
M2(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(2(D02(72=?#KMF8/'T0``$R-
M+:(8``!F#Q^$```````QP$B)WDR)[^B#\?__2(7`=#Y(BP7/=0``2&-4)`1(
MBP!(C0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(2(D02(72=?#KMF8/
M'T0``$R-/4P8``!F#Q^$```````QP$B)WDR)_^@C\?__2(7`=#Y(BP5O=0``
M2&-4)`1(BP!(C0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(2(D02(72
M=?#KMF8/'T0``$R-/8H7``!F#Q^$```````QP$B)WDR)_^C#\/__2(7`=#Y(
MBP4/=0``2&-4)`1(BP!(C0302(M0"$B)$$B%TG3/#Q^``````$B+4!!(@\`(
M2(D02(72=?#KMF8/'T0``$B)WDB-/9,7``#H<?#__TB%P'0-2(G&3(GO,<#H
M?_3__S'`2(G>2(T]@1<``.A.\/__2(7`=!%(B<9(C3U_%P``,<#H6/3__TB)
MWDB-/7D7```QP.@G\/__2(G&2(7`=#E,B><QP.@U]/__2(M$)`AD2"L$)2@`
M``!U1$B#Q!A(B>](C35C%P``,<!;74%<05U!7D%?Z0;T__\QP$B)WDB-/347
M``#HU>___TB%P'2[2(G&3(GW,<#HX_/__^NL_Q5K=0``9F8N#Q^$``````#S
M#Q[Z2(/L&$B-/:`6``!D2(L$)2@```!(B40D"#'`2(UT)`3HA^___TB%P'0R
M#[80@/HP#Y3`A-(/E,()T(/P`0^VP$B+5"0(9$@K%"4H````=1!(@\08PP\?
M@``````QP.O@_Q7V=```9@\?1```\P\>^E!82('LV````$B)="0H2(E4)#!(
MB4PD.$R)1"1`3(E,)$B$P'0W#RE$)%`/*4PD8`\I5"1P#RF<)(`````/*:0D
MD`````\IK"2@````#RFT)+`````/*;PDP````&1(BP0E*````$B)1"08,<!(
MC80DX````$B)^DB)X4B)1"0(2(U$)""^`0```$B)1"002(L%IG0``,<$)`@`
M``!(BP#'1"0$,````$B)Q_\5JW,``+__````_Q4(<P``05154TR-G"0``/__
M2('L`!```$B##"0`3#G<=>](@^P0N@(````Q]F1(BP0E*````$B)A"0(``$`
M,<!!B?Q(B>/_%2%T``!(C6C_2(GJ2,'Z/TC!ZC!(C405``^WP$@IT$@IQ0^(
MJ@```&8/'T0``#'22(GN1(GG_Q7J<P``2(/X_P^$MP```+H(``$`2(G>1(GG
M_Q5_<@``@_C_='](F$B#^`=V6DB-1`/X2#G8<E!(C14I=```ZQ`/'X``````
M2(/H`4@YV'(W2(L*2#D(=>](*=A(`>A(BY0D"``!`&1(*Q0E*````'5W2('$
M$``!`%M=05S#9BX/'X0``````$B![0```0!(@?T``/__#X5<____2,?`____
M_^NZ2(N$)`@``0!D2"L$)2@```!U,4B-/<04```QP&?H`?[__TB+A"0(``$`
M9$@K!"4H````=0](C3V5%```,<!GZ-_]____%<ER``!F#Q^$``````#S#Q[Z
M2(/L"/\5TG(``(G'_Q7:<0``2(7`=`-(BP!(@\0(PV9F+@\?A```````9I#S
M#Q[Z055!5$R-)5@4``!52(G]3(GG4TB#[!AD2(L$)2@```!(B40D"#'`2(UT
M)`3HJ^S__TB%P'0(@#@`2(G#=2Y(BT0D"&1(*P0E*`````^%A````$B)[DB#
MQ!A,B><QP%M=05Q!7>F3\/__#Q\`2(GO_Q5G<0``2(G?28G%_Q5;<0``28U\
M!0+_%?AP``!(@^P(28GHO@$```!32,?"_____TB)QTF)Q4R-#>T2``!(C0WL
M$P``,<#_%1AR``!86DB+1"0(9$@K!"4H````=0A,B>[I?_____\5L'$``/,/
M'OI!5TB-%9$3``!(C0TE$P``059F2`]NPDB-%303``!)B?Y!54B-/7H2``!!
M5%532('L.`$``&1(BP0E*````$B)A"0H`0``2(T%51(``$B-;"042,=$)%``
M````2,>$)(@`````````2(GN2,=$)#``````9D@/;LA(C04@$P``9@]LP69(
M#V[02(T%&!,```\I1"1`9D@/;L%F2`]NV$B-#083``!F#VS"2(T%`!,```\1
M1"1H9D@/;L)F2`]NX#'`9@]LPP\11"1X9D@/;L%F#VS$#RE$)"#H'^O__TB%
MP'0,@#@`2(G##X6^!```_Q78<```B<?_%>!O``!(A<`/A`<$``!,BR!-A>0/
MA/L#``!,B>?_%>)O``!(C7P``?\5?V\``$4/M@0D28G'2(G#183`=#=,C2U]
M$@``28/$`44/ML!,B?],B>E(Q\+_____O@$````QP$F#QP+_%9)P``!%#[8$
M)$6$P'703(UL)&!(C3W$$0``ZP]FD$F+?0A)@\4(2(7_=#M(B>XQP.AIZO__
M28G$2(7`=.&`.`!TW$B)Q^@DZ___A<!TT$R)Y_\5YV\``$F)QTB%P'4_#Q^`
M`````$R-9"1`38L\)$V%_W0J08`_``^$(`,``$R)_^CHZO__A<`/A1`!``!-
MBWPD"$F#Q`A-A?]UV&:03(G__Q7W;@``2(G?28G$_Q7K;@``18VD!`0$``!-
M8^1,B>?_%7]N``!338GX3(T-C!```$F)Q4B-!8T1``"^`0```$C'PO____]0
M2(T-<!$``$R)[S'`_Q6;;P``3(GOOL`!``#_%=UO``!?05B#^/]U$_\5'V\`
M`$2+"$&#^1$/A?@"``!(C80DD````$R)[TB)!"1(B<;_%2IO``")PX7`='Q(
MBP55;P``28L.38GH2(T5R!$``+X!````2(LX,<#_%:AM``!%,>1(BX0D*`$`
M`&1(*P0E*`````^%.P0``$B!Q#@!``!,B>!;74%<05U!7D%?PV8/'X0`````
M`$R)_TF#Q`C_%9-N``!)B<=(A<`/A+3^___IXO[__V:0BX0DJ````"4`\```
M/0!````/A6W___]$B[PDK````/\5BVX``$$YQP^%5O___XN$)*@````E_P$`
M`#W``0``#X4_____3(GG_Q4Z;0``2(GN2(T]&A```$F)Q#'`Z'[H__](A<`/
MA&4"``!)BSY(B<9GZ/GO__])B<=(A<`/A$T"``!,B?__%51M``!(B<*#^`-^
M/$B82(E4)`A(C34.$```28U\!_S_%>MM``"%P'4?2(M4)`B#^@0/A`$#``"-
M0OM(F$&`/`<O#X3Q`@``D$B)[DB-+04/```QP$B)[^C\Y___2(7`#X3S`0``
M#[8`A,`/A.@!```\,`^$X`$``$B)[TB--4(/```QP.COZ___2(TMW`\``$R-
M-<,/``#_%9-L``!52,?"_____TV)Z%!,C0UH#@``O@$```!,B>=(C0V-#P``
M,<#_%8EM``!86NM(#Q]$``#_%1IM``"#.!%U2_\53VP``(/#`4B#[`A-B>A5
M3(T-)0X``$R)\;X!````4TC'PO____],B>=0,<#_%4%M``!(@\0@OL`!``!,
MB>?_%7]M``"#^/]TJDR)[_\5(6P``$R)YDB-/<<-```QP.@XZ___3(GG9^@O
M^O__Z=7]__]F+@\?A```````13'_Z?C\__\/'X0``````$B+?"0@2(U<)"!(
MA?]T(I`QP$B)[NC6YO__2(7`=`6`.`!U-$B+>PA(@\,(2(7_==^_#0```$R-
M)4H.``#_%55K``!!N%,```!)B<=(B</IU?O__P\?0`!(B<?_%2]L``!)B<1(
MA<!TN^F7^___2(L%>VP``$F+#DV)Z$B-%:X.``"^`0```$B+.#'`_Q7.:@``
MZ2']__]F#Q^$``````!(B<=GZ&?Y__](BX0D*`$``&1(*P0E*`````^%30$`
M`$B!Q#@!``!(B=];74%<05U!7D%?_R6V:P``9@\?1```38L^Z:O]__\/'X0`
M`````#'V3(G_,<#_%5MK``!!B<:%P`^$"/[__XG'Z&'W__])B<=(A<`/B/7]
M__\QTDB-</9$B??_%:YK``!(C70D&KH&````1(GW_Q5+:@``2(/X!@^%S```
M`(%\)!H`0T%##X6[_?__9H%\)!Y(10^%KOW__S'228UWSD2)]_\59VL``$B+
M'"2Z*````$2)]TB)WO\5`FH``$B#^"@/A8,```!(C05M#0``O@$```!-B>A,
MB>?&A"2X`````$B-#20-``!,C0WX"P``2,?"_____U`QP%/_%1]K``!97NGN
M_?__38MV"$V%]@^$`_W__TR)]_\5"FH``$B#^`,/AO#\__])C7P&_$B--<D,
M``#_%:9J``"%P$T/1/[IT_S___\5C6H``$B+A"0H`0``9$@K!"4H````=>=(
MC3V=#```,<!GZ'OU__]F9BX/'X0``````/,/'OI!5$B-/88+``!54TB#[!!(
MBQW.:0``9$B+!"4H````2(E$)`@QP$R-9"0$3(GF9$B+*P^VA;````"(A;$`
M```QP.AFY/__2(7`=`F`.#$/A(@!```QP&?HD.___V1(BRM(B>^Y`P```+H!
M````2(TU2PP``/\5ZF@``$B)Q4B%P'0I2(M`$&1(BSM(C14P#```2(LP_Q7B
M:```2(M%$$B+,/9&#D`/A6$!``!D2(L[N0,```"Z`0```$B--08,``#_%9YH
M``!(B<5(A<!T+DR)YDB-/84*```QP.C-X___2(G"2(7`#X0!`0``2(M%$&1(
MBSM(BS#_%8!H``!D2(L[N@$```!(C36\"P``_Q7J9P``9$B+.TB-%<`+``!(
MB<;_%59H``!D2(L#3(GF2(T](`H``,:`L0`````QP.AJX___2(D%\VD``$B)
MPTB%P'1#OL`!``!(B=__%95I``"-4`&#^@%V"8/X$0^%JP```$B+1"0(9$@K
M!"4H````#X7%````2(/$$%M=05S##Q^``````$B+/:EI``!GZ`OW__](B064
M:0``2(G#2(7`=:%(BT0D"&1(*P0E*`````^%A0```$B-/;D+```QP/\5L6@`
M`&8/'X0``````(!X`0`/A'K^___I:?[__Y!(BP51:0``2(LP2(M%$$B+..CJ
MY?__Z?7^__\/'T0``&1(BSO_%3YH``#ID/[__TB+5"0(9$@K%"4H````=1Z)
MP4B+!1!I``!(B=I(C3UN"P``2(LP,<#_%3MH``#_%0UH``!F9BX/'X0`````
M`&:0\P\>^E532(G[2(T]/PD``$B#[!AD2(L$)2@```!(B40D"#'`2(UT)`3H
M(N+__TB%P'0<#[80@/HP#Y3`A-(/E,((T'4*2(7;=`6`.P!U(4B+1"0(9$@K
M!"4H````=79(@\086UW#9BX/'X0``````$B)WV?HQ^O__[XO````2(G%2(G'
M_Q4F9P``2(U0`4B%P$@/1>I(B>__%5)G``"Z!````$B--7$)``!(B<?_%3UF
M``"%P'682(M$)`AD2"L$)2@```!U#DB#Q!A(B=];7>G;XO___Q4E9P``#Q]$
M``#S#Q[Z2(/L"+^`````_Q4%9@``9@]O!35(``!(QT`P``````\1`&8/;P4R
M2```QT!X``````\10!!F#V\%+T@```\10"!(@\0(PV8/'T0``/,/'OI!5TAC
MPD%628G&055!5$F)_%5(C6\X4TB)\TB#[`A(BW<H2(M7,(T,QD@Y\4B)3RA(
M@](`2,'H'4@!T$B)1S!(8T=XA<`/A>D```!!@_X_#XXO`0``08/N0$6)]T'!
M[P9%C6\!2<'E!DD!W9#S#V\#3(GG2(/#0`\110#S#V]+T`\131#S#V]3X`\1
M52#S#V];\`\173#H$<K__TPYZW7,0<'G!D4I_DECQDB#^`AR7$F+50!)C4PD
M0$B#X?A)B50D.$F+5`7X2(E4!?A(*<U(`>A)*>U(@^#X2(/X"'(82(/@^#'2
M28MT%0!(B3012(/""$@YPG+N18ET)'A(@\0(6UU!7$%=05Y!7\.0J`1U?$B%
MP'3B00^V50!!B%0D.*@"=-1!#[=4!?YFB50%_NO'#Q]``$&_0````$B-?`4`
M2(G>02G'13GW10]/_DUC[TR)ZO\5_&4``$&+1"1X1`'X08E$)'B#^$!UDDR)
MYT4I_DP!Z^@TR?__Z<[^__\/'X``````28G=Z1O___]!BU4`08E4)#A!BU0%
M_(E4!?SI5/___V8/'X0``````/,/'OI!5DR-=CA!54%454B)_5-,BVXH2(GS
M3(MF,$R)Z$C!Z`.#X#^-4`'&1`8X@$ACRH/Z.`^.A`$``+Y`````3`'Q*=9T
M#S'`B<*#P`'&!!$`.?!R\TB)W^BAR/__9@_OP`\10SA)QT8P`````$$/$480
M00\11B!,B>)!#[;%3(GO3(GN2,'J&$C!X`A,B>%(P>\028G03(GJ2,'N&`^V
MUD`/MO]`#[;V2,'I$$@)T$R)X@^VR4C!X`A("?@/MOY!#[;02,'@"$@)\$$/
MMO1(P>`(2`GP2,'@"$@)^$B)WTC!X`A("<A(P>`(2`G02(E#<.@(R/__2(L#
M2(G?2,'H&(A%`$B+`TC!Z!"(10%(BP.(90)(BP.(10-(BT,(2,'H&(A%!$B+
M0PA(P>@0B$4%2(M#"(AE!DB+0PB(10=(BT,02,'H&(A%"$B+0Q!(P>@0B$4)
M2(M#$(AE"DB+0Q"(10M(BT,82,'H&(A%#$B+0QA(P>@0B$4-2(M#&(AE#DB+
M0QB(10](BT,@2,'H&(A%$$B+0R!(P>@0B$412(M#((AE$DB+0R"(11-;74%<
M05U!7O\EKV(```\?@`````"X.````$D!SC'_*="#^`AR1TF-3@B)PDG'!@``
M``!)QT06^`````!(@^'X22G.1`'P@^#X@_@(#X)\_O__@^#X,=*)UH/""$B)
M/#$YPG+SZ67^__\/'T``J`1U)(7`#X15_O__0<8&`*@"#X1)_O__,=)F08E4
M!O[I//[__P\?`$''!@````!!QT0&_`````#I)/[__V8N#Q^$``````"0\P\>
M^DB+#5UC``#_)8]A``````#S#Q[Z2(/L"$B#Q`C#````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````0`"`"5S+R5S`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!)
M;G1E<FYA;',Z.E!!4CHZ0D]/5``O<')O8R]S96QF+V5X90`O<')O8R\E:2\E
M<P`N`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`4$523#5,24(`4$523$Q)
M0@!015),-4]05`!015),24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$
M`%!!4E]#3$5!3@!005)?1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"
M54<`4$%27T=,3T)!3%]435!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%
M35``4$%27T=,3T)!3%]#3$5!3@`Q`&QS965K(&9A:6QE9`!R96%D(&9A:6QE
M9`!,1%],24)205)97U!!5$@`4UE35$5-`"]T;7``5$5-4$1)4@!435``55-%
M4@!54T523D%-10`E,#)X`"5S)7,E<R5S`'!A<BT`<&%R;``N<&%R`'-H;W)T
M(')E860`)7,E<V-A8VAE+25S)7,`)7,E<W1E;7`M)74E<P`E<R5S=&5M<"TE
M=2TE=25S`!@`<&5R;``P`%!!4CHZ4&%C:V5R.CI615)324].`#$N,#8S`"UE
M`"TM`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*````````56YA8FQE('1O(&-R96%T92!C86-H
M92!D:7)E8W1O<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A
M8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D
M8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2
M+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*
M)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S
M7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*
M;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE
M(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@
M)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@
M;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@
M)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H
M*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?
M34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D
M8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R
M;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%
M3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I
M<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H
M<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN
M9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL
M("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M
M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ
M5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I5
M3DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64[
M"G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@
M73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M
M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*
M9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P
M,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL
M(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG
M<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI
M.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F
M:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?
M96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?
M<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM0
M05)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S
M96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@
M>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO
M=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C
M(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA
M<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?
M1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD
M(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE
M(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P
M;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A
M9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B
M7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C
M:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H
M92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S
M("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,
M15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I
M.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?
M1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@
M("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B
M"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*
M<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA
M;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B
M+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX
M?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@
M*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B
M=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@
M*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI
M)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B
M=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@
M/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE
M?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M
M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L
M;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@
M7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF
M+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M
M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U
M<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE
M?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII
M9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W
M*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#
M86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD
M9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T
M=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I
M;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@
M/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N
M("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD
M(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E
M<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA
M;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA
M;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H
M)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII
M9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA
M;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^
M("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI
M<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*
M;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]
M("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II
M<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S
M86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@
M*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@
M)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P
M87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N
M7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=
M(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%
M55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E
M:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=6
M6S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF
M("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@
M*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G
M02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G
M3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L
M92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G
M86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS
M:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE
M('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D
M,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD[
M"FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG
M4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[
M(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I
M<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`
M24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@
M*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O
M<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE
M<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII
M9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY
M;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>
M7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I]
M.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D
M<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=
M.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I
M<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#
M.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/
M.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ
M1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI
M(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H
M+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H
M9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D
M;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O
M=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S
M92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7
M<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R
M:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]
M"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB
M*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T
M>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@
M/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A
M8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X]
M(")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M
M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@
M)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP
M<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L
M;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B
M.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S
M<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER
M92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U
M:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F
M:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F
M:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH
M*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@
M7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A
M9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@
M>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO
M=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@
M*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T
M;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X
M="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E
M>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H
M+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E
M9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I
M;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L
M<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D
M97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H
M)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@
M<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE
M(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"([
M"GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2![
M"F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP
M87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O
M9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`
M05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D
M97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN
M*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q
M=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R
M:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@
M0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I
M<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER
M92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R
M97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@
M871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U
M:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE
M=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![
M(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E
M<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@
M>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY
M("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2
M(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI
M('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD
M("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P
M=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E
M9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN
M,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S
M97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U53
M15).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP
M9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U
M<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER
M.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM
M>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS
M965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N
M9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L
M96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H
M+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C
M86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@
M*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[
M4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C
M86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-
M4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6
M>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U
M8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H
M<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II
M9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S
M>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R
M;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA
M;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T
M=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?
M<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!
M4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S
M+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D
M9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N
M86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@
M+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?
M<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!
M345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@
M>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@
M)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I
M"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H
M*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<
M+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E
M;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E
M='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF
M;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!
M344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D
M14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"
M04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[
M4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*
M:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP
M<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I
M9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF
M:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP
M('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP
M86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2
M.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A
M<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@
M<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE
M.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+
M7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/
M4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@
M)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH``$-!0TA%``!P97)L>'-I+F,`````
M````_____P````#_____``````$C16<`````B:O-[P````#^W+J8`````'94
M,A``````\.'2PP````````````````$;`SOL````'````&!___\<!@``@('_
M_P@!``"`@O__(`$``/"8__]P`0``P)G__[P!``!0FO__X`$``-";__\<`@``
M,)S__S@"```0G?__>`(``&"?___$`@``D)___]@"``!PH/__#`,``("@__\@
M`P``L*+__X0#``#@HO__H`,``#"D___<`P``0*G__R@$``"PJ?__1`0``'"J
M__]@!```X*O__Z`$```0K/__N`0``/"L__\`!0``0+3__[`%``"@MO__Y`4`
M`("W__](!@``T+?__V`&``"`N?__K`8``."[___P!@```````!0`````````
M`7I2``%X$`$;#`<(D`$``!0````<````<(#__R8`````1`<0`````$P````T
M````6('__VL6````0@X0CP))#AB.`T(.((T$0@XHC`5!#C"&!D$..(,'1P[0
M!0-"%@H..$$.,$$.*$(.($(.&$(.$$(."$$+````2````(0```!XE___S```
M``!"#A"/`D(.&(X#0@X@C01"#BB,!4$.,(8&00XX@P=$#D`";0H..$0.,$$.
M*$(.($(.&$(.$$(."$@+`"````#0````_)?__X0`````00X0@P)*#K`!`G$*
M#A!!#@A!"S@```#T````:)C__WT!````0@X0C0)"#AB,`T$.((8$00XH@P5'
M#M`!?@H.*$$.($$.&$(.$$(."$<+`!@````P`0``K)G__U$`````10X0@P("
M1@X(```\````3`$``/"9___;`````$(.$(T"0@X8C`-!#B"&!$$.*(,%1PZH
M($D.P"`">`H.*$$.($$.&$(.$$(."$0+2````(P!``"0FO__0@(```!"#A"/
M`D(.&(X#0@X@C01"#BB,!4$.,(8&1`XX@P='#F`#'0$*#CA!#C!!#BA"#B!"
M#AA"#A!"#@A%"Q````#8`0``E)S__RH`````````,````.P!``"PG/__V0``
M``!%#A"&`D$.&(,#1PZ8($P.F$!)#L!``JL*#AA$#A!!#@A!"Q`````@`@``
M7)W__PH`````````8````#0"``!8G?__)0(```!&#A"/`D(.&(X#0@X@C01"
M#BB,!4$.,(8&00XX@P='#K@@3`[P(0)R"@XX00XP00XH0@X@0@X80@X00@X(
M0PL"NP[X(4\.@")<#O@A00[P(0```!@```"8`@``))___R@`````10X0@P)=
M#@@````X````M`(``#B?__]"`0```$4.$(8"00X8@P-'#I@@20ZP(`+F"@X8
M00X000X(0@M:"@X82`X000X(30M(````\`(``$R@__\%!0```$8.$(\"0@X8
MC@-"#B"-!$(.*(P%00XPA@9(#CB#!T0.4`.F!`H..$T.,$$.*$(.($(.&$(.
M$$(."$4+&````#P#```0I?__:@````!(#B`"4`H."$@+`!@```!8`P``9*7_
M_\``````10X000X(1P[@`0`\````=`,```BF__]G`0```$(.$(P"00X8A@-!
M#B"#!$@,"Z"`!%$-!T0.L(`$`M`*#B!!#AA!#A!"#@A+"P``%````+0#```X
MI___(P````!(#A!:#@@`1````,P#``!0I___X`````!&#A"-`D(.&(P#2`X@
MA@1'#BB#!40.0`)""@XH1@X@00X80@X00@X(2`MD#DA)#E!D#DA!#D``K```
M`!0$``#HI___10<```!&#A"/`E`.&(X#40X@C01)#BB,!4$.,(8&00XX@P='
M#O`"`_$!#O@"80Z``V$.^`)"#O`"`GL*#CA$#C!!#BA"#B!"#AA"#A!"#@A*
M"P-&`0[X`DL.@`-?#O@"00[P`E\.^`)$#H`#4`Z(`TL.D`-,#O`"`QP!"@XX
M1`XP00XH0@X@0@X80@X00@X(3`L"X`[X`D,.@`-'#O@"00[P`@`P````Q`0`
M`(BN__]3`@```$8.$(P"2`X8A@-!#B"#!$0.,`-Y`0H.($$.&$$.$$(."$@+
M-````/@$``"TL/__VP````!%#A"&`D$.&(,#3@XP`D\*#AA!#A!!#@A+"P);
M"@X81`X000X(10LH````,`4``#QY__\@`@```$4.$(8"00X8@P-$#E`#=P$*
M#AA##A!!#@A!"Q0```!<!0``,+'__TH`````2`X0`D$."$@```!T!0``:+'_
M_Z<!````1@X0CP)%#AB.`T4.((T$0@XHC`5$#C"&!D4..(,'1PY``N(*#CA!
M#C!!#BA"#B!"#AA"#A!"#@A""P!`````P`4``,RR__]5`@```$8.$(X"1@X8
MC0-"#B",!$$.*(8%1`XP@P8#F0$*#BA!#B!"#AA"#A!"#@A-"P```````!``
M```$!@``Z+3__Q$`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````#`C````````X"(````````!`````````-4"
M`````````0````````#@`@```````!T`````````>P0````````,````````
M```@````````#0````````"T7````````!D`````````T+L````````;````
M``````@`````````&@````````#8NP```````!P`````````"`````````#U
M_O]O`````-`#````````!0````````#0#0````````8`````````&`4`````
M```*`````````)X$````````"P`````````8`````````!4`````````````
M```````'`````````+@3````````"``````````8!@````````D`````````
M&``````````>``````````@`````````^___;P`````!```(`````/[__V\`
M````*!,```````#___]O``````$`````````\/__;P````!N$@```````"0`
M````````T!D````````C`````````"``````````)0`````````(````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````."[````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````C`````````"E!!4BYP;0H``````````/AB
M````````1T-#.B`H1TY5*2`Q,RXR+C$@,C`R,S`X,#$``"YS:'-T<G1A8@`N
M:6YT97)P`"YN;W1E+F=N=2YP<F]P97)T>0`N;F]T92YG;G4N8G5I;&0M:60`
M+FYO=&4N04))+71A9P`N9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N
M=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L82YD>6X`+G)E;'(N9'EN`"YI
M;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+F5H7V9R86UE7VAD<@`N96A?9G)A
M;64`+FEN:71?87)R87D`+F9I;FE?87)R87D`+F1Y;F%M:6,`+F=O=``N9&%T
M80`N8G-S`"YC;VUM96YT````````````````````````````````````````
M```````````````````````````````````````````````+`````0````(`
M````````&`,````````8`P```````!P````````````````````!````````
M````````````$P````<````"`````````#@#````````.`,```````!0````
M````````````````"````````````````````"8````'`````@````````"(
M`P```````(@#````````)`````````````````````0`````````````````
M```Y````!P````(`````````K`,```````"L`P```````"``````````````
M```````$````````````````````1P```/;__V\"`````````-`#````````
MT`,```````!$`0````````8`````````"````````````````````%$````+
M`````@`````````8!0```````!@%````````N`@````````'`````0````@`
M````````&`````````!9`````P````(`````````T`T```````#0#0``````
M`)X$```````````````````!````````````````````80```/___V\"````
M`````&X2````````;A(```````"Z``````````8``````````@`````````"
M`````````&X```#^__]O`@`````````H$P```````"@3````````D```````
M```'`````0````@```````````````````!]````!`````(`````````N!,`
M``````"X$P```````!@&````````!@`````````(`````````!@`````````
MAP```!,````"`````````-`9````````T!D````````@````````````````
M````"``````````(`````````)$````!````!@``````````(``````````@
M````````&P````````````````````0```````````````````"7`````0``
M``8`````````("`````````@(````````)$\```````````````````0````
M````````````````G0````$````&`````````+1<````````M%P````````-
M````````````````````!````````````````````*,````!`````@``````
M````8`````````!@````````P$```````````````````!``````````````
M``````"K`````0````(`````````P*````````#`H````````.P`````````
M```````````$````````````````````N0````$````"`````````+"A````
M````L*$````````8!@``````````````````"````````````````````,,`
M```.`````P````````#0NP```````-"K````````"```````````````````
M``@`````````"`````````#/````#P````,`````````V+L```````#8JP``
M``````@````````````````````(``````````@`````````VP````8````#
M`````````."[````````X*L``````````@````````<`````````"```````
M```0`````````.0````!`````P````````#@O0```````."M````````(`(`
M``````````````````@`````````"`````````#I`````0````,`````````
M`,``````````L````````"@````````````````````(````````````````
M````[P````@````#`````````"C`````````*+`````````H````````````
M````````"````````````````````/0````!````,```````````````````
M`"BP````````&P````````````````````$``````````0`````````!````
M`P````````````````````````!#L````````/T````````````````````!
/````````````````````
