/***************************************************************************
 *                                                                         *
 *   LinuxSampler - modular, streaming capable sampler                     *
 *                                                                         *
 *   Copyright (C) 2003, 2004 by Benno Senoner and Christian Schoenebeck   *
 *   Copyright (C) 2005 - 2009 Christian Schoenebeck                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 ***************************************************************************/

#ifndef __LS_DEVICE_H__
#define __LS_DEVICE_H__

namespace LinuxSampler {

    class AudioOutputDeviceFactory;
    class MidiInputDeviceFactory;

    /**
     * Abstract base class for all kind of drivers in LinuxSampler.
     */
    class Device {
    public:
        /**
         * Determines whether the device is an autonomous instance of some
         * driver or not autonomous (that is bound to some other entity).
         * An autonomous device can be created and deleted separately.
         * A non autonomous device only exists in the context of some
         * entity, e.g. in the context of a host plugin
         * (VST, AU, LV2, DSSI, ...) instance, and thus such a device
         * cannot be created or deleted on its own.
         *
         * By default, this method returns @c true unless overridden by the
         * descendent.
         */
        virtual bool isAutonomousDevice();

        /**
         * Returns the numerical ID of this device instance. Every device
         * instance has a unique ID among all devices of its category, that
         * is e.g. every MIDI input device has a unique ID among all MIDI
         * input devices and every audio output device has a unique ID
         * among all audio output devices. The IDs are usually generated by
         * the respective factory class.
         */
        int deviceId() const;

        /**
         * Whether this is an autonomous device driver.
         *
         * @see isAutonomousDevice()
         */
        static bool isAutonomousDriver();

    protected:
        void setDeviceId(int id);

        //TODO: maybe it would be cleaner to pass the device id through the drivers' constructors instead of having a setDeviceId() method being available only for the factory classes
        friend class AudioOutputDeviceFactory;
        friend class MidiInputDeviceFactory;
    private:
        int iDeviceId; //TODO: there is not default initialization (e.g. with -1) yet, since there is no constructor yet, it would require all driver implementations to be updated with a constructor call
    };

} // namespace LinuxSampler

#endif // __LS_DEVICE_H__
