/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "ArgumentCoders.h"
#include "Connection.h"
#include "DataReference.h"
#include "MessageNames.h"
#include "WebTransportSession.h"
#include <optional>
#include <wtf/Forward.h>
#include <wtf/ThreadSafeRefCounted.h>


namespace Messages {
namespace NetworkTransportSession {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::NetworkTransportSession;
}

class SendDatagram {
public:
    using Arguments = std::tuple<IPC::DataReference>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_SendDatagram; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkTransportSession_SendDatagramReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    explicit SendDatagram(const IPC::DataReference& datagram)
        : m_arguments(datagram)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const IPC::DataReference&> m_arguments;
};

class CreateOutgoingUnidirectionalStream {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_CreateOutgoingUnidirectionalStream; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkTransportSession_CreateOutgoingUnidirectionalStreamReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::optional<WebKit::WebTransportStreamIdentifier>>;
    using Promise = WTF::NativePromise<std::optional<WebKit::WebTransportStreamIdentifier>, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class CreateBidirectionalStream {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_CreateBidirectionalStream; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkTransportSession_CreateBidirectionalStreamReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<std::optional<WebKit::WebTransportStreamIdentifier>>;
    using Promise = WTF::NativePromise<std::optional<WebKit::WebTransportStreamIdentifier>, IPC::Error>;
    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class DestroyOutgoingUnidirectionalStream {
public:
    using Arguments = std::tuple<WebKit::WebTransportStreamIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_DestroyOutgoingUnidirectionalStream; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit DestroyOutgoingUnidirectionalStream(const WebKit::WebTransportStreamIdentifier& identifier)
        : m_arguments(identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::WebTransportStreamIdentifier&> m_arguments;
};

class DestroyBidirectionalStream {
public:
    using Arguments = std::tuple<WebKit::WebTransportStreamIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_DestroyBidirectionalStream; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit DestroyBidirectionalStream(const WebKit::WebTransportStreamIdentifier& identifier)
        : m_arguments(identifier)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::WebTransportStreamIdentifier&> m_arguments;
};

class SendStreamSendBytes {
public:
    using Arguments = std::tuple<WebKit::WebTransportStreamIdentifier, IPC::DataReference, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_SendStreamSendBytes; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkTransportSession_SendStreamSendBytesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    SendStreamSendBytes(const WebKit::WebTransportStreamIdentifier& identifier, const IPC::DataReference& bytes, bool withFin)
        : m_arguments(identifier, bytes, withFin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::WebTransportStreamIdentifier&, const IPC::DataReference&, bool> m_arguments;
};

class StreamSendBytes {
public:
    using Arguments = std::tuple<WebKit::WebTransportStreamIdentifier, IPC::DataReference, bool>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_StreamSendBytes; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkTransportSession_StreamSendBytesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    StreamSendBytes(const WebKit::WebTransportStreamIdentifier& identifier, const IPC::DataReference& bytes, bool withFin)
        : m_arguments(identifier, bytes, withFin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebKit::WebTransportStreamIdentifier&, const IPC::DataReference&, bool> m_arguments;
};

class Terminate {
public:
    using Arguments = std::tuple<uint32_t, CString>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkTransportSession_Terminate; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    Terminate(uint32_t code, const CString& reason)
        : m_arguments(code, reason)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<uint32_t, const CString&> m_arguments;
};

} // namespace NetworkTransportSession
} // namespace Messages
