const char socket_lua[] =
"-- socket.lua (internal file)\n"
"\n"
"local TIMEOUT_INFINITY      = 500 * 365 * 86400\n"
"local LIMIT_INFINITY = 2147483647\n"
"\n"
"local ffi = require('ffi')\n"
"local boxerrno = require('errno')\n"
"local internal = require('socket.lib')\n"
"local fiber = require('fiber')\n"
"local fio = require('fio')\n"
"local log = require('log')\n"
"local buffer = require('buffer')\n"
"local cord_ibuf_take = buffer.internal.cord_ibuf_take\n"
"local cord_ibuf_put = buffer.internal.cord_ibuf_put\n"
"local cord_ibuf_drop = buffer.internal.cord_ibuf_drop\n"
"\n"
"local format = string.format\n"
"\n"
"ffi.cdef[[\n"
"    struct gc_socket {\n"
"        const int fd;\n"
"    };\n"
"    typedef uint32_t socklen_t;\n"
"    typedef ptrdiff_t ssize_t;\n"
"\n"
"    int connect(int sockfd, const struct sockaddr *addr,\n"
"                socklen_t addrlen);\n"
"    int bind(int sockfd, const struct sockaddr *addr,\n"
"                socklen_t addrlen);\n"
"\n"
"    ssize_t write(int fd, const char *octets, size_t len);\n"
"    ssize_t read(int fd, void *buf, size_t count);\n"
"    int listen(int fd, int backlog);\n"
"    int socket(int domain, int type, int protocol);\n"
"    int coio_close(int s);\n"
"    int shutdown(int s, int how);\n"
"    ssize_t send(int sockfd, const void *buf, size_t len, int flags);\n"
"    ssize_t recv(int s, void *buf, size_t len, int flags);\n"
"    int accept(int s, void *addr, void *addrlen);\n"
"    ssize_t sendto(int sockfd, const void *buf, size_t len, int flags,\n"
"                   const struct sockaddr *dest_addr, socklen_t addrlen);\n"
"\n"
"    int\n"
"    lbox_socket_local_resolve(const char *host, const char *port,\n"
"                         struct sockaddr *addr, socklen_t *socklen);\n"
"    int lbox_socket_nonblock(int fd, int mode);\n"
"\n"
"    int setsockopt(int s, int level, int iname, const void *opt, size_t optlen);\n"
"    int getsockopt(int s, int level, int iname, void *ptr, size_t *optlen);\n"
"\n"
"    typedef struct { int active; int timeout; } linger_t;\n"
"\n"
"    struct protoent {\n"
"        char  *p_name;       /* official protocol name */\n"
"        char **p_aliases;    /* alias list */\n"
"        int    p_proto;      /* protocol number */\n"
"    };\n"
"    struct protoent *getprotobyname(const char *name);\n"
"\n"
"    void *memmem(const void *haystack, size_t haystacklen,\n"
"        const void *needle, size_t needlelen);\n"
"]]\n"
"\n"
"local gc_socket_t = ffi.metatype(ffi.typeof('struct gc_socket'), {\n"
"    __gc = function (socket)\n"
"        if socket.fd < 0 then return end\n"
"        if ffi.C.coio_close(socket.fd) ~= 0 then\n"
"            log.error(\"socket: failed to close fd=%d on gc: %s\", socket.fd,\n"
"                boxerrno.strerror())\n"
"        end\n"
"    end\n"
"})\n"
"\n"
"local socket_mt\n"
"\n"
"local function socket_is_closed(socket)\n"
"    -- The type of 'socket' is not checked because the function\n"
"    -- is internal and the type must be checked outside if needed.\n"
"    return socket._gc_socket.fd < 0\n"
"end\n"
"\n"
"local function check_socket(socket)\n"
"    local gc_socket = type(socket) == 'table' and socket._gc_socket\n"
"    if ffi.istype(gc_socket_t, gc_socket) then\n"
"        if not socket_is_closed(socket) then\n"
"            return gc_socket.fd\n"
"        else\n"
"            error(\"attempt to use closed socket\")\n"
"        end\n"
"    else\n"
"        local msg = \"Usage: socket:method()\"\n"
"        if socket ~= nil then msg = msg .. \", called with non-socket\" end\n"
"        error(msg)\n"
"    end\n"
"end\n"
"\n"
"local function make_socket(fd, itype)\n"
"    assert(itype ~= nil)\n"
"    local socket = {\n"
"        _gc_socket = ffi.new(gc_socket_t, { fd = fd }),\n"
"        itype = itype,\n"
"    }\n"
"    return setmetatable(socket, socket_mt)\n"
"end\n"
"\n"
"local gc_socket_sentinel = ffi.new(gc_socket_t, { fd = -1 })\n"
"\n"
"local function socket_close(socket)\n"
"    local fd = check_socket(socket)\n"
"    socket._errno = nil\n"
"    local r = ffi.C.coio_close(fd)\n"
"    -- .fd is const to prevent tampering\n"
"    ffi.copy(socket._gc_socket, gc_socket_sentinel, ffi.sizeof(gc_socket_t))\n"
"    if r ~= 0 then\n"
"        socket._errno = boxerrno()\n"
"        return false\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"local soname_mt = {\n"
"    __tostring = function(si)\n"
"        if si.host == nil and si.port == nil then\n"
"            return ''\n"
"        end\n"
"        if si.host == nil then\n"
"            return format('%s:%s', '0', tostring(si.port))\n"
"        end\n"
"\n"
"        if si.port == nil then\n"
"            return format('%s:%', tostring(si.host), 0)\n"
"        end\n"
"        return format('%s:%s', tostring(si.host), tostring(si.port))\n"
"    end\n"
"}\n"
"\n"
"local function socket_name(self)\n"
"    local fd = check_socket(self)\n"
"    local aka = internal.name(fd)\n"
"    if aka == nil then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"    self._errno = nil\n"
"    setmetatable(aka, soname_mt)\n"
"    return aka\n"
"end\n"
"\n"
"local function socket_peer(self)\n"
"    local fd = check_socket(self)\n"
"    local peer = internal.peer(fd)\n"
"    if peer == nil then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"    self._errno = nil\n"
"    setmetatable(peer, soname_mt)\n"
"    return peer\n"
"end\n"
"\n"
"local function socket_fd(self)\n"
"    return check_socket(self)\n"
"end\n"
"\n"
"local function get_ivalue(table, key)\n"
"    if type(key) == 'number' then\n"
"        return key\n"
"    end\n"
"    return table[key]\n"
"end\n"
"\n"
"local function get_iflags(table, flags)\n"
"    if flags == nil then\n"
"        return 0\n"
"    end\n"
"    local res = 0\n"
"    if type(flags) ~= 'table' then\n"
"        flags = { flags }\n"
"    end\n"
"    for _, f in pairs(flags) do\n"
"        if table[f] == nil then\n"
"            return nil\n"
"        end\n"
"        res = bit.bor(res, table[f])\n"
"    end\n"
"    return res\n"
"end\n"
"\n"
"local function getprotobyname(name)\n"
"    if type(name) == 'number' then\n"
"        return name\n"
"    elseif type(name) ~= 'string' then\n"
"        boxerrno(boxerrno.EINVAL)\n"
"        return nil\n"
"    end\n"
"    local num = internal.protocols[name]\n"
"    if num ~= nil then\n"
"        return num\n"
"    end\n"
"    local p = ffi.C.getprotobyname(name)\n"
"    if p == nil then\n"
"        boxerrno(boxerrno.EPROTOTYPE)\n"
"        return nil\n"
"    end\n"
"    num = p.p_proto\n"
"    -- update cache\n"
"    internal.protocols[name] = num\n"
"    return num\n"
"end\n"
"\n"
"local function socket_errno(self)\n"
"    check_socket(self)\n"
"    if self['_errno'] == nil then\n"
"        return 0\n"
"    else\n"
"        return self['_errno']\n"
"    end\n"
"end\n"
"\n"
"local function socket_error(self)\n"
"    check_socket(self)\n"
"    if self['_errno'] == nil then\n"
"        return nil\n"
"    else\n"
"        return boxerrno.strerror(self._errno)\n"
"    end\n"
"end\n"
"\n"
"-- addrbuf is equivalent to struct sockaddr_storage\n"
"local addrbuf = ffi.new('char[128]') -- enough to fit any address\n"
"local addr = ffi.cast('struct sockaddr *', addrbuf)\n"
"local addr_len = ffi.new('socklen_t[1]')\n"
"local function socket_sysconnect(self, host, port)\n"
"    local fd = check_socket(self)\n"
"    self._errno = nil\n"
"\n"
"    host = tostring(host)\n"
"    port = tostring(port)\n"
"\n"
"    addr_len[0] = ffi.sizeof(addrbuf)\n"
"    local res = ffi.C.lbox_socket_local_resolve(host, port, addr, addr_len)\n"
"    if res == 0 then\n"
"        res = ffi.C.connect(fd, addr, addr_len[0]);\n"
"        if res == 0 then\n"
"            return true\n"
"        end\n"
"    end\n"
"    self._errno = boxerrno()\n"
"    return false\n"
"end\n"
"\n"
"local function syswrite(self, charptr, size)\n"
"    local fd = check_socket(self)\n"
"    self._errno = nil\n"
"    local done = ffi.C.write(fd, charptr, size)\n"
"    if done < 0 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"\n"
"    return tonumber(done)\n"
"end\n"
"\n"
"local function socket_syswrite(self, arg1, arg2)\n"
"    -- TODO: ffi.istype('char *', arg1) doesn't work for ffi.new('char[256]')\n"
"    if type(arg1) == 'cdata' and arg2 ~= nil then\n"
"        return syswrite(self, arg1, arg2)\n"
"    elseif type(arg1) == 'string' then\n"
"        return syswrite(self, arg1, #arg1)\n"
"    else\n"
"        error('Usage: socket:syswrite(data) or socket:syswrite(const char *, size)')\n"
"    end\n"
"end\n"
"\n"
"local function sysread(self, charptr, size)\n"
"    local fd = check_socket(self)\n"
"\n"
"    self._errno = nil\n"
"    local res = ffi.C.read(fd, charptr, size)\n"
"    if res < 0 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"\n"
"    return tonumber(res)\n"
"end\n"
"\n"
"local function socket_sysread(self, arg1, arg2)\n"
"    -- TODO: ffi.istype('char *', arg1) doesn't work for ffi.new('char[256]')\n"
"    if type(arg1) == 'cdata' and arg2 ~= nil then\n"
"        return sysread(self, arg1, arg2)\n"
"    end\n"
"\n"
"    local size = arg1 or buffer.READAHEAD\n"
"    if size < 0 then\n"
"        error('socket:sysread(): size can not be negative')\n"
"    end\n"
"\n"
"    local buf = cord_ibuf_take()\n"
"    local p = buf:alloc(size)\n"
"\n"
"    local res = sysread(self, p, size)\n"
"    if res then\n"
"        res = ffi.string(p, res)\n"
"    end\n"
"    cord_ibuf_drop(buf)\n"
"    return res\n"
"end\n"
"\n"
"local function socket_nonblock(self, nb)\n"
"    local fd = check_socket(self)\n"
"    self._errno = nil\n"
"\n"
"    local res\n"
"\n"
"    if nb == nil then\n"
"        res = ffi.C.lbox_socket_nonblock(fd, 0x80)\n"
"    elseif nb then\n"
"        res = ffi.C.lbox_socket_nonblock(fd, 1)\n"
"    else\n"
"        res = ffi.C.lbox_socket_nonblock(fd, 0)\n"
"    end\n"
"\n"
"    if res < 0 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"\n"
"    if res == 1 then\n"
"        return true\n"
"    else\n"
"        return false\n"
"    end\n"
"end\n"
"\n"
"local function do_wait(self, what, timeout)\n"
"    local fd = check_socket(self)\n"
"\n"
"    self._errno = nil\n"
"    timeout = timeout or TIMEOUT_INFINITY\n"
"\n"
"    repeat\n"
"        local started = fiber.clock()\n"
"        local events = internal.iowait(fd, what, timeout)\n"
"        fiber.testcancel()\n"
"        if events ~= 0 and events ~= '' then\n"
"            return events\n"
"        end\n"
"        timeout = timeout - (fiber.clock() - started)\n"
"    until timeout < 0\n"
"    self._errno = boxerrno.ETIMEDOUT\n"
"    return 0\n"
"end\n"
"\n"
"local function socket_readable(self, timeout)\n"
"    return do_wait(self, 1, timeout) ~= 0\n"
"end\n"
"\n"
"local function socket_writable(self, timeout)\n"
"    return do_wait(self, 2, timeout) ~= 0\n"
"end\n"
"\n"
"local function socket_wait(self, timeout)\n"
"    return do_wait(self, 'RW', timeout)\n"
"end\n"
"\n"
"local function socket_listen(self, backlog)\n"
"    local fd = check_socket(self)\n"
"    self._errno = nil\n"
"    if backlog == nil then\n"
"        backlog = 256\n"
"    end\n"
"    local res = ffi.C.listen(fd, backlog)\n"
"    if res < 0 then\n"
"        self._errno = boxerrno()\n"
"        return false\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"local function socket_bind(self, host, port)\n"
"    local fd = check_socket(self)\n"
"    self._errno = nil\n"
"\n"
"    host = tostring(host)\n"
"    port = tostring(port)\n"
"\n"
"    addr_len[0] = ffi.sizeof(addrbuf)\n"
"    local res = ffi.C.lbox_socket_local_resolve(host, port, addr, addr_len)\n"
"    if res == 0 then\n"
"        res = ffi.C.bind(fd, addr, addr_len[0]);\n"
"    end\n"
"    if res == 0 then\n"
"        return true\n"
"    end\n"
"\n"
"    self._errno = boxerrno()\n"
"    return false\n"
"end\n"
"\n"
"local function socket_shutdown(self, how)\n"
"    local fd = check_socket(self)\n"
"    local hvariants = {\n"
"        ['R']           = 0,\n"
"        ['READ']        = 0,\n"
"        ['receive']     = 0,\n"
"        ['W']           = 1,\n"
"        ['WRITE']       = 1,\n"
"        ['send']        = 1,\n"
"        ['RW']          = 2,\n"
"        ['READ_WRITE']  = 2,\n"
"        [\"both\"]        = 2,\n"
"\n"
"        [0]             = 0,\n"
"        [1]             = 1,\n"
"        [2]             = 2\n"
"    }\n"
"    local ihow = hvariants[how]\n"
"\n"
"    if ihow == nil then\n"
"        ihow = 2\n"
"    end\n"
"    self._errno = nil\n"
"    if ffi.C.shutdown(fd, ihow) < 0 then\n"
"        self._errno = boxerrno()\n"
"        return false\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"local function getsol(level)\n"
"    if type(level) == 'number' then\n"
"        return level\n"
"    elseif type(level) ~= 'string' then\n"
"        boxerrno(boxerrno.EINVAL)\n"
"        return nil\n"
"    elseif level == 'SOL_SOCKET' or level == 'socket' then\n"
"        return internal.SOL_SOCKET\n"
"    end\n"
"    level = (level:match('IPPROTO_([A-Z]*)') or\n"
"             level:match('SOL_([A-Z]*)') or\n"
"             level):lower()\n"
"    level = getprotobyname(level)\n"
"    if level == nil then\n"
"        return nil\n"
"    end\n"
"    return level\n"
"end\n"
"\n"
"local function socket_setsockopt(self, level, name, value)\n"
"    local fd = check_socket(self)\n"
"\n"
"    level = getsol(level)\n"
"    if level == nil then\n"
"        self._errno = boxerrno()\n"
"        return false\n"
"    end\n"
"\n"
"    local info = get_ivalue(internal.SO_OPT[level] or {}, name)\n"
"    if info == nil then\n"
"        error(format(\"Unknown socket option name: %s\", tostring(name)))\n"
"    end\n"
"\n"
"    if not info.rw then\n"
"        error(format(\"Socket option %s is read only\", name))\n"
"    end\n"
"\n"
"    self._errno = nil\n"
"\n"
"    if type(value) == 'boolean' then\n"
"        if value then\n"
"            value = 1\n"
"        else\n"
"            value = 0\n"
"        end\n"
"    end\n"
"\n"
"    if info.type == 1 then\n"
"        local ai = ffi.new('int[1]', value)\n"
"        local res = ffi.C.setsockopt(fd,\n"
"            level, info.iname, ai, ffi.sizeof('int'))\n"
"\n"
"        if res < 0 then\n"
"            self._errno = boxerrno()\n"
"            return false\n"
"        end\n"
"        return true\n"
"    end\n"
"\n"
"    if info.type == 2 then\n"
"        local res = ffi.C.setsockopt(fd,\n"
"            level, info.iname, value, ffi.sizeof('size_t'))\n"
"        if res < 0 then\n"
"            self._errno = boxerrno()\n"
"            return false\n"
"        end\n"
"        return true\n"
"    end\n"
"\n"
"    if name == 'SO_LINGER' then\n"
"        error(\"Use s:linger(active[, timeout])\")\n"
"    end\n"
"    error(format(\"Unsupported socket option: %s\", name))\n"
"end\n"
"\n"
"local function socket_getsockopt(self, level, name)\n"
"    local fd = check_socket(self)\n"
"\n"
"    level = getsol(level)\n"
"    if level == nil then\n"
"        self._errno = boxerrno()\n"
"        return false\n"
"    end\n"
"\n"
"    local info = get_ivalue(internal.SO_OPT[level] or {}, name)\n"
"    if info == nil then\n"
"        error(format(\"Unknown socket option name: %s\", tostring(name)))\n"
"    end\n"
"\n"
"    self._errno = nil\n"
"\n"
"    if info.type == 1 then\n"
"        local value = ffi.new('int[1]')\n"
"        local len = ffi.new('size_t[1]', ffi.sizeof('int'))\n"
"        local res = ffi.C.getsockopt(fd, level, info.iname, value, len)\n"
"\n"
"        if res < 0 then\n"
"            self._errno = boxerrno()\n"
"            return nil\n"
"        end\n"
"\n"
"        if len[0] ~= 4 then\n"
"            error(format(\"Internal error: unexpected optlen: %d\", len[0]))\n"
"        end\n"
"        return tonumber(value[0])\n"
"    end\n"
"\n"
"    if info.type == 2 then\n"
"        local ibuf = cord_ibuf_take()\n"
"        local value = ibuf:alloc(256)\n"
"        local len = ffi.new('size_t[1]', 256)\n"
"        local res = ffi.C.getsockopt(fd, level, info.iname, value, len)\n"
"        if res < 0 then\n"
"            self._errno = boxerrno()\n"
"            cord_ibuf_put(ibuf)\n"
"            return nil\n"
"        end\n"
"        value = ffi.string(value, tonumber(len[0]))\n"
"        cord_ibuf_put(ibuf)\n"
"        return value\n"
"    end\n"
"\n"
"    if name == 'SO_LINGER' then\n"
"        error(\"Use s:linger()\")\n"
"    end\n"
"    error(format(\"Unsupported socket option: %s\", name))\n"
"end\n"
"\n"
"local function socket_linger(self, active, timeout)\n"
"    local fd = check_socket(self)\n"
"\n"
"    local level = internal.SOL_SOCKET\n"
"    local info = internal.SO_OPT[level].SO_LINGER\n"
"    self._errno = nil\n"
"    if active == nil then\n"
"        local value = ffi.new('linger_t[1]')\n"
"        local len = ffi.new('size_t[1]', ffi.sizeof('linger_t'))\n"
"        local res = ffi.C.getsockopt(fd, level, info.iname, value, len)\n"
"        if res < 0 then\n"
"            self._errno = boxerrno()\n"
"            return nil\n"
"        end\n"
"        if value[0].active ~= 0 then\n"
"            active = true\n"
"        else\n"
"            active = false\n"
"        end\n"
"        return active, value[0].timeout\n"
"    end\n"
"\n"
"    if timeout == nil then\n"
"        timeout = 0\n"
"    end\n"
"\n"
"    local iactive\n"
"    if active then\n"
"        iactive = 1\n"
"    else\n"
"        iactive = 0\n"
"    end\n"
"\n"
"    local value = ffi.new(\"linger_t[1]\",\n"
"        { { active = iactive, timeout = timeout } })\n"
"    local len = 2 * ffi.sizeof('int')\n"
"    local res = ffi.C.setsockopt(fd, level, info.iname, value, len)\n"
"    if res < 0 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"\n"
"    return active, timeout\n"
"end\n"
"\n"
"local function socket_accept(self)\n"
"    local server_fd = check_socket(self)\n"
"    self._errno = nil\n"
"\n"
"    local client_fd, from = internal.accept(server_fd)\n"
"    if client_fd == nil then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"    local client = make_socket(client_fd, self.itype)\n"
"    if not client:nonblock(true) then\n"
"        client:close()\n"
"        return\n"
"    end\n"
"    return client, from\n"
"end\n"
"\n"
"local errno_is_transient = {\n"
"    [boxerrno.EAGAIN] = true;\n"
"    [boxerrno.EWOULDBLOCK] = true;\n"
"    [boxerrno.EINTR] = true;\n"
"}\n"
"\n"
"local errno_is_fatal = {\n"
"    [boxerrno.EBADF] = true;\n"
"    [boxerrno.EINVAL] = true;\n"
"    [boxerrno.EOPNOTSUPP] = true;\n"
"    [boxerrno.ENOTSOCK] = true;\n"
"}\n"
"\n"
"local function check_limit(self, limit)\n"
"    if self.rbuf:size() >= limit then\n"
"        return limit\n"
"    end\n"
"    return nil\n"
"end\n"
"\n"
"local function check_infinity(self, limit)\n"
"    if limit == 0 then\n"
"        return 0\n"
"    end\n"
"\n"
"    local rbuf = self.rbuf\n"
"    if rbuf:size() == 0 then\n"
"        return nil\n"
"    end\n"
"\n"
"    return rbuf:size()\n"
"end\n"
"\n"
"local function check_delimiter(self, limit, eols)\n"
"    if limit == 0 then\n"
"        return 0\n"
"    end\n"
"    local rbuf = self.rbuf\n"
"    if rbuf:size() == 0 then\n"
"        return nil\n"
"    end\n"
"\n"
"    local shortest\n"
"    for _, eol in ipairs(eols) do\n"
"        local data = ffi.C.memmem(rbuf.rpos, rbuf:size(), eol, #eol)\n"
"        if data ~= nil then\n"
"            local len = ffi.cast('char *', data) - rbuf.rpos + #eol\n"
"            if shortest == nil or shortest > len then\n"
"                shortest = len\n"
"            end\n"
"        end\n"
"    end\n"
"    if shortest ~= nil and shortest <= limit then\n"
"        return shortest\n"
"    elseif limit <= rbuf:size() then\n"
"        return limit\n"
"    end\n"
"    return nil\n"
"end\n"
"\n"
"local function read(self, limit, timeout, check, ...)\n"
"    if limit < 0 then\n"
"        error('socket:read(): limit can not be negative')\n"
"    end\n"
"\n"
"    limit = math.min(limit, LIMIT_INFINITY)\n"
"    local rbuf = self.rbuf\n"
"    if rbuf == nil then\n"
"        rbuf = buffer.ibuf()\n"
"        self.rbuf = rbuf\n"
"    end\n"
"\n"
"    local len = check(self, limit, ...)\n"
"    if len ~= nil then\n"
"        self._errno = nil\n"
"        local data = ffi.string(rbuf.rpos, len)\n"
"        rbuf.rpos = rbuf.rpos + len\n"
"        return data\n"
"    end\n"
"\n"
"    local deadline = fiber.clock() + timeout\n"
"    repeat\n"
"        assert(rbuf:size() < limit)\n"
"        local to_read = math.min(limit - rbuf:size(), buffer.READAHEAD)\n"
"        local data = rbuf:reserve(to_read)\n"
"        assert(rbuf:unused() >= to_read)\n"
"        local res = sysread(self, data, rbuf:unused())\n"
"        if res == 0 then -- eof\n"
"            self._errno = nil\n"
"            local len = rbuf:size()\n"
"            local data = ffi.string(rbuf.rpos, len)\n"
"            rbuf.rpos = rbuf.rpos + len\n"
"            return data\n"
"        elseif res ~= nil then\n"
"            rbuf.wpos = rbuf.wpos + res\n"
"            local len = check(self, limit, ...)\n"
"            if len ~= nil then\n"
"                self._errno = nil\n"
"                local data = ffi.string(rbuf.rpos, len)\n"
"                rbuf.rpos = rbuf.rpos + len\n"
"                return data\n"
"            end\n"
"        elseif not errno_is_transient[self._errno] then\n"
"            return nil\n"
"        end\n"
"    until not socket_readable(self, deadline - fiber.clock())\n"
"    return nil\n"
"end\n"
"\n"
"local function socket_read(self, opts, timeout)\n"
"    check_socket(self)\n"
"    timeout = timeout or TIMEOUT_INFINITY\n"
"    if type(opts) == 'number' then\n"
"        return read(self, opts, timeout, check_limit)\n"
"    elseif type(opts) == 'string' then\n"
"        return read(self, LIMIT_INFINITY, timeout, check_delimiter, { opts })\n"
"    elseif type(opts) == 'table' then\n"
"        local chunk = opts.chunk or opts.size or LIMIT_INFINITY\n"
"        local delimiter = opts.delimiter or opts.line\n"
"        if delimiter == nil then\n"
"            return read(self, chunk, timeout, check_limit)\n"
"        elseif type(delimiter) == 'string' then\n"
"            return read(self, chunk, timeout, check_delimiter, { delimiter })\n"
"        elseif type(delimiter) == 'table' then\n"
"            return read(self, chunk, timeout, check_delimiter, delimiter)\n"
"        end\n"
"    end\n"
"    error('Usage: s:read(delimiter|chunk|{delimiter = x, chunk = x}, timeout)')\n"
"end\n"
"\n"
"local function socket_write(self, octets, timeout)\n"
"    check_socket(self)\n"
"    if timeout == nil then\n"
"        timeout = TIMEOUT_INFINITY\n"
"    end\n"
"\n"
"    local s = ffi.cast('const char *', octets)\n"
"    local p = s\n"
"    local e = s + #octets\n"
"    if p == e then\n"
"        return 0\n"
"    end\n"
"\n"
"    local deadline = fiber.clock() + timeout\n"
"    repeat\n"
"        local written = syswrite(self, p, e - p)\n"
"        if written == 0 then\n"
"            return p - s -- eof\n"
"        elseif written ~= nil then\n"
"            p = p + written\n"
"            assert(p <= e)\n"
"            if p == e then\n"
"                return e - s\n"
"            end\n"
"        elseif not errno_is_transient[self._errno] then\n"
"            return nil\n"
"        end\n"
"\n"
"    until not socket_writable(self, deadline - fiber.clock())\n"
"    return nil\n"
"end\n"
"\n"
"local function socket_send(self, octets, flags)\n"
"    local fd = check_socket(self)\n"
"    local iflags = get_iflags(internal.SEND_FLAGS, flags)\n"
"\n"
"    self._errno = nil\n"
"    local res = ffi.C.send(fd, octets, string.len(octets), iflags)\n"
"    if res < 0 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"    return tonumber(res)\n"
"end\n"
"\n"
"-- Returns nil and sets EAGAIN when tries to determine the next\n"
"-- datagram length and there are no datagrams in the receive\n"
"-- queue.\n"
"local function get_recv_size(self, size)\n"
"    if size ~= nil then\n"
"        return size\n"
"    end\n"
"\n"
"    if self.itype ~= get_ivalue(internal.SO_TYPE, 'SOCK_DGRAM') then\n"
"        return 512\n"
"    end\n"
"\n"
"    -- Determine the next datagram length.\n"
"    local fd = check_socket(self)\n"
"    self._errno = nil\n"
"    if jit.os == 'OSX' then\n"
"        size = self:getsockopt('SOL_SOCKET', 'SO_NREAD')\n"
"        -- recv() with zero length buffer is always successful on\n"
"        -- Mac OS (at least for valid UDP sockets), so we need\n"
"        -- extra calls below to distinguish a zero length datagram\n"
"        -- from no datagram situation to set EAGAIN correctly.\n"
"        if size == 0 then\n"
"            -- No datagram or zero length datagram: distinguish\n"
"            -- them using message peek.\n"
"            local iflags = get_iflags(internal.SEND_FLAGS, {'MSG_PEEK'})\n"
"            assert(iflags ~= nil)\n"
"            size = tonumber(ffi.C.recv(fd, ffi.new('char[1]'), 1, iflags))\n"
"            -- Prevent race condition: proceed with the case when\n"
"            -- a datagram of length > 0 has been arrived after the\n"
"            -- getsockopt call above.\n"
"            if size > 0 then\n"
"                size = self:getsockopt('SOL_SOCKET', 'SO_NREAD')\n"
"                assert(size > 0)\n"
"            end\n"
"        end\n"
"    else\n"
"        local iflags = get_iflags(internal.SEND_FLAGS, {'MSG_TRUNC', 'MSG_PEEK'})\n"
"        assert(iflags ~= nil)\n"
"        size = tonumber(ffi.C.recv(fd, nil, 0, iflags))\n"
"    end\n"
"\n"
"    if size == -1 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"\n"
"    return size\n"
"end\n"
"\n"
"local function socket_recv(self, size, flags)\n"
"    local fd = check_socket(self)\n"
"    local iflags = get_iflags(internal.SEND_FLAGS, flags)\n"
"    if iflags == nil then\n"
"        self._errno = boxerrno.EINVAL\n"
"        return nil\n"
"    end\n"
"\n"
"    size = get_recv_size(self, size)\n"
"    if size == nil then\n"
"        return nil\n"
"    end\n"
"\n"
"    self._errno = nil\n"
"    local ibuf = cord_ibuf_take()\n"
"    local buf = ibuf:alloc(size)\n"
"    local res = ffi.C.recv(fd, buf, size, iflags)\n"
"\n"
"    if res == -1 then\n"
"        self._errno = boxerrno()\n"
"        cord_ibuf_put(ibuf)\n"
"        return nil\n"
"    end\n"
"    buf = ffi.string(buf, res)\n"
"    cord_ibuf_put(ibuf)\n"
"    return buf\n"
"end\n"
"\n"
"local function socket_recvfrom(self, size, flags)\n"
"    local fd = check_socket(self)\n"
"    local iflags = get_iflags(internal.SEND_FLAGS, flags)\n"
"    if iflags == nil then\n"
"        self._errno = boxerrno.EINVAL\n"
"        return nil\n"
"    end\n"
"\n"
"    size = get_recv_size(self, size)\n"
"    if size == nil then\n"
"        return nil\n"
"    end\n"
"\n"
"    self._errno = nil\n"
"    local res, from = internal.recvfrom(fd, size, iflags)\n"
"    if res == nil then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"    return res, from\n"
"end\n"
"\n"
"local function socket_sendto(self, host, port, octets, flags)\n"
"    local fd = check_socket(self)\n"
"    local iflags = get_iflags(internal.SEND_FLAGS, flags)\n"
"\n"
"    if iflags == nil then\n"
"        self._errno = boxerrno.EINVAL\n"
"        return nil\n"
"    end\n"
"\n"
"    self._errno = nil\n"
"    if octets == nil or octets == '' then\n"
"        return true\n"
"    end\n"
"\n"
"    host = tostring(host)\n"
"    port = tostring(port)\n"
"    octets = tostring(octets)\n"
"\n"
"    addr_len[0] = ffi.sizeof(addrbuf)\n"
"    local res = ffi.C.lbox_socket_local_resolve(host, port, addr, addr_len)\n"
"    if res == 0 then\n"
"        res = ffi.C.sendto(fd, octets, string.len(octets), iflags,\n"
"            addr, addr_len[0])\n"
"    end\n"
"    if res < 0 then\n"
"        self._errno = boxerrno()\n"
"        return nil\n"
"    end\n"
"    return tonumber(res)\n"
"end\n"
"\n"
"local function socket_new(domain, stype, proto)\n"
"    local idomain = get_ivalue(internal.DOMAIN, domain)\n"
"    if idomain == nil then\n"
"        boxerrno(boxerrno.EINVAL)\n"
"        return nil\n"
"    end\n"
"\n"
"    local itype = get_ivalue(internal.SO_TYPE, stype)\n"
"    if itype == nil then\n"
"        boxerrno(boxerrno.EINVAL)\n"
"        return nil\n"
"    end\n"
"\n"
"    local iproto = getprotobyname(proto)\n"
"    if iproto == nil then\n"
"        return nil\n"
"    end\n"
"\n"
"    local fd = ffi.C.socket(idomain, itype, iproto)\n"
"    if fd >= 0 then\n"
"        local socket = make_socket(fd, itype)\n"
"        if not socket:nonblock(true) then\n"
"            socket:close()\n"
"        else\n"
"            return socket\n"
"        end\n"
"    end\n"
"end\n"
"\n"
"local function getaddrinfo(host, port, timeout, opts)\n"
"    if type(timeout) == 'table' and opts == nil then\n"
"        opts = timeout\n"
"        timeout = TIMEOUT_INFINITY\n"
"    elseif timeout == nil then\n"
"        timeout = TIMEOUT_INFINITY\n"
"    end\n"
"    if port == nil then\n"
"        port = 0\n"
"    end\n"
"    local ga_opts = {}\n"
"    if opts ~= nil then\n"
"        if opts.type ~= nil then\n"
"            local itype = get_ivalue(internal.SO_TYPE, opts.type)\n"
"            if itype == nil then\n"
"                boxerrno(boxerrno.EINVAL)\n"
"                return nil, boxerrno.strerror()\n"
"            end\n"
"            ga_opts.type = itype\n"
"        end\n"
"\n"
"        if opts.family ~= nil then\n"
"            local ifamily = get_ivalue(internal.DOMAIN, opts.family)\n"
"            if ifamily == nil then\n"
"                boxerrno(boxerrno.EINVAL)\n"
"                return nil, boxerrno.strerror()\n"
"            end\n"
"            ga_opts.family = ifamily\n"
"        end\n"
"\n"
"        if opts.protocol ~= nil then\n"
"            local p = getprotobyname(opts.protocol)\n"
"            if p == nil then\n"
"                return nil, boxerrno.strerror()\n"
"            end\n"
"            ga_opts.protocol = p\n"
"        end\n"
"\n"
"        if opts.flags ~= nil then\n"
"            ga_opts.flags =\n"
"                get_iflags(internal.AI_FLAGS, opts.flags)\n"
"            if ga_opts.flags == nil then\n"
"                boxerrno(boxerrno.EINVAL)\n"
"                return nil, boxerrno.strerror()\n"
"            end\n"
"        end\n"
"\n"
"    end\n"
"    return internal.getaddrinfo(host, port, timeout, ga_opts)\n"
"end\n"
"\n"
"-- tcp connector\n"
"local function socket_tcp_connect(s, address, port, timeout)\n"
"    local res = socket_sysconnect(s, address, port)\n"
"    if res then\n"
"        -- Even through the socket is nonblocking, if the server to which we\n"
"        -- are connecting is on the same host, the connect is normally\n"
"        -- established immediately when we call connect (Stevens UNP).\n"
"        return true\n"
"    end\n"
"    if s._errno ~= boxerrno.EINPROGRESS then\n"
"        return nil\n"
"    end\n"
"    -- Wait until the connection is established or ultimately fails.\n"
"    -- In either condition the socket becomes writable. To tell these\n"
"    -- conditions appart SO_ERROR must be consulted (man connect).\n"
"    if socket_writable(s, timeout) then\n"
"        s._errno = socket_getsockopt(s, 'SOL_SOCKET', 'SO_ERROR')\n"
"    end\n"
"    if s._errno ~= 0 then\n"
"        return nil\n"
"    end\n"
"    -- Connected\n"
"    return true\n"
"end\n"
"\n"
"local function tcp_connect(host, port, timeout)\n"
"    if host == 'unix/' then\n"
"        local s = socket_new('AF_UNIX', 'SOCK_STREAM', 0)\n"
"        if not s then\n"
"            -- Address family is not supported by the host\n"
"            return nil, boxerrno.strerror()\n"
"        end\n"
"        if not socket_tcp_connect(s, host, port, timeout) then\n"
"            local save_errno = s._errno\n"
"            s:close()\n"
"            boxerrno(save_errno)\n"
"            return nil, boxerrno.strerror()\n"
"        end\n"
"        boxerrno(0)\n"
"        return s\n"
"    end\n"
"    local timeout = timeout or TIMEOUT_INFINITY\n"
"    local stop = fiber.clock() + timeout\n"
"    local dns, err = getaddrinfo(host, port, timeout, {\n"
"        protocol = 'tcp',\n"
"        type = 'SOCK_STREAM',\n"
"        flags = 'AI_ADDRCONFIG',\n"
"    })\n"
"    if dns == nil then\n"
"        return nil, err\n"
"    end\n"
"    if #dns == 0 then\n"
"        boxerrno(boxerrno.EINVAL)\n"
"        return nil, boxerrno.strerror()\n"
"    end\n"
"    for _, remote in pairs(dns) do\n"
"        timeout = stop - fiber.clock()\n"
"        if timeout <= 0 then\n"
"            boxerrno(boxerrno.ETIMEDOUT)\n"
"            return nil, boxerrno.strerror()\n"
"        end\n"
"        local s = socket_new(remote.family, remote.type, remote.protocol)\n"
"        if s then\n"
"            if socket_tcp_connect(s, remote.host, remote.port, timeout) then\n"
"                boxerrno(0)\n"
"                return s\n"
"            end\n"
"            local save_errno = s:errno()\n"
"            s:close()\n"
"            boxerrno(save_errno)\n"
"        end\n"
"    end\n"
"    -- errno is set by socket_tcp_connect()\n"
"    return nil, boxerrno.strerror()\n"
"end\n"
"\n"
"local function tcp_server_handler(server, sc, from)\n"
"    fiber.name(format(\"%s/%s:%s\", server.name, from.host, from.port), {truncate = true})\n"
"    local status, message = pcall(server.handler, sc, from)\n"
"    sc:shutdown()\n"
"    sc:close()\n"
"    if not status then\n"
"        error(message)\n"
"    end\n"
"end\n"
"\n"
"local function tcp_server_loop(server, s, addr)\n"
"    fiber.name(format(\"%s/%s:%s\", server.name, addr.host, addr.port), {truncate = true})\n"
"    log.info(\"started\")\n"
"    while socket_readable(s) do\n"
"        if socket_is_closed(s) then\n"
"            break\n"
"        end\n"
"        local sc, from = socket_accept(s)\n"
"        if sc == nil then\n"
"            local errno = s._errno\n"
"            if not errno_is_transient[errno] then\n"
"                log.error('accept(%s) failed: %s', tostring(s),\n"
"                          socket_error(s))\n"
"            end\n"
"            if  errno_is_fatal[errno] then\n"
"                break\n"
"            end\n"
"        else\n"
"            fiber.create(tcp_server_handler, server, sc, from)\n"
"        end\n"
"    end\n"
"    -- Socket was closed\n"
"    log.info(\"stopped\")\n"
"end\n"
"\n"
"local function tcp_server_usage()\n"
"    error('Usage: socket.tcp_server(host, port, handler | opts)')\n"
"end\n"
"\n"
"local function tcp_server_do_bind(s, addr)\n"
"    if socket_bind(s, addr.host, addr.port) then\n"
"        if addr.family == 'AF_UNIX' then\n"
"            -- Make close() remove the unix socket file created\n"
"            -- by bind(). Note, this must be done before closing\n"
"            -- the socket fd so that no other tcp server can\n"
"            -- reuse the same path before we remove the file.\n"
"            s.close = function(self)\n"
"                fio.unlink(addr.port)\n"
"                return socket_close(self)\n"
"            end\n"
"        end\n"
"        return true\n"
"    end\n"
"    return false\n"
"end\n"
"\n"
"local function tcp_server_bind_addr(s, addr)\n"
"    if tcp_server_do_bind(s, addr) then\n"
"        return true\n"
"    end\n"
"\n"
"    if addr.family ~= 'AF_UNIX' then\n"
"        return false\n"
"    end\n"
"\n"
"    if boxerrno() ~= boxerrno.EADDRINUSE then\n"
"        return false\n"
"    end\n"
"\n"
"    local save_errno = boxerrno()\n"
"\n"
"    local sc = tcp_connect(addr.host, addr.port)\n"
"    if sc ~= nil then\n"
"        sc:close()\n"
"        boxerrno(save_errno)\n"
"        return false\n"
"    end\n"
"\n"
"    if boxerrno() ~= boxerrno.ECONNREFUSED then\n"
"        boxerrno(save_errno)\n"
"        return false\n"
"    end\n"
"\n"
"    log.info(\"tcp_server: remove dead UNIX socket: %s\", addr.port)\n"
"    if not fio.unlink(addr.port) then\n"
"        log.warn(\"tcp_server: %s\", boxerrno.strerror())\n"
"        boxerrno(save_errno)\n"
"        return false\n"
"    end\n"
"    return tcp_server_do_bind(s, addr)\n"
"end\n"
"\n"
"\n"
"local function tcp_server_bind(host, port, prepare, timeout)\n"
"    timeout = timeout and tonumber(timeout) or TIMEOUT_INFINITY\n"
"    local dns, err\n"
"    if host == 'unix/' then\n"
"        dns = {{host = host, port = port, family = 'AF_UNIX', protocol = 0,\n"
"            type = 'SOCK_STREAM' }}\n"
"    else\n"
"        dns, err = getaddrinfo(host, port, timeout, {\n"
"            protocol = 'tcp',\n"
"            type = 'SOCK_STREAM',\n"
"            flags = {'AI_ADDRCONFIG', 'AI_PASSIVE'},\n"
"        })\n"
"        if dns == nil then\n"
"            return nil, err\n"
"        end\n"
"    end\n"
"\n"
"    for _, addr in ipairs(dns) do\n"
"        local s = socket_new(addr.family, addr.type, addr.protocol)\n"
"        if s ~= nil then\n"
"            local backlog\n"
"            if prepare then\n"
"                backlog = prepare(s)\n"
"            else\n"
"                socket_setsockopt(s, 'SOL_SOCKET', 'SO_REUSEADDR', 1) -- ignore error\n"
"            end\n"
"            if not tcp_server_bind_addr(s, addr) or not s:listen(backlog) then\n"
"                local save_errno = boxerrno()\n"
"                socket_close(s)\n"
"                boxerrno(save_errno)\n"
"                return nil, boxerrno.strerror()\n"
"            end\n"
"            return s, addr\n"
"       end\n"
"    end\n"
"    -- DNS resolved successfully, but addresss family is not supported\n"
"    boxerrno(boxerrno.EAFNOSUPPORT)\n"
"    return nil, boxerrno.strerror()\n"
"end\n"
"\n"
"local function tcp_server(host, port, opts, timeout)\n"
"    local server = {}\n"
"    if type(opts) == 'function' then\n"
"        server.handler = opts\n"
"    elseif type(opts) == 'table' then\n"
"        if type(opts.handler) ~='function' or (opts.prepare ~= nil and\n"
"            type(opts.prepare) ~= 'function') then\n"
"            tcp_server_usage()\n"
"        end\n"
"        for k, v in pairs(opts) do\n"
"            server[k] = v\n"
"        end\n"
"    else\n"
"        tcp_server_usage()\n"
"    end\n"
"    server.name = server.name or 'server'\n"
"    local s, addr = tcp_server_bind(host, port, server.prepare, timeout)\n"
"    if not s then\n"
"        -- addr is error message now.\n"
"        return nil, addr\n"
"    end\n"
"    fiber.create(tcp_server_loop, server, s, addr)\n"
"    return s, addr\n"
"end\n"
"\n"
"socket_mt   = {\n"
"    __index = {\n"
"        close = socket_close;\n"
"        errno = socket_errno;\n"
"        error = socket_error;\n"
"        sysconnect = socket_sysconnect;\n"
"        syswrite = socket_syswrite;\n"
"        sysread = socket_sysread;\n"
"        nonblock = socket_nonblock;\n"
"        readable = socket_readable;\n"
"        writable = socket_writable;\n"
"        wait = socket_wait;\n"
"        listen = socket_listen;\n"
"        bind = socket_bind;\n"
"        shutdown = socket_shutdown;\n"
"        setsockopt = socket_setsockopt;\n"
"        getsockopt = socket_getsockopt;\n"
"        linger = socket_linger;\n"
"        accept = socket_accept;\n"
"        read = socket_read;\n"
"        write = socket_write;\n"
"        send = socket_send;\n"
"        recv = socket_recv;\n"
"        recvfrom = socket_recvfrom;\n"
"        sendto = socket_sendto;\n"
"        name = socket_name;\n"
"        peer = socket_peer;\n"
"        fd = socket_fd;\n"
"    };\n"
"    __tostring  = function(self)\n"
"        local fd = check_socket(self)\n"
"\n"
"        local save_errno = self._errno\n"
"        local name = format(\"fd %d\", fd)\n"
"        local aka = socket_name(self)\n"
"        if aka ~= nil then\n"
"            name = format(\"%s, aka %s:%s\", name, aka.host, aka.port)\n"
"        end\n"
"        local peer = socket_peer(self)\n"
"        if peer ~= nil then\n"
"            name = format(\"%s, peer %s:%s\", name, peer.host, peer.port)\n"
"        end\n"
"        self._errno = save_errno\n"
"        return name\n"
"    end,\n"
"    __serialize = function(self)\n"
"        -- Allow YAML, MsgPack and JSON to dump objects with sockets\n"
"        local fd = check_socket(self)\n"
"        return { fd = fd, peer = socket_peer(self), name = socket_name(self) }\n"
"    end\n"
"}\n"
"\n"
"--------------------------------------------------------------------------------\n"
"-- Lua Socket Emulation\n"
"--------------------------------------------------------------------------------\n"
"\n"
"local lsocket_tcp_mt\n"
"local lsocket_tcp_server_mt\n"
"local lsocket_tcp_client_mt\n"
"\n"
"--\n"
"-- TCP Master Socket\n"
"--\n"
"\n"
"local function lsocket_tcp_tostring(self)\n"
"    local fd = check_socket(self)\n"
"    return string.format(\"tcp{master}: fd=%d\", fd)\n"
"end\n"
"\n"
"local function lsocket_tcp_close(self)\n"
"    if not socket_close(self) then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return 1\n"
"end\n"
"\n"
"local function lsocket_tcp_getsockname(self)\n"
"    local aka = socket_name(self)\n"
"    if aka == nil then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return aka.host, tostring(aka.port), aka.family:match(\"AF_(.*)\"):lower()\n"
"end\n"
"\n"
"local function lsocket_tcp_getpeername(self)\n"
"    local peer = socket_peer(self)\n"
"    if peer == nil then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return peer.host, tostring(peer.port), peer.family:match(\"AF_(.*)\"):lower()\n"
"end\n"
"\n"
"local function lsocket_tcp_settimeout(self, value)\n"
"    check_socket(self)\n"
"    self.timeout = value\n"
"    -- mode is effectively ignored\n"
"    return 1\n"
"end\n"
"\n"
"local function lsocket_tcp_setoption(self, option, value)\n"
"    local r\n"
"    if option == 'reuseaddr' then\n"
"        r = socket_setsockopt(self, 'socket', 'SO_REUSEADDR', value)\n"
"    elseif option == 'keepalive' then\n"
"        r = socket_setsockopt(self, 'socket', 'SO_KEEPALIVE', value)\n"
"    elseif option == 'linger' then\n"
"        value = type(value) == 'table' and value.on or value\n"
"        -- Sic: value.timeout is ignored\n"
"        r = socket_linger(self, value)\n"
"    elseif option == 'tcp-nodelay' then\n"
"        r = socket_setsockopt(self, 'tcp', 'TCP_NODELAY', value)\n"
"    else\n"
"        error(format(\"Unknown socket option name: %s\", tostring(option)))\n"
"    end\n"
"    if not r then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return 1\n"
"end\n"
"\n"
"local function lsocket_tcp_bind(self, address, port)\n"
"    if not socket_bind(self, address, port) then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return 1\n"
"end\n"
"\n"
"local function lsocket_tcp_listen(self, backlog)\n"
"    if not socket_listen(self, backlog) then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    setmetatable(self, lsocket_tcp_server_mt)\n"
"    return 1\n"
"end\n"
"\n"
"local function lsocket_tcp_connect(self, host, port)\n"
"    check_socket(self)\n"
"    local deadline = fiber.clock() + (self.timeout or TIMEOUT_INFINITY)\n"
"    -- This function is broken by design\n"
"    local ga_opts = { family = 'AF_INET', type = 'SOCK_STREAM' }\n"
"    local timeout = deadline - fiber.clock()\n"
"    local dns, err = getaddrinfo(host, port, timeout, ga_opts)\n"
"    if dns == nil then\n"
"        self._errno = boxerrno.EINVAL\n"
"        return nil, err\n"
"    end\n"
"    if #dns == 0 then\n"
"        self._errno = boxerrno.EINVAL\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    for _, remote in ipairs(dns) do\n"
"        timeout = deadline - fiber.clock()\n"
"        if socket_tcp_connect(self, remote.host, remote.port, timeout) then\n"
"            return 1\n"
"        end\n"
"    end\n"
"    return nil, socket_error(self)\n"
"end\n"
"\n"
"lsocket_tcp_mt = {\n"
"    __index = {\n"
"        close = lsocket_tcp_close;\n"
"        getsockname = lsocket_tcp_getsockname;\n"
"        getpeername = lsocket_tcp_getpeername;\n"
"        settimeout = lsocket_tcp_settimeout;\n"
"        setoption = lsocket_tcp_setoption;\n"
"        bind = lsocket_tcp_bind;\n"
"        listen = lsocket_tcp_listen;\n"
"        connect = lsocket_tcp_connect;\n"
"    };\n"
"    __tostring = lsocket_tcp_tostring;\n"
"    __serialize = lsocket_tcp_tostring;\n"
"};\n"
"\n"
"--\n"
"-- TCP Server Socket\n"
"--\n"
"\n"
"local function lsocket_tcp_server_tostring(self)\n"
"    local fd = check_socket(self)\n"
"    return string.format(\"tcp{server}: fd=%d\", fd)\n"
"end\n"
"\n"
"local function lsocket_tcp_accept(self)\n"
"    check_socket(self)\n"
"    local deadline = fiber.clock() + (self.timeout or TIMEOUT_INFINITY)\n"
"    repeat\n"
"        local client = socket_accept(self)\n"
"        if client then\n"
"            setmetatable(client, lsocket_tcp_client_mt)\n"
"            return client\n"
"        end\n"
"        local errno = socket_errno(self)\n"
"        if not errno_is_transient[errno] then\n"
"            break\n"
"        end\n"
"    until not socket_readable(self, deadline - fiber.clock())\n"
"    return nil, socket_error(self)\n"
"end\n"
"\n"
"lsocket_tcp_server_mt = {\n"
"    __index = {\n"
"        close = lsocket_tcp_close;\n"
"        getsockname = lsocket_tcp_getsockname;\n"
"        getpeername = lsocket_tcp_getpeername;\n"
"        settimeout = lsocket_tcp_settimeout;\n"
"        setoption = lsocket_tcp_setoption;\n"
"        accept = lsocket_tcp_accept;\n"
"    };\n"
"    __tostring = lsocket_tcp_server_tostring;\n"
"    __serialize = lsocket_tcp_server_tostring;\n"
"};\n"
"\n"
"--\n"
"-- TCP Client Socket\n"
"--\n"
"\n"
"local function lsocket_tcp_client_tostring(self)\n"
"    local fd = check_socket(self)\n"
"    return string.format(\"tcp{client}: fd=%d\", fd)\n"
"end\n"
"\n"
"local function lsocket_tcp_receive(self, pattern, prefix)\n"
"    check_socket(self)\n"
"    prefix = prefix or ''\n"
"    local timeout = self.timeout or TIMEOUT_INFINITY\n"
"    local data\n"
"    if type(pattern) == 'number' then\n"
"        data = read(self, pattern, timeout, check_limit)\n"
"        if data == nil then\n"
"            return nil, socket_error(self)\n"
"        elseif #data < pattern then\n"
"            -- eof\n"
"            return nil, 'closed', prefix..data\n"
"        else\n"
"            return prefix..data\n"
"        end\n"
"    elseif pattern == \"*l\" or pattern == nil then\n"
"        data = read(self, LIMIT_INFINITY, timeout, check_delimiter, {\"\\n\"})\n"
"        if data == nil then\n"
"            return nil, socket_error(self)\n"
"        elseif #data > 0 and data:byte(#data) == 10 then\n"
"            -- remove '\\n'\n"
"            return prefix..data:sub(1, #data - 1)\n"
"        else\n"
"            -- eof\n"
"            return nil, 'closed', prefix..data\n"
"        end\n"
"    elseif pattern == \"*a\" then\n"
"        local result = { prefix }\n"
"        local deadline = fiber.clock() + (self.timeout or TIMEOUT_INFINITY)\n"
"        repeat\n"
"            local data = read(self, LIMIT_INFINITY, timeout, check_infinity)\n"
"            if data == nil then\n"
"                if not errno_is_transient[self._errno] then\n"
"                    return nil, socket_error(self)\n"
"                end\n"
"            elseif data == '' then\n"
"                break\n"
"            else\n"
"                table.insert(result, data)\n"
"            end\n"
"        until not socket_readable(self, deadline - fiber.clock())\n"
"        if #result == 1 then\n"
"            return nil, 'closed', table.concat(result)\n"
"        end\n"
"        return table.concat(result)\n"
"    else\n"
"        error(\"Usage: socket:receive(pattern, [, prefix])\")\n"
"    end\n"
"end\n"
"\n"
"local function lsocket_tcp_send(self, data, i, j)\n"
"    if i ~= nil then\n"
"        data = string.sub(data, i, j)\n"
"    end\n"
"    local sent = socket_write(self, data, self.timeout)\n"
"    if not sent then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return (i or 1) + sent - 1\n"
"end\n"
"\n"
"local function lsocket_tcp_shutdown(self, how)\n"
"    if not socket_shutdown(self, how) then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return 1\n"
"end\n"
"\n"
"lsocket_tcp_client_mt = {\n"
"    __index = {\n"
"        close = lsocket_tcp_close;\n"
"        getsockname = lsocket_tcp_getsockname;\n"
"        getpeername = lsocket_tcp_getpeername;\n"
"        settimeout = lsocket_tcp_settimeout;\n"
"        setoption = lsocket_tcp_setoption;\n"
"        receive = lsocket_tcp_receive;\n"
"        send = lsocket_tcp_send;\n"
"        shutdown = lsocket_tcp_shutdown;\n"
"    };\n"
"    __tostring = lsocket_tcp_client_tostring;\n"
"    __serialize = lsocket_tcp_client_tostring;\n"
"};\n"
"\n"
"--\n"
"-- Unconnected tcp socket (tcp{master}) should not have receive() and\n"
"-- send methods according to LuaSocket documentation[1]. Unfortunally,\n"
"-- original implementation is buggy and doesn't match the documentation.\n"
"-- Some modules (e.g. MobDebug) rely on this bug and attempt to invoke\n"
"-- receive()/send() on unconnected sockets.\n"
"-- [1]: http://w3.impa.br/~diego/software/luasocket/tcp.html\n"
"--\n"
"lsocket_tcp_mt.__index.receive = lsocket_tcp_receive;\n"
"lsocket_tcp_mt.__index.send = lsocket_tcp_send;\n"
"\n"
"--\n"
"-- TCP Constructor and Shortcuts\n"
"--\n"
"\n"
"local function lsocket_tcp(self)\n"
"    local s = socket_new('AF_INET', 'SOCK_STREAM', 'tcp')\n"
"    if not s then\n"
"        return nil, socket_error(self)\n"
"    end\n"
"    return setmetatable(s, lsocket_tcp_mt)\n"
"end\n"
"\n"
"local function lsocket_connect(host, port)\n"
"    if host == nil or port == nil then\n"
"        error(\"Usage: luasocket.connect(host, port)\")\n"
"    end\n"
"    local s, err = tcp_connect(host, port)\n"
"    if not s then\n"
"        return nil, err\n"
"    end\n"
"    setmetatable(s, lsocket_tcp_client_mt)\n"
"    return s\n"
"end\n"
"\n"
"local function lsocket_bind(host, port, backlog)\n"
"    if host == nil or port == nil then\n"
"        error(\"Usage: luasocket.bind(host, port [, backlog])\")\n"
"    end\n"
"    local function prepare(s) return backlog end -- luacheck: no unused args\n"
"    local s, err = tcp_server_bind(host, port, prepare)\n"
"    if not s then\n"
"        return nil, err\n"
"    end\n"
"    return setmetatable(s, lsocket_tcp_server_mt)\n"
"end\n"
"\n"
"--------------------------------------------------------------------------------\n"
"-- Module Definition\n"
"--------------------------------------------------------------------------------\n"
"\n"
"return setmetatable({\n"
"    getaddrinfo = getaddrinfo,\n"
"    tcp_connect = tcp_connect,\n"
"    tcp_server = tcp_server,\n"
"    iowait = internal.iowait,\n"
"    internal = internal,\n"
"}, {\n"
"    __call = function(self, ...) return socket_new(...) end;\n"
"    __index = {\n"
"        tcp = lsocket_tcp;\n"
"        connect = lsocket_connect;\n"
"        bind = lsocket_bind;\n"
"    }\n"
"})\n"
""
;
