const char compat_lua[] =
"-- compat.lua -- module intended to solve compatibility problems in different\n"
"-- parts of Tarantool. Introduced in gh-7000, see also gh-6912.\n"
"\n"
"local NEW = true\n"
"local OLD = false\n"
"\n"
"local tweaks = require('internal.tweaks')\n"
"\n"
"local options_format = {\n"
"    default        = 'string',\n"
"    brief          = 'string',\n"
"    obsolete       = 'string/nil',\n"
"    run_action_now = 'boolean/nil',\n"
"    action         = 'function/nil',\n"
"}\n"
"\n"
"local JSON_ESCAPE_BRIEF = [[\n"
"Whether to escape the forward slash symbol '/' using a backslash in a\n"
"json.encode() result. The old and the new behavior produce a result, which\n"
"is compatible ith the JSON specification. However most of other JSON encoders\n"
"don't escape the forward slash, so the new behavior is considered more safe.\n"
"\n"
"https://tarantool.io/compat/json_escape_forward_slash\n"
"]]\n"
"\n"
"local YAML_PRETTY_MULTILINE_BRIEF = [[\n"
"Whether to encode in block scalar style all multiline strings or ones\n"
"containing \"\\n\\n\" substring. The new behavior makes all multiline string output\n"
"as single text block which is handier for the reader, but may be incompatible\n"
"with some existing applications that rely on the old style.\n"
"\n"
"https://tarantool.io/compat/yaml_pretty_multiline\n"
"]]\n"
"\n"
"local FIBER_CHANNEL_GRACEFUL_CLOSE_BRIEF = [[\n"
"Whether fiber channel should be marked read-only on close, instead of being\n"
"destroyed. The new behavior allows user to mark the channel as read-only. This\n"
"way if a message was in the buffer before close, it will remain accessible after\n"
"close. The read-only channel will be closed automatically when the buffer\n"
"becomes empty, or will be deleted by GC if all links to it are lost, as before.\n"
"The new behavior can break code that relies on `ch:get()` returning `nil` after\n"
"channel close.\n"
"\n"
"https://tarantool.io/compat/fiber_channel_close_mode\n"
"]]\n"
"\n"
"local SQL_SEQ_SCAN_DEFAULT_BRIEF = [[\n"
"Whether seq_seq_scan session setting should be set to true or false during\n"
"initialization or new session creation.\n"
"\n"
"https://tarantool.io/compat/sql_seq_scan_default\n"
"]]\n"
"\n"
"-- Returns an action callback that toggles a tweak.\n"
"local function tweak_action(tweak_name, old_tweak_value, new_tweak_value)\n"
"    return function(is_new)\n"
"        if is_new then\n"
"            tweaks[tweak_name] = new_tweak_value\n"
"        else\n"
"            tweaks[tweak_name] = old_tweak_value\n"
"        end\n"
"    end\n"
"end\n"
"\n"
"-- Contains options descriptions in following format:\n"
"-- * default  (string)\n"
"-- * brief    (string)\n"
"-- * obsolete (string or nil)\n"
"-- * current  (boolean, true for 'new')\n"
"-- * selected (boolean)\n"
"-- * action   (function)\n"
"local options = {\n"
"    json_escape_forward_slash = {\n"
"        default = 'old',\n"
"        obsolete = nil,\n"
"        brief = JSON_ESCAPE_BRIEF,\n"
"        run_action_now = true,\n"
"        action = tweak_action('json_escape_forward_slash', true, false),\n"
"    },\n"
"    yaml_pretty_multiline = {\n"
"        default = 'old',\n"
"        obsolete = nil,\n"
"        brief = YAML_PRETTY_MULTILINE_BRIEF,\n"
"        action = tweak_action('yaml_pretty_multiline', false, true),\n"
"    },\n"
"    fiber_channel_close_mode = {\n"
"        default = 'old',\n"
"        obsolete = nil,\n"
"        brief = FIBER_CHANNEL_GRACEFUL_CLOSE_BRIEF,\n"
"        action = tweak_action('fiber_channel_close_mode',\n"
"                              'forceful', 'graceful'),\n"
"    },\n"
"    sql_seq_scan_default = {\n"
"        default = 'old',\n"
"        obsolete = nil,\n"
"        brief = SQL_SEQ_SCAN_DEFAULT_BRIEF,\n"
"        action = tweak_action('sql_seq_scan_default', true, false),\n"
"    },\n"
"}\n"
"\n"
"-- Array with option names in order of addition.\n"
"local options_order = { }\n"
"\n"
"local help = [[\n"
"Tarantool compatibility module.\n"
"To get help, see the Tarantool manual at https://tarantool.io/compat\n"
"Available commands:\n"
"\n"
"    compat.help()                   -- show this help\n"
"    compat.<option_name>            -- list option info\n"
"    compat.<option_name> = 'old'    -- set desired value to option, could be\n"
"                                       'old', 'new', 'default'\n"
"    compat{<option_name> = 'new'}   -- set listed options to desired values\n"
"                                       ('old', 'new', 'default')\n"
"    compat.dump()                   -- get Lua command that sets up different\n"
"                                       compat with same options as current\n"
"    compat.add_option()             -- add new option by providing a table with\n"
"                                       name, default, brief, obsolete and action\n"
"                                       function\n"
"]]\n"
"\n"
"-- Returns table with all non-obsolete options from `options` and their values.\n"
"local function serialize_compat()\n"
"    -- The results of serialization should be ordered correctly.\n"
"    -- The only feasible way for it is using indexed table with\n"
"    -- values {option_name = val}.\n"
"    local result = { }\n"
"\n"
"    for _, name in pairs(options_order) do\n"
"        local option = options[name]\n"
"        if option.selected and option.current == NEW and\n"
"                not option.obsolete then\n"
"            table.insert(result, {[name] = 'new'})\n"
"        end\n"
"    end\n"
"    for _, name in pairs(options_order) do\n"
"        local option = options[name]\n"
"        if option.selected and option.current == OLD and\n"
"                not option.obsolete then\n"
"            table.insert(result, {[name] = 'old'})\n"
"        end\n"
"    end\n"
"    for _, name in pairs(options_order) do\n"
"        local option = options[name]\n"
"        if not option.selected and not option.obsolete then\n"
"            local val = option.current and 'new' or 'old'\n"
"            table.insert(result, {[name] = ('default (%s)'):format(val)})\n"
"        end\n"
"    end\n"
"\n"
"    -- Result is an indexed table but this way user can interact with it as if\n"
"    -- it was key-value.\n"
"    result = setmetatable(result, {\n"
"        __index = function(self, key)\n"
"            for i = 1, #self do\n"
"                if self[i][key] ~= nil then\n"
"                    return self[i][key]\n"
"                end\n"
"            end\n"
"        end\n"
"    })\n"
"\n"
"    return result\n"
"end\n"
"\n"
"-- Checks options correctness.\n"
"local function verify_option(name, option)\n"
"    if type(name) ~= 'string' then\n"
"        local msg = 'Option name must be a string (%s provided)'\n"
"        error(msg:format(type(name)))\n"
"    end\n"
"    for p, t in pairs(options_format) do\n"
"        if not t:find(type(option[p])) then\n"
"            local msg = 'Invalid option table for %s, bad %s (%s is expected)'\n"
"            error(msg:format(name, p, t))\n"
"        end\n"
"    end\n"
"    if option.default ~= 'new' and option.default ~= 'old' then\n"
"        local msg = \"Invalid option table for %s, bad default\" ..\n"
"                    \" ('new'/'old' is expected)\"\n"
"        error(msg:format(name))\n"
"    end\n"
"    if not option.obsolete and option.action == nil then\n"
"        local msg = \"Invalid option table for %s, bad action\" ..\n"
"                    \" (function is expected)\"\n"
"        error(msg:format(name))\n"
"    end\n"
"    if option.obsolete and option.default == 'old' then\n"
"        error(('Obsolete option %s default is wrong.'):format(name))\n"
"    end\n"
"end\n"
"\n"
"-- Checks if operation is valid, sets value to an option and runs postaction.\n"
"local function set_option(name, val)\n"
"    local option = options[name]\n"
"    if not option then\n"
"        error(('Invalid option %s'):format(name))\n"
"    end\n"
"    local selected\n"
"    if val == 'new' then\n"
"        val = NEW\n"
"        selected = true\n"
"    elseif val == 'old' then\n"
"        val = OLD\n"
"        selected = true\n"
"    elseif val == 'default' then\n"
"        val = option.default == 'new'\n"
"        selected = false\n"
"    else\n"
"        error(('Invalid argument %s in option %s'):format(val, name))\n"
"    end\n"
"    if option.obsolete and val == OLD then\n"
"        error(('Chosen option %s is no longer available'):format(name))\n"
"    end\n"
"    if val ~= option.current then\n"
"        option.action(val)\n"
"    end\n"
"    option.current = val\n"
"    option.selected = selected\n"
"end\n"
"\n"
"local compat = { }\n"
"\n"
"function compat.dump(mode)\n"
"    -- dump() works in one of the following modes:\n"
"    -- dump compat configuration as is with `default` if any\n"
"    local ACT_NIL     = 0\n"
"    -- dump compat configuration as is but replace `default` with\n"
"    -- its value (new/old)\n"
"    local ACT_CURRENT = 1\n"
"    -- dump everything as `new`\n"
"    local ACT_NEW     = 2\n"
"    -- dump everything as `old`, besides obsolete, that could be only `new`\n"
"    local ACT_OLD     = 3\n"
"    -- dump everything as `default`\n"
"    local ACT_DEFAULT = 4\n"
"    local action\n"
"    if mode == 'new' then\n"
"        action = ACT_NEW\n"
"    elseif mode == 'old' then\n"
"        action = ACT_OLD\n"
"    elseif mode == 'default' then\n"
"        action = ACT_DEFAULT\n"
"    elseif mode == 'current' then\n"
"        action = ACT_CURRENT\n"
"    elseif mode == nil then\n"
"        action = ACT_NIL\n"
"    else\n"
"        error(\"usage: compat.dump('new'/'old'/'default'/'current'/nil)\")\n"
"    end\n"
"    local result = [[require('compat')({]]\n"
"    local max_key_len = 0\n"
"    for _, key in pairs(options_order) do\n"
"        max_key_len = (#key > max_key_len) and #key or max_key_len\n"
"    end\n"
"    for _, key in pairs(options_order) do\n"
"        local val\n"
"        local comment\n"
"        if options[key].obsolete then\n"
"            if action == ACT_NEW or action == ACT_OLD or\n"
"                    action == ACT_CURRENT or\n"
"                    (action == ACT_NIL and options[key].selected) then\n"
"                val = \"'new'\"\n"
"            else\n"
"                val = \"'default'\"\n"
"            end\n"
"            comment = 'obsolete since ' .. options[key].obsolete\n"
"        else\n"
"            if action == ACT_CURRENT then\n"
"                if options[key].selected and options[key].current == NEW or\n"
"                        options[key].default == 'new' then\n"
"                    val = \"'new'\"\n"
"                else\n"
"                    val = \"'old'\"\n"
"                end\n"
"            elseif action == ACT_DEFAULT or (action == ACT_NIL and\n"
"                    not options[key].selected) then\n"
"                val = \"'default'\"\n"
"            elseif action == ACT_NEW or\n"
"                    (action == ACT_NIL and options[key].current == NEW) then\n"
"                val = \"'new'\"\n"
"            else\n"
"                val = \"'old'\"\n"
"            end\n"
"        end\n"
"        -- Can't use '\\t' due to gh-7681.\n"
"        local form = '%s\\n    %s%s = %s,'\n"
"        result = form:format(result, key,\n"
"                             string.rep(' ', max_key_len - #key), val)\n"
"        if comment then\n"
"            if (val ~= \"'default'\") then\n"
"                -- For alignment.\n"
"                result = result .. '    '\n"
"            end\n"
"            result = result .. ' -- ' .. comment\n"
"        end\n"
"    end\n"
"    if not result:find('\\n') then\n"
"        return result .. '})'\n"
"    else\n"
"        -- Need double '\\n' due to gh-3012.\n"
"        return result .. '\\n})\\n\\n'\n"
"    end\n"
"end\n"
"\n"
"-- option_def:\n"
"-- * name           (string)\n"
"-- * default        ('new'/'old')\n"
"-- * brief          (string)\n"
"-- * obsolete       (string/nil)\n"
"-- * action         (function/nil)\n"
"-- * run_action_now (boolean/nil)\n"
"function compat.add_option(option_def)\n"
"    if type(option_def) ~= 'table' then\n"
"        error(\"usage: compat.add_option({name = '...', default = 'new'/'old'\" ..\n"
"              \", brief = '...', action = func, run_action_now = true/false})\")\n"
"    end\n"
"    local name = option_def.name\n"
"    verify_option(name, option_def)\n"
"\n"
"    local current = option_def.default == 'new' and NEW or OLD\n"
"    -- If not hot reload, set `current` and `selected`.\n"
"    if not options[name] then\n"
"        options[name] = {\n"
"            current = current,\n"
"            selected = false\n"
"        }\n"
"        table.insert(options_order, name)\n"
"    -- If hot reload but option is set to 'default', update `current`.\n"
"    elseif not options[name].selected then\n"
"        options[name].current = current\n"
"    end\n"
"    -- Else keep `current` and `selected` as is.\n"
"\n"
"    -- Copy all other fields.\n"
"    local option = options[name]\n"
"    option.brief    = option_def.brief\n"
"    option.default  = option_def.default\n"
"    option.obsolete = option_def.obsolete\n"
"    option.action   = option_def.action\n"
"\n"
"    if not option.obsolete and option_def.run_action_now then\n"
"        option.action(options[name].current)\n"
"    end\n"
"end\n"
"\n"
"function compat.help()\n"
"    return help\n"
"end\n"
"\n"
"function compat.preload()\n"
"    for key, option in pairs(options) do\n"
"        verify_option(key, option)\n"
"        table.insert(options_order, key)\n"
"        option.current = option.default == 'new'\n"
"        option.selected = false\n"
"    end\n"
"    -- Preload should be run only once and is removed not to confuse users.\n"
"    compat.preload = nil\n"
"end\n"
"\n"
"function compat.postload()\n"
"    for _key, option in pairs(options) do\n"
"        if not option.obsolete and option.run_action_now then\n"
"            option.action(option.current)\n"
"        end\n"
"        option.run_action_now = nil\n"
"    end\n"
"    -- Postload should be run only once and is removed not to confuse users.\n"
"    compat.postload = nil\n"
"end\n"
"\n"
"local compat_mt = { }\n"
"\n"
"function compat_mt.__call(_, list)\n"
"    if type(list) ~= 'table' then\n"
"        error(\"usage: compat({<option_name> = 'new'/'old'/'default'})\")\n"
"    end\n"
"    for key, val in pairs(list) do\n"
"        set_option(key, val)\n"
"    end\n"
"end\n"
"\n"
"function compat_mt.__newindex(_, key, val)\n"
"    set_option(key, val)\n"
"end\n"
"\n"
"local compat_option_methods = {}\n"
"\n"
"-- Returns the effective compat option value - 'old' or 'new'.\n"
"local function option_value(option)\n"
"    if option.current == 'default' then\n"
"        return option.default\n"
"    else\n"
"        return option.current\n"
"    end\n"
"end\n"
"\n"
"-- Whether the effective value of the option is `new`.\n"
"function compat_option_methods.is_new(option)\n"
"    if type(option) ~= 'table' then\n"
"        error('usage: compat.<option_name>:is_new()')\n"
"    end\n"
"    return option_value(option) == 'new'\n"
"end\n"
"\n"
"-- Whether the effective value of the option is `old`.\n"
"function compat_option_methods.is_old(option)\n"
"    if type(option) ~= 'table' then\n"
"        error('usage: compat.<option_name>:is_old()')\n"
"    end\n"
"    return option_value(option) == 'old'\n"
"end\n"
"\n"
"local compat_option_mt = {\n"
"    __index = compat_option_methods,\n"
"}\n"
"\n"
"function compat_mt.__index(_, key)\n"
"    if not options[key] then\n"
"        error(('Invalid option %s'):format(key))\n"
"    end\n"
"    local result = {\n"
"        brief = options[key].brief,\n"
"        default = options[key].default,\n"
"        obsolete = options[key].obsolete,\n"
"    }\n"
"\n"
"    if not options[key].selected then\n"
"        result.current = 'default'\n"
"    elseif options[key].current == NEW then\n"
"        result.current = 'new'\n"
"    else\n"
"        result.current = 'old'\n"
"    end\n"
"\n"
"    return setmetatable(result, compat_option_mt)\n"
"end\n"
"\n"
"compat_mt.__serialize = serialize_compat\n"
"\n"
"function compat_mt.__tostring()\n"
"    -- YAML can't be required at the beginning as it isn't initialized by then.\n"
"    return require('yaml').encode(serialize_compat())\n"
"end\n"
"\n"
"function compat_mt.__autocomplete()\n"
"    local res = { }\n"
"    for key, option in pairs(options) do\n"
"        if not option.obsolete then\n"
"            res[key] = true\n"
"        end\n"
"    end\n"
"    return res\n"
"end\n"
"\n"
"\n"
"-- Preload reads hardcoded into `options` table options, verifies\n"
"-- them and sets other compat-specific values.\n"
"compat.preload()\n"
"-- Postload calls actions of hardcoded options if according\n"
"-- `run_action_now` is `true`. Most of the actions call `require`\n"
"-- but the required modules are unlikely to be loaded at compat\n"
"-- initialization, so it may be needed to move this call somewhere\n"
"-- else in the future.\n"
"compat.postload()\n"
"\n"
"compat = setmetatable(compat, compat_mt)\n"
"\n"
"return compat\n"
""
;
