/* Copyright (c) 2022 Advanced Micro Devices, Inc.

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE. */

#include <cassert>
#include "src/utils/helper.h"

#include "src/core/counters/basic/basic_counter.h"
#include "src/core/hardware/hsa_info.h"

#define ASSERTM(exp, msg) assert(((void)msg, exp))

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wmaybe-uninitialized"
namespace Counter {

BasicCounter::BasicCounter(uint64_t event_id, std::string block_id,
                           std::string name, std::string description,
                           std::string gpu_name)
    : Counter(name, description, gpu_name),
      event_id_(event_id),
      block_id_(block_id) {
  AddCounterToCounterMap();
}

BasicCounter::~BasicCounter() {}

uint64_t BasicCounter::GetBasicCounterID() {
  return GetCounterID();
}

uint64_t BasicCounter::GetEventId() { return event_id_; }
std::string BasicCounter::GetBlockId() { return block_id_; }
std::string BasicCounter::GetName() { return Counter::GetName(); }

bool BasicCounter::GetValue(uint64_t* value, int64_t instance_id = -1) {
  Agent::CounterHardwareInfo* agent_info =
      reinterpret_cast<Agent::CounterHardwareInfo*>(counter_hw_info);
  if ((agent_info->getNumInstances() > 1 && instance_id == -1) ||
      instance_id < -1 || instance_id >= agent_info->getNumInstances())
    return false;
  if (instance_id == -1) *value = instances_values_[0];
  *value = instances_values_[instance_id];
  return true;
}

uint64_t BasicCounter::GetValue(int64_t instance_id) {
  Agent::CounterHardwareInfo* agent_info =
      reinterpret_cast<Agent::CounterHardwareInfo*>(counter_hw_info);
  if ((agent_info->getNumInstances() > 1 && instance_id == -1) ||
      instance_id < -1 || instance_id >= agent_info->getNumInstances())
    throw(std::string("Error: Wrong number of instances (") +
                std::to_string(agent_info->getNumInstances()) +
                ") OR Instance ID is less than 0 ");
  if (instance_id == -1) return instances_values_[0];
  return instances_values_[instance_id];
}

uint64_t BasicCounter::avr(int64_t instances_count) {
  Agent::CounterHardwareInfo* agent_info =
      reinterpret_cast<Agent::CounterHardwareInfo*>(counter_hw_info);
  if (agent_info->getNumInstances() > instances_count)
    throw(std::string("Error: Number of instances (") +
                std::to_string(agent_info->getNumInstances()) +
                ") is greater than the given instance count(" +
                std::to_string(instances_count) + ")");
  uint64_t result = 0;
  int64_t instance_id;
  for (instance_id = 0; instance_id < instances_count; instance_id++) {
    uint64_t value;
    if (GetValue(&value, instance_id)) result += value;
  }
  return result / instances_count;
}
uint64_t BasicCounter::max(int64_t instances_count) {
  uint64_t result = 0;
  int64_t instance_id;
  for (instance_id = 0; instance_id < instances_count; instance_id++) {
    uint64_t value;
    if (GetValue(&value, instance_id) && result < value) result = value;
  }
  return result;
}
uint64_t BasicCounter::min(int64_t instances_count) {
  int64_t instance_id;
  uint64_t result = 0;
  for (instance_id = 0; instance_id < instances_count; instance_id++) {
    uint64_t value;
    if (GetValue(&value, instance_id) && result > value) result = value;
  }
  return result;
}
uint64_t BasicCounter::sum(int64_t instances_count) {
  int64_t instance_id;
  uint64_t result = 0;
  for (instance_id = 0; instance_id < instances_count; instance_id++) {
    uint64_t value;
    if (GetValue(&value, instance_id)) result += value;
  }
  return result;
}

uint64_t operator+(BasicCounter counter, const uint64_t number) {
  [[maybe_unused]] uint64_t value = 0;
  ASSERTM(counter.GetValue(&value), "Error: Counter has no value!");
  return number + value;
}
uint64_t operator-(BasicCounter counter, const uint64_t number) {
  [[maybe_unused]] uint64_t value = 0;
  ASSERTM(counter.GetValue(&value), "Error: Counter has no value!");
  return number - value;
}
uint64_t operator*(BasicCounter counter, const uint64_t number) {
  [[maybe_unused]] uint64_t value = 0;
  ASSERTM(counter.GetValue(&value), "Error: Counter has no value!");
  return number * value;
}
uint64_t operator/(BasicCounter counter, const uint64_t number) {
  [[maybe_unused]] uint64_t value = 0;
  ASSERTM(counter.GetValue(&value), "Error: Counter has no value!");
  return number / value;
}
uint64_t operator^(BasicCounter counter, const uint64_t number) {
  [[maybe_unused]] uint64_t value = 0;
  ASSERTM(counter.GetValue(&value), "Error: Counter has no value!");
  return number ^ value;
}

uint64_t operator+(BasicCounter counter1, BasicCounter counter2) {
  [[maybe_unused]] uint64_t value1 = 0;
  ASSERTM(counter1.GetValue(&value1), "Error: Counter has no value!");
  [[maybe_unused]] uint64_t value2 = 0;
  ASSERTM(counter2.GetValue(&value2), "Error: Counter has no value!");
  return value1 + value2;
}
uint64_t operator-(BasicCounter counter1, BasicCounter counter2) {
  [[maybe_unused]] uint64_t value1 = 0;
  ASSERTM(counter1.GetValue(&value1), "Error: Counter has no value!");
  [[maybe_unused]] uint64_t value2 = 0;
  ASSERTM(counter2.GetValue(&value2), "Error: Counter has no value!");
  return value1 - value2;
}
uint64_t operator*(BasicCounter counter1, BasicCounter counter2) {
  [[maybe_unused]] uint64_t value1 = 0;
  ASSERTM(counter1.GetValue(&value1), "Error: Counter has no value!");
  [[maybe_unused]] uint64_t value2 = 0;
  ASSERTM(counter2.GetValue(&value2), "Error: Counter has no value!");
  return value1 * value2;
}
uint64_t operator/(BasicCounter counter1, BasicCounter counter2) {
  [[maybe_unused]] uint64_t value1 = 0;
  ASSERTM(counter1.GetValue( & value1), "Error: Counter has no value!");
  [[maybe_unused]] uint64_t value2 = 0;
  ASSERTM(counter2.GetValue( & value2), "Error: Counter has no value!");
 return value1 / value2;
}
uint64_t operator^(BasicCounter counter1, BasicCounter counter2) {
  [[maybe_unused]] uint64_t value1 = 0;
  ASSERTM(counter1.GetValue(&value1), "Error: Counter has no value!");
  [[maybe_unused]] uint64_t value2 = 0;
  ASSERTM(counter2.GetValue(&value2), "Error: Counter has no value!");
  return value1 ^ value2;
}

static std::map<uint64_t, BasicCounter> basic_counters;

BasicCounter* GetGeneratedBasicCounter(uint64_t id) {
  return &basic_counters.at(id);
}

void ClearBasicCounters() {
  basic_counters.clear();
}

/**
 * @brief Basic Counters
 *
 * @{
 */
uint64_t GetBasicCounter(const char* name, const char* gpu_name) {
  std::string gpu;

	/**
	 * @brief Basic gfx8 counters
	 *
	 * @{
	 */
	gpu = "gfx8";

	if (strncmp(gpu_name, gpu.c_str(), gpu.length())==0) {
	/**
	 * Basic Counter: GRBM_COUNT
	 *
	 * Tie High - Count Number of Clocks
	 */
	if (strcmp(name, "GRBM_COUNT")==0) {
		basic_counters.emplace(0, BasicCounter{0, "GRBM", "GRBM_COUNT", "Tie High - Count Number of Clocks", "gfx8"});
		return 0;
	}
	/**
	 * Basic Counter: GRBM_GUI_ACTIVE
	 *
	 * The GUI is Active
	 */
	if (strcmp(name, "GRBM_GUI_ACTIVE")==0) {
		basic_counters.emplace(2, BasicCounter{2, "GRBM", "GRBM_GUI_ACTIVE", "The GUI is Active", "gfx8"});
		return 2;
	}
	/**
	 * Basic Counter: SQ_WAVES
	 *
	 * Count number of waves sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES")==0) {
		basic_counters.emplace(4, BasicCounter{4, "SQ", "SQ_WAVES", "Count number of waves sent to SQs. (per-simd, emulated, global)", "gfx8"});
		return 4;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU
	 *
	 * Number of VALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU")==0) {
		basic_counters.emplace(26, BasicCounter{26, "SQ", "SQ_INSTS_VALU", "Number of VALU instructions issued. (per-simd, emulated)", "gfx8"});
		return 26;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_WR
	 *
	 * Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_WR")==0) {
		basic_counters.emplace(27, BasicCounter{27, "SQ", "SQ_INSTS_VMEM_WR", "Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)", "gfx8"});
		return 27;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_RD
	 *
	 * Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_RD")==0) {
		basic_counters.emplace(28, BasicCounter{28, "SQ", "SQ_INSTS_VMEM_RD", "Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)", "gfx8"});
		return 28;
	}
	/**
	 * Basic Counter: SQ_INSTS_SALU
	 *
	 * Number of SALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SALU")==0) {
		basic_counters.emplace(30, BasicCounter{30, "SQ", "SQ_INSTS_SALU", "Number of SALU instructions issued. (per-simd, emulated)", "gfx8"});
		return 30;
	}
	/**
	 * Basic Counter: SQ_INSTS_SMEM
	 *
	 * Number of SMEM instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SMEM")==0) {
		basic_counters.emplace(31, BasicCounter{31, "SQ", "SQ_INSTS_SMEM", "Number of SMEM instructions issued. (per-simd, emulated)", "gfx8"});
		return 31;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT
	 *
	 * Number of FLAT instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT")==0) {
		basic_counters.emplace(32, BasicCounter{32, "SQ", "SQ_INSTS_FLAT", "Number of FLAT instructions issued. (per-simd, emulated)", "gfx8"});
		return 32;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT_LDS_ONLY
	 *
	 * Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT_LDS_ONLY")==0) {
		basic_counters.emplace(33, BasicCounter{33, "SQ", "SQ_INSTS_FLAT_LDS_ONLY", "Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)", "gfx8"});
		return 33;
	}
	/**
	 * Basic Counter: SQ_INSTS_LDS
	 *
	 * Number of LDS instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_LDS")==0) {
		basic_counters.emplace(34, BasicCounter{34, "SQ", "SQ_INSTS_LDS", "Number of LDS instructions issued (including FLAT). (per-simd, emulated)", "gfx8"});
		return 34;
	}
	/**
	 * Basic Counter: SQ_INSTS_GDS
	 *
	 * Number of GDS instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_GDS")==0) {
		basic_counters.emplace(35, BasicCounter{35, "SQ", "SQ_INSTS_GDS", "Number of GDS instructions issued. (per-simd, emulated)", "gfx8"});
		return 35;
	}
	/**
	 * Basic Counter: SQ_WAIT_INST_LDS
	 *
	 * Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)
	 */
	if (strcmp(name, "SQ_WAIT_INST_LDS")==0) {
		basic_counters.emplace(61, BasicCounter{61, "SQ", "SQ_WAIT_INST_LDS", "Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)", "gfx8"});
		return 61;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_VALU
	 *
	 * Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, nondeterministic). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_VALU")==0) {
		basic_counters.emplace(69, BasicCounter{69, "SQ", "SQ_ACTIVE_INST_VALU", "Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, nondeterministic). Units in quad-cycles(4 cycles)", "gfx8"});
		return 69;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_SALU
	 *
	 * Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_SALU")==0) {
		basic_counters.emplace(86, BasicCounter{86, "SQ", "SQ_INST_CYCLES_SALU", "Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated)", "gfx8"});
		return 86;
	}
	/**
	 * Basic Counter: SQ_THREAD_CYCLES_VALU
	 *
	 * Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)
	 */
	if (strcmp(name, "SQ_THREAD_CYCLES_VALU")==0) {
		basic_counters.emplace(89, BasicCounter{89, "SQ", "SQ_THREAD_CYCLES_VALU", "Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)", "gfx8"});
		return 89;
	}
	/**
	 * Basic Counter: SQ_LDS_BANK_CONFLICT
	 *
	 * Number of cycles LDS is stalled by bank conflicts. (emulated)
	 */
	if (strcmp(name, "SQ_LDS_BANK_CONFLICT")==0) {
		basic_counters.emplace(97, BasicCounter{97, "SQ", "SQ_LDS_BANK_CONFLICT", "Number of cycles LDS is stalled by bank conflicts. (emulated)", "gfx8"});
		return 97;
	}
	/**
	 * Basic Counter: TA_TA_BUSY
	 *
	 * TA block is busy. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_TA_BUSY")==0) {
		basic_counters.emplace(15, BasicCounter{15, "TA", "TA_TA_BUSY", "TA block is busy. Perf_Windowing not supported for this counter.", "gfx8"});
		return 15;
	}
	/**
	 * Basic Counter: TA_FLAT_READ_WAVEFRONTS
	 *
	 * Number of flat opcode reads processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_READ_WAVEFRONTS")==0) {
		basic_counters.emplace(101, BasicCounter{101, "TA", "TA_FLAT_READ_WAVEFRONTS", "Number of flat opcode reads processed by the TA.", "gfx8"});
		return 101;
	}
	/**
	 * Basic Counter: TA_FLAT_WRITE_WAVEFRONTS
	 *
	 * Number of flat opcode writes processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_WRITE_WAVEFRONTS")==0) {
		basic_counters.emplace(102, BasicCounter{102, "TA", "TA_FLAT_WRITE_WAVEFRONTS", "Number of flat opcode writes processed by the TA.", "gfx8"});
		return 102;
	}
	/**
	 * Basic Counter: TCC_HIT
	 *
	 * Number of cache hits.
	 */
	if (strcmp(name, "TCC_HIT")==0) {
		basic_counters.emplace(18, BasicCounter{18, "TCC", "TCC_HIT", "Number of cache hits.", "gfx8"});
		return 18;
	}
	/**
	 * Basic Counter: TCC_MISS
	 *
	 * Number of cache misses. UC reads count as misses.
	 */
	if (strcmp(name, "TCC_MISS")==0) {
		basic_counters.emplace(19, BasicCounter{19, "TCC", "TCC_MISS", "Number of cache misses. UC reads count as misses.", "gfx8"});
		return 19;
	}
	/**
	 * Basic Counter: TCC_MC_RDREQ
	 *
	 * Number of 32-byte reads. The hardware actually does 64-byte reads but the number is adjusted to provide uniformity.
	 */
	if (strcmp(name, "TCC_MC_RDREQ")==0) {
		basic_counters.emplace(35, BasicCounter{35, "TCC", "TCC_MC_RDREQ", "Number of 32-byte reads. The hardware actually does 64-byte reads but the number is adjusted to provide uniformity.", "gfx8"});
		return 35;
	}
	/**
	 * Basic Counter: TCC_MC_WRREQ
	 *
	 * Number of 32-byte transactions going over the TC_MC_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests.
	 */
	if (strcmp(name, "TCC_MC_WRREQ")==0) {
		basic_counters.emplace(26, BasicCounter{26, "TCC", "TCC_MC_WRREQ", "Number of 32-byte transactions going over the TC_MC_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests.", "gfx8"});
		return 26;
	}
	/**
	 * Basic Counter: TCC_MC_WRREQ_STALL
	 *
	 * Number of cycles a write request was stalled.
	 */
	if (strcmp(name, "TCC_MC_WRREQ_STALL")==0) {
		basic_counters.emplace(28, BasicCounter{28, "TCC", "TCC_MC_WRREQ_STALL", "Number of cycles a write request was stalled.", "gfx8"});
		return 28;
	}
	/**
	 * Basic Counter: TCP_TCP_TA_DATA_STALL_CYCLES
	 *
	 * TCP stalls TA data interface. Now Windowed.
	 */
	if (strcmp(name, "TCP_TCP_TA_DATA_STALL_CYCLES")==0) {
		basic_counters.emplace(3, BasicCounter{3, "TCP", "TCP_TCP_TA_DATA_STALL_CYCLES", "TCP stalls TA data interface. Now Windowed.", "gfx8"});
		return 3;
	}
	}

	/**
	 * @}
	 */

	/**
	 * @brief Basic gfx906 counters
	 *
	 * @{
	 */
	gpu = "gfx906";

	if (strncmp(gpu_name, gpu.c_str(), gpu.length())==0) {
	/**
	 * Basic Counter: GRBM_COUNT
	 *
	 * Tie High - Count Number of Clocks
	 */
	if (strcmp(name, "GRBM_COUNT")==0) {
		basic_counters.emplace(0, BasicCounter{0, "GRBM", "GRBM_COUNT", "Tie High - Count Number of Clocks", "gfx906"});
		return 0;
	}
	/**
	 * Basic Counter: GRBM_GUI_ACTIVE
	 *
	 * The GUI is Active
	 */
	if (strcmp(name, "GRBM_GUI_ACTIVE")==0) {
		basic_counters.emplace(2, BasicCounter{2, "GRBM", "GRBM_GUI_ACTIVE", "The GUI is Active", "gfx906"});
		return 2;
	}
	/**
	 * Basic Counter: SQ_WAVES
	 *
	 * Count number of waves sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES")==0) {
		basic_counters.emplace(4, BasicCounter{4, "SQ", "SQ_WAVES", "Count number of waves sent to SQs. (per-simd, emulated, global)", "gfx906"});
		return 4;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU
	 *
	 * Number of VALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU")==0) {
		basic_counters.emplace(26, BasicCounter{26, "SQ", "SQ_INSTS_VALU", "Number of VALU instructions issued. (per-simd, emulated)", "gfx906"});
		return 26;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_WR
	 *
	 * Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_WR")==0) {
		basic_counters.emplace(27, BasicCounter{27, "SQ", "SQ_INSTS_VMEM_WR", "Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)", "gfx906"});
		return 27;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_RD
	 *
	 * Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_RD")==0) {
		basic_counters.emplace(28, BasicCounter{28, "SQ", "SQ_INSTS_VMEM_RD", "Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)", "gfx906"});
		return 28;
	}
	/**
	 * Basic Counter: SQ_INSTS_SALU
	 *
	 * Number of SALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SALU")==0) {
		basic_counters.emplace(30, BasicCounter{30, "SQ", "SQ_INSTS_SALU", "Number of SALU instructions issued. (per-simd, emulated)", "gfx906"});
		return 30;
	}
	/**
	 * Basic Counter: SQ_INSTS_SMEM
	 *
	 * Number of SMEM instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SMEM")==0) {
		basic_counters.emplace(31, BasicCounter{31, "SQ", "SQ_INSTS_SMEM", "Number of SMEM instructions issued. (per-simd, emulated)", "gfx906"});
		return 31;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT
	 *
	 * Number of FLAT instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT")==0) {
		basic_counters.emplace(32, BasicCounter{32, "SQ", "SQ_INSTS_FLAT", "Number of FLAT instructions issued. (per-simd, emulated)", "gfx906"});
		return 32;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT_LDS_ONLY
	 *
	 * Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT_LDS_ONLY")==0) {
		basic_counters.emplace(33, BasicCounter{33, "SQ", "SQ_INSTS_FLAT_LDS_ONLY", "Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)", "gfx906"});
		return 33;
	}
	/**
	 * Basic Counter: SQ_INSTS_LDS
	 *
	 * Number of LDS instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_LDS")==0) {
		basic_counters.emplace(34, BasicCounter{34, "SQ", "SQ_INSTS_LDS", "Number of LDS instructions issued (including FLAT). (per-simd, emulated)", "gfx906"});
		return 34;
	}
	/**
	 * Basic Counter: SQ_INSTS_GDS
	 *
	 * Number of GDS instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_GDS")==0) {
		basic_counters.emplace(35, BasicCounter{35, "SQ", "SQ_INSTS_GDS", "Number of GDS instructions issued. (per-simd, emulated)", "gfx906"});
		return 35;
	}
	/**
	 * Basic Counter: SQ_WAIT_INST_LDS
	 *
	 * Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)
	 */
	if (strcmp(name, "SQ_WAIT_INST_LDS")==0) {
		basic_counters.emplace(63, BasicCounter{63, "SQ", "SQ_WAIT_INST_LDS", "Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)", "gfx906"});
		return 63;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_VALU
	 *
	 * regspec 71? Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, nondeterministic). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_VALU")==0) {
		basic_counters.emplace(71, BasicCounter{71, "SQ", "SQ_ACTIVE_INST_VALU", "regspec 71? Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, nondeterministic). Units in quad-cycles(4 cycles)", "gfx906"});
		return 71;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_SALU
	 *
	 * Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_SALU")==0) {
		basic_counters.emplace(84, BasicCounter{84, "SQ", "SQ_INST_CYCLES_SALU", "Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated)", "gfx906"});
		return 84;
	}
	/**
	 * Basic Counter: SQ_THREAD_CYCLES_VALU
	 *
	 * Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)
	 */
	if (strcmp(name, "SQ_THREAD_CYCLES_VALU")==0) {
		basic_counters.emplace(85, BasicCounter{85, "SQ", "SQ_THREAD_CYCLES_VALU", "Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)", "gfx906"});
		return 85;
	}
	/**
	 * Basic Counter: SQ_LDS_BANK_CONFLICT
	 *
	 * Number of cycles LDS is stalled by bank conflicts. (emulated)
	 */
	if (strcmp(name, "SQ_LDS_BANK_CONFLICT")==0) {
		basic_counters.emplace(93, BasicCounter{93, "SQ", "SQ_LDS_BANK_CONFLICT", "Number of cycles LDS is stalled by bank conflicts. (emulated)", "gfx906"});
		return 93;
	}
	/**
	 * Basic Counter: TA_TA_BUSY
	 *
	 * TA block is busy. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_TA_BUSY")==0) {
		basic_counters.emplace(15, BasicCounter{15, "TA", "TA_TA_BUSY", "TA block is busy. Perf_Windowing not supported for this counter.", "gfx906"});
		return 15;
	}
	/**
	 * Basic Counter: TA_FLAT_READ_WAVEFRONTS
	 *
	 * Number of flat opcode reads processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_READ_WAVEFRONTS")==0) {
		basic_counters.emplace(101, BasicCounter{101, "TA", "TA_FLAT_READ_WAVEFRONTS", "Number of flat opcode reads processed by the TA.", "gfx906"});
		return 101;
	}
	/**
	 * Basic Counter: TA_FLAT_WRITE_WAVEFRONTS
	 *
	 * Number of flat opcode writes processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_WRITE_WAVEFRONTS")==0) {
		basic_counters.emplace(102, BasicCounter{102, "TA", "TA_FLAT_WRITE_WAVEFRONTS", "Number of flat opcode writes processed by the TA.", "gfx906"});
		return 102;
	}
	/**
	 * Basic Counter: TCC_HIT
	 *
	 * Number of cache hits.
	 */
	if (strcmp(name, "TCC_HIT")==0) {
		basic_counters.emplace(20, BasicCounter{20, "TCC", "TCC_HIT", "Number of cache hits.", "gfx906"});
		return 20;
	}
	/**
	 * Basic Counter: TCC_MISS
	 *
	 * Number of cache misses. UC reads count as misses.
	 */
	if (strcmp(name, "TCC_MISS")==0) {
		basic_counters.emplace(22, BasicCounter{22, "TCC", "TCC_MISS", "Number of cache misses. UC reads count as misses.", "gfx906"});
		return 22;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ
	 *
	 * Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.
	 */
	if (strcmp(name, "TCC_EA_WRREQ")==0) {
		basic_counters.emplace(29, BasicCounter{29, "TCC", "TCC_EA_WRREQ", "Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.", "gfx906"});
		return 29;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_64B
	 *
	 * Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_64B")==0) {
		basic_counters.emplace(30, BasicCounter{30, "TCC", "TCC_EA_WRREQ_64B", "Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.", "gfx906"});
		return 30;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_STALL
	 *
	 * Number of cycles a write request was stalled.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_STALL")==0) {
		basic_counters.emplace(33, BasicCounter{33, "TCC", "TCC_EA_WRREQ_STALL", "Number of cycles a write request was stalled.", "gfx906"});
		return 33;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ
	 *
	 * Number of TCC/EA read requests (either 32-byte or 64-byte)
	 */
	if (strcmp(name, "TCC_EA_RDREQ")==0) {
		basic_counters.emplace(41, BasicCounter{41, "TCC", "TCC_EA_RDREQ", "Number of TCC/EA read requests (either 32-byte or 64-byte)", "gfx906"});
		return 41;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_32B
	 *
	 * Number of 32-byte TCC/EA read requests
	 */
	if (strcmp(name, "TCC_EA_RDREQ_32B")==0) {
		basic_counters.emplace(42, BasicCounter{42, "TCC", "TCC_EA_RDREQ_32B", "Number of 32-byte TCC/EA read requests", "gfx906"});
		return 42;
	}
	/**
	 * Basic Counter: TCP_TCP_TA_DATA_STALL_CYCLES
	 *
	 * TCP stalls TA data interface. Now Windowed.
	 */
	if (strcmp(name, "TCP_TCP_TA_DATA_STALL_CYCLES")==0) {
		basic_counters.emplace(6, BasicCounter{6, "TCP", "TCP_TCP_TA_DATA_STALL_CYCLES", "TCP stalls TA data interface. Now Windowed.", "gfx906"});
		return 6;
	}
	/**
	 * Basic Counter: TCC_EA1_WRREQ
	 *
	 * Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.
	 */
	if (strcmp(name, "TCC_EA1_WRREQ")==0) {
		basic_counters.emplace(256, BasicCounter{256, "TCC", "TCC_EA1_WRREQ", "Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.", "gfx906"});
		return 256;
	}
	/**
	 * Basic Counter: TCC_EA1_WRREQ_64B
	 *
	 * Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.
	 */
	if (strcmp(name, "TCC_EA1_WRREQ_64B")==0) {
		basic_counters.emplace(257, BasicCounter{257, "TCC", "TCC_EA1_WRREQ_64B", "Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.", "gfx906"});
		return 257;
	}
	/**
	 * Basic Counter: TCC_EA1_WRREQ_STALL
	 *
	 * Number of cycles a write request was stalled.
	 */
	if (strcmp(name, "TCC_EA1_WRREQ_STALL")==0) {
		basic_counters.emplace(260, BasicCounter{260, "TCC", "TCC_EA1_WRREQ_STALL", "Number of cycles a write request was stalled.", "gfx906"});
		return 260;
	}
	/**
	 * Basic Counter: TCC_EA1_RDREQ
	 *
	 * Number of TCC/EA read requests (either 32-byte or 64-byte)
	 */
	if (strcmp(name, "TCC_EA1_RDREQ")==0) {
		basic_counters.emplace(267, BasicCounter{267, "TCC", "TCC_EA1_RDREQ", "Number of TCC/EA read requests (either 32-byte or 64-byte)", "gfx906"});
		return 267;
	}
	/**
	 * Basic Counter: TCC_EA1_RDREQ_32B
	 *
	 * Number of 32-byte TCC/EA read requests
	 */
	if (strcmp(name, "TCC_EA1_RDREQ_32B")==0) {
		basic_counters.emplace(268, BasicCounter{268, "TCC", "TCC_EA1_RDREQ_32B", "Number of 32-byte TCC/EA read requests", "gfx906"});
		return 268;
	}
	}

	/**
	 * @}
	 */

	/**
	 * @brief Basic gfx908 counters
	 *
	 * @{
	 */
	gpu = "gfx908";

	if (strncmp(gpu_name, gpu.c_str(), gpu.length())==0) {
	/**
	 * Basic Counter: GRBM_COUNT
	 *
	 * Tie High - Count Number of Clocks
	 */
	if (strcmp(name, "GRBM_COUNT")==0) {
		basic_counters.emplace(0, BasicCounter{0, "GRBM", "GRBM_COUNT", "Tie High - Count Number of Clocks", "gfx908"});
		return 0;
	}
	/**
	 * Basic Counter: GRBM_GUI_ACTIVE
	 *
	 * The GUI is Active
	 */
	if (strcmp(name, "GRBM_GUI_ACTIVE")==0) {
		basic_counters.emplace(2, BasicCounter{2, "GRBM", "GRBM_GUI_ACTIVE", "The GUI is Active", "gfx908"});
		return 2;
	}
	/**
	 * Basic Counter: SQ_WAVES
	 *
	 * Count number of waves sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES")==0) {
		basic_counters.emplace(4, BasicCounter{4, "SQ", "SQ_WAVES", "Count number of waves sent to SQs. (per-simd, emulated, global)", "gfx908"});
		return 4;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU
	 *
	 * Number of VALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU")==0) {
		basic_counters.emplace(26, BasicCounter{26, "SQ", "SQ_INSTS_VALU", "Number of VALU instructions issued. (per-simd, emulated)", "gfx908"});
		return 26;
	}
	/**
	 * Basic Counter: TA_TA_BUSY
	 *
	 * TA block is busy. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_TA_BUSY")==0) {
		basic_counters.emplace(15, BasicCounter{15, "TA", "TA_TA_BUSY", "TA block is busy. Perf_Windowing not supported for this counter.", "gfx908"});
		return 15;
	}
	/**
	 * Basic Counter: TA_FLAT_READ_WAVEFRONTS
	 *
	 * Number of flat opcode reads processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_READ_WAVEFRONTS")==0) {
		basic_counters.emplace(101, BasicCounter{101, "TA", "TA_FLAT_READ_WAVEFRONTS", "Number of flat opcode reads processed by the TA.", "gfx908"});
		return 101;
	}
	/**
	 * Basic Counter: TA_FLAT_WRITE_WAVEFRONTS
	 *
	 * Number of flat opcode writes processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_WRITE_WAVEFRONTS")==0) {
		basic_counters.emplace(102, BasicCounter{102, "TA", "TA_FLAT_WRITE_WAVEFRONTS", "Number of flat opcode writes processed by the TA.", "gfx908"});
		return 102;
	}
	/**
	 * Basic Counter: TCP_TCP_TA_DATA_STALL_CYCLES
	 *
	 * TCP stalls TA data interface. Now Windowed.
	 */
	if (strcmp(name, "TCP_TCP_TA_DATA_STALL_CYCLES")==0) {
		basic_counters.emplace(6, BasicCounter{6, "TCP", "TCP_TCP_TA_DATA_STALL_CYCLES", "TCP stalls TA data interface. Now Windowed.", "gfx908"});
		return 6;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_WR
	 *
	 * Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_WR")==0) {
		basic_counters.emplace(28, BasicCounter{28, "SQ", "SQ_INSTS_VMEM_WR", "Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)", "gfx908"});
		return 28;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_RD
	 *
	 * Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_RD")==0) {
		basic_counters.emplace(29, BasicCounter{29, "SQ", "SQ_INSTS_VMEM_RD", "Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)", "gfx908"});
		return 29;
	}
	/**
	 * Basic Counter: SQ_INSTS_SALU
	 *
	 * Number of SALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SALU")==0) {
		basic_counters.emplace(31, BasicCounter{31, "SQ", "SQ_INSTS_SALU", "Number of SALU instructions issued. (per-simd, emulated)", "gfx908"});
		return 31;
	}
	/**
	 * Basic Counter: SQ_INSTS_SMEM
	 *
	 * Number of SMEM instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SMEM")==0) {
		basic_counters.emplace(32, BasicCounter{32, "SQ", "SQ_INSTS_SMEM", "Number of SMEM instructions issued. (per-simd, emulated)", "gfx908"});
		return 32;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT
	 *
	 * Number of FLAT instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT")==0) {
		basic_counters.emplace(33, BasicCounter{33, "SQ", "SQ_INSTS_FLAT", "Number of FLAT instructions issued. (per-simd, emulated)", "gfx908"});
		return 33;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT_LDS_ONLY
	 *
	 * Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT_LDS_ONLY")==0) {
		basic_counters.emplace(34, BasicCounter{34, "SQ", "SQ_INSTS_FLAT_LDS_ONLY", "Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)", "gfx908"});
		return 34;
	}
	/**
	 * Basic Counter: SQ_INSTS_LDS
	 *
	 * Number of LDS instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_LDS")==0) {
		basic_counters.emplace(35, BasicCounter{35, "SQ", "SQ_INSTS_LDS", "Number of LDS instructions issued (including FLAT). (per-simd, emulated)", "gfx908"});
		return 35;
	}
	/**
	 * Basic Counter: SQ_INSTS_GDS
	 *
	 * Number of GDS instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_GDS")==0) {
		basic_counters.emplace(36, BasicCounter{36, "SQ", "SQ_INSTS_GDS", "Number of GDS instructions issued. (per-simd, emulated)", "gfx908"});
		return 36;
	}
	/**
	 * Basic Counter: SQ_WAIT_INST_LDS
	 *
	 * Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)
	 */
	if (strcmp(name, "SQ_WAIT_INST_LDS")==0) {
		basic_counters.emplace(64, BasicCounter{64, "SQ", "SQ_WAIT_INST_LDS", "Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)", "gfx908"});
		return 64;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_VALU
	 *
	 * regspec 71? Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, nondeterministic). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_VALU")==0) {
		basic_counters.emplace(72, BasicCounter{72, "SQ", "SQ_ACTIVE_INST_VALU", "regspec 71? Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, nondeterministic). Units in quad-cycles(4 cycles)", "gfx908"});
		return 72;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_SALU
	 *
	 * Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_SALU")==0) {
		basic_counters.emplace(85, BasicCounter{85, "SQ", "SQ_INST_CYCLES_SALU", "Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated)", "gfx908"});
		return 85;
	}
	/**
	 * Basic Counter: SQ_THREAD_CYCLES_VALU
	 *
	 * Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)
	 */
	if (strcmp(name, "SQ_THREAD_CYCLES_VALU")==0) {
		basic_counters.emplace(86, BasicCounter{86, "SQ", "SQ_THREAD_CYCLES_VALU", "Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)", "gfx908"});
		return 86;
	}
	/**
	 * Basic Counter: SQ_LDS_BANK_CONFLICT
	 *
	 * Number of cycles LDS is stalled by bank conflicts. (emulated)
	 */
	if (strcmp(name, "SQ_LDS_BANK_CONFLICT")==0) {
		basic_counters.emplace(94, BasicCounter{94, "SQ", "SQ_LDS_BANK_CONFLICT", "Number of cycles LDS is stalled by bank conflicts. (emulated)", "gfx908"});
		return 94;
	}
	/**
	 * Basic Counter: TCC_HIT
	 *
	 * Number of cache hits.
	 */
	if (strcmp(name, "TCC_HIT")==0) {
		basic_counters.emplace(17, BasicCounter{17, "TCC", "TCC_HIT", "Number of cache hits.", "gfx908"});
		return 17;
	}
	/**
	 * Basic Counter: TCC_MISS
	 *
	 * Number of cache misses. UC reads count as misses.
	 */
	if (strcmp(name, "TCC_MISS")==0) {
		basic_counters.emplace(19, BasicCounter{19, "TCC", "TCC_MISS", "Number of cache misses. UC reads count as misses.", "gfx908"});
		return 19;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ
	 *
	 * Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.
	 */
	if (strcmp(name, "TCC_EA_WRREQ")==0) {
		basic_counters.emplace(26, BasicCounter{26, "TCC", "TCC_EA_WRREQ", "Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.", "gfx908"});
		return 26;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_64B
	 *
	 * Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_64B")==0) {
		basic_counters.emplace(27, BasicCounter{27, "TCC", "TCC_EA_WRREQ_64B", "Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.", "gfx908"});
		return 27;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_STALL
	 *
	 * Number of cycles a write request was stalled.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_STALL")==0) {
		basic_counters.emplace(30, BasicCounter{30, "TCC", "TCC_EA_WRREQ_STALL", "Number of cycles a write request was stalled.", "gfx908"});
		return 30;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ
	 *
	 * Number of TCC/EA read requests (either 32-byte or 64-byte)
	 */
	if (strcmp(name, "TCC_EA_RDREQ")==0) {
		basic_counters.emplace(38, BasicCounter{38, "TCC", "TCC_EA_RDREQ", "Number of TCC/EA read requests (either 32-byte or 64-byte)", "gfx908"});
		return 38;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_32B
	 *
	 * Number of 32-byte TCC/EA read requests
	 */
	if (strcmp(name, "TCC_EA_RDREQ_32B")==0) {
		basic_counters.emplace(39, BasicCounter{39, "TCC", "TCC_EA_RDREQ_32B", "Number of 32-byte TCC/EA read requests", "gfx908"});
		return 39;
	}
	}

	/**
	 * @}
	 */

	/**
	 * @brief Basic gfx90a counters
	 *
	 * @{
	 */
	gpu = "gfx90a";

	if (strncmp(gpu_name, gpu.c_str(), gpu.length())==0) {
	/**
	 * Basic Counter: SQ_WAIT_INST_LDS
	 *
	 * Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)
	 */
	if (strcmp(name, "SQ_WAIT_INST_LDS")==0) {
		basic_counters.emplace(91, BasicCounter{91, "SQ", "SQ_WAIT_INST_LDS", "Number of wave-cycles spent waiting for LDS instruction issue. In units of 4 cycles. (per-simd, nondeterministic)", "gfx90a"});
		return 91;
	}
	/**
	 * Basic Counter: TCP_TCP_TA_DATA_STALL_CYCLES
	 *
	 * TCP stalls TA data interface. Now Windowed.
	 */
	if (strcmp(name, "TCP_TCP_TA_DATA_STALL_CYCLES")==0) {
		basic_counters.emplace(6, BasicCounter{6, "TCP", "TCP_TCP_TA_DATA_STALL_CYCLES", "TCP stalls TA data interface. Now Windowed.", "gfx90a"});
		return 6;
	}
	/**
	 * Basic Counter: GRBM_COUNT
	 *
	 * Tie High - Count Number of Clocks
	 */
	if (strcmp(name, "GRBM_COUNT")==0) {
		basic_counters.emplace(0, BasicCounter{0, "GRBM", "GRBM_COUNT", "Tie High - Count Number of Clocks", "gfx90a"});
		return 0;
	}
	/**
	 * Basic Counter: GRBM_GUI_ACTIVE
	 *
	 * The GUI is Active
	 */
	if (strcmp(name, "GRBM_GUI_ACTIVE")==0) {
		basic_counters.emplace(2, BasicCounter{2, "GRBM", "GRBM_GUI_ACTIVE", "The GUI is Active", "gfx90a"});
		return 2;
	}
	/**
	 * Basic Counter: GRBM_CP_BUSY
	 *
	 * Any of the Command Processor (CPG/CPC/CPF) blocks are busy.
	 */
	if (strcmp(name, "GRBM_CP_BUSY")==0) {
		basic_counters.emplace(3, BasicCounter{3, "GRBM", "GRBM_CP_BUSY", "Any of the Command Processor (CPG/CPC/CPF) blocks are busy.", "gfx90a"});
		return 3;
	}
	/**
	 * Basic Counter: GRBM_SPI_BUSY
	 *
	 * Any of the Shader Pipe Interpolators (SPI) are busy in the shader engine(s).
	 */
	if (strcmp(name, "GRBM_SPI_BUSY")==0) {
		basic_counters.emplace(11, BasicCounter{11, "GRBM", "GRBM_SPI_BUSY", "Any of the Shader Pipe Interpolators (SPI) are busy in the shader engine(s).", "gfx90a"});
		return 11;
	}
	/**
	 * Basic Counter: GRBM_TA_BUSY
	 *
	 * Any of the Texture Pipes (TA) are busy in the shader engine(s).
	 */
	if (strcmp(name, "GRBM_TA_BUSY")==0) {
		basic_counters.emplace(13, BasicCounter{13, "GRBM", "GRBM_TA_BUSY", "Any of the Texture Pipes (TA) are busy in the shader engine(s).", "gfx90a"});
		return 13;
	}
	/**
	 * Basic Counter: GRBM_TC_BUSY
	 *
	 * Any of the Texture Cache Blocks (TCP/TCI/TCA/TCC) are busy.
	 */
	if (strcmp(name, "GRBM_TC_BUSY")==0) {
		basic_counters.emplace(28, BasicCounter{28, "GRBM", "GRBM_TC_BUSY", "Any of the Texture Cache Blocks (TCP/TCI/TCA/TCC) are busy.", "gfx90a"});
		return 28;
	}
	/**
	 * Basic Counter: GRBM_CPC_BUSY
	 *
	 * The Command Processor Compute (CPC) is busy.
	 */
	if (strcmp(name, "GRBM_CPC_BUSY")==0) {
		basic_counters.emplace(30, BasicCounter{30, "GRBM", "GRBM_CPC_BUSY", "The Command Processor Compute (CPC) is busy.", "gfx90a"});
		return 30;
	}
	/**
	 * Basic Counter: GRBM_CPF_BUSY
	 *
	 * The Command Processor Fetchers (CPF) is busy.
	 */
	if (strcmp(name, "GRBM_CPF_BUSY")==0) {
		basic_counters.emplace(31, BasicCounter{31, "GRBM", "GRBM_CPF_BUSY", "The Command Processor Fetchers (CPF) is busy.", "gfx90a"});
		return 31;
	}
	/**
	 * Basic Counter: GRBM_UTCL2_BUSY
	 *
	 * The Unified Translation Cache Level-2 (UTCL2) block is busy.
	 */
	if (strcmp(name, "GRBM_UTCL2_BUSY")==0) {
		basic_counters.emplace(34, BasicCounter{34, "GRBM", "GRBM_UTCL2_BUSY", "The Unified Translation Cache Level-2 (UTCL2) block is busy.", "gfx90a"});
		return 34;
	}
	/**
	 * Basic Counter: GRBM_EA_BUSY
	 *
	 * The Efficiency Arbiter (EA) block is busy.
	 */
	if (strcmp(name, "GRBM_EA_BUSY")==0) {
		basic_counters.emplace(35, BasicCounter{35, "GRBM", "GRBM_EA_BUSY", "The Efficiency Arbiter (EA) block is busy.", "gfx90a"});
		return 35;
	}
	/**
	 * Basic Counter: CPC_ME1_BUSY_FOR_PACKET_DECODE
	 *
	 * Me1 busy for packet decode.
	 */
	if (strcmp(name, "CPC_ME1_BUSY_FOR_PACKET_DECODE")==0) {
		basic_counters.emplace(13, BasicCounter{13, "CPC", "CPC_ME1_BUSY_FOR_PACKET_DECODE", "Me1 busy for packet decode.", "gfx90a"});
		return 13;
	}
	/**
	 * Basic Counter: CPC_UTCL1_STALL_ON_TRANSLATION
	 *
	 * One of the UTCL1s is stalled waiting on translation, XNACK or PENDING response.
	 */
	if (strcmp(name, "CPC_UTCL1_STALL_ON_TRANSLATION")==0) {
		basic_counters.emplace(24, BasicCounter{24, "CPC", "CPC_UTCL1_STALL_ON_TRANSLATION", "One of the UTCL1s is stalled waiting on translation, XNACK or PENDING response.", "gfx90a"});
		return 24;
	}
	/**
	 * Basic Counter: CPC_CPC_STAT_BUSY
	 *
	 * CPC Busy.
	 */
	if (strcmp(name, "CPC_CPC_STAT_BUSY")==0) {
		basic_counters.emplace(25, BasicCounter{25, "CPC", "CPC_CPC_STAT_BUSY", "CPC Busy.", "gfx90a"});
		return 25;
	}
	/**
	 * Basic Counter: CPC_CPC_STAT_IDLE
	 *
	 * CPC Idle.
	 */
	if (strcmp(name, "CPC_CPC_STAT_IDLE")==0) {
		basic_counters.emplace(26, BasicCounter{26, "CPC", "CPC_CPC_STAT_IDLE", "CPC Idle.", "gfx90a"});
		return 26;
	}
	/**
	 * Basic Counter: CPC_CPC_STAT_STALL
	 *
	 * CPC Stalled.
	 */
	if (strcmp(name, "CPC_CPC_STAT_STALL")==0) {
		basic_counters.emplace(27, BasicCounter{27, "CPC", "CPC_CPC_STAT_STALL", "CPC Stalled.", "gfx90a"});
		return 27;
	}
	/**
	 * Basic Counter: CPC_CPC_TCIU_BUSY
	 *
	 * CPC TCIU interface Busy.
	 */
	if (strcmp(name, "CPC_CPC_TCIU_BUSY")==0) {
		basic_counters.emplace(28, BasicCounter{28, "CPC", "CPC_CPC_TCIU_BUSY", "CPC TCIU interface Busy.", "gfx90a"});
		return 28;
	}
	/**
	 * Basic Counter: CPC_CPC_TCIU_IDLE
	 *
	 * CPC TCIU interface Idle.
	 */
	if (strcmp(name, "CPC_CPC_TCIU_IDLE")==0) {
		basic_counters.emplace(29, BasicCounter{29, "CPC", "CPC_CPC_TCIU_IDLE", "CPC TCIU interface Idle.", "gfx90a"});
		return 29;
	}
	/**
	 * Basic Counter: CPC_CPC_UTCL2IU_BUSY
	 *
	 * CPC UTCL2 interface Busy.
	 */
	if (strcmp(name, "CPC_CPC_UTCL2IU_BUSY")==0) {
		basic_counters.emplace(30, BasicCounter{30, "CPC", "CPC_CPC_UTCL2IU_BUSY", "CPC UTCL2 interface Busy.", "gfx90a"});
		return 30;
	}
	/**
	 * Basic Counter: CPC_CPC_UTCL2IU_IDLE
	 *
	 * CPC UTCL2 interface Idle.
	 */
	if (strcmp(name, "CPC_CPC_UTCL2IU_IDLE")==0) {
		basic_counters.emplace(31, BasicCounter{31, "CPC", "CPC_CPC_UTCL2IU_IDLE", "CPC UTCL2 interface Idle.", "gfx90a"});
		return 31;
	}
	/**
	 * Basic Counter: CPC_CPC_UTCL2IU_STALL
	 *
	 * CPC UTCL2 interface Stalled waiting on Free, Tags or Translation.
	 */
	if (strcmp(name, "CPC_CPC_UTCL2IU_STALL")==0) {
		basic_counters.emplace(32, BasicCounter{32, "CPC", "CPC_CPC_UTCL2IU_STALL", "CPC UTCL2 interface Stalled waiting on Free, Tags or Translation.", "gfx90a"});
		return 32;
	}
	/**
	 * Basic Counter: CPC_ME1_DC0_SPI_BUSY
	 *
	 * CPC Me1 Processor Busy.
	 */
	if (strcmp(name, "CPC_ME1_DC0_SPI_BUSY")==0) {
		basic_counters.emplace(33, BasicCounter{33, "CPC", "CPC_ME1_DC0_SPI_BUSY", "CPC Me1 Processor Busy.", "gfx90a"});
		return 33;
	}
	/**
	 * Basic Counter: CPF_CMP_UTCL1_STALL_ON_TRANSLATION
	 *
	 * One of the Compute UTCL1s is stalled waiting on translation, XNACK or PENDING response.
	 */
	if (strcmp(name, "CPF_CMP_UTCL1_STALL_ON_TRANSLATION")==0) {
		basic_counters.emplace(20, BasicCounter{20, "CPF", "CPF_CMP_UTCL1_STALL_ON_TRANSLATION", "One of the Compute UTCL1s is stalled waiting on translation, XNACK or PENDING response.", "gfx90a"});
		return 20;
	}
	/**
	 * Basic Counter: CPF_CPF_STAT_BUSY
	 *
	 * CPF Busy.
	 */
	if (strcmp(name, "CPF_CPF_STAT_BUSY")==0) {
		basic_counters.emplace(23, BasicCounter{23, "CPF", "CPF_CPF_STAT_BUSY", "CPF Busy.", "gfx90a"});
		return 23;
	}
	/**
	 * Basic Counter: CPF_CPF_STAT_IDLE
	 *
	 * CPF Idle.
	 */
	if (strcmp(name, "CPF_CPF_STAT_IDLE")==0) {
		basic_counters.emplace(24, BasicCounter{24, "CPF", "CPF_CPF_STAT_IDLE", "CPF Idle.", "gfx90a"});
		return 24;
	}
	/**
	 * Basic Counter: CPF_CPF_STAT_STALL
	 *
	 * CPF Stalled.
	 */
	if (strcmp(name, "CPF_CPF_STAT_STALL")==0) {
		basic_counters.emplace(25, BasicCounter{25, "CPF", "CPF_CPF_STAT_STALL", "CPF Stalled.", "gfx90a"});
		return 25;
	}
	/**
	 * Basic Counter: CPF_CPF_TCIU_BUSY
	 *
	 * CPF TCIU interface Busy.
	 */
	if (strcmp(name, "CPF_CPF_TCIU_BUSY")==0) {
		basic_counters.emplace(26, BasicCounter{26, "CPF", "CPF_CPF_TCIU_BUSY", "CPF TCIU interface Busy.", "gfx90a"});
		return 26;
	}
	/**
	 * Basic Counter: CPF_CPF_TCIU_IDLE
	 *
	 * CPF TCIU interface Idle.
	 */
	if (strcmp(name, "CPF_CPF_TCIU_IDLE")==0) {
		basic_counters.emplace(27, BasicCounter{27, "CPF", "CPF_CPF_TCIU_IDLE", "CPF TCIU interface Idle.", "gfx90a"});
		return 27;
	}
	/**
	 * Basic Counter: CPF_CPF_TCIU_STALL
	 *
	 * CPF TCIU interface Stalled waiting on Free, Tags.
	 */
	if (strcmp(name, "CPF_CPF_TCIU_STALL")==0) {
		basic_counters.emplace(28, BasicCounter{28, "CPF", "CPF_CPF_TCIU_STALL", "CPF TCIU interface Stalled waiting on Free, Tags.", "gfx90a"});
		return 28;
	}
	/**
	 * Basic Counter: SPI_CSN_WINDOW_VALID
	 *
	 * Clock count enabled by perfcounter_start event. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_WINDOW_VALID")==0) {
		basic_counters.emplace(47, BasicCounter{47, "SPI", "SPI_CSN_WINDOW_VALID", "Clock count enabled by perfcounter_start event. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx90a"});
		return 47;
	}
	/**
	 * Basic Counter: SPI_CSN_BUSY
	 *
	 * Number of clocks with outstanding waves (SPI or SH). Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_BUSY")==0) {
		basic_counters.emplace(48, BasicCounter{48, "SPI", "SPI_CSN_BUSY", "Number of clocks with outstanding waves (SPI or SH). Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx90a"});
		return 48;
	}
	/**
	 * Basic Counter: SPI_CSN_NUM_THREADGROUPS
	 *
	 * Number of threadgroups launched. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_NUM_THREADGROUPS")==0) {
		basic_counters.emplace(49, BasicCounter{49, "SPI", "SPI_CSN_NUM_THREADGROUPS", "Number of threadgroups launched. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx90a"});
		return 49;
	}
	/**
	 * Basic Counter: SPI_CSN_WAVE
	 *
	 * Number of waves. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_WAVE")==0) {
		basic_counters.emplace(52, BasicCounter{52, "SPI", "SPI_CSN_WAVE", "Number of waves. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx90a"});
		return 52;
	}
	/**
	 * Basic Counter: SPI_RA_REQ_NO_ALLOC
	 *
	 * Arb cycles with requests but no allocation. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_REQ_NO_ALLOC")==0) {
		basic_counters.emplace(79, BasicCounter{79, "SPI", "SPI_RA_REQ_NO_ALLOC", "Arb cycles with requests but no allocation. Source is RA0", "gfx90a"});
		return 79;
	}
	/**
	 * Basic Counter: SPI_RA_REQ_NO_ALLOC_CSN
	 *
	 * Arb cycles with CSn req and no CSn alloc. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_REQ_NO_ALLOC_CSN")==0) {
		basic_counters.emplace(85, BasicCounter{85, "SPI", "SPI_RA_REQ_NO_ALLOC_CSN", "Arb cycles with CSn req and no CSn alloc. Source is RA0", "gfx90a"});
		return 85;
	}
	/**
	 * Basic Counter: SPI_RA_RES_STALL_CSN
	 *
	 * Arb cycles with CSn req and no CSn fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_RES_STALL_CSN")==0) {
		basic_counters.emplace(91, BasicCounter{91, "SPI", "SPI_RA_RES_STALL_CSN", "Arb cycles with CSn req and no CSn fits. Source is RA0", "gfx90a"});
		return 91;
	}
	/**
	 * Basic Counter: SPI_RA_TMP_STALL_CSN
	 *
	 * Cycles where csn wants to req but does not fit in temp space.
	 */
	if (strcmp(name, "SPI_RA_TMP_STALL_CSN")==0) {
		basic_counters.emplace(97, BasicCounter{97, "SPI", "SPI_RA_TMP_STALL_CSN", "Cycles where csn wants to req but does not fit in temp space.", "gfx90a"});
		return 97;
	}
	/**
	 * Basic Counter: SPI_RA_WAVE_SIMD_FULL_CSN
	 *
	 * Sum of SIMD where WAVE can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_WAVE_SIMD_FULL_CSN")==0) {
		basic_counters.emplace(103, BasicCounter{103, "SPI", "SPI_RA_WAVE_SIMD_FULL_CSN", "Sum of SIMD where WAVE can't take csn wave when !fits. Source is RA0", "gfx90a"});
		return 103;
	}
	/**
	 * Basic Counter: SPI_RA_VGPR_SIMD_FULL_CSN
	 *
	 * Sum of SIMD where VGPR can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_VGPR_SIMD_FULL_CSN")==0) {
		basic_counters.emplace(109, BasicCounter{109, "SPI", "SPI_RA_VGPR_SIMD_FULL_CSN", "Sum of SIMD where VGPR can't take csn wave when !fits. Source is RA0", "gfx90a"});
		return 109;
	}
	/**
	 * Basic Counter: SPI_RA_SGPR_SIMD_FULL_CSN
	 *
	 * Sum of SIMD where SGPR can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_SGPR_SIMD_FULL_CSN")==0) {
		basic_counters.emplace(115, BasicCounter{115, "SPI", "SPI_RA_SGPR_SIMD_FULL_CSN", "Sum of SIMD where SGPR can't take csn wave when !fits. Source is RA0", "gfx90a"});
		return 115;
	}
	/**
	 * Basic Counter: SPI_RA_LDS_CU_FULL_CSN
	 *
	 * Sum of CU where LDS can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_LDS_CU_FULL_CSN")==0) {
		basic_counters.emplace(120, BasicCounter{120, "SPI", "SPI_RA_LDS_CU_FULL_CSN", "Sum of CU where LDS can't take csn wave when !fits. Source is RA0", "gfx90a"});
		return 120;
	}
	/**
	 * Basic Counter: SPI_RA_BAR_CU_FULL_CSN
	 *
	 * Sum of CU where BARRIER can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_BAR_CU_FULL_CSN")==0) {
		basic_counters.emplace(123, BasicCounter{123, "SPI", "SPI_RA_BAR_CU_FULL_CSN", "Sum of CU where BARRIER can't take csn wave when !fits. Source is RA0", "gfx90a"});
		return 123;
	}
	/**
	 * Basic Counter: SPI_RA_BULKY_CU_FULL_CSN
	 *
	 * Sum of CU where BULKY can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_BULKY_CU_FULL_CSN")==0) {
		basic_counters.emplace(125, BasicCounter{125, "SPI", "SPI_RA_BULKY_CU_FULL_CSN", "Sum of CU where BULKY can't take csn wave when !fits. Source is RA0", "gfx90a"});
		return 125;
	}
	/**
	 * Basic Counter: SPI_RA_TGLIM_CU_FULL_CSN
	 *
	 * Cycles where csn wants to req but all CU are at tg_limit
	 */
	if (strcmp(name, "SPI_RA_TGLIM_CU_FULL_CSN")==0) {
		basic_counters.emplace(127, BasicCounter{127, "SPI", "SPI_RA_TGLIM_CU_FULL_CSN", "Cycles where csn wants to req but all CU are at tg_limit", "gfx90a"});
		return 127;
	}
	/**
	 * Basic Counter: SPI_RA_WVLIM_STALL_CSN
	 *
	 * Number of clocks csn is stalled due to WAVE LIMIT.
	 */
	if (strcmp(name, "SPI_RA_WVLIM_STALL_CSN")==0) {
		basic_counters.emplace(133, BasicCounter{133, "SPI", "SPI_RA_WVLIM_STALL_CSN", "Number of clocks csn is stalled due to WAVE LIMIT.", "gfx90a"});
		return 133;
	}
	/**
	 * Basic Counter: SPI_SWC_CSC_WR
	 *
	 * Number of clocks to write CSC waves to SGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_SWC_CSC_WR")==0) {
		basic_counters.emplace(189, BasicCounter{189, "SPI", "SPI_SWC_CSC_WR", "Number of clocks to write CSC waves to SGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx90a"});
		return 189;
	}
	/**
	 * Basic Counter: SPI_VWC_CSC_WR
	 *
	 * Number of clocks to write CSC waves to VGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_VWC_CSC_WR")==0) {
		basic_counters.emplace(195, BasicCounter{195, "SPI", "SPI_VWC_CSC_WR", "Number of clocks to write CSC waves to VGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx90a"});
		return 195;
	}
	/**
	 * Basic Counter: SQ_ACCUM_PREV
	 *
	 * For counter N, increment by the value of counter N-1. Only accumulates once every 4 cycles.
	 */
	if (strcmp(name, "SQ_ACCUM_PREV")==0) {
		basic_counters.emplace(1, BasicCounter{1, "SQ", "SQ_ACCUM_PREV", "For counter N, increment by the value of counter N-1. Only accumulates once every 4 cycles.", "gfx90a"});
		return 1;
	}
	/**
	 * Basic Counter: SQ_CYCLES
	 *
	 * Clock cycles. (nondeterministic, per-simd, global)
	 */
	if (strcmp(name, "SQ_CYCLES")==0) {
		basic_counters.emplace(2, BasicCounter{2, "SQ", "SQ_CYCLES", "Clock cycles. (nondeterministic, per-simd, global)", "gfx90a"});
		return 2;
	}
	/**
	 * Basic Counter: SQ_BUSY_CYCLES
	 *
	 * Clock cycles while SQ is reporting that it is busy. (nondeterministic, per-simd, global)
	 */
	if (strcmp(name, "SQ_BUSY_CYCLES")==0) {
		basic_counters.emplace(3, BasicCounter{3, "SQ", "SQ_BUSY_CYCLES", "Clock cycles while SQ is reporting that it is busy. (nondeterministic, per-simd, global)", "gfx90a"});
		return 3;
	}
	/**
	 * Basic Counter: SQ_WAVES
	 *
	 * Count number of waves sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES")==0) {
		basic_counters.emplace(4, BasicCounter{4, "SQ", "SQ_WAVES", "Count number of waves sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 4;
	}
	/**
	 * Basic Counter: SQ_LEVEL_WAVES
	 *
	 * Track the number of waves. Set ACCUM_PREV for the next counter to use this. (level, per-simd, global)
	 */
	if (strcmp(name, "SQ_LEVEL_WAVES")==0) {
		basic_counters.emplace(5, BasicCounter{5, "SQ", "SQ_LEVEL_WAVES", "Track the number of waves. Set ACCUM_PREV for the next counter to use this. (level, per-simd, global)", "gfx90a"});
		return 5;
	}
	/**
	 * Basic Counter: SQ_WAVES_EQ_64
	 *
	 * Count number of waves with exactly 64 active threads sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_EQ_64")==0) {
		basic_counters.emplace(6, BasicCounter{6, "SQ", "SQ_WAVES_EQ_64", "Count number of waves with exactly 64 active threads sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 6;
	}
	/**
	 * Basic Counter: SQ_WAVES_LT_64
	 *
	 * Count number of waves with smaller than 64 active threads sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_LT_64")==0) {
		basic_counters.emplace(7, BasicCounter{7, "SQ", "SQ_WAVES_LT_64", "Count number of waves with smaller than 64 active threads sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 7;
	}
	/**
	 * Basic Counter: SQ_WAVES_LT_48
	 *
	 * Count number of waves with smaller than 48 active threads sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_LT_48")==0) {
		basic_counters.emplace(8, BasicCounter{8, "SQ", "SQ_WAVES_LT_48", "Count number of waves with smaller than 48 active threads sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 8;
	}
	/**
	 * Basic Counter: SQ_WAVES_LT_32
	 *
	 * Count number of waves sent smaller than 32 active threads sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_LT_32")==0) {
		basic_counters.emplace(9, BasicCounter{9, "SQ", "SQ_WAVES_LT_32", "Count number of waves sent smaller than 32 active threads sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 9;
	}
	/**
	 * Basic Counter: SQ_WAVES_LT_16
	 *
	 * Count number of waves sent smaller than 16 active threads sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_LT_16")==0) {
		basic_counters.emplace(10, BasicCounter{10, "SQ", "SQ_WAVES_LT_16", "Count number of waves sent smaller than 16 active threads sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 10;
	}
	/**
	 * Basic Counter: SQ_BUSY_CU_CYCLES
	 *
	 * Count quad-cycles each CU is busy. (nondeterministic, per-simd)
	 */
	if (strcmp(name, "SQ_BUSY_CU_CYCLES")==0) {
		basic_counters.emplace(13, BasicCounter{13, "SQ", "SQ_BUSY_CU_CYCLES", "Count quad-cycles each CU is busy. (nondeterministic, per-simd)", "gfx90a"});
		return 13;
	}
	/**
	 * Basic Counter: SQ_ITEMS
	 *
	 * Number of valid items per wave. (per-simd, global)
	 */
	if (strcmp(name, "SQ_ITEMS")==0) {
		basic_counters.emplace(14, BasicCounter{14, "SQ", "SQ_ITEMS", "Number of valid items per wave. (per-simd, global)", "gfx90a"});
		return 14;
	}
	/**
	 * Basic Counter: SQ_INSTS
	 *
	 * Number of instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS")==0) {
		basic_counters.emplace(25, BasicCounter{25, "SQ", "SQ_INSTS", "Number of instructions issued. (per-simd, emulated)", "gfx90a"});
		return 25;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU
	 *
	 * Number of VALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU")==0) {
		basic_counters.emplace(26, BasicCounter{26, "SQ", "SQ_INSTS_VALU", "Number of VALU instructions issued. (per-simd, emulated)", "gfx90a"});
		return 26;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_ADD_F16
	 *
	 * Number of VALU ADD/SUB instructions on float16. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_ADD_F16")==0) {
		basic_counters.emplace(27, BasicCounter{27, "SQ", "SQ_INSTS_VALU_ADD_F16", "Number of VALU ADD/SUB instructions on float16. (per-simd, emulated)", "gfx90a"});
		return 27;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MUL_F16
	 *
	 * Number of VALU MUL instructions on float16. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MUL_F16")==0) {
		basic_counters.emplace(28, BasicCounter{28, "SQ", "SQ_INSTS_VALU_MUL_F16", "Number of VALU MUL instructions on float16. (per-simd, emulated)", "gfx90a"});
		return 28;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_FMA_F16
	 *
	 * Number of VALU FMA/MAD instructions on float16. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_FMA_F16")==0) {
		basic_counters.emplace(29, BasicCounter{29, "SQ", "SQ_INSTS_VALU_FMA_F16", "Number of VALU FMA/MAD instructions on float16. (per-simd, emulated)", "gfx90a"});
		return 29;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_TRANS_F16
	 *
	 * Number of VALU transcendental instructions on float16. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_TRANS_F16")==0) {
		basic_counters.emplace(30, BasicCounter{30, "SQ", "SQ_INSTS_VALU_TRANS_F16", "Number of VALU transcendental instructions on float16. (per-simd, emulated)", "gfx90a"});
		return 30;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_ADD_F32
	 *
	 * Number of VALU ADD/SUB instructions on float32. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_ADD_F32")==0) {
		basic_counters.emplace(31, BasicCounter{31, "SQ", "SQ_INSTS_VALU_ADD_F32", "Number of VALU ADD/SUB instructions on float32. (per-simd, emulated)", "gfx90a"});
		return 31;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MUL_F32
	 *
	 * Number of VALU MUL instructions on float32. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MUL_F32")==0) {
		basic_counters.emplace(32, BasicCounter{32, "SQ", "SQ_INSTS_VALU_MUL_F32", "Number of VALU MUL instructions on float32. (per-simd, emulated)", "gfx90a"});
		return 32;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_FMA_F32
	 *
	 * Number of VALU FMA/MAD instructions on float32. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_FMA_F32")==0) {
		basic_counters.emplace(33, BasicCounter{33, "SQ", "SQ_INSTS_VALU_FMA_F32", "Number of VALU FMA/MAD instructions on float32. (per-simd, emulated)", "gfx90a"});
		return 33;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_TRANS_F32
	 *
	 * Number of VALU transcendental instructions on float32. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_TRANS_F32")==0) {
		basic_counters.emplace(34, BasicCounter{34, "SQ", "SQ_INSTS_VALU_TRANS_F32", "Number of VALU transcendental instructions on float32. (per-simd, emulated)", "gfx90a"});
		return 34;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_ADD_F64
	 *
	 * Number of VALU ADD/SUB instructions on float64. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_ADD_F64")==0) {
		basic_counters.emplace(35, BasicCounter{35, "SQ", "SQ_INSTS_VALU_ADD_F64", "Number of VALU ADD/SUB instructions on float64. (per-simd, emulated)", "gfx90a"});
		return 35;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MUL_F64
	 *
	 * Number of VALU MUL instructions on float64. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MUL_F64")==0) {
		basic_counters.emplace(36, BasicCounter{36, "SQ", "SQ_INSTS_VALU_MUL_F64", "Number of VALU MUL instructions on float64. (per-simd, emulated)", "gfx90a"});
		return 36;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_FMA_F64
	 *
	 * Number of VALU FMA/MAD instructions on float64. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_FMA_F64")==0) {
		basic_counters.emplace(37, BasicCounter{37, "SQ", "SQ_INSTS_VALU_FMA_F64", "Number of VALU FMA/MAD instructions on float64. (per-simd, emulated)", "gfx90a"});
		return 37;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_TRANS_F64
	 *
	 * Number of VALU transcendental instructions on float64. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_TRANS_F64")==0) {
		basic_counters.emplace(38, BasicCounter{38, "SQ", "SQ_INSTS_VALU_TRANS_F64", "Number of VALU transcendental instructions on float64. (per-simd, emulated)", "gfx90a"});
		return 38;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_INT32
	 *
	 * Number of VALU 32-bit integer (signed or unsigned) instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_INT32")==0) {
		basic_counters.emplace(39, BasicCounter{39, "SQ", "SQ_INSTS_VALU_INT32", "Number of VALU 32-bit integer (signed or unsigned) instructions. (per-simd, emulated)", "gfx90a"});
		return 39;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_INT64
	 *
	 * Number of VALU 64-bit integer (signed or unsigned) instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_INT64")==0) {
		basic_counters.emplace(40, BasicCounter{40, "SQ", "SQ_INSTS_VALU_INT64", "Number of VALU 64-bit integer (signed or unsigned) instructions. (per-simd, emulated)", "gfx90a"});
		return 40;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_CVT
	 *
	 * Number of VALU data conversion instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_CVT")==0) {
		basic_counters.emplace(41, BasicCounter{41, "SQ", "SQ_INSTS_VALU_CVT", "Number of VALU data conversion instructions. (per-simd, emulated)", "gfx90a"});
		return 41;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_I8
	 *
	 * Number of VALU V_MFMA_*_I8 instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_I8")==0) {
		basic_counters.emplace(42, BasicCounter{42, "SQ", "SQ_INSTS_VALU_MFMA_I8", "Number of VALU V_MFMA_*_I8 instructions. (per-simd, emulated)", "gfx90a"});
		return 42;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_F16
	 *
	 * Number of VALU V_MFMA_*_F16 instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_F16")==0) {
		basic_counters.emplace(43, BasicCounter{43, "SQ", "SQ_INSTS_VALU_MFMA_F16", "Number of VALU V_MFMA_*_F16 instructions. (per-simd, emulated)", "gfx90a"});
		return 43;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_BF16
	 *
	 * Number of VALU V_MFMA_*_BF16 instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_BF16")==0) {
		basic_counters.emplace(44, BasicCounter{44, "SQ", "SQ_INSTS_VALU_MFMA_BF16", "Number of VALU V_MFMA_*_BF16 instructions. (per-simd, emulated)", "gfx90a"});
		return 44;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_F32
	 *
	 * Number of VALU V_MFMA_*_F32 instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_F32")==0) {
		basic_counters.emplace(45, BasicCounter{45, "SQ", "SQ_INSTS_VALU_MFMA_F32", "Number of VALU V_MFMA_*_F32 instructions. (per-simd, emulated)", "gfx90a"});
		return 45;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_F64
	 *
	 * Number of VALU V_MFMA_*_F64 instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_F64")==0) {
		basic_counters.emplace(46, BasicCounter{46, "SQ", "SQ_INSTS_VALU_MFMA_F64", "Number of VALU V_MFMA_*_F64 instructions. (per-simd, emulated)", "gfx90a"});
		return 46;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_MOPS_I8
	 *
	 * Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type I8. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_MOPS_I8")==0) {
		basic_counters.emplace(47, BasicCounter{47, "SQ", "SQ_INSTS_VALU_MFMA_MOPS_I8", "Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type I8. (per-simd, emulated)", "gfx90a"});
		return 47;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_MOPS_F16
	 *
	 * Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type F16. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_MOPS_F16")==0) {
		basic_counters.emplace(48, BasicCounter{48, "SQ", "SQ_INSTS_VALU_MFMA_MOPS_F16", "Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type F16. (per-simd, emulated)", "gfx90a"});
		return 48;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_MOPS_BF16
	 *
	 * Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type BF16. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_MOPS_BF16")==0) {
		basic_counters.emplace(49, BasicCounter{49, "SQ", "SQ_INSTS_VALU_MFMA_MOPS_BF16", "Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type BF16. (per-simd, emulated)", "gfx90a"});
		return 49;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_MOPS_F32
	 *
	 * Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type F32. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_MOPS_F32")==0) {
		basic_counters.emplace(50, BasicCounter{50, "SQ", "SQ_INSTS_VALU_MFMA_MOPS_F32", "Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type F32. (per-simd, emulated)", "gfx90a"});
		return 50;
	}
	/**
	 * Basic Counter: SQ_INSTS_VALU_MFMA_MOPS_F64
	 *
	 * Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type F64. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VALU_MFMA_MOPS_F64")==0) {
		basic_counters.emplace(51, BasicCounter{51, "SQ", "SQ_INSTS_VALU_MFMA_MOPS_F64", "Number of VALU matrix math operations (add or mul) performed dividied by 512, assuming a full EXEC mask, of data type F64. (per-simd, emulated)", "gfx90a"});
		return 51;
	}
	/**
	 * Basic Counter: SQ_INSTS_MFMA
	 *
	 * Number of MFMA instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_MFMA")==0) {
		basic_counters.emplace(52, BasicCounter{52, "SQ", "SQ_INSTS_MFMA", "Number of MFMA instructions issued. (per-simd, emulated)", "gfx90a"});
		return 52;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_WR
	 *
	 * Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_WR")==0) {
		basic_counters.emplace(53, BasicCounter{53, "SQ", "SQ_INSTS_VMEM_WR", "Number of VMEM write instructions issued (including FLAT). (per-simd, emulated)", "gfx90a"});
		return 53;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM_RD
	 *
	 * Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM_RD")==0) {
		basic_counters.emplace(54, BasicCounter{54, "SQ", "SQ_INSTS_VMEM_RD", "Number of VMEM read instructions issued (including FLAT). (per-simd, emulated)", "gfx90a"});
		return 54;
	}
	/**
	 * Basic Counter: SQ_INSTS_VMEM
	 *
	 * Number of VMEM instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VMEM")==0) {
		basic_counters.emplace(55, BasicCounter{55, "SQ", "SQ_INSTS_VMEM", "Number of VMEM instructions issued. (per-simd, emulated)", "gfx90a"});
		return 55;
	}
	/**
	 * Basic Counter: SQ_INSTS_SALU
	 *
	 * Number of SALU instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SALU")==0) {
		basic_counters.emplace(56, BasicCounter{56, "SQ", "SQ_INSTS_SALU", "Number of SALU instructions issued. (per-simd, emulated)", "gfx90a"});
		return 56;
	}
	/**
	 * Basic Counter: SQ_INSTS_SMEM
	 *
	 * Number of SMEM instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SMEM")==0) {
		basic_counters.emplace(57, BasicCounter{57, "SQ", "SQ_INSTS_SMEM", "Number of SMEM instructions issued. (per-simd, emulated)", "gfx90a"});
		return 57;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT
	 *
	 * Number of FLAT instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT")==0) {
		basic_counters.emplace(58, BasicCounter{58, "SQ", "SQ_INSTS_FLAT", "Number of FLAT instructions issued. (per-simd, emulated)", "gfx90a"});
		return 58;
	}
	/**
	 * Basic Counter: SQ_INSTS_FLAT_LDS_ONLY
	 *
	 * Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_FLAT_LDS_ONLY")==0) {
		basic_counters.emplace(59, BasicCounter{59, "SQ", "SQ_INSTS_FLAT_LDS_ONLY", "Number of FLAT instructions issued that read/wrote only from/to LDS (only works if EARLY_TA_DONE is enabled). (per-simd, emulated)", "gfx90a"});
		return 59;
	}
	/**
	 * Basic Counter: SQ_INSTS_LDS
	 *
	 * Number of LDS instructions issued (including FLAT). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_LDS")==0) {
		basic_counters.emplace(60, BasicCounter{60, "SQ", "SQ_INSTS_LDS", "Number of LDS instructions issued (including FLAT). (per-simd, emulated)", "gfx90a"});
		return 60;
	}
	/**
	 * Basic Counter: SQ_INSTS_GDS
	 *
	 * Number of GDS instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_GDS")==0) {
		basic_counters.emplace(61, BasicCounter{61, "SQ", "SQ_INSTS_GDS", "Number of GDS instructions issued. (per-simd, emulated)", "gfx90a"});
		return 61;
	}
	/**
	 * Basic Counter: SQ_INSTS_EXP_GDS
	 *
	 * Number of EXP and GDS instructions issued, excluding skipped export instructions. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_EXP_GDS")==0) {
		basic_counters.emplace(63, BasicCounter{63, "SQ", "SQ_INSTS_EXP_GDS", "Number of EXP and GDS instructions issued, excluding skipped export instructions. (per-simd, emulated)", "gfx90a"});
		return 63;
	}
	/**
	 * Basic Counter: SQ_INSTS_BRANCH
	 *
	 * Number of Branch instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_BRANCH")==0) {
		basic_counters.emplace(64, BasicCounter{64, "SQ", "SQ_INSTS_BRANCH", "Number of Branch instructions issued. (per-simd, emulated)", "gfx90a"});
		return 64;
	}
	/**
	 * Basic Counter: SQ_INSTS_SENDMSG
	 *
	 * Number of Sendmsg instructions issued. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SENDMSG")==0) {
		basic_counters.emplace(65, BasicCounter{65, "SQ", "SQ_INSTS_SENDMSG", "Number of Sendmsg instructions issued. (per-simd, emulated)", "gfx90a"});
		return 65;
	}
	/**
	 * Basic Counter: SQ_INSTS_VSKIPPED
	 *
	 * Number of vector instructions skipped. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_VSKIPPED")==0) {
		basic_counters.emplace(66, BasicCounter{66, "SQ", "SQ_INSTS_VSKIPPED", "Number of vector instructions skipped. (per-simd, emulated)", "gfx90a"});
		return 66;
	}
	/**
	 * Basic Counter: SQ_INST_LEVEL_VMEM
	 *
	 * Number of in-flight VMEM instructions. Set next counter to ACCUM_PREV and divide by INSTS_VMEM for average latency. Includes FLAT instructions. (per-simd, level, nondeterministic)
	 */
	if (strcmp(name, "SQ_INST_LEVEL_VMEM")==0) {
		basic_counters.emplace(67, BasicCounter{67, "SQ", "SQ_INST_LEVEL_VMEM", "Number of in-flight VMEM instructions. Set next counter to ACCUM_PREV and divide by INSTS_VMEM for average latency. Includes FLAT instructions. (per-simd, level, nondeterministic)", "gfx90a"});
		return 67;
	}
	/**
	 * Basic Counter: SQ_INST_LEVEL_SMEM
	 *
	 * Number of in-flight SMEM instructions (*2 load/store; *2 atomic; *2 memtime; *4 wb/inv). Set next counter to ACCUM_PREV and divide by INSTS_SMEM for average latency per smem request. Falls slightly short of total request latency because some fetches are divided into two requests that may finish at different times and this counter collects the average latency of the two. (per-simd, level, nondeterministic)
	 */
	if (strcmp(name, "SQ_INST_LEVEL_SMEM")==0) {
		basic_counters.emplace(68, BasicCounter{68, "SQ", "SQ_INST_LEVEL_SMEM", "Number of in-flight SMEM instructions (*2 load/store; *2 atomic; *2 memtime; *4 wb/inv). Set next counter to ACCUM_PREV and divide by INSTS_SMEM for average latency per smem request. Falls slightly short of total request latency because some fetches are divided into two requests that may finish at different times and this counter collects the average latency of the two. (per-simd, level, nondeterministic)", "gfx90a"});
		return 68;
	}
	/**
	 * Basic Counter: SQ_INST_LEVEL_LDS
	 *
	 * Number of in-flight LDS instructions. Set next counter to ACCUM_PREV and divide by INSTS_LDS for average latency. Includes FLAT instructions. (per-simd, level, nondeterministic)
	 */
	if (strcmp(name, "SQ_INST_LEVEL_LDS")==0) {
		basic_counters.emplace(69, BasicCounter{69, "SQ", "SQ_INST_LEVEL_LDS", "Number of in-flight LDS instructions. Set next counter to ACCUM_PREV and divide by INSTS_LDS for average latency. Includes FLAT instructions. (per-simd, level, nondeterministic)", "gfx90a"});
		return 69;
	}
	/**
	 * Basic Counter: SQ_VALU_MFMA_BUSY_CYCLES
	 *
	 * Number of cycles the MFMA ALU is busy (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_VALU_MFMA_BUSY_CYCLES")==0) {
		basic_counters.emplace(72, BasicCounter{72, "SQ", "SQ_VALU_MFMA_BUSY_CYCLES", "Number of cycles the MFMA ALU is busy (per-simd, emulated)", "gfx90a"});
		return 72;
	}
	/**
	 * Basic Counter: SQ_WAVE_CYCLES
	 *
	 * Number of wave-cycles spent by waves in the CUs (per-simd, nondeterministic). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_WAVE_CYCLES")==0) {
		basic_counters.emplace(74, BasicCounter{74, "SQ", "SQ_WAVE_CYCLES", "Number of wave-cycles spent by waves in the CUs (per-simd, nondeterministic). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 74;
	}
	/**
	 * Basic Counter: SQ_WAIT_ANY
	 *
	 * Number of wave-cycles spent waiting for anything (per-simd, nondeterministic). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_WAIT_ANY")==0) {
		basic_counters.emplace(85, BasicCounter{85, "SQ", "SQ_WAIT_ANY", "Number of wave-cycles spent waiting for anything (per-simd, nondeterministic). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 85;
	}
	/**
	 * Basic Counter: SQ_WAIT_INST_ANY
	 *
	 * Number of wave-cycles spent waiting for any instruction issue. In units of 4 cycles. (per-simd, nondeterministic)
	 */
	if (strcmp(name, "SQ_WAIT_INST_ANY")==0) {
		basic_counters.emplace(88, BasicCounter{88, "SQ", "SQ_WAIT_INST_ANY", "Number of wave-cycles spent waiting for any instruction issue. In units of 4 cycles. (per-simd, nondeterministic)", "gfx90a"});
		return 88;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_ANY
	 *
	 * Number of cycles each wave is working on an instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_ANY")==0) {
		basic_counters.emplace(96, BasicCounter{96, "SQ", "SQ_ACTIVE_INST_ANY", "Number of cycles each wave is working on an instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 96;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_VMEM
	 *
	 * Number of cycles the SQ instruction arbiter is working on a VMEM instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_VMEM")==0) {
		basic_counters.emplace(97, BasicCounter{97, "SQ", "SQ_ACTIVE_INST_VMEM", "Number of cycles the SQ instruction arbiter is working on a VMEM instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 97;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_LDS
	 *
	 * Number of cycles the SQ instruction arbiter is working on a LDS instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_LDS")==0) {
		basic_counters.emplace(98, BasicCounter{98, "SQ", "SQ_ACTIVE_INST_LDS", "Number of cycles the SQ instruction arbiter is working on a LDS instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 98;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_VALU
	 *
	 * Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_VALU")==0) {
		basic_counters.emplace(99, BasicCounter{99, "SQ", "SQ_ACTIVE_INST_VALU", "Number of cycles the SQ instruction arbiter is working on a VALU instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 99;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_SCA
	 *
	 * Number of cycles the SQ instruction arbiter is working on a SALU or SMEM instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_SCA")==0) {
		basic_counters.emplace(100, BasicCounter{100, "SQ", "SQ_ACTIVE_INST_SCA", "Number of cycles the SQ instruction arbiter is working on a SALU or SMEM instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 100;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_EXP_GDS
	 *
	 * Number of cycles the SQ instruction arbiter is working on an EXPORT or GDS instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_EXP_GDS")==0) {
		basic_counters.emplace(101, BasicCounter{101, "SQ", "SQ_ACTIVE_INST_EXP_GDS", "Number of cycles the SQ instruction arbiter is working on an EXPORT or GDS instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 101;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_MISC
	 *
	 * Number of cycles the SQ instruction aribter is working on a BRANCH or SENDMSG instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_MISC")==0) {
		basic_counters.emplace(102, BasicCounter{102, "SQ", "SQ_ACTIVE_INST_MISC", "Number of cycles the SQ instruction aribter is working on a BRANCH or SENDMSG instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 102;
	}
	/**
	 * Basic Counter: SQ_ACTIVE_INST_FLAT
	 *
	 * Number of cycles the SQ instruction arbiter is working on a FLAT instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_ACTIVE_INST_FLAT")==0) {
		basic_counters.emplace(103, BasicCounter{103, "SQ", "SQ_ACTIVE_INST_FLAT", "Number of cycles the SQ instruction arbiter is working on a FLAT instruction. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 103;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_VMEM_WR
	 *
	 * Number of cycles needed to send addr and cmd data for VMEM write instructions. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_VMEM_WR")==0) {
		basic_counters.emplace(104, BasicCounter{104, "SQ", "SQ_INST_CYCLES_VMEM_WR", "Number of cycles needed to send addr and cmd data for VMEM write instructions. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 104;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_VMEM_RD
	 *
	 * Number of cycles needed to send addr and cmd data for VMEM read instructions. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_VMEM_RD")==0) {
		basic_counters.emplace(105, BasicCounter{105, "SQ", "SQ_INST_CYCLES_VMEM_RD", "Number of cycles needed to send addr and cmd data for VMEM read instructions. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 105;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_SMEM
	 *
	 * Number of cycles needed to execute scalar memory reads. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_SMEM")==0) {
		basic_counters.emplace(111, BasicCounter{111, "SQ", "SQ_INST_CYCLES_SMEM", "Number of cycles needed to execute scalar memory reads. (per-simd, emulated)", "gfx90a"});
		return 111;
	}
	/**
	 * Basic Counter: SQ_INST_CYCLES_SALU
	 *
	 * Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated). Units in quad-cycles(4 cycles)
	 */
	if (strcmp(name, "SQ_INST_CYCLES_SALU")==0) {
		basic_counters.emplace(112, BasicCounter{112, "SQ", "SQ_INST_CYCLES_SALU", "Number of cycles needed to execute non-memory read scalar operations. (per-simd, emulated). Units in quad-cycles(4 cycles)", "gfx90a"});
		return 112;
	}
	/**
	 * Basic Counter: SQ_THREAD_CYCLES_VALU
	 *
	 * Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)
	 */
	if (strcmp(name, "SQ_THREAD_CYCLES_VALU")==0) {
		basic_counters.emplace(113, BasicCounter{113, "SQ", "SQ_THREAD_CYCLES_VALU", "Number of thread-cycles used to execute VALU operations (similar to INST_CYCLES_VALU but multiplied by # of active threads). (per-simd)", "gfx90a"});
		return 113;
	}
	/**
	 * Basic Counter: SQ_IFETCH
	 *
	 * Number of instruction fetch requests from cache. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_IFETCH")==0) {
		basic_counters.emplace(115, BasicCounter{115, "SQ", "SQ_IFETCH", "Number of instruction fetch requests from cache. (per-simd, emulated)", "gfx90a"});
		return 115;
	}
	/**
	 * Basic Counter: SQ_IFETCH_LEVEL
	 *
	 * Number of instruction fetch requests from cache. (per-simd, level)
	 */
	if (strcmp(name, "SQ_IFETCH_LEVEL")==0) {
		basic_counters.emplace(116, BasicCounter{116, "SQ", "SQ_IFETCH_LEVEL", "Number of instruction fetch requests from cache. (per-simd, level)", "gfx90a"});
		return 116;
	}
	/**
	 * Basic Counter: SQ_LDS_BANK_CONFLICT
	 *
	 * Number of cycles LDS is stalled by bank conflicts. (emulated)
	 */
	if (strcmp(name, "SQ_LDS_BANK_CONFLICT")==0) {
		basic_counters.emplace(121, BasicCounter{121, "SQ", "SQ_LDS_BANK_CONFLICT", "Number of cycles LDS is stalled by bank conflicts. (emulated)", "gfx90a"});
		return 121;
	}
	/**
	 * Basic Counter: SQ_LDS_ADDR_CONFLICT
	 *
	 * Number of cycles LDS is stalled by address conflicts. (emulated,nondeterministic)
	 */
	if (strcmp(name, "SQ_LDS_ADDR_CONFLICT")==0) {
		basic_counters.emplace(122, BasicCounter{122, "SQ", "SQ_LDS_ADDR_CONFLICT", "Number of cycles LDS is stalled by address conflicts. (emulated,nondeterministic)", "gfx90a"});
		return 122;
	}
	/**
	 * Basic Counter: SQ_LDS_UNALIGNED_STALL
	 *
	 * Number of cycles LDS is stalled processing flat unaligned load/store ops. (emulated)
	 */
	if (strcmp(name, "SQ_LDS_UNALIGNED_STALL")==0) {
		basic_counters.emplace(123, BasicCounter{123, "SQ", "SQ_LDS_UNALIGNED_STALL", "Number of cycles LDS is stalled processing flat unaligned load/store ops. (emulated)", "gfx90a"});
		return 123;
	}
	/**
	 * Basic Counter: SQ_LDS_MEM_VIOLATIONS
	 *
	 * Number of threads that have a memory violation in the LDS.(emulated)
	 */
	if (strcmp(name, "SQ_LDS_MEM_VIOLATIONS")==0) {
		basic_counters.emplace(124, BasicCounter{124, "SQ", "SQ_LDS_MEM_VIOLATIONS", "Number of threads that have a memory violation in the LDS.(emulated)", "gfx90a"});
		return 124;
	}
	/**
	 * Basic Counter: SQ_LDS_ATOMIC_RETURN
	 *
	 * Number of atomic return cycles in LDS. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_LDS_ATOMIC_RETURN")==0) {
		basic_counters.emplace(125, BasicCounter{125, "SQ", "SQ_LDS_ATOMIC_RETURN", "Number of atomic return cycles in LDS. (per-simd, emulated)", "gfx90a"});
		return 125;
	}
	/**
	 * Basic Counter: SQ_LDS_IDX_ACTIVE
	 *
	 * Number of cycles LDS is used for indexed (non-direct,non-interpolation) operations. (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_LDS_IDX_ACTIVE")==0) {
		basic_counters.emplace(126, BasicCounter{126, "SQ", "SQ_LDS_IDX_ACTIVE", "Number of cycles LDS is used for indexed (non-direct,non-interpolation) operations. (per-simd, emulated)", "gfx90a"});
		return 126;
	}
	/**
	 * Basic Counter: SQ_ACCUM_PREV_HIRES
	 *
	 * For counter N, increment by the value of counter N-1.
	 */
	if (strcmp(name, "SQ_ACCUM_PREV_HIRES")==0) {
		basic_counters.emplace(185, BasicCounter{185, "SQ", "SQ_ACCUM_PREV_HIRES", "For counter N, increment by the value of counter N-1.", "gfx90a"});
		return 185;
	}
	/**
	 * Basic Counter: SQ_WAVES_RESTORED
	 *
	 * Count number of context-restored waves sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_RESTORED")==0) {
		basic_counters.emplace(186, BasicCounter{186, "SQ", "SQ_WAVES_RESTORED", "Count number of context-restored waves sent to SQs. (per-simd, emulated, global)", "gfx90a"});
		return 186;
	}
	/**
	 * Basic Counter: SQ_WAVES_SAVED
	 *
	 * Count number of context-saved waves. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_SAVED")==0) {
		basic_counters.emplace(187, BasicCounter{187, "SQ", "SQ_WAVES_SAVED", "Count number of context-saved waves. (per-simd, emulated, global)", "gfx90a"});
		return 187;
	}
	/**
	 * Basic Counter: SQ_INSTS_SMEM_NORM
	 *
	 * Number of SMEM instructions issued normalized to match smem_level (*2 load/store; *2 atomic; *2 memtime; *4 wb/inv). (per-simd, emulated)
	 */
	if (strcmp(name, "SQ_INSTS_SMEM_NORM")==0) {
		basic_counters.emplace(188, BasicCounter{188, "SQ", "SQ_INSTS_SMEM_NORM", "Number of SMEM instructions issued normalized to match smem_level (*2 load/store; *2 atomic; *2 memtime; *4 wb/inv). (per-simd, emulated)", "gfx90a"});
		return 188;
	}
	/**
	 * Basic Counter: SQC_DCACHE_INPUT_VALID_READYB
	 *
	 * Input stalled by SQC (per-SQ, nondeterministic, unwindowed)
	 */
	if (strcmp(name, "SQC_DCACHE_INPUT_VALID_READYB")==0) {
		basic_counters.emplace(260, BasicCounter{260, "SQ", "SQC_DCACHE_INPUT_VALID_READYB", "Input stalled by SQC (per-SQ, nondeterministic, unwindowed)", "gfx90a"});
		return 260;
	}
	/**
	 * Basic Counter: SQC_TC_REQ
	 *
	 * Total number of TC requests that were issued by instruction and constant caches. (No-Masking, nondeterministic)
	 */
	if (strcmp(name, "SQC_TC_REQ")==0) {
		basic_counters.emplace(262, BasicCounter{262, "SQ", "SQC_TC_REQ", "Total number of TC requests that were issued by instruction and constant caches. (No-Masking, nondeterministic)", "gfx90a"});
		return 262;
	}
	/**
	 * Basic Counter: SQC_TC_INST_REQ
	 *
	 * Number of insruction requests to the TC (No-Masking, nondeterministic)
	 */
	if (strcmp(name, "SQC_TC_INST_REQ")==0) {
		basic_counters.emplace(263, BasicCounter{263, "SQ", "SQC_TC_INST_REQ", "Number of insruction requests to the TC (No-Masking, nondeterministic)", "gfx90a"});
		return 263;
	}
	/**
	 * Basic Counter: SQC_TC_DATA_READ_REQ
	 *
	 * Number of data read requests to the TC (No-Masking, nondeterministic)
	 */
	if (strcmp(name, "SQC_TC_DATA_READ_REQ")==0) {
		basic_counters.emplace(264, BasicCounter{264, "SQ", "SQC_TC_DATA_READ_REQ", "Number of data read requests to the TC (No-Masking, nondeterministic)", "gfx90a"});
		return 264;
	}
	/**
	 * Basic Counter: SQC_TC_DATA_WRITE_REQ
	 *
	 * Number of data write requests to the TC (No-Masking, nondeterministic)
	 */
	if (strcmp(name, "SQC_TC_DATA_WRITE_REQ")==0) {
		basic_counters.emplace(265, BasicCounter{265, "SQ", "SQC_TC_DATA_WRITE_REQ", "Number of data write requests to the TC (No-Masking, nondeterministic)", "gfx90a"});
		return 265;
	}
	/**
	 * Basic Counter: SQC_TC_DATA_ATOMIC_REQ
	 *
	 * Number of data atomic requests to the TC (No-Masking, nondeterministic)
	 */
	if (strcmp(name, "SQC_TC_DATA_ATOMIC_REQ")==0) {
		basic_counters.emplace(266, BasicCounter{266, "SQ", "SQC_TC_DATA_ATOMIC_REQ", "Number of data atomic requests to the TC (No-Masking, nondeterministic)", "gfx90a"});
		return 266;
	}
	/**
	 * Basic Counter: SQC_TC_STALL
	 *
	 * Valid request stalled TC request interface (no-credits). (No-Masking, nondeterministic, unwindowed)
	 */
	if (strcmp(name, "SQC_TC_STALL")==0) {
		basic_counters.emplace(267, BasicCounter{267, "SQ", "SQC_TC_STALL", "Valid request stalled TC request interface (no-credits). (No-Masking, nondeterministic, unwindowed)", "gfx90a"});
		return 267;
	}
	/**
	 * Basic Counter: SQC_ICACHE_REQ
	 *
	 * Number of requests. (per-SQ, per-Bank)
	 */
	if (strcmp(name, "SQC_ICACHE_REQ")==0) {
		basic_counters.emplace(270, BasicCounter{270, "SQ", "SQC_ICACHE_REQ", "Number of requests. (per-SQ, per-Bank)", "gfx90a"});
		return 270;
	}
	/**
	 * Basic Counter: SQC_ICACHE_HITS
	 *
	 * Number of cache hits. (per-SQ, per-Bank, nondeterministic)
	 */
	if (strcmp(name, "SQC_ICACHE_HITS")==0) {
		basic_counters.emplace(271, BasicCounter{271, "SQ", "SQC_ICACHE_HITS", "Number of cache hits. (per-SQ, per-Bank, nondeterministic)", "gfx90a"});
		return 271;
	}
	/**
	 * Basic Counter: SQC_ICACHE_MISSES
	 *
	 * Number of cache misses, includes uncached requests. (per-SQ, per-Bank, nondeterministic)
	 */
	if (strcmp(name, "SQC_ICACHE_MISSES")==0) {
		basic_counters.emplace(272, BasicCounter{272, "SQ", "SQC_ICACHE_MISSES", "Number of cache misses, includes uncached requests. (per-SQ, per-Bank, nondeterministic)", "gfx90a"});
		return 272;
	}
	/**
	 * Basic Counter: SQC_ICACHE_MISSES_DUPLICATE
	 *
	 * Number of misses that were duplicates (access to a non-resident, miss pending CL). (per-SQ, per-Bank, nondeterministic)
	 */
	if (strcmp(name, "SQC_ICACHE_MISSES_DUPLICATE")==0) {
		basic_counters.emplace(273, BasicCounter{273, "SQ", "SQC_ICACHE_MISSES_DUPLICATE", "Number of misses that were duplicates (access to a non-resident, miss pending CL). (per-SQ, per-Bank, nondeterministic)", "gfx90a"});
		return 273;
	}
	/**
	 * Basic Counter: SQC_DCACHE_REQ
	 *
	 * Number of requests (post-bank-serialization). (per-SQ, per-Bank)
	 */
	if (strcmp(name, "SQC_DCACHE_REQ")==0) {
		basic_counters.emplace(290, BasicCounter{290, "SQ", "SQC_DCACHE_REQ", "Number of requests (post-bank-serialization). (per-SQ, per-Bank)", "gfx90a"});
		return 290;
	}
	/**
	 * Basic Counter: SQC_DCACHE_HITS
	 *
	 * Number of cache hits. (per-SQ, per-Bank, nondeterministic)
	 */
	if (strcmp(name, "SQC_DCACHE_HITS")==0) {
		basic_counters.emplace(291, BasicCounter{291, "SQ", "SQC_DCACHE_HITS", "Number of cache hits. (per-SQ, per-Bank, nondeterministic)", "gfx90a"});
		return 291;
	}
	/**
	 * Basic Counter: SQC_DCACHE_MISSES
	 *
	 * Number of cache misses, includes uncached requests. (per-SQ, per-Bank, nondeterministic)
	 */
	if (strcmp(name, "SQC_DCACHE_MISSES")==0) {
		basic_counters.emplace(292, BasicCounter{292, "SQ", "SQC_DCACHE_MISSES", "Number of cache misses, includes uncached requests. (per-SQ, per-Bank, nondeterministic)", "gfx90a"});
		return 292;
	}
	/**
	 * Basic Counter: SQC_DCACHE_MISSES_DUPLICATE
	 *
	 * Number of misses that were duplicates (access to a non-resident, miss pending CL). (per-SQ, per-Bank, nondeterministic)
	 */
	if (strcmp(name, "SQC_DCACHE_MISSES_DUPLICATE")==0) {
		basic_counters.emplace(293, BasicCounter{293, "SQ", "SQC_DCACHE_MISSES_DUPLICATE", "Number of misses that were duplicates (access to a non-resident, miss pending CL). (per-SQ, per-Bank, nondeterministic)", "gfx90a"});
		return 293;
	}
	/**
	 * Basic Counter: SQC_DCACHE_ATOMIC
	 *
	 * Number of atomic requests. (per-SQ, per-Bank)
	 */
	if (strcmp(name, "SQC_DCACHE_ATOMIC")==0) {
		basic_counters.emplace(298, BasicCounter{298, "SQ", "SQC_DCACHE_ATOMIC", "Number of atomic requests. (per-SQ, per-Bank)", "gfx90a"});
		return 298;
	}
	/**
	 * Basic Counter: SQC_DCACHE_REQ_READ_1
	 *
	 * Number of constant cache 1 dw read requests. (per-SQ)
	 */
	if (strcmp(name, "SQC_DCACHE_REQ_READ_1")==0) {
		basic_counters.emplace(323, BasicCounter{323, "SQ", "SQC_DCACHE_REQ_READ_1", "Number of constant cache 1 dw read requests. (per-SQ)", "gfx90a"});
		return 323;
	}
	/**
	 * Basic Counter: SQC_DCACHE_REQ_READ_2
	 *
	 * Number of constant cache 2 dw read requests. (per-SQ)
	 */
	if (strcmp(name, "SQC_DCACHE_REQ_READ_2")==0) {
		basic_counters.emplace(324, BasicCounter{324, "SQ", "SQC_DCACHE_REQ_READ_2", "Number of constant cache 2 dw read requests. (per-SQ)", "gfx90a"});
		return 324;
	}
	/**
	 * Basic Counter: SQC_DCACHE_REQ_READ_4
	 *
	 * Number of constant cache 4 dw read requests. (per-SQ)
	 */
	if (strcmp(name, "SQC_DCACHE_REQ_READ_4")==0) {
		basic_counters.emplace(325, BasicCounter{325, "SQ", "SQC_DCACHE_REQ_READ_4", "Number of constant cache 4 dw read requests. (per-SQ)", "gfx90a"});
		return 325;
	}
	/**
	 * Basic Counter: SQC_DCACHE_REQ_READ_8
	 *
	 * Number of constant cache 8 dw read requests. (per-SQ)
	 */
	if (strcmp(name, "SQC_DCACHE_REQ_READ_8")==0) {
		basic_counters.emplace(326, BasicCounter{326, "SQ", "SQC_DCACHE_REQ_READ_8", "Number of constant cache 8 dw read requests. (per-SQ)", "gfx90a"});
		return 326;
	}
	/**
	 * Basic Counter: SQC_DCACHE_REQ_READ_16
	 *
	 * Number of constant cache 16 dw read requests. (per-SQ)
	 */
	if (strcmp(name, "SQC_DCACHE_REQ_READ_16")==0) {
		basic_counters.emplace(327, BasicCounter{327, "SQ", "SQC_DCACHE_REQ_READ_16", "Number of constant cache 16 dw read requests. (per-SQ)", "gfx90a"});
		return 327;
	}
	/**
	 * Basic Counter: TA_TA_BUSY
	 *
	 * TA block is busy. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_TA_BUSY")==0) {
		basic_counters.emplace(15, BasicCounter{15, "TA", "TA_TA_BUSY", "TA block is busy. Perf_Windowing not supported for this counter.", "gfx90a"});
		return 15;
	}
	/**
	 * Basic Counter: TA_TOTAL_WAVEFRONTS
	 *
	 * Total number of wavefronts processed by TA.
	 */
	if (strcmp(name, "TA_TOTAL_WAVEFRONTS")==0) {
		basic_counters.emplace(32, BasicCounter{32, "TA", "TA_TOTAL_WAVEFRONTS", "Total number of wavefronts processed by TA.", "gfx90a"});
		return 32;
	}
	/**
	 * Basic Counter: TA_BUFFER_WAVEFRONTS
	 *
	 * Number of buffer wavefronts processed by TA.
	 */
	if (strcmp(name, "TA_BUFFER_WAVEFRONTS")==0) {
		basic_counters.emplace(44, BasicCounter{44, "TA", "TA_BUFFER_WAVEFRONTS", "Number of buffer wavefronts processed by TA.", "gfx90a"});
		return 44;
	}
	/**
	 * Basic Counter: TA_BUFFER_READ_WAVEFRONTS
	 *
	 * Number of buffer read wavefronts processed by TA.
	 */
	if (strcmp(name, "TA_BUFFER_READ_WAVEFRONTS")==0) {
		basic_counters.emplace(45, BasicCounter{45, "TA", "TA_BUFFER_READ_WAVEFRONTS", "Number of buffer read wavefronts processed by TA.", "gfx90a"});
		return 45;
	}
	/**
	 * Basic Counter: TA_BUFFER_WRITE_WAVEFRONTS
	 *
	 * Number of buffer write wavefronts processed by TA.
	 */
	if (strcmp(name, "TA_BUFFER_WRITE_WAVEFRONTS")==0) {
		basic_counters.emplace(46, BasicCounter{46, "TA", "TA_BUFFER_WRITE_WAVEFRONTS", "Number of buffer write wavefronts processed by TA.", "gfx90a"});
		return 46;
	}
	/**
	 * Basic Counter: TA_BUFFER_ATOMIC_WAVEFRONTS
	 *
	 * Number of buffer atomic wavefronts processed by TA.
	 */
	if (strcmp(name, "TA_BUFFER_ATOMIC_WAVEFRONTS")==0) {
		basic_counters.emplace(47, BasicCounter{47, "TA", "TA_BUFFER_ATOMIC_WAVEFRONTS", "Number of buffer atomic wavefronts processed by TA.", "gfx90a"});
		return 47;
	}
	/**
	 * Basic Counter: TA_BUFFER_TOTAL_CYCLES
	 *
	 * Number of buffer cycles issued to TC.
	 */
	if (strcmp(name, "TA_BUFFER_TOTAL_CYCLES")==0) {
		basic_counters.emplace(49, BasicCounter{49, "TA", "TA_BUFFER_TOTAL_CYCLES", "Number of buffer cycles issued to TC.", "gfx90a"});
		return 49;
	}
	/**
	 * Basic Counter: TA_BUFFER_COALESCED_READ_CYCLES
	 *
	 * Number of buffer coalesced read cycles issued to TC.
	 */
	if (strcmp(name, "TA_BUFFER_COALESCED_READ_CYCLES")==0) {
		basic_counters.emplace(52, BasicCounter{52, "TA", "TA_BUFFER_COALESCED_READ_CYCLES", "Number of buffer coalesced read cycles issued to TC.", "gfx90a"});
		return 52;
	}
	/**
	 * Basic Counter: TA_BUFFER_COALESCED_WRITE_CYCLES
	 *
	 * Number of buffer coalesced write cycles issued to TC.
	 */
	if (strcmp(name, "TA_BUFFER_COALESCED_WRITE_CYCLES")==0) {
		basic_counters.emplace(53, BasicCounter{53, "TA", "TA_BUFFER_COALESCED_WRITE_CYCLES", "Number of buffer coalesced write cycles issued to TC.", "gfx90a"});
		return 53;
	}
	/**
	 * Basic Counter: TA_ADDR_STALLED_BY_TC_CYCLES
	 *
	 * Number of cycles addr path stalled by TC. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_ADDR_STALLED_BY_TC_CYCLES")==0) {
		basic_counters.emplace(54, BasicCounter{54, "TA", "TA_ADDR_STALLED_BY_TC_CYCLES", "Number of cycles addr path stalled by TC. Perf_Windowing not supported for this counter.", "gfx90a"});
		return 54;
	}
	/**
	 * Basic Counter: TA_ADDR_STALLED_BY_TD_CYCLES
	 *
	 * Number of cycles addr path stalled by TD. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_ADDR_STALLED_BY_TD_CYCLES")==0) {
		basic_counters.emplace(55, BasicCounter{55, "TA", "TA_ADDR_STALLED_BY_TD_CYCLES", "Number of cycles addr path stalled by TD. Perf_Windowing not supported for this counter.", "gfx90a"});
		return 55;
	}
	/**
	 * Basic Counter: TA_DATA_STALLED_BY_TC_CYCLES
	 *
	 * Number of cycles data path stalled by TC. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TA_DATA_STALLED_BY_TC_CYCLES")==0) {
		basic_counters.emplace(56, BasicCounter{56, "TA", "TA_DATA_STALLED_BY_TC_CYCLES", "Number of cycles data path stalled by TC. Perf_Windowing not supported for this counter.", "gfx90a"});
		return 56;
	}
	/**
	 * Basic Counter: TA_FLAT_WAVEFRONTS
	 *
	 * Number of flat opcode wavfronts processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_WAVEFRONTS")==0) {
		basic_counters.emplace(100, BasicCounter{100, "TA", "TA_FLAT_WAVEFRONTS", "Number of flat opcode wavfronts processed by the TA.", "gfx90a"});
		return 100;
	}
	/**
	 * Basic Counter: TA_FLAT_READ_WAVEFRONTS
	 *
	 * Number of flat opcode reads processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_READ_WAVEFRONTS")==0) {
		basic_counters.emplace(101, BasicCounter{101, "TA", "TA_FLAT_READ_WAVEFRONTS", "Number of flat opcode reads processed by the TA.", "gfx90a"});
		return 101;
	}
	/**
	 * Basic Counter: TA_FLAT_WRITE_WAVEFRONTS
	 *
	 * Number of flat opcode writes processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_WRITE_WAVEFRONTS")==0) {
		basic_counters.emplace(102, BasicCounter{102, "TA", "TA_FLAT_WRITE_WAVEFRONTS", "Number of flat opcode writes processed by the TA.", "gfx90a"});
		return 102;
	}
	/**
	 * Basic Counter: TA_FLAT_ATOMIC_WAVEFRONTS
	 *
	 * Number of flat opcode atomics processed by the TA.
	 */
	if (strcmp(name, "TA_FLAT_ATOMIC_WAVEFRONTS")==0) {
		basic_counters.emplace(103, BasicCounter{103, "TA", "TA_FLAT_ATOMIC_WAVEFRONTS", "Number of flat opcode atomics processed by the TA.", "gfx90a"});
		return 103;
	}
	/**
	 * Basic Counter: TD_TD_BUSY
	 *
	 * TD is processing or waiting for data. Perf_Windowing not supported for this counter.
	 */
	if (strcmp(name, "TD_TD_BUSY")==0) {
		basic_counters.emplace(1, BasicCounter{1, "TD", "TD_TD_BUSY", "TD is processing or waiting for data. Perf_Windowing not supported for this counter.", "gfx90a"});
		return 1;
	}
	/**
	 * Basic Counter: TD_TC_STALL
	 *
	 * TD is stalled waiting for TC data.
	 */
	if (strcmp(name, "TD_TC_STALL")==0) {
		basic_counters.emplace(15, BasicCounter{15, "TD", "TD_TC_STALL", "TD is stalled waiting for TC data.", "gfx90a"});
		return 15;
	}
	/**
	 * Basic Counter: TD_SPI_STALL
	 *
	 * TD is stalled SPI vinit
	 */
	if (strcmp(name, "TD_SPI_STALL")==0) {
		basic_counters.emplace(18, BasicCounter{18, "TD", "TD_SPI_STALL", "TD is stalled SPI vinit", "gfx90a"});
		return 18;
	}
	/**
	 * Basic Counter: TD_LOAD_WAVEFRONT
	 *
	 * Count the wavefronts with opcode = load, include atomics and store.
	 */
	if (strcmp(name, "TD_LOAD_WAVEFRONT")==0) {
		basic_counters.emplace(25, BasicCounter{25, "TD", "TD_LOAD_WAVEFRONT", "Count the wavefronts with opcode = load, include atomics and store.", "gfx90a"});
		return 25;
	}
	/**
	 * Basic Counter: TD_ATOMIC_WAVEFRONT
	 *
	 * Count the wavefronts with opcode = atomic.
	 */
	if (strcmp(name, "TD_ATOMIC_WAVEFRONT")==0) {
		basic_counters.emplace(26, BasicCounter{26, "TD", "TD_ATOMIC_WAVEFRONT", "Count the wavefronts with opcode = atomic.", "gfx90a"});
		return 26;
	}
	/**
	 * Basic Counter: TD_STORE_WAVEFRONT
	 *
	 * Count the wavefronts with opcode = store.
	 */
	if (strcmp(name, "TD_STORE_WAVEFRONT")==0) {
		basic_counters.emplace(27, BasicCounter{27, "TD", "TD_STORE_WAVEFRONT", "Count the wavefronts with opcode = store.", "gfx90a"});
		return 27;
	}
	/**
	 * Basic Counter: TD_COALESCABLE_WAVEFRONT
	 *
	 * Count wavefronts that TA finds coalescable.
	 */
	if (strcmp(name, "TD_COALESCABLE_WAVEFRONT")==0) {
		basic_counters.emplace(32, BasicCounter{32, "TD", "TD_COALESCABLE_WAVEFRONT", "Count wavefronts that TA finds coalescable.", "gfx90a"});
		return 32;
	}
	/**
	 * Basic Counter: TCP_GATE_EN1
	 *
	 * TCP interface clocks are turned on. Not Windowed.
	 */
	if (strcmp(name, "TCP_GATE_EN1")==0) {
		basic_counters.emplace(0, BasicCounter{0, "TCP", "TCP_GATE_EN1", "TCP interface clocks are turned on. Not Windowed.", "gfx90a"});
		return 0;
	}
	/**
	 * Basic Counter: TCP_GATE_EN2
	 *
	 * TCP core clocks are turned on. Not Windowed.
	 */
	if (strcmp(name, "TCP_GATE_EN2")==0) {
		basic_counters.emplace(1, BasicCounter{1, "TCP", "TCP_GATE_EN2", "TCP core clocks are turned on. Not Windowed.", "gfx90a"});
		return 1;
	}
	/**
	 * Basic Counter: TCP_TD_TCP_STALL_CYCLES
	 *
	 * TD stalls TCP
	 */
	if (strcmp(name, "TCP_TD_TCP_STALL_CYCLES")==0) {
		basic_counters.emplace(7, BasicCounter{7, "TCP", "TCP_TD_TCP_STALL_CYCLES", "TD stalls TCP", "gfx90a"});
		return 7;
	}
	/**
	 * Basic Counter: TCP_TCR_TCP_STALL_CYCLES
	 *
	 * TCR stalls TCP_TCR_req interface
	 */
	if (strcmp(name, "TCP_TCR_TCP_STALL_CYCLES")==0) {
		basic_counters.emplace(8, BasicCounter{8, "TCP", "TCP_TCR_TCP_STALL_CYCLES", "TCR stalls TCP_TCR_req interface", "gfx90a"});
		return 8;
	}
	/**
	 * Basic Counter: TCP_READ_TAGCONFLICT_STALL_CYCLES
	 *
	 * Tagram conflict stall on a read
	 */
	if (strcmp(name, "TCP_READ_TAGCONFLICT_STALL_CYCLES")==0) {
		basic_counters.emplace(11, BasicCounter{11, "TCP", "TCP_READ_TAGCONFLICT_STALL_CYCLES", "Tagram conflict stall on a read", "gfx90a"});
		return 11;
	}
	/**
	 * Basic Counter: TCP_WRITE_TAGCONFLICT_STALL_CYCLES
	 *
	 * Tagram conflict stall on a write
	 */
	if (strcmp(name, "TCP_WRITE_TAGCONFLICT_STALL_CYCLES")==0) {
		basic_counters.emplace(12, BasicCounter{12, "TCP", "TCP_WRITE_TAGCONFLICT_STALL_CYCLES", "Tagram conflict stall on a write", "gfx90a"});
		return 12;
	}
	/**
	 * Basic Counter: TCP_ATOMIC_TAGCONFLICT_STALL_CYCLES
	 *
	 * Tagram conflict stall on an atomic
	 */
	if (strcmp(name, "TCP_ATOMIC_TAGCONFLICT_STALL_CYCLES")==0) {
		basic_counters.emplace(13, BasicCounter{13, "TCP", "TCP_ATOMIC_TAGCONFLICT_STALL_CYCLES", "Tagram conflict stall on an atomic", "gfx90a"});
		return 13;
	}
	/**
	 * Basic Counter: TCP_PENDING_STALL_CYCLES
	 *
	 * Stall due to data pending from L2
	 */
	if (strcmp(name, "TCP_PENDING_STALL_CYCLES")==0) {
		basic_counters.emplace(22, BasicCounter{22, "TCP", "TCP_PENDING_STALL_CYCLES", "Stall due to data pending from L2", "gfx90a"});
		return 22;
	}
	/**
	 * Basic Counter: TCP_TA_TCP_STATE_READ
	 *
	 * Number of state reads
	 */
	if (strcmp(name, "TCP_TA_TCP_STATE_READ")==0) {
		basic_counters.emplace(27, BasicCounter{27, "TCP", "TCP_TA_TCP_STATE_READ", "Number of state reads", "gfx90a"});
		return 27;
	}
	/**
	 * Basic Counter: TCP_VOLATILE
	 *
	 * Total number of L1 volatile pixels/buffers from TA
	 */
	if (strcmp(name, "TCP_VOLATILE")==0) {
		basic_counters.emplace(28, BasicCounter{28, "TCP", "TCP_VOLATILE", "Total number of L1 volatile pixels/buffers from TA", "gfx90a"});
		return 28;
	}
	/**
	 * Basic Counter: TCP_TOTAL_ACCESSES
	 *
	 * Total number of pixels/buffers from TA. Equals TCP_PERF_SEL_TOTAL_READ+TCP_PERF_SEL_TOTAL_NONREAD
	 */
	if (strcmp(name, "TCP_TOTAL_ACCESSES")==0) {
		basic_counters.emplace(29, BasicCounter{29, "TCP", "TCP_TOTAL_ACCESSES", "Total number of pixels/buffers from TA. Equals TCP_PERF_SEL_TOTAL_READ+TCP_PERF_SEL_TOTAL_NONREAD", "gfx90a"});
		return 29;
	}
	/**
	 * Basic Counter: TCP_TOTAL_READ
	 *
	 * Total number of read pixels/buffers from TA. Equals TCP_PERF_SEL_TOTAL_HIT_LRU_READ + TCP_PERF_SEL_TOTAL_MISS_LRU_READ + TCP_PERF_SEL_TOTAL_MISS_EVICT_READ
	 */
	if (strcmp(name, "TCP_TOTAL_READ")==0) {
		basic_counters.emplace(30, BasicCounter{30, "TCP", "TCP_TOTAL_READ", "Total number of read pixels/buffers from TA. Equals TCP_PERF_SEL_TOTAL_HIT_LRU_READ + TCP_PERF_SEL_TOTAL_MISS_LRU_READ + TCP_PERF_SEL_TOTAL_MISS_EVICT_READ", "gfx90a"});
		return 30;
	}
	/**
	 * Basic Counter: TCP_TOTAL_WRITE
	 *
	 * Total number of local write pixels/buffers from TA. Equals TCP_PERF_SEL_TOTAL_MISS_LRU_WRITE+ TCP_PERF_SEL_TOTAL_MISS_EVICT_WRITE
	 */
	if (strcmp(name, "TCP_TOTAL_WRITE")==0) {
		basic_counters.emplace(32, BasicCounter{32, "TCP", "TCP_TOTAL_WRITE", "Total number of local write pixels/buffers from TA. Equals TCP_PERF_SEL_TOTAL_MISS_LRU_WRITE+ TCP_PERF_SEL_TOTAL_MISS_EVICT_WRITE", "gfx90a"});
		return 32;
	}
	/**
	 * Basic Counter: TCP_TOTAL_ATOMIC_WITH_RET
	 *
	 * Total number of atomic with return pixels/buffers from TA
	 */
	if (strcmp(name, "TCP_TOTAL_ATOMIC_WITH_RET")==0) {
		basic_counters.emplace(38, BasicCounter{38, "TCP", "TCP_TOTAL_ATOMIC_WITH_RET", "Total number of atomic with return pixels/buffers from TA", "gfx90a"});
		return 38;
	}
	/**
	 * Basic Counter: TCP_TOTAL_ATOMIC_WITHOUT_RET
	 *
	 * Total number of atomic without return pixels/buffers from TA
	 */
	if (strcmp(name, "TCP_TOTAL_ATOMIC_WITHOUT_RET")==0) {
		basic_counters.emplace(39, BasicCounter{39, "TCP", "TCP_TOTAL_ATOMIC_WITHOUT_RET", "Total number of atomic without return pixels/buffers from TA", "gfx90a"});
		return 39;
	}
	/**
	 * Basic Counter: TCP_TOTAL_WRITEBACK_INVALIDATES
	 *
	 * Total number of cache invalidates. Equals TCP_PERF_SEL_TOTAL_WBINVL1+ TCP_PERF_SEL_TOTAL_WBINVL1_VOL+ TCP_PERF_SEL_CP_TCP_INVALIDATE+ TCP_PERF_SEL_SQ_TCP_INVALIDATE_VOL. Not Windowed.
	 */
	if (strcmp(name, "TCP_TOTAL_WRITEBACK_INVALIDATES")==0) {
		basic_counters.emplace(45, BasicCounter{45, "TCP", "TCP_TOTAL_WRITEBACK_INVALIDATES", "Total number of cache invalidates. Equals TCP_PERF_SEL_TOTAL_WBINVL1+ TCP_PERF_SEL_TOTAL_WBINVL1_VOL+ TCP_PERF_SEL_CP_TCP_INVALIDATE+ TCP_PERF_SEL_SQ_TCP_INVALIDATE_VOL. Not Windowed.", "gfx90a"});
		return 45;
	}
	/**
	 * Basic Counter: TCP_UTCL1_REQUEST
	 *
	 * Total CLIENT_UTCL1 NORMAL requests
	 */
	if (strcmp(name, "TCP_UTCL1_REQUEST")==0) {
		basic_counters.emplace(47, BasicCounter{47, "TCP", "TCP_UTCL1_REQUEST", "Total CLIENT_UTCL1 NORMAL requests", "gfx90a"});
		return 47;
	}
	/**
	 * Basic Counter: TCP_UTCL1_TRANSLATION_MISS
	 *
	 * Total utcl1 translation misses
	 */
	if (strcmp(name, "TCP_UTCL1_TRANSLATION_MISS")==0) {
		basic_counters.emplace(48, BasicCounter{48, "TCP", "TCP_UTCL1_TRANSLATION_MISS", "Total utcl1 translation misses", "gfx90a"});
		return 48;
	}
	/**
	 * Basic Counter: TCP_UTCL1_TRANSLATION_HIT
	 *
	 * Total utcl1 translation hits
	 */
	if (strcmp(name, "TCP_UTCL1_TRANSLATION_HIT")==0) {
		basic_counters.emplace(49, BasicCounter{49, "TCP", "TCP_UTCL1_TRANSLATION_HIT", "Total utcl1 translation hits", "gfx90a"});
		return 49;
	}
	/**
	 * Basic Counter: TCP_UTCL1_PERMISSION_MISS
	 *
	 * Total utcl1 permission misses
	 */
	if (strcmp(name, "TCP_UTCL1_PERMISSION_MISS")==0) {
		basic_counters.emplace(50, BasicCounter{50, "TCP", "TCP_UTCL1_PERMISSION_MISS", "Total utcl1 permission misses", "gfx90a"});
		return 50;
	}
	/**
	 * Basic Counter: TCP_TOTAL_CACHE_ACCESSES
	 *
	 * Count of total cache line (tag) accesses (includes hits and misses).
	 */
	if (strcmp(name, "TCP_TOTAL_CACHE_ACCESSES")==0) {
		basic_counters.emplace(60, BasicCounter{60, "TCP", "TCP_TOTAL_CACHE_ACCESSES", "Count of total cache line (tag) accesses (includes hits and misses).", "gfx90a"});
		return 60;
	}
	/**
	 * Basic Counter: TCP_TCP_LATENCY
	 *
	 * Total TCP wave latency (from first clock of wave entering to first clock of wave leaving), divide by TA_TCP_STATE_READ to avg wave latency
	 */
	if (strcmp(name, "TCP_TCP_LATENCY")==0) {
		basic_counters.emplace(65, BasicCounter{65, "TCP", "TCP_TCP_LATENCY", "Total TCP wave latency (from first clock of wave entering to first clock of wave leaving), divide by TA_TCP_STATE_READ to avg wave latency", "gfx90a"});
		return 65;
	}
	/**
	 * Basic Counter: TCP_TCC_READ_REQ_LATENCY
	 *
	 * Total TCP->TCC request latency for reads and atomics with return. Not Windowed.
	 */
	if (strcmp(name, "TCP_TCC_READ_REQ_LATENCY")==0) {
		basic_counters.emplace(66, BasicCounter{66, "TCP", "TCP_TCC_READ_REQ_LATENCY", "Total TCP->TCC request latency for reads and atomics with return. Not Windowed.", "gfx90a"});
		return 66;
	}
	/**
	 * Basic Counter: TCP_TCC_WRITE_REQ_LATENCY
	 *
	 * Total TCP->TCC request latency for writes and atomics without return. Not Windowed.
	 */
	if (strcmp(name, "TCP_TCC_WRITE_REQ_LATENCY")==0) {
		basic_counters.emplace(67, BasicCounter{67, "TCP", "TCP_TCC_WRITE_REQ_LATENCY", "Total TCP->TCC request latency for writes and atomics without return. Not Windowed.", "gfx90a"});
		return 67;
	}
	/**
	 * Basic Counter: TCP_TCC_READ_REQ
	 *
	 * Total read requests from TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_READ_REQ")==0) {
		basic_counters.emplace(69, BasicCounter{69, "TCP", "TCP_TCC_READ_REQ", "Total read requests from TCP to all TCCs", "gfx90a"});
		return 69;
	}
	/**
	 * Basic Counter: TCP_TCC_WRITE_REQ
	 *
	 * Total write requests from TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_WRITE_REQ")==0) {
		basic_counters.emplace(70, BasicCounter{70, "TCP", "TCP_TCC_WRITE_REQ", "Total write requests from TCP to all TCCs", "gfx90a"});
		return 70;
	}
	/**
	 * Basic Counter: TCP_TCC_ATOMIC_WITH_RET_REQ
	 *
	 * Total atomic with return requests from TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_ATOMIC_WITH_RET_REQ")==0) {
		basic_counters.emplace(71, BasicCounter{71, "TCP", "TCP_TCC_ATOMIC_WITH_RET_REQ", "Total atomic with return requests from TCP to all TCCs", "gfx90a"});
		return 71;
	}
	/**
	 * Basic Counter: TCP_TCC_ATOMIC_WITHOUT_RET_REQ
	 *
	 * Total atomic without return requests from TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_ATOMIC_WITHOUT_RET_REQ")==0) {
		basic_counters.emplace(72, BasicCounter{72, "TCP", "TCP_TCC_ATOMIC_WITHOUT_RET_REQ", "Total atomic without return requests from TCP to all TCCs", "gfx90a"});
		return 72;
	}
	/**
	 * Basic Counter: TCP_TCC_NC_READ_REQ
	 *
	 * Total read requests with NC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_NC_READ_REQ")==0) {
		basic_counters.emplace(75, BasicCounter{75, "TCP", "TCP_TCC_NC_READ_REQ", "Total read requests with NC mtype from this TCP to all TCCs", "gfx90a"});
		return 75;
	}
	/**
	 * Basic Counter: TCP_TCC_NC_WRITE_REQ
	 *
	 * Total write requests with NC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_NC_WRITE_REQ")==0) {
		basic_counters.emplace(76, BasicCounter{76, "TCP", "TCP_TCC_NC_WRITE_REQ", "Total write requests with NC mtype from this TCP to all TCCs", "gfx90a"});
		return 76;
	}
	/**
	 * Basic Counter: TCP_TCC_NC_ATOMIC_REQ
	 *
	 * Total atomic requests with NC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_NC_ATOMIC_REQ")==0) {
		basic_counters.emplace(77, BasicCounter{77, "TCP", "TCP_TCC_NC_ATOMIC_REQ", "Total atomic requests with NC mtype from this TCP to all TCCs", "gfx90a"});
		return 77;
	}
	/**
	 * Basic Counter: TCP_TCC_UC_READ_REQ
	 *
	 * Total read requests with UC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_UC_READ_REQ")==0) {
		basic_counters.emplace(78, BasicCounter{78, "TCP", "TCP_TCC_UC_READ_REQ", "Total read requests with UC mtype from this TCP to all TCCs", "gfx90a"});
		return 78;
	}
	/**
	 * Basic Counter: TCP_TCC_UC_WRITE_REQ
	 *
	 * Total write requests with UC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_UC_WRITE_REQ")==0) {
		basic_counters.emplace(79, BasicCounter{79, "TCP", "TCP_TCC_UC_WRITE_REQ", "Total write requests with UC mtype from this TCP to all TCCs", "gfx90a"});
		return 79;
	}
	/**
	 * Basic Counter: TCP_TCC_UC_ATOMIC_REQ
	 *
	 * Total atomic requests with UC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_UC_ATOMIC_REQ")==0) {
		basic_counters.emplace(80, BasicCounter{80, "TCP", "TCP_TCC_UC_ATOMIC_REQ", "Total atomic requests with UC mtype from this TCP to all TCCs", "gfx90a"});
		return 80;
	}
	/**
	 * Basic Counter: TCP_TCC_CC_READ_REQ
	 *
	 * Total write requests with CC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_CC_READ_REQ")==0) {
		basic_counters.emplace(81, BasicCounter{81, "TCP", "TCP_TCC_CC_READ_REQ", "Total write requests with CC mtype from this TCP to all TCCs", "gfx90a"});
		return 81;
	}
	/**
	 * Basic Counter: TCP_TCC_CC_WRITE_REQ
	 *
	 * Total write requests with CC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_CC_WRITE_REQ")==0) {
		basic_counters.emplace(82, BasicCounter{82, "TCP", "TCP_TCC_CC_WRITE_REQ", "Total write requests with CC mtype from this TCP to all TCCs", "gfx90a"});
		return 82;
	}
	/**
	 * Basic Counter: TCP_TCC_CC_ATOMIC_REQ
	 *
	 * Total atomic requests with CC mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_CC_ATOMIC_REQ")==0) {
		basic_counters.emplace(83, BasicCounter{83, "TCP", "TCP_TCC_CC_ATOMIC_REQ", "Total atomic requests with CC mtype from this TCP to all TCCs", "gfx90a"});
		return 83;
	}
	/**
	 * Basic Counter: TCP_TCC_RW_READ_REQ
	 *
	 * Total write requests with RW mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_RW_READ_REQ")==0) {
		basic_counters.emplace(85, BasicCounter{85, "TCP", "TCP_TCC_RW_READ_REQ", "Total write requests with RW mtype from this TCP to all TCCs", "gfx90a"});
		return 85;
	}
	/**
	 * Basic Counter: TCP_TCC_RW_WRITE_REQ
	 *
	 * Total write requests with RW mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_RW_WRITE_REQ")==0) {
		basic_counters.emplace(86, BasicCounter{86, "TCP", "TCP_TCC_RW_WRITE_REQ", "Total write requests with RW mtype from this TCP to all TCCs", "gfx90a"});
		return 86;
	}
	/**
	 * Basic Counter: TCP_TCC_RW_ATOMIC_REQ
	 *
	 * Total atomic requests with RW mtype from this TCP to all TCCs
	 */
	if (strcmp(name, "TCP_TCC_RW_ATOMIC_REQ")==0) {
		basic_counters.emplace(87, BasicCounter{87, "TCP", "TCP_TCC_RW_ATOMIC_REQ", "Total atomic requests with RW mtype from this TCP to all TCCs", "gfx90a"});
		return 87;
	}
	/**
	 * Basic Counter: TCA_CYCLE
	 *
	 * Number of cycles. Not windowable.
	 */
	if (strcmp(name, "TCA_CYCLE")==0) {
		basic_counters.emplace(1, BasicCounter{1, "TCA", "TCA_CYCLE", "Number of cycles. Not windowable.", "gfx90a"});
		return 1;
	}
	/**
	 * Basic Counter: TCA_BUSY
	 *
	 * Number of cycles we have a request pending. Not windowable.
	 */
	if (strcmp(name, "TCA_BUSY")==0) {
		basic_counters.emplace(2, BasicCounter{2, "TCA", "TCA_BUSY", "Number of cycles we have a request pending. Not windowable.", "gfx90a"});
		return 2;
	}
	/**
	 * Basic Counter: TCC_CYCLE
	 *
	 * Number of cycles. Not windowable.
	 */
	if (strcmp(name, "TCC_CYCLE")==0) {
		basic_counters.emplace(1, BasicCounter{1, "TCC", "TCC_CYCLE", "Number of cycles. Not windowable.", "gfx90a"});
		return 1;
	}
	/**
	 * Basic Counter: TCC_BUSY
	 *
	 * Number of cycles we have a request pending. Not windowable.
	 */
	if (strcmp(name, "TCC_BUSY")==0) {
		basic_counters.emplace(2, BasicCounter{2, "TCC", "TCC_BUSY", "Number of cycles we have a request pending. Not windowable.", "gfx90a"});
		return 2;
	}
	/**
	 * Basic Counter: TCC_REQ
	 *
	 * Number of requests of all types. This is measured at the tag block. This may be more than the number of requests arriving at the TCC, but it is a good indication of the total amount of work that needs to be performed.
	 */
	if (strcmp(name, "TCC_REQ")==0) {
		basic_counters.emplace(3, BasicCounter{3, "TCC", "TCC_REQ", "Number of requests of all types. This is measured at the tag block. This may be more than the number of requests arriving at the TCC, but it is a good indication of the total amount of work that needs to be performed.", "gfx90a"});
		return 3;
	}
	/**
	 * Basic Counter: TCC_STREAMING_REQ
	 *
	 * Number of streaming requests. This is measured at the tag block.
	 */
	if (strcmp(name, "TCC_STREAMING_REQ")==0) {
		basic_counters.emplace(4, BasicCounter{4, "TCC", "TCC_STREAMING_REQ", "Number of streaming requests. This is measured at the tag block.", "gfx90a"});
		return 4;
	}
	/**
	 * Basic Counter: TCC_NC_REQ
	 *
	 * The number of noncoherently cached requests. This is measured at the tag block.
	 */
	if (strcmp(name, "TCC_NC_REQ")==0) {
		basic_counters.emplace(5, BasicCounter{5, "TCC", "TCC_NC_REQ", "The number of noncoherently cached requests. This is measured at the tag block.", "gfx90a"});
		return 5;
	}
	/**
	 * Basic Counter: TCC_UC_REQ
	 *
	 * The number of uncached requests. This is measured at the tag block.
	 */
	if (strcmp(name, "TCC_UC_REQ")==0) {
		basic_counters.emplace(6, BasicCounter{6, "TCC", "TCC_UC_REQ", "The number of uncached requests. This is measured at the tag block.", "gfx90a"});
		return 6;
	}
	/**
	 * Basic Counter: TCC_CC_REQ
	 *
	 * The number of coherently cached requests. This is measured at the tag block.
	 */
	if (strcmp(name, "TCC_CC_REQ")==0) {
		basic_counters.emplace(7, BasicCounter{7, "TCC", "TCC_CC_REQ", "The number of coherently cached requests. This is measured at the tag block.", "gfx90a"});
		return 7;
	}
	/**
	 * Basic Counter: TCC_RW_REQ
	 *
	 * The number of RW requests. This is measured at the tag block.
	 */
	if (strcmp(name, "TCC_RW_REQ")==0) {
		basic_counters.emplace(8, BasicCounter{8, "TCC", "TCC_RW_REQ", "The number of RW requests. This is measured at the tag block.", "gfx90a"});
		return 8;
	}
	/**
	 * Basic Counter: TCC_PROBE
	 *
	 * Number of probe requests. Not windowable.
	 */
	if (strcmp(name, "TCC_PROBE")==0) {
		basic_counters.emplace(9, BasicCounter{9, "TCC", "TCC_PROBE", "Number of probe requests. Not windowable.", "gfx90a"});
		return 9;
	}
	/**
	 * Basic Counter: TCC_PROBE_ALL
	 *
	 * Number of external probe requests with with EA_TCC_preq_all== 1. Not windowable.
	 */
	if (strcmp(name, "TCC_PROBE_ALL")==0) {
		basic_counters.emplace(10, BasicCounter{10, "TCC", "TCC_PROBE_ALL", "Number of external probe requests with with EA_TCC_preq_all== 1. Not windowable.", "gfx90a"});
		return 10;
	}
	/**
	 * Basic Counter: TCC_READ
	 *
	 * Number of read requests. Compressed reads are included in this, but metadata reads are not included.
	 */
	if (strcmp(name, "TCC_READ")==0) {
		basic_counters.emplace(12, BasicCounter{12, "TCC", "TCC_READ", "Number of read requests. Compressed reads are included in this, but metadata reads are not included.", "gfx90a"});
		return 12;
	}
	/**
	 * Basic Counter: TCC_WRITE
	 *
	 * Number of write requests.
	 */
	if (strcmp(name, "TCC_WRITE")==0) {
		basic_counters.emplace(13, BasicCounter{13, "TCC", "TCC_WRITE", "Number of write requests.", "gfx90a"});
		return 13;
	}
	/**
	 * Basic Counter: TCC_ATOMIC
	 *
	 * Number of atomic requests of all types.
	 */
	if (strcmp(name, "TCC_ATOMIC")==0) {
		basic_counters.emplace(14, BasicCounter{14, "TCC", "TCC_ATOMIC", "Number of atomic requests of all types.", "gfx90a"});
		return 14;
	}
	/**
	 * Basic Counter: TCC_HIT
	 *
	 * Number of cache hits.
	 */
	if (strcmp(name, "TCC_HIT")==0) {
		basic_counters.emplace(17, BasicCounter{17, "TCC", "TCC_HIT", "Number of cache hits.", "gfx90a"});
		return 17;
	}
	/**
	 * Basic Counter: TCC_MISS
	 *
	 * Number of cache misses. UC reads count as misses.
	 */
	if (strcmp(name, "TCC_MISS")==0) {
		basic_counters.emplace(19, BasicCounter{19, "TCC", "TCC_MISS", "Number of cache misses. UC reads count as misses.", "gfx90a"});
		return 19;
	}
	/**
	 * Basic Counter: TCC_WRITEBACK
	 *
	 * Number of lines written back to main memory. This includes writebacks of dirty lines and uncached write/atomic requests.
	 */
	if (strcmp(name, "TCC_WRITEBACK")==0) {
		basic_counters.emplace(22, BasicCounter{22, "TCC", "TCC_WRITEBACK", "Number of lines written back to main memory. This includes writebacks of dirty lines and uncached write/atomic requests.", "gfx90a"});
		return 22;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ
	 *
	 * Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.
	 */
	if (strcmp(name, "TCC_EA_WRREQ")==0) {
		basic_counters.emplace(26, BasicCounter{26, "TCC", "TCC_EA_WRREQ", "Number of transactions (either 32-byte or 64-byte) going over the TC_EA_wrreq interface. Atomics may travel over the same interface and are generally classified as write requests. This does not include probe commands.", "gfx90a"});
		return 26;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_64B
	 *
	 * Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_64B")==0) {
		basic_counters.emplace(27, BasicCounter{27, "TCC", "TCC_EA_WRREQ_64B", "Number of 64-byte transactions going (64-byte write or CMPSWAP) over the TC_EA_wrreq interface.", "gfx90a"});
		return 27;
	}
	/**
	 * Basic Counter: TCC_EA_WR_UNCACHED_32B
	 *
	 * Number of 32-byte write/atomic going over the TC_EA_wrreq interface due to uncached traffic. Note that CC mtypes can produce uncached requests, and those are included in this. A 64-byte request will be counted as 2
	 */
	if (strcmp(name, "TCC_EA_WR_UNCACHED_32B")==0) {
		basic_counters.emplace(29, BasicCounter{29, "TCC", "TCC_EA_WR_UNCACHED_32B", "Number of 32-byte write/atomic going over the TC_EA_wrreq interface due to uncached traffic. Note that CC mtypes can produce uncached requests, and those are included in this. A 64-byte request will be counted as 2", "gfx90a"});
		return 29;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_STALL
	 *
	 * Number of cycles a write request was stalled.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_STALL")==0) {
		basic_counters.emplace(30, BasicCounter{30, "TCC", "TCC_EA_WRREQ_STALL", "Number of cycles a write request was stalled.", "gfx90a"});
		return 30;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_IO_CREDIT_STALL
	 *
	 * Number of cycles a EA write request was stalled because the interface was out of IO credits.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_IO_CREDIT_STALL")==0) {
		basic_counters.emplace(31, BasicCounter{31, "TCC", "TCC_EA_WRREQ_IO_CREDIT_STALL", "Number of cycles a EA write request was stalled because the interface was out of IO credits.", "gfx90a"});
		return 31;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_GMI_CREDIT_STALL
	 *
	 * Number of cycles a EA write request was stalled because the interface was out of GMI credits.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_GMI_CREDIT_STALL")==0) {
		basic_counters.emplace(32, BasicCounter{32, "TCC", "TCC_EA_WRREQ_GMI_CREDIT_STALL", "Number of cycles a EA write request was stalled because the interface was out of GMI credits.", "gfx90a"});
		return 32;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_DRAM_CREDIT_STALL
	 *
	 * Number of cycles a EA write request was stalled because the interface was out of DRAM credits.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_DRAM_CREDIT_STALL")==0) {
		basic_counters.emplace(33, BasicCounter{33, "TCC", "TCC_EA_WRREQ_DRAM_CREDIT_STALL", "Number of cycles a EA write request was stalled because the interface was out of DRAM credits.", "gfx90a"});
		return 33;
	}
	/**
	 * Basic Counter: TCC_TOO_MANY_EA_WRREQS_STALL
	 *
	 * Number of cycles the TCC could not send a EA write request because it already reached its maximum number of pending EA write requests.
	 */
	if (strcmp(name, "TCC_TOO_MANY_EA_WRREQS_STALL")==0) {
		basic_counters.emplace(34, BasicCounter{34, "TCC", "TCC_TOO_MANY_EA_WRREQS_STALL", "Number of cycles the TCC could not send a EA write request because it already reached its maximum number of pending EA write requests.", "gfx90a"});
		return 34;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_LEVEL
	 *
	 * The sum of the number of EA write requests in flight. This is primarily meant for measure average EA write latency. Average write latency = TCC_PERF_SEL_EA_WRREQ_LEVEL/TCC_PERF_SEL_EA_WRREQ.
	 */
	if (strcmp(name, "TCC_EA_WRREQ_LEVEL")==0) {
		basic_counters.emplace(35, BasicCounter{35, "TCC", "TCC_EA_WRREQ_LEVEL", "The sum of the number of EA write requests in flight. This is primarily meant for measure average EA write latency. Average write latency = TCC_PERF_SEL_EA_WRREQ_LEVEL/TCC_PERF_SEL_EA_WRREQ.", "gfx90a"});
		return 35;
	}
	/**
	 * Basic Counter: TCC_EA_ATOMIC
	 *
	 * Number of transactions going over the TC_EA_wrreq interface that are actually atomic requests.
	 */
	if (strcmp(name, "TCC_EA_ATOMIC")==0) {
		basic_counters.emplace(36, BasicCounter{36, "TCC", "TCC_EA_ATOMIC", "Number of transactions going over the TC_EA_wrreq interface that are actually atomic requests.", "gfx90a"});
		return 36;
	}
	/**
	 * Basic Counter: TCC_EA_ATOMIC_LEVEL
	 *
	 * The sum of the number of EA atomics in flight. This is primarily meant for measure average EA atomic latency. Average atomic latency = TCC_PERF_SEL_EA_WRREQ_ATOMIC_LEVEL/TCC_PERF_SEL_EA_WRREQ_ATOMIC.
	 */
	if (strcmp(name, "TCC_EA_ATOMIC_LEVEL")==0) {
		basic_counters.emplace(37, BasicCounter{37, "TCC", "TCC_EA_ATOMIC_LEVEL", "The sum of the number of EA atomics in flight. This is primarily meant for measure average EA atomic latency. Average atomic latency = TCC_PERF_SEL_EA_WRREQ_ATOMIC_LEVEL/TCC_PERF_SEL_EA_WRREQ_ATOMIC.", "gfx90a"});
		return 37;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ
	 *
	 * Number of TCC/EA read requests (either 32-byte or 64-byte)
	 */
	if (strcmp(name, "TCC_EA_RDREQ")==0) {
		basic_counters.emplace(38, BasicCounter{38, "TCC", "TCC_EA_RDREQ", "Number of TCC/EA read requests (either 32-byte or 64-byte)", "gfx90a"});
		return 38;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_32B
	 *
	 * Number of 32-byte TCC/EA read requests
	 */
	if (strcmp(name, "TCC_EA_RDREQ_32B")==0) {
		basic_counters.emplace(39, BasicCounter{39, "TCC", "TCC_EA_RDREQ_32B", "Number of 32-byte TCC/EA read requests", "gfx90a"});
		return 39;
	}
	/**
	 * Basic Counter: TCC_EA_RD_UNCACHED_32B
	 *
	 * Number of 32-byte TCC/EA read due to uncached traffic. A 64-byte request will be counted as 2
	 */
	if (strcmp(name, "TCC_EA_RD_UNCACHED_32B")==0) {
		basic_counters.emplace(40, BasicCounter{40, "TCC", "TCC_EA_RD_UNCACHED_32B", "Number of 32-byte TCC/EA read due to uncached traffic. A 64-byte request will be counted as 2", "gfx90a"});
		return 40;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_IO_CREDIT_STALL
	 *
	 * Number of cycles there was a stall because the read request interface was out of IO credits. Stalls occur regardless of whether a read needed to be performed or not.
	 */
	if (strcmp(name, "TCC_EA_RDREQ_IO_CREDIT_STALL")==0) {
		basic_counters.emplace(41, BasicCounter{41, "TCC", "TCC_EA_RDREQ_IO_CREDIT_STALL", "Number of cycles there was a stall because the read request interface was out of IO credits. Stalls occur regardless of whether a read needed to be performed or not.", "gfx90a"});
		return 41;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_GMI_CREDIT_STALL
	 *
	 * Number of cycles there was a stall because the read request interface was out of GMI credits. Stalls occur regardless of whether a read needed to be performed or not.
	 */
	if (strcmp(name, "TCC_EA_RDREQ_GMI_CREDIT_STALL")==0) {
		basic_counters.emplace(42, BasicCounter{42, "TCC", "TCC_EA_RDREQ_GMI_CREDIT_STALL", "Number of cycles there was a stall because the read request interface was out of GMI credits. Stalls occur regardless of whether a read needed to be performed or not.", "gfx90a"});
		return 42;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_DRAM_CREDIT_STALL
	 *
	 * Number of cycles there was a stall because the read request interface was out of DRAM credits. Stalls occur regardless of whether a read needed to be performed or not.
	 */
	if (strcmp(name, "TCC_EA_RDREQ_DRAM_CREDIT_STALL")==0) {
		basic_counters.emplace(43, BasicCounter{43, "TCC", "TCC_EA_RDREQ_DRAM_CREDIT_STALL", "Number of cycles there was a stall because the read request interface was out of DRAM credits. Stalls occur regardless of whether a read needed to be performed or not.", "gfx90a"});
		return 43;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_LEVEL
	 *
	 * The sum of the number of TCC/EA read requests in flight. This is primarily meant for measure average EA read latency. Average read latency = TCC_PERF_SEL_EA_RDREQ_LEVEL/TCC_PERF_SEL_EA_RDREQ.
	 */
	if (strcmp(name, "TCC_EA_RDREQ_LEVEL")==0) {
		basic_counters.emplace(44, BasicCounter{44, "TCC", "TCC_EA_RDREQ_LEVEL", "The sum of the number of TCC/EA read requests in flight. This is primarily meant for measure average EA read latency. Average read latency = TCC_PERF_SEL_EA_RDREQ_LEVEL/TCC_PERF_SEL_EA_RDREQ.", "gfx90a"});
		return 44;
	}
	/**
	 * Basic Counter: TCC_TAG_STALL
	 *
	 * Number of cycles the normal request pipeline in the tag was stalled for any reason. Normally, stalls of this nature are measured exactly from one point the pipeline, but that is not the case for this counter. Probes can stall the pipeline at a variety of places, and there is no single point that can reasonably measure the total stalls accurately.
	 */
	if (strcmp(name, "TCC_TAG_STALL")==0) {
		basic_counters.emplace(45, BasicCounter{45, "TCC", "TCC_TAG_STALL", "Number of cycles the normal request pipeline in the tag was stalled for any reason. Normally, stalls of this nature are measured exactly from one point the pipeline, but that is not the case for this counter. Probes can stall the pipeline at a variety of places, and there is no single point that can reasonably measure the total stalls accurately.", "gfx90a"});
		return 45;
	}
	/**
	 * Basic Counter: TCC_NORMAL_WRITEBACK
	 *
	 * Number of writebacks due to requests that are not writeback requests.
	 */
	if (strcmp(name, "TCC_NORMAL_WRITEBACK")==0) {
		basic_counters.emplace(68, BasicCounter{68, "TCC", "TCC_NORMAL_WRITEBACK", "Number of writebacks due to requests that are not writeback requests.", "gfx90a"});
		return 68;
	}
	/**
	 * Basic Counter: TCC_ALL_TC_OP_WB_WRITEBACK
	 *
	 * Number of writebacks due to all TC_OP writeback requests.
	 */
	if (strcmp(name, "TCC_ALL_TC_OP_WB_WRITEBACK")==0) {
		basic_counters.emplace(73, BasicCounter{73, "TCC", "TCC_ALL_TC_OP_WB_WRITEBACK", "Number of writebacks due to all TC_OP writeback requests.", "gfx90a"});
		return 73;
	}
	/**
	 * Basic Counter: TCC_NORMAL_EVICT
	 *
	 * Number of evictions due to requests that are not invalidate or probe requests.
	 */
	if (strcmp(name, "TCC_NORMAL_EVICT")==0) {
		basic_counters.emplace(74, BasicCounter{74, "TCC", "TCC_NORMAL_EVICT", "Number of evictions due to requests that are not invalidate or probe requests.", "gfx90a"});
		return 74;
	}
	/**
	 * Basic Counter: TCC_ALL_TC_OP_INV_EVICT
	 *
	 * Number of evictions due to all TC_OP invalidate requests.
	 */
	if (strcmp(name, "TCC_ALL_TC_OP_INV_EVICT")==0) {
		basic_counters.emplace(80, BasicCounter{80, "TCC", "TCC_ALL_TC_OP_INV_EVICT", "Number of evictions due to all TC_OP invalidate requests.", "gfx90a"});
		return 80;
	}
	/**
	 * Basic Counter: TCC_EA_RDREQ_DRAM
	 *
	 * Number of TCC/EA read requests (either 32-byte or 64-byte) destined for DRAM (MC).
	 */
	if (strcmp(name, "TCC_EA_RDREQ_DRAM")==0) {
		basic_counters.emplace(102, BasicCounter{102, "TCC", "TCC_EA_RDREQ_DRAM", "Number of TCC/EA read requests (either 32-byte or 64-byte) destined for DRAM (MC).", "gfx90a"});
		return 102;
	}
	/**
	 * Basic Counter: TCC_EA_WRREQ_DRAM
	 *
	 * Number of TCC/EA write requests (either 32-byte of 64-byte) destined for DRAM (MC).
	 */
	if (strcmp(name, "TCC_EA_WRREQ_DRAM")==0) {
		basic_counters.emplace(103, BasicCounter{103, "TCC", "TCC_EA_WRREQ_DRAM", "Number of TCC/EA write requests (either 32-byte of 64-byte) destined for DRAM (MC).", "gfx90a"});
		return 103;
	}
	}

	/**
	 * @}
	 */

	/**
	 * @brief Basic gfx940 counters
	 *
	 * @{
	 */
	gpu = "gfx940";

	if (strncmp(gpu_name, gpu.c_str(), gpu.length())==0) {
	/**
	 * Basic Counter: SQ_WAIT_INST_LDS
	 *
	 * Number of quad-cycles spent waiting for LDS instruction issue. (per-simd, nondeterministic)
	 */
	if (strcmp(name, "SQ_WAIT_INST_LDS")==0) {
		basic_counters.emplace(96, BasicCounter{96, "SQ", "SQ_WAIT_INST_LDS", "Number of quad-cycles spent waiting for LDS instruction issue. (per-simd, nondeterministic)", "gfx940"});
		return 96;
	}
	/**
	 * Basic Counter: TCP_TCP_TA_DATA_STALL_CYCLES
	 *
	 * TCP stalls TA data interface. Not Windowed.
	 */
	if (strcmp(name, "TCP_TCP_TA_DATA_STALL_CYCLES")==0) {
		basic_counters.emplace(6, BasicCounter{6, "TCP", "TCP_TCP_TA_DATA_STALL_CYCLES", "TCP stalls TA data interface. Not Windowed.", "gfx940"});
		return 6;
	}
	/**
	 * Basic Counter: GRBM_COUNT
	 *
	 * Tie High
	 */
	if (strcmp(name, "GRBM_COUNT")==0) {
		basic_counters.emplace(0, BasicCounter{0, "GRBM", "GRBM_COUNT", "Tie High", "gfx940"});
		return 0;
	}
	/**
	 * Basic Counter: GRBM_GUI_ACTIVE
	 *
	 * The GUI is Active
	 */
	if (strcmp(name, "GRBM_GUI_ACTIVE")==0) {
		basic_counters.emplace(2, BasicCounter{2, "GRBM", "GRBM_GUI_ACTIVE", "The GUI is Active", "gfx940"});
		return 2;
	}
	/**
	 * Basic Counter: GRBM_CP_BUSY
	 *
	 * Any of the Command Processor (CPG/CPC/CPF) blocks are busy.
	 */
	if (strcmp(name, "GRBM_CP_BUSY")==0) {
		basic_counters.emplace(3, BasicCounter{3, "GRBM", "GRBM_CP_BUSY", "Any of the Command Processor (CPG/CPC/CPF) blocks are busy.", "gfx940"});
		return 3;
	}
	/**
	 * Basic Counter: GRBM_SPI_BUSY
	 *
	 * Any of the Shader Pipe Interpolators (SPI) are busy in the shader engine(s).
	 */
	if (strcmp(name, "GRBM_SPI_BUSY")==0) {
		basic_counters.emplace(11, BasicCounter{11, "GRBM", "GRBM_SPI_BUSY", "Any of the Shader Pipe Interpolators (SPI) are busy in the shader engine(s).", "gfx940"});
		return 11;
	}
	/**
	 * Basic Counter: GRBM_TA_BUSY
	 *
	 * Any of the Texture Pipes (TA) are busy in the shader engine(s).
	 */
	if (strcmp(name, "GRBM_TA_BUSY")==0) {
		basic_counters.emplace(13, BasicCounter{13, "GRBM", "GRBM_TA_BUSY", "Any of the Texture Pipes (TA) are busy in the shader engine(s).", "gfx940"});
		return 13;
	}
	/**
	 * Basic Counter: GRBM_TC_BUSY
	 *
	 * Any of the Texture Cache Blocks (TCP/TCI/TCA/TCC) are busy.
	 */
	if (strcmp(name, "GRBM_TC_BUSY")==0) {
		basic_counters.emplace(28, BasicCounter{28, "GRBM", "GRBM_TC_BUSY", "Any of the Texture Cache Blocks (TCP/TCI/TCA/TCC) are busy.", "gfx940"});
		return 28;
	}
	/**
	 * Basic Counter: GRBM_CPC_BUSY
	 *
	 * The Command Processor Compute (CPC) is busy.
	 */
	if (strcmp(name, "GRBM_CPC_BUSY")==0) {
		basic_counters.emplace(30, BasicCounter{30, "GRBM", "GRBM_CPC_BUSY", "The Command Processor Compute (CPC) is busy.", "gfx940"});
		return 30;
	}
	/**
	 * Basic Counter: GRBM_CPF_BUSY
	 *
	 * The Command Processor Fetchers (CPF) is busy.
	 */
	if (strcmp(name, "GRBM_CPF_BUSY")==0) {
		basic_counters.emplace(31, BasicCounter{31, "GRBM", "GRBM_CPF_BUSY", "The Command Processor Fetchers (CPF) is busy.", "gfx940"});
		return 31;
	}
	/**
	 * Basic Counter: GRBM_UTCL2_BUSY
	 *
	 * The Unified Translation Cache Level-2 (UTCL2) block is busy.
	 */
	if (strcmp(name, "GRBM_UTCL2_BUSY")==0) {
		basic_counters.emplace(34, BasicCounter{34, "GRBM", "GRBM_UTCL2_BUSY", "The Unified Translation Cache Level-2 (UTCL2) block is busy.", "gfx940"});
		return 34;
	}
	/**
	 * Basic Counter: GRBM_EA_BUSY
	 *
	 * The Efficiency Arbiter (EA) block is busy.
	 */
	if (strcmp(name, "GRBM_EA_BUSY")==0) {
		basic_counters.emplace(35, BasicCounter{35, "GRBM", "GRBM_EA_BUSY", "The Efficiency Arbiter (EA) block is busy.", "gfx940"});
		return 35;
	}
	/**
	 * Basic Counter: CPC_ME1_BUSY_FOR_PACKET_DECODE
	 *
	 * Me1 busy for packet decode.
	 */
	if (strcmp(name, "CPC_ME1_BUSY_FOR_PACKET_DECODE")==0) {
		basic_counters.emplace(13, BasicCounter{13, "CPC", "CPC_ME1_BUSY_FOR_PACKET_DECODE", "Me1 busy for packet decode.", "gfx940"});
		return 13;
	}
	/**
	 * Basic Counter: CPC_UTCL1_STALL_ON_TRANSLATION
	 *
	 * One of the UTCL1s is stalled waiting on translation, XNACK or PENDING response.
	 */
	if (strcmp(name, "CPC_UTCL1_STALL_ON_TRANSLATION")==0) {
		basic_counters.emplace(24, BasicCounter{24, "CPC", "CPC_UTCL1_STALL_ON_TRANSLATION", "One of the UTCL1s is stalled waiting on translation, XNACK or PENDING response.", "gfx940"});
		return 24;
	}
	/**
	 * Basic Counter: CPC_CPC_STAT_BUSY
	 *
	 * CPC Busy.
	 */
	if (strcmp(name, "CPC_CPC_STAT_BUSY")==0) {
		basic_counters.emplace(25, BasicCounter{25, "CPC", "CPC_CPC_STAT_BUSY", "CPC Busy.", "gfx940"});
		return 25;
	}
	/**
	 * Basic Counter: CPC_CPC_STAT_IDLE
	 *
	 * CPC Idle.
	 */
	if (strcmp(name, "CPC_CPC_STAT_IDLE")==0) {
		basic_counters.emplace(26, BasicCounter{26, "CPC", "CPC_CPC_STAT_IDLE", "CPC Idle.", "gfx940"});
		return 26;
	}
	/**
	 * Basic Counter: CPC_CPC_STAT_STALL
	 *
	 * CPC Stalled.
	 */
	if (strcmp(name, "CPC_CPC_STAT_STALL")==0) {
		basic_counters.emplace(27, BasicCounter{27, "CPC", "CPC_CPC_STAT_STALL", "CPC Stalled.", "gfx940"});
		return 27;
	}
	/**
	 * Basic Counter: CPC_CPC_TCIU_BUSY
	 *
	 * CPC TCIU interface Busy.
	 */
	if (strcmp(name, "CPC_CPC_TCIU_BUSY")==0) {
		basic_counters.emplace(28, BasicCounter{28, "CPC", "CPC_CPC_TCIU_BUSY", "CPC TCIU interface Busy.", "gfx940"});
		return 28;
	}
	/**
	 * Basic Counter: CPC_CPC_TCIU_IDLE
	 *
	 * CPC TCIU interface Idle.
	 */
	if (strcmp(name, "CPC_CPC_TCIU_IDLE")==0) {
		basic_counters.emplace(29, BasicCounter{29, "CPC", "CPC_CPC_TCIU_IDLE", "CPC TCIU interface Idle.", "gfx940"});
		return 29;
	}
	/**
	 * Basic Counter: CPC_CPC_UTCL2IU_BUSY
	 *
	 * CPC UTCL2 interface Busy.
	 */
	if (strcmp(name, "CPC_CPC_UTCL2IU_BUSY")==0) {
		basic_counters.emplace(30, BasicCounter{30, "CPC", "CPC_CPC_UTCL2IU_BUSY", "CPC UTCL2 interface Busy.", "gfx940"});
		return 30;
	}
	/**
	 * Basic Counter: CPC_CPC_UTCL2IU_IDLE
	 *
	 * CPC UTCL2 interface Idle.
	 */
	if (strcmp(name, "CPC_CPC_UTCL2IU_IDLE")==0) {
		basic_counters.emplace(31, BasicCounter{31, "CPC", "CPC_CPC_UTCL2IU_IDLE", "CPC UTCL2 interface Idle.", "gfx940"});
		return 31;
	}
	/**
	 * Basic Counter: CPC_CPC_UTCL2IU_STALL
	 *
	 * CPC UTCL2 interface Stalled waiting on Free, Tags or Translation.
	 */
	if (strcmp(name, "CPC_CPC_UTCL2IU_STALL")==0) {
		basic_counters.emplace(32, BasicCounter{32, "CPC", "CPC_CPC_UTCL2IU_STALL", "CPC UTCL2 interface Stalled waiting on Free, Tags or Translation.", "gfx940"});
		return 32;
	}
	/**
	 * Basic Counter: CPC_ME1_DC0_SPI_BUSY
	 *
	 * CPC Me1 Processor Busy.
	 */
	if (strcmp(name, "CPC_ME1_DC0_SPI_BUSY")==0) {
		basic_counters.emplace(33, BasicCounter{33, "CPC", "CPC_ME1_DC0_SPI_BUSY", "CPC Me1 Processor Busy.", "gfx940"});
		return 33;
	}
	/**
	 * Basic Counter: CPF_CMP_UTCL1_STALL_ON_TRANSLATION
	 *
	 * One of the Compute UTCL1s is stalled waiting on translation, XNACK or PENDING response.
	 */
	if (strcmp(name, "CPF_CMP_UTCL1_STALL_ON_TRANSLATION")==0) {
		basic_counters.emplace(20, BasicCounter{20, "CPF", "CPF_CMP_UTCL1_STALL_ON_TRANSLATION", "One of the Compute UTCL1s is stalled waiting on translation, XNACK or PENDING response.", "gfx940"});
		return 20;
	}
	/**
	 * Basic Counter: CPF_CPF_STAT_BUSY
	 *
	 * CPF Busy.
	 */
	if (strcmp(name, "CPF_CPF_STAT_BUSY")==0) {
		basic_counters.emplace(23, BasicCounter{23, "CPF", "CPF_CPF_STAT_BUSY", "CPF Busy.", "gfx940"});
		return 23;
	}
	/**
	 * Basic Counter: CPF_CPF_STAT_IDLE
	 *
	 * CPF Idle.
	 */
	if (strcmp(name, "CPF_CPF_STAT_IDLE")==0) {
		basic_counters.emplace(24, BasicCounter{24, "CPF", "CPF_CPF_STAT_IDLE", "CPF Idle.", "gfx940"});
		return 24;
	}
	/**
	 * Basic Counter: CPF_CPF_STAT_STALL
	 *
	 * CPF Stalled.
	 */
	if (strcmp(name, "CPF_CPF_STAT_STALL")==0) {
		basic_counters.emplace(25, BasicCounter{25, "CPF", "CPF_CPF_STAT_STALL", "CPF Stalled.", "gfx940"});
		return 25;
	}
	/**
	 * Basic Counter: CPF_CPF_TCIU_BUSY
	 *
	 * CPF TCIU interface Busy.
	 */
	if (strcmp(name, "CPF_CPF_TCIU_BUSY")==0) {
		basic_counters.emplace(26, BasicCounter{26, "CPF", "CPF_CPF_TCIU_BUSY", "CPF TCIU interface Busy.", "gfx940"});
		return 26;
	}
	/**
	 * Basic Counter: CPF_CPF_TCIU_IDLE
	 *
	 * CPF TCIU interface Idle.
	 */
	if (strcmp(name, "CPF_CPF_TCIU_IDLE")==0) {
		basic_counters.emplace(27, BasicCounter{27, "CPF", "CPF_CPF_TCIU_IDLE", "CPF TCIU interface Idle.", "gfx940"});
		return 27;
	}
	/**
	 * Basic Counter: CPF_CPF_TCIU_STALL
	 *
	 * CPF TCIU interface Stalled waiting on Free, Tags.
	 */
	if (strcmp(name, "CPF_CPF_TCIU_STALL")==0) {
		basic_counters.emplace(28, BasicCounter{28, "CPF", "CPF_CPF_TCIU_STALL", "CPF TCIU interface Stalled waiting on Free, Tags.", "gfx940"});
		return 28;
	}
	/**
	 * Basic Counter: SPI_CSN_WINDOW_VALID
	 *
	 * Clock count enabled by perfcounter_start event. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_WINDOW_VALID")==0) {
		basic_counters.emplace(47, BasicCounter{47, "SPI", "SPI_CSN_WINDOW_VALID", "Clock count enabled by perfcounter_start event. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx940"});
		return 47;
	}
	/**
	 * Basic Counter: SPI_CSN_BUSY
	 *
	 * Number of clocks with outstanding waves (SPI or SH). Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_BUSY")==0) {
		basic_counters.emplace(48, BasicCounter{48, "SPI", "SPI_CSN_BUSY", "Number of clocks with outstanding waves (SPI or SH). Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx940"});
		return 48;
	}
	/**
	 * Basic Counter: SPI_CSN_NUM_THREADGROUPS
	 *
	 * Number of threadgroups launched. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_NUM_THREADGROUPS")==0) {
		basic_counters.emplace(49, BasicCounter{49, "SPI", "SPI_CSN_NUM_THREADGROUPS", "Number of threadgroups launched. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx940"});
		return 49;
	}
	/**
	 * Basic Counter: SPI_CSN_WAVE
	 *
	 * Number of waves. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_CSN_WAVE")==0) {
		basic_counters.emplace(52, BasicCounter{52, "SPI", "SPI_CSN_WAVE", "Number of waves. Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx940"});
		return 52;
	}
	/**
	 * Basic Counter: SPI_RA_REQ_NO_ALLOC
	 *
	 * Arb cycles with requests but no allocation. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_REQ_NO_ALLOC")==0) {
		basic_counters.emplace(79, BasicCounter{79, "SPI", "SPI_RA_REQ_NO_ALLOC", "Arb cycles with requests but no allocation. Source is RA0", "gfx940"});
		return 79;
	}
	/**
	 * Basic Counter: SPI_RA_REQ_NO_ALLOC_CSN
	 *
	 * Arb cycles with CSn req and no CSn alloc. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_REQ_NO_ALLOC_CSN")==0) {
		basic_counters.emplace(85, BasicCounter{85, "SPI", "SPI_RA_REQ_NO_ALLOC_CSN", "Arb cycles with CSn req and no CSn alloc. Source is RA0", "gfx940"});
		return 85;
	}
	/**
	 * Basic Counter: SPI_RA_RES_STALL_CSN
	 *
	 * Arb cycles with CSn req and no CSn fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_RES_STALL_CSN")==0) {
		basic_counters.emplace(91, BasicCounter{91, "SPI", "SPI_RA_RES_STALL_CSN", "Arb cycles with CSn req and no CSn fits. Source is RA0", "gfx940"});
		return 91;
	}
	/**
	 * Basic Counter: SPI_RA_TMP_STALL_CSN
	 *
	 * Cycles where csn wants to req but does not fit in temp space.
	 */
	if (strcmp(name, "SPI_RA_TMP_STALL_CSN")==0) {
		basic_counters.emplace(97, BasicCounter{97, "SPI", "SPI_RA_TMP_STALL_CSN", "Cycles where csn wants to req but does not fit in temp space.", "gfx940"});
		return 97;
	}
	/**
	 * Basic Counter: SPI_RA_WAVE_SIMD_FULL_CSN
	 *
	 * Sum of SIMD where WAVE can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_WAVE_SIMD_FULL_CSN")==0) {
		basic_counters.emplace(103, BasicCounter{103, "SPI", "SPI_RA_WAVE_SIMD_FULL_CSN", "Sum of SIMD where WAVE can't take csn wave when !fits. Source is RA0", "gfx940"});
		return 103;
	}
	/**
	 * Basic Counter: SPI_RA_VGPR_SIMD_FULL_CSN
	 *
	 * Sum of SIMD where VGPR can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_VGPR_SIMD_FULL_CSN")==0) {
		basic_counters.emplace(109, BasicCounter{109, "SPI", "SPI_RA_VGPR_SIMD_FULL_CSN", "Sum of SIMD where VGPR can't take csn wave when !fits. Source is RA0", "gfx940"});
		return 109;
	}
	/**
	 * Basic Counter: SPI_RA_SGPR_SIMD_FULL_CSN
	 *
	 * Sum of SIMD where SGPR can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_SGPR_SIMD_FULL_CSN")==0) {
		basic_counters.emplace(115, BasicCounter{115, "SPI", "SPI_RA_SGPR_SIMD_FULL_CSN", "Sum of SIMD where SGPR can't take csn wave when !fits. Source is RA0", "gfx940"});
		return 115;
	}
	/**
	 * Basic Counter: SPI_RA_LDS_CU_FULL_CSN
	 *
	 * Sum of CU where LDS can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_LDS_CU_FULL_CSN")==0) {
		basic_counters.emplace(120, BasicCounter{120, "SPI", "SPI_RA_LDS_CU_FULL_CSN", "Sum of CU where LDS can't take csn wave when !fits. Source is RA0", "gfx940"});
		return 120;
	}
	/**
	 * Basic Counter: SPI_RA_BAR_CU_FULL_CSN
	 *
	 * Sum of CU where BARRIER can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_BAR_CU_FULL_CSN")==0) {
		basic_counters.emplace(123, BasicCounter{123, "SPI", "SPI_RA_BAR_CU_FULL_CSN", "Sum of CU where BARRIER can't take csn wave when !fits. Source is RA0", "gfx940"});
		return 123;
	}
	/**
	 * Basic Counter: SPI_RA_BULKY_CU_FULL_CSN
	 *
	 * Sum of CU where BULKY can't take csn wave when !fits. Source is RA0
	 */
	if (strcmp(name, "SPI_RA_BULKY_CU_FULL_CSN")==0) {
		basic_counters.emplace(125, BasicCounter{125, "SPI", "SPI_RA_BULKY_CU_FULL_CSN", "Sum of CU where BULKY can't take csn wave when !fits. Source is RA0", "gfx940"});
		return 125;
	}
	/**
	 * Basic Counter: SPI_RA_TGLIM_CU_FULL_CSN
	 *
	 * Cycles where csn wants to req but all CU are at tg_limit
	 */
	if (strcmp(name, "SPI_RA_TGLIM_CU_FULL_CSN")==0) {
		basic_counters.emplace(127, BasicCounter{127, "SPI", "SPI_RA_TGLIM_CU_FULL_CSN", "Cycles where csn wants to req but all CU are at tg_limit", "gfx940"});
		return 127;
	}
	/**
	 * Basic Counter: SPI_RA_WVLIM_STALL_CSN
	 *
	 * Number of clocks csn is stalled due to WAVE LIMIT.
	 */
	if (strcmp(name, "SPI_RA_WVLIM_STALL_CSN")==0) {
		basic_counters.emplace(133, BasicCounter{133, "SPI", "SPI_RA_WVLIM_STALL_CSN", "Number of clocks csn is stalled due to WAVE LIMIT.", "gfx940"});
		return 133;
	}
	/**
	 * Basic Counter: SPI_SWC_CSC_WR
	 *
	 * Number of clocks to write CSC waves to SGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_SWC_CSC_WR")==0) {
		basic_counters.emplace(189, BasicCounter{189, "SPI", "SPI_SWC_CSC_WR", "Number of clocks to write CSC waves to SGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx940"});
		return 189;
	}
	/**
	 * Basic Counter: SPI_VWC_CSC_WR
	 *
	 * Number of clocks to write CSC waves to VGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;
	 */
	if (strcmp(name, "SPI_VWC_CSC_WR")==0) {
		basic_counters.emplace(195, BasicCounter{195, "SPI", "SPI_VWC_CSC_WR", "Number of clocks to write CSC waves to VGPRs (need to multiply this value by 4) Requires SPI_DEBUG_CNTL.DEBUG_PIPE_SEL to select source, DEBUG_PIPE_SEL = 1, source is CS1; DEBUG_PIPE_SEL = 2, source is CS2; DEBUG_PIPE_SEL = 3, source is CS3; default, source is CS0;", "gfx940"});
		return 195;
	}
	/**
	 * Basic Counter: SQ_ACCUM_PREV
	 *
	 * For counter N, increment by the value of counter N-1. Only accumulates once every 4 cycles.
	 */
	if (strcmp(name, "SQ_ACCUM_PREV")==0) {
		basic_counters.emplace(1, BasicCounter{1, "SQ", "SQ_ACCUM_PREV", "For counter N, increment by the value of counter N-1. Only accumulates once every 4 cycles.", "gfx940"});
		return 1;
	}
	/**
	 * Basic Counter: SQ_CYCLES
	 *
	 * Clock cycles. (nondeterministic, per-simd, global)
	 */
	if (strcmp(name, "SQ_CYCLES")==0) {
		basic_counters.emplace(2, BasicCounter{2, "SQ", "SQ_CYCLES", "Clock cycles. (nondeterministic, per-simd, global)", "gfx940"});
		return 2;
	}
	/**
	 * Basic Counter: SQ_BUSY_CYCLES
	 *
	 * Clock cycles while SQ is reporting that it is busy. (nondeterministic, per-simd, global)
	 */
	if (strcmp(name, "SQ_BUSY_CYCLES")==0) {
		basic_counters.emplace(3, BasicCounter{3, "SQ", "SQ_BUSY_CYCLES", "Clock cycles while SQ is reporting that it is busy. (nondeterministic, per-simd, global)", "gfx940"});
		return 3;
	}
	/**
	 * Basic Counter: SQ_WAVES
	 *
	 * Count number of waves sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES")==0) {
		basic_counters.emplace(4, BasicCounter{4, "SQ", "SQ_WAVES", "Count number of waves sent to SQs. (per-simd, emulated, global)", "gfx940"});
		return 4;
	}
	/**
	 * Basic Counter: SQ_LEVEL_WAVES
	 *
	 * Track the number of waves. Set ACCUM_PREV for the next counter to use this. (level, per-simd, global)
	 */
	if (strcmp(name, "SQ_LEVEL_WAVES")==0) {
		basic_counters.emplace(5, BasicCounter{5, "SQ", "SQ_LEVEL_WAVES", "Track the number of waves. Set ACCUM_PREV for the next counter to use this. (level, per-simd, global)", "gfx940"});
		return 5;
	}
	/**
	 * Basic Counter: SQ_WAVES_EQ_64
	 *
	 * Count number of waves with exactly 64 active threads sent to SQs. (per-simd, emulated, global)
	 */
	if (strcmp(name, "SQ_WAVES_EQ_64")==0) {
		basic_counters.emplace(6, BasicCounter{6, "SQ", "SQ_WAVES_EQ_64", "Count number of waves with exactly 64 active threads sent to SQs. (per-simd, emulated, global)", "gfx940"});
		return 6;
	}
	/**
	 * Basic Counter: SQ_WAVES_LT_64
	 *
	 * Count number of waves with 
	 */
	if (strcmp(name, "SQ_WAVES_LT_64")==0) {
		basic_counters.emplace(7, BasicCounter{7, "SQ", "SQ_WAVES_LT_64", "Count number of waves with ", "gfx940"});
		return 7;
	}
	}

	/**
	 * @}
	 */
  throw("Couldn't find the required Counter name for the mentioned GPU!");
  return 0;
}
/**
 * @}
 */

}  // namespace Counter

#pragma GCC diagnostic pop