/*
 *  This file is part of rmlint.
 *
 *  rmlint is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  rmlint is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with rmlint.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *
 *  - Christopher <sahib> Pahl 2010-2017 (https://github.com/sahib)
 *  - Daniel <SeeSpotRun> T.   2014-2017 (https://github.com/SeeSpotRun)
 *
 * Hosted on http://github.com/sahib/rmlint
 *
 */

#include "../config.h"
#include "../formats.h"
#include "../preprocess.h"

#include <glib.h>
#include <stdio.h>
#include <string.h>


typedef struct RmFmtHandlerShScript {
    RmFmtHandler parent;
    RmFile *last_original;
    RmSession *session;

    bool allow_user_cmd : 1;
    bool allow_clone : 1;
    bool allow_reflink : 1;
    bool allow_symlink : 1;
    bool allow_hardlink : 1;
    bool allow_remove : 1;

    const char *user_cmd;

    GByteArray *order;
    RmOff line_count;
} RmFmtHandlerShScript;

static const char *SH_SCRIPT_TEMPLATE_HEAD = "#!/bin/sh\n"
"\n"
"PROGRESS_CURR=0\n"
"PROGRESS_TOTAL=0                           \n"
"\n"
"# This file was autowritten by rmlint\n"
"# rmlint was executed from: %s\n"
"# Your command line was: %s\n"
"\n"
"RMLINT_BINARY=\"%s\"\n"
"\n"
"# Only use sudo if we're not root yet:\n"
"# (See: https://github.com/sahib/rmlint/issues/27://github.com/sahib/rmlint/issues/271)\n"
"SUDO_COMMAND=\"sudo\"\n"
"if [ \"$(id -u)\" -eq \"0\" ]\n"
"then\n"
"  SUDO_COMMAND=\"\"\n"
"fi\n"
"\n"
"USER='%s'\n"
"GROUP='%s'\n"
"\n"
"# Set to true on -n\n"
"DO_DRY_RUN=\n"
"\n"
"# Set to true on -p\n"
"DO_PARANOID_CHECK=\n"
"\n"
"# Set to true on -r\n"
"DO_CLONE_READONLY=\n"
"\n"
"# Set to true on -q\n"
"DO_SHOW_PROGRESS=true\n"
"\n"
"# Set to true on -c\n"
"DO_DELETE_EMPTY_DIRS=\n"
"\n"
"# Set to true on -k\n"
"DO_KEEP_DIR_TIMESTAMPS=\n"
"\n"
"# Set to true on -i\n"
"DO_ASK_BEFORE_DELETE=\n"
"\n"
"##################################\n"
"# GENERAL LINT HANDLER FUNCTIONS #\n"
"##################################\n"
"\n"
"COL_RED='\e[0;31m'\n"
"COL_BLUE='\e[1;34m'\n"
"COL_GREEN='\e[0;32m'\n"
"COL_YELLOW='\e[0;33m'\n"
"COL_RESET='\e[0m'\n"
"\n"
"print_progress_prefix() {\n"
"    if [ -n \"$DO_SHOW_PROGRESS\" ]; then\n"
"        PROGRESS_PERC=0\n"
"        if [ $((PROGRESS_TOTAL)) -gt 0 ]; then\n"
"            PROGRESS_PERC=$((PROGRESS_CURR * 100 / PROGRESS_TOTAL))\n"
"        fi\n"
"        printf %s \"${COL_BLUE}\" \"$PROGRESS_PERC\" \"${COL_RESET}\"\n"
"        if [ $# -eq \"1\" ]; then\n"
"            PROGRESS_CURR=$((PROGRESS_CURR+$1))\n"
"        else\n"
"            PROGRESS_CURR=$((PROGRESS_CURR+1))\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"handle_emptyfile() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN}Deleting empty file:${COL_RESET} $1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        rm -f \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"handle_emptydir() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN}Deleting empty directory: ${COL_RESET}$1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        rmdir \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"handle_bad_symlink() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN} Deleting symlink pointing nowhere: ${COL_RESET}$1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        rm -f \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"handle_unstripped_binary() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN} Stripping debug symbols of: ${COL_RESET}$1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        strip -s \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"handle_bad_user_id() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN}chown ${USER}${COL_RESET} $1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        chown \"$USER\" \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"handle_bad_group_id() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN}chgrp ${GROUP}${COL_RESET} $1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        chgrp \"$GROUP\" \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"handle_bad_user_and_group_id() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN}chown ${USER}:${GROUP}${COL_RESET} $1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        chown \"$USER:$GROUP\" \"$1\"\n"
"    fi\n"
"}\n"
"\n"
"###############################\n"
"# DUPLICATE HANDLER FUNCTIONS #\n"
"###############################\n"
"\n"
"check_for_equality() {\n"
"    if [ -f \"$1\" ]; then\n"
"        # Use the more lightweight builtin `cmp` for regular files:\n"
"        cmp -s \"$1\" \"$2\"\n"
"        echo $?\n"
"    else\n"
"        # Fallback to `rmlint --equal` for directories:\n"
"        \"$RMLINT_BINARY\" -p --equal %s \"$1\" \"$2\"\n"
"        echo $?\n"
"    fi\n"
"}\n"
"\n"
"original_check() {\n"
"    if [ ! -e \"$2\" ]; then\n"
"        echo \"${COL_RED}^^^^^^ Error: original has disappeared - cancelling.....${COL_RESET}\"\n"
"        return 1\n"
"    fi\n"
"\n"
"    if [ ! -e \"$1\" ]; then\n"
"        echo \"${COL_RED}^^^^^^ Error: duplicate has disappeared - cancelling.....${COL_RESET}\"\n"
"        return 1\n"
"    fi\n"
"\n"
"    # Check they are not the exact same file (hardlinks allowed):\n"
"    if [ \"$1\" = \"$2\" ]; then\n"
"        echo \"${COL_RED}^^^^^^ Error: original and duplicate point to the *same* path - cancelling.....${COL_RESET}\"\n"
"        return 1\n"
"    fi\n"
"\n"
"    # Do double-check if requested:\n"
"    if [ -z \"$DO_PARANOID_CHECK\" ]; then\n"
"        return 0\n"
"    else\n"
"        if [ \"$(check_for_equality \"$1\" \"$2\")\" -ne \"0\" ]; then\n"
"            echo \"${COL_RED}^^^^^^ Error: files no longer identical - cancelling.....${COL_RESET}\"\n"
"            return 1\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"cp_symlink() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_YELLOW}Symlinking to original: ${COL_RESET}$1\"\n"
"    if original_check \"$1\" \"$2\"; then\n"
"        if [ -z \"$DO_DRY_RUN\" ]; then\n"
"            # replace duplicate with symlink\n"
"            rm -rf \"$1\"\n"
"            ln -s \"$2\" \"$1\"\n"
"            # make the symlink's mtime the same as the original\n"
"            touch -mr \"$2\" -h \"$1\"\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"cp_hardlink() {\n"
"    if [ -d \"$1\" ]; then\n"
"        # for duplicate dir's, can't hardlink so use symlink\n"
"        cp_symlink \"$@\"\n"
"        return $?\n"
"    fi\n"
"    print_progress_prefix\n"
"    echo \"${COL_YELLOW}Hardlinking to original: ${COL_RESET}$1\"\n"
"    if original_check \"$1\" \"$2\"; then\n"
"        if [ -z \"$DO_DRY_RUN\" ]; then\n"
"            # replace duplicate with hardlink\n"
"            rm -rf \"$1\"\n"
"            ln \"$2\" \"$1\"\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"cp_reflink() {\n"
"    if [ -d \"$1\" ]; then\n"
"        # for duplicate dir's, can't clone so use symlink\n"
"        cp_symlink \"$@\"\n"
"        return $?\n"
"    fi\n"
"    print_progress_prefix\n"
"    # reflink $1 to $2's data, preserving $1's  mtime\n"
"    echo \"${COL_YELLOW}Reflinking to original: ${COL_RESET}$1\"\n"
"    if original_check \"$1\" \"$2\"; then\n"
"        if [ -z \"$DO_DRY_RUN\" ]; then\n"
"            touch -mr \"$1\" \"$0\"\n"
"            if [ -d \"$1\" ]; then\n"
"                rm -rf \"$1\"\n"
"            fi\n"
"            cp --archive --reflink=always \"$2\" \"$1\"\n"
"            touch -mr \"$0\" \"$1\"\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"clone() {\n"
"    print_progress_prefix\n"
"    # clone $1 from $2's data\n"
"    # note: no original_check() call because rmlint --dedupe takes care of this\n"
"    echo \"${COL_YELLOW}Cloning to: ${COL_RESET}$1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        if [ -n \"$DO_CLONE_READONLY\" ]; then\n"
"            $SUDO_COMMAND $RMLINT_BINARY --dedupe %s --dedupe-readonly \"$2\" \"$1\"\n"
"        else\n"
"            $RMLINT_BINARY --dedupe %s \"$2\" \"$1\"\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"skip_hardlink() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_BLUE}Leaving as-is (already hardlinked to original): ${COL_RESET}$1\"\n"
"}\n"
"\n"
"skip_reflink() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_BLUE}Leaving as-is (already reflinked to original): ${COL_RESET}$1\"\n"
"}\n"
"\n"
"user_command() {\n"
"    print_progress_prefix\n"
"\n"
"    echo \"${COL_YELLOW}Executing user command: ${COL_RESET}$1\"\n"
"    if [ -z \"$DO_DRY_RUN\" ]; then\n"
"        # You can define this function to do what you want:\n"
"        %s\n"
"    fi\n"
"}\n"
"\n"
"remove_cmd() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_YELLOW}Deleting: ${COL_RESET}$1\"\n"
"    if original_check \"$1\" \"$2\"; then\n"
"        if [ -z \"$DO_DRY_RUN\" ]; then\n"
"            if [ -n \"$DO_KEEP_DIR_TIMESTAMPS\" ]; then\n"
"                touch -r \"$(dirname \"$1\")\" \"$STAMPFILE\"\n"
"            fi\n"
"            if [ -n \"$DO_ASK_BEFORE_DELETE\" ]; then\n"
"              rm -ri \"$1\"\n"
"            else\n"
"              rm -rf \"$1\"\n"
"            fi\n"
"            if [ -n \"$DO_KEEP_DIR_TIMESTAMPS\" ]; then\n"
"                # Swap back old directory timestamp:\n"
"                touch -r \"$STAMPFILE\" \"$(dirname \"$1\")\"\n"
"                rm \"$STAMPFILE\"\n"
"            fi\n"
"\n"
"            if [ -n \"$DO_DELETE_EMPTY_DIRS\" ]; then\n"
"                DIR=$(dirname \"$1\")\n"
"                while [ ! \"$(ls -A \"$DIR\")\" ]; do\n"
"                    print_progress_prefix 0\n"
"                    echo \"${COL_GREEN}Deleting resulting empty dir: ${COL_RESET}$DIR\"\n"
"                    rmdir \"$DIR\"\n"
"                    DIR=$(dirname \"$DIR\")\n"
"                done\n"
"            fi\n"
"        fi\n"
"    fi\n"
"}\n"
"\n"
"original_cmd() {\n"
"    print_progress_prefix\n"
"    echo \"${COL_GREEN}Keeping:  ${COL_RESET}$1\"\n"
"}\n"
"\n"
"##################\n"
"# OPTION PARSING #\n"
"##################\n"
"\n"
"ask() {\n"
"    cat << EOF\n"
"\n"
"This script will delete certain files rmlint found.\n"
"It is highly advisable to view the script first!\n"
"\n"
"Rmlint was executed in the following way:\n"
"\n"
"   $ %s\n"
"\n"
"Execute this script with -d to disable this informational message.\n"
"Type any string to continue; CTRL-C, Enter or CTRL-D to abort immediately\n"
"EOF\n"
"    read -r eof_check\n"
"    if [ -z \"$eof_check\" ]\n"
"    then\n"
"        # Count Ctrl-D and Enter as aborted too.\n"
"        echo \"${COL_RED}Aborted on behalf of the user.${COL_RESET}\"\n"
"        exit 1;\n"
"    fi\n"
"}\n"
"\n"
"usage() {\n"
"    cat << EOF\n"
"usage: $0 OPTIONS\n"
"\n"
"OPTIONS:\n"
"\n"
"  -h   Show this message.\n"
"  -d   Do not ask before running.\n"
"  -x   Keep rmlint.sh; do not autodelete it.\n"
"  -p   Recheck that files are still identical before removing duplicates.\n"
"  -r   Allow deduplication of files on read-only btrfs snapshots. (requires sudo)\n"
"  -n   Do not perform any modifications, just print what would be done. (implies -d and -x)\n"
"  -c   Clean up empty directories while deleting duplicates.\n"
"  -q   Do not show progress.\n"
"  -k   Keep the timestamp of directories when removing duplicates.\n"
"  -i   Ask before deleting each file\n"
"EOF\n"
"}\n"
"\n"
"DO_REMOVE=\n"
"DO_ASK=\n"
"\n"
"while getopts \"dhxnrpqcki\" OPTION\n"
"do\n"
"  case $OPTION in\n"
"     h)\n"
"       usage\n"
"       exit 0\n"
"       ;;\n"
"     d)\n"
"       DO_ASK=false\n"
"       ;;\n"
"     x)\n"
"       DO_REMOVE=false\n"
"       ;;\n"
"     n)\n"
"       DO_DRY_RUN=true\n"
"       DO_REMOVE=false\n"
"       DO_ASK=false\n"
"       DO_ASK_BEFORE_DELETE=false\n"
"       ;;\n"
"     r)\n"
"       DO_CLONE_READONLY=true\n"
"       ;;\n"
"     p)\n"
"       DO_PARANOID_CHECK=true\n"
"       ;;\n"
"     c)\n"
"       DO_DELETE_EMPTY_DIRS=true\n"
"       ;;\n"
"     q)\n"
"       DO_SHOW_PROGRESS=\n"
"       ;;\n"
"     k)\n"
"       DO_KEEP_DIR_TIMESTAMPS=true\n"
"       STAMPFILE=$(mktemp 'rmlint.XXXXXXXX.stamp')\n"
"       ;;\n"
"     i)\n"
"       DO_ASK_BEFORE_DELETE=true\n"
"       ;;\n"
"     *)\n"
"       usage\n"
"       exit 1\n"
"  esac\n"
"done\n"
"\n"
"if [ -z $DO_REMOVE ]\n"
"then\n"
"    echo \"#${COL_YELLOW} ///${COL_RESET}This script will be deleted after it runs${COL_YELLOW}///${COL_RESET}\"\n"
"fi\n"
"\n"
"if [ -z $DO_ASK ]\n"
"then\n"
"  usage\n"
"  ask\n"
"fi\n"
"\n"
"if [ -n \"$DO_DRY_RUN\" ]\n"
"then\n"
"    echo \"#${COL_YELLOW} ////////////////////////////////////////////////////////////${COL_RESET}\"\n"
"    echo \"#${COL_YELLOW} /// ${COL_RESET} This is only a dry run; nothing will be modified! ${COL_YELLOW}///${COL_RESET}\"\n"
"    echo \"#${COL_YELLOW} ////////////////////////////////////////////////////////////${COL_RESET}\"\n"
"fi\n"
"\n"
"######### START OF AUTOGENERATED OUTPUT #########\n"
"\n"
"";
static const char *SH_SCRIPT_TEMPLATE_FOOT =
    "                                               \n"
    "                                               \n"
    "                                               \n"
    "######### END OF AUTOGENERATED OUTPUT #########\n"
    "                                               \n"
    "if [ $PROGRESS_CURR -le $PROGRESS_TOTAL ]; then\n"
    "    print_progress_prefix                      \n"
    "    echo \"${COL_BLUE}Done!${COL_RESET}\"      \n"
    "fi                                             \n"
    "                                               \n"
    "if [ -z $DO_REMOVE ] && [ -z $DO_DRY_RUN ]     \n"
    "then                                           \n"
    "  echo \"Deleting script \" \"$0\"             \n"
    "  %s '%s';                                     \n"
    "fi                                             \n"
    ;

typedef bool (* RmShOrderEmitFunc)(RmFmtHandlerShScript *self, char **out, RmFile *file, char *dupe_path, char *orig_path, char *dupe_escaped, char *orig_escaped);


static bool rm_sh_emit_handler_user(RmFmtHandlerShScript *self, char **out, _UNUSED RmFile *file, _UNUSED char *dupe_path, _UNUSED char *orig_path, char *dupe_escaped, char *orig_escaped) {
    if(self->user_cmd == NULL || !self->allow_user_cmd) {
        return false;
    }

    *out = g_strdup_printf("user_command  '%s' '%s'", dupe_escaped, orig_escaped);
    return true;
}

static bool rm_sh_emit_handler_clone(RmFmtHandlerShScript *self, char **out, RmFile *file, char *dupe_path, char *orig_path, char *dupe_escaped, char *orig_escaped) {
    if(!self->allow_clone || self->session->mounts == NULL) {
        return false;
    }

    if (!rm_mounts_can_reflink(self->session->mounts, file->dev, self->last_original->dev) ) {
        return false;
    }

    /* Needs to have at least kernel 4.2 */
    if(!rm_session_check_kernel_version(4, 2)) {
        return false;
    }

    int link_type = rm_util_link_type(dupe_path, orig_path);
    switch(link_type) {
    case RM_LINK_REFLINK:
        *out = g_strdup_printf("skip_reflink  '%s' '%s'", dupe_escaped, orig_escaped);
        return TRUE;
    case RM_LINK_SAME_FILE:
    case RM_LINK_NOT_FILE:
    case RM_LINK_WRONG_SIZE:
    case RM_LINK_PATH_DOUBLE:
    case RM_LINK_ERROR:
    case RM_LINK_XDEV:
    case RM_LINK_SYMLINK:
        rm_log_warning_line("Unexpected return code %d from rm_util_link_type()", link_type);
        return FALSE;
    case RM_LINK_HARDLINK:
    case RM_LINK_MAYBE_REFLINK:
    case RM_LINK_NONE:
        *out = g_strdup_printf("clone         '%s' '%s'", dupe_escaped, orig_escaped);
        return TRUE;
    default:
        g_assert_not_reached();
        return FALSE;
    }
}

static bool rm_sh_emit_handler_reflink(RmFmtHandlerShScript *self, char **out, RmFile *file, char *dupe_path, char *orig_path, char *dupe_escaped, char *orig_escaped) {
    if(!self->allow_reflink || self->session->mounts == NULL) {
        return false;
    }

    if(!rm_mounts_can_reflink(self->session->mounts, self->last_original->dev, file->dev)) {
        return false;
    }

    int link_type = rm_util_link_type(dupe_path, orig_path);
    switch(link_type) {
    case RM_LINK_REFLINK:
        *out = g_strdup_printf("skip_reflink  '%s' '%s'", dupe_escaped, orig_escaped);
        return TRUE;
    case RM_LINK_SAME_FILE:
    case RM_LINK_NOT_FILE:
    case RM_LINK_WRONG_SIZE:
    case RM_LINK_PATH_DOUBLE:
    case RM_LINK_XDEV:
    case RM_LINK_ERROR:
        rm_log_warning_line("Unexpected return code %d from rm_util_link_type()", link_type);
        return FALSE;
    case RM_LINK_HARDLINK:
    case RM_LINK_SYMLINK:
    case RM_LINK_MAYBE_REFLINK:
    case RM_LINK_NONE:
        *out = g_strdup_printf("cp_reflink    '%s' '%s'", dupe_escaped, orig_escaped);
        return TRUE;
    default:
        g_assert_not_reached();
        return FALSE;
    }
}

static bool rm_sh_emit_handler_symlink(RmFmtHandlerShScript *self, char **out, _UNUSED RmFile *file, _UNUSED char *dupe_path, _UNUSED char *orig_path, char *dupe_escaped, char *orig_escaped) {
    if(!self->allow_symlink) {
        return false;
    }

    *out = g_strdup_printf("cp_symlink    '%s' '%s'", dupe_escaped, orig_escaped);
    return true;
}

static bool rm_sh_emit_handler_hardlink(RmFmtHandlerShScript *self, char **out, _UNUSED RmFile *file, _UNUSED char *dupe_path, _UNUSED char *orig_path, char *dupe_escaped, char *orig_escaped) {
    if(!self->allow_hardlink || self->last_original->dev != file->dev) {
        return false;
    }

    if (self->last_original && self->last_original->inode == file->inode) {
        *out = g_strdup_printf("skip_hardlink '%s' '%s'", dupe_escaped, orig_escaped);
    } else {
        *out = g_strdup_printf("cp_hardlink   '%s' '%s'", dupe_escaped, orig_escaped);
    }
    return true;
}

static bool rm_sh_emit_handler_remove(RmFmtHandlerShScript *self, char **out, _UNUSED RmFile *file, _UNUSED char *dupe_path, _UNUSED char *orig_path, char *dupe_escaped, char *orig_escaped) {
    if(!self->allow_remove) {
        return false;
    }

    *out = g_strdup_printf("remove_cmd    '%s' '%s'", dupe_escaped, orig_escaped);
    return true;
}

typedef enum RmShHandler {
    RM_SH_HANDLER_UNKNOWN = 0,
    RM_SH_HANDLER_USER_COMMAND,
    RM_SH_HANDLER_CLONE,
    RM_SH_HANDLER_REFLINK,
    RM_SH_HANDLER_SYMLINK,
    RM_SH_HANDLER_HARDLINK,
    RM_SH_HANDLER_REMOVE,
    RM_SH_HANDLER_N
} RmShHandler;

static const char *ORDER_TO_STRING[] = {
    [RM_SH_HANDLER_UNKNOWN] = NULL,
    [RM_SH_HANDLER_USER_COMMAND] = "cmd",
    [RM_SH_HANDLER_CLONE] = "clone",
    [RM_SH_HANDLER_REFLINK] = "reflink",
    [RM_SH_HANDLER_SYMLINK] = "symlink",
    [RM_SH_HANDLER_HARDLINK] = "hardlink",
    [RM_SH_HANDLER_REMOVE] = "remove",
    [RM_SH_HANDLER_N] = NULL
};

static const RmShOrderEmitFunc ORDER_TO_FUNC[] = {
    [RM_SH_HANDLER_UNKNOWN] = NULL,
    [RM_SH_HANDLER_USER_COMMAND] = rm_sh_emit_handler_user,
    [RM_SH_HANDLER_CLONE] = rm_sh_emit_handler_clone,
    [RM_SH_HANDLER_REFLINK] = rm_sh_emit_handler_reflink,
    [RM_SH_HANDLER_SYMLINK] = rm_sh_emit_handler_symlink,
    [RM_SH_HANDLER_HARDLINK] = rm_sh_emit_handler_hardlink,
    [RM_SH_HANDLER_REMOVE] = rm_sh_emit_handler_remove
};

static void rm_sh_warn_if_reflink_not_compiled_in(void) {
#if !HAVE_BLKID || !HAVE_GIO_UNIX
    g_printerr("\n%sWARNING:%s reflink will not be emitted: please compile with blkid and gio-unix-2.0.\n", YELLOW, RESET);
#endif
}

static void rm_sh_parse_handlers(RmFmtHandlerShScript *self, const char *handler_cfg) {
    static GOnce log_once = G_ONCE_INIT;
    self->order = g_byte_array_new();

    char **order_vec = g_strsplit(handler_cfg, ",", -1);
    for(int i = 0; order_vec && order_vec[i]; ++i) {
        bool found = false;
        for(RmShHandler n = 0; n < RM_SH_HANDLER_N; ++n) {
            if(ORDER_TO_STRING[n] == NULL) {
                continue;
            }

            if(strcasecmp(order_vec[i], ORDER_TO_STRING[n]) == 0) {
                switch(n) {
                case RM_SH_HANDLER_USER_COMMAND:
                    self->allow_user_cmd = true;
                    break;
                case RM_SH_HANDLER_CLONE:
                    self->allow_clone = true;
                    g_once(&log_once, (GThreadFunc)rm_sh_warn_if_reflink_not_compiled_in, NULL);
                    break;
                case RM_SH_HANDLER_REFLINK:
                    self->allow_reflink = true;
                    g_once(&log_once, (GThreadFunc)rm_sh_warn_if_reflink_not_compiled_in, NULL);
                    break;
                case RM_SH_HANDLER_SYMLINK:
                    self->allow_symlink = true;
                    break;
                case RM_SH_HANDLER_HARDLINK:
                    self->allow_hardlink = true;
                    break;
                case RM_SH_HANDLER_REMOVE:
                    self->allow_remove = true;
                    break;
                default:
                    g_assert_not_reached();
                }

                /* we found the id */
                g_byte_array_append(self->order, (guint8 *)&n, 1);
                found = true;
                break;
            }
        }

        if(!found) {
            rm_log_error_line(_("%s is an invalid handler."), order_vec[i]);
        }
    }

    g_strfreev(order_vec);
}

static char *rm_fmt_sh_get_extra_equal_args(RmSession *session) {
    RmCfg *cfg = session->cfg;
    GString *buf = g_string_new(NULL);

    if(cfg->see_symlinks == false) {
        if(cfg->follow_symlinks) {
            g_string_append(buf, " --followlinks");
        } else {
            g_string_append(buf, " --no-followlinks");
        }
    }

    if(cfg->find_hardlinked_dupes == false) {
        g_string_append(buf, " --no-hardlinked");
    }

    if(cfg->honour_dir_layout == true) {
        g_string_append(buf, " --honour-dir-layout");
    }

    return g_string_free(buf, false);
}

static char *rm_fmt_sh_get_extra_dedupe_args(RmSession *session) {
    RmCfg *cfg = session->cfg;

    if(cfg->write_cksum_to_xattr) {
        return " --dedupe-xattr";
    }

    return "";
}

static void rm_fmt_head(RmSession *session, RmFmtHandler *parent, FILE *out) {
    RmFmtHandlerShScript *self = (RmFmtHandlerShScript *)parent;

    self->line_count = 0;
    self->session = session;
    self->user_cmd = rm_fmt_get_config_value(session->formats, "sh", "cmd");

    const char *handler_cfg = rm_fmt_get_config_value(session->formats, "sh", "handler");
    if(handler_cfg != NULL) {
        /* user specified handlers */
        rm_sh_parse_handlers(self, handler_cfg);
    } else if(rm_fmt_get_config_value(session->formats, "sh", "clone") != NULL) {
        /* Preset: try clone, then reflinks, then hardlinks then symlinks */
        rm_sh_parse_handlers(self, "clone,reflink,hardlink,symlink");
    } else if(rm_fmt_get_config_value(session->formats, "sh", "link") != NULL) {
        /* Preset: try reflinks, then then hardlinks then symlinks */
        rm_sh_parse_handlers(self, "reflink,hardlink,symlink");
    } else if(rm_fmt_get_config_value(session->formats, "sh", "reflink") != NULL) {
        /* Preset: same as above, just an alias */
        rm_sh_parse_handlers(self, "reflink,hardlink,symlink");
    } else if(rm_fmt_get_config_value(session->formats, "sh", "hardlink") != NULL) {
        /* Preset: try hardlinks before using symlinks */
        rm_sh_parse_handlers(self, "hardlink,symlink");
    } else if(rm_fmt_get_config_value(session->formats, "sh", "symlink") != NULL) {
        /* Preset: only do symlinks */
        rm_sh_parse_handlers(self, "symlink");
    } else {
        /* Default: remove the file */
        rm_sh_parse_handlers(self, "cmd,remove");
    }

    if(fchmod(fileno(out), S_IRUSR | S_IWUSR | S_IXUSR) == -1) {
        rm_log_perror("Could not chmod +x sh script");
    }

    char *equal_extra_args = rm_fmt_sh_get_extra_equal_args(session);
    char *dedupe_extra_args = rm_fmt_sh_get_extra_dedupe_args(session);

    /* Fill in all placeholders in the script template */
    /* This is a brittle hack of which the author should be ashamed. */
    fprintf(
        out, SH_SCRIPT_TEMPLATE_HEAD,
        session->cfg->iwd,
        (session->cfg->joined_argv) ? (session->cfg->joined_argv) : "[unknown]",
        (session->cfg->full_argv0_path) ? (session->cfg->full_argv0_path) : "$(which rmlint)",
        rm_util_get_username(),
        rm_util_get_groupname(),
        "'%s[%3d%%]%s '", /* The progress format */
        equal_extra_args,
        dedupe_extra_args,
        dedupe_extra_args,
        (self->user_cmd) ? self->user_cmd : "echo 'no user command defined.'",
        (session->cfg->joined_argv) ? (session->cfg->joined_argv) : "unknown_commandline"
    );

    g_free(equal_extra_args);
}

static char *rm_fmt_sh_escape_path(char *path) {
    /* See http://stackoverflow.com/questions/1250079/bash-escaping-single-quotes-inside-of-single-quoted-strings
     * for more info on this
     * */
    return rm_util_strsub(path, "'", "'\"'\"'");
}

static void rm_fmt_write_duplicate(RmFmtHandlerShScript *self, FILE *out, RmFile *file) {
    bool is_dir = (file->lint_type == RM_LINT_TYPE_DUPE_DIR_CANDIDATE);

    RM_DEFINE_PATH(file);

    char *file_escaped = rm_fmt_sh_escape_path(file_path);
    char *prefix = NULL;
    const char *comment = NULL;

    if(file->is_original) {
        if(is_dir) {
            comment = "# original directory";
        } else {
            comment = "# original";
        }

        prefix = g_strdup_printf("\noriginal_cmd  '%s'", file_escaped);
        self->last_original = file;
    } else if(self->last_original) {

        RmFile *last_original = self->last_original;
        RM_DEFINE_PATH(last_original);

        char *orig_escaped = rm_fmt_sh_escape_path(last_original_path);
        if(is_dir) {
            comment = "# duplicate directory";
        } else {
            comment = "# duplicate";
        }

        for(gsize n = 0; n < self->order->len; ++n) {
            RmShOrderEmitFunc func = ORDER_TO_FUNC[self->order->data[n]];
            if(func == NULL) {
                rm_log_error_line("null-func in sh formatter. Should not happen.");
                continue;
            }

            if(func(self, &prefix, file, file_path, last_original_path, file_escaped, orig_escaped) && prefix) {
                break;
            }
        }

        g_free(orig_escaped);
    }

    if(prefix != NULL) {
        fprintf(out, "%s %s\n", prefix, comment);
        g_free(prefix);
    }

    g_free(file_escaped);
}

static void rm_fmt_elem(_UNUSED RmSession *session, _UNUSED RmFmtHandler *parent, FILE *out, RmFile *file) {
    RmFmtHandlerShScript *self = (RmFmtHandlerShScript *)parent;
    if(file->lint_type == RM_LINT_TYPE_UNIQUE_FILE) {
        return;
    }

    RM_DEFINE_PATH(file);
    char *path = rm_fmt_sh_escape_path(file_path);

    switch(file->lint_type) {
    case RM_LINT_TYPE_EMPTY_DIR:
        fprintf(out, "handle_emptydir '%s' # empty folder\n", path);
        break;
    case RM_LINT_TYPE_EMPTY_FILE:
        fprintf(out, "handle_emptyfile '%s' # empty file\n", path);
        break;
    case RM_LINT_TYPE_BADLINK:
        fprintf(out, "handle_bad_symlink '%s' # bad symlink pointing nowhere\n", path);
        break;
    case RM_LINT_TYPE_NONSTRIPPED:
        fprintf(out, "handle_unstripped_binary '%s' # binary with debugsymbols\n", path);
        break;
    case RM_LINT_TYPE_BADUID:
        fprintf(out, "handle_bad_user_id '%s' # bad uid\n", path);
        break;
    case RM_LINT_TYPE_BADGID:
        fprintf(out, "handle_bad_group_id '%s' # bad gid\n", path);
        break;
    case RM_LINT_TYPE_BADUGID:
        fprintf(out, "handle_bad_user_and_group_id '%s' # bad gid\n", path);
        break;
    case RM_LINT_TYPE_DUPE_DIR_CANDIDATE:
    case RM_LINT_TYPE_DUPE_CANDIDATE:
        rm_fmt_write_duplicate(self, out, file);
        break;
    case RM_LINT_TYPE_PART_OF_DIRECTORY:
        break;
    default:
        rm_log_warning("warning: sh: unknown type in encountered: %d\n", file->lint_type);
        break;
    }

    self->line_count++;
    g_free(path);
}

static void rm_fmt_foot(_UNUSED RmSession *session, RmFmtHandler *parent, FILE *out) {
    RmFmtHandlerShScript *self = (RmFmtHandlerShScript *)parent;
    g_byte_array_free(self->order, true);

    if(rm_fmt_is_stream(session->formats, parent)) {
        /* You will have a hard time deleting standard streams. */
        return;
    }

    char *escaped_path = rm_fmt_sh_escape_path(parent->path);
    fprintf(out, SH_SCRIPT_TEMPLATE_FOOT, "rm -f", escaped_path);

    const char progress_marker_text[] = "PROGRESS_TOTAL=";
    char *progress_marker = strstr(SH_SCRIPT_TEMPLATE_HEAD, progress_marker_text);
    if(progress_marker != NULL) {
        gsize offset = (progress_marker - SH_SCRIPT_TEMPLATE_HEAD) +
                       sizeof(progress_marker_text) - 1;

        /* Go to the exact write position */
        if(fseek(out, offset, SEEK_SET) != -1) {
            if(fprintf(out, "%"LLU, self->line_count) < 0) {
                rm_log_perror("writing total progress failed; there will be no proress bar");
            }
        } else {
            rm_log_perror("Seeking in script failed; there will be no progress bar");
        }
    }

    g_free(escaped_path);
}

static RmFmtHandlerShScript SH_SCRIPT_HANDLER_IMPL = {
    .parent = {
        .size = sizeof(SH_SCRIPT_HANDLER_IMPL),
        .name = "sh",
        .head = rm_fmt_head,
        .elem = rm_fmt_elem,
        .prog = NULL,
        .foot = rm_fmt_foot,
        .valid_keys = {"handler", "cmd", "clone", "link", "hardlink", "symlink", "reflink", NULL},
    },
    .last_original = NULL,
    .line_count = 0
};

RmFmtHandler *SH_SCRIPT_HANDLER = (RmFmtHandler *) &SH_SCRIPT_HANDLER_IMPL;
