
/*  This file is part of the program psim.

    Copyright (C) 1994-1995, Andrew Cagney <cagney@highland.com.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see <http://www.gnu.org/licenses/>.

    --

    This file was generated by the program igen */

#ifndef _MODEL_C_
#define _MODEL_C_


#include "cpu.h"
#include "mon.h"

#include <stdlib.h>

#line 159 "../../../sim/ppc/powerpc.igen"
typedef enum _ppc_function_unit {
  PPC_UNIT_BAD,				/* unknown function unit */
  PPC_UNIT_IU,				/* integer unit (601/603 style) */
  PPC_UNIT_SRU,				/* system register unit (601/603 style) */
  PPC_UNIT_SCIU1,			/* 1st single cycle integer unit (604 style) */
  PPC_UNIT_SCIU2,			/* 2nd single cycle integer unit (604 style) */
  PPC_UNIT_MCIU,			/* multiple cycle integer unit (604 style) */
  PPC_UNIT_FPU,				/* floating point unit */
  PPC_UNIT_LSU,				/* load/store unit */
  PPC_UNIT_BPU,				/* branch unit */
  nr_ppc_function_units
} ppc_function_unit;

/* Structure to hold timing information on a per instruction basis */
struct _model_time {
  ppc_function_unit first_unit;			/* first functional unit this insn could use */
  ppc_function_unit second_unit;		/* second functional unit this insn could use */
  int16_t	    issue;			/* # cycles before function unit can process other insns */
  int16_t	    done;			/* # cycles before insn is done */
  uint32_t	    flags;			/* any flags that are needed */
};

/* Register mappings in status masks */
#define PPC_CR_REG	0			/* start of CR0 .. CR7 */
#define PPC_FPSCR_REG	(PPC_CR_REG + 8)	/* start of fpscr register */

#define PPC_NO_SPR	(-1)			/* flag for no SPR register */

/* Return if 1 bit set */
#define PPC_ONE_BIT_SET_P(x) (((x) & ((x)-1)) == 0)

/* Structure for each functional unit that is busy */
typedef struct _model_busy model_busy;
struct _model_busy {
  model_busy *next;				/* next function unit */
  ppc_function_unit unit;			/* function unit name */
  uint32_t int_busy;				/* int registers that are busy */
  uint32_t fp_busy;				/* floating point registers that are busy */
  uint32_t cr_fpscr_busy;			/* CR/FPSCR registers that are busy */
  int16_t spr_busy;				/* SPR register that is busy or PPC_NO_SPR */
  uint32_t vr_busy;				/* AltiVec registers that are busy */
  int16_t vscr_busy;				/* AltiVec status register busy */
  int16_t issue;				/* # of cycles until unit can accept another insn */
  int16_t done;				/* # of cycles until insn is done */
  int16_t nr_writebacks;			/* # of registers this unit writes back */
};

/* Structure to hold the current state information for the simulated CPU model */
struct _model_data {
  cpu *processor;				/* point back to processor */
  const char *name;				/* model name */
  const model_time *timing;			/* timing information */
  model_busy busy_head;				/* dummy entry to head list of busy function units */
  model_busy *busy_tail;			/* tail of list of busy function units */
  model_busy *free_list;			/* list of model_busy structs not in use */
  count_type nr_cycles;				/* # cycles */
  count_type nr_branches;			/* # branches */
  count_type nr_branches_fallthrough;		/* # conditional branches that fell through */
  count_type nr_branch_predict_trues;		/* # branches predicted correctly */
  count_type nr_branch_predict_falses;		/* # branches predicted incorrectly */
  count_type nr_branch_conditional[32];		/* # of each type of bc */
  count_type nr_mtcrf_crs[9];			/* # of CR's moved in a mtcrf instruction */
  count_type nr_stalls_data;			/* # of stalls for data */
  count_type nr_stalls_unit;			/* # of stalls waiting for a function unit */
  count_type nr_stalls_serialize;		/* # of stalls waiting for things to quiet down */
  count_type nr_stalls_writeback;		/* # of stalls waiting for a writeback slot */
  count_type nr_units[nr_ppc_function_units];	/* function unit counts */
  int max_nr_writebacks;			/* max # of writeback slots available */
  uint32_t int_busy;				/* int registers that are busy */
  uint32_t fp_busy;				/* floating point registers that are busy */
  uint32_t cr_fpscr_busy;			/* CR/FPSCR registers that are busy */
  uint8_t spr_busy[nr_of_sprs];		/* SPR registers that are busy */
  uint32_t vr_busy;				/* AltiVec registers that are busy */
  uint8_t vscr_busy;				/* AltiVec SC register busy */
  uint8_t busy[nr_ppc_function_units];	/* whether a function is busy or not */
};

static const char *const ppc_function_unit_name[ (int)nr_ppc_function_units ] = {
  "unknown functional unit instruction",
  "integer functional unit instruction",
  "system register functional unit instruction",
  "1st single cycle integer functional unit instruction",
  "2nd single cycle integer functional unit instruction",
  "multiple cycle integer functional unit instruction",
  "floating point functional unit instruction",
  "load/store functional unit instruction",
  "branch functional unit instruction",
};

static const char *const ppc_branch_conditional_name[32] = {
  "branch if --CTR != 0 and condition is FALSE",				/* 0000y */
  "branch if --CTR != 0 and condition is FALSE, reverse branch likely",
  "branch if --CTR == 0 and condition is FALSE",				/* 0001y */
  "branch if --CTR == 0 and condition is FALSE, reverse branch likely",
  "branch if the condition is FALSE",						/* 001zy */
  "branch if the condition is FALSE, reverse branch likely",
  "branch if the condition is FALSE (ignored bit 1 set to 1)",			/* 001zy */
  "branch if the condition is FALSE, reverse branch likely (ignored bit 4 set to 1)",
  "branch if --CTR != 0 and condition is TRUE",					/* 0100y */
  "branch if --CTR != 0 and condition is TRUE, reverse branch likely",
  "branch if --CTR == 0 and condition is TRUE",					/* 0101y */
  "branch if --CTR == 0 and condition is TRUE, reverse branch likely",
  "branch if the condition is TRUE",						/* 011zy */
  "branch if the condition is TRUE, reverse branch likely",
  "branch if the condition is TRUE (ignored bit 1 set to 1)",			/* 011zy */
  "branch if the condition is TRUE, reverse branch likely (ignored bit 4 set to 1)",
  "branch if --CTR != 0",							/* 1z00y */
  "branch if --CTR != 0, reverse branch likely",
  "branch if --CTR == 0",							/* 1z01y */
  "branch if --CTR == 0, reverse branch likely",
  "branch always",								/* 1z1zz */
  "branch always (ignored bit 5 set to 1)",
  "branch always (ignored bit 4 set to 1)",					/* 1z1zz */
  "branch always (ignored bits 4,5 set to 1)",
  "branch if --CTR != 0 (ignored bit 1 set to 1)",				/* 1z00y */
  "branch if --CTR != 0, reverse branch likely (ignored bit 1 set to 1)",
  "branch if --CTR == 0 (ignored bit 1 set to 1)",				/* 1z01y */
  "branch if --CTR == 0, reverse branch likely (ignored bit 1 set to 1)",
  "branch always (ignored bit 1 set to 1)",					/* 1z1zz */
  "branch always (ignored bits 1,5 set to 1)",
  "branch always (ignored bits 1,4 set to 1)",					/* 1z1zz */
  "branch always (ignored bits 1,4,5 set to 1)",
};

static const char *const ppc_nr_mtcrf_crs[9] = {
  "mtcrf moving 0 CRs",
  "mtcrf moving 1 CR",
  "mtcrf moving 2 CRs",
  "mtcrf moving 3 CRs",
  "mtcrf moving 4 CRs",
  "mtcrf moving 5 CRs",
  "mtcrf moving 6 CRs",
  "mtcrf moving 7 CRs",
  "mtcrf moving all CRs",
};
#line 169 "model.c"


/*h*/STATIC\
(void) model_trace_release
(model_data *model_ptr, model_busy *busy);


/*h*/STATIC\
(void) model_trace_make_busy
(model_data *model_ptr, uint32_t int_mask, uint32_t fp_mask, uint32_t cr_mask);


/*h*/STATIC\
(void) model_trace_busy_p
(model_data *model_ptr, uint32_t int_busy, uint32_t fp_busy, uint32_t cr_or_fpscr_busy, int spr_busy);


/*h*/STATIC\
(void) model_trace_altivec_busy_p
(model_data *model_ptr, uint32_t vr_busy);


/*h*/STATIC\
(void) model_trace_altivec_make_busy
(model_data *model_ptr, uint32_t vr_mask, uint32_t cr_mask);


STATIC_INLINE_MODEL\
(void) model_new_cycle
(model_data *model_ptr);


STATIC_INLINE_MODEL\
(model_busy *) model_make_busy
(model_data *model_ptr, ppc_function_unit unit, int issue, int done);


STATIC_INLINE_MODEL\
(model_busy *) model_wait_for_unit
(itable_index index, model_data *const model_ptr, const model_time *const time_ptr);


/*c*/STATIC\
(void)
model_trace_release(model_data *model_ptr, model_busy *busy)
#line 297 "../../../sim/ppc/powerpc.igen"
{
  int i;
  TRACE(trace_model,("done, %s, %d writeback%s\n", ppc_function_unit_name[busy->unit],
  		   busy->nr_writebacks, busy->nr_writebacks == 1 ? "" : "s"));
  if (busy->int_busy) {
    for(i = 0; i < 32; i++) {
      if (((1 << i) & busy->int_busy) != 0) {
        TRACE(trace_model, ("Register r%d is now available.\n", i));
      }
    }
  }
  if (busy->fp_busy) {
    for(i = 0; i < 32; i++) {
      if (((1 << i) & busy->fp_busy) != 0) {
        TRACE(trace_model, ("Register f%d is now available.\n", i));
      }
    }
  }
  if (busy->cr_fpscr_busy) {
    for(i = 0; i < 8; i++) {
      if (((1 << i) & busy->cr_fpscr_busy) != 0) {
        TRACE(trace_model, ("Register cr%d is now available.\n", i));
      }
    }
    if (busy->cr_fpscr_busy & 0x100)
      TRACE(trace_model, ("Register fpscr is now available.\n"));
  }
  if (busy->spr_busy != PPC_NO_SPR)
    TRACE(trace_model, ("Register %s is now available.\n", spr_name(busy->spr_busy)));
  if (busy->vr_busy) {
    for(i = 0; i < 32; i++) {
      if (((1 << i) & busy->vr_busy) != 0) {
        TRACE(trace_model, ("Register v%d is now available.\n", i));
      }
    }
  }
  if (busy->vscr_busy)
    TRACE(trace_model, ("VSCR Register %s is now available.\n", spr_name(busy->spr_busy)));
}
#line 255 "model.c"


/*c*/STATIC\
(void)
model_trace_make_busy(model_data *model_ptr, uint32_t int_mask, uint32_t fp_mask, uint32_t cr_mask)
#line 337 "../../../sim/ppc/powerpc.igen"
{
  int i;
  if (int_mask) {
    for(i = 0; i < 32; i++) {
      if (((1 << i) & int_mask) != 0) {
        TRACE(trace_model, ("Register r%d is now busy.\n", i));
      }
    }
  }
  if (fp_mask) {
    for(i = 0; i < 32; i++) {
      if (((1 << i) & fp_mask) != 0) {
        TRACE(trace_model, ("Register f%d is now busy.\n", i));
      }
    }
  }
  if (cr_mask) {
    for(i = 0; i < 8; i++) {
      if (((1 << i) & cr_mask) != 0) {
        TRACE(trace_model, ("Register cr%d is now busy.\n", i));
      }
    }
  }
}
#line 286 "model.c"


/*c*/STATIC\
(void)
model_trace_busy_p(model_data *model_ptr, uint32_t int_busy, uint32_t fp_busy, uint32_t cr_or_fpscr_busy, int spr_busy)
#line 362 "../../../sim/ppc/powerpc.igen"
{
  int i;
  if (int_busy) {
    int_busy &= model_ptr->int_busy;
    for(i = 0; i < 32; i++) {
      if (((1 << i) & int_busy) != 0) {
        TRACE(trace_model, ("Waiting for register r%d.\n", i));
      }
    }
  }
  if (fp_busy) {
    fp_busy &= model_ptr->fp_busy;
    for(i = 0; i < 32; i++) {
      if (((1 << i) & fp_busy) != 0) {
        TRACE(trace_model, ("Waiting for register f%d.\n", i));
      }
    }
  }
  if (cr_or_fpscr_busy) {
    cr_or_fpscr_busy &= model_ptr->cr_fpscr_busy;
    for(i = 0; i < 8; i++) {
      if (((1 << i) & cr_or_fpscr_busy) != 0) {
        TRACE(trace_model, ("Waiting for register cr%d.\n", i));
      }
    }
    if (cr_or_fpscr_busy & 0x100)
      TRACE(trace_model, ("Waiting for register fpscr.\n"));
  }
  if (spr_busy != PPC_NO_SPR && model_ptr->spr_busy[spr_busy])
    TRACE(trace_model, ("Waiting for register %s.\n", spr_name(spr_busy)));
}
#line 324 "model.c"


/*c*/STATIC\
(void)
model_trace_altivec_busy_p(model_data *model_ptr, uint32_t vr_busy)
#line 80 "../../../sim/ppc/altivec.igen"
{
  int i;
  if (vr_busy) {
    vr_busy &= model_ptr->vr_busy;
    for(i = 0; i < 32; i++) {
      if (((1 << i) & vr_busy) != 0) {
        TRACE(trace_model, ("Waiting for register v%d.\n", i));
      }
    }
  }
  if (model_ptr->vscr_busy)
    TRACE(trace_model, ("Waiting for VSCR\n"));
}
#line 344 "model.c"


/*c*/STATIC\
(void)
model_trace_altivec_make_busy(model_data *model_ptr, uint32_t vr_mask, uint32_t cr_mask)
#line 94 "../../../sim/ppc/altivec.igen"
{
  int i;
  if (vr_mask) {
    for(i = 0; i < 32; i++) {
      if (((1 << i) & vr_mask) != 0) {
        TRACE(trace_model, ("Register v%d is now busy.\n", i));
      }
    }
  }
  if (cr_mask) {
    for(i = 0; i < 8; i++) {
      if (((1 << i) & cr_mask) != 0) {
        TRACE(trace_model, ("Register cr%d is now busy.\n", i));
      }
    }
  }
}
#line 368 "model.c"


STATIC_INLINE_MODEL\
(void)
model_new_cycle(model_data *model_ptr)
#line 394 "../../../sim/ppc/powerpc.igen"
{
  model_busy *cur_busy  = model_ptr->busy_head.next;
  model_busy *free_list = model_ptr->free_list;
  model_busy *busy_tail = &model_ptr->busy_head;
  int nr_writebacks     = model_ptr->max_nr_writebacks;
  model_busy *next;

  model_ptr->nr_cycles++;
  TRACE(trace_model,("New cycle %lu\n", (unsigned long)model_ptr->nr_cycles));
  for ( ; cur_busy; cur_busy = next) {
    next = cur_busy->next;
    if (--cur_busy->done <= 0) {		/* function unit done, release registers if we have writeback slots */
      nr_writebacks -= cur_busy->nr_writebacks;
      if (nr_writebacks >= 0) {
        model_ptr->int_busy &= ~cur_busy->int_busy;
        model_ptr->fp_busy &= ~cur_busy->fp_busy;
        model_ptr->cr_fpscr_busy &= ~cur_busy->cr_fpscr_busy;
        if (cur_busy->spr_busy != PPC_NO_SPR)
  	model_ptr->spr_busy[cur_busy->spr_busy] = 0;
        model_ptr->vr_busy &= ~cur_busy->vr_busy;
        model_ptr->vscr_busy = ~cur_busy->vscr_busy;

        if (WITH_TRACE && ppc_trace[trace_model])
  	model_trace_release(model_ptr, cur_busy);

        model_ptr->busy[cur_busy->unit] = 0;
        cur_busy->next = free_list;
        free_list = cur_busy;
      }
      else {	/* writeback slots not available */
        TRACE(trace_model,("%d writeback slot%s not available for %s\n",
  			 cur_busy->nr_writebacks,
  			 cur_busy->nr_writebacks == 1 ? " is" : "s are",
  			 ppc_function_unit_name[cur_busy->unit]));
        cur_busy->done++;			/* undo -- above */
        model_ptr->nr_stalls_writeback++;
        busy_tail->next = cur_busy;
        busy_tail = cur_busy;
      }
    }
    else if (--cur_busy->issue <= 0) {	/* function unit pipelined, allow new use */
      TRACE(trace_model,("pipeline, %s ready for next client\n", ppc_function_unit_name[cur_busy->unit]));
      model_ptr->busy[cur_busy->unit] = 0;
      busy_tail->next = cur_busy;
      busy_tail = cur_busy;
    }
    else {
      TRACE(trace_model,("%s still working, issue = %d, done = %d\n",
  		       ppc_function_unit_name[cur_busy->unit],
  		       cur_busy->issue,
  		       cur_busy->done));
      busy_tail->next = cur_busy;
      busy_tail = cur_busy;
    }
  }

  busy_tail->next = (model_busy *)0;
  model_ptr->busy_tail = busy_tail;
  model_ptr->free_list = free_list;
}
#line 435 "model.c"


STATIC_INLINE_MODEL\
(model_busy *)
model_make_busy(model_data *model_ptr, ppc_function_unit unit, int issue, int done)
#line 455 "../../../sim/ppc/powerpc.igen"
{
  model_busy *busy;

  TRACE(trace_model,("unit = %s, issue = %d, done = %d\n", ppc_function_unit_name[unit], issue, done));

  if (!model_ptr->free_list) {
    busy = ZALLOC(model_busy);
  }
  else {
    busy = model_ptr->free_list;
    model_ptr->free_list = busy->next;
    busy->next = (model_busy *)0;
    busy->int_busy = 0;
    busy->fp_busy = 0;
    busy->cr_fpscr_busy = 0;
    busy->nr_writebacks = 0;
    busy->vr_busy = 0;
    busy->vscr_busy = 0;
  }

  busy->unit = unit;
  busy->issue = issue;
  busy->done = done;
  busy->spr_busy = PPC_NO_SPR;
  model_ptr->busy_tail->next = busy;
  model_ptr->busy_tail = busy;
  model_ptr->busy[unit] = 1;
  model_ptr->nr_units[unit]++;
  return busy;
}
#line 472 "model.c"


STATIC_INLINE_MODEL\
(model_busy *)
model_wait_for_unit(itable_index index, model_data *const model_ptr, const model_time *const time_ptr)
#line 486 "../../../sim/ppc/powerpc.igen"
{
  ppc_function_unit first_unit = time_ptr->first_unit;
  ppc_function_unit second_unit = time_ptr->second_unit;
  int stall_increment = 0;

  for (;;) {
    if (!model_ptr->busy[first_unit])
      return model_make_busy(model_ptr, first_unit,
  			   model_ptr->timing[index].issue,
  			   model_ptr->timing[index].done);

    if (!model_ptr->busy[second_unit])
      return model_make_busy(model_ptr, second_unit,
  			   model_ptr->timing[index].issue,
  			   model_ptr->timing[index].done);

    TRACE(trace_model,("all function units are busy for %s\n", itable[index].name));
    model_ptr->nr_stalls_unit += stall_increment;		/* don't count first stall */
    stall_increment = 1;
    model_new_cycle(model_ptr);
  }
}
#line 501 "model.c"


INLINE_MODEL\
(void)
model_serialize(itable_index index, model_data *model_ptr)
#line 509 "../../../sim/ppc/powerpc.igen"
{
  while (model_ptr->busy_head.next) {
    TRACE(trace_model,("waiting for pipeline to empty\n"));
    model_ptr->nr_stalls_serialize++;
    model_new_cycle(model_ptr);
  }
  (void) model_make_busy(model_ptr,
  		       model_ptr->timing[index].first_unit,
  		       model_ptr->timing[index].issue,
  		       model_ptr->timing[index].done);
}
#line 519 "model.c"


INLINE_MODEL\
(void)
model_wait_for_cr(model_data *model_ptr, unsigned CRBIT)
#line 521 "../../../sim/ppc/powerpc.igen"
{
  unsigned u;
  uint32_t cr_mask;
  int cr_var = 0;
  for (u = 0xc0000000; (u != 0) && (CRBIT & u) == 0; u >>= 4 )
    cr_var++;

  cr_mask = (1 << cr_var);
  while ((model_ptr->cr_fpscr_busy & cr_mask) != 0) {
    TRACE(trace_model,("waiting for CR %d\n", cr_var));
    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }
}
#line 540 "model.c"


INLINE_MODEL\
(void)
ppc_insn_int(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask)
#line 536 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t int_mask = out_mask | in_mask;
  model_busy *busy_ptr;

  if ((model_ptr->int_busy & int_mask) != 0) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->int_busy & int_mask) != 0) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_busy_p(model_ptr, int_mask, 0, 0, PPC_NO_SPR);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->int_busy |= out_mask;
  busy_ptr->int_busy |= out_mask;
  if (out_mask)
    busy_ptr->nr_writebacks = (PPC_ONE_BIT_SET_P(out_mask)) ? 1 : 2;

  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, out_mask, 0, 0);
}
#line 572 "model.c"


INLINE_MODEL\
(void)
ppc_insn_int_cr(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask, const uint32_t cr_mask)
#line 562 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t int_mask = out_mask | in_mask;
  model_busy *busy_ptr;

  if ((model_ptr->int_busy & int_mask) || (model_ptr->cr_fpscr_busy & cr_mask)) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->int_busy & int_mask) || (model_ptr->cr_fpscr_busy & cr_mask)) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_busy_p(model_ptr, int_mask, 0, cr_mask, PPC_NO_SPR);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->int_busy |= out_mask;
  busy_ptr->int_busy |= out_mask;
  model_ptr->cr_fpscr_busy |= cr_mask;
  busy_ptr->cr_fpscr_busy |= cr_mask;
  if (out_mask)
    busy_ptr->nr_writebacks = (PPC_ONE_BIT_SET_P(out_mask)) ? 1 : 2;

  if (cr_mask)
    busy_ptr->nr_writebacks++;

  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, out_mask, 0, cr_mask);
}
#line 609 "model.c"


INLINE_MODEL\
(void)
ppc_insn_cr(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask)
#line 594 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t cr_mask = out_mask | in_mask;
  model_busy *busy_ptr;

  if ((model_ptr->cr_fpscr_busy & cr_mask) != 0) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->cr_fpscr_busy & cr_mask) != 0) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_busy_p(model_ptr, 0, 0, cr_mask, PPC_NO_SPR);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->cr_fpscr_busy |= out_mask;
  busy_ptr->cr_fpscr_busy |= out_mask;
  if (out_mask)
    busy_ptr->nr_writebacks = 1;

  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, 0, 0, out_mask);
}
#line 641 "model.c"


INLINE_MODEL\
(void)
ppc_insn_float(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask)
#line 621 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t fp_mask = out_mask | in_mask;
  model_busy *busy_ptr;

  if ((model_ptr->fp_busy & fp_mask) != 0) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->fp_busy & fp_mask) != 0) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_busy_p(model_ptr, 0, fp_mask, 0, PPC_NO_SPR);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->fp_busy |= out_mask;
  busy_ptr->fp_busy |= out_mask;
  busy_ptr->nr_writebacks = 1;
  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, 0, out_mask, 0);
}
#line 671 "model.c"


INLINE_MODEL\
(void)
ppc_insn_float_cr(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask, const uint32_t cr_mask)
#line 646 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t fp_mask = out_mask | in_mask;
  model_busy *busy_ptr;

  if ((model_ptr->fp_busy & fp_mask) || (model_ptr->cr_fpscr_busy & cr_mask)) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->fp_busy & fp_mask) || (model_ptr->cr_fpscr_busy & cr_mask)) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_busy_p(model_ptr, 0, fp_mask, cr_mask, PPC_NO_SPR);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->fp_busy |= out_mask;
  busy_ptr->fp_busy |= out_mask;
  model_ptr->cr_fpscr_busy |= cr_mask;
  busy_ptr->cr_fpscr_busy |= cr_mask;
  busy_ptr->nr_writebacks = (cr_mask) ? 2 : 1;
  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, 0, out_mask, cr_mask);
}
#line 703 "model.c"


INLINE_MODEL\
(void)
ppc_insn_int_float(itable_index index, model_data *model_ptr, const uint32_t out_int_mask, const uint32_t out_fp_mask, const uint32_t in_int_mask, const uint32_t in_fp_mask)
#line 673 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t int_mask = out_int_mask | in_int_mask;
  const uint32_t fp_mask = out_fp_mask | in_fp_mask;
  model_busy *busy_ptr;

  if ((model_ptr->int_busy & int_mask) || (model_ptr->fp_busy & fp_mask)) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->int_busy & int_mask) || (model_ptr->fp_busy & fp_mask)) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_busy_p(model_ptr, int_mask, fp_mask, 0, PPC_NO_SPR);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }

    busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
    model_ptr->int_busy |= out_int_mask;
    busy_ptr->int_busy |= out_int_mask;
    model_ptr->fp_busy |= out_fp_mask;
    busy_ptr->fp_busy |= out_fp_mask;
    busy_ptr->nr_writebacks = ((out_int_mask) ? 1 : 0) + ((out_fp_mask) ? 1 : 0);
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_make_busy(model_ptr, out_int_mask, out_fp_mask, 0);
    return;
  }
}
#line 737 "model.c"


INLINE_MODEL\
(void)
ppc_insn_from_spr(itable_index index, model_data *model_ptr, const uint32_t int_mask, const unsigned nSPR)
#line 701 "../../../sim/ppc/powerpc.igen"
{
  model_busy *busy_ptr;

  while ((model_ptr->int_busy & int_mask) != 0 || model_ptr->spr_busy[nSPR] != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_busy_p(model_ptr, int_mask, 0, 0, nSPR);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->int_busy |= int_mask;
  busy_ptr->int_busy |= int_mask;
  busy_ptr->nr_writebacks = 1;
  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, int_mask, 0, 0);
}
#line 762 "model.c"


INLINE_MODEL\
(void)
ppc_insn_to_spr(itable_index index, model_data *model_ptr, const uint32_t int_mask, const unsigned nSPR)
#line 720 "../../../sim/ppc/powerpc.igen"
{
  model_busy *busy_ptr;

  while ((model_ptr->int_busy & int_mask) != 0 || model_ptr->spr_busy[nSPR] != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_busy_p(model_ptr, int_mask, 0, 0, nSPR);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  busy_ptr->spr_busy = nSPR;
  model_ptr->spr_busy[nSPR] = 1;
  busy_ptr->nr_writebacks = 1;
  TRACE(trace_model,("Making register %s busy.\n", spr_name(nSPR)));
}
#line 786 "model.c"


INLINE_MODEL\
(void)
ppc_insn_mfcr(itable_index index, model_data *model_ptr, uint32_t int_mask)
#line 738 "../../../sim/ppc/powerpc.igen"
{
  const uint32_t cr_mask = 0xff;
  model_busy *busy_ptr;

  while (((model_ptr->int_busy & int_mask) | (model_ptr->cr_fpscr_busy & cr_mask)) != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_busy_p(model_ptr, int_mask, 0, cr_mask, PPC_NO_SPR);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->int_busy |= int_mask;
  busy_ptr->int_busy |= int_mask;
  busy_ptr->nr_writebacks = 1;
  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, int_mask, 0, 0);
}
#line 812 "model.c"


INLINE_MODEL\
(void)
ppc_insn_mtcr(itable_index index, model_data *model_ptr, uint32_t int_mask, unsigned FXM)
#line 758 "../../../sim/ppc/powerpc.igen"
{
  int f;
  int nr_crs = 0;
  uint32_t cr_mask = 0;
  const model_time *normal_time = &model_ptr->timing[index];
  static const model_time ppc604_1bit_time = { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 };
  model_busy *busy_ptr;

  for (f = 0; f < 8; f++) {
    if (FXM & (0x80 >> f)) {
      cr_mask |= (1 << f);
      nr_crs++;
    }
  }

  while (((model_ptr->int_busy & int_mask) | (model_ptr->cr_fpscr_busy & cr_mask)) != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_busy_p(model_ptr, int_mask, 0, cr_mask, PPC_NO_SPR);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }

  /* If only one CR is being moved, use the SCIU, not the MCIU on the 604 */
  if (CURRENT_MODEL == MODEL_ppc604 && nr_crs == 1) {
    normal_time = &ppc604_1bit_time;
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, normal_time);
  busy_ptr->cr_fpscr_busy |= cr_mask;
  model_ptr->cr_fpscr_busy |= cr_mask;
  model_ptr->nr_mtcrf_crs[nr_crs]++;
  busy_ptr->nr_writebacks = 1;
  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_make_busy(model_ptr, 0, 0, cr_mask);
}
#line 855 "model.c"


INLINE_MODEL\
(model_data *)
model_create(cpu *processor)
#line 794 "../../../sim/ppc/powerpc.igen"
{
  model_data *model_ptr = ZALLOC(model_data);
  model_ptr->name = model_name[CURRENT_MODEL];
  model_ptr->timing = model_time_mapping[CURRENT_MODEL];
  model_ptr->processor = processor;
  model_ptr->nr_cycles = 1;
  model_ptr->busy_tail = &model_ptr->busy_head;
  switch (CURRENT_MODEL) {
  case MODEL_ppc601:  model_ptr->max_nr_writebacks = 1; break;	/* ??? */
  case MODEL_ppc603:  model_ptr->max_nr_writebacks = 2; break;
  case MODEL_ppc603e: model_ptr->max_nr_writebacks = 2; break;
  case MODEL_ppc604:  model_ptr->max_nr_writebacks = 2; break;
  default: error ("Unknown model %d\n", CURRENT_MODEL);
  }
  return model_ptr;
}
#line 878 "model.c"


INLINE_MODEL\
(void)
model_init(model_data *model_ptr)
#line 810 "../../../sim/ppc/powerpc.igen"
{
}
#line 887 "model.c"


INLINE_MODEL\
(void)
model_halt(model_data *model_ptr)
#line 812 "../../../sim/ppc/powerpc.igen"
{
  /* Let pipeline drain */
  while (model_ptr->busy_head.next)
    model_new_cycle(model_ptr);
}
#line 899 "model.c"


INLINE_MODEL\
(unsigned_word)
model_get_number_of_stalls(model_data *model_ptr)
#line 817 "../../../sim/ppc/powerpc.igen"
{
  return (model_ptr->nr_stalls_data
          + model_ptr->nr_stalls_unit
          + model_ptr->nr_stalls_serialize
          + model_ptr->nr_stalls_writeback);
}
#line 912 "model.c"


INLINE_MODEL\
(unsigned_word)
model_get_number_of_cycles(model_data *model_ptr)
#line 823 "../../../sim/ppc/powerpc.igen"
{
  return (model_ptr->nr_cycles);
}
#line 922 "model.c"


INLINE_MODEL\
(model_print *)
model_mon_info(model_data *model_ptr)
#line 826 "../../../sim/ppc/powerpc.igen"
{
  model_print *head;
  model_print *tail;
  ppc_function_unit i;
  count_type nr_insns;
  int j;

  head = tail = ZALLOC(model_print);
  tail->count = model_ptr->nr_cycles;
  tail->name = "cycle";
  tail->suffix_plural = "s";
  tail->suffix_singular = "";

  if (model_ptr->nr_stalls_data) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_stalls_data;
    tail->name = "stall";
    tail->suffix_plural = "s waiting for data";
    tail->suffix_singular = " waiting for data";
  }

  if (model_ptr->nr_stalls_unit) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_stalls_unit;
    tail->name = "stall";
    tail->suffix_plural = "s waiting for a function unit";
    tail->suffix_singular = " waiting for a function unit";
  }

  if (model_ptr->nr_stalls_serialize) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_stalls_serialize;
    tail->name = "stall";
    tail->suffix_plural = "s waiting for serialization";
    tail->suffix_singular = " waiting for serialization";
  }

  if (model_ptr->nr_stalls_writeback) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_stalls_writeback;
    tail->name = "";
    tail->suffix_plural = "times a write-back slot was unavailable";
    tail->suffix_singular = "time a writeback was unavailable";
  }

  if (model_ptr->nr_branches) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_branches;
    tail->name = "branch";
    tail->suffix_plural = "es";
    tail->suffix_singular = "";
  }

  if (model_ptr->nr_branches_fallthrough) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_branches_fallthrough;
    tail->name = "conditional branch";
    tail->suffix_plural = "es fell through";
    tail->suffix_singular = " fell through";
  }

  if (model_ptr->nr_branch_predict_trues) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_branch_predict_trues;
    tail->name = "successful branch prediction";
    tail->suffix_plural = "s";
    tail->suffix_singular = "";
  }

  if (model_ptr->nr_branch_predict_falses) {
    tail->next = ZALLOC(model_print);
    tail = tail->next;
    tail->count = model_ptr->nr_branch_predict_falses;
    tail->name = "unsuccessful branch prediction";
    tail->suffix_plural = "s";
    tail->suffix_singular = "";
  }

  for (j = 0; j < ARRAY_SIZE (ppc_branch_conditional_name); j++) {
    if (model_ptr->nr_branch_conditional[j]) {
      tail->next = ZALLOC(model_print);
      tail = tail->next;
      tail->count = model_ptr->nr_branch_conditional[j];
      tail->name = ppc_branch_conditional_name[j];
      tail->suffix_plural = " conditional branches";
      tail->suffix_singular = " conditional branch";
    }
  }

  for (j = 0; j < 9; j++) {
    if (model_ptr->nr_mtcrf_crs[j]) {
      tail->next = ZALLOC(model_print);
      tail = tail->next;
      tail->count = model_ptr->nr_mtcrf_crs[j];
      tail->name = ppc_nr_mtcrf_crs[j];
      tail->suffix_plural = " instructions";
      tail->suffix_singular = " instruction";
    }
  }

  nr_insns = 0;
  for (i = PPC_UNIT_BAD; i < nr_ppc_function_units; i++) {
    if (model_ptr->nr_units[i]) {
      nr_insns += model_ptr->nr_units[i];
      tail->next = ZALLOC(model_print);
      tail = tail->next;
      tail->count = model_ptr->nr_units[i];
      tail->name = ppc_function_unit_name[i];
      tail->suffix_plural = "s";
      tail->suffix_singular = "";
    }
  }

  tail->next = ZALLOC(model_print);
  tail = tail->next;
  tail->count = nr_insns;
  tail->name = "instruction";
  tail->suffix_plural = "s that were accounted for in timing info";
  tail->suffix_singular = " that was accounted for in timing info";

  tail->next = (model_print *)0;
  return head;
}
#line 1059 "model.c"


INLINE_MODEL\
(void)
model_mon_info_free(model_data *model_ptr, model_print *ptr)
#line 956 "../../../sim/ppc/powerpc.igen"
{
  while (ptr) {
    model_print *next = ptr->next;
    free((void *)ptr);
    ptr = next;
  }
}
#line 1073 "model.c"


INLINE_MODEL\
(void)
model_branches(model_data *model_ptr, int failed, int conditional)
#line 963 "../../../sim/ppc/powerpc.igen"
{
  model_ptr->nr_units[PPC_UNIT_BPU]++;
  if (failed)
    model_ptr->nr_branches_fallthrough++;
  else
    model_ptr->nr_branches++;
  if (conditional >= 0)
    model_ptr->nr_branch_conditional[conditional]++;
  model_new_cycle(model_ptr);	/* A branch always ends the current cycle */
}
#line 1090 "model.c"


INLINE_MODEL\
(void)
model_branch_predict(model_data *model_ptr, int success)
#line 973 "../../../sim/ppc/powerpc.igen"
{
  if (success)
    model_ptr->nr_branch_predict_trues++;
  else
    model_ptr->nr_branch_predict_falses++;
}
#line 1103 "model.c"


INLINE_MODEL\
(void)
ppc_insn_int_vr(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask, const uint32_t out_vmask, const uint32_t in_vmask)
#line 112 "../../../sim/ppc/altivec.igen"
{
  const uint32_t int_mask = out_mask | in_mask;
  const uint32_t vr_mask = out_vmask | in_vmask;
  model_busy *busy_ptr;

  if ((model_ptr->int_busy & int_mask) != 0 || (model_ptr->vr_busy & vr_mask)) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->int_busy & int_mask) != 0 || (model_ptr->vr_busy & vr_mask)) {
      if (WITH_TRACE && ppc_trace[trace_model]) {
        model_trace_busy_p(model_ptr, int_mask, 0, 0, PPC_NO_SPR);
        model_trace_altivec_busy_p(model_ptr, vr_mask);
      }

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->int_busy |= out_mask;
  busy_ptr->int_busy |= out_mask;
  model_ptr->vr_busy |= out_vmask;
  busy_ptr->vr_busy |= out_vmask;

  if (out_mask)
    busy_ptr->nr_writebacks = (PPC_ONE_BIT_SET_P(out_vmask)) ? 1 : 2;

  if (out_vmask)
    busy_ptr->nr_writebacks += (PPC_ONE_BIT_SET_P(out_vmask)) ? 1 : 2;

  if (WITH_TRACE && ppc_trace[trace_model]) {
    model_trace_make_busy(model_ptr, out_mask, 0, 0);
    model_trace_altivec_make_busy(model_ptr, vr_mask, 0);
  }
}
#line 1146 "model.c"


INLINE_MODEL\
(void)
ppc_insn_vr(itable_index index, model_data *model_ptr, const uint32_t out_vmask, const uint32_t in_vmask)
#line 149 "../../../sim/ppc/altivec.igen"
{
  const uint32_t vr_mask = out_vmask | in_vmask;
  model_busy *busy_ptr;

  if (model_ptr->vr_busy & vr_mask) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while (model_ptr->vr_busy & vr_mask) {
      if (WITH_TRACE && ppc_trace[trace_model]) {
        model_trace_altivec_busy_p(model_ptr, vr_mask);
      }

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->vr_busy |= out_vmask;
  busy_ptr->vr_busy |= out_vmask;
  if (out_vmask)
    busy_ptr->nr_writebacks = (PPC_ONE_BIT_SET_P(out_vmask)) ? 1 : 2;

  if (WITH_TRACE && ppc_trace[trace_model]) {
    model_trace_altivec_make_busy(model_ptr, vr_mask, 0);
  }
}
#line 1180 "model.c"


INLINE_MODEL\
(void)
ppc_insn_vr_cr(itable_index index, model_data *model_ptr, const uint32_t out_vmask, const uint32_t in_vmask, const uint32_t cr_mask)
#line 177 "../../../sim/ppc/altivec.igen"
{
  const uint32_t vr_mask = out_vmask | in_vmask;
  model_busy *busy_ptr;

  if ((model_ptr->vr_busy & vr_mask) || (model_ptr->cr_fpscr_busy & cr_mask)) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->vr_busy & vr_mask) || (model_ptr->cr_fpscr_busy & cr_mask)) {
      if (WITH_TRACE && ppc_trace[trace_model]) {
        model_trace_busy_p(model_ptr, 0, 0, cr_mask, PPC_NO_SPR);
        model_trace_altivec_busy_p(model_ptr, vr_mask);
      }

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->cr_fpscr_busy |= cr_mask;
  busy_ptr->cr_fpscr_busy |= cr_mask;
  model_ptr->vr_busy |= out_vmask;
  busy_ptr->vr_busy |= out_vmask;

  if (out_vmask)
    busy_ptr->nr_writebacks = (PPC_ONE_BIT_SET_P(out_vmask)) ? 1 : 2;

  if (cr_mask)
    busy_ptr->nr_writebacks++;

  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_altivec_make_busy(model_ptr, vr_mask, cr_mask);
}
#line 1220 "model.c"


INLINE_MODEL\
(void)
ppc_insn_vr_vscr(itable_index index, model_data *model_ptr, const uint32_t out_vmask, const uint32_t in_vmask)
#line 211 "../../../sim/ppc/altivec.igen"
{
  const uint32_t vr_mask = out_vmask | in_vmask;
  model_busy *busy_ptr;

  if ((model_ptr->vr_busy & vr_mask) != 0 || model_ptr->vscr_busy != 0) {
    model_new_cycle(model_ptr);			/* don't count first dependency as a stall */

    while ((model_ptr->vr_busy & vr_mask) != 0 || model_ptr->vscr_busy != 0) {
      if (WITH_TRACE && ppc_trace[trace_model])
        model_trace_altivec_busy_p(model_ptr, vr_mask);

      model_ptr->nr_stalls_data++;
      model_new_cycle(model_ptr);
    }
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->vr_busy |= out_vmask;
  busy_ptr->vr_busy |= out_vmask;
  model_ptr->vscr_busy = 1;
  busy_ptr->vscr_busy = 1;

  if (out_vmask)
    busy_ptr->nr_writebacks = 1 + (PPC_ONE_BIT_SET_P (out_vmask) ? 1 : 2);

  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_altivec_make_busy(model_ptr, vr_mask, 0);
}
#line 1255 "model.c"


INLINE_MODEL\
(void)
ppc_insn_from_vscr(itable_index index, model_data *model_ptr, const uint32_t vr_mask)
#line 240 "../../../sim/ppc/altivec.igen"
{
  model_busy *busy_ptr;

  while ((model_ptr->vr_busy & vr_mask) != 0 || model_ptr->vscr_busy != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_altivec_busy_p(model_ptr, vr_mask);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }
  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  model_ptr->cr_fpscr_busy |= vr_mask;
  busy_ptr->cr_fpscr_busy |= vr_mask;

  if (vr_mask)
    busy_ptr->nr_writebacks = 1;

  model_ptr->vr_busy |= vr_mask;
  if (WITH_TRACE && ppc_trace[trace_model])
    model_trace_altivec_make_busy(model_ptr, vr_mask, 0);
}
#line 1283 "model.c"


INLINE_MODEL\
(void)
ppc_insn_to_vscr(itable_index index, model_data *model_ptr, const uint32_t vr_mask)
#line 262 "../../../sim/ppc/altivec.igen"
{
  model_busy *busy_ptr;

  while ((model_ptr->vr_busy & vr_mask) != 0 || model_ptr->vscr_busy != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_altivec_busy_p(model_ptr, vr_mask);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }
  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  busy_ptr ->vscr_busy = 1;
  model_ptr->vscr_busy = 1;
  busy_ptr->nr_writebacks = 1;

  TRACE(trace_model,("Making VSCR busy.\n"));
}
#line 1307 "model.c"


INLINE_MODEL\
(int8_t)
altivec_signed_saturate_8(int16_t val, int *sat)
#line 281 "../../../sim/ppc/altivec.igen"
{
    int8_t rv;
    if (val > 127) {
      rv = 127;
      *sat = 1;
    } else if (val < -128) {
      rv = -128;
      *sat = 1;
    } else {
      rv = val;
      *sat = 0;
    }
    return rv;
}
#line 1328 "model.c"


INLINE_MODEL\
(int16_t)
altivec_signed_saturate_16(int32_t val, int *sat)
#line 295 "../../../sim/ppc/altivec.igen"
{
    int16_t rv;
    if (val > 32767) {
      rv = 32767;
      *sat = 1;
    } else if (val < -32768) {
      rv = -32768;
      *sat = 1;
    } else {
      rv = val;
      *sat = 0;
    }
    return rv;
}
#line 1349 "model.c"


INLINE_MODEL\
(int32_t)
altivec_signed_saturate_32(int64_t val, int *sat)
#line 309 "../../../sim/ppc/altivec.igen"
{
    int32_t rv;
    if (val > 2147483647) {
      rv = 2147483647;
      *sat = 1;
    } else if (val < -2147483648LL) {
      rv = -2147483648LL;
      *sat = 1;
    } else {
      rv = val;
      *sat = 0;
    }
    return rv;
}
#line 1370 "model.c"


INLINE_MODEL\
(uint8_t)
altivec_unsigned_saturate_8(int16_t val, int *sat)
#line 323 "../../../sim/ppc/altivec.igen"
{
    uint8_t rv;
    if (val > 255) {
      rv = 255;
      *sat = 1;
    } else if (val < 0) {
      rv = 0;
      *sat = 1;
    } else {
      rv = val;
      *sat = 0;
    }
    return rv;
}
#line 1391 "model.c"


INLINE_MODEL\
(uint16_t)
altivec_unsigned_saturate_16(int32_t val, int *sat)
#line 337 "../../../sim/ppc/altivec.igen"
{
    uint16_t rv;
    if (val > 65535) {
      rv = 65535;
      *sat = 1;
    } else if (val < 0) {
      rv = 0;
      *sat = 1;
    } else {
      rv = val;
      *sat = 0;
    }
    return rv;
}
#line 1412 "model.c"


INLINE_MODEL\
(uint32_t)
altivec_unsigned_saturate_32(int64_t val, int *sat)
#line 351 "../../../sim/ppc/altivec.igen"
{
    uint32_t rv;
    if (val > 4294967295LL) {
      rv = 4294967295LL;
      *sat = 1;
    } else if (val < 0) {
      rv = 0;
      *sat = 1;
    } else {
      rv = val;
      *sat = 0;
    }
    return rv;
}
#line 1433 "model.c"


INLINE_MODEL\
(void)
ppc_insn_int_spr(itable_index index, model_data *model_ptr, const uint32_t out_mask, const uint32_t in_mask, const unsigned nSPR)
#line 40 "../../../sim/ppc/e500.igen"
{
  const uint32_t int_mask = out_mask | in_mask;
  model_busy *busy_ptr;

  while ((model_ptr->int_busy & int_mask) != 0 || model_ptr->spr_busy[nSPR] != 0) {
    if (WITH_TRACE && ppc_trace[trace_model])
      model_trace_busy_p(model_ptr, int_mask, 0, 0, nSPR);

    model_ptr->nr_stalls_data++;
    model_new_cycle(model_ptr);
  }

  busy_ptr = model_wait_for_unit(index, model_ptr, &model_ptr->timing[index]);
  busy_ptr->int_busy |= out_mask;
  model_ptr->int_busy |= out_mask;
  busy_ptr->spr_busy = nSPR;
  model_ptr->spr_busy[nSPR] = 1;
  busy_ptr->nr_writebacks = (PPC_ONE_BIT_SET_P(out_mask)) ? 3 : 2;
  TRACE(trace_model,("Making register %s busy.\n", spr_name(nSPR)));
}
#line 1460 "model.c"

/* Insn functional unit info */
static const model_time model_time_ppc604[] = {
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* instruction_call */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Trap Word Immediate */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  3,  3,  0 },  /* Multiply Low Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Subtract From Immediate Carrying */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Compare Logical Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Compare Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add Immediate Carrying */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add Immediate Carrying and Record */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add Immediate Shifted */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* System Call */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Move Condition Register Field */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Link Register */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register NOR */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  3,  3,  0 },  /* Return From Interrupt */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register AND with Complement */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  1,  0 },  /* Instruction Synchronize */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register XOR */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register NAND */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register AND */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register Equivalent */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register OR with Complement */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Condition Register OR */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Count Register */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Rotate Left Word Immediate then Mask Insert */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Rotate Left Word Immediate then AND with Mask */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Rotate Left Word then AND with Mask */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* OR Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* OR Immediate Shifted */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* XOR Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* XOR Immediate Shifted */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* AND Immediate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* AND Immediate Shifted */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Compare */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Trap Word */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Subtract From Carrying */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add Carrying */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  4,  4,  0 },  /* Multiply High Word Unsigned */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  3,  3,  0 },  /* Move From Condition Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Word And Reserve Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Shift Left Word */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Count Leading Zeros Word */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* AND */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Compare Logical */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Subtract From */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  1,  0 },  /* Data Cache Block Store */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero with Update Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* AND with Complement */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  4,  4,  0 },  /* Multiply High Word */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  3,  3,  0 },  /* Move From Machine State Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  1,  0 },  /* Data Cache Block Flush */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Negate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero with Update Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* NOR */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Subtract From Extended */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add Extended */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move to Condition Register Fields */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move To Machine State Register */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  3,  0 },  /* Store Word Conditional Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Word with Update Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Subtract from Zero Extended */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add to Zero Extended */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move To Segment Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Byte Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Subtract From Minus One Extended */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add to Minus One Extended */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  4,  4,  0 },  /* Multiply Low Word */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move To Segment Register Indirect */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Data Cache Block Touch for Store */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Byte with Update Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Add */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  1,  0 },  /* Data Cache Block Touch */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Equivalent */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate Entry */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero with Update Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* XOR */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  3,  3,  0 },  /* Move from Special Purpose Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate All */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  3,  3,  0 },  /* Move From Time Base */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic with Update Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Half Word Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* OR with Complement */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Half Word with Update Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* OR */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  20, 20, 0 },  /* Divide Word Unsigned */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move to Special Purpose Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Data Cache Block Invalidate */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* NAND */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  20, 20, 0 },  /* Divide Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from XER */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Word Byte-Reverse Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Single Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Shift Right Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Synchronize */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Single with Update Indexed */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move From Segment Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Immediate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  1,  0 },  /* Synchronize */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Double Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Double with Update Indexed */
  { PPC_UNIT_MCIU,  PPC_UNIT_MCIU,  1,  1,  0 },  /* Move From Segment Register Indirect */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Word Byte-Reversed Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Single Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Single with Update Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Immedate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Double Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Double with Update Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Halfword Byte-Reverse Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Shift Right Algebraic Word */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Shift Right Algebraic Word Immediate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Enforce In-order Execution of I/O */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Half Word Byte-Reversed Indexed */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Extend Sign Half Word */
  { PPC_UNIT_SCIU1, PPC_UNIT_SCIU2, 1,  1,  0 },  /* Extend Sign Byte */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  1,  0 },  /* Instruction Cache Block Invalidate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Integer Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Data Cache Block set to Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Word with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Byte */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Byte with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Half Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Half Word with Update */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load Multiple Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store Multiple Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Single */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Single with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Double */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Load Floating-Point Double with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Single */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Single with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Double */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Store Floating-Point Double with Update */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   18, 18, 0 },  /* Floating Divide Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Add Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Estimate Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Compare Unordered */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Round to Single-Precision */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Convert To Integer Word */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Convert To Integer Word with round towards Zero */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   32, 32, 0 },  /* Floating Divide */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Add */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Select */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Square Root Estimate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Compare Ordered */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 1 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 0 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Move Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Field Immediate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Absolute Value */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Absolute Value */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move From FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Fields */
};


static const model_time model_time_ppc603e[] = {
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* instruction_call */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Trap Word Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Multiply Low Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Immediate Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Compare Logical Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Compare Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Carrying and Record */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Add Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Add Immediate Shifted */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* System Call */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move Condition Register Field */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Link Register */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register NOR */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* Return From Interrupt */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register AND with Complement */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Instruction Synchronize */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register XOR */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register NAND */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register AND */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register Equivalent */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register OR with Complement */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register OR */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Count Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Rotate Left Word Immediate then Mask Insert */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Rotate Left Word Immediate then AND with Mask */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Rotate Left Word then AND with Mask */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Compare */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Trap Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    6,  6,  0 },  /* Multiply High Word Unsigned */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move From Condition Register */
  { PPC_UNIT_LSU,   PPC_UNIT_IU,    1,  2,  0 },  /* Load Word And Reserve Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Left Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Count Leading Zeros Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Compare Logical */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   5,  5,  0 },  /* Data Cache Block Store */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND with Complement */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply High Word */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move From Machine State Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   5,  5,  0 },  /* Data Cache Block Flush */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Negate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* NOR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Extended */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move to Condition Register Fields */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move To Machine State Register */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   8,  8,  0 },  /* Store Word Conditional Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract from Zero Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add to Zero Extended */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move To Segment Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Minus One Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add to Minus One Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply Low Word */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move To Segment Register Indirect */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   2,  2,  0 },  /* Data Cache Block Touch for Store */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_SRU,   1,  1,  0 },  /* Add */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   2,  2,  0 },  /* Data Cache Block Touch */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Equivalent */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate Entry */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move from Special Purpose Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate All */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move From Time Base */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic with Update Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR with Complement */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    37, 37, 0 },  /* Divide Word Unsigned */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move to Special Purpose Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   2,  2,  0 },  /* Data Cache Block Invalidate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* NAND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    37, 37, 0 },  /* Divide Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from XER */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word Byte-Reverse Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Synchronize */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single with Update Indexed */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* Move From Segment Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Immediate */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Synchronize */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double with Update Indexed */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* Move From Segment Register Indirect */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word Byte-Reversed Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single with Update Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Immedate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double with Update Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Byte-Reverse Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Algebraic Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Algebraic Word Immediate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Enforce In-order Execution of I/O */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word Byte-Reversed Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Extend Sign Half Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Extend Sign Byte */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Instruction Cache Block Invalidate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Integer Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   10, 10, 0 },  /* Data Cache Block set to Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word with Update */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load Multiple Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store Multiple Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double with Update */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   18, 18, 0 },  /* Floating Divide Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Add Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Estimate Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Compare Unordered */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Round to Single-Precision */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Convert To Integer Word */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Convert To Integer Word with round towards Zero */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   33, 33, 0 },  /* Floating Divide */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Add */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Select */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Multiply */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Square Root Estimate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Negative Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Negative Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Compare Ordered */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 1 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 0 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Move Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Field Immediate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Absolute Value */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Absolute Value */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move From FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Fields */
};


static const model_time model_time_ppc603[] = {
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* instruction_call */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Trap Word Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Multiply Low Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Immediate Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare Logical Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Carrying and Record */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Shifted */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* System Call */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move Condition Register Field */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Link Register */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register NOR */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* Return From Interrupt */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register AND with Complement */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Instruction Synchronize */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register XOR */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register NAND */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register AND */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register Equivalent */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register OR with Complement */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Condition Register OR */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Count Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Rotate Left Word Immediate then Mask Insert */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Rotate Left Word Immediate then AND with Mask */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Rotate Left Word then AND with Mask */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Trap Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    6,  6,  0 },  /* Multiply High Word Unsigned */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move From Condition Register */
  { PPC_UNIT_LSU,   PPC_UNIT_IU,    1,  2,  0 },  /* Load Word And Reserve Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Left Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Count Leading Zeros Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare Logical */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   5,  5,  0 },  /* Data Cache Block Store */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND with Complement */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply High Word */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move From Machine State Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   5,  5,  0 },  /* Data Cache Block Flush */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Negate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* NOR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Extended */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move to Condition Register Fields */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move To Machine State Register */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   8,  8,  0 },  /* Store Word Conditional Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract from Zero Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add to Zero Extended */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move To Segment Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Minus One Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add to Minus One Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply Low Word */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move To Segment Register Indirect */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   2,  2,  0 },  /* Data Cache Block Touch for Store */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   2,  2,  0 },  /* Data Cache Block Touch */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Equivalent */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate Entry */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move from Special Purpose Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate All */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Move From Time Base */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic with Update Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR with Complement */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    37, 37, 0 },  /* Divide Word Unsigned */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   2,  2,  0 },  /* Move to Special Purpose Register */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   2,  2,  0 },  /* Data Cache Block Invalidate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* NAND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    37, 37, 0 },  /* Divide Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from XER */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word Byte-Reverse Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Synchronize */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single with Update Indexed */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* Move From Segment Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Immediate */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   1,  1,  0 },  /* Synchronize */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double with Update Indexed */
  { PPC_UNIT_SRU,   PPC_UNIT_SRU,   3,  3,  0 },  /* Move From Segment Register Indirect */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word Byte-Reversed Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single with Update Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Immedate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double with Update Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Byte-Reverse Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Algebraic Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Algebraic Word Immediate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Enforce In-order Execution of I/O */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word Byte-Reversed Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Extend Sign Half Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Extend Sign Byte */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  3,  0 },  /* Instruction Cache Block Invalidate */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Integer Word Indexed */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   10, 10, 0 },  /* Data Cache Block set to Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Word and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Byte and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Word with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Byte with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword and Zero with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Halfword Algebraic with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Half Word with Update */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load Multiple Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store Multiple Word */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Single with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Load Floating-Point Double with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Single with Update */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double */
  { PPC_UNIT_LSU,   PPC_UNIT_LSU,   1,  2,  0 },  /* Store Floating-Point Double with Update */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   18, 18, 0 },  /* Floating Divide Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Add Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Estimate Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Compare Unordered */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Round to Single-Precision */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Convert To Integer Word */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Convert To Integer Word with round towards Zero */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   33, 33, 0 },  /* Floating Divide */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Add */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Select */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Multiply */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Square Root Estimate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Negative Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   2,  4,  0 },  /* Floating Negative Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Compare Ordered */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 1 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 0 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Move Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Field Immediate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Negative Absolute Value */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   1,  3,  0 },  /* Floating Absolute Value */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move From FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Fields */
};


static const model_time model_time_ppc601[] = {
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* instruction_call */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Trap Word Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply Low Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Immediate Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare Logical Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Carrying and Record */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Immediate Shifted */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* System Call */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move Condition Register Field */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Link Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register NOR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Return From Interrupt */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register AND with Complement */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Instruction Synchronize */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register XOR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register NAND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register AND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register Equivalent */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register OR with Complement */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Condition Register OR */
  { PPC_UNIT_BPU,   PPC_UNIT_BPU,   1,  1,  0 },  /* Branch Conditional to Count Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Rotate Left Word Immediate then Mask Insert */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Rotate Left Word Immediate then AND with Mask */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Rotate Left Word then AND with Mask */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND Immediate Shifted */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Trap Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Carrying */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    10, 10, 0 },  /* Multiply High Word Unsigned */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move From Condition Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Load Word And Reserve Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Word and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Left Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Count Leading Zeros Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Compare Logical */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Data Cache Block Store */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Word and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* AND with Complement */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply High Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move From Machine State Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Data Cache Block Flush */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Byte and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Negate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Byte and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* NOR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Move to Condition Register Fields */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move To Machine State Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Word Conditional Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Word with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract from Zero Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add to Zero Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move To Segment Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Byte Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Subtract From Minus One Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add to Minus One Extended */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    5,  5,  0 },  /* Multiply Low Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move To Segment Register Indirect */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Data Cache Block Touch for Store */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Byte with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Add */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Data Cache Block Touch */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword and Zero Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Equivalent */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate Entry */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword and Zero with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* XOR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move from Special Purpose Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword Algebraic Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Invalidate All */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move From Time Base */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword Algebraic with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Half Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR with Complement */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Half Word with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* OR */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    36, 36, 0 },  /* Divide Word Unsigned */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Move to Special Purpose Register */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Data Cache Block Invalidate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* NAND */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    36, 36, 0 },  /* Divide Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from XER */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Load Word Byte-Reverse Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Single Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* TLB Synchronize */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Single with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Move From Segment Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load String Word Immediate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Synchronize */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Double Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Double with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    2,  2,  0 },  /* Move From Segment Register Indirect */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Word Byte-Reversed Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Single Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Single with Update Indexed */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store String Word Immedate */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Double Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Double with Update Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Load Halfword Byte-Reverse Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Algebraic Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Shift Right Algebraic Word Immediate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Enforce In-order Execution of I/O */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Half Word Byte-Reversed Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Extend Sign Half Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Extend Sign Byte */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Instruction Cache Block Invalidate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store Floating-Point Integer Word Indexed */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Data Cache Block set to Zero */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Word and Zero */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Word and Zero with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Byte and Zero */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Byte and Zero with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Word with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Byte */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Byte with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword and Zero */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword and Zero with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword Algebraic */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  2,  0 },  /* Load Halfword Algebraic with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Half Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Half Word with Update */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Load Multiple Word */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Store Multiple Word */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Single */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Single with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Double */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    3,  3,  0 },  /* Load Floating-Point Double with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Single */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Single with Update */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Double */
  { PPC_UNIT_IU,    PPC_UNIT_IU,    1,  1,  0 },  /* Store Floating-Point Double with Update */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   17, 17, 0 },  /* Floating Divide Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Add Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root Single */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Estimate Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Multiply Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Negative Multiply-Subtract Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Negative Multiply-Add Single */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Compare Unordered */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Round to Single-Precision */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Convert To Integer Word */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Convert To Integer Word with round towards Zero */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   31, 31, 0 },  /* Floating Divide */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Add */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Square Root */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   0,  0,  0 },  /* Floating Select */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   5,  5,  0 },  /* Floating Multiply */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Floating Reciprocal Square Root Estimate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   5,  5,  0 },  /* Floating Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   5,  5,  0 },  /* Floating Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   5,  5,  0 },  /* Floating Negative Multiply-Subtract */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   5,  5,  0 },  /* Floating Negative Multiply-Add */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Compare Ordered */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 1 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Negate */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move to Condition Register from FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Bit 0 */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Move Register */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Field Immediate */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Negative Absolute Value */
  { PPC_UNIT_FPU,   PPC_UNIT_FPU,   4,  4,  0 },  /* Floating Absolute Value */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move From FPSCR */
  { PPC_UNIT_BAD,   PPC_UNIT_BAD,   1,  1,  0 },  /* Move To FPSCR Fields */
};


#ifndef _INLINE_C_
const model_time *const model_time_mapping[ (int)nr_models ] = {
  (const model_time *const)0,
  model_time_ppc604,
  model_time_ppc603e,
  model_time_ppc603,
  model_time_ppc601,
};
#endif


/* map model enumeration into printable string */
#ifndef _INLINE_C_
const char *model_name[ (int)nr_models ] = {
  "NONE",
  "604",
  "603e",
  "603",
  "601",
};
#endif

INLINE_MODEL\
(void)
model_set(const char *name)
{
  model_enum model;
  for(model = MODEL_ppc604; model < nr_models; model++) {
    if(strcmp(name, model_name[model]) == 0) {
      current_model = model;
      return;
    }
  }

  error("Unknown model '%s', Models which are known are:%s\n",
        name,
        "\n\t604\n\t603e\n\t603\n\t601");
}

#endif /* _MODEL_C_*/
