/*-----------------------------------------*\
|  AsusAuraTUFKeyboardController.h          |
|                                           |
|  Definitions and types for ASUS Aura      |
|  USB RGB lighting controller              |
|                                           |
|  Mola19 10/02/2021                        |
\*-----------------------------------------*/

#include "RGBControllerKeyNames.h"
#include <vector>
#include <map>
#include "RGBController.h"

#pragma once

enum
{
    ASUS_TUF_K7_LAYOUT_CA    = 1,
    ASUS_TUF_K7_LAYOUT_AR    = 2,
    ASUS_TUF_K7_LAYOUT_DE    = 3,
    ASUS_TUF_K7_LAYOUT_UK    = 4,
    ASUS_TUF_K7_LAYOUT_FR    = 5,
    ASUS_TUF_K7_LAYOUT_CN    = 6,
    ASUS_TUF_K7_LAYOUT_HU    = 7,
    ASUS_TUF_K7_LAYOUT_IT    = 8,
    ASUS_TUF_K7_LAYOUT_TH    = 9,
    ASUS_TUF_K7_LAYOUT_UA    = 10,
    ASUS_TUF_K7_LAYOUT_NO    = 11,
    ASUS_TUF_K7_LAYOUT_PT    = 12,
    ASUS_TUF_K7_LAYOUT_HE    = 13,
    ASUS_TUF_K7_LAYOUT_RU    = 14,
    ASUS_TUF_K7_LAYOUT_ES    = 15,
    ASUS_TUF_K7_LAYOUT_TW    = 16,
    ASUS_TUF_K7_LAYOUT_US    = 17,
    ASUS_TUF_K7_LAYOUT_TR    = 18,
    ASUS_TUF_K7_LAYOUT_CZ    = 19,
    ASUS_TUF_K7_LAYOUT_BE    = 20,
    ASUS_TUF_K7_LAYOUT_JP    = 21,
    ASUS_TUF_K7_LAYOUT_KR    = 22,
    ASUS_TUF_K7_LAYOUT_IS    = 23,
    ASUS_TUF_K7_LAYOUT_WB    = 24,
    ASUS_TUF_K7_LAYOUT_SW_CH = 25
};

#define NA  0xFFFFFFFF

struct led_value
{
    const char*         name;
    unsigned char       id;
};

struct layout_info
{
    unsigned int*               matrix_map;
    int                         size;
    int                         rows;
    int                         cols;
    std::vector<led_value>      led_names;
};

static unsigned int ASUS_TUF_K7_LAYOUT_KEYS_US[6][24] =
{
    {   0,  NA,   8,  14,  19,  24,  NA,  34,  39,  44,  49,  55,  61,  66,  70,  NA,  74,  78,  83,  NA,  NA,  NA,  NA,  NA },
    {   1,   6,   9,  15,  20,  25,  29,  35,  40,  45,  50,  56,  62,  67,  NA,  NA,  75,  79,  84,  NA,  87,  92,  96, 101 },
    {   2,  NA,  10,  16,  21,  26,  30,  36,  41,  46,  51,  57,  63,  68,  71,  NA,  76,  80,  85,  NA,  88,  93,  97, 102 },
    {   3,  NA,  11,  17,  22,  27,  31,  37,  42,  47,  52,  58,  64,  NA,  72,  NA,  NA,  NA,  NA,  NA,  89,  94,  98,  NA },
    {   4,  NA,  12,  18,  23,  28,  32,  38,  43,  48,  53,  59,  NA,  69,  NA,  NA,  NA,  81,  NA,  NA,  90,  95,  99, 103 },
    {   5,   7,  13,  NA,  NA,  NA,  33,  NA,  NA,  NA,  54,  60,  65,  NA,  73,  NA,  77,  82,  86,  NA,  91,  NA, 100,  NA }
};


static unsigned int ASUS_TUF_K7_LAYOUT_KEYS_UK[6][24] =
{
    {   0,  NA,   9,  15,  20,  25,  NA,  35,  40,  45,  50,  56,  62,  67,  72,  NA,  75,  79,  84,  NA,  NA,  NA,  NA,  NA },
    {   1,   6,  10,  16,  21,  26,  30,  36,  41,  46,  51,  57,  63,  68,  NA,  NA,  76,  80,  85,  NA,  88,  93,  97, 102 },
    {   2,  NA,  11,  17,  22,  27,  31,  37,  42,  47,  52,  58,  64,  69,  NA,  NA,  77,  81,  86,  NA,  89,  94,  98, 103 },
    {   3,  NA,  12,  18,  23,  28,  32,  38,  43,  48,  53,  59,  65,  70,  73,  NA,  NA,  NA,  NA,  NA,  90,  95,  99,  NA },
    {   4,   7,  13,  19,  24,  29,  33,  39,  44,  49,  54,  60,  NA,  71,  NA,  NA,  NA,  82,  NA,  NA,  91,  96, 100, 104 },
    {   5,   8,  14,  NA,  NA,  NA,  34,  NA,  NA,  NA,  55,  61,  66,  NA,  74,  NA,  78,  83,  87,  NA,  92,  NA, 101,  NA }
};

static unsigned int ASUS_FALCHION_LAYOUT_KEYS_US[5][16] =
{
    {   0,   5,   7,  12,  16,  20,  24,  29,  33,  37,  41,  46,  51,  56,  NA,  63 },
    {   1,  NA,   8,  13,  17,  21,  25,  30,  34,  38,  42,  47,  52,  57,  59,  64 },
    {   2,  NA,   9,  14,  18,  22,  26,  31,  35,  39,  43,  48,  53,  NA,  60,  65 }, 
    {   3,  NA,  10,  15,  19,  23,  27,  32,  36,  40,  44,  49,  54,  NA,  61,  66 },
    {   4,   6,  11,  NA,  NA,  NA,  28,  NA,  NA,  NA,  45,  50,  55,  58,  62,  67 }
};

static unsigned int ASUS_FALCHION_LAYOUT_KEYS_UK[5][16] =
{
    {   0,   5,   8,  13,  17,  21,  25,  30,  34,  38,  42,  47,  52,  57,  NA,  64 },
    {   1,  NA,   9,  14,  18,  22,  26,  31,  35,  39,  43,  48,  53,  58,  61,  65 },
    {   2,  NA,  10,  15,  19,  23,  27,  32,  36,  40,  44,  49,  54,  59,  NA,  66 }, 
    {   3,   6,  11,  16,  20,  24,  28,  33,  37,  41,  45,  50,  55,  NA,  62,  67 },
    {   4,   7,  12,  NA,  NA,  NA,  29,  NA,  NA,  NA,  46,  51,  56,  60,  63,  68 }
};

static unsigned int ASUS_CLAYMORE_NO_NUMPAD_LAYOUT_KEYS_US[7][19] =
{
    {   0,  NA,   8,  14,  19,  24,  NA,  34,  39,  45,  50,  56,  62,  67,  71,  NA,  75,  79,  84 },
    {   1,   6,   9,  15,  20,  25,  29,  35,  40,  46,  51,  57,  63,  68,  NA,  NA,  76,  80,  85 },
    {   2,  NA,  10,  16,  21,  26,  30,  36,  41,  47,  52,  58,  64,  69,  72,  NA,  77,  81,  86 },
    {   3,  NA,  11,  17,  22,  27,  31,  37,  42,  48,  53,  59,  65,  NA,  73,  NA,  NA,  NA,  NA },
    {   4,  NA,  12,  18,  23,  28,  32,  38,  43,  49,  54,  60,  NA,  70,  NA,  NA,  NA,  82,  NA },
    {   5,   7,  13,  NA,  NA,  NA,  33,  NA,  NA,  NA,  55,  61,  66,  NA,  74,  NA,  78,  83,  87 },
    {  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  44,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA }
};

static unsigned int ASUS_CLAYMORE_NO_NUMPAD_LAYOUT_KEYS_UK[7][19] =
{
    {   0,  NA,   9,  15,  20,  25,  NA,  35,  40,  46,  51,  57,  63,  68,  73,  NA,  76,  80,  85 },
    {   1,   6,  10,  16,  21,  26,  30,  36,  41,  47,  52,  58,  64,  69,  NA,  NA,  77,  81,  86 },
    {   2,  NA,  11,  17,  22,  27,  31,  37,  42,  48,  53,  59,  65,  70,  NA,  NA,  78,  82,  87 },
    {   3,  NA,  12,  18,  23,  28,  32,  38,  43,  49,  54,  60,  66,  71,  74,  NA,  NA,  NA,  NA },
    {   4,   7,  13,  19,  24,  29,  33,  39,  44,  50,  55,  61,  NA,  72,  NA,  NA,  NA,  83,  NA },
    {   5,   8,  14,  NA,  NA,  NA,  34,  NA,  NA,  NA,  56,  62,  67,  NA,  75,  NA,  79,  84,  88 },
    {  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  45,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA }
};

static unsigned int ASUS_CLAYMORE_NUMPAD_RIGHT_LAYOUT_KEYS_US[7][24] =
{
    {   0,  NA,   8,  14,  19,  24,  NA,  34,  39,  45,  50,  56,  62,  67,  71,  NA,  75,  79,  84,  NA,  NA,  NA,  NA,  NA },
    {   1,   6,   9,  15,  20,  25,  29,  35,  40,  46,  51,  57,  63,  68,  NA,  NA,  76,  80,  85,  NA,  88,  93,  97, 102 },
    {   2,  NA,  10,  16,  21,  26,  30,  36,  41,  47,  52,  58,  64,  69,  72,  NA,  77,  81,  86,  NA,  89,  94,  98, 103 },
    {   3,  NA,  11,  17,  22,  27,  31,  37,  42,  48,  53,  59,  65,  NA,  73,  NA,  NA,  NA,  NA,  NA,  90,  95,  99,  NA },
    {   4,  NA,  12,  18,  23,  28,  32,  38,  43,  49,  54,  60,  NA,  70,  NA,  NA,  NA,  82,  NA,  NA,  91,  96, 100, 104 },
    {   5,   7,  13,  NA,  NA,  NA,  33,  NA,  NA,  NA,  55,  61,  66,  NA,  74,  NA,  78,  83,  87,  NA,  92,  NA, 101,  NA },
    {  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  44,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA }
};

static unsigned int ASUS_CLAYMORE_NUMPAD_RIGHT_LAYOUT_KEYS_UK[7][24] =
{
    {   0,  NA,   9,  15,  20,  25,  NA,  35,  40,  46,  51,  57,  63,  68,  73,  NA,  76,  80,  85,  NA,  NA,  NA,  NA,  NA },
    {   1,   6,  10,  16,  21,  26,  30,  36,  41,  47,  52,  58,  64,  69,  NA,  NA,  77,  81,  86,  NA,  89,  94,  98, 103 },
    {   2,  NA,  11,  17,  22,  27,  31,  37,  42,  48,  53,  59,  65,  70,  NA,  NA,  78,  82,  87,  NA,  90,  95,  99, 104 },
    {   3,  NA,  12,  18,  23,  28,  32,  38,  43,  49,  54,  60,  66,  71,  74,  NA,  NA,  NA,  NA,  NA,  91,  96, 100,  NA },
    {   4,   7,  13,  19,  24,  29,  33,  39,  44,  50,  55,  61,  NA,  72,  NA,  NA,  NA,  83,  NA,  NA,  92,  97, 101, 105 },
    {   5,   8,  14,  NA,  NA,  NA,  34,  NA,  NA,  NA,  56,  62,  67,  NA,  75,  NA,  79,  84,  88,  NA,  93,  NA, 102,  NA },
    {  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  45,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA }
};

static unsigned int ASUS_CLAYMORE_NUMPAD_LEFT_LAYOUT_KEYS_US[7][24] =
{
    {  NA,  NA,  NA,  NA,  NA,  17,  NA,  25,  31,  36,  41,  NA,  51,  56,  62,  67,  73,  79,  84,  88,  NA,  92,  96, 101 },
    {   0,   5,   9,  14,  NA,  18,  23,  26,  32,  37,  42,  46,  52,  57,  63,  68,  74,  80,  85,  NA,  NA,  93,  97, 102 },
    {   1,   6,  10,  15,  NA,  19,  NA,  27,  33,  38,  43,  47,  53,  58,  64,  69,  75,  81,  86,  89,  NA,  94,  98, 103 },
    {   2,   7,  11,  NA,  NA,  20,  NA,  28,  34,  39,  44,  48,  54,  59,  65,  70,  76,  82,  NA,  90,  NA,  NA,  NA,  NA },
    {   3,   8,  12,  16,  NA,  21,  NA,  29,  35,  40,  45,  49,  55,  60,  66,  71,  77,  NA,  87,  NA,  NA,  NA,  99,  NA },
    {   4,  NA,  13,  NA,  NA,  22,  24,  30,  NA,  NA,  NA,  50,  NA,  NA,  NA,  72,  78,  83,  NA,  91,  NA,  95, 100, 104 },
    {  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  61,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA }
};

static unsigned int ASUS_CLAYMORE_NUMPAD_LEFT_LAYOUT_KEYS_UK[7][24] =
{
    {  NA,  NA,  NA,  NA,  NA,  17,  NA,  26,  32,  37,  42,  NA,  52,  57,  63,  68,  74,  80,  85,  90,  NA,  93,  97, 102 },
    {   0,   5,   9,  14,  NA,  18,  23,  27,  33,  38,  43,  47,  53,  58,  64,  69,  75,  81,  86,  NA,  NA,  94,  98, 103 },
    {   1,   6,  10,  15,  NA,  19,  NA,  28,  34,  39,  44,  48,  54,  59,  65,  70,  76,  82,  87,  NA,  NA,  95,  99, 104 },
    {   2,   7,  11,  NA,  NA,  20,  NA,  29,  35,  40,  45,  49,  55,  60,  66,  71,  77,  83,  88,  91,  NA,  NA,  NA,  NA },
    {   3,   8,  12,  16,  NA,  21,  24,  30,  36,  41,  46,  50,  56,  61,  67,  72,  78,  NA,  89,  NA,  NA,  NA, 100,  NA },
    {   4,  NA,  13,  NA,  NA,  22,  25,  31,  NA,  NA,  NA,  51,  NA,  NA,  NA,  73,  79,  84,  NA,  92,  NA,  96, 101, 105 },
    {  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  62,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA,  NA }
};

static unsigned int ASUS_TUF_K1_LAYOUT_KEYS[1][5] =
{
    {  0,  1,  2,  3,  4 },
};

static std::map<int,layout_info> AsusTUFK7Layouts =
{
    {
        ASUS_TUF_K7_LAYOUT_UK,
        {
            *ASUS_TUF_K7_LAYOUT_KEYS_UK,
            105,
            6,
            24,
            {
                { KEY_EN_ESCAPE,              0x00 },
                { KEY_EN_BACK_TICK,           0x01 },
                { KEY_EN_TAB,                 0x02 },
                { KEY_EN_CAPS_LOCK,           0x03 },
                { KEY_EN_LEFT_SHIFT,          0x04 },
                { KEY_EN_LEFT_CONTROL,        0x05 },

                { KEY_EN_1,                   0x11 },
                { KEY_EN_ISO_BACK_SLASH,      0x0C },
                { KEY_EN_LEFT_WINDOWS,        0x0D },

                { KEY_EN_F1,                  0x18 },
                { KEY_EN_2,                   0x19 },
                { KEY_EN_Q,                   0x12 },
                { KEY_EN_A,                   0x13 },
                { KEY_EN_Z,                   0x14 },
                { KEY_EN_LEFT_ALT,            0x15 },

                { KEY_EN_F2,                  0x20 },
                { KEY_EN_3,                   0x21 },
                { KEY_EN_W,                   0x1A },
                { KEY_EN_S,                   0x1B },
                { KEY_EN_X,                   0x1C },

                { KEY_EN_F3,                  0x28 },
                { KEY_EN_4,                   0x29 },
                { KEY_EN_E,                   0x22 },
                { KEY_EN_D,                   0x23 },
                { KEY_EN_C,                   0x24 },
                
                { KEY_EN_F4,                  0x30 },
                { KEY_EN_5,                   0x31 },
                { KEY_EN_R,                   0x2A },
                { KEY_EN_F,                   0x2B },
                { KEY_EN_V,                   0x2C },

                { KEY_EN_6,                   0x39 },
                { KEY_EN_T,                   0x32 },
                { KEY_EN_G,                   0x33 },
                { KEY_EN_B,                   0x34 },
                { KEY_EN_SPACE,               0x35 },

                { KEY_EN_F5,                  0x40 },
                { KEY_EN_7,                   0x41 },
                { KEY_EN_Y,                   0x3A },
                { KEY_EN_H,                   0x3B },
                { KEY_EN_N,                   0x3C },

                { KEY_EN_F6,                  0x48 },
                { KEY_EN_8,                   0x49 },
                { KEY_EN_U,                   0x42 },
                { KEY_EN_J,                   0x43 },
                { KEY_EN_M,                   0x44 },

                { KEY_EN_F7,                  0x50 },
                { KEY_EN_9,                   0x51 },
                { KEY_EN_I,                   0x4A },
                { KEY_EN_K,                   0x4B },
                { KEY_EN_COMMA,               0x4C },

                { KEY_EN_F8,                  0x58 },
                { KEY_EN_0,                   0x59 },
                { KEY_EN_O,                   0x52 },
                { KEY_EN_L,                   0x53 },
                { KEY_EN_PERIOD,              0x54 },
                { KEY_EN_RIGHT_ALT,           0x4D },

                { KEY_EN_F9,                  0x60 },
                { KEY_EN_MINUS,               0x61 },
                { KEY_EN_P,                   0x5A },
                { KEY_EN_SEMICOLON,           0x5B },
                { KEY_EN_FORWARD_SLASH,       0x5C },
                { KEY_EN_RIGHT_FUNCTION,      0x5D },

                { KEY_EN_F10,                 0x68 },
                { KEY_EN_EQUALS,              0x69 },
                { KEY_EN_LEFT_BRACKET,        0x62 },
                { KEY_EN_QUOTE,               0x63 },
                { KEY_EN_MENU,                0x65 },

                { KEY_EN_F11,                 0x70 },
                { KEY_EN_BACKSPACE,           0x79 },
                { KEY_EN_RIGHT_BRACKET,       0x6A },
                { KEY_EN_POUND,               0x6B },
                { KEY_EN_RIGHT_SHIFT,         0x7C },

                { KEY_EN_F12,                 0x78 },
                { KEY_EN_ISO_ENTER,           0x7B },
                { KEY_EN_RIGHT_CONTROL,       0x7D },

                { KEY_EN_PRINT_SCREEN,        0x80 },
                { KEY_EN_INSERT,              0x81 },
                { KEY_EN_DELETE,              0x82 },
                { KEY_EN_LEFT_ARROW,          0x85 },

                { KEY_EN_SCROLL_LOCK,         0x88 },
                { KEY_EN_HOME,                0x89 },
                { KEY_EN_END,                 0x8A },
                { KEY_EN_UP_ARROW,            0x8C },
                { KEY_EN_DOWN_ARROW,          0x8D },

                { KEY_EN_PAUSE_BREAK,         0x90 },
                { KEY_EN_PAGE_UP,             0x91 },
                { KEY_EN_PAGE_DOWN,           0x92 },
                { KEY_EN_RIGHT_ARROW,         0x95 },

                { KEY_EN_NUMPAD_LOCK,         0x99 },
                { KEY_EN_NUMPAD_7,            0x9A },
                { KEY_EN_NUMPAD_4,            0x9B },
                { KEY_EN_NUMPAD_1,            0x9C },
                { KEY_EN_NUMPAD_0,            0x9D },

                { KEY_EN_NUMPAD_DIVIDE,       0xA1 },
                { KEY_EN_NUMPAD_8,            0xA2 },
                { KEY_EN_NUMPAD_5,            0xA3 },
                { KEY_EN_NUMPAD_2,            0xA4 },

                { KEY_EN_NUMPAD_TIMES,        0xA9 },
                { KEY_EN_NUMPAD_9,            0xAA },
                { KEY_EN_NUMPAD_6,            0xAB },
                { KEY_EN_NUMPAD_3,            0xAC },
                { KEY_EN_NUMPAD_PERIOD,       0xAD },

                { KEY_EN_NUMPAD_MINUS,        0xB1 },
                { KEY_EN_NUMPAD_PLUS,         0xB2 },
                { KEY_EN_NUMPAD_ENTER,        0xB4 }
            }
        }
    },
    {
        ASUS_TUF_K7_LAYOUT_US,
        {
            *ASUS_TUF_K7_LAYOUT_KEYS_US,
            104,
            6,
            24,
            {
                { KEY_EN_ESCAPE,              0x00 },
                { KEY_EN_BACK_TICK,           0x01 },
                { KEY_EN_TAB,                 0x02 },
                { KEY_EN_CAPS_LOCK,           0x03 },
                { KEY_EN_LEFT_SHIFT,          0x04 },
                { KEY_EN_LEFT_CONTROL,        0x05 },

                { KEY_EN_1,                   0x11 },
                { KEY_EN_LEFT_WINDOWS,        0x0D },

                { KEY_EN_F1,                  0x18 },
                { KEY_EN_2,                   0x19 },
                { KEY_EN_Q,                   0x12 },
                { KEY_EN_A,                   0x13 },
                { KEY_EN_Z,                   0x14 },
                { KEY_EN_LEFT_ALT,            0x15 },

                { KEY_EN_F2,                  0x20 },
                { KEY_EN_3,                   0x21 },
                { KEY_EN_W,                   0x1A },
                { KEY_EN_S,                   0x1B },
                { KEY_EN_X,                   0x1C },

                { KEY_EN_F3,                  0x28 },
                { KEY_EN_4,                   0x29 },
                { KEY_EN_E,                   0x22 },
                { KEY_EN_D,                   0x23 },
                { KEY_EN_C,                   0x24 },

                { KEY_EN_F4,                  0x30 },
                { KEY_EN_5,                   0x31 },
                { KEY_EN_R,                   0x2A },
                { KEY_EN_F,                   0x2B },
                { KEY_EN_V,                   0x2C },

                { KEY_EN_6,                   0x39 },
                { KEY_EN_T,                   0x32 },
                { KEY_EN_G,                   0x33 },
                { KEY_EN_B,                   0x34 },
                { KEY_EN_SPACE,               0x35 },

                { KEY_EN_F5,                  0x40 },
                { KEY_EN_7,                   0x41 },
                { KEY_EN_Y,                   0x3A },
                { KEY_EN_H,                   0x3B },
                { KEY_EN_N,                   0x3C },

                { KEY_EN_F6,                  0x48 },
                { KEY_EN_8,                   0x49 },
                { KEY_EN_U,                   0x42 },
                { KEY_EN_J,                   0x43 },
                { KEY_EN_M,                   0x44 },

                { KEY_EN_F7,                  0x50 },
                { KEY_EN_9,                   0x51 },
                { KEY_EN_I,                   0x4A },
                { KEY_EN_K,                   0x4B },
                { KEY_EN_COMMA,               0x4C },

                { KEY_EN_F8,                  0x58 },
                { KEY_EN_0,                   0x59 },
                { KEY_EN_O,                   0x52 },
                { KEY_EN_L,                   0x53 },
                { KEY_EN_PERIOD,              0x54 },
                { KEY_EN_RIGHT_ALT,           0x4D },

                { KEY_EN_F9,                  0x60 },
                { KEY_EN_MINUS,               0x61 },
                { KEY_EN_P,                   0x5A },
                { KEY_EN_SEMICOLON,           0x5B },
                { KEY_EN_FORWARD_SLASH,       0x5C },
                { KEY_EN_RIGHT_FUNCTION,      0x5D },

                { KEY_EN_F10,                 0x68 },
                { KEY_EN_EQUALS,              0x69 },
                { KEY_EN_LEFT_BRACKET,        0x62 },
                { KEY_EN_QUOTE,               0x63 },
                { KEY_EN_MENU,                0x65 },

                { KEY_EN_F11,                 0x70 },
                { KEY_EN_BACKSPACE,           0x79 },
                { KEY_EN_RIGHT_BRACKET,       0x6A },
                { KEY_EN_RIGHT_SHIFT,         0x7C },

                { KEY_EN_F12,                 0x78 },
                { KEY_EN_ANSI_BACK_SLASH,     0x7A },
                { KEY_EN_ANSI_ENTER,          0x7B },
                { KEY_EN_RIGHT_CONTROL,       0x7D },

                { KEY_EN_PRINT_SCREEN,        0x80 },
                { KEY_EN_INSERT,              0x81 },
                { KEY_EN_DELETE,              0x82 },
                { KEY_EN_LEFT_ARROW,          0x85 },

                { KEY_EN_SCROLL_LOCK,         0x88 },
                { KEY_EN_HOME,                0x89 },
                { KEY_EN_END,                 0x8A },
                { KEY_EN_UP_ARROW,            0x8C },
                { KEY_EN_DOWN_ARROW,          0x8D },

                { KEY_EN_PAUSE_BREAK,         0x90 },
                { KEY_EN_PAGE_UP,             0x91 },
                { KEY_EN_PAGE_DOWN,           0x92 },
                { KEY_EN_RIGHT_ARROW,         0x95 },

                { KEY_EN_NUMPAD_LOCK,         0x99 },
                { KEY_EN_NUMPAD_7,            0x9A },
                { KEY_EN_NUMPAD_4,            0x9B },
                { KEY_EN_NUMPAD_1,            0x9C },
                { KEY_EN_NUMPAD_0,            0x9D },

                { KEY_EN_NUMPAD_DIVIDE,       0xA1 },
                { KEY_EN_NUMPAD_8,            0xA2 },
                { KEY_EN_NUMPAD_5,            0xA3 },
                { KEY_EN_NUMPAD_2,            0xA4 },

                { KEY_EN_NUMPAD_TIMES,        0xA9 },
                { KEY_EN_NUMPAD_9,            0xAA },
                { KEY_EN_NUMPAD_6,            0xAB },
                { KEY_EN_NUMPAD_3,            0xAC },
                { KEY_EN_NUMPAD_PERIOD,       0xAD },

                { KEY_EN_NUMPAD_MINUS,        0xB1 },
                { KEY_EN_NUMPAD_PLUS,         0xB2 },
                { KEY_EN_NUMPAD_ENTER,        0xB4 }
            }
        }
    },
};

static std::map<int,layout_info> AsusFalchionLayouts =
{
    {
        ASUS_TUF_K7_LAYOUT_UK,
        {
            *ASUS_FALCHION_LAYOUT_KEYS_UK,
            69,
            5,
            16,
            {
                { KEY_EN_ESCAPE,            0x00    },
                { KEY_EN_TAB,               0x01    },
                { KEY_EN_CAPS_LOCK,         0x02    },
                { KEY_EN_LEFT_SHIFT,        0x03    },
                { KEY_EN_LEFT_CONTROL,      0x04    },

                { KEY_EN_1,                 0x08    },
                { KEY_EN_ISO_BACK_SLASH,    0x0B    },
                { KEY_EN_LEFT_WINDOWS,      0x0C    },

                { KEY_EN_2,                 0x10    },
                { KEY_EN_Q,                 0x09    },
                { KEY_EN_A,                 0x0A    },
                { KEY_EN_Z,                 0x13    },
                { KEY_EN_LEFT_ALT,          0x14    },

                { KEY_EN_3,                 0x18    },
                { KEY_EN_W,                 0x11    },
                { KEY_EN_S,                 0x12    },
                { KEY_EN_X,                 0x1B    },

                { KEY_EN_4,                 0x20    },
                { KEY_EN_E,                 0x19    },
                { KEY_EN_D,                 0x1A    },
                { KEY_EN_C,                 0x23    },

                { KEY_EN_5,                 0x28    },
                { KEY_EN_R,                 0x21    },
                { KEY_EN_F,                 0x22    },
                { KEY_EN_V,                 0x2B    },

                { KEY_EN_6,                 0x30    },
                { KEY_EN_T,                 0x29    },
                { KEY_EN_G,                 0x2A    },
                { KEY_EN_B,                 0x33    },
                { KEY_EN_SPACE,             0x34    },

                { KEY_EN_7,                 0x38    },
                { KEY_EN_Y,                 0x31    },
                { KEY_EN_H,                 0x32    },
                { KEY_EN_N,                 0x3B    },

                { KEY_EN_8,                 0x40    },
                { KEY_EN_U,                 0x39    },
                { KEY_EN_J,                 0x3A    },
                { KEY_EN_M,                 0x43    },

                { KEY_EN_9,                 0x48    },
                { KEY_EN_I,                 0x41    },
                { KEY_EN_K,                 0x42    },
                { KEY_EN_COMMA,             0x4B    },

                { KEY_EN_0,                 0x50    },
                { KEY_EN_O,                 0x49    },
                { KEY_EN_L,                 0x4A    },
                { KEY_EN_PERIOD,            0x53    },
                { KEY_EN_RIGHT_ALT,         0x4C    },

                { KEY_EN_MINUS,             0x58    },
                { KEY_EN_P,                 0x51    },
                { KEY_EN_SEMICOLON,         0x52    },
                { KEY_EN_FORWARD_SLASH,     0x5B    },
                { KEY_EN_RIGHT_FUNCTION,    0x54    },

                { KEY_EN_EQUALS,            0x60    },
                { KEY_EN_LEFT_BRACKET,      0x59    },
                { KEY_EN_QUOTE,             0x5A    },
                { KEY_EN_RIGHT_SHIFT,       0x63    },
                { KEY_EN_RIGHT_CONTROL,     0x5C    },

                { KEY_EN_BACKSPACE,         0x68    },
                { KEY_EN_RIGHT_BRACKET,     0x61    },
                { KEY_EN_POUND,             0x62    },
                { KEY_EN_LEFT_ARROW,        0x64    },

                { KEY_EN_ISO_ENTER,         0x6A    },
                { KEY_EN_UP_ARROW,          0x6B    },
                { KEY_EN_DOWN_ARROW,        0x6C    },

                { KEY_EN_INSERT,            0x70    },
                { KEY_EN_DELETE,            0x71    },
                { KEY_EN_PAGE_UP,           0x72    },
                { KEY_EN_PAGE_DOWN,         0x73    },
                { KEY_EN_RIGHT_ARROW,       0x74    },
            }
        }
    },
    {
        ASUS_TUF_K7_LAYOUT_US,
        {
            *ASUS_FALCHION_LAYOUT_KEYS_US,
            68,
            5,
            16,
            {
                { KEY_EN_ESCAPE,            0x00    },
                { KEY_EN_TAB,               0x01    },
                { KEY_EN_CAPS_LOCK,         0x02    },
                { KEY_EN_LEFT_SHIFT,        0x03    },
                { KEY_EN_LEFT_CONTROL,      0x04    },

                { KEY_EN_1,                 0x08    },
                { KEY_EN_LEFT_WINDOWS,      0x0C    },

                { KEY_EN_2,                 0x10    },
                { KEY_EN_Q,                 0x09    },
                { KEY_EN_A,                 0x0A    },
                { KEY_EN_Z,                 0x13    },
                { KEY_EN_LEFT_ALT,          0x14    },

                { KEY_EN_3,                 0x18    },
                { KEY_EN_W,                 0x11    },
                { KEY_EN_S,                 0x12    },
                { KEY_EN_X,                 0x1B    },

                { KEY_EN_4,                 0x20    },
                { KEY_EN_E,                 0x19    },
                { KEY_EN_D,                 0x1A    },
                { KEY_EN_C,                 0x23    },

                { KEY_EN_5,                 0x28    },
                { KEY_EN_R,                 0x21    },
                { KEY_EN_F,                 0x22    },
                { KEY_EN_V,                 0x2B    },

                { KEY_EN_6,                 0x30    },
                { KEY_EN_T,                 0x29    },
                { KEY_EN_G,                 0x2A    },
                { KEY_EN_B,                 0x33    },
                { KEY_EN_SPACE,             0x34    },

                { KEY_EN_7,                 0x38    },
                { KEY_EN_Y,                 0x31    },
                { KEY_EN_H,                 0x32    },
                { KEY_EN_N,                 0x3B    },

                { KEY_EN_8,                 0x40    },
                { KEY_EN_U,                 0x39    },
                { KEY_EN_J,                 0x3A    },
                { KEY_EN_M,                 0x43    },

                { KEY_EN_9,                 0x48    },
                { KEY_EN_I,                 0x41    },
                { KEY_EN_K,                 0x42    },
                { KEY_EN_COMMA,             0x4B    },

                { KEY_EN_0,                 0x50    },
                { KEY_EN_O,                 0x49    },
                { KEY_EN_L,                 0x4A    },
                { KEY_EN_PERIOD,            0x53    },
                { KEY_EN_RIGHT_ALT,         0x4C    },

                { KEY_EN_MINUS,             0x58    },
                { KEY_EN_P,                 0x51    },
                { KEY_EN_SEMICOLON,         0x52    },
                { KEY_EN_FORWARD_SLASH,     0x5B    },
                { KEY_EN_RIGHT_FUNCTION,    0x54    },

                { KEY_EN_EQUALS,            0x60    },
                { KEY_EN_LEFT_BRACKET,      0x59    },
                { KEY_EN_QUOTE,             0x5A    },
                { KEY_EN_RIGHT_SHIFT,       0x63    },
                { KEY_EN_RIGHT_CONTROL,     0x5C    },

                { KEY_EN_BACKSPACE,         0x68    },
                { KEY_EN_RIGHT_BRACKET,     0x61    },
                { KEY_EN_LEFT_ARROW,        0x64    },

                { KEY_EN_ANSI_BACK_SLASH,   0x69    },
                { KEY_EN_ANSI_ENTER,        0x6A    },
                { KEY_EN_UP_ARROW,          0x6B    },
                { KEY_EN_DOWN_ARROW,        0x6C    },

                { KEY_EN_INSERT,            0x70    },
                { KEY_EN_DELETE,            0x71    },
                { KEY_EN_PAGE_UP,           0x72    },
                { KEY_EN_PAGE_DOWN,         0x73    },
                { KEY_EN_RIGHT_ARROW,       0x74    },
            }
        }
    },
};


static std::map<int,layout_info> AsusClaymoreNoNumpadLayouts =
{
    {
        ASUS_TUF_K7_LAYOUT_UK,
        {
            *ASUS_CLAYMORE_NO_NUMPAD_LAYOUT_KEYS_UK,
            89,
            7,
            19,
            {
                { KEY_EN_ESCAPE,              0x00 },
                { KEY_EN_BACK_TICK,           0x01 },
                { KEY_EN_TAB,                 0x02 },
                { KEY_EN_CAPS_LOCK,           0x03 },
                { KEY_EN_LEFT_SHIFT,          0x04 },
                { KEY_EN_LEFT_CONTROL,        0x05 },
                { KEY_EN_1,                   0x09 },
                { KEY_EN_ISO_BACK_SLASH,      0x0C },
                { KEY_EN_LEFT_WINDOWS,        0x0D },
                { KEY_EN_F1,                  0x10 },
                { KEY_EN_2,                   0x11 },
                { KEY_EN_Q,                   0x0A },
                { KEY_EN_A,                   0x0B },
                { KEY_EN_Z,                   0x14 },
                { KEY_EN_LEFT_ALT,            0x15 },
                { KEY_EN_F2,                  0x18 },
                { KEY_EN_3,                   0x19 },
                { KEY_EN_W,                   0x12 },
                { KEY_EN_S,                   0x13 },
                { KEY_EN_X,                   0x1C },
                { KEY_EN_F3,                  0x20 },
                { KEY_EN_4,                   0x21 },
                { KEY_EN_E,                   0x1A },
                { KEY_EN_D,                   0x1B },
                { KEY_EN_C,                   0x24 },
                { KEY_EN_F4,                  0x28 },
                { KEY_EN_5,                   0x29 },
                { KEY_EN_R,                   0x22 },
                { KEY_EN_F,                   0x23 },
                { KEY_EN_V,                   0x2C },
                { KEY_EN_6,                   0x31 },
                { KEY_EN_T,                   0x2A },
                { KEY_EN_G,                   0x2B },
                { KEY_EN_B,                   0x34 },
                { KEY_EN_SPACE,               0x25 },
                { KEY_EN_F5,                  0x38 },
                { KEY_EN_7,                   0x39 },
                { KEY_EN_Y,                   0x32 },
                { KEY_EN_H,                   0x33 },
                { KEY_EN_N,                   0x3C },
                { KEY_EN_F6,                  0x40 },
                { KEY_EN_8,                   0x41 },
                { KEY_EN_U,                   0x3A },
                { KEY_EN_J,                   0x3B },
                { KEY_EN_M,                   0x44 },
                { "Logo",                     0x45 },
                { KEY_EN_F7,                  0x48 },
                { KEY_EN_9,                   0x49 },
                { KEY_EN_I,                   0x42 },
                { KEY_EN_K,                   0x43 },
                { KEY_EN_COMMA,               0x4C },
                { KEY_EN_F8,                  0x50 },
                { KEY_EN_0,                   0x51 },
                { KEY_EN_O,                   0x4A },
                { KEY_EN_L,                   0x4B },
                { KEY_EN_PERIOD,              0x54 },
                { KEY_EN_RIGHT_ALT,           0x4D },
                { KEY_EN_F9,                  0x60 },
                { KEY_EN_MINUS,               0x59 },
                { KEY_EN_P,                   0x52 },
                { KEY_EN_SEMICOLON,           0x53 },
                { KEY_EN_FORWARD_SLASH,       0x5C },
                { KEY_EN_RIGHT_FUNCTION,      0x55 },
                { KEY_EN_F10,                 0x68 },
                { KEY_EN_EQUALS,              0x61 },
                { KEY_EN_LEFT_BRACKET,        0x5A },
                { KEY_EN_QUOTE,               0x5B },
                { KEY_EN_MENU,                0x5D },
                { KEY_EN_F11,                 0x70 },
                { KEY_EN_BACKSPACE,           0x69 },
                { KEY_EN_RIGHT_BRACKET,       0x62 },
                { KEY_EN_POUND,               0x63 },
                { KEY_EN_RIGHT_SHIFT,         0x6C },
                { KEY_EN_F12,                 0x78 },
                { KEY_EN_ISO_ENTER,           0x6B },
                { KEY_EN_RIGHT_CONTROL,       0x6D },
                { KEY_EN_PRINT_SCREEN,        0x80 },
                { KEY_EN_INSERT,              0x81 },
                { KEY_EN_DELETE,              0x82 },
                { KEY_EN_LEFT_ARROW,          0x85 },
                { KEY_EN_SCROLL_LOCK,         0x88 },
                { KEY_EN_HOME,                0x89 },
                { KEY_EN_END,                 0x8A },
                { KEY_EN_UP_ARROW,            0x8C },
                { KEY_EN_DOWN_ARROW,          0x8D },
                { KEY_EN_PAUSE_BREAK,         0x90 },
                { KEY_EN_PAGE_UP,             0x91 },
                { KEY_EN_PAGE_DOWN,           0x92 },
                { KEY_EN_RIGHT_ARROW,         0x95 }
            }
        }
    },
    {
        ASUS_TUF_K7_LAYOUT_US,
        {
            *ASUS_CLAYMORE_NO_NUMPAD_LAYOUT_KEYS_US,
            88,
            7,
            19,
            {
                { KEY_EN_ESCAPE,              0x00 },
                { KEY_EN_BACK_TICK,           0x01 },
                { KEY_EN_TAB,                 0x02 },
                { KEY_EN_CAPS_LOCK,           0x03 },
                { KEY_EN_LEFT_SHIFT,          0x04 },
                { KEY_EN_LEFT_CONTROL,        0x05 },
                { KEY_EN_1,                   0x09 },
                { KEY_EN_LEFT_WINDOWS,        0x0D },
                { KEY_EN_F1,                  0x10 },
                { KEY_EN_2,                   0x11 },
                { KEY_EN_Q,                   0x0A },
                { KEY_EN_A,                   0x0B },
                { KEY_EN_Z,                   0x0C },
                { KEY_EN_LEFT_ALT,            0x15 },
                { KEY_EN_F2,                  0x18 },
                { KEY_EN_3,                   0x19 },
                { KEY_EN_W,                   0x12 },
                { KEY_EN_S,                   0x13 },
                { KEY_EN_X,                   0x14 },
                { KEY_EN_F3,                  0x20 },
                { KEY_EN_4,                   0x21 },
                { KEY_EN_E,                   0x1A },
                { KEY_EN_D,                   0x1B },
                { KEY_EN_C,                   0x1C },
                { KEY_EN_F4,                  0x28 },
                { KEY_EN_5,                   0x29 },
                { KEY_EN_R,                   0x22 },
                { KEY_EN_F,                   0x23 },
                { KEY_EN_V,                   0x24 },
                { KEY_EN_6,                   0x31 },
                { KEY_EN_T,                   0x2A },
                { KEY_EN_G,                   0x2B },
                { KEY_EN_B,                   0x2C },
                { KEY_EN_SPACE,               0x25 },
                { KEY_EN_F5,                  0x38 },
                { KEY_EN_7,                   0x39 },
                { KEY_EN_Y,                   0x32 },
                { KEY_EN_H,                   0x33 },
                { KEY_EN_N,                   0x34 },
                { KEY_EN_F6,                  0x40 },
                { KEY_EN_8,                   0x41 },
                { KEY_EN_U,                   0x3A },
                { KEY_EN_J,                   0x3B },
                { KEY_EN_M,                   0x3C },
                { "Logo",                     0x3D },
                { KEY_EN_F7,                  0x48 },
                { KEY_EN_9,                   0x49 },
                { KEY_EN_I,                   0x42 },
                { KEY_EN_K,                   0x43 },
                { KEY_EN_COMMA,               0x44 },
                { KEY_EN_F8,                  0x50 },
                { KEY_EN_0,                   0x51 },
                { KEY_EN_O,                   0x4A },
                { KEY_EN_L,                   0x4B },
                { KEY_EN_PERIOD,              0x4C },
                { KEY_EN_RIGHT_ALT,           0x4D },
                { KEY_EN_F9,                  0x60 },
                { KEY_EN_MINUS,               0x59 },
                { KEY_EN_P,                   0x52 },
                { KEY_EN_SEMICOLON,           0x53 },
                { KEY_EN_FORWARD_SLASH,       0x54 },
                { KEY_EN_RIGHT_FUNCTION,      0x55 },
                { KEY_EN_F10,                 0x68 },
                { KEY_EN_EQUALS,              0x61 },
                { KEY_EN_LEFT_BRACKET,        0x5A },
                { KEY_EN_QUOTE,               0x5B },
                { KEY_EN_MENU,                0x5D },
                { KEY_EN_F11,                 0x70 },
                { KEY_EN_BACKSPACE,           0x69 },
                { KEY_EN_RIGHT_BRACKET,       0x62 },
                { KEY_EN_RIGHT_SHIFT,         0x6C },
                { KEY_EN_F12,                 0x78 },
                { KEY_EN_ANSI_BACK_SLASH,     0x6A },
                { KEY_EN_ANSI_ENTER,          0x6B },
                { KEY_EN_RIGHT_CONTROL,       0x6D },
                { KEY_EN_PRINT_SCREEN,        0x80 },
                { KEY_EN_INSERT,              0x81 },
                { KEY_EN_DELETE,              0x82 },
                { KEY_EN_LEFT_ARROW,          0x85 },
                { KEY_EN_SCROLL_LOCK,         0x88 },
                { KEY_EN_HOME,                0x89 },
                { KEY_EN_END,                 0x8A },
                { KEY_EN_UP_ARROW,            0x8C },
                { KEY_EN_DOWN_ARROW,          0x8D },
                { KEY_EN_PAUSE_BREAK,         0x90 },
                { KEY_EN_PAGE_UP,             0x91 },
                { KEY_EN_PAGE_DOWN,           0x92 },
                { KEY_EN_RIGHT_ARROW,         0x95 }
            }
        }
    },
};

static std::map<int,layout_info> AsusClaymoreNumpadRightLayouts =
{
    {
        ASUS_TUF_K7_LAYOUT_UK,
        {
            *ASUS_CLAYMORE_NUMPAD_RIGHT_LAYOUT_KEYS_UK,
            106,
            7,
            24,
            {
                { KEY_EN_ESCAPE,              0x00 },
                { KEY_EN_BACK_TICK,           0x01 },
                { KEY_EN_TAB,                 0x02 },
                { KEY_EN_CAPS_LOCK,           0x03 },
                { KEY_EN_LEFT_SHIFT,          0x04 },
                { KEY_EN_LEFT_CONTROL,        0x05 },
                { KEY_EN_1,                   0x09 },
                { KEY_EN_ISO_BACK_SLASH,      0x0C },
                { KEY_EN_LEFT_WINDOWS,        0x0D },
                { KEY_EN_F1,                  0x10 },
                { KEY_EN_2,                   0x11 },
                { KEY_EN_Q,                   0x0A },
                { KEY_EN_A,                   0x0B },
                { KEY_EN_Z,                   0x14 },
                { KEY_EN_LEFT_ALT,            0x15 },
                { KEY_EN_F2,                  0x18 },
                { KEY_EN_3,                   0x19 },
                { KEY_EN_W,                   0x12 },
                { KEY_EN_S,                   0x13 },
                { KEY_EN_X,                   0x1C },
                { KEY_EN_F3,                  0x20 },
                { KEY_EN_4,                   0x21 },
                { KEY_EN_E,                   0x1A },
                { KEY_EN_D,                   0x1B },
                { KEY_EN_C,                   0x24 },
                { KEY_EN_F4,                  0x28 },
                { KEY_EN_5,                   0x29 },
                { KEY_EN_R,                   0x22 },
                { KEY_EN_F,                   0x23 },
                { KEY_EN_V,                   0x2C },
                { KEY_EN_6,                   0x31 },
                { KEY_EN_T,                   0x2A },
                { KEY_EN_G,                   0x2B },
                { KEY_EN_B,                   0x34 },
                { KEY_EN_SPACE,               0x25 },
                { KEY_EN_F5,                  0x38 },
                { KEY_EN_7,                   0x39 },
                { KEY_EN_Y,                   0x32 },
                { KEY_EN_H,                   0x33 },
                { KEY_EN_N,                   0x3C },
                { KEY_EN_F6,                  0x40 },
                { KEY_EN_8,                   0x41 },
                { KEY_EN_U,                   0x3A },
                { KEY_EN_J,                   0x3B },
                { KEY_EN_M,                   0x44 },
                { "Logo",                     0x45 },
                { KEY_EN_F7,                  0x48 },
                { KEY_EN_9,                   0x49 },
                { KEY_EN_I,                   0x42 },
                { KEY_EN_K,                   0x43 },
                { KEY_EN_COMMA,               0x4C },
                { KEY_EN_F8,                  0x50 },
                { KEY_EN_0,                   0x51 },
                { KEY_EN_O,                   0x4A },
                { KEY_EN_L,                   0x4B },
                { KEY_EN_PERIOD,              0x54 },
                { KEY_EN_RIGHT_ALT,           0x4D },
                { KEY_EN_F9,                  0x60 },
                { KEY_EN_MINUS,               0x59 },
                { KEY_EN_P,                   0x52 },
                { KEY_EN_SEMICOLON,           0x53 },
                { KEY_EN_FORWARD_SLASH,       0x5C },
                { KEY_EN_RIGHT_FUNCTION,      0x55 },
                { KEY_EN_F10,                 0x68 },
                { KEY_EN_EQUALS,              0x61 },
                { KEY_EN_LEFT_BRACKET,        0x5A },
                { KEY_EN_QUOTE,               0x5B },
                { KEY_EN_MENU,                0x5D },
                { KEY_EN_F11,                 0x70 },
                { KEY_EN_BACKSPACE,           0x69 },
                { KEY_EN_RIGHT_BRACKET,       0x62 },
                { KEY_EN_POUND,               0x63 },
                { KEY_EN_RIGHT_SHIFT,         0x6C },
                { KEY_EN_F12,                 0x78 },
                { KEY_EN_ISO_ENTER,           0x6B },
                { KEY_EN_RIGHT_CONTROL,       0x6D },
                { KEY_EN_PRINT_SCREEN,        0x80 },
                { KEY_EN_INSERT,              0x81 },
                { KEY_EN_DELETE,              0x82 },
                { KEY_EN_LEFT_ARROW,          0x85 },
                { KEY_EN_SCROLL_LOCK,         0x88 },
                { KEY_EN_HOME,                0x89 },
                { KEY_EN_END,                 0x8A },
                { KEY_EN_UP_ARROW,            0x8C },
                { KEY_EN_DOWN_ARROW,          0x8D },
                { KEY_EN_PAUSE_BREAK,         0x90 },
                { KEY_EN_PAGE_UP,             0x91 },
                { KEY_EN_PAGE_DOWN,           0x92 },
                { KEY_EN_RIGHT_ARROW,         0x95 },
                { KEY_EN_NUMPAD_LOCK,         0x99 },
                { KEY_EN_NUMPAD_7,            0x9A },
                { KEY_EN_NUMPAD_4,            0x9B },
                { KEY_EN_NUMPAD_1,            0x9C },
                { KEY_EN_NUMPAD_0,            0x9D },
                { KEY_EN_NUMPAD_DIVIDE,       0xA1 },
                { KEY_EN_NUMPAD_8,            0xA2 },
                { KEY_EN_NUMPAD_5,            0xA3 },
                { KEY_EN_NUMPAD_2,            0xA4 },
                { KEY_EN_NUMPAD_TIMES,        0xA9 },
                { KEY_EN_NUMPAD_9,            0xAA },
                { KEY_EN_NUMPAD_6,            0xAB },
                { KEY_EN_NUMPAD_3,            0xAC },
                { KEY_EN_NUMPAD_PERIOD,       0xAD },
                { KEY_EN_NUMPAD_MINUS,        0xB1 },
                { KEY_EN_NUMPAD_PLUS,         0xB2 },
                { KEY_EN_NUMPAD_ENTER,        0xB4 }
            }
        }
    },
    {
        ASUS_TUF_K7_LAYOUT_US,
        {
            *ASUS_CLAYMORE_NUMPAD_RIGHT_LAYOUT_KEYS_US,
            105,
            7,
            24,
            {
                { KEY_EN_ESCAPE,              0x00 },
                { KEY_EN_BACK_TICK,           0x01 },
                { KEY_EN_TAB,                 0x02 },
                { KEY_EN_CAPS_LOCK,           0x03 },
                { KEY_EN_LEFT_SHIFT,          0x04 },
                { KEY_EN_LEFT_CONTROL,        0x05 },
                { KEY_EN_1,                   0x09 },
                { KEY_EN_LEFT_WINDOWS,        0x0D },
                { KEY_EN_F1,                  0x10 },
                { KEY_EN_2,                   0x11 },
                { KEY_EN_Q,                   0x0A },
                { KEY_EN_A,                   0x0B },
                { KEY_EN_Z,                   0x0C },
                { KEY_EN_LEFT_ALT,            0x15 },
                { KEY_EN_F2,                  0x18 },
                { KEY_EN_3,                   0x19 },
                { KEY_EN_W,                   0x12 },
                { KEY_EN_S,                   0x13 },
                { KEY_EN_X,                   0x14 },
                { KEY_EN_F3,                  0x20 },
                { KEY_EN_4,                   0x21 },
                { KEY_EN_E,                   0x1A },
                { KEY_EN_D,                   0x1B },
                { KEY_EN_C,                   0x1C },
                { KEY_EN_F4,                  0x28 },
                { KEY_EN_5,                   0x29 },
                { KEY_EN_R,                   0x22 },
                { KEY_EN_F,                   0x23 },
                { KEY_EN_V,                   0x24 },
                { KEY_EN_6,                   0x31 },
                { KEY_EN_T,                   0x2A },
                { KEY_EN_G,                   0x2B },
                { KEY_EN_B,                   0x2C },
                { KEY_EN_SPACE,               0x25 },
                { KEY_EN_F5,                  0x38 },
                { KEY_EN_7,                   0x39 },
                { KEY_EN_Y,                   0x32 },
                { KEY_EN_H,                   0x33 },
                { KEY_EN_N,                   0x34 },
                { KEY_EN_F6,                  0x40 },
                { KEY_EN_8,                   0x41 },
                { KEY_EN_U,                   0x3A },
                { KEY_EN_J,                   0x3B },
                { KEY_EN_M,                   0x3C },
                { "Logo",                     0x3D },
                { KEY_EN_F7,                  0x48 },
                { KEY_EN_9,                   0x49 },
                { KEY_EN_I,                   0x42 },
                { KEY_EN_K,                   0x43 },
                { KEY_EN_COMMA,               0x44 },
                { KEY_EN_F8,                  0x50 },
                { KEY_EN_0,                   0x51 },
                { KEY_EN_O,                   0x4A },
                { KEY_EN_L,                   0x4B },
                { KEY_EN_PERIOD,              0x4C },
                { KEY_EN_RIGHT_ALT,           0x4D },
                { KEY_EN_F9,                  0x60 },
                { KEY_EN_MINUS,               0x59 },
                { KEY_EN_P,                   0x52 },
                { KEY_EN_SEMICOLON,           0x53 },
                { KEY_EN_FORWARD_SLASH,       0x54 },
                { KEY_EN_RIGHT_FUNCTION,      0x55 },
                { KEY_EN_F10,                 0x68 },
                { KEY_EN_EQUALS,              0x61 },
                { KEY_EN_LEFT_BRACKET,        0x5A },
                { KEY_EN_QUOTE,               0x5B },
                { KEY_EN_MENU,                0x5D },
                { KEY_EN_F11,                 0x70 },
                { KEY_EN_BACKSPACE,           0x69 },
                { KEY_EN_RIGHT_BRACKET,       0x62 },
                { KEY_EN_RIGHT_SHIFT,         0x6C },
                { KEY_EN_F12,                 0x78 },
                { KEY_EN_ANSI_BACK_SLASH,     0x6A },
                { KEY_EN_ANSI_ENTER,          0x6B },
                { KEY_EN_RIGHT_CONTROL,       0x6D },
                { KEY_EN_PRINT_SCREEN,        0x80 },
                { KEY_EN_INSERT,              0x81 },
                { KEY_EN_DELETE,              0x82 },
                { KEY_EN_LEFT_ARROW,          0x85 },
                { KEY_EN_SCROLL_LOCK,         0x88 },
                { KEY_EN_HOME,                0x89 },
                { KEY_EN_END,                 0x8A },
                { KEY_EN_UP_ARROW,            0x8C },
                { KEY_EN_DOWN_ARROW,          0x8D },
                { KEY_EN_PAUSE_BREAK,         0x90 },
                { KEY_EN_PAGE_UP,             0x91 },
                { KEY_EN_PAGE_DOWN,           0x92 },
                { KEY_EN_RIGHT_ARROW,         0x95 },
                { KEY_EN_NUMPAD_LOCK,         0x99 },
                { KEY_EN_NUMPAD_7,            0x9A },
                { KEY_EN_NUMPAD_4,            0x9B },
                { KEY_EN_NUMPAD_1,            0x9C },
                { KEY_EN_NUMPAD_0,            0x9D },
                { KEY_EN_NUMPAD_DIVIDE,       0xA1 },
                { KEY_EN_NUMPAD_8,            0xA2 },
                { KEY_EN_NUMPAD_5,            0xA3 },
                { KEY_EN_NUMPAD_2,            0xA4 },
                { KEY_EN_NUMPAD_TIMES,        0xA9 },
                { KEY_EN_NUMPAD_9,            0xAA },
                { KEY_EN_NUMPAD_6,            0xAB },
                { KEY_EN_NUMPAD_3,            0xAC },
                { KEY_EN_NUMPAD_PERIOD,       0xAD },
                { KEY_EN_NUMPAD_MINUS,        0xB1 },
                { KEY_EN_NUMPAD_PLUS,         0xB2 },
                { KEY_EN_NUMPAD_ENTER,        0xB4 }
            }
        }
    },
};

static std::map<int,layout_info> AsusClaymoreNumpadLeftLayouts =
{
    {
        ASUS_TUF_K7_LAYOUT_UK,
        {
            *ASUS_CLAYMORE_NUMPAD_LEFT_LAYOUT_KEYS_UK,
            106,
            7,
            24,
            {
                { KEY_EN_NUMPAD_LOCK,         0x01 },
                { KEY_EN_NUMPAD_7,            0x02 },
                { KEY_EN_NUMPAD_4,            0x03 },
                { KEY_EN_NUMPAD_1,            0x04 },
                { KEY_EN_NUMPAD_0,            0x05 },
                { KEY_EN_NUMPAD_DIVIDE,       0x09 },
                { KEY_EN_NUMPAD_8,            0x0A },
                { KEY_EN_NUMPAD_5,            0x0B },
                { KEY_EN_NUMPAD_2,            0x0C },
                { KEY_EN_NUMPAD_TIMES,        0x11 },
                { KEY_EN_NUMPAD_9,            0x12 },
                { KEY_EN_NUMPAD_6,            0x13 },
                { KEY_EN_NUMPAD_3,            0x14 },
                { KEY_EN_NUMPAD_PERIOD,       0x15 },
                { KEY_EN_NUMPAD_MINUS,        0x19 },
                { KEY_EN_NUMPAD_PLUS,         0x1A },
                { KEY_EN_NUMPAD_ENTER,        0x1C },
                { KEY_EN_ESCAPE,              0x20 },
                { KEY_EN_BACK_TICK,           0x21 },
                { KEY_EN_TAB,                 0x22 },
                { KEY_EN_CAPS_LOCK,           0x23 },
                { KEY_EN_LEFT_SHIFT,          0x24 },
                { KEY_EN_LEFT_CONTROL,        0x25 },
                { KEY_EN_1,                   0x29 },
                { KEY_EN_ISO_BACK_SLASH,      0x2C },
                { KEY_EN_LEFT_WINDOWS,        0x2D },
                { KEY_EN_F1,                  0x30 },
                { KEY_EN_2,                   0x31 },
                { KEY_EN_Q,                   0x2A },
                { KEY_EN_A,                   0x2B },
                { KEY_EN_Z,                   0x34 },
                { KEY_EN_LEFT_ALT,            0x35 },
                { KEY_EN_F2,                  0x38 },
                { KEY_EN_3,                   0x39 },
                { KEY_EN_W,                   0x32 },
                { KEY_EN_S,                   0x33 },
                { KEY_EN_X,                   0x3C },
                { KEY_EN_F3,                  0x40 },
                { KEY_EN_4,                   0x41 },
                { KEY_EN_E,                   0x3A },
                { KEY_EN_D,                   0x3B },
                { KEY_EN_C,                   0x44 },
                { KEY_EN_F4,                  0x48 },
                { KEY_EN_5,                   0x49 },
                { KEY_EN_R,                   0x42 },
                { KEY_EN_F,                   0x43 },
                { KEY_EN_V,                   0x4C },
                { KEY_EN_6,                   0x51 },
                { KEY_EN_T,                   0x4A },
                { KEY_EN_G,                   0x4B },
                { KEY_EN_B,                   0x54 },
                { KEY_EN_SPACE,               0x45 },
                { KEY_EN_F5,                  0x58 },
                { KEY_EN_7,                   0x59 },
                { KEY_EN_Y,                   0x52 },
                { KEY_EN_H,                   0x53 },
                { KEY_EN_N,                   0x5C },
                { KEY_EN_F6,                  0x60 },
                { KEY_EN_8,                   0x61 },
                { KEY_EN_U,                   0x5A },
                { KEY_EN_J,                   0x5B },
                { KEY_EN_M,                   0x64 },
                { "Logo",                     0x65 },
                { KEY_EN_F7,                  0x68 },
                { KEY_EN_9,                   0x69 },
                { KEY_EN_I,                   0x62 },
                { KEY_EN_K,                   0x63 },
                { KEY_EN_COMMA,               0x6C },
                { KEY_EN_F8,                  0x70 },
                { KEY_EN_0,                   0x71 },
                { KEY_EN_O,                   0x6A },
                { KEY_EN_L,                   0x6B },
                { KEY_EN_PERIOD,              0x74 },
                { KEY_EN_RIGHT_ALT,           0x6D },
                { KEY_EN_F9,                  0x80 },
                { KEY_EN_MINUS,               0x79 },
                { KEY_EN_P,                   0x72 },
                { KEY_EN_SEMICOLON,           0x73 },
                { KEY_EN_FORWARD_SLASH,       0x7C },
                { KEY_EN_RIGHT_FUNCTION,      0x75 },
                { KEY_EN_F10,                 0x88 },
                { KEY_EN_EQUALS,              0x81 },
                { KEY_EN_LEFT_BRACKET,        0x7A },
                { KEY_EN_QUOTE,               0x7B },
                { KEY_EN_MENU,                0x7D },
                { KEY_EN_F11,                 0x90 },
                { KEY_EN_BACKSPACE,           0x89 },
                { KEY_EN_RIGHT_BRACKET,       0x82 },
                { KEY_EN_POUND,               0x83 },
                { KEY_EN_RIGHT_SHIFT,         0x8C },
                { KEY_EN_F12,                 0x98 },
                { KEY_EN_ISO_ENTER,           0x8B },
                { KEY_EN_RIGHT_CONTROL,       0x8D },
                { KEY_EN_PRINT_SCREEN,        0xA0 },
                { KEY_EN_INSERT,              0xA1 },
                { KEY_EN_DELETE,              0xA2 },
                { KEY_EN_LEFT_ARROW,          0xA5 },
                { KEY_EN_SCROLL_LOCK,         0xA8 },
                { KEY_EN_HOME,                0xA9 },
                { KEY_EN_END,                 0xAA },
                { KEY_EN_UP_ARROW,            0xAC },
                { KEY_EN_DOWN_ARROW,          0xAD },
                { KEY_EN_PAUSE_BREAK,         0xB0 },
                { KEY_EN_PAGE_UP,             0xB1 },
                { KEY_EN_PAGE_DOWN,           0xB2 },
                { KEY_EN_RIGHT_ARROW,         0xB5 }
            }
        }
    },
    {
        ASUS_TUF_K7_LAYOUT_US,
        {
            *ASUS_CLAYMORE_NUMPAD_LEFT_LAYOUT_KEYS_US,
            105,
            7,
            24,
            {
                { KEY_EN_NUMPAD_LOCK,         0x01 },
                { KEY_EN_NUMPAD_7,            0x02 },
                { KEY_EN_NUMPAD_4,            0x03 },
                { KEY_EN_NUMPAD_1,            0x04 },
                { KEY_EN_NUMPAD_0,            0x05 },
                { KEY_EN_NUMPAD_DIVIDE,       0x09 },
                { KEY_EN_NUMPAD_8,            0x0A },
                { KEY_EN_NUMPAD_5,            0x0B },
                { KEY_EN_NUMPAD_2,            0x0C },
                { KEY_EN_NUMPAD_TIMES,        0x11 },
                { KEY_EN_NUMPAD_9,            0x12 },
                { KEY_EN_NUMPAD_6,            0x13 },
                { KEY_EN_NUMPAD_3,            0x14 },
                { KEY_EN_NUMPAD_PERIOD,       0x15 },
                { KEY_EN_NUMPAD_MINUS,        0x19 },
                { KEY_EN_NUMPAD_PLUS,         0x1A },
                { KEY_EN_NUMPAD_ENTER,        0x1C },
                { KEY_EN_ESCAPE,              0x20 },
                { KEY_EN_BACK_TICK,           0x21 },
                { KEY_EN_TAB,                 0x22 },
                { KEY_EN_CAPS_LOCK,           0x23 },
                { KEY_EN_LEFT_SHIFT,          0x24 },
                { KEY_EN_LEFT_CONTROL,        0x25 },
                { KEY_EN_1,                   0x29 },
                { KEY_EN_LEFT_WINDOWS,        0x2D },
                { KEY_EN_F1,                  0x30 },
                { KEY_EN_2,                   0x31 },
                { KEY_EN_Q,                   0x2A },
                { KEY_EN_A,                   0x2B },
                { KEY_EN_Z,                   0x2C },
                { KEY_EN_LEFT_ALT,            0x35 },
                { KEY_EN_F2,                  0x38 },
                { KEY_EN_3,                   0x39 },
                { KEY_EN_W,                   0x32 },
                { KEY_EN_S,                   0x33 },
                { KEY_EN_X,                   0x34 },
                { KEY_EN_F3,                  0x40 },
                { KEY_EN_4,                   0x41 },
                { KEY_EN_E,                   0x3A },
                { KEY_EN_D,                   0x3B },
                { KEY_EN_C,                   0x3C },
                { KEY_EN_F4,                  0x48 },
                { KEY_EN_5,                   0x49 },
                { KEY_EN_R,                   0x42 },
                { KEY_EN_F,                   0x43 },
                { KEY_EN_V,                   0x44 },
                { KEY_EN_6,                   0x51 },
                { KEY_EN_T,                   0x4A },
                { KEY_EN_G,                   0x4B },
                { KEY_EN_B,                   0x4C },
                { KEY_EN_SPACE,               0x45 },
                { KEY_EN_F5,                  0x58 },
                { KEY_EN_7,                   0x59 },
                { KEY_EN_Y,                   0x52 },
                { KEY_EN_H,                   0x53 },
                { KEY_EN_N,                   0x54 },
                { KEY_EN_F6,                  0x60 },
                { KEY_EN_8,                   0x61 },
                { KEY_EN_U,                   0x5A },
                { KEY_EN_J,                   0x5B },
                { KEY_EN_M,                   0x5C },
                { "Logo",                     0x5D },
                { KEY_EN_F7,                  0x68 },
                { KEY_EN_9,                   0x69 },
                { KEY_EN_I,                   0x62 },
                { KEY_EN_K,                   0x63 },
                { KEY_EN_COMMA,               0x64 },
                { KEY_EN_F8,                  0x70 },
                { KEY_EN_0,                   0x71 },
                { KEY_EN_O,                   0x6A },
                { KEY_EN_L,                   0x6B },
                { KEY_EN_PERIOD,              0x6C },
                { KEY_EN_RIGHT_ALT,           0x6D },
                { KEY_EN_F9,                  0x80 },
                { KEY_EN_MINUS,               0x79 },
                { KEY_EN_P,                   0x72 },
                { KEY_EN_SEMICOLON,           0x73 },
                { KEY_EN_FORWARD_SLASH,       0x74 },
                { KEY_EN_RIGHT_FUNCTION,      0x75 },
                { KEY_EN_F10,                 0x88 },
                { KEY_EN_EQUALS,              0x81 },
                { KEY_EN_LEFT_BRACKET,        0x7A },
                { KEY_EN_QUOTE,               0x7B },
                { KEY_EN_MENU,                0x7D },
                { KEY_EN_F11,                 0x90 },
                { KEY_EN_BACKSPACE,           0x89 },
                { KEY_EN_RIGHT_BRACKET,       0x82 },
                { KEY_EN_RIGHT_SHIFT,         0x8C },
                { KEY_EN_F12,                 0x98 },
                { KEY_EN_ANSI_BACK_SLASH,     0x8A },
                { KEY_EN_ANSI_ENTER,          0x8B },
                { KEY_EN_RIGHT_CONTROL,       0x8D },
                { KEY_EN_PRINT_SCREEN,        0xA0 },
                { KEY_EN_INSERT,              0xA1 },
                { KEY_EN_DELETE,              0xA2 },
                { KEY_EN_LEFT_ARROW,          0xA5 },
                { KEY_EN_SCROLL_LOCK,         0xA8 },
                { KEY_EN_HOME,                0xA9 },
                { KEY_EN_END,                 0xAA },
                { KEY_EN_UP_ARROW,            0xAC },
                { KEY_EN_DOWN_ARROW,          0xAD },
                { KEY_EN_PAUSE_BREAK,         0xB0 },
                { KEY_EN_PAGE_UP,             0xB1 },
                { KEY_EN_PAGE_DOWN,           0xB2 },
                { KEY_EN_RIGHT_ARROW,         0xB5 }
            }
        }
    },
};

static std::map<int,layout_info> AsusTufK1Layouts =
{
    {
        ASUS_TUF_K7_LAYOUT_UK,
        {
            *ASUS_TUF_K1_LAYOUT_KEYS,
            5,
            1,
            5,
            {
                { "Keyboard LED 1",             0x00 },
                { "Keyboard LED 2",             0x01 },
                { "Keyboard LED 3",             0x02 },
                { "Keyboard LED 4",             0x03 },
                { "Keyboard LED 5",             0x04 },
            }
        }
    },
    {
        ASUS_TUF_K7_LAYOUT_US,
        {
            *ASUS_TUF_K1_LAYOUT_KEYS,
            5,
            1,
            5,
            {
                { "Keyboard LED 1",             0x00 },
                { "Keyboard LED 2",             0x01 },
                { "Keyboard LED 3",             0x02 },
                { "Keyboard LED 4",             0x03 },
                { "Keyboard LED 5",             0x04 },
            }
        }
    },
};
