use crate::util::constants;
use chrono::NaiveDateTime;
use chrono::{Duration, Utc};
use diffus::{Diffable, Same};
use news_flash::models::{Article, ArticleID, Feed, FeedID, Marked, Read, Tag};
use std::sync::Arc;

#[derive(Debug, Clone)]
pub struct ArticleListArticleModel {
    // immutable data
    pub id: ArticleID,
    pub title: Arc<String>,
    pub feed_id: FeedID,
    pub feed_title: Arc<String>,
    pub summary: Arc<String>,

    // mutable data
    pub date: NaiveDateTime,
    pub read: Read,
    pub marked: Marked,
    pub tags: Vec<Tag>,
}

impl ArticleListArticleModel {
    pub fn new(article: Article, feed: Option<&Feed>, tags: Vec<&Tag>) -> Self {
        let Article {
            article_id,
            title,
            author: _,
            feed_id,
            date,
            summary,
            direction: _,
            marked,
            thumbnail_url: _,
            url: _,
            synced: _,
            unread,
        } = article;

        ArticleListArticleModel {
            id: article_id,
            title: Arc::new(match title {
                Some(title) => title,
                None => constants::NO_TITLE.to_owned(),
            }),
            feed_id,
            feed_title: Arc::new(feed.map(|f| f.label.clone()).unwrap_or(constants::UNKNOWN_FEED.into())),
            summary: Arc::new(match summary {
                Some(summary) => summary,
                None => "No Summary".to_owned(),
            }),
            date,
            read: unread,
            marked,
            tags: tags.into_iter().cloned().collect(),
        }
    }
}

impl PartialEq for ArticleListArticleModel {
    fn eq(&self, other: &ArticleListArticleModel) -> bool {
        self.id == other.id
    }
}

impl Same for ArticleListArticleModel {
    fn same(&self, other: &Self) -> bool {
        self.id == other.id
    }
}

impl<'a> Diffable<'a> for ArticleListArticleModel {
    type Diff = ArticleDiff;

    fn diff(&'a self, other: &'a Self) -> diffus::edit::Edit<Self> {
        let date = if self.date == other.date {
            let diff_days = Utc::now().naive_utc().date() - self.date.date();
            if diff_days > Duration::days(0) && diff_days <= Duration::days(1) {
                Some(other.date)
            } else {
                None
            }
        } else {
            Some(other.date)
        };

        let read = if self.read == other.read {
            None
        } else {
            Some(other.read)
        };

        let marked = if self.marked == other.marked {
            None
        } else {
            Some(other.marked)
        };

        let tags = if self.tags.eq(&other.tags) {
            None
        } else {
            Some(other.tags.clone())
        };

        if self == other && date.is_none() && read.is_none() && marked.is_none() {
            diffus::edit::Edit::Copy(self)
        } else {
            diffus::edit::Edit::Change(ArticleDiff {
                id: self.id.clone(),
                read,
                marked,
                date,
                tags,
            })
        }
    }
}

#[derive(Debug, Clone)]
pub struct ArticleDiff {
    pub id: ArticleID,
    pub read: Option<Read>,
    pub marked: Option<Marked>,
    pub date: Option<NaiveDateTime>,
    pub tags: Option<Vec<Tag>>,
}
