
/*  Copyright (C) 2022 CZ.NIC, z.s.p.o. <knot-dns@labs.nic.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <math.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "libzscanner/scanner.h"
#include "libzscanner/functions.h"
#include "libknot/descriptor.h"

/*! \brief Maximal length of rdata item. */
#define MAX_ITEM_LENGTH		255
#define MAX_ITEM_LENGTH2	65535

/*! \brief Latitude value for equator (2^31). */
#define LOC_LAT_ZERO	(uint32_t)2147483648
/*! \brief Longitude value for meridian (2^31). */
#define LOC_LONG_ZERO	(uint32_t)2147483648
/*! \brief Zero level altitude value. */
#define LOC_ALT_ZERO	(uint32_t)10000000

/*! \brief Shorthand for setting warning data. */
#define WARN(err_code) { s->error.code = err_code; }
/*! \brief Shorthand for setting error data. */
#define ERR(err_code) { WARN(err_code); s->error.fatal = true; }
/*! \brief Shorthand for error reset. */
#define NOERR { WARN(ZS_OK); s->error.fatal = false; }

/*!
 * \brief Writes record type number to r_data.
 *
 * \param type		Type number.
 * \param rdata_tail	Position where to write type number to.
 */
static inline void type_num(const uint16_t type, uint8_t **rdata_tail)
{
	*((uint16_t *)*rdata_tail) = htons(type);
	*rdata_tail += 2;
}

/*!
 * \brief Sets bit to bitmap window.
 *
 * \param type		Type number.
 * \param s		Scanner context.
 */
static inline void window_add_bit(const uint16_t type, zs_scanner_t *s) {
	uint8_t win      = type / 256;
	uint8_t bit_pos  = type % 256;
	uint8_t byte_pos = bit_pos / 8;

	((s->windows[win]).bitmap)[byte_pos] |= 128 >> (bit_pos % 8);

	if ((s->windows[win]).length < byte_pos + 1) {
		(s->windows[win]).length = byte_pos + 1;
	}

	if (s->last_window < win) {
		s->last_window = win;
	}
}

// Include scanner file (in Ragel).






__attribute__((visibility("default")))
int zs_init(
	zs_scanner_t *s,
	const char *origin,
	const uint16_t rclass,
	const uint32_t ttl)
{
	if (s == NULL) {
		return -1;
	}

	memset(s, 0, sizeof(*s));

	// Nonzero initial scanner state.
	s->cs = 1396;

	// Reset the file descriptor.
	s->file.descriptor = -1;

	// Use the root zone as origin if not specified.
	if (origin == NULL || strlen(origin) == 0) {
		origin = ".";
	}
	size_t origin_len = strlen(origin);

	// Prepare a zone settings header.
	const char *format;
	if (origin[origin_len - 1] != '.') {
		format = "$ORIGIN %s.\n";
	} else {
		format = "$ORIGIN %s\n";
	}

	char settings[1024];
	int ret = snprintf(settings, sizeof(settings), format, origin);
	if (ret <= 0 || ret >= sizeof(settings)) {
		ERR(ZS_ENOMEM);
		return -1;
	}

	// Parse the settings to set up the scanner origin.
	if (zs_set_input_string(s, settings, ret) != 0 ||
	    zs_parse_all(s) != 0) {
		return -1;
	}

	// Set scanner defaults.
	s->path = strdup(".");
	if (s->path == NULL) {
		ERR(ZS_ENOMEM);
		return -1;
	}
	s->default_class = rclass;
	s->default_ttl = ttl;
	s->line_counter = 1;

	s->state = ZS_STATE_NONE;
	s->process.automatic = false;

	return 0;
}

static void input_deinit(
	zs_scanner_t *s,
	bool keep_filename)
{
	// Deinit the file input.
	if (s->file.descriptor != -1) {
		// Unmap the file content.
		if (s->input.start != NULL) {
			if (s->input.mmaped) {
				munmap((void *)s->input.start,
				       s->input.end - s->input.start);
			} else {
				free((void *)s->input.start);
			}
		}

		// Close the opened file.
		close(s->file.descriptor);
		s->file.descriptor = -1;
	}

	// Keep file name for possible trailing error report.
	if (!keep_filename) {
		free(s->file.name);
		s->file.name = NULL;
	}

	// Unset the input limits.
	s->input.start   = NULL;
	s->input.current = NULL;
	s->input.end     = NULL;
	s->input.eof     = false;
}

__attribute__((visibility("default")))
void zs_deinit(
	zs_scanner_t *s)
{
	if (s == NULL) {
		return;
	}

	input_deinit(s, false);
	free(s->path);
}

static int set_input_string(
	zs_scanner_t *s,
	const char *input,
	size_t size,
	bool final_block)
{
	if (s == NULL) {
		return -1;
	}

	if (input == NULL) {
		ERR(ZS_EINVAL);
		return -1;
	}

	// Deinit possibly opened file.
	input_deinit(s, final_block);

	// Set the scanner input limits.
	s->input.start   = input;
	s->input.current = input;
	s->input.end     = input + size;
	s->input.eof     = final_block;

	return 0;
}

static char *read_file_to_buf(
	int fd,
	size_t *bufsize)
{
	size_t bufs = 0, newbufs = 8192;
	char *buf = malloc(bufs + newbufs);
	int ret = 0;

	while (buf != NULL && (ret = read(fd, buf + bufs, newbufs)) == newbufs) {
		bufs += newbufs;
		newbufs = bufs;
		char *newbuf = realloc(buf, bufs + newbufs);
		if (newbuf == NULL) {
			free(buf);
		}
		buf = newbuf;
	}
	if (ret < 0) {
		free(buf);
		return NULL;
	}

	*bufsize = bufs + ret;
	return buf;
}

__attribute__((visibility("default")))
int zs_set_input_string(
	zs_scanner_t *s,
	const char *input,
	size_t size)
{
	s->state = ZS_STATE_NONE;

	return set_input_string(s, input, size, false);
}

__attribute__((visibility("default")))
int zs_set_input_file(
	zs_scanner_t *s,
	const char *file_name)
{
	if (s == NULL) {
		return -1;
	}

	if (file_name == NULL) {
		ERR(ZS_EINVAL);
		return -1;
	}

	// Deinit possibly opened file.
	input_deinit(s, false);

	// Try to open the file.
	s->file.descriptor = open(file_name, O_RDONLY);
	if (s->file.descriptor == -1) {
		ERR(errno == EACCES ? ZS_FILE_ACCESS : ZS_FILE_OPEN);
		return -1;
	}

	char *start = NULL;
	size_t size = 0;

	// Check the input.
	struct stat file_stat;
	if (fstat(s->file.descriptor, &file_stat) == -1) {
		ERR(ZS_FILE_INVALID);
		input_deinit(s, false);
		return -1;
	} else if (S_ISCHR(file_stat.st_mode) ||
	           S_ISBLK(file_stat.st_mode) ||
	           S_ISFIFO(file_stat.st_mode)) {
		// Workaround if cannot mmap, read to memory.
		start = read_file_to_buf(s->file.descriptor, &size);
		if (start == NULL) {
			ERR(ZS_FILE_INVALID);
			input_deinit(s, false);
			return -1;
		}
	} else if (!S_ISREG(file_stat.st_mode)) { // Require regular file.
		ERR(ZS_FILE_INVALID);
		input_deinit(s, false);
		return -1;
	} else if (file_stat.st_size > 0) { // Mmap non-empty file.
		start = mmap(0, file_stat.st_size, PROT_READ, MAP_SHARED,
		             s->file.descriptor, 0);
		if (start == MAP_FAILED) {
			ERR(ZS_FILE_INVALID);
			input_deinit(s, false);
			return -1;
		}

		size = file_stat.st_size;
		s->input.mmaped = true;

		// Try to set the mapped memory advise to sequential.
#if defined(MADV_SEQUENTIAL) && !defined(__sun)
		(void)madvise(start, size, MADV_SEQUENTIAL);
#else
#ifdef POSIX_MADV_SEQUENTIAL
		(void)posix_madvise(start, size, POSIX_MADV_SEQUENTIAL);
#endif /* POSIX_MADV_SEQUENTIAL */
#endif /* MADV_SEQUENTIAL && !__sun */
	}

	// Set the scanner input limits.
	s->input.start   = start;
	s->input.current = start;
	s->input.end     = (start != NULL) ? start + size : start;

	// Get absolute path of the zone file if possible.
	char *full_name = realpath(file_name, NULL);
	if (full_name != NULL) {
		free(s->path);
		s->path = strdup(dirname(full_name));
		free(full_name);
		if (s->path == NULL) {
			ERR(ZS_ENOMEM);
			input_deinit(s, false);
			return -1;
		}
	}

	s->file.name = strdup(file_name);
	if (s->file.name == NULL) {
		ERR(ZS_ENOMEM);
		input_deinit(s, false);
		return -1;
	}

	s->state = ZS_STATE_NONE;

	return 0;
}

__attribute__((visibility("default")))
int zs_set_processing(
	zs_scanner_t *s,
	void (*process_record)(zs_scanner_t *),
	void (*process_error)(zs_scanner_t *),
	void *data)
{
	if (s == NULL) {
		return -1;
	}

	s->process.record = process_record;
	s->process.error = process_error;
	s->process.data = data;

	return 0;
}

__attribute__((visibility("default")))
int zs_set_processing_comment(
	zs_scanner_t *s,
	void (*process_comment)(zs_scanner_t *))
{
	if (s == NULL) {
		return -1;
	}

	s->process.comment = process_comment;

	return 0;
}

typedef enum {
	WRAP_NONE,     // Initial state.
	WRAP_DETECTED, // Input block end is a first '\' in rdata.
	WRAP_PROCESS   // Parsing of auxiliary block = "\".
} wrap_t;

static void parse(
	zs_scanner_t *s,
	wrap_t *wrap)
{
	// Restore scanner input limits (Ragel internals).
	const char *p = s->input.current;
	const char *pe = s->input.end;
	const char *eof = s->input.eof ? pe : NULL;

	// Restore state variables (Ragel internals).
	int cs  = s->cs;
	int top = s->top;
	int stack[ZS_RAGEL_STACK_SIZE];
	memcpy(stack, s->stack, sizeof(stack));

	// Next 2 variables are for better performance.
	// Restoring r_data pointer to next free space.
	uint8_t *rdata_tail = s->r_data + s->r_data_tail;
	// Initialization of the last r_data byte.
	uint8_t *rdata_stop = s->r_data + ZS_MAX_RDATA_LENGTH - 1;

	// Write scanner body (in C).

	{
	short _widec;
	if ( p == pe )
		goto _test_eof;
	goto _resume;

_again:
	switch ( cs ) {
		case 1396: goto st1396;
		case 0: goto st0;
		case 1: goto st1;
		case 2: goto st2;
		case 3: goto st3;
		case 4: goto st4;
		case 5: goto st5;
		case 6: goto st6;
		case 7: goto st7;
		case 8: goto st8;
		case 9: goto st9;
		case 10: goto st10;
		case 11: goto st11;
		case 12: goto st12;
		case 13: goto st13;
		case 1397: goto st1397;
		case 14: goto st14;
		case 15: goto st15;
		case 16: goto st16;
		case 17: goto st17;
		case 18: goto st18;
		case 19: goto st19;
		case 20: goto st20;
		case 21: goto st21;
		case 22: goto st22;
		case 23: goto st23;
		case 24: goto st24;
		case 25: goto st25;
		case 26: goto st26;
		case 27: goto st27;
		case 28: goto st28;
		case 29: goto st29;
		case 30: goto st30;
		case 31: goto st31;
		case 32: goto st32;
		case 33: goto st33;
		case 34: goto st34;
		case 35: goto st35;
		case 36: goto st36;
		case 37: goto st37;
		case 38: goto st38;
		case 39: goto st39;
		case 40: goto st40;
		case 41: goto st41;
		case 42: goto st42;
		case 43: goto st43;
		case 44: goto st44;
		case 45: goto st45;
		case 46: goto st46;
		case 47: goto st47;
		case 48: goto st48;
		case 49: goto st49;
		case 50: goto st50;
		case 51: goto st51;
		case 52: goto st52;
		case 53: goto st53;
		case 54: goto st54;
		case 55: goto st55;
		case 56: goto st56;
		case 57: goto st57;
		case 58: goto st58;
		case 59: goto st59;
		case 60: goto st60;
		case 61: goto st61;
		case 62: goto st62;
		case 63: goto st63;
		case 64: goto st64;
		case 65: goto st65;
		case 66: goto st66;
		case 67: goto st67;
		case 68: goto st68;
		case 69: goto st69;
		case 70: goto st70;
		case 71: goto st71;
		case 72: goto st72;
		case 73: goto st73;
		case 74: goto st74;
		case 75: goto st75;
		case 76: goto st76;
		case 77: goto st77;
		case 78: goto st78;
		case 79: goto st79;
		case 80: goto st80;
		case 81: goto st81;
		case 82: goto st82;
		case 83: goto st83;
		case 84: goto st84;
		case 85: goto st85;
		case 86: goto st86;
		case 87: goto st87;
		case 88: goto st88;
		case 89: goto st89;
		case 90: goto st90;
		case 91: goto st91;
		case 92: goto st92;
		case 93: goto st93;
		case 94: goto st94;
		case 95: goto st95;
		case 96: goto st96;
		case 97: goto st97;
		case 98: goto st98;
		case 99: goto st99;
		case 100: goto st100;
		case 101: goto st101;
		case 102: goto st102;
		case 103: goto st103;
		case 104: goto st104;
		case 105: goto st105;
		case 106: goto st106;
		case 107: goto st107;
		case 108: goto st108;
		case 109: goto st109;
		case 110: goto st110;
		case 111: goto st111;
		case 112: goto st112;
		case 113: goto st113;
		case 114: goto st114;
		case 115: goto st115;
		case 116: goto st116;
		case 117: goto st117;
		case 118: goto st118;
		case 119: goto st119;
		case 120: goto st120;
		case 121: goto st121;
		case 122: goto st122;
		case 123: goto st123;
		case 124: goto st124;
		case 125: goto st125;
		case 126: goto st126;
		case 127: goto st127;
		case 128: goto st128;
		case 129: goto st129;
		case 130: goto st130;
		case 131: goto st131;
		case 132: goto st132;
		case 133: goto st133;
		case 134: goto st134;
		case 135: goto st135;
		case 136: goto st136;
		case 137: goto st137;
		case 138: goto st138;
		case 139: goto st139;
		case 140: goto st140;
		case 141: goto st141;
		case 142: goto st142;
		case 143: goto st143;
		case 144: goto st144;
		case 145: goto st145;
		case 146: goto st146;
		case 147: goto st147;
		case 148: goto st148;
		case 149: goto st149;
		case 150: goto st150;
		case 151: goto st151;
		case 152: goto st152;
		case 153: goto st153;
		case 154: goto st154;
		case 155: goto st155;
		case 156: goto st156;
		case 157: goto st157;
		case 158: goto st158;
		case 159: goto st159;
		case 160: goto st160;
		case 161: goto st161;
		case 162: goto st162;
		case 163: goto st163;
		case 164: goto st164;
		case 165: goto st165;
		case 166: goto st166;
		case 167: goto st167;
		case 168: goto st168;
		case 169: goto st169;
		case 170: goto st170;
		case 171: goto st171;
		case 172: goto st172;
		case 173: goto st173;
		case 1398: goto st1398;
		case 174: goto st174;
		case 175: goto st175;
		case 176: goto st176;
		case 177: goto st177;
		case 178: goto st178;
		case 179: goto st179;
		case 180: goto st180;
		case 181: goto st181;
		case 182: goto st182;
		case 183: goto st183;
		case 1399: goto st1399;
		case 184: goto st184;
		case 185: goto st185;
		case 186: goto st186;
		case 187: goto st187;
		case 188: goto st188;
		case 189: goto st189;
		case 190: goto st190;
		case 191: goto st191;
		case 1400: goto st1400;
		case 192: goto st192;
		case 193: goto st193;
		case 194: goto st194;
		case 1401: goto st1401;
		case 195: goto st195;
		case 196: goto st196;
		case 197: goto st197;
		case 198: goto st198;
		case 199: goto st199;
		case 200: goto st200;
		case 201: goto st201;
		case 202: goto st202;
		case 203: goto st203;
		case 204: goto st204;
		case 205: goto st205;
		case 1402: goto st1402;
		case 206: goto st206;
		case 207: goto st207;
		case 208: goto st208;
		case 209: goto st209;
		case 1403: goto st1403;
		case 210: goto st210;
		case 211: goto st211;
		case 212: goto st212;
		case 213: goto st213;
		case 214: goto st214;
		case 215: goto st215;
		case 216: goto st216;
		case 217: goto st217;
		case 218: goto st218;
		case 219: goto st219;
		case 220: goto st220;
		case 221: goto st221;
		case 222: goto st222;
		case 223: goto st223;
		case 224: goto st224;
		case 225: goto st225;
		case 226: goto st226;
		case 227: goto st227;
		case 1404: goto st1404;
		case 228: goto st228;
		case 229: goto st229;
		case 230: goto st230;
		case 231: goto st231;
		case 232: goto st232;
		case 233: goto st233;
		case 234: goto st234;
		case 235: goto st235;
		case 236: goto st236;
		case 237: goto st237;
		case 238: goto st238;
		case 239: goto st239;
		case 240: goto st240;
		case 241: goto st241;
		case 242: goto st242;
		case 243: goto st243;
		case 1405: goto st1405;
		case 244: goto st244;
		case 245: goto st245;
		case 246: goto st246;
		case 247: goto st247;
		case 248: goto st248;
		case 249: goto st249;
		case 250: goto st250;
		case 251: goto st251;
		case 252: goto st252;
		case 253: goto st253;
		case 254: goto st254;
		case 255: goto st255;
		case 256: goto st256;
		case 257: goto st257;
		case 258: goto st258;
		case 259: goto st259;
		case 260: goto st260;
		case 261: goto st261;
		case 262: goto st262;
		case 263: goto st263;
		case 264: goto st264;
		case 265: goto st265;
		case 266: goto st266;
		case 267: goto st267;
		case 268: goto st268;
		case 269: goto st269;
		case 270: goto st270;
		case 271: goto st271;
		case 272: goto st272;
		case 273: goto st273;
		case 274: goto st274;
		case 275: goto st275;
		case 276: goto st276;
		case 277: goto st277;
		case 278: goto st278;
		case 279: goto st279;
		case 280: goto st280;
		case 281: goto st281;
		case 282: goto st282;
		case 283: goto st283;
		case 284: goto st284;
		case 285: goto st285;
		case 286: goto st286;
		case 287: goto st287;
		case 288: goto st288;
		case 289: goto st289;
		case 290: goto st290;
		case 291: goto st291;
		case 292: goto st292;
		case 293: goto st293;
		case 294: goto st294;
		case 295: goto st295;
		case 296: goto st296;
		case 297: goto st297;
		case 298: goto st298;
		case 299: goto st299;
		case 300: goto st300;
		case 301: goto st301;
		case 302: goto st302;
		case 303: goto st303;
		case 304: goto st304;
		case 305: goto st305;
		case 306: goto st306;
		case 1406: goto st1406;
		case 307: goto st307;
		case 308: goto st308;
		case 1407: goto st1407;
		case 309: goto st309;
		case 310: goto st310;
		case 1408: goto st1408;
		case 311: goto st311;
		case 312: goto st312;
		case 313: goto st313;
		case 314: goto st314;
		case 315: goto st315;
		case 316: goto st316;
		case 317: goto st317;
		case 318: goto st318;
		case 319: goto st319;
		case 1409: goto st1409;
		case 1410: goto st1410;
		case 320: goto st320;
		case 321: goto st321;
		case 322: goto st322;
		case 323: goto st323;
		case 324: goto st324;
		case 325: goto st325;
		case 326: goto st326;
		case 327: goto st327;
		case 328: goto st328;
		case 329: goto st329;
		case 330: goto st330;
		case 331: goto st331;
		case 332: goto st332;
		case 333: goto st333;
		case 334: goto st334;
		case 1411: goto st1411;
		case 335: goto st335;
		case 336: goto st336;
		case 337: goto st337;
		case 338: goto st338;
		case 339: goto st339;
		case 340: goto st340;
		case 341: goto st341;
		case 342: goto st342;
		case 343: goto st343;
		case 344: goto st344;
		case 1412: goto st1412;
		case 345: goto st345;
		case 346: goto st346;
		case 347: goto st347;
		case 348: goto st348;
		case 349: goto st349;
		case 350: goto st350;
		case 351: goto st351;
		case 352: goto st352;
		case 353: goto st353;
		case 354: goto st354;
		case 355: goto st355;
		case 356: goto st356;
		case 357: goto st357;
		case 358: goto st358;
		case 1413: goto st1413;
		case 359: goto st359;
		case 360: goto st360;
		case 361: goto st361;
		case 362: goto st362;
		case 363: goto st363;
		case 364: goto st364;
		case 365: goto st365;
		case 1414: goto st1414;
		case 366: goto st366;
		case 367: goto st367;
		case 368: goto st368;
		case 369: goto st369;
		case 370: goto st370;
		case 371: goto st371;
		case 372: goto st372;
		case 373: goto st373;
		case 374: goto st374;
		case 1415: goto st1415;
		case 1416: goto st1416;
		case 1417: goto st1417;
		case 375: goto st375;
		case 376: goto st376;
		case 377: goto st377;
		case 378: goto st378;
		case 379: goto st379;
		case 380: goto st380;
		case 381: goto st381;
		case 382: goto st382;
		case 1418: goto st1418;
		case 1419: goto st1419;
		case 383: goto st383;
		case 384: goto st384;
		case 385: goto st385;
		case 1420: goto st1420;
		case 386: goto st386;
		case 387: goto st387;
		case 388: goto st388;
		case 389: goto st389;
		case 390: goto st390;
		case 391: goto st391;
		case 392: goto st392;
		case 393: goto st393;
		case 394: goto st394;
		case 395: goto st395;
		case 396: goto st396;
		case 397: goto st397;
		case 398: goto st398;
		case 399: goto st399;
		case 400: goto st400;
		case 401: goto st401;
		case 402: goto st402;
		case 403: goto st403;
		case 404: goto st404;
		case 405: goto st405;
		case 406: goto st406;
		case 407: goto st407;
		case 408: goto st408;
		case 409: goto st409;
		case 410: goto st410;
		case 411: goto st411;
		case 412: goto st412;
		case 413: goto st413;
		case 414: goto st414;
		case 415: goto st415;
		case 416: goto st416;
		case 417: goto st417;
		case 418: goto st418;
		case 419: goto st419;
		case 420: goto st420;
		case 421: goto st421;
		case 422: goto st422;
		case 423: goto st423;
		case 424: goto st424;
		case 425: goto st425;
		case 426: goto st426;
		case 427: goto st427;
		case 428: goto st428;
		case 429: goto st429;
		case 430: goto st430;
		case 431: goto st431;
		case 432: goto st432;
		case 433: goto st433;
		case 434: goto st434;
		case 435: goto st435;
		case 436: goto st436;
		case 437: goto st437;
		case 438: goto st438;
		case 439: goto st439;
		case 440: goto st440;
		case 441: goto st441;
		case 442: goto st442;
		case 443: goto st443;
		case 444: goto st444;
		case 445: goto st445;
		case 446: goto st446;
		case 447: goto st447;
		case 448: goto st448;
		case 449: goto st449;
		case 450: goto st450;
		case 451: goto st451;
		case 452: goto st452;
		case 453: goto st453;
		case 454: goto st454;
		case 455: goto st455;
		case 456: goto st456;
		case 457: goto st457;
		case 458: goto st458;
		case 459: goto st459;
		case 460: goto st460;
		case 461: goto st461;
		case 462: goto st462;
		case 463: goto st463;
		case 464: goto st464;
		case 465: goto st465;
		case 466: goto st466;
		case 467: goto st467;
		case 468: goto st468;
		case 469: goto st469;
		case 470: goto st470;
		case 471: goto st471;
		case 472: goto st472;
		case 473: goto st473;
		case 474: goto st474;
		case 475: goto st475;
		case 476: goto st476;
		case 477: goto st477;
		case 478: goto st478;
		case 479: goto st479;
		case 480: goto st480;
		case 481: goto st481;
		case 482: goto st482;
		case 483: goto st483;
		case 484: goto st484;
		case 485: goto st485;
		case 486: goto st486;
		case 487: goto st487;
		case 488: goto st488;
		case 489: goto st489;
		case 490: goto st490;
		case 491: goto st491;
		case 492: goto st492;
		case 493: goto st493;
		case 494: goto st494;
		case 495: goto st495;
		case 496: goto st496;
		case 497: goto st497;
		case 498: goto st498;
		case 499: goto st499;
		case 500: goto st500;
		case 501: goto st501;
		case 502: goto st502;
		case 503: goto st503;
		case 504: goto st504;
		case 505: goto st505;
		case 506: goto st506;
		case 507: goto st507;
		case 508: goto st508;
		case 509: goto st509;
		case 510: goto st510;
		case 511: goto st511;
		case 512: goto st512;
		case 513: goto st513;
		case 514: goto st514;
		case 515: goto st515;
		case 516: goto st516;
		case 517: goto st517;
		case 518: goto st518;
		case 519: goto st519;
		case 520: goto st520;
		case 521: goto st521;
		case 522: goto st522;
		case 523: goto st523;
		case 524: goto st524;
		case 525: goto st525;
		case 526: goto st526;
		case 527: goto st527;
		case 528: goto st528;
		case 529: goto st529;
		case 530: goto st530;
		case 531: goto st531;
		case 532: goto st532;
		case 533: goto st533;
		case 534: goto st534;
		case 535: goto st535;
		case 536: goto st536;
		case 537: goto st537;
		case 538: goto st538;
		case 539: goto st539;
		case 540: goto st540;
		case 541: goto st541;
		case 542: goto st542;
		case 543: goto st543;
		case 544: goto st544;
		case 545: goto st545;
		case 546: goto st546;
		case 1421: goto st1421;
		case 1422: goto st1422;
		case 1423: goto st1423;
		case 547: goto st547;
		case 548: goto st548;
		case 549: goto st549;
		case 550: goto st550;
		case 551: goto st551;
		case 1424: goto st1424;
		case 552: goto st552;
		case 553: goto st553;
		case 554: goto st554;
		case 555: goto st555;
		case 1425: goto st1425;
		case 1426: goto st1426;
		case 1427: goto st1427;
		case 556: goto st556;
		case 557: goto st557;
		case 1428: goto st1428;
		case 558: goto st558;
		case 559: goto st559;
		case 560: goto st560;
		case 561: goto st561;
		case 562: goto st562;
		case 563: goto st563;
		case 1429: goto st1429;
		case 564: goto st564;
		case 565: goto st565;
		case 566: goto st566;
		case 567: goto st567;
		case 568: goto st568;
		case 569: goto st569;
		case 570: goto st570;
		case 571: goto st571;
		case 572: goto st572;
		case 573: goto st573;
		case 574: goto st574;
		case 575: goto st575;
		case 576: goto st576;
		case 577: goto st577;
		case 578: goto st578;
		case 579: goto st579;
		case 580: goto st580;
		case 581: goto st581;
		case 582: goto st582;
		case 583: goto st583;
		case 584: goto st584;
		case 585: goto st585;
		case 586: goto st586;
		case 587: goto st587;
		case 588: goto st588;
		case 589: goto st589;
		case 590: goto st590;
		case 591: goto st591;
		case 592: goto st592;
		case 593: goto st593;
		case 594: goto st594;
		case 595: goto st595;
		case 596: goto st596;
		case 597: goto st597;
		case 598: goto st598;
		case 599: goto st599;
		case 600: goto st600;
		case 601: goto st601;
		case 602: goto st602;
		case 603: goto st603;
		case 604: goto st604;
		case 605: goto st605;
		case 606: goto st606;
		case 607: goto st607;
		case 608: goto st608;
		case 609: goto st609;
		case 610: goto st610;
		case 611: goto st611;
		case 1430: goto st1430;
		case 1431: goto st1431;
		case 612: goto st612;
		case 613: goto st613;
		case 614: goto st614;
		case 615: goto st615;
		case 616: goto st616;
		case 617: goto st617;
		case 618: goto st618;
		case 619: goto st619;
		case 620: goto st620;
		case 621: goto st621;
		case 1432: goto st1432;
		case 622: goto st622;
		case 623: goto st623;
		case 624: goto st624;
		case 625: goto st625;
		case 626: goto st626;
		case 627: goto st627;
		case 628: goto st628;
		case 629: goto st629;
		case 630: goto st630;
		case 631: goto st631;
		case 632: goto st632;
		case 633: goto st633;
		case 634: goto st634;
		case 635: goto st635;
		case 636: goto st636;
		case 637: goto st637;
		case 638: goto st638;
		case 639: goto st639;
		case 640: goto st640;
		case 641: goto st641;
		case 642: goto st642;
		case 643: goto st643;
		case 644: goto st644;
		case 645: goto st645;
		case 646: goto st646;
		case 647: goto st647;
		case 648: goto st648;
		case 649: goto st649;
		case 650: goto st650;
		case 651: goto st651;
		case 652: goto st652;
		case 653: goto st653;
		case 654: goto st654;
		case 655: goto st655;
		case 656: goto st656;
		case 657: goto st657;
		case 658: goto st658;
		case 659: goto st659;
		case 660: goto st660;
		case 661: goto st661;
		case 662: goto st662;
		case 663: goto st663;
		case 664: goto st664;
		case 665: goto st665;
		case 666: goto st666;
		case 667: goto st667;
		case 668: goto st668;
		case 669: goto st669;
		case 670: goto st670;
		case 671: goto st671;
		case 672: goto st672;
		case 673: goto st673;
		case 674: goto st674;
		case 675: goto st675;
		case 676: goto st676;
		case 677: goto st677;
		case 678: goto st678;
		case 679: goto st679;
		case 680: goto st680;
		case 681: goto st681;
		case 682: goto st682;
		case 683: goto st683;
		case 684: goto st684;
		case 685: goto st685;
		case 686: goto st686;
		case 687: goto st687;
		case 688: goto st688;
		case 689: goto st689;
		case 690: goto st690;
		case 691: goto st691;
		case 692: goto st692;
		case 693: goto st693;
		case 694: goto st694;
		case 695: goto st695;
		case 696: goto st696;
		case 697: goto st697;
		case 698: goto st698;
		case 699: goto st699;
		case 700: goto st700;
		case 1433: goto st1433;
		case 701: goto st701;
		case 702: goto st702;
		case 703: goto st703;
		case 704: goto st704;
		case 705: goto st705;
		case 706: goto st706;
		case 707: goto st707;
		case 708: goto st708;
		case 709: goto st709;
		case 710: goto st710;
		case 711: goto st711;
		case 712: goto st712;
		case 713: goto st713;
		case 714: goto st714;
		case 715: goto st715;
		case 716: goto st716;
		case 717: goto st717;
		case 718: goto st718;
		case 719: goto st719;
		case 720: goto st720;
		case 721: goto st721;
		case 722: goto st722;
		case 723: goto st723;
		case 724: goto st724;
		case 725: goto st725;
		case 726: goto st726;
		case 727: goto st727;
		case 728: goto st728;
		case 729: goto st729;
		case 730: goto st730;
		case 731: goto st731;
		case 732: goto st732;
		case 733: goto st733;
		case 734: goto st734;
		case 735: goto st735;
		case 736: goto st736;
		case 737: goto st737;
		case 738: goto st738;
		case 739: goto st739;
		case 740: goto st740;
		case 741: goto st741;
		case 742: goto st742;
		case 743: goto st743;
		case 744: goto st744;
		case 745: goto st745;
		case 746: goto st746;
		case 747: goto st747;
		case 748: goto st748;
		case 749: goto st749;
		case 750: goto st750;
		case 751: goto st751;
		case 752: goto st752;
		case 753: goto st753;
		case 754: goto st754;
		case 755: goto st755;
		case 756: goto st756;
		case 757: goto st757;
		case 758: goto st758;
		case 759: goto st759;
		case 760: goto st760;
		case 761: goto st761;
		case 762: goto st762;
		case 763: goto st763;
		case 764: goto st764;
		case 765: goto st765;
		case 766: goto st766;
		case 767: goto st767;
		case 768: goto st768;
		case 769: goto st769;
		case 770: goto st770;
		case 771: goto st771;
		case 772: goto st772;
		case 773: goto st773;
		case 774: goto st774;
		case 775: goto st775;
		case 776: goto st776;
		case 777: goto st777;
		case 778: goto st778;
		case 779: goto st779;
		case 780: goto st780;
		case 781: goto st781;
		case 782: goto st782;
		case 783: goto st783;
		case 784: goto st784;
		case 785: goto st785;
		case 786: goto st786;
		case 787: goto st787;
		case 788: goto st788;
		case 789: goto st789;
		case 790: goto st790;
		case 791: goto st791;
		case 792: goto st792;
		case 793: goto st793;
		case 794: goto st794;
		case 795: goto st795;
		case 796: goto st796;
		case 797: goto st797;
		case 798: goto st798;
		case 799: goto st799;
		case 800: goto st800;
		case 801: goto st801;
		case 802: goto st802;
		case 803: goto st803;
		case 804: goto st804;
		case 1434: goto st1434;
		case 805: goto st805;
		case 806: goto st806;
		case 807: goto st807;
		case 808: goto st808;
		case 809: goto st809;
		case 810: goto st810;
		case 811: goto st811;
		case 812: goto st812;
		case 813: goto st813;
		case 814: goto st814;
		case 815: goto st815;
		case 816: goto st816;
		case 817: goto st817;
		case 818: goto st818;
		case 819: goto st819;
		case 820: goto st820;
		case 821: goto st821;
		case 822: goto st822;
		case 823: goto st823;
		case 824: goto st824;
		case 825: goto st825;
		case 826: goto st826;
		case 827: goto st827;
		case 828: goto st828;
		case 829: goto st829;
		case 830: goto st830;
		case 831: goto st831;
		case 832: goto st832;
		case 833: goto st833;
		case 834: goto st834;
		case 835: goto st835;
		case 836: goto st836;
		case 837: goto st837;
		case 838: goto st838;
		case 839: goto st839;
		case 840: goto st840;
		case 841: goto st841;
		case 842: goto st842;
		case 843: goto st843;
		case 844: goto st844;
		case 845: goto st845;
		case 1435: goto st1435;
		case 846: goto st846;
		case 847: goto st847;
		case 1436: goto st1436;
		case 848: goto st848;
		case 849: goto st849;
		case 850: goto st850;
		case 851: goto st851;
		case 852: goto st852;
		case 853: goto st853;
		case 854: goto st854;
		case 855: goto st855;
		case 856: goto st856;
		case 857: goto st857;
		case 858: goto st858;
		case 859: goto st859;
		case 860: goto st860;
		case 861: goto st861;
		case 1437: goto st1437;
		case 862: goto st862;
		case 863: goto st863;
		case 864: goto st864;
		case 865: goto st865;
		case 866: goto st866;
		case 867: goto st867;
		case 868: goto st868;
		case 869: goto st869;
		case 870: goto st870;
		case 871: goto st871;
		case 872: goto st872;
		case 873: goto st873;
		case 874: goto st874;
		case 875: goto st875;
		case 876: goto st876;
		case 877: goto st877;
		case 878: goto st878;
		case 879: goto st879;
		case 880: goto st880;
		case 881: goto st881;
		case 882: goto st882;
		case 883: goto st883;
		case 1438: goto st1438;
		case 884: goto st884;
		case 885: goto st885;
		case 886: goto st886;
		case 887: goto st887;
		case 888: goto st888;
		case 1439: goto st1439;
		case 889: goto st889;
		case 890: goto st890;
		case 891: goto st891;
		case 892: goto st892;
		case 893: goto st893;
		case 1440: goto st1440;
		case 894: goto st894;
		case 895: goto st895;
		case 896: goto st896;
		case 897: goto st897;
		case 898: goto st898;
		case 1441: goto st1441;
		case 1442: goto st1442;
		case 1443: goto st1443;
		case 899: goto st899;
		case 900: goto st900;
		case 1444: goto st1444;
		case 901: goto st901;
		case 902: goto st902;
		case 903: goto st903;
		case 904: goto st904;
		case 905: goto st905;
		case 906: goto st906;
		case 907: goto st907;
		case 908: goto st908;
		case 909: goto st909;
		case 910: goto st910;
		case 911: goto st911;
		case 912: goto st912;
		case 913: goto st913;
		case 914: goto st914;
		case 915: goto st915;
		case 916: goto st916;
		case 917: goto st917;
		case 918: goto st918;
		case 919: goto st919;
		case 920: goto st920;
		case 921: goto st921;
		case 922: goto st922;
		case 923: goto st923;
		case 924: goto st924;
		case 925: goto st925;
		case 926: goto st926;
		case 927: goto st927;
		case 1445: goto st1445;
		case 1446: goto st1446;
		case 1447: goto st1447;
		case 928: goto st928;
		case 929: goto st929;
		case 930: goto st930;
		case 1448: goto st1448;
		case 1449: goto st1449;
		case 931: goto st931;
		case 932: goto st932;
		case 933: goto st933;
		case 934: goto st934;
		case 1450: goto st1450;
		case 1451: goto st1451;
		case 935: goto st935;
		case 936: goto st936;
		case 937: goto st937;
		case 938: goto st938;
		case 1452: goto st1452;
		case 1453: goto st1453;
		case 939: goto st939;
		case 940: goto st940;
		case 941: goto st941;
		case 942: goto st942;
		case 943: goto st943;
		case 944: goto st944;
		case 945: goto st945;
		case 946: goto st946;
		case 947: goto st947;
		case 948: goto st948;
		case 949: goto st949;
		case 950: goto st950;
		case 951: goto st951;
		case 952: goto st952;
		case 953: goto st953;
		case 954: goto st954;
		case 955: goto st955;
		case 956: goto st956;
		case 957: goto st957;
		case 958: goto st958;
		case 959: goto st959;
		case 960: goto st960;
		case 961: goto st961;
		case 962: goto st962;
		case 963: goto st963;
		case 1454: goto st1454;
		case 964: goto st964;
		case 965: goto st965;
		case 966: goto st966;
		case 967: goto st967;
		case 968: goto st968;
		case 969: goto st969;
		case 970: goto st970;
		case 971: goto st971;
		case 972: goto st972;
		case 973: goto st973;
		case 974: goto st974;
		case 975: goto st975;
		case 976: goto st976;
		case 977: goto st977;
		case 978: goto st978;
		case 1455: goto st1455;
		case 979: goto st979;
		case 980: goto st980;
		case 981: goto st981;
		case 982: goto st982;
		case 983: goto st983;
		case 984: goto st984;
		case 985: goto st985;
		case 986: goto st986;
		case 987: goto st987;
		case 988: goto st988;
		case 989: goto st989;
		case 990: goto st990;
		case 991: goto st991;
		case 1456: goto st1456;
		case 992: goto st992;
		case 993: goto st993;
		case 994: goto st994;
		case 995: goto st995;
		case 996: goto st996;
		case 997: goto st997;
		case 998: goto st998;
		case 999: goto st999;
		case 1000: goto st1000;
		case 1001: goto st1001;
		case 1002: goto st1002;
		case 1457: goto st1457;
		case 1458: goto st1458;
		case 1003: goto st1003;
		case 1004: goto st1004;
		case 1005: goto st1005;
		case 1459: goto st1459;
		case 1006: goto st1006;
		case 1007: goto st1007;
		case 1008: goto st1008;
		case 1009: goto st1009;
		case 1010: goto st1010;
		case 1011: goto st1011;
		case 1012: goto st1012;
		case 1013: goto st1013;
		case 1014: goto st1014;
		case 1015: goto st1015;
		case 1460: goto st1460;
		case 1461: goto st1461;
		case 1462: goto st1462;
		case 1016: goto st1016;
		case 1017: goto st1017;
		case 1018: goto st1018;
		case 1019: goto st1019;
		case 1020: goto st1020;
		case 1021: goto st1021;
		case 1022: goto st1022;
		case 1023: goto st1023;
		case 1024: goto st1024;
		case 1025: goto st1025;
		case 1026: goto st1026;
		case 1463: goto st1463;
		case 1464: goto st1464;
		case 1465: goto st1465;
		case 1027: goto st1027;
		case 1028: goto st1028;
		case 1029: goto st1029;
		case 1030: goto st1030;
		case 1031: goto st1031;
		case 1032: goto st1032;
		case 1033: goto st1033;
		case 1034: goto st1034;
		case 1035: goto st1035;
		case 1036: goto st1036;
		case 1037: goto st1037;
		case 1038: goto st1038;
		case 1466: goto st1466;
		case 1039: goto st1039;
		case 1040: goto st1040;
		case 1041: goto st1041;
		case 1467: goto st1467;
		case 1468: goto st1468;
		case 1042: goto st1042;
		case 1469: goto st1469;
		case 1470: goto st1470;
		case 1043: goto st1043;
		case 1471: goto st1471;
		case 1472: goto st1472;
		case 1044: goto st1044;
		case 1045: goto st1045;
		case 1046: goto st1046;
		case 1047: goto st1047;
		case 1048: goto st1048;
		case 1049: goto st1049;
		case 1050: goto st1050;
		case 1051: goto st1051;
		case 1052: goto st1052;
		case 1053: goto st1053;
		case 1054: goto st1054;
		case 1055: goto st1055;
		case 1056: goto st1056;
		case 1057: goto st1057;
		case 1058: goto st1058;
		case 1059: goto st1059;
		case 1060: goto st1060;
		case 1061: goto st1061;
		case 1062: goto st1062;
		case 1063: goto st1063;
		case 1064: goto st1064;
		case 1065: goto st1065;
		case 1066: goto st1066;
		case 1067: goto st1067;
		case 1068: goto st1068;
		case 1069: goto st1069;
		case 1070: goto st1070;
		case 1071: goto st1071;
		case 1072: goto st1072;
		case 1073: goto st1073;
		case 1074: goto st1074;
		case 1075: goto st1075;
		case 1076: goto st1076;
		case 1077: goto st1077;
		case 1078: goto st1078;
		case 1079: goto st1079;
		case 1080: goto st1080;
		case 1081: goto st1081;
		case 1082: goto st1082;
		case 1083: goto st1083;
		case 1084: goto st1084;
		case 1085: goto st1085;
		case 1473: goto st1473;
		case 1086: goto st1086;
		case 1087: goto st1087;
		case 1088: goto st1088;
		case 1089: goto st1089;
		case 1090: goto st1090;
		case 1091: goto st1091;
		case 1092: goto st1092;
		case 1093: goto st1093;
		case 1094: goto st1094;
		case 1095: goto st1095;
		case 1096: goto st1096;
		case 1097: goto st1097;
		case 1098: goto st1098;
		case 1099: goto st1099;
		case 1100: goto st1100;
		case 1101: goto st1101;
		case 1102: goto st1102;
		case 1103: goto st1103;
		case 1104: goto st1104;
		case 1105: goto st1105;
		case 1106: goto st1106;
		case 1107: goto st1107;
		case 1108: goto st1108;
		case 1109: goto st1109;
		case 1110: goto st1110;
		case 1111: goto st1111;
		case 1112: goto st1112;
		case 1113: goto st1113;
		case 1114: goto st1114;
		case 1115: goto st1115;
		case 1116: goto st1116;
		case 1117: goto st1117;
		case 1118: goto st1118;
		case 1119: goto st1119;
		case 1120: goto st1120;
		case 1121: goto st1121;
		case 1122: goto st1122;
		case 1123: goto st1123;
		case 1124: goto st1124;
		case 1125: goto st1125;
		case 1126: goto st1126;
		case 1127: goto st1127;
		case 1128: goto st1128;
		case 1129: goto st1129;
		case 1130: goto st1130;
		case 1131: goto st1131;
		case 1132: goto st1132;
		case 1133: goto st1133;
		case 1134: goto st1134;
		case 1135: goto st1135;
		case 1136: goto st1136;
		case 1137: goto st1137;
		case 1138: goto st1138;
		case 1139: goto st1139;
		case 1140: goto st1140;
		case 1141: goto st1141;
		case 1142: goto st1142;
		case 1143: goto st1143;
		case 1144: goto st1144;
		case 1145: goto st1145;
		case 1146: goto st1146;
		case 1147: goto st1147;
		case 1148: goto st1148;
		case 1149: goto st1149;
		case 1150: goto st1150;
		case 1151: goto st1151;
		case 1152: goto st1152;
		case 1153: goto st1153;
		case 1154: goto st1154;
		case 1155: goto st1155;
		case 1156: goto st1156;
		case 1157: goto st1157;
		case 1158: goto st1158;
		case 1159: goto st1159;
		case 1160: goto st1160;
		case 1161: goto st1161;
		case 1162: goto st1162;
		case 1163: goto st1163;
		case 1164: goto st1164;
		case 1165: goto st1165;
		case 1166: goto st1166;
		case 1167: goto st1167;
		case 1168: goto st1168;
		case 1169: goto st1169;
		case 1170: goto st1170;
		case 1171: goto st1171;
		case 1172: goto st1172;
		case 1173: goto st1173;
		case 1174: goto st1174;
		case 1175: goto st1175;
		case 1176: goto st1176;
		case 1177: goto st1177;
		case 1178: goto st1178;
		case 1179: goto st1179;
		case 1180: goto st1180;
		case 1181: goto st1181;
		case 1182: goto st1182;
		case 1183: goto st1183;
		case 1184: goto st1184;
		case 1185: goto st1185;
		case 1186: goto st1186;
		case 1187: goto st1187;
		case 1188: goto st1188;
		case 1189: goto st1189;
		case 1190: goto st1190;
		case 1191: goto st1191;
		case 1192: goto st1192;
		case 1193: goto st1193;
		case 1194: goto st1194;
		case 1195: goto st1195;
		case 1196: goto st1196;
		case 1197: goto st1197;
		case 1198: goto st1198;
		case 1199: goto st1199;
		case 1200: goto st1200;
		case 1201: goto st1201;
		case 1202: goto st1202;
		case 1203: goto st1203;
		case 1204: goto st1204;
		case 1205: goto st1205;
		case 1206: goto st1206;
		case 1207: goto st1207;
		case 1208: goto st1208;
		case 1209: goto st1209;
		case 1210: goto st1210;
		case 1211: goto st1211;
		case 1212: goto st1212;
		case 1213: goto st1213;
		case 1214: goto st1214;
		case 1215: goto st1215;
		case 1216: goto st1216;
		case 1217: goto st1217;
		case 1218: goto st1218;
		case 1219: goto st1219;
		case 1220: goto st1220;
		case 1221: goto st1221;
		case 1222: goto st1222;
		case 1223: goto st1223;
		case 1224: goto st1224;
		case 1225: goto st1225;
		case 1226: goto st1226;
		case 1227: goto st1227;
		case 1228: goto st1228;
		case 1229: goto st1229;
		case 1230: goto st1230;
		case 1231: goto st1231;
		case 1232: goto st1232;
		case 1233: goto st1233;
		case 1234: goto st1234;
		case 1235: goto st1235;
		case 1236: goto st1236;
		case 1237: goto st1237;
		case 1238: goto st1238;
		case 1239: goto st1239;
		case 1240: goto st1240;
		case 1241: goto st1241;
		case 1242: goto st1242;
		case 1243: goto st1243;
		case 1244: goto st1244;
		case 1245: goto st1245;
		case 1246: goto st1246;
		case 1247: goto st1247;
		case 1248: goto st1248;
		case 1249: goto st1249;
		case 1250: goto st1250;
		case 1251: goto st1251;
		case 1252: goto st1252;
		case 1253: goto st1253;
		case 1254: goto st1254;
		case 1255: goto st1255;
		case 1256: goto st1256;
		case 1474: goto st1474;
		case 1257: goto st1257;
		case 1258: goto st1258;
		case 1259: goto st1259;
		case 1260: goto st1260;
		case 1261: goto st1261;
		case 1262: goto st1262;
		case 1263: goto st1263;
		case 1264: goto st1264;
		case 1475: goto st1475;
		case 1265: goto st1265;
		case 1266: goto st1266;
		case 1267: goto st1267;
		case 1268: goto st1268;
		case 1269: goto st1269;
		case 1476: goto st1476;
		case 1270: goto st1270;
		case 1271: goto st1271;
		case 1272: goto st1272;
		case 1273: goto st1273;
		case 1274: goto st1274;
		case 1275: goto st1275;
		case 1276: goto st1276;
		case 1277: goto st1277;
		case 1278: goto st1278;
		case 1279: goto st1279;
		case 1280: goto st1280;
		case 1281: goto st1281;
		case 1282: goto st1282;
		case 1283: goto st1283;
		case 1284: goto st1284;
		case 1285: goto st1285;
		case 1286: goto st1286;
		case 1287: goto st1287;
		case 1477: goto st1477;
		case 1288: goto st1288;
		case 1289: goto st1289;
		case 1290: goto st1290;
		case 1291: goto st1291;
		case 1292: goto st1292;
		case 1293: goto st1293;
		case 1294: goto st1294;
		case 1295: goto st1295;
		case 1296: goto st1296;
		case 1297: goto st1297;
		case 1298: goto st1298;
		case 1299: goto st1299;
		case 1300: goto st1300;
		case 1301: goto st1301;
		case 1302: goto st1302;
		case 1303: goto st1303;
		case 1304: goto st1304;
		case 1305: goto st1305;
		case 1306: goto st1306;
		case 1478: goto st1478;
		case 1307: goto st1307;
		case 1308: goto st1308;
		case 1309: goto st1309;
		case 1310: goto st1310;
		case 1311: goto st1311;
		case 1312: goto st1312;
		case 1313: goto st1313;
		case 1314: goto st1314;
		case 1315: goto st1315;
		case 1316: goto st1316;
		case 1317: goto st1317;
		case 1318: goto st1318;
		case 1319: goto st1319;
		case 1320: goto st1320;
		case 1321: goto st1321;
		case 1479: goto st1479;
		case 1480: goto st1480;
		case 1481: goto st1481;
		case 1322: goto st1322;
		case 1323: goto st1323;
		case 1324: goto st1324;
		case 1325: goto st1325;
		case 1326: goto st1326;
		case 1327: goto st1327;
		case 1328: goto st1328;
		case 1329: goto st1329;
		case 1482: goto st1482;
		case 1330: goto st1330;
		case 1331: goto st1331;
		case 1332: goto st1332;
		case 1333: goto st1333;
		case 1334: goto st1334;
		case 1335: goto st1335;
		case 1336: goto st1336;
		case 1337: goto st1337;
		case 1338: goto st1338;
		case 1339: goto st1339;
		case 1340: goto st1340;
		case 1483: goto st1483;
		case 1484: goto st1484;
		case 1485: goto st1485;
		case 1341: goto st1341;
		case 1342: goto st1342;
		case 1343: goto st1343;
		case 1344: goto st1344;
		case 1345: goto st1345;
		case 1346: goto st1346;
		case 1347: goto st1347;
		case 1486: goto st1486;
		case 1348: goto st1348;
		case 1349: goto st1349;
		case 1350: goto st1350;
		case 1351: goto st1351;
		case 1352: goto st1352;
		case 1353: goto st1353;
		case 1354: goto st1354;
		case 1355: goto st1355;
		case 1356: goto st1356;
		case 1357: goto st1357;
		case 1358: goto st1358;
		case 1359: goto st1359;
		case 1360: goto st1360;
		case 1487: goto st1487;
		case 1361: goto st1361;
		case 1362: goto st1362;
		case 1363: goto st1363;
		case 1364: goto st1364;
		case 1365: goto st1365;
		case 1366: goto st1366;
		case 1367: goto st1367;
		case 1488: goto st1488;
		case 1368: goto st1368;
		case 1369: goto st1369;
		case 1370: goto st1370;
		case 1371: goto st1371;
		case 1372: goto st1372;
		case 1373: goto st1373;
		case 1489: goto st1489;
		case 1374: goto st1374;
		case 1375: goto st1375;
		case 1376: goto st1376;
		case 1377: goto st1377;
		case 1378: goto st1378;
		case 1379: goto st1379;
		case 1490: goto st1490;
		case 1380: goto st1380;
		case 1381: goto st1381;
		case 1382: goto st1382;
		case 1383: goto st1383;
		case 1384: goto st1384;
		case 1385: goto st1385;
		case 1386: goto st1386;
		case 1387: goto st1387;
		case 1491: goto st1491;
		case 1388: goto st1388;
		case 1389: goto st1389;
		case 1390: goto st1390;
		case 1391: goto st1391;
		case 1392: goto st1392;
		case 1393: goto st1393;
		case 1394: goto st1394;
		case 1492: goto st1492;
		case 1395: goto st1395;
	default: break;
	}

	if ( ++p == pe )
		goto _test_eof;
_resume:
	switch ( cs )
	{
tr22:
	{
		s->line_counter++;
	}
	goto st1396;
tr87:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1396; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1396; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1396;
tr93:
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1396; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1396; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1396;
tr96:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1396; goto _out;}
			}
		}
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1396; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1396; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1396;
tr754:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1396; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st1396;
tr848:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1396; goto _out;}
			}
		}
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1396; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1396; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1396;
tr966:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1396; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1396; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1396; goto _out;}
			}
		}
	}
	goto st1396;
tr970:
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1396; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1396; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1396; goto _out;}
			}
		}
	}
	goto st1396;
tr4211:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->line_counter++;
	}
	goto st1396;
tr4275:
	{
		NOERR;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->line_counter++;
	}
	goto st1396;
st1396:
	if ( ++p == pe )
		goto _test_eof1396;
case 1396:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr4206;
		case 32: goto tr4206;
		case 36: goto st184;
		case 40: goto tr4208;
		case 41: goto tr4209;
		case 42: goto tr4210;
		case 92: goto tr4210;
		case 95: goto tr4210;
		case 1802: goto tr4211;
		case 1851: goto tr4212;
		case 2058: goto tr4213;
		case 2107: goto tr4214;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr4210;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr4210;
	} else
		goto tr4210;
	goto tr4205;
tr0:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr26:
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	goto st0;
tr38:
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	goto st0;
tr61:
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr75:
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr83:
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr89:
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr118:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr145:
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr151:
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr656:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	goto st0;
tr658:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr755:
	{
		ERR(ZS_BAD_DIRECTIVE);
		p--; {goto st307;}
	}
	goto st0;
tr766:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	goto st0;
tr780:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	goto st0;
tr808:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr823:
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr861:
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_DIRECTIVE);
		p--; {goto st307;}
	}
	goto st0;
tr871:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr877:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr890:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr904:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr996:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	goto st0;
tr1011:
	{
		WARN(ZS_BAD_DNAME_CHAR);
		p--; {goto st307;}
	}
	goto st0;
tr1024:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_DNAME_CHAR);
		p--; {goto st307;}
	}
	goto st0;
tr1032:
	{
		WARN(ZS_BAD_TEXT_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_TEXT);
		p--; {goto st307;}
	}
	goto st0;
tr1040:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_TEXT_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_TEXT);
		p--; {goto st307;}
	}
	goto st0;
tr1056:
	{
		WARN(ZS_BAD_TEXT);
		p--; {goto st307;}
	}
	goto st0;
tr1065:
	{
		ERR(ZS_BAD_TTL);
		p--; {goto st307;}
	}
	goto st0;
tr1071:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_TTL);
		p--; {goto st307;}
	}
	goto st0;
tr1073:
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_TTL);
		p--; {goto st307;}
	}
	goto st0;
tr1085:
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_TTL);
		p--; {goto st307;}
	}
	goto st0;
tr1094:
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_TTL);
		p--; {goto st307;}
	}
	goto st0;
tr1109:
	{
		ERR(ZS_BAD_ORIGIN);
		p--; {goto st307;}
	}
	goto st0;
tr1121:
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_ORIGIN);
		p--; {goto st307;}
	}
	goto st0;
tr1137:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_ORIGIN);
		p--; {goto st307;}
	}
	goto st0;
tr1151:
	{
		ERR(ZS_BAD_INCLUDE_FILENAME);
		p--; {goto st307;}
	}
	goto st0;
tr1152:
	{
		ERR(ZS_BAD_INCLUDE_FILENAME);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr1159:
	{
		ERR(ZS_BAD_INCLUDE_ORIGIN);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr1169:
	{
		ERR(ZS_BAD_INCLUDE_ORIGIN);
		p--; {goto st307;}
	}
	goto st0;
tr1186:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_INCLUDE_ORIGIN);
		p--; {goto st307;}
	}
	goto st0;
tr1203:
	{
		WARN(ZS_BAD_BASE64_CHAR);
		p--; {goto st307;}
	}
	goto st0;
tr1219:
	{
		WARN(ZS_BAD_BITMAP);
		p--; {goto st307;}
	}
	goto st0;
tr1792:
	{
		WARN(ZS_BAD_HEX_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr1798:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_HEX_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr1806:
	{
		WARN(ZS_BAD_HEX_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_HEX_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_HEX_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr1868:
	{
		WARN(ZS_BAD_SVCB_MANDATORY);
		p--; {goto st307;}
	}
	goto st0;
tr1898:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_SVCB_MANDATORY);
		p--; {goto st307;}
	}
	goto st0;
tr1921:
	{
		WARN(ZS_BAD_SVCB_PARAM);
		p--; {goto st307;}
	}
	goto st0;
tr2005:
	{
		WARN(ZS_BAD_ADDRESS_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_SVCB_PARAM);
		p--; {goto st307;}
	}
	goto st0;
tr2049:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_SVCB_PARAM);
		p--; {goto st307;}
	}
	goto st0;
tr2129:
	{
		WARN(ZS_BAD_ALGORITHM);
		p--; {goto st307;}
	}
	goto st0;
tr2251:
	{
		WARN(ZS_BAD_CERT_TYPE);
		p--; {goto st307;}
	}
	goto st0;
tr2304:
	{
		WARN(ZS_BAD_ADDRESS_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2327:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2346:
	{
		WARN(ZS_BAD_TIME_UNIT);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2512:
	{
		s->long_string = false;
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2535:
	{
		WARN(ZS_BAD_LOC_DATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2938:
	{
		WARN(ZS_BAD_APL);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2953:
	{
		WARN(ZS_BAD_ADDRESS_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_APL);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr2998:
	{
		WARN(ZS_BAD_HEX_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3069:
	{
		WARN(ZS_BAD_GATEWAY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3084:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_GATEWAY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3098:
	{
		WARN(ZS_BAD_GATEWAY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_GATEWAY_KEY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3121:
	{
		WARN(ZS_BAD_GATEWAY_KEY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3145:
	{
		WARN(ZS_BAD_ADDRESS_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_GATEWAY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3152:
	{
		WARN(ZS_BAD_ADDRESS_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_GATEWAY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_GATEWAY_KEY);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3271:
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3327:
	{
		WARN(ZS_BAD_TIMESTAMP_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3756:
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr3867:
	{
		WARN(ZS_BAD_BASE32HEX_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr4106:
	{
		WARN(ZS_BAD_HEX_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_CHAR_COLON);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr4112:
	{
		WARN(ZS_BAD_CHAR_COLON);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_HEX_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr4118:
	{
		WARN(ZS_BAD_CHAR_DASH);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_HEX_CHAR);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	goto st0;
tr4205:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr4227:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr4250:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr4269:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		ERR(ZS_BAD_DIRECTIVE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
tr4293:
	{
		s->r_owner_length = 0;
		WARN(ZS_BAD_OWNER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_NUMBER);
		p--; {goto st307;}
	}
	{
		WARN(ZS_UNSUPPORTED_TYPE);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_RDATA);
		p--; {goto st307;}
	}
	{
		WARN(ZS_BAD_REST);
		p--; {goto st307;}
	}
	goto st0;
st0:
cs = 0;
	goto _out;
tr2:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st1;
tr3:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st1;
tr4206:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4208:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4209:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4279:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4280:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4270:
	{
		NOERR;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4272:
	{
		NOERR;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
tr4273:
	{
		NOERR;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1;
st1:
	if ( ++p == pe )
		goto _test_eof1;
case 1:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st1;
		case 32: goto st1;
		case 40: goto tr2;
		case 41: goto tr3;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 1802: goto tr22;
		case 1851: goto tr23;
		case 2058: goto tr24;
		case 2107: goto tr25;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr4:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st2;
tr30:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st2;
st2:
	if ( ++p == pe )
		goto _test_eof2;
case 2:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr27;
		case 32: goto tr27;
		case 40: goto tr28;
		case 41: goto tr29;
		case 68: goto tr31;
		case 72: goto tr32;
		case 77: goto tr33;
		case 83: goto st198;
		case 87: goto tr35;
		case 100: goto tr31;
		case 104: goto tr32;
		case 109: goto tr33;
		case 115: goto st198;
		case 119: goto tr35;
		case 2058: goto tr36;
		case 2107: goto tr37;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr30;
	goto tr26;
tr40:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st3;
tr41:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st3;
tr59:
	{
		s->line_counter++;
	}
	goto st3;
tr27:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st3;
tr28:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st3;
tr29:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st3;
tr36:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	goto st3;
tr807:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 3; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st3;
tr800:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st3;
tr801:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st3;
tr802:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st3;
tr804:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	goto st3;
st3:
	if ( ++p == pe )
		goto _test_eof3;
case 3:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st3;
		case 32: goto st3;
		case 40: goto tr40;
		case 41: goto tr41;
		case 65: goto tr42;
		case 67: goto tr43;
		case 68: goto tr44;
		case 69: goto tr45;
		case 72: goto tr46;
		case 73: goto tr47;
		case 75: goto tr48;
		case 76: goto tr49;
		case 77: goto tr50;
		case 78: goto tr51;
		case 79: goto tr52;
		case 80: goto tr53;
		case 82: goto tr54;
		case 83: goto tr55;
		case 84: goto tr56;
		case 85: goto tr57;
		case 90: goto tr58;
		case 97: goto tr42;
		case 99: goto tr43;
		case 100: goto tr44;
		case 101: goto tr45;
		case 104: goto tr46;
		case 105: goto tr47;
		case 107: goto tr48;
		case 108: goto tr49;
		case 109: goto tr50;
		case 110: goto tr51;
		case 111: goto tr52;
		case 112: goto tr53;
		case 114: goto tr54;
		case 115: goto tr55;
		case 116: goto tr56;
		case 117: goto tr57;
		case 122: goto tr58;
		case 2058: goto tr59;
		case 2107: goto tr60;
	}
	goto tr38;
tr5:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st4;
tr42:
	{
		s->r_class = s->default_class;
	}
	goto st4;
tr708:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st4;
st4:
	if ( ++p == pe )
		goto _test_eof4;
case 4:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr62;
		case 32: goto tr62;
		case 40: goto tr63;
		case 41: goto tr64;
		case 65: goto st266;
		case 70: goto st269;
		case 80: goto st273;
		case 97: goto st266;
		case 102: goto st269;
		case 112: goto st273;
		case 3082: goto tr68;
		case 3131: goto tr69;
		case 3338: goto tr70;
		case 3387: goto tr70;
		case 3594: goto tr71;
		case 3643: goto tr72;
	}
	goto tr61;
tr76:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr77:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr79:
	{
		s->line_counter++;
	}
	goto st5;
tr62:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr63:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr64:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr68:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr100:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 5; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st5;
tr163:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr164:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr165:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr166:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr185:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr186:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr187:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr188:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr193:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr194:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr195:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr196:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr203:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr204:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr205:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr206:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr214:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr215:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr216:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr217:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr225:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr226:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr227:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr228:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr239:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr240:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr241:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr242:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr251:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr252:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr253:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr254:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr262:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr263:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr264:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr265:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr270:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr271:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr272:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr273:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr283:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr284:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr285:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr286:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr292:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr293:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr294:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr295:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr305:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr306:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr307:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr308:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr316:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr317:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr318:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr319:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr359:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr360:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr361:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr362:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr370:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr371:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr372:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr373:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr378:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr379:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr380:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr381:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr391:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr392:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr393:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr394:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr400:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr401:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr402:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr403:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr409:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr410:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr411:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr412:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr417:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr418:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr419:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr420:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr430:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr431:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr432:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr433:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr438:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr439:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr440:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr441:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr452:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr453:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr454:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr455:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr461:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr462:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr463:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr464:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr469:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr470:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr471:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr473:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr479:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr480:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr481:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr483:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr488:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr489:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr490:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr492:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr501:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr502:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr503:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr504:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr518:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr519:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr520:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr521:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr528:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr529:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr530:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr531:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr539:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr540:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr541:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr542:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr550:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr551:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr552:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr553:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr558:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr559:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr560:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr561:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr576:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr577:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr578:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr579:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr585:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr586:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr587:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr588:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr594:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr595:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr596:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr597:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr603:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr604:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr605:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr606:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr614:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr615:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr616:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr617:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr624:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr625:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr626:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr627:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr637:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr638:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr639:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr640:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr646:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr647:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr648:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr649:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr659:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr660:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr661:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr663:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr670:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr671:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr672:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr673:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr683:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr684:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr685:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr686:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr933:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr934:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr935:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr936:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr944:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr945:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr946:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr947:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
tr953:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr954:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st5;
tr955:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st5;
tr956:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	goto st5;
st5:
	if ( ++p == pe )
		goto _test_eof5;
case 5:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st5;
		case 32: goto st5;
		case 40: goto tr76;
		case 41: goto tr77;
		case 92: goto tr78;
		case 3082: goto tr79;
		case 3131: goto tr80;
		case 3338: goto tr73;
		case 3387: goto tr73;
		case 3594: goto tr81;
		case 3643: goto tr82;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr73;
	} else
		goto tr73;
	goto tr75;
tr70:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr73:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr97:
	{
		if (*wrap == WRAP_NONE) {
			p--;
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr98:
	{
		switch (s->r_type) {
		// Next types must not have empty rdata.
		case KNOT_RRTYPE_A:
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
		case KNOT_RRTYPE_SOA:
		case KNOT_RRTYPE_HINFO:
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
		case KNOT_RRTYPE_RP:
		case KNOT_RRTYPE_AAAA:
		case KNOT_RRTYPE_LOC:
		case KNOT_RRTYPE_SRV:
		case KNOT_RRTYPE_NAPTR:
		case KNOT_RRTYPE_CERT:
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_SSHFP:
		case KNOT_RRTYPE_IPSECKEY:
		case KNOT_RRTYPE_RRSIG:
		case KNOT_RRTYPE_NSEC:
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_NSEC3:
		case KNOT_RRTYPE_NSEC3PARAM:
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
		case KNOT_RRTYPE_CDS:
		case KNOT_RRTYPE_CDNSKEY:
		case KNOT_RRTYPE_OPENPGPKEY:
		case KNOT_RRTYPE_CSYNC:
		case KNOT_RRTYPE_ZONEMD:
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L32:
		case KNOT_RRTYPE_L64:
		case KNOT_RRTYPE_LP:
		case KNOT_RRTYPE_EUI48:
		case KNOT_RRTYPE_EUI64:
		case KNOT_RRTYPE_URI:
		case KNOT_RRTYPE_CAA:
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st540;}
		// Next types can have empty rdata.
		case KNOT_RRTYPE_APL:
		default:
			{stack[top++] = 6;goto st549;}
		}
	}
	goto st6;
tr168:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr190:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr198:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr208:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr219:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr230:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr244:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr256:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr267:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr275:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr288:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr297:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr310:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr321:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr364:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr375:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr383:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr396:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr405:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr414:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr422:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr435:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr443:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr457:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr466:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr475:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr485:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr494:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr506:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr523:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr533:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr544:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr555:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr563:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr581:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr590:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr599:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr608:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr619:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr629:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr642:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr651:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr665:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr675:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr688:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr938:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr949:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
tr958:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 6;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 6;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 6;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 6;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 6;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 6;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 6;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 6;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 6;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st6;
st6:
	if ( ++p == pe )
		goto _test_eof6;
case 6:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr83;
tr91:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st7;
tr92:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st7;
tr84:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st7;
tr85:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st7;
tr86:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st7;
st7:
	if ( ++p == pe )
		goto _test_eof7;
case 7:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr91;
		case 41: goto tr92;
		case 1802: goto tr93;
		case 1851: goto tr94;
		case 2058: goto tr93;
		case 2107: goto tr94;
	}
	goto tr89;
tr94:
	{
		s->buffer_length = 0;
	}
	goto st8;
tr88:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st8;
tr95:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st8;
tr849:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st8;
tr853:
	{
		s->buffer_length = 0;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st8;
st8:
	if ( ++p == pe )
		goto _test_eof8;
case 8:
	if ( (*p) == 10 )
		goto tr96;
	goto tr95;
tr78:
	{
		if (pe - p == 1) {
			*wrap = WRAP_DETECTED;
		}
	}
	goto st9;
st9:
	if ( ++p == pe )
		goto _test_eof9;
case 9:
	if ( (*p) == 35 )
		goto tr98;
	goto tr97;
tr80:
	{
		s->buffer_length = 0;
	}
	goto st10;
tr69:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr99:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st10;
tr167:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr189:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr197:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr207:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr218:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr229:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr243:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr255:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr266:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr274:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr287:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr296:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr309:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr320:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr363:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr374:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr382:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr395:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr404:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr413:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr421:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr434:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr442:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr456:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr465:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr474:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr484:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr493:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr505:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr522:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr532:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr543:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr554:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr562:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr580:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr589:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr598:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr607:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr618:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr628:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr641:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr650:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr664:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr674:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr687:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr937:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr948:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
tr957:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	goto st10;
st10:
	if ( ++p == pe )
		goto _test_eof10;
case 10:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr100;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr99;
	goto tr75;
tr71:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr81:
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr169:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr191:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr199:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr209:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr220:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr231:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr245:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr257:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr268:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr276:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr289:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr298:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr311:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr322:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr365:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr376:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr384:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr397:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr406:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr415:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr423:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr436:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr444:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr458:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr467:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr476:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr486:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr495:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr507:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr524:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr534:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr545:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr556:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr564:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr582:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr591:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr600:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr609:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr620:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr630:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr643:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr652:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr666:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr676:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr689:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr939:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr950:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
tr959:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 11;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 11;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 11;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 11;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 11;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 11;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 11;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 11;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 11;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st11;
st11:
	if ( ++p == pe )
		goto _test_eof11;
case 11:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr101;
		case 32: goto tr101;
		case 40: goto tr102;
		case 41: goto tr103;
		case 92: goto tr78;
		case 2826: goto tr87;
		case 2875: goto tr88;
		case 3082: goto tr87;
		case 3131: goto tr104;
		case 3338: goto tr105;
		case 3387: goto tr106;
		case 3594: goto tr87;
		case 3643: goto tr107;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr73;
	} else
		goto tr73;
	goto tr83;
tr109:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st12;
tr110:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st12;
tr101:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st12;
tr102:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st12;
tr103:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st12;
st12:
	if ( ++p == pe )
		goto _test_eof12;
case 12:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st12;
		case 32: goto st12;
		case 40: goto tr109;
		case 41: goto tr110;
		case 92: goto tr78;
		case 2826: goto tr93;
		case 2875: goto tr94;
		case 3082: goto tr93;
		case 3131: goto tr111;
		case 3338: goto tr112;
		case 3387: goto tr113;
		case 3594: goto tr93;
		case 3643: goto tr114;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr73;
	} else
		goto tr73;
	goto tr83;
tr111:
	{
		s->buffer_length = 0;
	}
	goto st13;
tr115:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st13;
tr104:
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st13;
tr174:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st13;
tr178:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st13;
tr828:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st13;
st13:
	if ( ++p == pe )
		goto _test_eof13;
case 13:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr96;
		case 2058: goto tr116;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr115;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr83;
tr116:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1397; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1397; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1397; goto _out;}
		}
	}
	goto st1397;
tr836:
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1397; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1397; goto _out;}
		}
	}
	goto st1397;
tr827:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1397; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1397; goto _out;}
		}
	}
	goto st1397;
st1397:
	if ( ++p == pe )
		goto _test_eof1397;
case 1397:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr4215;
		case 32: goto tr4215;
		case 36: goto tr4216;
		case 40: goto tr4217;
		case 41: goto tr4218;
		case 42: goto tr4219;
		case 58: goto tr73;
		case 92: goto tr4220;
		case 95: goto tr4219;
		case 2826: goto tr4211;
		case 2875: goto tr4212;
		case 3082: goto tr4221;
		case 3131: goto tr4222;
		case 3338: goto tr4223;
		case 3387: goto tr4224;
		case 3594: goto tr4225;
		case 3643: goto tr4226;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr73;
		} else if ( _widec > 44 ) {
			if ( 45 <= _widec && _widec <= 57 )
				goto tr4219;
		} else
			goto tr73;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 90 )
				goto tr4219;
		} else if ( _widec > 96 ) {
			if ( _widec > 122 ) {
				if ( 123 <= _widec )
					goto tr73;
			} else if ( _widec >= 97 )
				goto tr4219;
		} else
			goto tr73;
	} else
		goto tr73;
	goto tr871;
tr119:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st14;
tr120:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st14;
tr4215:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st14;
tr4217:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st14;
tr4218:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st14;
st14:
	if ( ++p == pe )
		goto _test_eof14;
case 14:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st14;
		case 32: goto st14;
		case 40: goto tr119;
		case 41: goto tr120;
		case 58: goto tr73;
		case 65: goto tr122;
		case 67: goto tr123;
		case 68: goto tr124;
		case 69: goto tr125;
		case 72: goto tr126;
		case 73: goto tr127;
		case 75: goto tr128;
		case 76: goto tr129;
		case 77: goto tr130;
		case 78: goto tr131;
		case 79: goto tr132;
		case 80: goto tr133;
		case 82: goto tr134;
		case 83: goto tr135;
		case 84: goto tr136;
		case 85: goto tr137;
		case 90: goto tr138;
		case 92: goto tr78;
		case 97: goto tr122;
		case 99: goto tr123;
		case 100: goto tr124;
		case 101: goto tr125;
		case 104: goto tr126;
		case 105: goto tr127;
		case 107: goto tr128;
		case 108: goto tr129;
		case 109: goto tr130;
		case 110: goto tr131;
		case 111: goto tr132;
		case 112: goto tr133;
		case 114: goto tr134;
		case 115: goto tr135;
		case 116: goto tr136;
		case 117: goto tr137;
		case 122: goto tr138;
		case 2826: goto tr22;
		case 2875: goto tr23;
		case 3082: goto tr139;
		case 3131: goto tr140;
		case 3338: goto tr141;
		case 3387: goto tr142;
		case 3594: goto tr143;
		case 3643: goto tr144;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr73;
		} else if ( _widec >= 48 )
			goto tr121;
	} else
		goto tr73;
	goto tr118;
tr121:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 15;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 15;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 15;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 15;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 15;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 15;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 15;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 15;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 15;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 15;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 15;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 15;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 15;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 15;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 15;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 15;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 15;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 15;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 15;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 15;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 15;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 15;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 15;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 15;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 15;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 15;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 15;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 15;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 15;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 15;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 15;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 15;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st15;
st15:
	if ( ++p == pe )
		goto _test_eof15;
case 15:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr146;
		case 32: goto tr146;
		case 40: goto tr147;
		case 41: goto tr148;
		case 68: goto tr31;
		case 72: goto tr32;
		case 77: goto tr33;
		case 83: goto st198;
		case 87: goto tr35;
		case 100: goto tr31;
		case 104: goto tr32;
		case 109: goto tr33;
		case 115: goto st198;
		case 119: goto tr35;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr149;
		case 2107: goto tr150;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr30;
	goto tr145;
tr153:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st16;
tr154:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st16;
tr146:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st16;
tr147:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st16;
tr148:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st16;
st16:
	if ( ++p == pe )
		goto _test_eof16;
case 16:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st16;
		case 32: goto st16;
		case 40: goto tr153;
		case 41: goto tr154;
		case 65: goto tr42;
		case 67: goto tr43;
		case 68: goto tr44;
		case 69: goto tr45;
		case 72: goto tr46;
		case 73: goto tr47;
		case 75: goto tr48;
		case 76: goto tr49;
		case 77: goto tr50;
		case 78: goto tr51;
		case 79: goto tr52;
		case 80: goto tr53;
		case 82: goto tr54;
		case 83: goto tr55;
		case 84: goto tr56;
		case 85: goto tr57;
		case 90: goto tr58;
		case 97: goto tr42;
		case 99: goto tr43;
		case 100: goto tr44;
		case 101: goto tr45;
		case 104: goto tr46;
		case 105: goto tr47;
		case 107: goto tr48;
		case 108: goto tr49;
		case 109: goto tr50;
		case 110: goto tr51;
		case 111: goto tr52;
		case 112: goto tr53;
		case 114: goto tr54;
		case 115: goto tr55;
		case 116: goto tr56;
		case 117: goto tr57;
		case 122: goto tr58;
		case 1802: goto tr93;
		case 1851: goto tr94;
		case 2058: goto tr155;
		case 2107: goto tr156;
	}
	goto tr151;
tr6:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st17;
tr43:
	{
		s->r_class = s->default_class;
	}
	goto st17;
tr709:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st17;
st17:
	if ( ++p == pe )
		goto _test_eof17;
case 17:
	switch( (*p) ) {
		case 65: goto st18;
		case 68: goto st22;
		case 69: goto st29;
		case 78: goto st32;
		case 83: goto st36;
		case 97: goto st18;
		case 100: goto st22;
		case 101: goto st29;
		case 110: goto st32;
		case 115: goto st36;
	}
	goto tr38;
st18:
	if ( ++p == pe )
		goto _test_eof18;
case 18:
	switch( (*p) ) {
		case 65: goto st19;
		case 97: goto st19;
	}
	goto tr38;
st19:
	if ( ++p == pe )
		goto _test_eof19;
case 19:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr163;
		case 32: goto tr163;
		case 40: goto tr164;
		case 41: goto tr165;
		case 3082: goto tr166;
		case 3131: goto tr167;
		case 3338: goto tr168;
		case 3387: goto tr168;
		case 3594: goto tr169;
		case 3643: goto tr170;
	}
	goto tr61;
tr72:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr82:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr170:
	{ s->r_type = KNOT_RRTYPE_CAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr192:
	{ s->r_type = KNOT_RRTYPE_CDNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr200:
	{ s->r_type = KNOT_RRTYPE_CDS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr210:
	{ s->r_type = KNOT_RRTYPE_CERT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr221:
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr232:
	{ s->r_type = KNOT_RRTYPE_CSYNC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr246:
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr258:
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr269:
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr277:
	{ s->r_type = KNOT_RRTYPE_DS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr290:
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr299:
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr312:
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr323:
	{ s->r_type = KNOT_RRTYPE_HTTPS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr366:
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr377:
	{ s->r_type = KNOT_RRTYPE_KEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr385:
	{ s->r_type = KNOT_RRTYPE_KX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr398:
	{ s->r_type = KNOT_RRTYPE_L32; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr407:
	{ s->r_type = KNOT_RRTYPE_L64; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr416:
	{ s->r_type = KNOT_RRTYPE_LOC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr424:
	{ s->r_type = KNOT_RRTYPE_LP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr437:
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr445:
	{ s->r_type = KNOT_RRTYPE_MX; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr459:
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr468:
	{ s->r_type = KNOT_RRTYPE_NID; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr477:
	{ s->r_type = KNOT_RRTYPE_NS; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr487:
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr496:
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr508:
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr525:
	{ s->r_type = KNOT_RRTYPE_OPENPGPKEY; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr535:
	{ s->r_type = KNOT_RRTYPE_PTR; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr546:
	{ s->r_type = KNOT_RRTYPE_RP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr557:
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr565:
	{ s->r_type = KNOT_RRTYPE_RT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr583:
	{ s->r_type = KNOT_RRTYPE_SMIMEA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr592:
	{ s->r_type = KNOT_RRTYPE_SOA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr601:
	{ s->r_type = KNOT_RRTYPE_SPF; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr610:
	{ s->r_type = KNOT_RRTYPE_SRV; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr621:
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr631:
	{ s->r_type = KNOT_RRTYPE_SVCB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr644:
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr653:
	{ s->r_type = KNOT_RRTYPE_TXT; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr667:
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZS_NUMBER16_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr677:
	{ s->r_type = KNOT_RRTYPE_URI; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr690:
	{ s->r_type = KNOT_RRTYPE_ZONEMD; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr940:
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr951:
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
tr960:
	{ s->r_type = KNOT_RRTYPE_APL; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 20;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 20;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 20;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 20;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 20;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 20;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 20;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 20;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 20;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st20;
st20:
	if ( ++p == pe )
		goto _test_eof20;
case 20:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr84;
		case 1802: goto tr87;
		case 1824: goto tr84;
		case 1832: goto tr85;
		case 1833: goto tr86;
		case 1851: goto tr88;
		case 2057: goto tr171;
		case 2058: goto tr87;
		case 2080: goto tr171;
		case 2088: goto tr172;
		case 2089: goto tr173;
		case 2107: goto tr174;
	}
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr99;
	goto tr83;
tr175:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st21;
tr171:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st21;
tr172:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st21;
tr173:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st21;
tr176:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st21;
tr177:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st21;
st21:
	if ( ++p == pe )
		goto _test_eof21;
case 21:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto st7;
		case 1802: goto tr93;
		case 1824: goto st7;
		case 1832: goto tr91;
		case 1833: goto tr92;
		case 1851: goto tr94;
		case 2057: goto tr175;
		case 2058: goto tr93;
		case 2080: goto tr175;
		case 2088: goto tr176;
		case 2089: goto tr177;
		case 2107: goto tr178;
	}
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr99;
	goto tr83;
st22:
	if ( ++p == pe )
		goto _test_eof22;
case 22:
	switch( (*p) ) {
		case 78: goto st23;
		case 83: goto st28;
		case 110: goto st23;
		case 115: goto st28;
	}
	goto tr38;
st23:
	if ( ++p == pe )
		goto _test_eof23;
case 23:
	switch( (*p) ) {
		case 83: goto st24;
		case 115: goto st24;
	}
	goto tr38;
st24:
	if ( ++p == pe )
		goto _test_eof24;
case 24:
	switch( (*p) ) {
		case 75: goto st25;
		case 107: goto st25;
	}
	goto tr38;
st25:
	if ( ++p == pe )
		goto _test_eof25;
case 25:
	switch( (*p) ) {
		case 69: goto st26;
		case 101: goto st26;
	}
	goto tr38;
st26:
	if ( ++p == pe )
		goto _test_eof26;
case 26:
	switch( (*p) ) {
		case 89: goto st27;
		case 121: goto st27;
	}
	goto tr38;
st27:
	if ( ++p == pe )
		goto _test_eof27;
case 27:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr185;
		case 32: goto tr185;
		case 40: goto tr186;
		case 41: goto tr187;
		case 3082: goto tr188;
		case 3131: goto tr189;
		case 3338: goto tr190;
		case 3387: goto tr190;
		case 3594: goto tr191;
		case 3643: goto tr192;
	}
	goto tr61;
st28:
	if ( ++p == pe )
		goto _test_eof28;
case 28:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr193;
		case 32: goto tr193;
		case 40: goto tr194;
		case 41: goto tr195;
		case 3082: goto tr196;
		case 3131: goto tr197;
		case 3338: goto tr198;
		case 3387: goto tr198;
		case 3594: goto tr199;
		case 3643: goto tr200;
	}
	goto tr61;
st29:
	if ( ++p == pe )
		goto _test_eof29;
case 29:
	switch( (*p) ) {
		case 82: goto st30;
		case 114: goto st30;
	}
	goto tr38;
st30:
	if ( ++p == pe )
		goto _test_eof30;
case 30:
	switch( (*p) ) {
		case 84: goto st31;
		case 116: goto st31;
	}
	goto tr38;
st31:
	if ( ++p == pe )
		goto _test_eof31;
case 31:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr203;
		case 32: goto tr203;
		case 40: goto tr204;
		case 41: goto tr205;
		case 3082: goto tr206;
		case 3131: goto tr207;
		case 3338: goto tr208;
		case 3387: goto tr208;
		case 3594: goto tr209;
		case 3643: goto tr210;
	}
	goto tr61;
st32:
	if ( ++p == pe )
		goto _test_eof32;
case 32:
	switch( (*p) ) {
		case 65: goto st33;
		case 97: goto st33;
	}
	goto tr38;
st33:
	if ( ++p == pe )
		goto _test_eof33;
case 33:
	switch( (*p) ) {
		case 77: goto st34;
		case 109: goto st34;
	}
	goto tr38;
st34:
	if ( ++p == pe )
		goto _test_eof34;
case 34:
	switch( (*p) ) {
		case 69: goto st35;
		case 101: goto st35;
	}
	goto tr38;
st35:
	if ( ++p == pe )
		goto _test_eof35;
case 35:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr214;
		case 32: goto tr214;
		case 40: goto tr215;
		case 41: goto tr216;
		case 3082: goto tr217;
		case 3131: goto tr218;
		case 3338: goto tr219;
		case 3387: goto tr219;
		case 3594: goto tr220;
		case 3643: goto tr221;
	}
	goto tr61;
st36:
	if ( ++p == pe )
		goto _test_eof36;
case 36:
	switch( (*p) ) {
		case 89: goto st37;
		case 121: goto st37;
	}
	goto tr38;
st37:
	if ( ++p == pe )
		goto _test_eof37;
case 37:
	switch( (*p) ) {
		case 78: goto st38;
		case 110: goto st38;
	}
	goto tr38;
st38:
	if ( ++p == pe )
		goto _test_eof38;
case 38:
	switch( (*p) ) {
		case 67: goto st39;
		case 99: goto st39;
	}
	goto tr38;
st39:
	if ( ++p == pe )
		goto _test_eof39;
case 39:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr225;
		case 32: goto tr225;
		case 40: goto tr226;
		case 41: goto tr227;
		case 3082: goto tr228;
		case 3131: goto tr229;
		case 3338: goto tr230;
		case 3387: goto tr230;
		case 3594: goto tr231;
		case 3643: goto tr232;
	}
	goto tr61;
tr7:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st40;
tr44:
	{
		s->r_class = s->default_class;
	}
	goto st40;
tr710:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st40;
st40:
	if ( ++p == pe )
		goto _test_eof40;
case 40:
	switch( (*p) ) {
		case 72: goto st41;
		case 78: goto st45;
		case 83: goto st53;
		case 104: goto st41;
		case 110: goto st45;
		case 115: goto st53;
	}
	goto tr38;
st41:
	if ( ++p == pe )
		goto _test_eof41;
case 41:
	switch( (*p) ) {
		case 67: goto st42;
		case 99: goto st42;
	}
	goto tr38;
st42:
	if ( ++p == pe )
		goto _test_eof42;
case 42:
	switch( (*p) ) {
		case 73: goto st43;
		case 105: goto st43;
	}
	goto tr38;
st43:
	if ( ++p == pe )
		goto _test_eof43;
case 43:
	switch( (*p) ) {
		case 68: goto st44;
		case 100: goto st44;
	}
	goto tr38;
st44:
	if ( ++p == pe )
		goto _test_eof44;
case 44:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr239;
		case 32: goto tr239;
		case 40: goto tr240;
		case 41: goto tr241;
		case 3082: goto tr242;
		case 3131: goto tr243;
		case 3338: goto tr244;
		case 3387: goto tr244;
		case 3594: goto tr245;
		case 3643: goto tr246;
	}
	goto tr61;
st45:
	if ( ++p == pe )
		goto _test_eof45;
case 45:
	switch( (*p) ) {
		case 65: goto st46;
		case 83: goto st49;
		case 97: goto st46;
		case 115: goto st49;
	}
	goto tr38;
st46:
	if ( ++p == pe )
		goto _test_eof46;
case 46:
	switch( (*p) ) {
		case 77: goto st47;
		case 109: goto st47;
	}
	goto tr38;
st47:
	if ( ++p == pe )
		goto _test_eof47;
case 47:
	switch( (*p) ) {
		case 69: goto st48;
		case 101: goto st48;
	}
	goto tr38;
st48:
	if ( ++p == pe )
		goto _test_eof48;
case 48:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr251;
		case 32: goto tr251;
		case 40: goto tr252;
		case 41: goto tr253;
		case 3082: goto tr254;
		case 3131: goto tr255;
		case 3338: goto tr256;
		case 3387: goto tr256;
		case 3594: goto tr257;
		case 3643: goto tr258;
	}
	goto tr61;
st49:
	if ( ++p == pe )
		goto _test_eof49;
case 49:
	switch( (*p) ) {
		case 75: goto st50;
		case 107: goto st50;
	}
	goto tr38;
st50:
	if ( ++p == pe )
		goto _test_eof50;
case 50:
	switch( (*p) ) {
		case 69: goto st51;
		case 101: goto st51;
	}
	goto tr38;
st51:
	if ( ++p == pe )
		goto _test_eof51;
case 51:
	switch( (*p) ) {
		case 89: goto st52;
		case 121: goto st52;
	}
	goto tr38;
st52:
	if ( ++p == pe )
		goto _test_eof52;
case 52:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr262;
		case 32: goto tr262;
		case 40: goto tr263;
		case 41: goto tr264;
		case 3082: goto tr265;
		case 3131: goto tr266;
		case 3338: goto tr267;
		case 3387: goto tr267;
		case 3594: goto tr268;
		case 3643: goto tr269;
	}
	goto tr61;
st53:
	if ( ++p == pe )
		goto _test_eof53;
case 53:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr270;
		case 32: goto tr270;
		case 40: goto tr271;
		case 41: goto tr272;
		case 3082: goto tr273;
		case 3131: goto tr274;
		case 3338: goto tr275;
		case 3387: goto tr275;
		case 3594: goto tr276;
		case 3643: goto tr277;
	}
	goto tr61;
tr8:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st54;
tr45:
	{
		s->r_class = s->default_class;
	}
	goto st54;
tr711:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st54;
st54:
	if ( ++p == pe )
		goto _test_eof54;
case 54:
	switch( (*p) ) {
		case 85: goto st55;
		case 117: goto st55;
	}
	goto tr38;
st55:
	if ( ++p == pe )
		goto _test_eof55;
case 55:
	switch( (*p) ) {
		case 73: goto st56;
		case 105: goto st56;
	}
	goto tr38;
st56:
	if ( ++p == pe )
		goto _test_eof56;
case 56:
	switch( (*p) ) {
		case 52: goto st57;
		case 54: goto st59;
	}
	goto tr38;
st57:
	if ( ++p == pe )
		goto _test_eof57;
case 57:
	if ( (*p) == 56 )
		goto st58;
	goto tr38;
st58:
	if ( ++p == pe )
		goto _test_eof58;
case 58:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr283;
		case 32: goto tr283;
		case 40: goto tr284;
		case 41: goto tr285;
		case 3082: goto tr286;
		case 3131: goto tr287;
		case 3338: goto tr288;
		case 3387: goto tr288;
		case 3594: goto tr289;
		case 3643: goto tr290;
	}
	goto tr61;
st59:
	if ( ++p == pe )
		goto _test_eof59;
case 59:
	if ( (*p) == 52 )
		goto st60;
	goto tr38;
st60:
	if ( ++p == pe )
		goto _test_eof60;
case 60:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr292;
		case 32: goto tr292;
		case 40: goto tr293;
		case 41: goto tr294;
		case 3082: goto tr295;
		case 3131: goto tr296;
		case 3338: goto tr297;
		case 3387: goto tr297;
		case 3594: goto tr298;
		case 3643: goto tr299;
	}
	goto tr61;
tr9:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st61;
tr46:
	{
		s->r_class = s->default_class;
	}
	goto st61;
tr712:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st61;
st61:
	if ( ++p == pe )
		goto _test_eof61;
case 61:
	switch( (*p) ) {
		case 73: goto st62;
		case 84: goto st66;
		case 105: goto st62;
		case 116: goto st66;
	}
	goto tr38;
st62:
	if ( ++p == pe )
		goto _test_eof62;
case 62:
	switch( (*p) ) {
		case 78: goto st63;
		case 110: goto st63;
	}
	goto tr38;
st63:
	if ( ++p == pe )
		goto _test_eof63;
case 63:
	switch( (*p) ) {
		case 70: goto st64;
		case 102: goto st64;
	}
	goto tr38;
st64:
	if ( ++p == pe )
		goto _test_eof64;
case 64:
	switch( (*p) ) {
		case 79: goto st65;
		case 111: goto st65;
	}
	goto tr38;
st65:
	if ( ++p == pe )
		goto _test_eof65;
case 65:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr305;
		case 32: goto tr305;
		case 40: goto tr306;
		case 41: goto tr307;
		case 3082: goto tr308;
		case 3131: goto tr309;
		case 3338: goto tr310;
		case 3387: goto tr310;
		case 3594: goto tr311;
		case 3643: goto tr312;
	}
	goto tr61;
st66:
	if ( ++p == pe )
		goto _test_eof66;
case 66:
	switch( (*p) ) {
		case 84: goto st67;
		case 116: goto st67;
	}
	goto tr38;
st67:
	if ( ++p == pe )
		goto _test_eof67;
case 67:
	switch( (*p) ) {
		case 80: goto st68;
		case 112: goto st68;
	}
	goto tr38;
st68:
	if ( ++p == pe )
		goto _test_eof68;
case 68:
	switch( (*p) ) {
		case 83: goto st69;
		case 115: goto st69;
	}
	goto tr38;
st69:
	if ( ++p == pe )
		goto _test_eof69;
case 69:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr316;
		case 32: goto tr316;
		case 40: goto tr317;
		case 41: goto tr318;
		case 3082: goto tr319;
		case 3131: goto tr320;
		case 3338: goto tr321;
		case 3387: goto tr321;
		case 3594: goto tr322;
		case 3643: goto tr323;
	}
	goto tr61;
tr47:
	{
		s->r_class = s->default_class;
	}
	goto st70;
st70:
	if ( ++p == pe )
		goto _test_eof70;
case 70:
	switch( (*p) ) {
		case 78: goto st71;
		case 80: goto st74;
		case 110: goto st71;
		case 112: goto st74;
	}
	goto tr38;
st71:
	if ( ++p == pe )
		goto _test_eof71;
case 71:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr326;
		case 32: goto tr326;
		case 40: goto tr327;
		case 41: goto tr328;
		case 2058: goto tr329;
		case 2107: goto tr330;
	}
	goto tr38;
tr332:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st72;
tr333:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st72;
tr351:
	{
		s->line_counter++;
	}
	goto st72;
tr727:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st72;
tr728:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st72;
tr729:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st72;
tr736:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	goto st72;
tr692:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 72; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st72;
tr326:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	goto st72;
tr327:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st72;
tr328:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st72;
tr329:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		s->line_counter++;
	}
	goto st72;
tr745:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st72;
tr746:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st72;
tr747:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st72;
tr749:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	goto st72;
st72:
	if ( ++p == pe )
		goto _test_eof72;
case 72:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st72;
		case 32: goto st72;
		case 40: goto tr332;
		case 41: goto tr333;
		case 65: goto st4;
		case 67: goto st17;
		case 68: goto st40;
		case 69: goto st54;
		case 72: goto st61;
		case 73: goto st73;
		case 75: goto st81;
		case 76: goto st85;
		case 77: goto st93;
		case 78: goto st99;
		case 79: goto st115;
		case 80: goto st125;
		case 82: goto st128;
		case 83: goto st135;
		case 84: goto st154;
		case 85: goto st164;
		case 90: goto st167;
		case 97: goto st4;
		case 99: goto st17;
		case 100: goto st40;
		case 101: goto st54;
		case 104: goto st61;
		case 105: goto st73;
		case 107: goto st81;
		case 108: goto st85;
		case 109: goto st93;
		case 110: goto st99;
		case 111: goto st115;
		case 112: goto st125;
		case 114: goto st128;
		case 115: goto st135;
		case 116: goto st154;
		case 117: goto st164;
		case 122: goto st167;
		case 2058: goto tr351;
		case 2107: goto tr352;
	}
	goto tr38;
tr713:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st73;
st73:
	if ( ++p == pe )
		goto _test_eof73;
case 73:
	switch( (*p) ) {
		case 80: goto st74;
		case 112: goto st74;
	}
	goto tr38;
st74:
	if ( ++p == pe )
		goto _test_eof74;
case 74:
	switch( (*p) ) {
		case 83: goto st75;
		case 115: goto st75;
	}
	goto tr38;
st75:
	if ( ++p == pe )
		goto _test_eof75;
case 75:
	switch( (*p) ) {
		case 69: goto st76;
		case 101: goto st76;
	}
	goto tr38;
st76:
	if ( ++p == pe )
		goto _test_eof76;
case 76:
	switch( (*p) ) {
		case 67: goto st77;
		case 99: goto st77;
	}
	goto tr38;
st77:
	if ( ++p == pe )
		goto _test_eof77;
case 77:
	switch( (*p) ) {
		case 75: goto st78;
		case 107: goto st78;
	}
	goto tr38;
st78:
	if ( ++p == pe )
		goto _test_eof78;
case 78:
	switch( (*p) ) {
		case 69: goto st79;
		case 101: goto st79;
	}
	goto tr38;
st79:
	if ( ++p == pe )
		goto _test_eof79;
case 79:
	switch( (*p) ) {
		case 89: goto st80;
		case 121: goto st80;
	}
	goto tr38;
st80:
	if ( ++p == pe )
		goto _test_eof80;
case 80:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr359;
		case 32: goto tr359;
		case 40: goto tr360;
		case 41: goto tr361;
		case 3082: goto tr362;
		case 3131: goto tr363;
		case 3338: goto tr364;
		case 3387: goto tr364;
		case 3594: goto tr365;
		case 3643: goto tr366;
	}
	goto tr61;
tr11:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st81;
tr48:
	{
		s->r_class = s->default_class;
	}
	goto st81;
tr714:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st81;
st81:
	if ( ++p == pe )
		goto _test_eof81;
case 81:
	switch( (*p) ) {
		case 69: goto st82;
		case 88: goto st84;
		case 101: goto st82;
		case 120: goto st84;
	}
	goto tr38;
st82:
	if ( ++p == pe )
		goto _test_eof82;
case 82:
	switch( (*p) ) {
		case 89: goto st83;
		case 121: goto st83;
	}
	goto tr38;
st83:
	if ( ++p == pe )
		goto _test_eof83;
case 83:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr370;
		case 32: goto tr370;
		case 40: goto tr371;
		case 41: goto tr372;
		case 3082: goto tr373;
		case 3131: goto tr374;
		case 3338: goto tr375;
		case 3387: goto tr375;
		case 3594: goto tr376;
		case 3643: goto tr377;
	}
	goto tr61;
st84:
	if ( ++p == pe )
		goto _test_eof84;
case 84:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr378;
		case 32: goto tr378;
		case 40: goto tr379;
		case 41: goto tr380;
		case 3082: goto tr381;
		case 3131: goto tr382;
		case 3338: goto tr383;
		case 3387: goto tr383;
		case 3594: goto tr384;
		case 3643: goto tr385;
	}
	goto tr61;
tr12:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st85;
tr49:
	{
		s->r_class = s->default_class;
	}
	goto st85;
tr715:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st85;
st85:
	if ( ++p == pe )
		goto _test_eof85;
case 85:
	switch( (*p) ) {
		case 51: goto st86;
		case 54: goto st88;
		case 79: goto st90;
		case 80: goto st92;
		case 111: goto st90;
		case 112: goto st92;
	}
	goto tr38;
st86:
	if ( ++p == pe )
		goto _test_eof86;
case 86:
	if ( (*p) == 50 )
		goto st87;
	goto tr38;
st87:
	if ( ++p == pe )
		goto _test_eof87;
case 87:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr391;
		case 32: goto tr391;
		case 40: goto tr392;
		case 41: goto tr393;
		case 3082: goto tr394;
		case 3131: goto tr395;
		case 3338: goto tr396;
		case 3387: goto tr396;
		case 3594: goto tr397;
		case 3643: goto tr398;
	}
	goto tr61;
st88:
	if ( ++p == pe )
		goto _test_eof88;
case 88:
	if ( (*p) == 52 )
		goto st89;
	goto tr38;
st89:
	if ( ++p == pe )
		goto _test_eof89;
case 89:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr400;
		case 32: goto tr400;
		case 40: goto tr401;
		case 41: goto tr402;
		case 3082: goto tr403;
		case 3131: goto tr404;
		case 3338: goto tr405;
		case 3387: goto tr405;
		case 3594: goto tr406;
		case 3643: goto tr407;
	}
	goto tr61;
st90:
	if ( ++p == pe )
		goto _test_eof90;
case 90:
	switch( (*p) ) {
		case 67: goto st91;
		case 99: goto st91;
	}
	goto tr38;
st91:
	if ( ++p == pe )
		goto _test_eof91;
case 91:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr409;
		case 32: goto tr409;
		case 40: goto tr410;
		case 41: goto tr411;
		case 3082: goto tr412;
		case 3131: goto tr413;
		case 3338: goto tr414;
		case 3387: goto tr414;
		case 3594: goto tr415;
		case 3643: goto tr416;
	}
	goto tr61;
st92:
	if ( ++p == pe )
		goto _test_eof92;
case 92:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr417;
		case 32: goto tr417;
		case 40: goto tr418;
		case 41: goto tr419;
		case 3082: goto tr420;
		case 3131: goto tr421;
		case 3338: goto tr422;
		case 3387: goto tr422;
		case 3594: goto tr423;
		case 3643: goto tr424;
	}
	goto tr61;
tr13:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st93;
tr50:
	{
		s->r_class = s->default_class;
	}
	goto st93;
tr716:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st93;
st93:
	if ( ++p == pe )
		goto _test_eof93;
case 93:
	switch( (*p) ) {
		case 73: goto st94;
		case 88: goto st98;
		case 105: goto st94;
		case 120: goto st98;
	}
	goto tr38;
st94:
	if ( ++p == pe )
		goto _test_eof94;
case 94:
	switch( (*p) ) {
		case 78: goto st95;
		case 110: goto st95;
	}
	goto tr38;
st95:
	if ( ++p == pe )
		goto _test_eof95;
case 95:
	switch( (*p) ) {
		case 70: goto st96;
		case 102: goto st96;
	}
	goto tr38;
st96:
	if ( ++p == pe )
		goto _test_eof96;
case 96:
	switch( (*p) ) {
		case 79: goto st97;
		case 111: goto st97;
	}
	goto tr38;
st97:
	if ( ++p == pe )
		goto _test_eof97;
case 97:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr430;
		case 32: goto tr430;
		case 40: goto tr431;
		case 41: goto tr432;
		case 3082: goto tr433;
		case 3131: goto tr434;
		case 3338: goto tr435;
		case 3387: goto tr435;
		case 3594: goto tr436;
		case 3643: goto tr437;
	}
	goto tr61;
st98:
	if ( ++p == pe )
		goto _test_eof98;
case 98:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr438;
		case 32: goto tr438;
		case 40: goto tr439;
		case 41: goto tr440;
		case 3082: goto tr441;
		case 3131: goto tr442;
		case 3338: goto tr443;
		case 3387: goto tr443;
		case 3594: goto tr444;
		case 3643: goto tr445;
	}
	goto tr61;
tr14:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st99;
tr51:
	{
		s->r_class = s->default_class;
	}
	goto st99;
tr717:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st99;
st99:
	if ( ++p == pe )
		goto _test_eof99;
case 99:
	switch( (*p) ) {
		case 65: goto st100;
		case 73: goto st104;
		case 83: goto st106;
		case 97: goto st100;
		case 105: goto st104;
		case 115: goto st106;
	}
	goto tr38;
st100:
	if ( ++p == pe )
		goto _test_eof100;
case 100:
	switch( (*p) ) {
		case 80: goto st101;
		case 112: goto st101;
	}
	goto tr38;
st101:
	if ( ++p == pe )
		goto _test_eof101;
case 101:
	switch( (*p) ) {
		case 84: goto st102;
		case 116: goto st102;
	}
	goto tr38;
st102:
	if ( ++p == pe )
		goto _test_eof102;
case 102:
	switch( (*p) ) {
		case 82: goto st103;
		case 114: goto st103;
	}
	goto tr38;
st103:
	if ( ++p == pe )
		goto _test_eof103;
case 103:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr452;
		case 32: goto tr452;
		case 40: goto tr453;
		case 41: goto tr454;
		case 3082: goto tr455;
		case 3131: goto tr456;
		case 3338: goto tr457;
		case 3387: goto tr457;
		case 3594: goto tr458;
		case 3643: goto tr459;
	}
	goto tr61;
st104:
	if ( ++p == pe )
		goto _test_eof104;
case 104:
	switch( (*p) ) {
		case 68: goto st105;
		case 100: goto st105;
	}
	goto tr38;
st105:
	if ( ++p == pe )
		goto _test_eof105;
case 105:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr461;
		case 32: goto tr461;
		case 40: goto tr462;
		case 41: goto tr463;
		case 3082: goto tr464;
		case 3131: goto tr465;
		case 3338: goto tr466;
		case 3387: goto tr466;
		case 3594: goto tr467;
		case 3643: goto tr468;
	}
	goto tr61;
st106:
	if ( ++p == pe )
		goto _test_eof106;
case 106:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr469;
		case 32: goto tr469;
		case 40: goto tr470;
		case 41: goto tr471;
		case 69: goto st107;
		case 101: goto st107;
		case 3082: goto tr473;
		case 3131: goto tr474;
		case 3338: goto tr475;
		case 3387: goto tr475;
		case 3594: goto tr476;
		case 3643: goto tr477;
	}
	goto tr61;
st107:
	if ( ++p == pe )
		goto _test_eof107;
case 107:
	switch( (*p) ) {
		case 67: goto st108;
		case 99: goto st108;
	}
	goto tr38;
st108:
	if ( ++p == pe )
		goto _test_eof108;
case 108:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr479;
		case 32: goto tr479;
		case 40: goto tr480;
		case 41: goto tr481;
		case 51: goto st109;
		case 3082: goto tr483;
		case 3131: goto tr484;
		case 3338: goto tr485;
		case 3387: goto tr485;
		case 3594: goto tr486;
		case 3643: goto tr487;
	}
	goto tr61;
st109:
	if ( ++p == pe )
		goto _test_eof109;
case 109:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr488;
		case 32: goto tr488;
		case 40: goto tr489;
		case 41: goto tr490;
		case 80: goto st110;
		case 112: goto st110;
		case 3082: goto tr492;
		case 3131: goto tr493;
		case 3338: goto tr494;
		case 3387: goto tr494;
		case 3594: goto tr495;
		case 3643: goto tr496;
	}
	goto tr61;
st110:
	if ( ++p == pe )
		goto _test_eof110;
case 110:
	switch( (*p) ) {
		case 65: goto st111;
		case 97: goto st111;
	}
	goto tr38;
st111:
	if ( ++p == pe )
		goto _test_eof111;
case 111:
	switch( (*p) ) {
		case 82: goto st112;
		case 114: goto st112;
	}
	goto tr38;
st112:
	if ( ++p == pe )
		goto _test_eof112;
case 112:
	switch( (*p) ) {
		case 65: goto st113;
		case 97: goto st113;
	}
	goto tr38;
st113:
	if ( ++p == pe )
		goto _test_eof113;
case 113:
	switch( (*p) ) {
		case 77: goto st114;
		case 109: goto st114;
	}
	goto tr38;
st114:
	if ( ++p == pe )
		goto _test_eof114;
case 114:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr501;
		case 32: goto tr501;
		case 40: goto tr502;
		case 41: goto tr503;
		case 3082: goto tr504;
		case 3131: goto tr505;
		case 3338: goto tr506;
		case 3387: goto tr506;
		case 3594: goto tr507;
		case 3643: goto tr508;
	}
	goto tr61;
tr15:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st115;
tr52:
	{
		s->r_class = s->default_class;
	}
	goto st115;
tr718:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st115;
st115:
	if ( ++p == pe )
		goto _test_eof115;
case 115:
	switch( (*p) ) {
		case 80: goto st116;
		case 112: goto st116;
	}
	goto tr38;
st116:
	if ( ++p == pe )
		goto _test_eof116;
case 116:
	switch( (*p) ) {
		case 69: goto st117;
		case 101: goto st117;
	}
	goto tr38;
st117:
	if ( ++p == pe )
		goto _test_eof117;
case 117:
	switch( (*p) ) {
		case 78: goto st118;
		case 110: goto st118;
	}
	goto tr38;
st118:
	if ( ++p == pe )
		goto _test_eof118;
case 118:
	switch( (*p) ) {
		case 80: goto st119;
		case 112: goto st119;
	}
	goto tr38;
st119:
	if ( ++p == pe )
		goto _test_eof119;
case 119:
	switch( (*p) ) {
		case 71: goto st120;
		case 103: goto st120;
	}
	goto tr38;
st120:
	if ( ++p == pe )
		goto _test_eof120;
case 120:
	switch( (*p) ) {
		case 80: goto st121;
		case 112: goto st121;
	}
	goto tr38;
st121:
	if ( ++p == pe )
		goto _test_eof121;
case 121:
	switch( (*p) ) {
		case 75: goto st122;
		case 107: goto st122;
	}
	goto tr38;
st122:
	if ( ++p == pe )
		goto _test_eof122;
case 122:
	switch( (*p) ) {
		case 69: goto st123;
		case 101: goto st123;
	}
	goto tr38;
st123:
	if ( ++p == pe )
		goto _test_eof123;
case 123:
	switch( (*p) ) {
		case 89: goto st124;
		case 121: goto st124;
	}
	goto tr38;
st124:
	if ( ++p == pe )
		goto _test_eof124;
case 124:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr518;
		case 32: goto tr518;
		case 40: goto tr519;
		case 41: goto tr520;
		case 3082: goto tr521;
		case 3131: goto tr522;
		case 3338: goto tr523;
		case 3387: goto tr523;
		case 3594: goto tr524;
		case 3643: goto tr525;
	}
	goto tr61;
tr16:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st125;
tr53:
	{
		s->r_class = s->default_class;
	}
	goto st125;
tr719:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st125;
st125:
	if ( ++p == pe )
		goto _test_eof125;
case 125:
	switch( (*p) ) {
		case 84: goto st126;
		case 116: goto st126;
	}
	goto tr38;
st126:
	if ( ++p == pe )
		goto _test_eof126;
case 126:
	switch( (*p) ) {
		case 82: goto st127;
		case 114: goto st127;
	}
	goto tr38;
st127:
	if ( ++p == pe )
		goto _test_eof127;
case 127:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr528;
		case 32: goto tr528;
		case 40: goto tr529;
		case 41: goto tr530;
		case 3082: goto tr531;
		case 3131: goto tr532;
		case 3338: goto tr533;
		case 3387: goto tr533;
		case 3594: goto tr534;
		case 3643: goto tr535;
	}
	goto tr61;
tr17:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st128;
tr54:
	{
		s->r_class = s->default_class;
	}
	goto st128;
tr720:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st128;
st128:
	if ( ++p == pe )
		goto _test_eof128;
case 128:
	switch( (*p) ) {
		case 80: goto st129;
		case 82: goto st130;
		case 84: goto st134;
		case 112: goto st129;
		case 114: goto st130;
		case 116: goto st134;
	}
	goto tr38;
st129:
	if ( ++p == pe )
		goto _test_eof129;
case 129:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr539;
		case 32: goto tr539;
		case 40: goto tr540;
		case 41: goto tr541;
		case 3082: goto tr542;
		case 3131: goto tr543;
		case 3338: goto tr544;
		case 3387: goto tr544;
		case 3594: goto tr545;
		case 3643: goto tr546;
	}
	goto tr61;
st130:
	if ( ++p == pe )
		goto _test_eof130;
case 130:
	switch( (*p) ) {
		case 83: goto st131;
		case 115: goto st131;
	}
	goto tr38;
st131:
	if ( ++p == pe )
		goto _test_eof131;
case 131:
	switch( (*p) ) {
		case 73: goto st132;
		case 105: goto st132;
	}
	goto tr38;
st132:
	if ( ++p == pe )
		goto _test_eof132;
case 132:
	switch( (*p) ) {
		case 71: goto st133;
		case 103: goto st133;
	}
	goto tr38;
st133:
	if ( ++p == pe )
		goto _test_eof133;
case 133:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr550;
		case 32: goto tr550;
		case 40: goto tr551;
		case 41: goto tr552;
		case 3082: goto tr553;
		case 3131: goto tr554;
		case 3338: goto tr555;
		case 3387: goto tr555;
		case 3594: goto tr556;
		case 3643: goto tr557;
	}
	goto tr61;
st134:
	if ( ++p == pe )
		goto _test_eof134;
case 134:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr558;
		case 32: goto tr558;
		case 40: goto tr559;
		case 41: goto tr560;
		case 3082: goto tr561;
		case 3131: goto tr562;
		case 3338: goto tr563;
		case 3387: goto tr563;
		case 3594: goto tr564;
		case 3643: goto tr565;
	}
	goto tr61;
tr18:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st135;
tr55:
	{
		s->r_class = s->default_class;
	}
	goto st135;
tr721:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st135;
st135:
	if ( ++p == pe )
		goto _test_eof135;
case 135:
	switch( (*p) ) {
		case 77: goto st136;
		case 79: goto st141;
		case 80: goto st143;
		case 82: goto st145;
		case 83: goto st147;
		case 86: goto st151;
		case 109: goto st136;
		case 111: goto st141;
		case 112: goto st143;
		case 114: goto st145;
		case 115: goto st147;
		case 118: goto st151;
	}
	goto tr38;
st136:
	if ( ++p == pe )
		goto _test_eof136;
case 136:
	switch( (*p) ) {
		case 73: goto st137;
		case 105: goto st137;
	}
	goto tr38;
st137:
	if ( ++p == pe )
		goto _test_eof137;
case 137:
	switch( (*p) ) {
		case 77: goto st138;
		case 109: goto st138;
	}
	goto tr38;
st138:
	if ( ++p == pe )
		goto _test_eof138;
case 138:
	switch( (*p) ) {
		case 69: goto st139;
		case 101: goto st139;
	}
	goto tr38;
st139:
	if ( ++p == pe )
		goto _test_eof139;
case 139:
	switch( (*p) ) {
		case 65: goto st140;
		case 97: goto st140;
	}
	goto tr38;
st140:
	if ( ++p == pe )
		goto _test_eof140;
case 140:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr576;
		case 32: goto tr576;
		case 40: goto tr577;
		case 41: goto tr578;
		case 3082: goto tr579;
		case 3131: goto tr580;
		case 3338: goto tr581;
		case 3387: goto tr581;
		case 3594: goto tr582;
		case 3643: goto tr583;
	}
	goto tr61;
st141:
	if ( ++p == pe )
		goto _test_eof141;
case 141:
	switch( (*p) ) {
		case 65: goto st142;
		case 97: goto st142;
	}
	goto tr38;
st142:
	if ( ++p == pe )
		goto _test_eof142;
case 142:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr585;
		case 32: goto tr585;
		case 40: goto tr586;
		case 41: goto tr587;
		case 3082: goto tr588;
		case 3131: goto tr589;
		case 3338: goto tr590;
		case 3387: goto tr590;
		case 3594: goto tr591;
		case 3643: goto tr592;
	}
	goto tr61;
st143:
	if ( ++p == pe )
		goto _test_eof143;
case 143:
	switch( (*p) ) {
		case 70: goto st144;
		case 102: goto st144;
	}
	goto tr38;
st144:
	if ( ++p == pe )
		goto _test_eof144;
case 144:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr594;
		case 32: goto tr594;
		case 40: goto tr595;
		case 41: goto tr596;
		case 3082: goto tr597;
		case 3131: goto tr598;
		case 3338: goto tr599;
		case 3387: goto tr599;
		case 3594: goto tr600;
		case 3643: goto tr601;
	}
	goto tr61;
st145:
	if ( ++p == pe )
		goto _test_eof145;
case 145:
	switch( (*p) ) {
		case 86: goto st146;
		case 118: goto st146;
	}
	goto tr38;
st146:
	if ( ++p == pe )
		goto _test_eof146;
case 146:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 3082: goto tr606;
		case 3131: goto tr607;
		case 3338: goto tr608;
		case 3387: goto tr608;
		case 3594: goto tr609;
		case 3643: goto tr610;
	}
	goto tr61;
st147:
	if ( ++p == pe )
		goto _test_eof147;
case 147:
	switch( (*p) ) {
		case 72: goto st148;
		case 104: goto st148;
	}
	goto tr38;
st148:
	if ( ++p == pe )
		goto _test_eof148;
case 148:
	switch( (*p) ) {
		case 70: goto st149;
		case 102: goto st149;
	}
	goto tr38;
st149:
	if ( ++p == pe )
		goto _test_eof149;
case 149:
	switch( (*p) ) {
		case 80: goto st150;
		case 112: goto st150;
	}
	goto tr38;
st150:
	if ( ++p == pe )
		goto _test_eof150;
case 150:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr614;
		case 32: goto tr614;
		case 40: goto tr615;
		case 41: goto tr616;
		case 3082: goto tr617;
		case 3131: goto tr618;
		case 3338: goto tr619;
		case 3387: goto tr619;
		case 3594: goto tr620;
		case 3643: goto tr621;
	}
	goto tr61;
st151:
	if ( ++p == pe )
		goto _test_eof151;
case 151:
	switch( (*p) ) {
		case 67: goto st152;
		case 99: goto st152;
	}
	goto tr38;
st152:
	if ( ++p == pe )
		goto _test_eof152;
case 152:
	switch( (*p) ) {
		case 66: goto st153;
		case 98: goto st153;
	}
	goto tr38;
st153:
	if ( ++p == pe )
		goto _test_eof153;
case 153:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr624;
		case 32: goto tr624;
		case 40: goto tr625;
		case 41: goto tr626;
		case 3082: goto tr627;
		case 3131: goto tr628;
		case 3338: goto tr629;
		case 3387: goto tr629;
		case 3594: goto tr630;
		case 3643: goto tr631;
	}
	goto tr61;
tr19:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st154;
tr56:
	{
		s->r_class = s->default_class;
	}
	goto st154;
tr722:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st154;
st154:
	if ( ++p == pe )
		goto _test_eof154;
case 154:
	switch( (*p) ) {
		case 76: goto st155;
		case 88: goto st158;
		case 89: goto st160;
		case 108: goto st155;
		case 120: goto st158;
		case 121: goto st160;
	}
	goto tr38;
st155:
	if ( ++p == pe )
		goto _test_eof155;
case 155:
	switch( (*p) ) {
		case 83: goto st156;
		case 115: goto st156;
	}
	goto tr38;
st156:
	if ( ++p == pe )
		goto _test_eof156;
case 156:
	switch( (*p) ) {
		case 65: goto st157;
		case 97: goto st157;
	}
	goto tr38;
st157:
	if ( ++p == pe )
		goto _test_eof157;
case 157:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr637;
		case 32: goto tr637;
		case 40: goto tr638;
		case 41: goto tr639;
		case 3082: goto tr640;
		case 3131: goto tr641;
		case 3338: goto tr642;
		case 3387: goto tr642;
		case 3594: goto tr643;
		case 3643: goto tr644;
	}
	goto tr61;
st158:
	if ( ++p == pe )
		goto _test_eof158;
case 158:
	switch( (*p) ) {
		case 84: goto st159;
		case 116: goto st159;
	}
	goto tr38;
st159:
	if ( ++p == pe )
		goto _test_eof159;
case 159:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr646;
		case 32: goto tr646;
		case 40: goto tr647;
		case 41: goto tr648;
		case 3082: goto tr649;
		case 3131: goto tr650;
		case 3338: goto tr651;
		case 3387: goto tr651;
		case 3594: goto tr652;
		case 3643: goto tr653;
	}
	goto tr61;
st160:
	if ( ++p == pe )
		goto _test_eof160;
case 160:
	switch( (*p) ) {
		case 80: goto st161;
		case 112: goto st161;
	}
	goto tr38;
st161:
	if ( ++p == pe )
		goto _test_eof161;
case 161:
	switch( (*p) ) {
		case 69: goto st162;
		case 101: goto st162;
	}
	goto tr38;
st162:
	if ( ++p == pe )
		goto _test_eof162;
case 162:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr657;
	goto tr656;
tr657:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st163;
tr662:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st163;
st163:
	if ( ++p == pe )
		goto _test_eof163;
case 163:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr659;
		case 32: goto tr659;
		case 40: goto tr660;
		case 41: goto tr661;
		case 3082: goto tr663;
		case 3131: goto tr664;
		case 3338: goto tr665;
		case 3387: goto tr665;
		case 3594: goto tr666;
		case 3643: goto tr667;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr662;
	goto tr658;
tr20:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st164;
tr57:
	{
		s->r_class = s->default_class;
	}
	goto st164;
tr723:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st164;
st164:
	if ( ++p == pe )
		goto _test_eof164;
case 164:
	switch( (*p) ) {
		case 82: goto st165;
		case 114: goto st165;
	}
	goto tr38;
st165:
	if ( ++p == pe )
		goto _test_eof165;
case 165:
	switch( (*p) ) {
		case 73: goto st166;
		case 105: goto st166;
	}
	goto tr38;
st166:
	if ( ++p == pe )
		goto _test_eof166;
case 166:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr670;
		case 32: goto tr670;
		case 40: goto tr671;
		case 41: goto tr672;
		case 3082: goto tr673;
		case 3131: goto tr674;
		case 3338: goto tr675;
		case 3387: goto tr675;
		case 3594: goto tr676;
		case 3643: goto tr677;
	}
	goto tr61;
tr21:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st167;
tr58:
	{
		s->r_class = s->default_class;
	}
	goto st167;
tr724:
	{
		s->r_ttl = s->default_ttl;
	}
	goto st167;
st167:
	if ( ++p == pe )
		goto _test_eof167;
case 167:
	switch( (*p) ) {
		case 79: goto st168;
		case 111: goto st168;
	}
	goto tr38;
st168:
	if ( ++p == pe )
		goto _test_eof168;
case 168:
	switch( (*p) ) {
		case 78: goto st169;
		case 110: goto st169;
	}
	goto tr38;
st169:
	if ( ++p == pe )
		goto _test_eof169;
case 169:
	switch( (*p) ) {
		case 69: goto st170;
		case 101: goto st170;
	}
	goto tr38;
st170:
	if ( ++p == pe )
		goto _test_eof170;
case 170:
	switch( (*p) ) {
		case 77: goto st171;
		case 109: goto st171;
	}
	goto tr38;
st171:
	if ( ++p == pe )
		goto _test_eof171;
case 171:
	switch( (*p) ) {
		case 68: goto st172;
		case 100: goto st172;
	}
	goto tr38;
st172:
	if ( ++p == pe )
		goto _test_eof172;
case 172:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr683;
		case 32: goto tr683;
		case 40: goto tr684;
		case 41: goto tr685;
		case 3082: goto tr686;
		case 3131: goto tr687;
		case 3338: goto tr688;
		case 3387: goto tr688;
		case 3594: goto tr689;
		case 3643: goto tr690;
	}
	goto tr61;
tr352:
	{
		s->buffer_length = 0;
	}
	goto st173;
tr737:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st173;
tr691:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st173;
tr330:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		s->buffer_length = 0;
	}
	goto st173;
tr750:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st173;
st173:
	if ( ++p == pe )
		goto _test_eof173;
case 173:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr692;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr691;
	goto tr38;
tr1002:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1398; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1398; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1398; goto _out;}
		}
	}
	goto st1398;
tr149:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1398; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1398; goto _out;}
		}
	}
	goto st1398;
tr155:
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1398; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1398; goto _out;}
		}
	}
	goto st1398;
st1398:
	if ( ++p == pe )
		goto _test_eof1398;
case 1398:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr4228;
		case 32: goto tr4228;
		case 36: goto st184;
		case 40: goto tr4229;
		case 41: goto tr4230;
		case 42: goto tr4210;
		case 65: goto tr4231;
		case 67: goto tr4232;
		case 68: goto tr4233;
		case 69: goto tr4234;
		case 72: goto tr4235;
		case 73: goto tr4236;
		case 75: goto tr4237;
		case 76: goto tr4238;
		case 77: goto tr4239;
		case 78: goto tr4240;
		case 79: goto tr4241;
		case 80: goto tr4242;
		case 82: goto tr4243;
		case 83: goto tr4244;
		case 84: goto tr4245;
		case 85: goto tr4246;
		case 90: goto tr4247;
		case 92: goto tr4210;
		case 95: goto tr4210;
		case 97: goto tr4231;
		case 99: goto tr4232;
		case 100: goto tr4233;
		case 101: goto tr4234;
		case 104: goto tr4235;
		case 105: goto tr4236;
		case 107: goto tr4237;
		case 108: goto tr4238;
		case 109: goto tr4239;
		case 110: goto tr4240;
		case 111: goto tr4241;
		case 112: goto tr4242;
		case 114: goto tr4243;
		case 115: goto tr4244;
		case 116: goto tr4245;
		case 117: goto tr4246;
		case 122: goto tr4247;
		case 1802: goto tr4211;
		case 1851: goto tr4212;
		case 2058: goto tr4248;
		case 2107: goto tr4249;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr4210;
	} else if ( _widec > 89 ) {
		if ( 98 <= _widec && _widec <= 121 )
			goto tr4210;
	} else
		goto tr4210;
	goto tr4227;
tr694:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st174;
tr695:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st174;
tr4228:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st174;
tr4229:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st174;
tr4230:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st174;
st174:
	if ( ++p == pe )
		goto _test_eof174;
case 174:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st174;
		case 32: goto st174;
		case 40: goto tr694;
		case 41: goto tr695;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 1802: goto tr22;
		case 1851: goto tr23;
		case 2058: goto tr696;
		case 2107: goto tr697;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr10:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	goto st175;
st175:
	if ( ++p == pe )
		goto _test_eof175;
case 175:
	switch( (*p) ) {
		case 78: goto st176;
		case 80: goto st74;
		case 110: goto st176;
		case 112: goto st74;
	}
	goto tr38;
st176:
	if ( ++p == pe )
		goto _test_eof176;
case 176:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr699;
		case 32: goto tr699;
		case 40: goto tr700;
		case 41: goto tr701;
		case 2058: goto tr702;
		case 2107: goto tr703;
	}
	goto tr38;
tr705:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st177;
tr706:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st177;
tr725:
	{
		s->line_counter++;
	}
	goto st177;
tr752:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 177; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st177;
tr699:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	goto st177;
tr700:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st177;
tr701:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st177;
tr702:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		s->line_counter++;
	}
	goto st177;
st177:
	if ( ++p == pe )
		goto _test_eof177;
case 177:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st177;
		case 32: goto st177;
		case 40: goto tr705;
		case 41: goto tr706;
		case 65: goto tr708;
		case 67: goto tr709;
		case 68: goto tr710;
		case 69: goto tr711;
		case 72: goto tr712;
		case 73: goto tr713;
		case 75: goto tr714;
		case 76: goto tr715;
		case 77: goto tr716;
		case 78: goto tr717;
		case 79: goto tr718;
		case 80: goto tr719;
		case 82: goto tr720;
		case 83: goto tr721;
		case 84: goto tr722;
		case 85: goto tr723;
		case 90: goto tr724;
		case 97: goto tr708;
		case 99: goto tr709;
		case 100: goto tr710;
		case 101: goto tr711;
		case 104: goto tr712;
		case 105: goto tr713;
		case 107: goto tr714;
		case 108: goto tr715;
		case 109: goto tr716;
		case 110: goto tr717;
		case 111: goto tr718;
		case 112: goto tr719;
		case 114: goto tr720;
		case 115: goto tr721;
		case 116: goto tr722;
		case 117: goto tr723;
		case 122: goto tr724;
		case 2058: goto tr725;
		case 2107: goto tr726;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr707;
	goto tr656;
tr707:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st178;
tr730:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st178;
st178:
	if ( ++p == pe )
		goto _test_eof178;
case 178:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr727;
		case 32: goto tr727;
		case 40: goto tr728;
		case 41: goto tr729;
		case 68: goto tr731;
		case 72: goto tr732;
		case 77: goto tr733;
		case 83: goto st179;
		case 87: goto tr735;
		case 100: goto tr731;
		case 104: goto tr732;
		case 109: goto tr733;
		case 115: goto st179;
		case 119: goto tr735;
		case 2058: goto tr736;
		case 2107: goto tr737;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr730;
	goto tr26;
tr731:
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st179;
tr732:
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st179;
tr733:
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st179;
tr735:
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st179;
st179:
	if ( ++p == pe )
		goto _test_eof179;
case 179:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr727;
		case 32: goto tr727;
		case 40: goto tr728;
		case 41: goto tr729;
		case 2058: goto tr736;
		case 2107: goto tr737;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr738;
	goto tr26;
tr739:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st180;
tr738:
	{
		s->number64_tmp = s->number64;
	}
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st180;
tr748:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->number64_tmp = s->number64;
	}
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st180;
st180:
	if ( ++p == pe )
		goto _test_eof180;
case 180:
	switch( (*p) ) {
		case 68: goto tr740;
		case 72: goto tr741;
		case 77: goto tr742;
		case 83: goto st181;
		case 87: goto tr744;
		case 100: goto tr740;
		case 104: goto tr741;
		case 109: goto tr742;
		case 115: goto st181;
		case 119: goto tr744;
	}
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr739;
	goto tr26;
tr740:
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st181;
tr741:
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st181;
tr742:
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st181;
tr744:
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st181;
st181:
	if ( ++p == pe )
		goto _test_eof181;
case 181:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr745;
		case 32: goto tr745;
		case 40: goto tr746;
		case 41: goto tr747;
		case 2058: goto tr749;
		case 2107: goto tr750;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr748;
	goto tr26;
tr726:
	{
		s->buffer_length = 0;
	}
	goto st182;
tr751:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st182;
tr703:
	{
		s->r_class = KNOT_CLASS_IN;
	}
	{
		s->buffer_length = 0;
	}
	goto st182;
st182:
	if ( ++p == pe )
		goto _test_eof182;
case 182:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr752;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr751;
	goto tr38;
tr23:
	{
		s->buffer_length = 0;
	}
	goto st183;
tr4212:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st183;
tr753:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st183;
tr4276:
	{
		NOERR;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st183;
st183:
	if ( ++p == pe )
		goto _test_eof183;
case 183:
	if ( (*p) == 10 )
		goto tr754;
	goto tr753;
tr696:
	{
		s->line_counter++;
	}
	goto st1399;
tr1000:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1399; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st1399;
tr889:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1399; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1399; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1399; goto _out;}
		}
	}
	goto st1399;
tr886:
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1399; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1399; goto _out;}
		}
	}
	goto st1399;
tr881:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1399; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1399; goto _out;}
		}
	}
	goto st1399;
tr4248:
	{
		s->line_counter++;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1399;
st1399:
	if ( ++p == pe )
		goto _test_eof1399;
case 1399:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr4228;
		case 32: goto tr4228;
		case 36: goto st184;
		case 40: goto tr4229;
		case 41: goto tr4230;
		case 42: goto tr4210;
		case 65: goto tr4252;
		case 67: goto tr4253;
		case 68: goto tr4254;
		case 69: goto tr4255;
		case 72: goto tr4256;
		case 73: goto tr4257;
		case 75: goto tr4258;
		case 76: goto tr4259;
		case 77: goto tr4260;
		case 78: goto tr4261;
		case 79: goto tr4262;
		case 80: goto tr4263;
		case 82: goto tr4264;
		case 83: goto tr4265;
		case 84: goto tr4266;
		case 85: goto tr4267;
		case 90: goto tr4268;
		case 92: goto tr4210;
		case 95: goto tr4210;
		case 97: goto tr4252;
		case 99: goto tr4253;
		case 100: goto tr4254;
		case 101: goto tr4255;
		case 104: goto tr4256;
		case 105: goto tr4257;
		case 107: goto tr4258;
		case 108: goto tr4259;
		case 109: goto tr4260;
		case 110: goto tr4261;
		case 111: goto tr4262;
		case 112: goto tr4263;
		case 114: goto tr4264;
		case 115: goto tr4265;
		case 116: goto tr4266;
		case 117: goto tr4267;
		case 122: goto tr4268;
		case 1802: goto tr4211;
		case 1851: goto tr4212;
		case 2058: goto tr4248;
		case 2107: goto tr4249;
	}
	if ( _widec < 48 ) {
		if ( 45 <= _widec && _widec <= 47 )
			goto tr4210;
	} else if ( _widec > 57 ) {
		if ( _widec > 89 ) {
			if ( 98 <= _widec && _widec <= 121 )
				goto tr4210;
		} else if ( _widec >= 64 )
			goto tr4210;
	} else
		goto tr4251;
	goto tr4250;
tr4271:
	{
		NOERR;
	}
	goto st184;
st184:
	if ( ++p == pe )
		goto _test_eof184;
case 184:
	switch( (*p) ) {
		case 73: goto tr756;
		case 79: goto tr757;
		case 84: goto tr758;
		case 105: goto tr756;
		case 111: goto tr757;
		case 116: goto tr758;
	}
	goto tr755;
tr756:
	{
		ERR(ZS_OK);
	}
	goto st185;
st185:
	if ( ++p == pe )
		goto _test_eof185;
case 185:
	switch( (*p) ) {
		case 78: goto st186;
		case 110: goto st186;
	}
	goto tr755;
st186:
	if ( ++p == pe )
		goto _test_eof186;
case 186:
	switch( (*p) ) {
		case 67: goto st187;
		case 99: goto st187;
	}
	goto tr755;
st187:
	if ( ++p == pe )
		goto _test_eof187;
case 187:
	switch( (*p) ) {
		case 76: goto st188;
		case 108: goto st188;
	}
	goto tr755;
st188:
	if ( ++p == pe )
		goto _test_eof188;
case 188:
	switch( (*p) ) {
		case 85: goto st189;
		case 117: goto st189;
	}
	goto tr755;
st189:
	if ( ++p == pe )
		goto _test_eof189;
case 189:
	switch( (*p) ) {
		case 68: goto st190;
		case 100: goto st190;
	}
	goto tr755;
st190:
	if ( ++p == pe )
		goto _test_eof190;
case 190:
	switch( (*p) ) {
		case 69: goto st191;
		case 101: goto st191;
	}
	goto tr755;
st191:
	if ( ++p == pe )
		goto _test_eof191;
case 191:
	switch( (*p) ) {
		case 32: goto tr765;
		case 59: goto tr765;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr765;
	} else if ( (*p) >= 9 )
		goto tr765;
	goto tr755;
tr765:
	{ p--; {stack[top++] = 1400;goto st352;} }
	goto st1400;
tr867:
	{ p--; {stack[top++] = 1400;goto st340;} }
	goto st1400;
tr870:
	{ p--; {stack[top++] = 1400;goto st331;} }
	goto st1400;
st1400:
	if ( ++p == pe )
		goto _test_eof1400;
case 1400:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr4270;
		case 32: goto tr4270;
		case 36: goto tr4271;
		case 40: goto tr4272;
		case 41: goto tr4273;
		case 42: goto tr4274;
		case 92: goto tr4274;
		case 95: goto tr4274;
		case 1802: goto tr4275;
		case 1851: goto tr4276;
		case 2058: goto tr4277;
		case 2107: goto tr4278;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr4274;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr4274;
	} else
		goto tr4274;
	goto tr4269;
tr4210:
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 192;goto st309;} }
	goto st192;
tr4274:
	{
		NOERR;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 192;goto st309;} }
	goto st192;
st192:
	if ( ++p == pe )
		goto _test_eof192;
case 192:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr766;
tr773:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st193;
tr774:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st193;
tr775:
	{
		s->line_counter++;
	}
	goto st193;
tr779:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 193; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st193;
tr767:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	goto st193;
tr768:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st193;
tr769:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st193;
tr770:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	goto st193;
st193:
	if ( ++p == pe )
		goto _test_eof193;
case 193:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st193;
		case 32: goto st193;
		case 40: goto tr773;
		case 41: goto tr774;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 2058: goto tr775;
		case 2107: goto tr776;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr656;
tr776:
	{
		s->buffer_length = 0;
	}
	goto st194;
tr777:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st194;
tr771:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	goto st194;
st194:
	if ( ++p == pe )
		goto _test_eof194;
case 194:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr779;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr777;
	goto st0;
tr24:
	{
		s->line_counter++;
	}
	goto st1401;
tr998:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1401; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st1401;
tr844:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1401; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	goto st1401;
tr841:
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	goto st1401;
tr875:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	goto st1401;
tr912:
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	goto st1401;
tr915:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1401; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	goto st1401;
tr4213:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1401;
tr4281:
	{
		s->line_counter++;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1401;
tr4277:
	{
		NOERR;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1401;
tr4286:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st1401;
tr4312:
	{
		s->line_counter++;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1401; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1401; goto _out;}
		}
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st1401;
st1401:
	if ( ++p == pe )
		goto _test_eof1401;
case 1401:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr4206;
		case 32: goto tr4206;
		case 36: goto st184;
		case 40: goto tr4279;
		case 41: goto tr4280;
		case 42: goto tr4210;
		case 65: goto tr4252;
		case 67: goto tr4253;
		case 68: goto tr4254;
		case 69: goto tr4255;
		case 72: goto tr4256;
		case 73: goto tr4257;
		case 75: goto tr4258;
		case 76: goto tr4259;
		case 77: goto tr4260;
		case 78: goto tr4261;
		case 79: goto tr4262;
		case 80: goto tr4263;
		case 82: goto tr4264;
		case 83: goto tr4265;
		case 84: goto tr4266;
		case 85: goto tr4267;
		case 90: goto tr4268;
		case 92: goto tr4210;
		case 95: goto tr4210;
		case 97: goto tr4252;
		case 99: goto tr4253;
		case 100: goto tr4254;
		case 101: goto tr4255;
		case 104: goto tr4256;
		case 105: goto tr4257;
		case 107: goto tr4258;
		case 108: goto tr4259;
		case 109: goto tr4260;
		case 110: goto tr4261;
		case 111: goto tr4262;
		case 112: goto tr4263;
		case 114: goto tr4264;
		case 115: goto tr4265;
		case 116: goto tr4266;
		case 117: goto tr4267;
		case 122: goto tr4268;
		case 1802: goto tr4211;
		case 1851: goto tr4212;
		case 2058: goto tr4281;
		case 2107: goto tr4282;
	}
	if ( _widec < 48 ) {
		if ( 45 <= _widec && _widec <= 47 )
			goto tr4210;
	} else if ( _widec > 57 ) {
		if ( _widec > 89 ) {
			if ( 98 <= _widec && _widec <= 121 )
				goto tr4210;
		} else if ( _widec >= 64 )
			goto tr4210;
	} else
		goto tr4251;
	goto tr4250;
tr4251:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 195;goto st309;} }
	goto st195;
st195:
	if ( ++p == pe )
		goto _test_eof195;
case 195:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr781;
		case 32: goto tr781;
		case 40: goto tr782;
		case 41: goto tr783;
		case 68: goto tr31;
		case 72: goto tr32;
		case 77: goto tr33;
		case 83: goto st198;
		case 87: goto tr35;
		case 100: goto tr31;
		case 104: goto tr32;
		case 109: goto tr33;
		case 115: goto st198;
		case 119: goto tr35;
		case 2058: goto tr784;
		case 2107: goto tr785;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr30;
	goto tr780;
tr787:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st196;
tr788:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st196;
tr789:
	{
		s->line_counter++;
	}
	goto st196;
tr792:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 196; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st196;
tr781:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st196;
tr782:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st196;
tr783:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st196;
tr784:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st196;
st196:
	if ( ++p == pe )
		goto _test_eof196;
case 196:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st196;
		case 32: goto st196;
		case 40: goto tr787;
		case 41: goto tr788;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 2058: goto tr789;
		case 2107: goto tr790;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr656;
tr790:
	{
		s->buffer_length = 0;
	}
	goto st197;
tr791:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st197;
tr785:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st197;
st197:
	if ( ++p == pe )
		goto _test_eof197;
case 197:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr792;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr791;
	goto tr38;
tr31:
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st198;
tr32:
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st198;
tr33:
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st198;
tr35:
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st198;
st198:
	if ( ++p == pe )
		goto _test_eof198;
case 198:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr27;
		case 32: goto tr27;
		case 40: goto tr28;
		case 41: goto tr29;
		case 2058: goto tr36;
		case 2107: goto tr37;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr793;
	goto tr26;
tr794:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st199;
tr793:
	{
		s->number64_tmp = s->number64;
	}
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st199;
tr803:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->number64_tmp = s->number64;
	}
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st199;
st199:
	if ( ++p == pe )
		goto _test_eof199;
case 199:
	switch( (*p) ) {
		case 68: goto tr795;
		case 72: goto tr796;
		case 77: goto tr797;
		case 83: goto st200;
		case 87: goto tr799;
		case 100: goto tr795;
		case 104: goto tr796;
		case 109: goto tr797;
		case 115: goto st200;
		case 119: goto tr799;
	}
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr794;
	goto tr26;
tr795:
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st200;
tr796:
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st200;
tr797:
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st200;
tr799:
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st200;
st200:
	if ( ++p == pe )
		goto _test_eof200;
case 200:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr800;
		case 32: goto tr800;
		case 40: goto tr801;
		case 41: goto tr802;
		case 2058: goto tr804;
		case 2107: goto tr805;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr803;
	goto tr26;
tr60:
	{
		s->buffer_length = 0;
	}
	goto st201;
tr37:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st201;
tr806:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st201;
tr805:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st201;
st201:
	if ( ++p == pe )
		goto _test_eof201;
case 201:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr807;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr806;
	goto tr38;
tr4231:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 202;goto st309;} }
	goto st202;
tr4252:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 202;goto st309;} }
	goto st202;
st202:
	if ( ++p == pe )
		goto _test_eof202;
case 202:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr809;
		case 32: goto tr809;
		case 40: goto tr810;
		case 41: goto tr811;
		case 65: goto st266;
		case 70: goto st269;
		case 80: goto st273;
		case 97: goto st266;
		case 102: goto st269;
		case 112: goto st273;
		case 3082: goto tr812;
		case 3131: goto tr813;
		case 3338: goto tr70;
		case 3387: goto tr70;
		case 3594: goto tr814;
		case 3643: goto tr815;
	}
	goto tr808;
tr817:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st203;
tr818:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st203;
tr819:
	{
		s->line_counter++;
	}
	goto st203;
tr984:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 203; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st203;
tr809:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	goto st203;
tr810:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	goto st203;
tr811:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	goto st203;
tr812:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	goto st203;
st203:
	if ( ++p == pe )
		goto _test_eof203;
case 203:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st203;
		case 32: goto st203;
		case 40: goto tr817;
		case 41: goto tr818;
		case 58: goto tr73;
		case 65: goto tr122;
		case 67: goto tr123;
		case 68: goto tr124;
		case 69: goto tr125;
		case 72: goto tr126;
		case 73: goto tr127;
		case 75: goto tr128;
		case 76: goto tr129;
		case 77: goto tr130;
		case 78: goto tr131;
		case 79: goto tr132;
		case 80: goto tr133;
		case 82: goto tr134;
		case 83: goto tr135;
		case 84: goto tr136;
		case 85: goto tr137;
		case 90: goto tr138;
		case 92: goto tr78;
		case 97: goto tr122;
		case 99: goto tr123;
		case 100: goto tr124;
		case 101: goto tr125;
		case 104: goto tr126;
		case 105: goto tr127;
		case 107: goto tr128;
		case 108: goto tr129;
		case 109: goto tr130;
		case 110: goto tr131;
		case 111: goto tr132;
		case 112: goto tr133;
		case 114: goto tr134;
		case 115: goto tr135;
		case 116: goto tr136;
		case 117: goto tr137;
		case 122: goto tr138;
		case 3082: goto tr819;
		case 3131: goto tr820;
		case 3338: goto tr73;
		case 3387: goto tr73;
		case 3594: goto tr821;
		case 3643: goto tr822;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr73;
		} else if ( _widec >= 48 )
			goto tr121;
	} else
		goto tr73;
	goto tr658;
tr122:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 204;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 204;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 204;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 204;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 204;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 204;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 204;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 204;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 204;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 204;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 204;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 204;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 204;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 204;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 204;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 204;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 204;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 204;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 204;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 204;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 204;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 204;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 204;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 204;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 204;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 204;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 204;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 204;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 204;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 204;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 204;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 204;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st204;
st204:
	if ( ++p == pe )
		goto _test_eof204;
case 204:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr824;
		case 32: goto tr824;
		case 40: goto tr825;
		case 41: goto tr826;
		case 65: goto st266;
		case 70: goto st269;
		case 80: goto st273;
		case 97: goto st266;
		case 102: goto st269;
		case 112: goto st273;
		case 2826: goto tr87;
		case 2875: goto tr88;
		case 3082: goto tr827;
		case 3131: goto tr828;
		case 3338: goto tr829;
		case 3387: goto tr830;
		case 3594: goto tr831;
		case 3643: goto tr832;
	}
	goto tr823;
tr834:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st205;
tr835:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st205;
tr824:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st205;
tr825:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st205;
tr826:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st205;
st205:
	if ( ++p == pe )
		goto _test_eof205;
case 205:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st205;
		case 32: goto st205;
		case 40: goto tr834;
		case 41: goto tr835;
		case 92: goto tr78;
		case 2826: goto tr93;
		case 2875: goto tr94;
		case 3082: goto tr836;
		case 3131: goto tr111;
		case 3338: goto tr112;
		case 3387: goto tr113;
		case 3594: goto tr837;
		case 3643: goto tr114;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr73;
	} else
		goto tr73;
	goto tr83;
tr105:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1402;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1402;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1402;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1402;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1402;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1402;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1402;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1402;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1402;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1402;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1402;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1402;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1402;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1402;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1402;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1402;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1402;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1402;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1402;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1402;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1402;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1402;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1402;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1402;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1402;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1402;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1402;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1402;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1402;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1402;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1402;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1402;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1402; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1402; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1402;
tr112:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1402;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1402;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1402;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1402;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1402;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1402;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1402;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1402;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1402;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1402;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1402;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1402;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1402;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1402;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1402;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1402;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1402;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1402;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1402;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1402;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1402;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1402;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1402;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1402;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1402;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1402;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1402;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1402;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1402;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1402;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1402;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1402;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1402; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1402; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1402;
tr141:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1402;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1402;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1402;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1402;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1402;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1402;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1402;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1402;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1402;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1402;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1402;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1402;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1402;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1402;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1402;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1402;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1402;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1402;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1402;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1402;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1402;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1402;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1402;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1402;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1402;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1402;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1402;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1402;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1402;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1402;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1402;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1402;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1402;
tr829:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1402;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1402;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1402;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1402;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1402;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1402;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1402;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1402;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1402;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1402;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1402;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1402;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1402;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1402;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1402;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1402;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1402;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1402;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1402;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1402;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1402;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1402;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1402;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1402;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1402;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1402;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1402;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1402;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1402;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1402;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1402;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1402;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1402; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1402; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto st1402;
tr4223:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1402;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1402;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1402;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1402;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1402;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1402;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1402;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1402;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1402;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1402;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1402;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1402;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1402;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1402;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1402;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1402;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1402;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1402;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1402;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1402;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1402;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1402;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1402;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1402;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1402;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1402;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1402;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1402;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1402;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1402;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1402;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1402;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->line_counter++;
	}
	goto st1402;
st1402:
	if ( ++p == pe )
		goto _test_eof1402;
case 1402:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr4283;
		case 32: goto tr4283;
		case 36: goto st184;
		case 40: goto tr4284;
		case 41: goto tr4285;
		case 42: goto tr4210;
		case 92: goto tr4210;
		case 95: goto tr4210;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr4286;
		case 2107: goto tr4287;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr4210;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr4210;
	} else
		goto tr4210;
	goto tr871;
tr839:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st206;
tr840:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st206;
tr909:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st206;
tr910:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st206;
tr911:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st206;
tr872:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st206;
tr873:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st206;
tr874:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st206;
tr4283:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st206;
tr4284:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st206;
tr4285:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st206;
st206:
	if ( ++p == pe )
		goto _test_eof206;
case 206:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st206;
		case 32: goto st206;
		case 40: goto tr839;
		case 41: goto tr840;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 1802: goto tr93;
		case 1851: goto tr94;
		case 2058: goto tr841;
		case 2107: goto tr842;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr842:
	{
		s->buffer_length = 0;
	}
	goto st207;
tr843:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st207;
tr913:
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st207;
tr918:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st207;
tr922:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st207;
tr876:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st207;
tr4287:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st207;
st207:
	if ( ++p == pe )
		goto _test_eof207;
case 207:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr96;
		case 2058: goto tr844;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr843;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr89;
tr106:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 208;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 208;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 208;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 208;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 208;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 208;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 208;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 208;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 208;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 208;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 208;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 208;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 208;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 208;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 208;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 208;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 208;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 208;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 208;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 208;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 208;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 208;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 208;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 208;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 208;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 208;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 208;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 208;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 208;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 208;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 208;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 208;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st208;
tr113:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 208;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 208;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 208;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 208;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 208;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 208;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 208;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 208;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 208;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 208;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 208;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 208;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 208;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 208;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 208;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 208;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 208;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 208;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 208;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 208;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 208;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 208;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 208;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 208;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 208;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 208;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 208;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 208;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 208;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 208;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 208;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 208;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st208;
tr830:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 208;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 208;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 208;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 208;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 208;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 208;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 208;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 208;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 208;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 208;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 208;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 208;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 208;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 208;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 208;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 208;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 208;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 208;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 208;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 208;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 208;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 208;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 208;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 208;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 208;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 208;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 208;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 208;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 208;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 208;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 208;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 208;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st208;
st208:
	if ( ++p == pe )
		goto _test_eof208;
case 208:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr845;
		case 32: goto tr845;
		case 40: goto tr846;
		case 41: goto tr847;
		case 1802: goto tr848;
		case 1851: goto tr849;
		case 2058: goto tr848;
		case 2107: goto tr849;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr95;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr95;
	} else
		goto tr95;
	goto tr83;
tr850:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st209;
tr845:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st209;
tr846:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st209;
tr847:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st209;
tr851:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st209;
tr852:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st209;
st209:
	if ( ++p == pe )
		goto _test_eof209;
case 209:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr850;
		case 32: goto tr850;
		case 40: goto tr851;
		case 41: goto tr852;
		case 1802: goto tr96;
		case 1851: goto tr853;
		case 2058: goto tr96;
		case 2107: goto tr853;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr95;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr95;
	} else
		goto tr95;
	goto tr89;
tr831:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1403;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1403;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1403;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1403;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1403;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1403;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1403;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1403;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1403;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1403;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1403;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1403;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1403;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1403;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1403;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1403;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1403;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1403;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1403;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1403;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1403;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1403;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1403;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1403;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1403;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1403;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1403;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1403;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1403;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1403;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1403;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1403;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1403; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1403; goto _out;}
		}
	}
	goto st1403;
tr837:
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1403;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1403;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1403;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1403;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1403;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1403;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1403;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1403;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1403;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1403;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1403;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1403;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1403;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1403;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1403;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1403;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1403;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1403;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1403;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1403;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1403;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1403;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1403;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1403;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1403;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1403;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1403;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1403;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1403;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1403;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1403;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1403;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1403; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1403; goto _out;}
		}
	}
	goto st1403;
st1403:
	if ( ++p == pe )
		goto _test_eof1403;
case 1403:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr4288;
		case 32: goto tr4288;
		case 36: goto tr4216;
		case 40: goto tr4289;
		case 41: goto tr4290;
		case 42: goto tr4219;
		case 58: goto tr73;
		case 92: goto tr4220;
		case 95: goto tr4219;
		case 2826: goto tr87;
		case 2875: goto tr88;
		case 3082: goto tr4286;
		case 3131: goto tr4291;
		case 3338: goto tr105;
		case 3387: goto tr106;
		case 3594: goto tr4286;
		case 3643: goto tr4292;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr73;
		} else if ( _widec > 44 ) {
			if ( 45 <= _widec && _widec <= 57 )
				goto tr4219;
		} else
			goto tr73;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 90 )
				goto tr4219;
		} else if ( _widec > 96 ) {
			if ( _widec > 122 ) {
				if ( 123 <= _widec )
					goto tr73;
			} else if ( _widec >= 97 )
				goto tr4219;
		} else
			goto tr73;
	} else
		goto tr73;
	goto tr871;
tr855:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st210;
tr856:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st210;
tr985:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st210;
tr986:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st210;
tr987:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st210;
tr4288:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st210;
tr4289:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st210;
tr4290:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st210;
st210:
	if ( ++p == pe )
		goto _test_eof210;
case 210:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st210;
		case 32: goto st210;
		case 40: goto tr855;
		case 41: goto tr856;
		case 58: goto tr73;
		case 65: goto tr122;
		case 67: goto tr123;
		case 68: goto tr124;
		case 69: goto tr125;
		case 72: goto tr126;
		case 73: goto tr127;
		case 75: goto tr128;
		case 76: goto tr129;
		case 77: goto tr130;
		case 78: goto tr131;
		case 79: goto tr132;
		case 80: goto tr133;
		case 82: goto tr134;
		case 83: goto tr135;
		case 84: goto tr136;
		case 85: goto tr137;
		case 90: goto tr138;
		case 92: goto tr78;
		case 97: goto tr122;
		case 99: goto tr123;
		case 100: goto tr124;
		case 101: goto tr125;
		case 104: goto tr126;
		case 105: goto tr127;
		case 107: goto tr128;
		case 108: goto tr129;
		case 109: goto tr130;
		case 110: goto tr131;
		case 111: goto tr132;
		case 112: goto tr133;
		case 114: goto tr134;
		case 115: goto tr135;
		case 116: goto tr136;
		case 117: goto tr137;
		case 122: goto tr138;
		case 2826: goto tr93;
		case 2875: goto tr94;
		case 3082: goto tr841;
		case 3131: goto tr857;
		case 3338: goto tr112;
		case 3387: goto tr113;
		case 3594: goto tr841;
		case 3643: goto tr858;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr73;
		} else if ( _widec >= 48 )
			goto tr121;
	} else
		goto tr73;
	goto tr118;
tr123:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 211;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 211;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 211;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 211;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 211;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 211;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 211;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 211;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 211;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 211;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 211;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 211;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 211;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 211;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 211;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 211;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 211;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 211;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 211;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 211;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 211;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 211;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 211;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 211;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 211;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 211;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 211;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 211;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 211;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 211;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 211;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 211;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st211;
st211:
	if ( ++p == pe )
		goto _test_eof211;
case 211:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 65: goto st18;
		case 68: goto st22;
		case 69: goto st29;
		case 78: goto st32;
		case 83: goto st36;
		case 97: goto st18;
		case 100: goto st22;
		case 101: goto st29;
		case 110: goto st32;
		case 115: goto st36;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr124:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 212;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 212;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 212;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 212;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 212;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 212;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 212;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 212;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 212;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 212;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 212;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 212;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 212;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 212;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 212;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 212;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 212;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 212;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 212;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 212;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 212;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 212;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 212;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 212;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 212;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 212;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 212;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 212;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 212;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 212;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 212;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 212;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st212;
st212:
	if ( ++p == pe )
		goto _test_eof212;
case 212:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 72: goto st41;
		case 78: goto st45;
		case 83: goto st53;
		case 104: goto st41;
		case 110: goto st45;
		case 115: goto st53;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr125:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 213;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 213;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 213;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 213;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 213;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 213;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 213;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 213;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 213;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 213;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 213;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 213;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 213;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 213;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 213;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 213;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 213;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 213;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 213;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 213;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 213;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 213;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 213;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 213;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 213;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 213;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 213;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 213;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 213;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 213;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 213;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 213;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st213;
st213:
	if ( ++p == pe )
		goto _test_eof213;
case 213:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 85: goto st55;
		case 117: goto st55;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr126:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 214;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 214;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 214;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 214;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 214;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 214;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 214;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 214;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 214;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 214;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 214;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 214;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 214;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 214;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 214;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 214;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 214;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 214;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 214;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 214;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 214;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 214;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 214;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 214;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 214;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 214;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 214;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 214;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 214;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 214;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 214;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 214;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st214;
st214:
	if ( ++p == pe )
		goto _test_eof214;
case 214:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 73: goto st62;
		case 84: goto st66;
		case 105: goto st62;
		case 116: goto st66;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr127:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 215;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 215;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 215;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 215;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 215;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 215;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 215;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 215;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 215;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 215;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 215;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 215;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 215;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 215;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 215;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 215;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 215;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 215;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 215;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 215;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 215;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 215;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 215;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 215;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 215;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 215;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 215;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 215;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 215;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 215;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 215;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 215;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st215;
st215:
	if ( ++p == pe )
		goto _test_eof215;
case 215:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 78: goto st176;
		case 80: goto st74;
		case 110: goto st176;
		case 112: goto st74;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr128:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 216;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 216;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 216;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 216;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 216;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 216;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 216;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 216;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 216;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 216;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 216;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 216;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 216;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 216;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 216;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 216;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 216;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 216;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 216;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 216;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 216;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 216;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 216;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 216;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 216;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 216;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 216;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 216;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 216;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 216;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 216;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 216;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st216;
st216:
	if ( ++p == pe )
		goto _test_eof216;
case 216:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 69: goto st82;
		case 88: goto st84;
		case 101: goto st82;
		case 120: goto st84;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr129:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 217;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 217;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 217;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 217;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 217;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 217;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 217;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 217;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 217;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 217;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 217;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 217;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 217;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 217;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 217;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 217;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 217;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 217;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 217;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 217;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 217;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 217;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 217;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 217;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 217;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 217;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 217;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 217;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 217;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 217;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 217;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 217;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st217;
st217:
	if ( ++p == pe )
		goto _test_eof217;
case 217:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 51: goto st86;
		case 54: goto st88;
		case 79: goto st90;
		case 80: goto st92;
		case 111: goto st90;
		case 112: goto st92;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr130:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 218;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 218;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 218;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 218;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 218;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 218;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 218;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 218;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 218;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 218;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 218;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 218;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 218;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 218;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 218;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 218;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 218;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 218;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 218;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 218;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 218;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 218;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 218;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 218;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 218;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 218;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 218;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 218;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 218;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 218;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 218;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 218;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st218;
st218:
	if ( ++p == pe )
		goto _test_eof218;
case 218:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 73: goto st94;
		case 88: goto st98;
		case 105: goto st94;
		case 120: goto st98;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr131:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 219;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 219;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 219;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 219;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 219;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 219;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 219;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 219;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 219;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 219;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 219;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 219;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 219;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 219;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 219;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 219;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 219;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 219;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 219;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 219;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 219;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 219;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 219;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 219;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 219;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 219;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 219;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 219;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 219;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 219;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 219;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 219;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st219;
st219:
	if ( ++p == pe )
		goto _test_eof219;
case 219:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 65: goto st100;
		case 73: goto st104;
		case 83: goto st106;
		case 97: goto st100;
		case 105: goto st104;
		case 115: goto st106;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr132:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 220;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 220;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 220;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 220;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 220;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 220;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 220;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 220;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 220;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 220;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 220;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 220;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 220;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 220;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 220;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 220;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 220;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 220;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 220;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 220;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 220;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 220;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 220;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 220;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 220;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 220;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 220;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 220;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 220;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 220;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 220;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 220;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st220;
st220:
	if ( ++p == pe )
		goto _test_eof220;
case 220:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 80: goto st116;
		case 112: goto st116;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr133:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 221;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 221;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 221;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 221;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 221;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 221;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 221;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 221;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 221;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 221;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 221;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 221;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 221;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 221;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 221;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 221;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 221;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 221;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 221;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 221;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 221;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 221;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 221;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 221;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 221;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 221;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 221;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 221;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 221;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 221;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 221;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 221;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st221;
st221:
	if ( ++p == pe )
		goto _test_eof221;
case 221:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 84: goto st126;
		case 116: goto st126;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr134:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 222;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 222;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 222;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 222;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 222;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 222;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 222;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 222;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 222;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 222;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 222;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 222;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 222;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 222;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 222;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 222;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 222;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 222;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 222;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 222;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 222;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 222;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 222;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 222;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 222;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 222;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 222;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 222;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 222;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 222;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 222;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 222;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st222;
st222:
	if ( ++p == pe )
		goto _test_eof222;
case 222:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 80: goto st129;
		case 82: goto st130;
		case 84: goto st134;
		case 112: goto st129;
		case 114: goto st130;
		case 116: goto st134;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr135:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 223;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 223;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 223;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 223;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 223;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 223;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 223;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 223;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 223;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 223;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 223;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 223;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 223;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 223;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 223;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 223;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 223;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 223;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 223;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 223;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 223;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 223;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 223;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 223;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 223;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 223;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 223;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 223;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 223;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 223;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 223;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 223;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st223;
st223:
	if ( ++p == pe )
		goto _test_eof223;
case 223:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 77: goto st136;
		case 79: goto st141;
		case 80: goto st143;
		case 82: goto st145;
		case 83: goto st147;
		case 86: goto st151;
		case 109: goto st136;
		case 111: goto st141;
		case 112: goto st143;
		case 114: goto st145;
		case 115: goto st147;
		case 118: goto st151;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr136:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 224;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 224;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 224;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 224;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 224;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 224;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 224;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 224;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 224;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 224;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 224;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 224;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 224;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 224;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 224;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 224;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 224;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 224;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 224;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 224;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 224;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 224;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 224;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 224;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 224;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 224;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 224;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 224;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 224;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 224;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 224;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 224;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st224;
st224:
	if ( ++p == pe )
		goto _test_eof224;
case 224:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 76: goto st155;
		case 88: goto st158;
		case 89: goto st160;
		case 108: goto st155;
		case 120: goto st158;
		case 121: goto st160;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr137:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 225;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 225;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 225;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 225;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 225;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 225;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 225;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 225;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 225;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 225;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 225;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 225;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 225;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 225;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 225;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 225;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 225;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 225;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 225;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 225;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 225;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 225;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 225;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 225;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 225;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 225;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 225;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 225;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 225;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 225;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 225;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 225;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st225;
st225:
	if ( ++p == pe )
		goto _test_eof225;
case 225:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 82: goto st165;
		case 114: goto st165;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr138:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 226;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 226;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 226;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 226;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 226;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 226;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 226;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 226;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 226;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 226;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 226;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 226;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 226;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 226;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 226;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 226;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 226;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 226;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 226;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 226;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 226;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 226;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 226;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 226;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 226;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 226;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 226;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 226;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 226;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 226;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 226;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 226;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st226;
st226:
	if ( ++p == pe )
		goto _test_eof226;
case 226:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 79: goto st168;
		case 111: goto st168;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr823;
tr857:
	{
		s->buffer_length = 0;
	}
	goto st227;
tr859:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st227;
tr988:
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st227;
tr926:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st227;
tr930:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st227;
tr895:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st227;
tr4291:
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st227;
st227:
	if ( ++p == pe )
		goto _test_eof227;
case 227:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr96;
		case 2058: goto tr860;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr859;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr83;
tr139:
	{
		s->line_counter++;
	}
	goto st1404;
tr962:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1404; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st1404;
tr860:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1404; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1404; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1404; goto _out;}
		}
	}
	goto st1404;
tr901:
	{
		s->line_counter++;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1404; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1404; goto _out;}
		}
	}
	goto st1404;
tr894:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1404; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1404; goto _out;}
		}
	}
	goto st1404;
tr4221:
	{
		s->line_counter++;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1404;
st1404:
	if ( ++p == pe )
		goto _test_eof1404;
case 1404:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr4215;
		case 32: goto tr4215;
		case 36: goto tr4216;
		case 40: goto tr4217;
		case 41: goto tr4218;
		case 42: goto tr4219;
		case 58: goto tr73;
		case 65: goto tr4295;
		case 67: goto tr4296;
		case 68: goto tr4297;
		case 69: goto tr4298;
		case 72: goto tr4299;
		case 73: goto tr4300;
		case 75: goto tr4301;
		case 76: goto tr4302;
		case 77: goto tr4303;
		case 78: goto tr4304;
		case 79: goto tr4305;
		case 80: goto tr4306;
		case 82: goto tr4307;
		case 83: goto tr4308;
		case 84: goto tr4309;
		case 85: goto tr4310;
		case 90: goto tr4311;
		case 92: goto tr4220;
		case 95: goto tr4219;
		case 97: goto tr4295;
		case 99: goto tr4296;
		case 100: goto tr4297;
		case 101: goto tr4298;
		case 104: goto tr4299;
		case 105: goto tr4300;
		case 107: goto tr4301;
		case 108: goto tr4302;
		case 109: goto tr4303;
		case 110: goto tr4304;
		case 111: goto tr4305;
		case 112: goto tr4306;
		case 114: goto tr4307;
		case 115: goto tr4308;
		case 116: goto tr4309;
		case 117: goto tr4310;
		case 122: goto tr4311;
		case 2826: goto tr4211;
		case 2875: goto tr4212;
		case 3082: goto tr4221;
		case 3131: goto tr4222;
		case 3338: goto tr4223;
		case 3387: goto tr4224;
		case 3594: goto tr4225;
		case 3643: goto tr4226;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr73;
		} else if ( _widec > 44 ) {
			if ( _widec > 47 ) {
				if ( 48 <= _widec && _widec <= 57 )
					goto tr4294;
			} else if ( _widec >= 45 )
				goto tr4219;
		} else
			goto tr73;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 89 )
				goto tr4219;
		} else if ( _widec > 96 ) {
			if ( _widec > 121 ) {
				if ( 123 <= _widec )
					goto tr73;
			} else if ( _widec >= 98 )
				goto tr4219;
		} else
			goto tr73;
	} else
		goto tr73;
	goto tr4293;
tr4216:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 228;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 228;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 228;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 228;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 228;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 228;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 228;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 228;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 228;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 228;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 228;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 228;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 228;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 228;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 228;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 228;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 228;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 228;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 228;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 228;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 228;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 228;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 228;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 228;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 228;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 228;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 228;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 228;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 228;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 228;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 228;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 228;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st228;
st228:
	if ( ++p == pe )
		goto _test_eof228;
case 228:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr84;
		case 32: goto tr84;
		case 40: goto tr85;
		case 41: goto tr86;
		case 73: goto tr756;
		case 79: goto tr757;
		case 84: goto tr758;
		case 105: goto tr756;
		case 111: goto tr757;
		case 116: goto tr758;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr87;
		case 2107: goto tr88;
	}
	goto tr861;
tr757:
	{
		ERR(ZS_OK);
	}
	goto st229;
st229:
	if ( ++p == pe )
		goto _test_eof229;
case 229:
	switch( (*p) ) {
		case 82: goto st230;
		case 114: goto st230;
	}
	goto tr755;
st230:
	if ( ++p == pe )
		goto _test_eof230;
case 230:
	switch( (*p) ) {
		case 73: goto st231;
		case 105: goto st231;
	}
	goto tr755;
st231:
	if ( ++p == pe )
		goto _test_eof231;
case 231:
	switch( (*p) ) {
		case 71: goto st232;
		case 103: goto st232;
	}
	goto tr755;
st232:
	if ( ++p == pe )
		goto _test_eof232;
case 232:
	switch( (*p) ) {
		case 73: goto st233;
		case 105: goto st233;
	}
	goto tr755;
st233:
	if ( ++p == pe )
		goto _test_eof233;
case 233:
	switch( (*p) ) {
		case 78: goto st234;
		case 110: goto st234;
	}
	goto tr755;
st234:
	if ( ++p == pe )
		goto _test_eof234;
case 234:
	switch( (*p) ) {
		case 32: goto tr867;
		case 59: goto tr867;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr867;
	} else if ( (*p) >= 9 )
		goto tr867;
	goto tr755;
tr758:
	{
		ERR(ZS_OK);
	}
	goto st235;
st235:
	if ( ++p == pe )
		goto _test_eof235;
case 235:
	switch( (*p) ) {
		case 84: goto st236;
		case 116: goto st236;
	}
	goto tr755;
st236:
	if ( ++p == pe )
		goto _test_eof236;
case 236:
	switch( (*p) ) {
		case 76: goto st237;
		case 108: goto st237;
	}
	goto tr755;
st237:
	if ( ++p == pe )
		goto _test_eof237;
case 237:
	switch( (*p) ) {
		case 32: goto tr870;
		case 59: goto tr870;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr870;
	} else if ( (*p) >= 9 )
		goto tr870;
	goto tr755;
tr4219:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 238;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 238;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 238;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 238;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 238;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 238;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 238;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 238;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 238;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 238;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 238;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 238;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 238;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 238;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 238;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 238;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 238;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 238;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 238;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 238;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 238;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 238;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 238;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 238;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 238;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 238;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 238;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 238;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 238;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 238;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 238;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 238;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 238;goto st309;} }
	goto st238;
st238:
	if ( ++p == pe )
		goto _test_eof238;
case 238:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr871;
tr4294:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 239;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 239;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 239;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 239;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 239;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 239;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 239;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 239;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 239;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 239;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 239;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 239;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 239;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 239;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 239;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 239;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 239;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 239;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 239;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 239;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 239;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 239;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 239;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 239;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 239;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 239;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 239;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 239;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 239;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 239;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 239;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 239;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 239;goto st309;} }
	goto st239;
st239:
	if ( ++p == pe )
		goto _test_eof239;
case 239:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr878;
		case 32: goto tr878;
		case 40: goto tr879;
		case 41: goto tr880;
		case 68: goto tr31;
		case 72: goto tr32;
		case 77: goto tr33;
		case 83: goto st198;
		case 87: goto tr35;
		case 100: goto tr31;
		case 104: goto tr32;
		case 109: goto tr33;
		case 115: goto st198;
		case 119: goto tr35;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr881;
		case 2107: goto tr882;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr30;
	goto tr877;
tr884:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st240;
tr885:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st240;
tr878:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st240;
tr879:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st240;
tr880:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st240;
st240:
	if ( ++p == pe )
		goto _test_eof240;
case 240:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st240;
		case 32: goto st240;
		case 40: goto tr884;
		case 41: goto tr885;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 1802: goto tr93;
		case 1851: goto tr94;
		case 2058: goto tr886;
		case 2107: goto tr887;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr887:
	{
		s->buffer_length = 0;
	}
	goto st241;
tr888:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st241;
tr882:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st241;
st241:
	if ( ++p == pe )
		goto _test_eof241;
case 241:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr96;
		case 2058: goto tr889;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr888;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr151;
tr4295:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 242;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 242;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 242;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 242;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 242;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 242;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 242;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 242;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 242;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 242;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 242;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 242;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 242;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 242;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 242;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 242;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 242;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 242;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 242;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 242;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 242;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 242;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 242;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 242;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 242;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 242;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 242;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 242;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 242;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 242;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 242;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 242;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 242;goto st309;} }
	goto st242;
st242:
	if ( ++p == pe )
		goto _test_eof242;
case 242:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr891;
		case 32: goto tr891;
		case 40: goto tr892;
		case 41: goto tr893;
		case 65: goto st266;
		case 70: goto st269;
		case 80: goto st273;
		case 97: goto st266;
		case 102: goto st269;
		case 112: goto st273;
		case 2826: goto tr87;
		case 2875: goto tr88;
		case 3082: goto tr894;
		case 3131: goto tr895;
		case 3338: goto tr829;
		case 3387: goto tr830;
		case 3594: goto tr896;
		case 3643: goto tr897;
	}
	goto tr890;
tr899:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st243;
tr900:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st243;
tr891:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st243;
tr892:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st243;
tr893:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st243;
st243:
	if ( ++p == pe )
		goto _test_eof243;
case 243:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st243;
		case 32: goto st243;
		case 40: goto tr899;
		case 41: goto tr900;
		case 58: goto tr73;
		case 65: goto tr122;
		case 67: goto tr123;
		case 68: goto tr124;
		case 69: goto tr125;
		case 72: goto tr126;
		case 73: goto tr127;
		case 75: goto tr128;
		case 76: goto tr129;
		case 77: goto tr130;
		case 78: goto tr131;
		case 79: goto tr132;
		case 80: goto tr133;
		case 82: goto tr134;
		case 83: goto tr135;
		case 84: goto tr136;
		case 85: goto tr137;
		case 90: goto tr138;
		case 92: goto tr78;
		case 97: goto tr122;
		case 99: goto tr123;
		case 100: goto tr124;
		case 101: goto tr125;
		case 104: goto tr126;
		case 105: goto tr127;
		case 107: goto tr128;
		case 108: goto tr129;
		case 109: goto tr130;
		case 110: goto tr131;
		case 111: goto tr132;
		case 112: goto tr133;
		case 114: goto tr134;
		case 115: goto tr135;
		case 116: goto tr136;
		case 117: goto tr137;
		case 122: goto tr138;
		case 2826: goto tr93;
		case 2875: goto tr94;
		case 3082: goto tr901;
		case 3131: goto tr857;
		case 3338: goto tr112;
		case 3387: goto tr113;
		case 3594: goto tr902;
		case 3643: goto tr858;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr73;
		} else if ( _widec >= 48 )
			goto tr121;
	} else
		goto tr73;
	goto tr118;
tr143:
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1405;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1405;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1405;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1405;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1405;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1405;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1405;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1405;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1405;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1405;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1405;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1405;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1405;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1405;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1405;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1405;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1405;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1405;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1405;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1405;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1405;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1405;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1405;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1405;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1405;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1405;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1405;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1405;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1405;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1405;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1405;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1405;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st1405;
tr902:
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1405;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1405;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1405;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1405;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1405;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1405;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1405;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1405;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1405;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1405;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1405;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1405;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1405;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1405;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1405;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1405;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1405;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1405;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1405;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1405;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1405;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1405;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1405;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1405;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1405;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1405;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1405;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1405;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1405;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1405;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1405;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1405;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1405; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1405; goto _out;}
		}
	}
	goto st1405;
tr896:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1405;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1405;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1405;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1405;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1405;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1405;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1405;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1405;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1405;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1405;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1405;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1405;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1405;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1405;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1405;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1405;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1405;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1405;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1405;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1405;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1405;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1405;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1405;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1405;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1405;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1405;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1405;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1405;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1405;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1405;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1405;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1405;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZS_RDATA_OVERFLOW);
			p--; {goto st307;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->state = ZS_STATE_DATA;

		// Execute the record callback.
		if (s->process.automatic) {
			if (s->process.record != NULL) {
				s->process.record(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; cs = 1405; goto _out;}
				}
			}
		} else {
			// Return if external processing.
			p--; {p++; cs = 1405; goto _out;}
		}
	}
	goto st1405;
tr4225:
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1405;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1405;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1405;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1405;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1405;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1405;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1405;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1405;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1405;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1405;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1405;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1405;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1405;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 1405;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1405;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1405;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1405;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1405;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 1405;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 1405;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1405;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1405;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 1405;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 1405;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 1405;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1405;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1405;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1405;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1405;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 1405;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 1405;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 1405;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st1405;
st1405:
	if ( ++p == pe )
		goto _test_eof1405;
case 1405:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr4288;
		case 32: goto tr4288;
		case 36: goto tr4216;
		case 40: goto tr4289;
		case 41: goto tr4290;
		case 42: goto tr4219;
		case 58: goto tr73;
		case 65: goto tr4295;
		case 67: goto tr4296;
		case 68: goto tr4297;
		case 69: goto tr4298;
		case 72: goto tr4299;
		case 73: goto tr4300;
		case 75: goto tr4301;
		case 76: goto tr4302;
		case 77: goto tr4303;
		case 78: goto tr4304;
		case 79: goto tr4305;
		case 80: goto tr4306;
		case 82: goto tr4307;
		case 83: goto tr4308;
		case 84: goto tr4309;
		case 85: goto tr4310;
		case 90: goto tr4311;
		case 92: goto tr4220;
		case 95: goto tr4219;
		case 97: goto tr4295;
		case 99: goto tr4296;
		case 100: goto tr4297;
		case 101: goto tr4298;
		case 104: goto tr4299;
		case 105: goto tr4300;
		case 107: goto tr4301;
		case 108: goto tr4302;
		case 109: goto tr4303;
		case 110: goto tr4304;
		case 111: goto tr4305;
		case 112: goto tr4306;
		case 114: goto tr4307;
		case 115: goto tr4308;
		case 116: goto tr4309;
		case 117: goto tr4310;
		case 122: goto tr4311;
		case 2826: goto tr87;
		case 2875: goto tr88;
		case 3082: goto tr4312;
		case 3131: goto tr4291;
		case 3338: goto tr105;
		case 3387: goto tr106;
		case 3594: goto tr4312;
		case 3643: goto tr4292;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr73;
		} else if ( _widec > 44 ) {
			if ( _widec > 47 ) {
				if ( 48 <= _widec && _widec <= 57 )
					goto tr4294;
			} else if ( _widec >= 45 )
				goto tr4219;
		} else
			goto tr73;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 89 )
				goto tr4219;
		} else if ( _widec > 96 ) {
			if ( _widec > 121 ) {
				if ( 123 <= _widec )
					goto tr73;
			} else if ( _widec >= 98 )
				goto tr4219;
		} else
			goto tr73;
	} else
		goto tr73;
	goto tr4293;
tr4296:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 244;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 244;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 244;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 244;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 244;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 244;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 244;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 244;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 244;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 244;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 244;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 244;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 244;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 244;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 244;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 244;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 244;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 244;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 244;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 244;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 244;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 244;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 244;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 244;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 244;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 244;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 244;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 244;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 244;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 244;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 244;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 244;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 244;goto st309;} }
	goto st244;
st244:
	if ( ++p == pe )
		goto _test_eof244;
case 244:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 65: goto st18;
		case 68: goto st22;
		case 69: goto st29;
		case 78: goto st32;
		case 83: goto st36;
		case 97: goto st18;
		case 100: goto st22;
		case 101: goto st29;
		case 110: goto st32;
		case 115: goto st36;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4297:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 245;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 245;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 245;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 245;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 245;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 245;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 245;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 245;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 245;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 245;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 245;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 245;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 245;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 245;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 245;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 245;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 245;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 245;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 245;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 245;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 245;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 245;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 245;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 245;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 245;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 245;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 245;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 245;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 245;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 245;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 245;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 245;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 245;goto st309;} }
	goto st245;
st245:
	if ( ++p == pe )
		goto _test_eof245;
case 245:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 72: goto st41;
		case 78: goto st45;
		case 83: goto st53;
		case 104: goto st41;
		case 110: goto st45;
		case 115: goto st53;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4298:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 246;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 246;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 246;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 246;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 246;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 246;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 246;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 246;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 246;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 246;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 246;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 246;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 246;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 246;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 246;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 246;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 246;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 246;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 246;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 246;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 246;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 246;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 246;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 246;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 246;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 246;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 246;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 246;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 246;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 246;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 246;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 246;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 246;goto st309;} }
	goto st246;
st246:
	if ( ++p == pe )
		goto _test_eof246;
case 246:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 85: goto st55;
		case 117: goto st55;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4299:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 247;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 247;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 247;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 247;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 247;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 247;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 247;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 247;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 247;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 247;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 247;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 247;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 247;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 247;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 247;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 247;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 247;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 247;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 247;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 247;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 247;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 247;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 247;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 247;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 247;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 247;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 247;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 247;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 247;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 247;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 247;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 247;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 247;goto st309;} }
	goto st247;
st247:
	if ( ++p == pe )
		goto _test_eof247;
case 247:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 73: goto st62;
		case 84: goto st66;
		case 105: goto st62;
		case 116: goto st66;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4300:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 248;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 248;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 248;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 248;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 248;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 248;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 248;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 248;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 248;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 248;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 248;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 248;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 248;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 248;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 248;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 248;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 248;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 248;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 248;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 248;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 248;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 248;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 248;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 248;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 248;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 248;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 248;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 248;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 248;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 248;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 248;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 248;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 248;goto st309;} }
	goto st248;
st248:
	if ( ++p == pe )
		goto _test_eof248;
case 248:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 78: goto st176;
		case 80: goto st74;
		case 110: goto st176;
		case 112: goto st74;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4301:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 249;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 249;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 249;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 249;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 249;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 249;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 249;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 249;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 249;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 249;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 249;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 249;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 249;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 249;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 249;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 249;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 249;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 249;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 249;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 249;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 249;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 249;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 249;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 249;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 249;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 249;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 249;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 249;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 249;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 249;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 249;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 249;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 249;goto st309;} }
	goto st249;
st249:
	if ( ++p == pe )
		goto _test_eof249;
case 249:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 69: goto st82;
		case 88: goto st84;
		case 101: goto st82;
		case 120: goto st84;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4302:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 250;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 250;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 250;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 250;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 250;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 250;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 250;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 250;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 250;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 250;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 250;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 250;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 250;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 250;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 250;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 250;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 250;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 250;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 250;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 250;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 250;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 250;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 250;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 250;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 250;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 250;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 250;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 250;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 250;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 250;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 250;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 250;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 250;goto st309;} }
	goto st250;
st250:
	if ( ++p == pe )
		goto _test_eof250;
case 250:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 51: goto st86;
		case 54: goto st88;
		case 79: goto st90;
		case 80: goto st92;
		case 111: goto st90;
		case 112: goto st92;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4303:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 251;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 251;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 251;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 251;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 251;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 251;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 251;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 251;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 251;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 251;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 251;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 251;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 251;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 251;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 251;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 251;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 251;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 251;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 251;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 251;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 251;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 251;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 251;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 251;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 251;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 251;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 251;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 251;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 251;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 251;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 251;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 251;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 251;goto st309;} }
	goto st251;
st251:
	if ( ++p == pe )
		goto _test_eof251;
case 251:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 73: goto st94;
		case 88: goto st98;
		case 105: goto st94;
		case 120: goto st98;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4304:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 252;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 252;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 252;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 252;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 252;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 252;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 252;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 252;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 252;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 252;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 252;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 252;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 252;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 252;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 252;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 252;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 252;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 252;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 252;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 252;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 252;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 252;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 252;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 252;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 252;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 252;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 252;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 252;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 252;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 252;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 252;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 252;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 252;goto st309;} }
	goto st252;
st252:
	if ( ++p == pe )
		goto _test_eof252;
case 252:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 65: goto st100;
		case 73: goto st104;
		case 83: goto st106;
		case 97: goto st100;
		case 105: goto st104;
		case 115: goto st106;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4305:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 253;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 253;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 253;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 253;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 253;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 253;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 253;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 253;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 253;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 253;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 253;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 253;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 253;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 253;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 253;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 253;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 253;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 253;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 253;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 253;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 253;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 253;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 253;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 253;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 253;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 253;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 253;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 253;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 253;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 253;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 253;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 253;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 253;goto st309;} }
	goto st253;
st253:
	if ( ++p == pe )
		goto _test_eof253;
case 253:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 80: goto st116;
		case 112: goto st116;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4306:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 254;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 254;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 254;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 254;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 254;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 254;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 254;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 254;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 254;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 254;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 254;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 254;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 254;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 254;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 254;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 254;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 254;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 254;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 254;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 254;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 254;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 254;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 254;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 254;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 254;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 254;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 254;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 254;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 254;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 254;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 254;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 254;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 254;goto st309;} }
	goto st254;
st254:
	if ( ++p == pe )
		goto _test_eof254;
case 254:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 84: goto st126;
		case 116: goto st126;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4307:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 255;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 255;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 255;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 255;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 255;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 255;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 255;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 255;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 255;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 255;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 255;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 255;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 255;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 255;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 255;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 255;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 255;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 255;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 255;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 255;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 255;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 255;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 255;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 255;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 255;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 255;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 255;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 255;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 255;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 255;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 255;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 255;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 255;goto st309;} }
	goto st255;
st255:
	if ( ++p == pe )
		goto _test_eof255;
case 255:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 80: goto st129;
		case 82: goto st130;
		case 84: goto st134;
		case 112: goto st129;
		case 114: goto st130;
		case 116: goto st134;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4308:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 256;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 256;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 256;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 256;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 256;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 256;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 256;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 256;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 256;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 256;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 256;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 256;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 256;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 256;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 256;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 256;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 256;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 256;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 256;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 256;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 256;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 256;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 256;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 256;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 256;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 256;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 256;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 256;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 256;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 256;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 256;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 256;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 256;goto st309;} }
	goto st256;
st256:
	if ( ++p == pe )
		goto _test_eof256;
case 256:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 77: goto st136;
		case 79: goto st141;
		case 80: goto st143;
		case 82: goto st145;
		case 83: goto st147;
		case 86: goto st151;
		case 109: goto st136;
		case 111: goto st141;
		case 112: goto st143;
		case 114: goto st145;
		case 115: goto st147;
		case 118: goto st151;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4309:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 257;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 257;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 257;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 257;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 257;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 257;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 257;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 257;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 257;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 257;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 257;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 257;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 257;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 257;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 257;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 257;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 257;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 257;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 257;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 257;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 257;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 257;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 257;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 257;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 257;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 257;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 257;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 257;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 257;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 257;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 257;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 257;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 257;goto st309;} }
	goto st257;
st257:
	if ( ++p == pe )
		goto _test_eof257;
case 257:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 76: goto st155;
		case 88: goto st158;
		case 89: goto st160;
		case 108: goto st155;
		case 120: goto st158;
		case 121: goto st160;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4310:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 258;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 258;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 258;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 258;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 258;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 258;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 258;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 258;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 258;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 258;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 258;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 258;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 258;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 258;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 258;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 258;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 258;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 258;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 258;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 258;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 258;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 258;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 258;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 258;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 258;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 258;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 258;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 258;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 258;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 258;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 258;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 258;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 258;goto st309;} }
	goto st258;
st258:
	if ( ++p == pe )
		goto _test_eof258;
case 258:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 82: goto st165;
		case 114: goto st165;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4311:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 259;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 259;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 259;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 259;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 259;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 259;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 259;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 259;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 259;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 259;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 259;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 259;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 259;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 259;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 259;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 259;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 259;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 259;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 259;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 259;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 259;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 259;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 259;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 259;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 259;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 259;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 259;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 259;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 259;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 259;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 259;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 259;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 259;goto st309;} }
	goto st259;
st259:
	if ( ++p == pe )
		goto _test_eof259;
case 259:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 79: goto st168;
		case 111: goto st168;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr875;
		case 2107: goto tr876;
	}
	goto tr890;
tr4220:
	{
		if (pe - p == 1) {
			*wrap = WRAP_DETECTED;
		}
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 260;goto st309;} }
	goto st260;
st260:
	if ( ++p == pe )
		goto _test_eof260;
case 260:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr903;
		case 32: goto tr903;
		case 35: goto tr98;
		case 40: goto tr905;
		case 41: goto tr906;
		case 1802: goto tr97;
		case 1851: goto tr97;
		case 2058: goto tr907;
		case 2107: goto tr908;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr97;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr97;
	} else
		goto tr97;
	goto tr904;
tr903:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (*wrap == WRAP_NONE) {
			p--;
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 261;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 261;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 261;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 261;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 261;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 261;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 261;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 261;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 261;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 261;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 261;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 261;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 261;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 261;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 261;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 261;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 261;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 261;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 261;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 261;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 261;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 261;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 261;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 261;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 261;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 261;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 261;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 261;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 261;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 261;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 261;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 261;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st261;
tr905:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (*wrap == WRAP_NONE) {
			p--;
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 261;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 261;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 261;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 261;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 261;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 261;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 261;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 261;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 261;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 261;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 261;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 261;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 261;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 261;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 261;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 261;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 261;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 261;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 261;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 261;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 261;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 261;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 261;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 261;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 261;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 261;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 261;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 261;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 261;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 261;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 261;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 261;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st261;
tr906:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (*wrap == WRAP_NONE) {
			p--;
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 261;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 261;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 261;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 261;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 261;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 261;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 261;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 261;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 261;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 261;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 261;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 261;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 261;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 261;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 261;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 261;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 261;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 261;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 261;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 261;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 261;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 261;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 261;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 261;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 261;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 261;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 261;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 261;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 261;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 261;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 261;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 261;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st261;
tr907:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{
		if (*wrap == WRAP_NONE) {
			p--;
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 261;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 261;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 261;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 261;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 261;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 261;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 261;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 261;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 261;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 261;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 261;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 261;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 261;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 261;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 261;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 261;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 261;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 261;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 261;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 261;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 261;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 261;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 261;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 261;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 261;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 261;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 261;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 261;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 261;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 261;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 261;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 261;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st261;
st261:
	if ( ++p == pe )
		goto _test_eof261;
case 261:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr909;
		case 32: goto tr909;
		case 40: goto tr910;
		case 41: goto tr911;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 79: goto tr15;
		case 80: goto tr16;
		case 82: goto tr17;
		case 83: goto tr18;
		case 84: goto tr19;
		case 85: goto tr20;
		case 90: goto tr21;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 111: goto tr15;
		case 112: goto tr16;
		case 114: goto tr17;
		case 115: goto tr18;
		case 116: goto tr19;
		case 117: goto tr20;
		case 122: goto tr21;
		case 1802: goto tr87;
		case 1851: goto tr88;
		case 2058: goto tr912;
		case 2107: goto tr913;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr118;
tr908:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{
		if (*wrap == WRAP_NONE) {
			p--;
		}
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 262;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 262;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 262;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 262;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 262;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 262;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 262;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 262;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 262;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 262;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 262;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 262;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 262;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 262;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 262;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 262;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 262;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 262;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 262;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 262;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 262;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 262;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 262;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 262;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 262;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 262;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 262;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 262;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 262;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 262;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 262;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 262;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st262;
st262:
	if ( ++p == pe )
		goto _test_eof262;
case 262:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr84;
		case 1802: goto tr87;
		case 1824: goto tr84;
		case 1832: goto tr85;
		case 1833: goto tr86;
		case 1851: goto tr88;
		case 2057: goto tr914;
		case 2058: goto tr915;
		case 2080: goto tr914;
		case 2088: goto tr916;
		case 2089: goto tr917;
		case 2107: goto tr918;
	}
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr777;
	goto tr83;
tr919:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st263;
tr914:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st263;
tr916:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st263;
tr917:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st263;
tr920:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st263;
tr921:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st263;
st263:
	if ( ++p == pe )
		goto _test_eof263;
case 263:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto st7;
		case 1802: goto tr93;
		case 1824: goto st7;
		case 1832: goto tr91;
		case 1833: goto tr92;
		case 1851: goto tr94;
		case 2057: goto tr919;
		case 2058: goto tr844;
		case 2080: goto tr919;
		case 2088: goto tr920;
		case 2089: goto tr921;
		case 2107: goto tr922;
	}
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr777;
	goto tr89;
tr858:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 264;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 264;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 264;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 264;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 264;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 264;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 264;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 264;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 264;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 264;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 264;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 264;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 264;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 264;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 264;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 264;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 264;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 264;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 264;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 264;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 264;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 264;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 264;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 264;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 264;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 264;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 264;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 264;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 264;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 264;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 264;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 264;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st264;
tr989:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 264;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 264;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 264;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 264;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 264;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 264;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 264;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 264;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 264;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 264;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 264;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 264;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 264;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 264;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 264;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 264;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 264;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 264;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 264;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 264;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 264;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 264;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 264;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 264;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 264;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 264;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 264;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 264;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 264;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 264;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 264;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 264;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st264;
tr897:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 264;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 264;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 264;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 264;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 264;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 264;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 264;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 264;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 264;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 264;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 264;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 264;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 264;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 264;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 264;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 264;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 264;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 264;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 264;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 264;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 264;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 264;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 264;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 264;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 264;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 264;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 264;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 264;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 264;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 264;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 264;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 264;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st264;
tr4292:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 264;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 264;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 264;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 264;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 264;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 264;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 264;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 264;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 264;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 264;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 264;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 264;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 264;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 264;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 264;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 264;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 264;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 264;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 264;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 264;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 264;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 264;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 264;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 264;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 264;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 264;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 264;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 264;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 264;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 264;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 264;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 264;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	goto st264;
st264:
	if ( ++p == pe )
		goto _test_eof264;
case 264:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr845;
		case 1802: goto tr848;
		case 1824: goto tr845;
		case 1832: goto tr846;
		case 1833: goto tr847;
		case 1851: goto tr849;
		case 2057: goto tr923;
		case 2058: goto tr915;
		case 2080: goto tr923;
		case 2088: goto tr924;
		case 2089: goto tr925;
		case 2107: goto tr926;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr859;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr83;
tr927:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st265;
tr923:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st265;
tr924:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st265;
tr925:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st265;
tr928:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st265;
tr929:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st265;
st265:
	if ( ++p == pe )
		goto _test_eof265;
case 265:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr850;
		case 1802: goto tr96;
		case 1824: goto tr850;
		case 1832: goto tr851;
		case 1833: goto tr852;
		case 1851: goto tr853;
		case 2057: goto tr927;
		case 2058: goto tr844;
		case 2080: goto tr927;
		case 2088: goto tr928;
		case 2089: goto tr929;
		case 2107: goto tr930;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr859;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr83;
st266:
	if ( ++p == pe )
		goto _test_eof266;
case 266:
	switch( (*p) ) {
		case 65: goto st267;
		case 97: goto st267;
	}
	goto tr38;
st267:
	if ( ++p == pe )
		goto _test_eof267;
case 267:
	switch( (*p) ) {
		case 65: goto st268;
		case 97: goto st268;
	}
	goto tr38;
st268:
	if ( ++p == pe )
		goto _test_eof268;
case 268:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr933;
		case 32: goto tr933;
		case 40: goto tr934;
		case 41: goto tr935;
		case 3082: goto tr936;
		case 3131: goto tr937;
		case 3338: goto tr938;
		case 3387: goto tr938;
		case 3594: goto tr939;
		case 3643: goto tr940;
	}
	goto tr61;
st269:
	if ( ++p == pe )
		goto _test_eof269;
case 269:
	switch( (*p) ) {
		case 83: goto st270;
		case 115: goto st270;
	}
	goto tr38;
st270:
	if ( ++p == pe )
		goto _test_eof270;
case 270:
	switch( (*p) ) {
		case 68: goto st271;
		case 100: goto st271;
	}
	goto tr38;
st271:
	if ( ++p == pe )
		goto _test_eof271;
case 271:
	switch( (*p) ) {
		case 66: goto st272;
		case 98: goto st272;
	}
	goto tr38;
st272:
	if ( ++p == pe )
		goto _test_eof272;
case 272:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr944;
		case 32: goto tr944;
		case 40: goto tr945;
		case 41: goto tr946;
		case 3082: goto tr947;
		case 3131: goto tr948;
		case 3338: goto tr949;
		case 3387: goto tr949;
		case 3594: goto tr950;
		case 3643: goto tr951;
	}
	goto tr61;
st273:
	if ( ++p == pe )
		goto _test_eof273;
case 273:
	switch( (*p) ) {
		case 76: goto st274;
		case 108: goto st274;
	}
	goto tr38;
st274:
	if ( ++p == pe )
		goto _test_eof274;
case 274:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr953;
		case 32: goto tr953;
		case 40: goto tr954;
		case 41: goto tr955;
		case 3082: goto tr956;
		case 3131: goto tr957;
		case 3338: goto tr958;
		case 3387: goto tr958;
		case 3594: goto tr959;
		case 3643: goto tr960;
	}
	goto tr61;
tr140:
	{
		s->buffer_length = 0;
	}
	goto st275;
tr961:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st275;
tr4222:
	{
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st275;
st275:
	if ( ++p == pe )
		goto _test_eof275;
case 275:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr754;
		case 2058: goto tr962;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr961;
	} else if ( _widec >= 1664 )
		goto tr753;
	goto tr83;
tr4224:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 276;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 276;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 276;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 276;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 276;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 276;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 276;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 276;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 276;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 276;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 276;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 276;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 276;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 276;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 276;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 276;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 276;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 276;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 276;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 276;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 276;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 276;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 276;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 276;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 276;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 276;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 276;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 276;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 276;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 276;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 276;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 276;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st276;
tr142:
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 276;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 276;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 276;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 276;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 276;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 276;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 276;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 276;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 276;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 276;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 276;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 276;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 276;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 276;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 276;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 276;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 276;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 276;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 276;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 276;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 276;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 276;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 276;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 276;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 276;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 276;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 276;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 276;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 276;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 276;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 276;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 276;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	goto st276;
st276:
	if ( ++p == pe )
		goto _test_eof276;
case 276:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr963;
		case 32: goto tr963;
		case 40: goto tr964;
		case 41: goto tr965;
		case 1802: goto tr966;
		case 1851: goto tr849;
		case 2058: goto tr966;
		case 2107: goto tr849;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr753;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr753;
	} else
		goto tr753;
	goto tr83;
tr967:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st277;
tr963:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st277;
tr964:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st277;
tr965:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st277;
tr968:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st277;
tr969:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st277;
st277:
	if ( ++p == pe )
		goto _test_eof277;
case 277:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr967;
		case 32: goto tr967;
		case 40: goto tr968;
		case 41: goto tr969;
		case 1802: goto tr970;
		case 1851: goto tr853;
		case 2058: goto tr970;
		case 2107: goto tr853;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr753;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr753;
	} else
		goto tr753;
	goto tr89;
tr144:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 278;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 278;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 278;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 278;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 278;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 278;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 278;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 278;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 278;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 278;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 278;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 278;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 278;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 278;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 278;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 278;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 278;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 278;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 278;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 278;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 278;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 278;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 278;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 278;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 278;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 278;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 278;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 278;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 278;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 278;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 278;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 278;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st278;
tr4226:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 278;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 278;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 278;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 278;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 278;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 278;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 278;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 278;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 278;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 278;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 278;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 278;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 278;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 278;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 278;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 278;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 278;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 278;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 278;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 278;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 278;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 278;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 278;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 278;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 278;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 278;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 278;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 278;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 278;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 278;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 278;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 278;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st278;
st278:
	if ( ++p == pe )
		goto _test_eof278;
case 278:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr963;
		case 1802: goto tr966;
		case 1824: goto tr963;
		case 1832: goto tr964;
		case 1833: goto tr965;
		case 1851: goto tr849;
		case 2057: goto tr971;
		case 2058: goto tr915;
		case 2080: goto tr971;
		case 2088: goto tr972;
		case 2089: goto tr973;
		case 2107: goto tr926;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr961;
	} else if ( _widec >= 1664 )
		goto tr753;
	goto tr83;
tr974:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st279;
tr971:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st279;
tr972:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st279;
tr973:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st279;
tr975:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st279;
tr976:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st279;
st279:
	if ( ++p == pe )
		goto _test_eof279;
case 279:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr967;
		case 1802: goto tr970;
		case 1824: goto tr967;
		case 1832: goto tr968;
		case 1833: goto tr969;
		case 1851: goto tr853;
		case 2057: goto tr974;
		case 2058: goto tr844;
		case 2080: goto tr974;
		case 2088: goto tr975;
		case 2089: goto tr976;
		case 2107: goto tr930;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr961;
	} else if ( _widec >= 1664 )
		goto tr753;
	goto tr83;
tr114:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 280;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 280;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 280;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 280;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 280;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 280;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 280;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 280;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 280;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 280;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 280;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 280;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 280;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 280;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 280;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 280;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 280;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 280;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 280;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 280;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 280;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 280;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 280;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 280;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 280;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 280;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 280;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 280;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 280;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 280;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 280;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 280;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st280;
tr107:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 280;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 280;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 280;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 280;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 280;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 280;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 280;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 280;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 280;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 280;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 280;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 280;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 280;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 280;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 280;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 280;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 280;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 280;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 280;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 280;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 280;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 280;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 280;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 280;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 280;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 280;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 280;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 280;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 280;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 280;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 280;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 280;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st280;
tr832:
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 280;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 280;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 280;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 280;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 280;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 280;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 280;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 280;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 280;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 280;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 280;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 280;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 280;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 280;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 280;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 280;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 280;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 280;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 280;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 280;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 280;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 280;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 280;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 280;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 280;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 280;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 280;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 280;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 280;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 280;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 280;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 280;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st280;
st280:
	if ( ++p == pe )
		goto _test_eof280;
case 280:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr845;
		case 1802: goto tr848;
		case 1824: goto tr845;
		case 1832: goto tr846;
		case 1833: goto tr847;
		case 1851: goto tr849;
		case 2057: goto tr977;
		case 2058: goto tr848;
		case 2080: goto tr977;
		case 2088: goto tr978;
		case 2089: goto tr979;
		case 2107: goto tr174;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr115;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr83;
tr980:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st281;
tr977:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st281;
tr978:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st281;
tr979:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st281;
tr981:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st281;
tr982:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st281;
st281:
	if ( ++p == pe )
		goto _test_eof281;
case 281:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr850;
		case 1802: goto tr96;
		case 1824: goto tr850;
		case 1832: goto tr851;
		case 1833: goto tr852;
		case 1851: goto tr853;
		case 2057: goto tr980;
		case 2058: goto tr96;
		case 2080: goto tr980;
		case 2088: goto tr981;
		case 2089: goto tr982;
		case 2107: goto tr178;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr115;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr83;
tr820:
	{
		s->buffer_length = 0;
	}
	goto st282;
tr983:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st282;
tr813:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	goto st282;
st282:
	if ( ++p == pe )
		goto _test_eof282;
case 282:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr984;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr983;
	goto tr75;
tr821:
	{
		s->line_counter++;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 283;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 283;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 283;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 283;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 283;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 283;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 283;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 283;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 283;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 283;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 283;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 283;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 283;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 283;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 283;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 283;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 283;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 283;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 283;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 283;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 283;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 283;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 283;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 283;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 283;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 283;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 283;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 283;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 283;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 283;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 283;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 283;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st283;
tr814:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->line_counter++;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 283;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 283;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 283;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 283;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 283;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 283;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 283;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 283;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 283;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 283;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 283;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 283;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 283;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 283;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 283;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 283;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 283;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 283;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 283;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 283;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 283;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 283;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 283;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 283;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 283;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 283;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 283;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 283;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 283;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 283;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 283;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 283;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st283;
st283:
	if ( ++p == pe )
		goto _test_eof283;
case 283:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(2688 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
			if (
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(2688 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
		if (
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr985;
		case 32: goto tr985;
		case 40: goto tr986;
		case 41: goto tr987;
		case 58: goto tr73;
		case 65: goto tr122;
		case 67: goto tr123;
		case 68: goto tr124;
		case 69: goto tr125;
		case 72: goto tr126;
		case 73: goto tr127;
		case 75: goto tr128;
		case 76: goto tr129;
		case 77: goto tr130;
		case 78: goto tr131;
		case 79: goto tr132;
		case 80: goto tr133;
		case 82: goto tr134;
		case 83: goto tr135;
		case 84: goto tr136;
		case 85: goto tr137;
		case 90: goto tr138;
		case 92: goto tr78;
		case 97: goto tr122;
		case 99: goto tr123;
		case 100: goto tr124;
		case 101: goto tr125;
		case 104: goto tr126;
		case 105: goto tr127;
		case 107: goto tr128;
		case 108: goto tr129;
		case 109: goto tr130;
		case 110: goto tr131;
		case 111: goto tr132;
		case 112: goto tr133;
		case 114: goto tr134;
		case 115: goto tr135;
		case 116: goto tr136;
		case 117: goto tr137;
		case 122: goto tr138;
		case 2826: goto tr87;
		case 2875: goto tr88;
		case 3082: goto tr912;
		case 3131: goto tr988;
		case 3338: goto tr105;
		case 3387: goto tr106;
		case 3594: goto tr912;
		case 3643: goto tr989;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr73;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr73;
		} else if ( _widec >= 48 )
			goto tr121;
	} else
		goto tr73;
	goto tr118;
tr822:
	{
		s->buffer_length = 0;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 284;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 284;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 284;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 284;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 284;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 284;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 284;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 284;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 284;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 284;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 284;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 284;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 284;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 284;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 284;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 284;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 284;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 284;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 284;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 284;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 284;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 284;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 284;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 284;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 284;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 284;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 284;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 284;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 284;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 284;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 284;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 284;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st284;
tr815:
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	{
		s->buffer_length = 0;
	}
	{ s->r_type = KNOT_RRTYPE_A; }
	{
		rdata_tail = s->r_data;
	}
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 284;goto st844;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 284;goto st846;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 284;goto st848;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 284;goto st880;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 284;goto st885;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 284;goto st890;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 284;goto st895;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 284;goto st899;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 284;goto st901;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 284;goto st956;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 284;goto st967;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 284;goto st984;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 284;goto st995;}
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_CDS:
			{stack[top++] = 284;goto st1006;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 284;goto st1019;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 284;goto st1029;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 284;goto st1068;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 284;goto st1254;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_CDNSKEY:
			{stack[top++] = 284;goto st1257;}
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_OPENPGPKEY:
			{stack[top++] = 284;goto st1268;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 284;goto st1270;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 284;goto st1299;}
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_SMIMEA:
			{stack[top++] = 284;goto st1312;}
		case KNOT_RRTYPE_CSYNC:
			{stack[top++] = 284;goto st1325;}
		case KNOT_RRTYPE_ZONEMD:
			{stack[top++] = 284;goto st1331;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 284;goto st1349;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 284;goto st1344;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 284;goto st1362;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 284;goto st1368;}
		case KNOT_RRTYPE_URI:
			{stack[top++] = 284;goto st1374;}
		case KNOT_RRTYPE_CAA:
			{stack[top++] = 284;goto st1382;}
		case KNOT_RRTYPE_SVCB:
		case KNOT_RRTYPE_HTTPS:
			{stack[top++] = 284;goto st1390;}
		default:
			WARN(ZS_CANNOT_TEXT_DATA);
			{goto st307;}
		}
	}
	goto st284;
st284:
	if ( ++p == pe )
		goto _test_eof284;
case 284:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto tr84;
		case 1802: goto tr87;
		case 1824: goto tr84;
		case 1832: goto tr85;
		case 1833: goto tr86;
		case 1851: goto tr88;
		case 2057: goto tr990;
		case 2058: goto tr915;
		case 2080: goto tr990;
		case 2088: goto tr991;
		case 2089: goto tr992;
		case 2107: goto tr926;
	}
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr983;
	goto tr83;
tr993:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st285;
tr990:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st285;
tr991:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st285;
tr992:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st285;
tr994:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st285;
tr995:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st285;
st285:
	if ( ++p == pe )
		goto _test_eof285;
case 285:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(1664 + ((*p) - -128));
				if (
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1801: goto st7;
		case 1802: goto tr93;
		case 1824: goto st7;
		case 1832: goto tr91;
		case 1833: goto tr92;
		case 1851: goto tr94;
		case 2057: goto tr993;
		case 2058: goto tr844;
		case 2080: goto tr993;
		case 2088: goto tr994;
		case 2089: goto tr995;
		case 2107: goto tr930;
	}
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr983;
	goto tr83;
tr4232:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 286;goto st309;} }
	goto st286;
tr4253:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 286;goto st309;} }
	goto st286;
st286:
	if ( ++p == pe )
		goto _test_eof286;
case 286:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 65: goto st18;
		case 68: goto st22;
		case 69: goto st29;
		case 78: goto st32;
		case 83: goto st36;
		case 97: goto st18;
		case 100: goto st22;
		case 101: goto st29;
		case 110: goto st32;
		case 115: goto st36;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4233:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 287;goto st309;} }
	goto st287;
tr4254:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 287;goto st309;} }
	goto st287;
st287:
	if ( ++p == pe )
		goto _test_eof287;
case 287:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 72: goto st41;
		case 78: goto st45;
		case 83: goto st53;
		case 104: goto st41;
		case 110: goto st45;
		case 115: goto st53;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4234:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 288;goto st309;} }
	goto st288;
tr4255:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 288;goto st309;} }
	goto st288;
st288:
	if ( ++p == pe )
		goto _test_eof288;
case 288:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 85: goto st55;
		case 117: goto st55;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4235:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 289;goto st309;} }
	goto st289;
tr4256:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 289;goto st309;} }
	goto st289;
st289:
	if ( ++p == pe )
		goto _test_eof289;
case 289:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 73: goto st62;
		case 84: goto st66;
		case 105: goto st62;
		case 116: goto st66;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4257:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 290;goto st309;} }
	goto st290;
st290:
	if ( ++p == pe )
		goto _test_eof290;
case 290:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 78: goto st176;
		case 80: goto st74;
		case 110: goto st176;
		case 112: goto st74;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4237:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 291;goto st309;} }
	goto st291;
tr4258:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 291;goto st309;} }
	goto st291;
st291:
	if ( ++p == pe )
		goto _test_eof291;
case 291:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 69: goto st82;
		case 88: goto st84;
		case 101: goto st82;
		case 120: goto st84;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4238:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 292;goto st309;} }
	goto st292;
tr4259:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 292;goto st309;} }
	goto st292;
st292:
	if ( ++p == pe )
		goto _test_eof292;
case 292:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 51: goto st86;
		case 54: goto st88;
		case 79: goto st90;
		case 80: goto st92;
		case 111: goto st90;
		case 112: goto st92;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4239:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 293;goto st309;} }
	goto st293;
tr4260:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 293;goto st309;} }
	goto st293;
st293:
	if ( ++p == pe )
		goto _test_eof293;
case 293:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 73: goto st94;
		case 88: goto st98;
		case 105: goto st94;
		case 120: goto st98;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4240:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 294;goto st309;} }
	goto st294;
tr4261:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 294;goto st309;} }
	goto st294;
st294:
	if ( ++p == pe )
		goto _test_eof294;
case 294:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 65: goto st100;
		case 73: goto st104;
		case 83: goto st106;
		case 97: goto st100;
		case 105: goto st104;
		case 115: goto st106;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4241:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 295;goto st309;} }
	goto st295;
tr4262:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 295;goto st309;} }
	goto st295;
st295:
	if ( ++p == pe )
		goto _test_eof295;
case 295:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 80: goto st116;
		case 112: goto st116;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4242:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 296;goto st309;} }
	goto st296;
tr4263:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 296;goto st309;} }
	goto st296;
st296:
	if ( ++p == pe )
		goto _test_eof296;
case 296:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 84: goto st126;
		case 116: goto st126;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4243:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 297;goto st309;} }
	goto st297;
tr4264:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 297;goto st309;} }
	goto st297;
st297:
	if ( ++p == pe )
		goto _test_eof297;
case 297:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 80: goto st129;
		case 82: goto st130;
		case 84: goto st134;
		case 112: goto st129;
		case 114: goto st130;
		case 116: goto st134;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4244:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 298;goto st309;} }
	goto st298;
tr4265:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 298;goto st309;} }
	goto st298;
st298:
	if ( ++p == pe )
		goto _test_eof298;
case 298:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 77: goto st136;
		case 79: goto st141;
		case 80: goto st143;
		case 82: goto st145;
		case 83: goto st147;
		case 86: goto st151;
		case 109: goto st136;
		case 111: goto st141;
		case 112: goto st143;
		case 114: goto st145;
		case 115: goto st147;
		case 118: goto st151;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4245:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 299;goto st309;} }
	goto st299;
tr4266:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 299;goto st309;} }
	goto st299;
st299:
	if ( ++p == pe )
		goto _test_eof299;
case 299:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 76: goto st155;
		case 88: goto st158;
		case 89: goto st160;
		case 108: goto st155;
		case 120: goto st158;
		case 121: goto st160;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4246:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 300;goto st309;} }
	goto st300;
tr4267:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 300;goto st309;} }
	goto st300;
st300:
	if ( ++p == pe )
		goto _test_eof300;
case 300:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 82: goto st165;
		case 114: goto st165;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr4247:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 301;goto st309;} }
	goto st301;
tr4268:
	{
		s->r_class = s->default_class;
	}
	{
		s->r_ttl = s->default_ttl;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 301;goto st309;} }
	goto st301;
st301:
	if ( ++p == pe )
		goto _test_eof301;
case 301:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 79: goto st168;
		case 111: goto st168;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr25:
	{
		s->buffer_length = 0;
	}
	goto st302;
tr997:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st302;
tr4214:
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st302;
tr4282:
	{
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st302;
tr4278:
	{
		NOERR;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st302;
st302:
	if ( ++p == pe )
		goto _test_eof302;
case 302:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr754;
		case 2058: goto tr998;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr997;
	} else if ( _widec >= 1664 )
		goto tr753;
	goto tr89;
tr697:
	{
		s->buffer_length = 0;
	}
	goto st303;
tr999:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st303;
tr4249:
	{
		s->buffer_length = 0;
	}
	{
		if (s->r_owner_length == 0) {
			WARN(ZS_BAD_PREVIOUS_OWNER);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st303;
st303:
	if ( ++p == pe )
		goto _test_eof303;
case 303:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr754;
		case 2058: goto tr1000;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr999;
	} else if ( _widec >= 1664 )
		goto tr753;
	goto tr151;
tr4236:
	{
		s->r_class = s->default_class;
	}
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	{ p--; {stack[top++] = 304;goto st309;} }
	goto st304;
st304:
	if ( ++p == pe )
		goto _test_eof304;
case 304:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 40: goto tr768;
		case 41: goto tr769;
		case 78: goto st71;
		case 80: goto st74;
		case 110: goto st71;
		case 112: goto st74;
		case 2058: goto tr770;
		case 2107: goto tr771;
	}
	goto tr996;
tr156:
	{
		s->buffer_length = 0;
	}
	goto st305;
tr1001:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st305;
tr150:
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer_length = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st305;
st305:
	if ( ++p == pe )
		goto _test_eof305;
case 305:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 1802: goto tr96;
		case 2058: goto tr1002;
	}
	if ( _widec > 1919 ) {
		if ( 1920 <= _widec && _widec <= 2175 )
			goto tr1001;
	} else if ( _widec >= 1664 )
		goto tr95;
	goto tr151;
tr1006:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st306;
tr1005:
	{
		s->line_counter++;
	}
	goto st306;
st306:
	if ( ++p == pe )
		goto _test_eof306;
case 306:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 41 <= (*p) && (*p) <= 41 ) {
			_widec = (short)(640 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 266: goto tr1004;
		case 522: goto tr1005;
		case 1065: goto tr1006;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 9 )
			goto st306;
	} else if ( _widec > 40 ) {
		if ( 42 <= _widec )
			goto st306;
	} else
		goto st306;
	goto st0;
tr1004:
	{ p--; {stack[top++] = 1406;goto st1396;} }
	{
		s->line_counter++;
	}
	goto st1406;
st1406:
	if ( ++p == pe )
		goto _test_eof1406;
case 1406:
	goto st0;
st307:
	if ( ++p == pe )
		goto _test_eof307;
case 307:
	if ( (*p) == 10 )
		goto tr1008;
	goto tr1007;
tr1007:
	{
		s->buffer_length = 0;
	}
	{
		if ((*p) == '\r') {
			ERR(ZS_DOS_NEWLINE);
		}

		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st308;
tr1009:
	{
		if ((*p) == '\r') {
			ERR(ZS_DOS_NEWLINE);
		}

		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st308;
st308:
	if ( ++p == pe )
		goto _test_eof308;
case 308:
	if ( (*p) == 10 )
		goto tr1010;
	goto tr1009;
tr1008:
	cs = 1407;
	{
		s->buffer_length = 0;
	}
	{
		// Terminate the error context string.
		s->buffer[s->buffer_length++] = 0;

		// Error counter incrementation.
		s->error.counter++;

		// Initialize the fcall stack.
		top = 0;

		// Reset per-record contexts.
		s->long_string = false;
		s->comma_list = false;
		s->pending_backslash = false;

		s->state = ZS_STATE_ERROR;

		// Execute the error callback.
		if (s->process.automatic) {
			p--;
			if (s->process.error != NULL) {
				s->process.error(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; goto _out;}
				}
			}

			// Stop the scanner if fatal error.
			if (s->error.fatal) {
				{p++; goto _out;}
			}
			{goto st306;}
		} else {
			// Return if external processing.
			p--; cs = 306; {p++; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto _again;
tr1010:
	cs = 1407;
	{
		// Terminate the error context string.
		s->buffer[s->buffer_length++] = 0;

		// Error counter incrementation.
		s->error.counter++;

		// Initialize the fcall stack.
		top = 0;

		// Reset per-record contexts.
		s->long_string = false;
		s->comma_list = false;
		s->pending_backslash = false;

		s->state = ZS_STATE_ERROR;

		// Execute the error callback.
		if (s->process.automatic) {
			p--;
			if (s->process.error != NULL) {
				s->process.error(s);

				// Stop if required from the callback.
				if (s->state == ZS_STATE_STOP) {
					{p++; goto _out;}
				}
			}

			// Stop the scanner if fatal error.
			if (s->error.fatal) {
				{p++; goto _out;}
			}
			{goto st306;}
		} else {
			// Return if external processing.
			p--; cs = 306; {p++; goto _out;}
		}
	}
	{
		s->line_counter++;
	}
	goto _again;
st1407:
	if ( ++p == pe )
		goto _test_eof1407;
case 1407:
	goto st0;
st309:
	if ( ++p == pe )
		goto _test_eof309;
case 309:
	switch( (*p) ) {
		case 42: goto tr1012;
		case 46: goto tr1013;
		case 64: goto st317;
		case 92: goto tr1015;
		case 95: goto tr1012;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr1012;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr1012;
	} else
		goto tr1012;
	goto tr1011;
tr1012:
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st310;
tr1017:
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st310;
tr1021:
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st310;
tr1028:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st310;
st310:
	if ( ++p == pe )
		goto _test_eof310;
case 310:
	switch( (*p) ) {
		case 32: goto tr1016;
		case 42: goto tr1017;
		case 46: goto tr1018;
		case 59: goto tr1016;
		case 92: goto st312;
		case 95: goto tr1017;
	}
	if ( (*p) < 45 ) {
		if ( (*p) > 10 ) {
			if ( 40 <= (*p) && (*p) <= 41 )
				goto tr1016;
		} else if ( (*p) >= 9 )
			goto tr1016;
	} else if ( (*p) > 57 ) {
		if ( (*p) > 90 ) {
			if ( 97 <= (*p) && (*p) <= 122 )
				goto tr1017;
		} else if ( (*p) >= 65 )
			goto tr1017;
	} else
		goto tr1017;
	goto tr1011;
tr1016:
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		// Check for (relative + origin) dname length overflow.
		if (s->dname_tmp_length + s->zone_origin_length <= ZS_MAX_DNAME_LENGTH) {
			memcpy(s->dname + s->dname_tmp_length,
			       s->zone_origin,
			       s->zone_origin_length);

			s->dname_tmp_length += s->zone_origin_length;
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1408;
tr1020:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1408;
tr1027:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		// Check for (relative + origin) dname length overflow.
		if (s->dname_tmp_length + s->zone_origin_length <= ZS_MAX_DNAME_LENGTH) {
			memcpy(s->dname + s->dname_tmp_length,
			       s->zone_origin,
			       s->zone_origin_length);

			s->dname_tmp_length += s->zone_origin_length;
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1408;
tr1031:
	{
		// Copy already verified zone origin.
		memcpy(s->dname,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length = s->zone_origin_length;
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1408;
st1408:
	if ( ++p == pe )
		goto _test_eof1408;
case 1408:
	goto st0;
tr1018:
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st311;
tr1029:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st311;
st311:
	if ( ++p == pe )
		goto _test_eof311;
case 311:
	switch( (*p) ) {
		case 32: goto tr1020;
		case 42: goto tr1021;
		case 45: goto tr1021;
		case 59: goto tr1020;
		case 92: goto tr1022;
		case 95: goto tr1021;
	}
	if ( (*p) < 47 ) {
		if ( (*p) > 10 ) {
			if ( 40 <= (*p) && (*p) <= 41 )
				goto tr1020;
		} else if ( (*p) >= 9 )
			goto tr1020;
	} else if ( (*p) > 57 ) {
		if ( (*p) > 90 ) {
			if ( 97 <= (*p) && (*p) <= 122 )
				goto tr1021;
		} else if ( (*p) >= 65 )
			goto tr1021;
	} else
		goto tr1021;
	goto tr1011;
tr1015:
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st312;
tr1022:
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st312;
tr1030:
	{
		s->dname_tmp_length++;
	}
	goto st312;
st312:
	if ( ++p == pe )
		goto _test_eof312;
case 312:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1023;
	goto tr1017;
tr1023:
	{
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st313;
st313:
	if ( ++p == pe )
		goto _test_eof313;
case 313:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1025;
	goto tr1024;
tr1025:
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st314;
st314:
	if ( ++p == pe )
		goto _test_eof314;
case 314:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1026;
	goto tr1024;
tr1026:
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st315;
st315:
	if ( ++p == pe )
		goto _test_eof315;
case 315:
	switch( (*p) ) {
		case 32: goto tr1027;
		case 42: goto tr1028;
		case 46: goto tr1029;
		case 59: goto tr1027;
		case 92: goto tr1030;
		case 95: goto tr1028;
	}
	if ( (*p) < 45 ) {
		if ( (*p) > 10 ) {
			if ( 40 <= (*p) && (*p) <= 41 )
				goto tr1027;
		} else if ( (*p) >= 9 )
			goto tr1027;
	} else if ( (*p) > 57 ) {
		if ( (*p) > 90 ) {
			if ( 97 <= (*p) && (*p) <= 122 )
				goto tr1028;
		} else if ( (*p) >= 65 )
			goto tr1028;
	} else
		goto tr1028;
	goto tr1024;
tr1013:
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	goto st316;
st316:
	if ( ++p == pe )
		goto _test_eof316;
case 316:
	switch( (*p) ) {
		case 32: goto tr1020;
		case 59: goto tr1020;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr1020;
	} else if ( (*p) >= 9 )
		goto tr1020;
	goto tr1011;
st317:
	if ( ++p == pe )
		goto _test_eof317;
case 317:
	switch( (*p) ) {
		case 32: goto tr1031;
		case 59: goto tr1031;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr1031;
	} else if ( (*p) >= 9 )
		goto tr1031;
	goto tr1011;
st318:
	if ( ++p == pe )
		goto _test_eof318;
case 318:
	switch( (*p) ) {
		case 34: goto st324;
		case 92: goto st320;
	}
	if ( (*p) > 58 ) {
		if ( 60 <= (*p) && (*p) <= 126 )
			goto tr1033;
	} else if ( (*p) >= 33 )
		goto tr1033;
	goto tr1032;
tr1033:
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st319;
tr1037:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st319;
tr1045:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st319;
st319:
	if ( ++p == pe )
		goto _test_eof319;
case 319:
	switch( (*p) ) {
		case 32: goto tr1036;
		case 33: goto tr1037;
		case 59: goto tr1036;
		case 92: goto tr1039;
	}
	if ( (*p) < 35 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto tr1036;
	} else if ( (*p) > 39 ) {
		if ( (*p) > 41 ) {
			if ( 42 <= (*p) && (*p) <= 126 )
				goto tr1037;
		} else if ( (*p) >= 40 )
			goto tr1038;
	} else
		goto tr1037;
	goto tr1032;
tr1036:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1409;
tr1044:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1409;
tr1057:
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1409;
st1409:
	if ( ++p == pe )
		goto _test_eof1409;
case 1409:
	goto st0;
tr1038:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1410;
tr1046:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1410;
st1410:
	if ( ++p == pe )
		goto _test_eof1410;
case 1410:
	switch( (*p) ) {
		case 32: goto tr1036;
		case 33: goto tr1037;
		case 59: goto tr1036;
		case 92: goto tr1039;
	}
	if ( (*p) < 35 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto tr1036;
	} else if ( (*p) > 39 ) {
		if ( (*p) > 41 ) {
			if ( 42 <= (*p) && (*p) <= 126 )
				goto tr1037;
		} else if ( (*p) >= 40 )
			goto tr1038;
	} else
		goto tr1037;
	goto tr1032;
tr1039:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	goto st320;
tr1047:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	goto st320;
st320:
	if ( ++p == pe )
		goto _test_eof320;
case 320:
	if ( (*p) < 48 ) {
		if ( 32 <= (*p) && (*p) <= 47 )
			goto tr1033;
	} else if ( (*p) > 57 ) {
		if ( 58 <= (*p) && (*p) <= 126 )
			goto tr1033;
	} else
		goto tr1041;
	goto tr1040;
tr1041:
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*rdata_tail = 0;
			s->item_length++;
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + '0')
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER8_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st321;
st321:
	if ( ++p == pe )
		goto _test_eof321;
case 321:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1042;
	goto tr1040;
tr1042:
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + '0')
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER8_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st322;
st322:
	if ( ++p == pe )
		goto _test_eof322;
case 322:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1043;
	goto tr1040;
tr1043:
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + '0')
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER8_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st323;
st323:
	if ( ++p == pe )
		goto _test_eof323;
case 323:
	switch( (*p) ) {
		case 32: goto tr1044;
		case 33: goto tr1045;
		case 59: goto tr1044;
		case 92: goto tr1047;
	}
	if ( (*p) < 35 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto tr1044;
	} else if ( (*p) > 39 ) {
		if ( (*p) > 41 ) {
			if ( 42 <= (*p) && (*p) <= 126 )
				goto tr1045;
		} else if ( (*p) >= 40 )
			goto tr1046;
	} else
		goto tr1045;
	goto tr1040;
tr1048:
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st324;
tr1052:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st324;
tr1061:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st324;
st324:
	if ( ++p == pe )
		goto _test_eof324;
case 324:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(1152 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1048;
		case 32: goto tr1048;
		case 34: goto st326;
		case 59: goto tr1048;
		case 92: goto st327;
		case 1546: goto tr1048;
	}
	if ( 33 <= _widec && _widec <= 126 )
		goto tr1049;
	goto tr1032;
tr1049:
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st325;
tr1053:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st325;
tr1062:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*(rdata_tail++) = (*p);
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st325;
st325:
	if ( ++p == pe )
		goto _test_eof325;
case 325:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(1152 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1052;
		case 32: goto tr1052;
		case 34: goto tr1054;
		case 59: goto tr1052;
		case 92: goto tr1055;
		case 1546: goto tr1052;
	}
	if ( 33 <= _widec && _widec <= 126 )
		goto tr1053;
	goto tr1032;
tr1054:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	goto st326;
tr1063:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	goto st326;
st326:
	if ( ++p == pe )
		goto _test_eof326;
case 326:
	switch( (*p) ) {
		case 32: goto tr1057;
		case 59: goto tr1057;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr1057;
	} else if ( (*p) >= 9 )
		goto tr1057;
	goto tr1056;
tr1055:
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	goto st327;
tr1064:
	{
		rdata_tail++;
	}
	{
		if (s->comma_list) {
			uint8_t *last_two = rdata_tail - 2;
			uint16_t current_len = rdata_tail - s->item_length_location - 2;
			if (last_two[1] == ',') {
				if (current_len <= 1) {
					WARN(ZS_EMPTY_LIST_ITEM);
					p--; {goto st307;}
				} else if (last_two[0] != '\\' || !s->pending_backslash) { // Start a new item.
					*(s->item_length_location) = current_len;
					s->item_length_location = rdata_tail - 1;
				} else { // Remove backslash.
					last_two[0] = ',';
					rdata_tail--;
					s->pending_backslash = false;
				}
			} else if (last_two[1] == '\\') {
				if (s->pending_backslash) { // Remove backslash.
					rdata_tail--;
					s->pending_backslash = false;
				} else {
					s->pending_backslash = true;
				}
			} else if (s->pending_backslash) {
				WARN(ZS_BAD_ALPN_BACKSLASH);
				p--; {goto st307;}
			}
		}
	}
	goto st327;
st327:
	if ( ++p == pe )
		goto _test_eof327;
case 327:
	if ( (*p) < 48 ) {
		if ( 32 <= (*p) && (*p) <= 47 )
			goto tr1049;
	} else if ( (*p) > 57 ) {
		if ( 58 <= (*p) && (*p) <= 126 )
			goto tr1049;
	} else
		goto tr1058;
	goto tr1040;
tr1058:
	{
		if (rdata_tail <= rdata_stop) {
			// Split long string.
			if (s->long_string &&
			    rdata_tail - s->item_length_location == 1 + MAX_ITEM_LENGTH) {
				// _item_length_exit equivalent.
				*(s->item_length_location) = MAX_ITEM_LENGTH;
				// _item_length_init equivalent.
				s->item_length_location = rdata_tail++;

				if (rdata_tail > rdata_stop) {
					WARN(ZS_TEXT_OVERFLOW);
					p--; {goto st307;}
				}
			}

			*rdata_tail = 0;
			s->item_length++;
		} else {
			WARN(ZS_TEXT_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + '0')
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER8_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st328;
st328:
	if ( ++p == pe )
		goto _test_eof328;
case 328:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1059;
	goto tr1040;
tr1059:
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + '0')
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER8_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st329;
st329:
	if ( ++p == pe )
		goto _test_eof329;
case 329:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1060;
	goto tr1040;
tr1060:
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + '0')
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER8_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st330;
st330:
	if ( ++p == pe )
		goto _test_eof330;
case 330:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(1152 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1061;
		case 32: goto tr1061;
		case 34: goto tr1063;
		case 59: goto tr1061;
		case 92: goto tr1064;
		case 1546: goto tr1061;
	}
	if ( 33 <= _widec && _widec <= 126 )
		goto tr1062;
	goto tr1040;
st331:
	if ( ++p == pe )
		goto _test_eof331;
case 331:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st332;
		case 32: goto st332;
		case 40: goto tr1067;
		case 41: goto tr1068;
		case 2058: goto tr1069;
		case 2107: goto tr1070;
	}
	goto tr1065;
tr1067:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st332;
tr1068:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st332;
tr1069:
	{
		s->line_counter++;
	}
	goto st332;
tr1108:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 332; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st332;
st332:
	if ( ++p == pe )
		goto _test_eof332;
case 332:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st332;
		case 32: goto st332;
		case 40: goto tr1067;
		case 41: goto tr1068;
		case 2058: goto tr1069;
		case 2107: goto tr1070;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1072;
	goto tr1071;
tr1072:
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st333;
tr1077:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st333;
st333:
	if ( ++p == pe )
		goto _test_eof333;
case 333:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1074;
		case 32: goto tr1074;
		case 40: goto tr1075;
		case 41: goto tr1076;
		case 68: goto tr1078;
		case 72: goto tr1079;
		case 77: goto tr1080;
		case 83: goto st336;
		case 87: goto tr1082;
		case 100: goto tr1078;
		case 104: goto tr1079;
		case 109: goto tr1080;
		case 115: goto st336;
		case 119: goto tr1082;
		case 1802: goto tr1083;
		case 1851: goto tr1084;
		case 2058: goto tr1083;
		case 2107: goto tr1084;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1077;
	goto tr1073;
tr1087:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st334;
tr1088:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st334;
tr1074:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st334;
tr1075:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st334;
tr1076:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st334;
tr1101:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st334;
tr1102:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st334;
tr1103:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st334;
st334:
	if ( ++p == pe )
		goto _test_eof334;
case 334:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st334;
		case 32: goto st334;
		case 40: goto tr1087;
		case 41: goto tr1088;
		case 1802: goto tr1089;
		case 1851: goto tr1090;
		case 2058: goto tr1089;
		case 2107: goto tr1090;
	}
	goto tr1085;
tr1083:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1411;
tr1089:
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1411;
tr1092:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1411; goto _out;}
			}
		}
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1411;
tr1105:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1411;
st1411:
	if ( ++p == pe )
		goto _test_eof1411;
case 1411:
	goto st0;
tr1090:
	{
		s->buffer_length = 0;
	}
	goto st335;
tr1084:
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st335;
tr1091:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st335;
tr1106:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st335;
st335:
	if ( ++p == pe )
		goto _test_eof335;
case 335:
	if ( (*p) == 10 )
		goto tr1092;
	goto tr1091;
tr1078:
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st336;
tr1079:
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st336;
tr1080:
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st336;
tr1082:
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st336;
st336:
	if ( ++p == pe )
		goto _test_eof336;
case 336:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1074;
		case 32: goto tr1074;
		case 40: goto tr1075;
		case 41: goto tr1076;
		case 1802: goto tr1083;
		case 1851: goto tr1084;
		case 2058: goto tr1083;
		case 2107: goto tr1084;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1093;
	goto tr1073;
tr1095:
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st337;
tr1093:
	{
		s->number64_tmp = s->number64;
	}
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st337;
tr1104:
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZS_NUMBER32_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		s->number64_tmp = s->number64;
	}
	{
		s->number64 = 0;
	}
	{
		// Overflow check: 10*(s->number64) + fc - '0' <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + '0')
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZS_NUMBER64_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st337;
st337:
	if ( ++p == pe )
		goto _test_eof337;
case 337:
	switch( (*p) ) {
		case 68: goto tr1096;
		case 72: goto tr1097;
		case 77: goto tr1098;
		case 83: goto st338;
		case 87: goto tr1100;
		case 100: goto tr1096;
		case 104: goto tr1097;
		case 109: goto tr1098;
		case 115: goto st338;
		case 119: goto tr1100;
	}
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1095;
	goto tr1094;
tr1096:
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st338;
tr1097:
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st338;
tr1098:
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st338;
tr1100:
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZS_NUMBER32_OVERFLOW);
	                  p--; {goto st307;}
	              }
	            }
	goto st338;
st338:
	if ( ++p == pe )
		goto _test_eof338;
case 338:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1101;
		case 32: goto tr1101;
		case 40: goto tr1102;
		case 41: goto tr1103;
		case 1802: goto tr1105;
		case 1851: goto tr1106;
		case 2058: goto tr1105;
		case 2107: goto tr1106;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1104;
	goto tr1073;
tr1070:
	{
		s->buffer_length = 0;
	}
	goto st339;
tr1107:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st339;
st339:
	if ( ++p == pe )
		goto _test_eof339;
case 339:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr1108;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr1107;
	goto tr1065;
st340:
	if ( ++p == pe )
		goto _test_eof340;
case 340:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st341;
		case 32: goto st341;
		case 40: goto tr1111;
		case 41: goto tr1112;
		case 2058: goto tr1113;
		case 2107: goto tr1114;
	}
	goto tr1109;
tr1111:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st341;
tr1112:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st341;
tr1113:
	{
		s->line_counter++;
	}
	goto st341;
tr1144:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 341; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st341;
st341:
	if ( ++p == pe )
		goto _test_eof341;
case 341:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st341;
		case 32: goto st341;
		case 40: goto tr1111;
		case 41: goto tr1112;
		case 42: goto tr1115;
		case 46: goto tr1116;
		case 92: goto tr1117;
		case 95: goto tr1115;
		case 2058: goto tr1113;
		case 2107: goto tr1114;
	}
	if ( _widec < 65 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr1115;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr1115;
	} else
		goto tr1115;
	goto tr1109;
tr1118:
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st342;
tr1125:
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st342;
tr1140:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st342;
tr1115:
	{
		s->dname = s->zone_origin;
	}
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st342;
st342:
	if ( ++p == pe )
		goto _test_eof342;
case 342:
	switch( (*p) ) {
		case 42: goto tr1118;
		case 46: goto tr1119;
		case 92: goto st346;
		case 95: goto tr1118;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr1118;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr1118;
	} else
		goto tr1118;
	goto tr1109;
tr1119:
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st343;
tr1141:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st343;
st343:
	if ( ++p == pe )
		goto _test_eof343;
case 343:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1122;
		case 32: goto tr1122;
		case 40: goto tr1123;
		case 41: goto tr1124;
		case 42: goto tr1125;
		case 45: goto tr1125;
		case 92: goto tr1126;
		case 95: goto tr1125;
		case 1802: goto tr1127;
		case 1851: goto tr1128;
		case 2058: goto tr1127;
		case 2107: goto tr1128;
	}
	if ( _widec < 65 ) {
		if ( 47 <= _widec && _widec <= 57 )
			goto tr1125;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr1125;
	} else
		goto tr1125;
	goto tr1121;
tr1130:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st344;
tr1131:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st344;
tr1122:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st344;
tr1123:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st344;
tr1124:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st344;
st344:
	if ( ++p == pe )
		goto _test_eof344;
case 344:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st344;
		case 32: goto st344;
		case 40: goto tr1130;
		case 41: goto tr1131;
		case 1802: goto tr1132;
		case 1851: goto tr1133;
		case 2058: goto tr1132;
		case 2107: goto tr1133;
	}
	goto tr1121;
tr1127:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1412;
tr1132:
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1412;
tr1135:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 1412; goto _out;}
			}
		}
	}
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto st1412;
st1412:
	if ( ++p == pe )
		goto _test_eof1412;
case 1412:
	goto st0;
tr1133:
	{
		s->buffer_length = 0;
	}
	goto st345;
tr1134:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st345;
tr1128:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		s->buffer_length = 0;
	}
	goto st345;
st345:
	if ( ++p == pe )
		goto _test_eof345;
case 345:
	if ( (*p) == 10 )
		goto tr1135;
	goto tr1134;
tr1126:
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st346;
tr1142:
	{
		s->dname_tmp_length++;
	}
	goto st346;
tr1117:
	{
		s->dname = s->zone_origin;
	}
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st346;
st346:
	if ( ++p == pe )
		goto _test_eof346;
case 346:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1136;
	goto tr1118;
tr1136:
	{
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st347;
st347:
	if ( ++p == pe )
		goto _test_eof347;
case 347:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1138;
	goto tr1137;
tr1138:
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st348;
st348:
	if ( ++p == pe )
		goto _test_eof348;
case 348:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1139;
	goto tr1137;
tr1139:
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st349;
st349:
	if ( ++p == pe )
		goto _test_eof349;
case 349:
	switch( (*p) ) {
		case 42: goto tr1140;
		case 46: goto tr1141;
		case 92: goto tr1142;
		case 95: goto tr1140;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr1140;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr1140;
	} else
		goto tr1140;
	goto tr1137;
tr1116:
	{
		s->dname = s->zone_origin;
	}
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	goto st350;
st350:
	if ( ++p == pe )
		goto _test_eof350;
case 350:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1122;
		case 32: goto tr1122;
		case 40: goto tr1123;
		case 41: goto tr1124;
		case 1802: goto tr1127;
		case 1851: goto tr1128;
		case 2058: goto tr1127;
		case 2107: goto tr1128;
	}
	goto tr1121;
tr1114:
	{
		s->buffer_length = 0;
	}
	goto st351;
tr1143:
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st351;
st351:
	if ( ++p == pe )
		goto _test_eof351;
case 351:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 2058 )
		goto tr1144;
	if ( 1920 <= _widec && _widec <= 2175 )
		goto tr1143;
	goto tr1109;
st352:
	if ( ++p == pe )
		goto _test_eof352;
case 352:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st353;
		case 32: goto st353;
		case 40: goto tr1146;
		case 41: goto tr1147;
		case 2058: goto tr1148;
		case 2107: goto tr1149;
	}
	goto st0;
tr1146:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st353;
tr1147:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st353;
tr1148:
	{
		s->line_counter++;
	}
	goto st353;
tr1202:
	{
		s->buffer[s->buffer_length++] = 0;

		// Execute the comment callback.
		if (s->process.automatic && s->process.comment != NULL) {
			s->process.comment(s);

			// Stop if required from the callback.
			if (s->state == ZS_STATE_STOP) {
				{p++; cs = 353; goto _out;}
			}
		}
	}
	{
		s->line_counter++;
	}
	goto st353;
st353:
	if ( ++p == pe )
		goto _test_eof353;
case 353:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st353;
		case 32: goto st353;
		case 40: goto tr1146;
		case 41: goto tr1147;
		case 2058: goto tr1148;
		case 2107: goto tr1149;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1150;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr1150;
	} else
		goto tr1150;
	goto tr1151;
tr1150:
	{
		rdata_tail = s->r_data;
	}
	{ p--; {stack[top++] = 354;goto st318;} }
	goto st354;
st354:
	if ( ++p == pe )
		goto _test_eof354;
case 354:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1153;
		case 32: goto tr1153;
		case 40: goto tr1154;
		case 41: goto tr1155;
		case 1802: goto tr1156;
		case 1851: goto tr1157;
		case 2058: goto tr1156;
		case 2107: goto tr1158;
	}
	goto tr1152;
tr1161:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st355;
tr1162:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st355;
tr1153:
	{
		size_t len = rdata_tail - s->r_data;
		if (len >= sizeof(s->include_filename)) {
			ERR(ZS_BAD_INCLUDE_FILENAME);
			p--; {goto st307;}
		}

		// Store zero terminated include filename.
		memcpy(s->include_filename, s->r_data, len);
		s->include_filename[len] = '\0';

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st355;
tr1154:
	{
		size_t len = rdata_tail - s->r_data;
		if (len >= sizeof(s->include_filename)) {
			ERR(ZS_BAD_INCLUDE_FILENAME);
			p--; {goto st307;}
		}

		// Store zero terminated include filename.
		memcpy(s->include_filename, s->r_data, len);
		s->include_filename[len] = '\0';

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st355;
tr1155:
	{
		size_t len = rdata_tail - s->r_data;
		if (len >= sizeof(s->include_filename)) {
			ERR(ZS_BAD_INCLUDE_FILENAME);
			p--; {goto st307;}
		}

		// Store zero terminated include filename.
		memcpy(s->include_filename, s->r_data, len);
		s->include_filename[len] = '\0';

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st355;
st355:
	if ( ++p == pe )
		goto _test_eof355;
case 355:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st355;
		case 32: goto st355;
		case 40: goto tr1161;
		case 41: goto tr1162;
		case 42: goto tr1163;
		case 46: goto tr1164;
		case 92: goto tr1165;
		case 95: goto tr1163;
		case 1802: goto tr1166;
		case 1851: goto tr1167;
		case 2058: goto tr1166;
		case 2107: goto tr1168;
	}
	if ( _widec < 65 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr1163;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr1163;
	} else
		goto tr1163;
	goto tr1159;
tr1170:
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st356;
tr1176:
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st356;
tr1189:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st356;
tr1163:
	{
		s->dname = s->r_data;
	}
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	{
		// Check for maximum dname label length.
		if (s->item_length < ZS_MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZS_LABEL_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st356;
st356:
	if ( ++p == pe )
		goto _test_eof356;
case 356:
	switch( (*p) ) {
		case 42: goto tr1170;
		case 46: goto tr1171;
		case 92: goto st360;
		case 95: goto tr1170;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr1170;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr1170;
	} else
		goto tr1170;
	goto tr1169;
tr1171:
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st357;
tr1190:
	{
		s->dname_tmp_length++;
	}
	{
		// Check for maximum dname length overflow after each label.
		// (at least the next label length must follow).
		if (s->dname_tmp_length < ZS_MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZS_DNAME_OVERFLOW);
			p--; {goto st307;}
		}
	}
	goto st357;
st357:
	if ( ++p == pe )
		goto _test_eof357;
case 357:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1173;
		case 32: goto tr1173;
		case 40: goto tr1174;
		case 41: goto tr1175;
		case 42: goto tr1176;
		case 45: goto tr1176;
		case 92: goto tr1177;
		case 95: goto tr1176;
		case 1802: goto tr1178;
		case 1851: goto tr1179;
		case 2058: goto tr1178;
		case 2107: goto tr1179;
	}
	if ( _widec < 65 ) {
		if ( 47 <= _widec && _widec <= 57 )
			goto tr1176;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr1176;
	} else
		goto tr1176;
	goto tr1159;
tr1181:
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st358;
tr1182:
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st358;
tr1173:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->r_data_length = s->dname_tmp_length;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	goto st358;
tr1174:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->r_data_length = s->dname_tmp_length;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == true) {
			ERR(ZS_LEFT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = true;
	}
	goto st358;
tr1175:
	{
		// Enough room for the terminal label is guaranteed (_label_exit).
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	{
		s->r_data_length = s->dname_tmp_length;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		if (s->multiline == false) {
			ERR(ZS_RIGHT_PARENTHESIS);
			p--; {goto st307;}
		}
		s->multiline = false;
	}
	goto st358;
st358:
	if ( ++p == pe )
		goto _test_eof358;
case 358:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if (
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if (
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st358;
		case 32: goto st358;
		case 40: goto tr1181;
		case 41: goto tr1182;
		case 1802: goto tr1166;
		case 1851: goto tr1167;
		case 2058: goto tr1166;
		case 2107: goto tr1167;
	}
	goto tr89;
tr1156:
	cs = 1413;
	{
		size_t len = rdata_tail - s->r_data;
		if (len >= sizeof(s->include_filename)) {
			ERR(ZS_BAD_INCLUDE_FILENAME);
			p--; {goto st307;}
		}

		// Store zero terminated include filename.
		memcpy(s->include_filename, s->r_data, len);
		s->include_filename[len] = '\0';

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	{
		s->buffer[0] = 0;
		s->buffer_length = 0;
	}
	{
		// Extend relative file path.
		if (s->include_filename[0] != '/') {
			int ret = snprintf((char *)(s->buffer), sizeof(s->buffer),
			                   "%s/%s", s->path, s->include_filename);
			if (ret <= 0 || ret >= sizeof(s->buffer)) {
				ERR(ZS_BAD_INCLUDE_FILENAME);
				p--; {goto st307;}
			}
			memcpy(s->include_filename, s->buffer, ret + 1);
		}

		// Origin conversion from wire to text form in \DDD notation.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
			                  (char *)s->buffer);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
			                  (char *)s->buffer);
		}

		// Let the caller to solve the include.
		if (s->process.automatic) {
			// Create new scanner for included zone file.
			zs_scanner_t *ss = malloc(sizeof(zs_scanner_t));
			if (ss == NULL) {
				ERR(ZS_UNPROCESSED_INCLUDE);
				p--; {goto st307;}
			}

			// Parse included zone file.
			if (zs_init(ss, (char *)s->buffer, s->default_class,
			            s->default_ttl) != 0 ||
			    zs_set_input_file(ss, (char *)(s->include_filename)) != 0 ||
			    zs_set_processing(ss, s->process.record, s->process.error,
			                      s->process.data) != 0 ||
			    zs_parse_all(ss) != 0) {
				// File internal errors are handled by error callback.
				if (ss->error.counter > 0) {
					s->error.counter += ss->error.counter;
					ERR(ZS_UNPROCESSED_INCLUDE);
				// General include file error.
				} else {
					ERR(ss->error.code);
				}
				zs_deinit(ss);
				free(ss);
				p--; {goto st307;}
			}
			zs_deinit(ss);
			free(ss);
		} else {
			s->state = ZS_STATE_INCLUDE;
			p--; cs = 1396; {p++; goto _out;}
		}
	}
	{
		p--; {cs = stack[--top];goto _again;}
	}
	{
		s->line_counter++;
	}
	goto _again;
tr1166:
	cs = 1413;
	{
		// Extend relative file path.
		if (s->include_filename[0] != '/') {
			int ret = snprintf((char *)(s->buffer), sizeof(s->buffer),
			                   "%s/%s", s->path, s->include_filename);
			if (ret <= 0 || ret >= sizeof(s->buffer)) {
				ERR(ZS_BAD_INCLUDE_FILENAME);
				p--; {goto st307;}
			}
			memcpy(s->include_filename, s->buffer, ret + 1);
		}

		// Origin conversion from wire to text form in \DDD notation.
		if (s->dname ==