/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include <config.h>
#include <stdlib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <glib/gstdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "main_window.h"
#include "main_window_cb.h"
#include "gwget_data.h"
#include "utils.h"
#include "systray.h"
#include "new_window.h"

static gboolean there_are_completed_on_startup = FALSE;
 
/* Global Preferences */
Preferences gwget_pref;

static void 
show_prefered_columns(void);

static gboolean
gwget_destination_file_exists(GwgetData *data);

static gboolean
view_selection_func (GtkTreeSelection *selection,
                        GtkTreeModel     *model,
                        GtkTreePath      *path,
                        gboolean          path_currently_selected,
                        gpointer          userdata);

void 
main_window(void) 
{
	GtkWidget * window = NULL;
	gchar *builder_file = NULL;
	GtkWidget *treev,*toolbar,*menu_item, *combo;
	GtkTreeSelection *select;
	GError* error = NULL;

	if (!builder) {
		builder_file = g_build_filename(DATADIR,"gwget.ui",NULL);
		builder = gtk_builder_new();
		if (!gtk_builder_add_from_file (builder, builder_file, &error)) {
				g_warning ("Couldn't load builder file: %s", builder_file);
				g_error_free (error);
		}
		gtk_builder_connect_signals(builder, NULL);
	}
	
	window = GTK_WIDGET (gtk_builder_get_object(builder,"main_window"));
	treev = GTK_WIDGET (gtk_builder_get_object(builder,"treeview1"));
	model = create_model();
	gtk_tree_view_set_model(GTK_TREE_VIEW(treev),GTK_TREE_MODEL(model));
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treev));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	/* Set a callback when a selection is made for update the menu Download */
	gtk_tree_selection_set_select_function(select, view_selection_func, NULL, NULL);

	/*
	 * Update window size in gsettings on resize 
	 */
	g_signal_connect(GTK_WIDGET(window),
			 "configure-event",
			 G_CALLBACK(gwget_remember_window_size_and_position),
			 NULL);
	
	/* add the columns titles to the tree view */
	add_columns (GTK_TREE_VIEW (treev));

	settings = g_settings_new("org.gwget");
	
	/* gwget_pref.download_dir=NULL; */
	gwget_get_defaults_from_gsettings();
	
	/* Drag and drop set up */
	gtk_drag_dest_set(GTK_WIDGET(window), 
			  GTK_DEST_DEFAULT_ALL | GTK_DEST_DEFAULT_HIGHLIGHT,
			  dragtypes, 
			  sizeof(dragtypes) / sizeof(dragtypes[0]),
                          GDK_ACTION_COPY);
						
	g_signal_connect(G_OBJECT(window), 
			 "drag_data_received",
			 G_CALLBACK(on_gwget_drag_received),
			 GUINT_TO_POINTER(dnd_type));	
					 
	/* Show the toolbar ? */
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (gtk_builder_get_object(builder,"view_toolbar")),gwget_pref.view_toolbar);
	toolbar = GTK_WIDGET (gtk_builder_get_object(builder,"toolbar1"));
	menu_item = GTK_WIDGET (gtk_builder_get_object(builder,"view_toolbar"));
	if (gwget_pref.trayonly) {
		gtk_widget_show(GTK_WIDGET(window));
		gtk_widget_hide(GTK_WIDGET(window));
	}
	else 
		gtk_widget_show(GTK_WIDGET(window));

	if (!gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu_item))) {
		gtk_widget_hide(GTK_WIDGET(toolbar));
	} else {
		gtk_widget_show(GTK_WIDGET(toolbar));
	}

	/* Show the statusbar ? */
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (gtk_builder_get_object(builder,"view_statusbar")), gwget_pref.view_statusbar);
	menu_item = GTK_WIDGET (gtk_builder_get_object(builder,"view_statusbar"));
	if (gwget_pref.view_statusbar)
	{
		gtk_widget_show (GTK_WIDGET(GTK_WIDGET (gtk_builder_get_object(builder, "statusbar"))));
	} else {
		gtk_widget_hide (GTK_WIDGET(GTK_WIDGET (gtk_builder_get_object(builder, "statusbar"))));
	}

	if (there_are_completed_on_startup) {
		gtk_widget_set_sensitive (GTK_WIDGET (gtk_builder_get_object (builder, "clear_button")), TRUE);
	} else {
		gtk_widget_set_sensitive (GTK_WIDGET (gtk_builder_get_object (builder, "clear_button")), FALSE);
	}

	systray_load();
	gwget_pref.docked=TRUE;

	/* Create the model for the "save in" option in new download dialog */
	save_in_model = (GtkTreeModel*)gtk_list_store_new (1, G_TYPE_STRING);
	combo = GTK_WIDGET (gtk_builder_get_object (builder, "save_in_comboboxentry"));
	gtk_combo_box_set_model(GTK_COMBO_BOX(combo), save_in_model);
	gtk_combo_box_set_entry_text_column (GTK_COMBO_BOX(combo), 0);
	gtk_entry_set_text(GTK_ENTRY(gtk_bin_get_child(GTK_BIN(combo))), gwget_pref.download_dir);
		
}

static 
gboolean view_selection_func (GtkTreeSelection *selection,
			GtkTreeModel     *model,
			GtkTreePath      *path,
			gboolean          path_currently_selected,
			gpointer          userdata)
{
	GtkTreeIter iter;

	if (gwget_data_get_selected() == NULL) {
		return TRUE;
	}

	if (gtk_tree_model_get_iter(model, &iter, path))
	{
		gchar *name;

		gtk_tree_model_get(model, &iter, FILENAME_COLUMN, &name, -1);

		g_free(name);
	}

	return TRUE; /* allow selection state to change */
}




static gboolean
gwget_destination_file_exists(GwgetData *data)
{
	struct stat s;
	
	return ( g_stat(data->local_filename,&s) == 0 );
}


void 
gwget_get_defaults_from_gsettings(void)
{
	gint num_dl, i;
	gint64 total_size;
	GwgetData *data;
	gchar *path,*url,*dir,*name;
	DlState state;
	gint default_width, default_height;
	GSettingsSchemaSource *schema_source;
	GSettingsSchema *http_proxy_schema;
	GSettings *http_proxy_settings,*download_settings;
	
	gwget_pref.http_proxy=g_settings_get_string(settings,"http-proxy");
	gwget_pref.http_proxy_port=g_settings_get_int(settings,"http-proxy-port");
	gwget_pref.proxy_uses_auth=g_settings_get_boolean(settings,"proxy-uses-auth");
	gwget_pref.proxy_user=g_settings_get_string(settings,"proxy-user");
	gwget_pref.proxy_password=g_settings_get_string(settings,"proxy-password");
	
	schema_source=g_settings_schema_source_get_default();
	http_proxy_schema=g_settings_schema_source_lookup(schema_source,"org.gnome.system.proxy.http",FALSE);
	
	if (http_proxy_schema != NULL) {
		http_proxy_settings=g_settings_new("org.gnome.system.proxy.http");
		gwget_pref.gnome_http_proxy=g_settings_get_string(http_proxy_settings,"host");
		gwget_pref.gnome_http_proxy_port=g_settings_get_int(http_proxy_settings,"port");
		gwget_pref.gnome_proxy_uses_auth=g_settings_get_boolean(http_proxy_settings,"use-authentication");
		gwget_pref.gnome_proxy_user=g_settings_get_string(http_proxy_settings,"authentication-user");
		gwget_pref.gnome_proxy_password=g_settings_get_string(http_proxy_settings,"authentication-password");
		gwget_pref.gnome_use_proxy=g_settings_get_boolean(http_proxy_settings,"enabled");
		g_settings_schema_unref(http_proxy_schema);
	}
	
	gwget_pref.network_mode=g_settings_get_string(settings,"network-mode");
	gwget_pref.download_dir=g_settings_get_string(settings,"download-dir");
	gwget_pref.num_retries=g_settings_get_int(settings,"num-retries");
	gwget_pref.resume_at_start=g_settings_get_boolean(settings,"resume-at-start");
	gwget_pref.open_after_dl=g_settings_get_boolean(settings,"open-after-dl");
	gwget_pref.no_create_directories=g_settings_get_boolean(settings,"no-create-directories");
	gwget_pref.follow_relative=g_settings_get_boolean(settings,"follow-relative");
	gwget_pref.convert_links = g_settings_get_boolean(settings,"convert-links");	
	gwget_pref.dl_page_requisites = g_settings_get_boolean(settings,"dl-page-requisites");
	gwget_pref.max_depth=g_settings_get_int(settings,"max-depth");
	gwget_pref.view_actual_size=g_settings_get_boolean(settings,"view-actual-size");
	gwget_pref.view_total_size=g_settings_get_boolean(settings,"view-total-size");
	gwget_pref.view_percentage=g_settings_get_boolean(settings,"view-percentage");
	gwget_pref.view_elapse_time=g_settings_get_boolean(settings,"view-elapse-time");
	gwget_pref.view_down_speed=g_settings_get_boolean(settings,"view-down-speed");
	gwget_pref.view_toolbar=g_settings_get_boolean(settings,"view-toolbar");
	gwget_pref.view_statusbar=g_settings_get_boolean(settings,"view-statusbar");
	gwget_pref.limit_speed = g_settings_get_boolean(settings,"limit-speed");
	gwget_pref.max_speed=g_settings_get_int(settings,"max-speed");
	gwget_pref.limit_simultaneousdownloads = g_settings_get_boolean(settings,"limit-simultaneousdownloads");
	gwget_pref.max_simultaneousdownloads=g_settings_get_int(settings,"max-simultaneousdownloads");
	gwget_pref.ask_save_each_dl = g_settings_get_boolean(settings,"ask-save-each-dl");
	
	if (!gwget_pref.download_dir) {
		gwget_pref.download_dir = g_strdup(g_get_home_dir());
		g_settings_set_string (settings, "download-dir", g_strdup(g_get_home_dir()));
	}

	if (!gwget_pref.network_mode) {
		gwget_pref.network_mode = g_strdup("direct");
		g_settings_set_string (settings, "network-mode", "direct");
	}

	/* Put in the list of save_in_paths (used by new dialog window) the initial download directory */
	save_in_paths = g_list_append (save_in_paths, gwget_pref.download_dir);
		
	num_dl=g_settings_get_int(settings,"n-downloads");
	for (i=0;i<num_dl;i++) {
		path=g_strdup_printf("/org/gwget2/downloads-data/%d/",i);
		download_settings = g_settings_new_with_path("org.gwget.download", path);
		url=g_settings_get_string(download_settings,"url");
		dir=g_settings_get_string(download_settings,"dir");

		if (url!=NULL) {
			data=gwget_data_create(url,dir);
		
			name=g_settings_get_string(download_settings,"filename");
			gwget_data_set_filename(data,name);
		
			state=g_settings_get_int(download_settings,"state"); 
		
			total_size = atoll(g_settings_get_string (download_settings, "file-size"));
			gwget_data_set_total_size (data, total_size);

			if ( state != DL_COMPLETED ) {
			/*
			 * If the download is not completed - add it , no questions
			 */	
				new_download(data);
			} else {
			/* 
			 * If the download is completed , then:
			 * if the file we want to write to is missing , the user
			 * has (re)moved it , so quietly forget download , otherwise
			 * add it
		 	 */
				if ( !gwget_destination_file_exists(data) ) {
					/*
					 * We do not add download, the gwget_remember_downloads
					 * call after the cycle will flush gsettings
					 */
					continue;
				} else {
					there_are_completed_on_startup = TRUE;
					new_download(data);
					gwget_data_set_state_no_sync(data,DL_COMPLETED);
					continue;
				}
			}
		}
		    		
		g_object_unref (download_settings);

		gwget_data_set_state_no_sync(data,DL_NOT_RUNNING); 
		if (gwget_pref.resume_at_start && data->state!=DL_COMPLETED) {
			gwget_data_start_download(data);
		} 
		/* FIXME: put all the cases (error, retriving...) */
		
		if (state==DL_COMPLETED) {
			gwget_data_set_state_no_sync(data, state);
		}
	}
	/*
	 * We may have altered the download list so we sync gsettings
	 */
	gwget_remember_downloads();
	
	show_prefered_columns();

	/* Default width and height */
	default_width=g_settings_get_int (settings,"default-width");
	default_height=g_settings_get_int (settings,"default-height");
	gtk_window_resize (GTK_WINDOW (gtk_builder_get_object(builder,"main_window")),default_width,default_height);
	
	/* Default position */
	gtk_window_move(GTK_WINDOW(GTK_WIDGET (gtk_builder_get_object(builder,"main_window"))),
					g_settings_get_int (settings,"position-x"),
					g_settings_get_int (settings,"position-y")
					);
}			


void 
on_main_window_delete_event(GtkWidget *widget, gpointer data)
{
	if (gwget_pref.docked) {
		gtk_widget_hide(widget);
	} else {
		gwget_quit();
	}
}

GtkTreeModel *
create_model (void)
{
	GtkListStore *model;

	model = gtk_list_store_new (NUM_COLUMNS,
				    GDK_TYPE_PIXBUF,
				    G_TYPE_STRING, /* File name */
			 	    G_TYPE_STRING, /* URL */
				    G_TYPE_STRING, /* State */
				    G_TYPE_STRING, /* Current size */
				    G_TYPE_STRING, /* Total Size */
				    G_TYPE_INT,    /* Percentage */
				    G_TYPE_STRING, /* Percentage */
				    G_TYPE_STRING, /* Elapse Time */
				    G_TYPE_STRING, /* Current time  */
				    G_TYPE_STRING, /* Estimated time  */
				    G_TYPE_STRING, /* Remain Time */
				    /* Not viewable columns */ 
				    G_TYPE_INT,    /* Pid */
				    G_TYPE_INT,	   /* State int column */
				    G_TYPE_STRING  /* Speed */
				);
				  
	return GTK_TREE_MODEL (model);
}

void
add_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	PangoContext *context;
	PangoLayout *layout;
	
	
	/* File Name Column */
	
	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start (column,
					renderer,
					FALSE);
	gtk_tree_view_column_add_attribute(column,
					  renderer,
					  "pixbuf", 
					  IMAGE_COLUMN);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column,
					renderer,
					FALSE);
	gtk_tree_view_column_add_attribute (column,
					  renderer, 
					  "text",
					  FILENAME_COLUMN);
	
	gtk_tree_view_column_set_title (GTK_TREE_VIEW_COLUMN(column),_("File Name"));
	
	context = gtk_widget_get_pango_context(GTK_WIDGET(treeview));
	layout = pango_layout_new (PANGO_CONTEXT(context));
	pango_layout_set_ellipsize (PANGO_LAYOUT(layout), PANGO_ELLIPSIZE_END);
	
	gtk_tree_view_column_set_sort_column_id (column, FILENAME_COLUMN);
	gtk_tree_view_column_set_resizable (column, TRUE); 
	gtk_tree_view_column_set_reorderable (column, TRUE);
	gtk_tree_view_append_column (treeview, column);


	
	/* State Column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("State"),
							  renderer,
							  "text",
							  STATE_COLUMN,
							  NULL);
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_sort_column_id (column, STATE_COLUMN);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_reorderable (column, TRUE);
	gtk_tree_view_append_column (treeview, column);
	
	/* Current Size */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Current Size"),
							renderer,
						     	"text",
							CURRENTSIZE_COLUMN,
							NULL);

	gtk_tree_view_column_set_sort_column_id (column, CURRENTSIZE_COLUMN);
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_reorderable (column, TRUE);
 	gtk_tree_view_append_column (treeview, column);
	
	/* Total Size */
	renderer = gtk_cell_renderer_text_new ();
    	column = gtk_tree_view_column_new_with_attributes (_("Total Size"),
							renderer,
							"text",
							TOTALSIZE_COLUMN,
							NULL);
	gtk_tree_view_column_set_sort_column_id (column, TOTALSIZE_COLUMN);
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_reorderable (column, TRUE);
 	gtk_tree_view_append_column (treeview, column);
	
	/* Percentage */
	renderer = gtk_cell_renderer_progress_new();
	column = gtk_tree_view_column_new_with_attributes (_("Percentage"),
							renderer,
							"value", 
							PERCENTAGE_COLUMN,
							NULL);

	gtk_tree_view_column_set_title (column, (_("Percentage"))); 
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_sort_column_id (column, PERCENTAGE_COLUMN);
	gtk_tree_view_column_set_reorderable (column, TRUE);	
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview),column); 
		
	/* Elapse Time */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Elapsed Time"),
							renderer,
							"text",
							ELAPSETIME_COLUMN,
							NULL);
	gtk_tree_view_column_set_sort_column_id (column, ELAPSETIME_COLUMN);
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_reorderable (column, TRUE);
	gtk_tree_view_append_column (treeview, column);
	
	/* REMAINTIME_COLUMN */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Remaining Time"),
														renderer,
														"text",
														REMAINTIME_COLUMN,
														NULL);
	gtk_tree_view_column_set_sort_column_id (column, REMAINTIME_COLUMN);
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_reorderable (column, TRUE);
	gtk_tree_view_append_column (treeview, column);
	
	/* Speed */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Speed"),
														renderer,
														"text",
														SPEED_COLUMN,
														NULL);
	gtk_tree_view_column_set_sort_column_id (column, SPEED_COLUMN);
	gtk_tree_view_column_set_resizable(column, TRUE); 
	gtk_tree_view_column_set_reorderable (column, TRUE);
	gtk_tree_view_append_column (treeview, column);
	
}

void 
on_gwget_drag_received (GtkWidget * widget, GdkDragContext * context, int x,
							int y, GtkSelectionData * seldata, guint info,	
							guint time, gpointer data)
{
	gchar *file;
	gchar **uris;
	GList *files = NULL;
	GwgetData *gwgetdata;
	gboolean drop_ok;
	int i;
	
	g_return_if_fail(gwget_pref.download_dir != NULL);

	dnd_type = GPOINTER_TO_UINT(data);
	
	drop_ok = FALSE;
	if (dnd_type==TARGET_URI_LIST) {
		uris = g_uri_list_extract_uris ((gchar *)gtk_selection_data_get_data(seldata));
		for (i=0; uris[i] != NULL; i++) {
		  files = g_list_prepend (files, uris[i]);
		}
		g_strfreev (uris); 
		file = g_list_first(files)->data;
		drop_ok = TRUE;
	} else if (dnd_type==TARGET_NETSCAPE_URL) {
		file=((gchar *) (gtk_selection_data_get_data(seldata)));
		drop_ok = TRUE;
	} else 	{
		gtk_drag_finish(context, FALSE, TRUE, time);
	}

	if (drop_ok && gwget_pref.ask_save_each_dl) {
		create_new_window_with_url (file);
	} else if (drop_ok && !gwget_pref.ask_save_each_dl) {
		gwgetdata = gwget_data_create(file, gwget_pref.download_dir);
		if (gwgetdata) {
			new_download(gwgetdata);
			gwget_data_start_download(gwgetdata);
		} else {
			run_dialog_error(_("Error starting the download"), _("There was an unexpected error starting the download"));
		}
		gtk_drag_finish(context, TRUE, FALSE, time);
	}
}

static void 
show_prefered_columns(void)
{	
	GtkWidget *treev,*column,*checkitem;
	gchar *builder_file = NULL;
	GError* error = NULL;
	
	if (!builder_pref) {
		builder_file = g_build_filename (DATADIR, "preferences.ui", NULL);
		builder_pref = gtk_builder_new();
		if (!gtk_builder_add_from_file (builder_pref, builder_file, &error)) {
				g_warning ("Couldn't load builder file: %s", builder_file);
				g_error_free (error);
		}
		gtk_builder_connect_signals (builder_pref, NULL);
	}
	
	treev = GTK_WIDGET (gtk_builder_get_object(builder,"treeview1"));
	
	column=(GtkWidget *)gtk_tree_view_get_column(GTK_TREE_VIEW(treev),CURRENTSIZE_COLUMN-2);
	checkitem=GTK_WIDGET (gtk_builder_get_object(builder_pref,"check_actual_size"));
	if (gwget_pref.view_actual_size) {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column),TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	} else {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column),FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),FALSE);
	}
	
	column=(GtkWidget *)gtk_tree_view_get_column(GTK_TREE_VIEW(treev),TOTALSIZE_COLUMN-2);
	checkitem=GTK_WIDGET (gtk_builder_get_object(builder_pref,"check_total_size"));
	if (gwget_pref.view_total_size) {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column),TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	} else {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),FALSE);
	}
	
	column=(GtkWidget *)gtk_tree_view_get_column(GTK_TREE_VIEW(treev),PERCENTAGE_COLUMN-2);
	checkitem=GTK_WIDGET (gtk_builder_get_object(builder_pref,"check_percentage"));
	if (gwget_pref.view_percentage) {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem), TRUE);
	} else {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem), FALSE);
	}
	
	column=(GtkWidget *)gtk_tree_view_get_column(GTK_TREE_VIEW(treev),ELAPSETIME_COLUMN-3);
	checkitem=GTK_WIDGET (gtk_builder_get_object(builder_pref,"check_elapse_time"));
	if (gwget_pref.view_elapse_time) {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	} else {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),FALSE);
	}
	
	column=(GtkWidget *)gtk_tree_view_get_column(GTK_TREE_VIEW(treev),REMAINTIME_COLUMN-5);
	checkitem=GTK_WIDGET (gtk_builder_get_object(builder_pref,"check_rem_time"));
	if (gwget_pref.view_rem_time) {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	} else {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	}
	
	column=(GtkWidget *)gtk_tree_view_get_column(GTK_TREE_VIEW(treev),SPEED_COLUMN-7);
	checkitem=GTK_WIDGET (gtk_builder_get_object(builder_pref,"check_down_speed"));
	if (gwget_pref.view_down_speed) {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem), TRUE);
	} else {
		gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem), FALSE);
	}
	
	checkitem = GTK_WIDGET (gtk_builder_get_object(builder_pref,"limit_speed_check"));
	if (gwget_pref.limit_speed) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),FALSE);
	}
	checkitem = GTK_WIDGET (gtk_builder_get_object(builder_pref,"limit_speed_spin"));
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(checkitem), (gdouble)gwget_pref.max_speed);


	checkitem = GTK_WIDGET (gtk_builder_get_object(builder_pref,"limit_simultaneousdownloads_check"));
	if (gwget_pref.limit_simultaneousdownloads) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),TRUE);
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkitem),FALSE);
	}
	checkitem = GTK_WIDGET (gtk_builder_get_object(builder_pref,"limit_simultaneousdownloads_spin"));
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(checkitem), (gdouble)gwget_pref.max_simultaneousdownloads);

}

/*
 * This is called as a GTK singal handler , so the return value tells
 * the singal was not handled and should propagate further
 */
gboolean
gwget_remember_window_size_and_position(void)
{
	GtkWidget *main_window;
	GtkAllocation allocation;
	gint position_x,position_y;

	main_window=GTK_WIDGET (gtk_builder_get_object(builder,"main_window"));

	if (!gtk_widget_get_visible(main_window))
		return FALSE;

	/* Remember the size of the window */
	gtk_widget_get_allocation (main_window, &allocation);
	g_settings_set_int (settings, "default-height", allocation.height);
	g_settings_set_int (settings, "default-width", allocation.width);
	
	/* Remember the position */
	gtk_window_get_position(GTK_WINDOW(main_window), &position_x, &position_y);
	g_settings_set_int (settings, "position-x", position_x);
	g_settings_set_int (settings, "position-y", position_y);

	return FALSE;
}

/*
 * The return value is used in gwget_quit
 */
gboolean 
gwget_remember_downloads(void)
{
	gchar *url,*path;
	GwgetData *gwgetdata;
	GtkTreeIter iter;
	gint i,length;
	gboolean running = FALSE;
	gchar down_size[2048];
	GSettings *download_settings;
	
	/* calculate the number of items in the treeview */
	length=gtk_tree_model_iter_n_children(GTK_TREE_MODEL(model),NULL);
	
	/* Save the number of current downloads in the treev */
	/* When load again we can known the number of directories to load */
	g_settings_set_int(settings,"n-downloads",length);
		
	gtk_tree_model_get_iter_first(model,&iter);
	/* Save current downloads in GSettings */
	/* Calculate if there are any dl in retriving state */
	for (i=0;i<length;i++) {
		gtk_tree_model_get (model, &iter, URL_COLUMN, &url, -1);
		gwgetdata=g_object_get_data(G_OBJECT(model),url);
	
		path=g_strdup_printf("/org/gwget2/downloads-data/%d/",i);
		download_settings = g_settings_new_with_path("org.gwget.download", path);

		g_settings_set_string(download_settings,"url",gwgetdata->url);
		g_settings_set_string(download_settings,"filename",gwgetdata->filename);
		g_settings_set_string(download_settings,"dir",gwgetdata->dir);
		g_settings_set_int(download_settings,"state",gwgetdata->state);

		g_snprintf(down_size,2047, "%" G_GINT64_FORMAT,gwgetdata->total_size);
		g_settings_set_string(download_settings,"file-size",down_size);
		    		
		g_object_unref (download_settings);
		
		if (gwgetdata->log_tag != -1) 
		{
			running=TRUE;
		}
		
		gtk_tree_model_iter_next(model,&iter);
	}

	return running;
}

void
gwget_quit(void)
{
	gboolean running;
	gint response;
	
	running = gwget_remember_downloads();
	
	if (running) {
		response = run_dialog(_("Stop current downloads?"),
				      _("There is at least one active download left. Really cancel all running transfers?"), _("No"), _("Yes"));
		if (response == GTK_RESPONSE_OK) {
			stop_all_downloads();
			g_settings_sync();
			gtk_main_quit();
		}
	} else {
		g_settings_sync();
		gtk_main_quit();
	}
}

