#include "trezor.h"
#include "helpers.h"
#include "ethereum/tx.h"
#include <QDebug>
#include <QByteArray>

namespace Trezor {

    TrezorDevice::TrezorDevice() : QObject(nullptr),
        fDevice(), fWorker(fDevice), fQueue(), fDeviceID(), fDevicePresent(false)
    {
        connect(&fWorker, &TrezorWorker::finished, this, &TrezorDevice::workerDone);
    }

    TrezorDevice::~TrezorDevice()
    {
    }

    void TrezorDevice::checkPresence()
    {
        // don't check while busy
        if ( getBusy() ) {
            return;
        }

        bool wasPresent = fDevicePresent;
        fDevicePresent = fDevice.isPresent();

        if ( wasPresent != fDevicePresent ) {
            emit presenceChanged(fDevicePresent);

            // we if inserted
            if ( fDevicePresent ) {
                initialize();
            } else { // if we removed
                fDeviceID = QString();
                fDevice.close();
                emit initializedChanged(false);
            }
        }
    }

    bool TrezorDevice::isPresent()
    {
        return fDevicePresent;
    }

    bool TrezorDevice::isInitialized()
    {
        return !fDeviceID.isEmpty();
    }

    void TrezorDevice::initialize()
    {
        if ( !isPresent() ) {
            return;
        }

        try {
            fDevice.init();
        } catch ( Trezor::Wire::Device::wire_error& err ) {
            return bail("Error opening TREZOR device: " + QString(err.what()));
        }

        Initialize request;
        sendMessage(request, MessageType_Initialize);
    }

    void TrezorDevice::onDeviceInserted()
    {
        checkPresence();
    }

    void TrezorDevice::onDeviceRemoved()
    {
        checkPresence();
    }

    void TrezorDevice::onDirectoryChanged(const QString &path)
    {
        Q_UNUSED(path)
        initialize();
    }

    void TrezorDevice::getAddress(const HDPath& hdPath)
    {
        if ( !isPresent() ) {
            bail("getAddress called when trezor not present");
            return;
        }

        if ( !hdPath.valid() ) {
            bail("hd path invalid");
            return;
        }

        EthereumGetAddress request;
        request.set_show_display(false);

        quint32 segment;
        int index = 0;
        while ( hdPath.getSegment(index++, segment) ) {
            request.add_address_n(segment);
        }

        sendMessage(request, MessageType_EthereumGetAddress, hdPath.toString());        
    }

    const QString TrezorDevice::getDeviceID() const
    {
        return fDeviceID;
    }

    const QString TrezorDevice::getVersion() const
    {
        return fVersion;
    }

    void TrezorDevice::submitPin(const QString &pin)
    {
        PinMatrixAck request;
        request.set_pin(pin.toUtf8().data());
        sendMessage(request, MessageType_PinMatrixAck);
    }

    void TrezorDevice::submitPassphrase(const QString &pw)
    {
        PassphraseAck request;
        request.set_passphrase(pw.toStdString());
        sendMessage(request, MessageType_PassphraseAck);
    }

    void TrezorDevice::signTransaction(quint32 chaindID, const QString& hdPath, const QString& from, const QString &to, const QString &valStr,
                                       quint64 nonce, const QString &gas, const QString &gasPrice, const QString &data)
    {
        fPendingTx.init(from, to, valStr, nonce, gas, gasPrice, data);

        EthereumSignTx request;

        HDPath path(hdPath);
        quint32 segment;
        int index = 0;
        while ( path.getSegment(index++, segment) ) {
            request.add_address_n(segment);
        }

        request.set_chain_id(chaindID);
        request.set_to(fPendingTx.toStr().toStdString());
        if ( fPendingTx.hasValue() ) {
            request.set_value(fPendingTx.valueBytes());
        }
        if ( nonce > 0 ) { // seems like a protobuf/trezor bug
            request.set_nonce(fPendingTx.nonceBytes());
        }
        request.set_gas_limit(fPendingTx.gasBytes());
        request.set_gas_price(fPendingTx.gasPriceBytes());

        if ( fPendingTx.dataByteSize() > 0 ) {
            request.set_data_length(fPendingTx.dataByteSize());
            request.set_data_initial_chunk(fPendingTx.dataNext(1024));
        }

        sendMessage(request, MessageType_EthereumSignTx);
    }

    void TrezorDevice::workerDone()
    {
        handleResponse(fWorker.getReply()); // handle first so we can have "inserts" for the meta workflows
        sendNext();
        emit busyChanged(getBusy());
    }

    bool TrezorDevice::getBusy() const
    {
        return fWorker.isRunning() || !fQueue.empty();
    }

    void TrezorDevice::cancel()
    {
        Cancel request;
        fQueue.unlock(); // ensure we don't try and wait for user response
        sendMessage(request, MessageType_Cancel);
    }

    void TrezorDevice::bail(const QString& err)
    {
        if ( fWorker.isRunning() ) {
            fWorker.terminate();
        }

        if ( !fQueue.empty() ) {
            fQueue.clear();
        }

        emit error(err);
    }

    const Wire::Message TrezorDevice::serializeMessage(google::protobuf::Message &msg, MessageType type, const QVariant& index)
    {
        Wire::Message msg_wire;
        msg_wire.id = type;
        msg_wire.index = index;
        msg_wire.data = std::vector<uint8_t>(msg.ByteSizeLong());
        if ( !msg.SerializeToArray(msg_wire.data.data(), msg.ByteSizeLong()) ) {
            bail("Could not serialize getAddress msg");
            return msg_wire;
        }

        return msg_wire;
    }

    bool TrezorDevice::parseMessage(const Wire::Message &msg_in, google::protobuf::Message& parsed) const
    {
        return parsed.ParseFromArray(msg_in.data.data(), msg_in.data.size());
    }

    void TrezorDevice::sendMessage(google::protobuf::Message& msg, MessageType type, const QVariant index)
    {
        const Wire::Message wireMsg = serializeMessage(msg, type, index);

        if ( type == MessageType_ButtonAck ||
             type == MessageType_PassphraseAck ||
             type == MessageType_PassphraseStateAck ||
             type == MessageType_EthereumTxAck ||
             type == MessageType_Cancel ) { // these msgs need to always go right after, no matter what we have queued already
            fQueue.prepend(wireMsg); // no need to check for lock here
        } else {
            fQueue.push(wireMsg);
        }

        sendNext();
    }

    void TrezorDevice::sendNext()
    {
        if ( !fDevice.isInitialized() ) {
            return;
        }

        if ( fWorker.isRunning() ) {
            return;
        }

        Wire::Message request;

        if ( !fQueue.pop(request)) {
            return;
        }

        fWorker.setRequest(request);
        fWorker.start();
        emit busyChanged(getBusy());
    }

    void TrezorDevice::handleResponse(const Wire::Message &msg_in)
    {
        switch ( msg_in.id ) {
            case MessageType_Failure: handleFailure(msg_in); return;
            case MessageType_PinMatrixRequest: handleMatrixRequest(msg_in); return;
            case MessageType_ButtonRequest: handleButtonRequest(msg_in); return;
            case MessageType_PassphraseRequest: handlePassphrase(msg_in); return;
            case MessageType_PassphraseStateRequest: handlePassphraseStateRequest(msg_in); return;
            case MessageType_Features: handleFeatures(msg_in); return;
            case MessageType_EthereumAddress: handleAddress(msg_in); return;
            case MessageType_EthereumTxRequest: handleTxRequest(msg_in); return;
        }

        bail("Unknown msg response: " + QString::number(msg_in.id));
    }

    void TrezorDevice::handleFailure(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_Failure ) {
            bail("Unexpected failure response: " + QString::number(msg_in.id));
            return;
        }

        Failure response;
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing failure response");
            return;
        }

        const QString error = QString::fromStdString(response.message());
        emit failure(error);
        fQueue.clear(); // we cannot continue after failure!
    }

    void TrezorDevice::handleMatrixRequest(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_PinMatrixRequest ) {
            bail("Unexpected pin matrix response: " + QString::number(msg_in.id));
            return;
        }

        PinMatrixRequest response;
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing matrix response");
            return;
        }

        emit matrixRequest(response.type());
        fQueue.lock(MessageType_PinMatrixAck, fWorker.getIndex()); // we need to wait for this call before making others, saving the index
    }

    void TrezorDevice::handleButtonRequest(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_ButtonRequest ) {
            bail("Unexpected button response: " + QString::number(msg_in.id));
            return;
        }

        ButtonRequest response;
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing button response");
            return;
        }
        emit buttonRequest(response.code());
        // we have to ack right away, worker will wait for actual reply
        ButtonAck request;
        sendMessage(request, MessageType_ButtonAck, fWorker.getIndex());
    }

    void TrezorDevice::handlePassphrase(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_PassphraseRequest ) {
            bail("Unexpected pin matrix response: " + QString::number(msg_in.id));
            return;
        }

        PassphraseRequest response;
        
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing passphrase response");
            return;
        }
        
        emit passphraseRequest(response.on_device());
        if (response.on_device()) {
           PassphraseAck request;
           sendMessage(request, MessageType_PassphraseAck, fWorker.getIndex());
        } else {
           fQueue.lock(MessageType_PassphraseAck, fWorker.getIndex()); // we need to wait for this call before making others, saving the index
        }
    }
    
    void TrezorDevice::handlePassphraseStateRequest(const Wire::Message &msg_in) {
       if ( msg_in.id != MessageType_PassphraseStateRequest ) {
          bail("Unexpected passphrase state response: " + QString::number(msg_in.id));
       }
              
       PassphraseStateAck ack;
       sendMessage(ack, MessageType_PassphraseStateAck, fWorker.getIndex());
    }

    void TrezorDevice::handleFeatures(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_Features ) {
            bail("Unexpected init response: " + QString::number(msg_in.id));
            return;
        }

        Features response;
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing features response");
            return;
        }

        fDeviceID = QString::fromStdString(response.device_id());
        quint32 mV = response.major_version();
        quint32 nV = response.minor_version();
        quint32 pV = response.patch_version();

        fVersion = QString::number(mV) + "." + QString::number(nV) + "." + QString::number(pV);
        // pre-1.8 check
        if (mV < 1 || (mV == 1 && nV < 8)) {
            qDebug() << "Device outdated, current version: " << fVersion << " need at least 1.8.0";
            emit deviceOutdated("1.8.0", fVersion);
        }

        emit initialized(fDeviceID);
        emit initializedChanged(true);
    }

    void TrezorDevice::handleAddress(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_EthereumAddress ) {
            bail("Unexpected get address response: " + QString::number(msg_in.id));
            return;
        }

        if ( fWorker.getIndex().toInt() < 0 ) {
            bail("Address index lost on reply");
            return;
        }

        EthereumAddress response;
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing address response");
            return;
        }

        const QString addressHex = Etherwall::Helpers::hexPrefix(QByteArray::fromStdString(response.address()));
        const QString hdPath = fWorker.getIndex().toString();
        emit addressRetrieved(addressHex, hdPath);
    }

    void TrezorDevice::handleTxRequest(const Wire::Message &msg_in)
    {
        if ( msg_in.id != MessageType_EthereumTxRequest ) {
            bail("Unexpected transaction response: " + QString::number(msg_in.id));
            return;
        }

        EthereumTxRequest response;
        if ( !parseMessage(msg_in, response) ) {
            bail("error parsing txsign response");
            return;
        }

        if ( response.data_length() > 0 ) {
            if ( response.data_length() > fPendingTx.dataByteSize() ) {
                bail("TREZOR requested more bytes than are in the pending tx data");
                return;
            }

            EthereumTxAck request;
            request.set_data_chunk(fPendingTx.dataNext(response.data_length()));
            sendMessage(request, MessageType_EthereumTxAck, fWorker.getIndex());
            return;
        }

        quint32 v = response.signature_v();
        std::string r = response.signature_r();
        std::string s = response.signature_s();

        fPendingTx.sign(v, r, s);
        emit transactionReady(fPendingTx);
    }

    // TrezorWorker

    TrezorWorker::TrezorWorker(Wire::Device &device): QThread(0),
        fDevice(device)
    {

    }

    void TrezorWorker::setRequest(const Wire::Message &request)
    {
        fRequest = request;
    }

    const Wire::Message &TrezorWorker::getReply() const
    {
        return fReply;
    }

    const QVariant TrezorWorker::getIndex() const
    {
        return fRequest.index;
    }

    // TrezorWorker

    void TrezorWorker::run()
    {
        fRequest.write_to(fDevice);
        fReply.read_from(fDevice);
    }

    // MessageQueue

    MessageQueue::MessageQueue() :
        QQueue(),
        fLockType(-1), fIndex(-1)
    {

    }

    void MessageQueue::lock(int type, const QVariant& index)
    {
        fLockType = type;
        fIndex = index;
    }

    void MessageQueue::unlock()
    {
        fLockType = -1;
    }

    void MessageQueue::push(const Wire::Message &msg)
    {
        if ( msg.id == fLockType ) {
            prepend(msg);
            return;
        }

        enqueue(msg);
    }

    bool MessageQueue::pop(Wire::Message& popped)
    {
        if ( empty() ) {
            return false;
        }

        if ( fLockType >= 0 && head().id != fLockType ) {
            return false;
        }

        popped = dequeue();

        if ( popped.id == fLockType ) {
            popped.index = fIndex;
            fIndex = QVariant(); // "forget"
            unlock();
        }
        return true;
    }

    const QString MessageQueue::toString() const
    {
        QString result;
        foreach ( const Wire::Message& msg, *this ) {
            result += QString::number(msg.id) + ",";
        }
        if ( result.endsWith(',') ) {
            result.remove(result.length() - 1, 1);
        }

        return result;
    }

}
