/*
 * Copyright (C) 2014-2024 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QMap>
#include <QMutex>
#include <QReadWriteLock>
#include <QString>

namespace Isds {
	/* Forward class declaration. */
	class Session;
}

/*!
 * @brief Container holding context structures of libdatovka.
 */
class IsdsSessions {

public:
	/*!
	 * @brief Constructor.
	 *
	 * @note Calls Isds::init().
	 */
	IsdsSessions(void);

	/*!
	 * @brief Destructor.
	 *
	 * @note Calls Isds::cleanup().
	 */
	~IsdsSessions(void);

	/*!
	 * @brief Check whether username has a non-terminated session.
	 *
	 * @param[in] username Username identifying the account.
	 * @return True if session exists, false else.
	 */
	bool holdsSession(const QString &username) const;

	/*!
	 * @brief Obtain associated non-terminated session.
	 *
	 * @param[in] username Username identifying the account.
	 * @return Non-null pointer if session exists.
	 */
	Isds::Session *session(const QString &username) const;

	/*!
	 * @brief Ping ISDS. Test whether connection is active.
	 *
	 * @param[in] username Username identifying the account.
	 * @return True if communication with ISDS is established and working,
	 *     false on any error.
	 */
	bool isConnectedToIsds(const QString &username) const;

	/*!
	 * @brief Creates new session.
	 *
	 * @param[in] username Username identifying the newly created account.
	 * @param[in] connectionTimeoutMs Connection timeout in milliseconds.
	 * @return Pointer to new session or Q_NULLPTR on failure.
	 */
	Isds::Session *createCleanSession(const QString &username,
	    unsigned int connectionTimeoutMs);

	/*!
	 * @brief Set time-out in milliseconds to session associated to
	 *     username.
	 *
	 * @param[in] username Username identifying the newly created account.
	 * @param[in] timeoutMs Connection timeout in milliseconds.
	 * @return True on success.
	 */
	bool setSessionTimeout(const QString &username, unsigned int timeoutMs);

	/*!
	 * @brief Remove session for username.
	 *
	 * @param[in] username Username identifying the account.
	 * @return True if non-terminated session exists and has been removed,
	 *     false otherwise.
	 */
	bool quitSession(const QString &username);

private:
	mutable QReadWriteLock m_lock; /*!< Read/write mutual exclusion. */
	mutable QMutex m_terminatingLock; /*!< Mutual exclusion for terminating sessions container. */

	QMap<QString, Isds::Session *> m_sessions; /*!< Holds sessions. */
	QMap<QString, Isds::Session *> m_terminatingSessions; /*!< Holds sessions to be terminated. */
};
