/*
 * Copyright (C) 2014-2024 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QMutexLocker>
#include <QReadLocker>
#include <QWriteLocker>

#include "src/common.h"
#include "src/datovka_shared/isds/error.h"
#include "src/datovka_shared/log/log.h"
#include "src/global.h"
#include "src/io/imports.h"
#include "src/io/isds_sessions.h"
#include "src/isds/initialisation.h"
#include "src/isds/services.h"
#include "src/isds/session.h"
#include "src/models/accounts_model.h"
#include "src/settings/prefs_specific.h"

IsdsSessions::IsdsSessions(void)
    : m_lock(),
    m_terminatingLock(),
    m_sessions(),
    m_terminatingSessions()
{
	/* Initialise libdatovka. */
	Isds::init();
}

IsdsSessions::~IsdsSessions(void)
{
	/* Free all contexts. */
	{
		QWriteLocker locker(&m_lock);
		for (Isds::Session *session : m_sessions) {
			delete session;
		}
		m_sessions.clear();
	}
	{
		QMutexLocker locker(&m_terminatingLock);
		for (Isds::Session *session : m_terminatingSessions) {
			delete session;
		}
		m_terminatingSessions.clear();
	}

	Isds::cleanup();
}

bool IsdsSessions::holdsSession(const QString &username) const
{
	QReadLocker locker(&m_lock);
	return Q_NULLPTR != m_sessions.value(username, Q_NULLPTR);
}

Isds::Session *IsdsSessions::session(const QString &username) const
{
	QReadLocker locker(&m_lock);
	return m_sessions.value(username, Q_NULLPTR);
}

/*!
 * @brief Set time-out in milliseconds to session associated to
 *     username.
 *
 * @param[in] session Session pointer.
 * @param[in] username Username identifying the newly created account.
 * @param[in] timeoutMs Connection timeout in milliseconds.
 * @return True on success.
 */
static inline
bool _setSessionTimeout(Isds::Session *session, const QString &username,
    unsigned int timeoutMs)
{
	if (Q_UNLIKELY(Q_NULLPTR == session)) {
		Q_ASSERT(0);
		return false;
	}

	bool ret = session->setTimeout(timeoutMs);
	if (Q_UNLIKELY(!ret)) {
		logErrorNL("Error setting time-out for username '%s'.",
		    username.toUtf8().constData());
	}

	return ret;
}

bool IsdsSessions::isConnectedToIsds(const QString &username) const
{
	Isds::Session *s = session(username); /* Uses QReadLocker. */
	if (Q_UNLIKELY(Q_NULLPTR == s)) {
		return false;
	}

	Isds::Error pingErr;

	_setSessionTimeout(s, username, ISDS_PING_TIMEOUT_MS);
	pingErr = Isds::Service::dummyOperation(s);
	_setSessionTimeout(s, username,
	    PrefsSpecific::isdsDownloadTimeoutMs(*GlobInstcs::prefsPtr));

	return Isds::Type::ERR_SUCCESS == pingErr.code();
}

Isds::Session *IsdsSessions::createCleanSession(const QString &username,
    unsigned int connectionTimeoutMs)
{
	QWriteLocker locker(&m_lock);

	/* Username shouldn't exist. */
	if (Q_UNLIKELY(Q_NULLPTR != m_sessions.value(username, Q_NULLPTR))) {
		Q_ASSERT(0);
		return Q_NULLPTR;
	}

	Isds::Session *session = Isds::Session::createSession(connectionTimeoutMs);
	if (Q_UNLIKELY(Q_NULLPTR == session)) {
		logErrorNL("Error creating ISDS session for username '%s'.",
		    username.toUtf8().constData());
		return Q_NULLPTR;
	}

	m_sessions.insert(username, session);
	return session;
}

bool IsdsSessions::setSessionTimeout(const QString &username,
    unsigned int timeoutMs)
{
	Isds::Session *s = session(username); /* Uses QReadLocker. */
	if (Q_UNLIKELY(Q_NULLPTR == s)) {
		Q_ASSERT(0);
		return false;
	}

	return _setSessionTimeout(s, username, timeoutMs);
}

bool IsdsSessions::quitSession(const QString &username)
{
	/*
	 * Don't delete the session directly because it may still be used in
	 * other threads.
	 */
	Isds::Session *session = Q_NULLPTR;

	{
		QWriteLocker locker(&m_lock);

		session = m_sessions.value(username, Q_NULLPTR);
		if (Q_NULLPTR != session) {
			m_sessions.remove(username);
		}
	}

	if (Q_NULLPTR != session) {
		QMutexLocker locker(&m_terminatingLock);

		{
			/* Delete any existent old terminating session. */
			Isds::Session *terminatingSession =
			    m_terminatingSessions.value(username, Q_NULLPTR);
			if (Q_UNLIKELY(Q_NULLPTR != terminatingSession)) {
				m_terminatingSessions.remove(username);
			}
			delete terminatingSession;
		}

		m_terminatingSessions.insert(username, session);
		return true;
	}

	return false;
}
