/*
	Copyright (C) 2003-2014 by David White <davewx7@gmail.com>

	This software is provided 'as-is', without any express or implied
	warranty. In no event will the authors be held liable for any damages
	arising from the use of this software.

	Permission is granted to anyone to use this software for any purpose,
	including commercial applications, and to alter it and redistribute it
	freely, subject to the following restrictions:

	   1. The origin of this software must not be misrepresented; you must not
	   claim that you wrote the original software. If you use this software
	   in a product, an acknowledgement in the product documentation would be
	   appreciated but is not required.

	   2. Altered source versions must be plainly marked as such, and must not be
	   misrepresented as being the original software.

	   3. This notice may not be removed or altered from any source
	   distribution.
*/

#include <iostream>
#include <map>
#include <vector>
#include <string>
#include <time.h>

#ifdef USE_SVG
#include "cairo.hpp"
#endif

#include "code_editor_dialog.hpp"
#include "ColorTransform.hpp"
#include "RenderTarget.hpp"
#include "WindowManager.hpp"

#include "achievements.hpp"
#include "asserts.hpp"
#include "blur.hpp"
#include "clipboard.hpp"
#include "collision_utils.hpp"
#include "controls.hpp"
#include "current_generator.hpp"
#include "custom_object_functions.hpp"
#include "custom_object.hpp"
#include "debug_console.hpp"
#include "draw_scene.hpp"
#include "editor.hpp"
#include "entity.hpp"
#include "ffl_weak_ptr.hpp"
#include "filesystem.hpp"
#include "formatter.hpp"
#include "formula_callable_definition.hpp"
#include "formula_function_registry.hpp"
#include "formula_profiler.hpp"
#include "formula_vm.hpp"
#include "haptic.hpp"
#include "i18n.hpp"
#include "input.hpp"
#include "json_parser.hpp"
#include "level.hpp"
#include "level_runner.hpp"
#include "load_level.hpp"
#include "object_events.hpp"
#include "pause_game_dialog.hpp"
#include "player_info.hpp"
#include "tbs_client.hpp"
#include "tbs_internal_client.hpp"
#include "tbs_ipc_client.hpp"
#include "message_dialog.hpp"
#include "playable_custom_object.hpp"
#include "preferences.hpp"
#include "profile_timer.hpp"
#include "random.hpp"
#include "sound.hpp"
#include "speech_dialog.hpp"
#include "stats.hpp"
#include "string_utils.hpp"
#include "tbs_internal_server.hpp"
#include "thread.hpp"
#include "unit_test.hpp"
#include "preferences.hpp"
#include "module.hpp"
#include "variant_utils.hpp"
#include "widget_factory.hpp"
#include "graphical_font.hpp"
#include "user_voxel_object.hpp"

namespace {
std::set<ObjectTypesSpawnedTracker*> g_spawned_tracker_scopes;
void trackObjectSpawned(const std::string& id)
{
	for(auto s : g_spawned_tracker_scopes) {
		s->spawned.insert(id);
	}
}
}

ObjectTypesSpawnedTracker::ObjectTypesSpawnedTracker()
{
	g_spawned_tracker_scopes.insert(this);
}

ObjectTypesSpawnedTracker::~ObjectTypesSpawnedTracker()
{
	g_spawned_tracker_scopes.erase(this);
}

bool g_mouse_event_swallowed;

void run_auto_updater();

PREF_BOOL(tbs_use_shared_mem, true, "Use shared memory for tbs comms");

void EntityCommandCallable::setExpression(const game_logic::FormulaExpression* expr)
{
	expr_ = expr;
	expr_holder_.reset(expr);
}

void CustomObjectCommandCallable::setExpression(const game_logic::FormulaExpression* expr)
{
	expr_ = expr;
	expr_holder_.reset(expr);
}

void EntityCommandCallable::runCommand(Level& lvl, Entity& obj) const
{
	if(expr_) {
//		try {
//			fatal_assert_scope scope;
			execute(lvl, obj);
//		} catch(fatal_assert_failure_exception& e) {
//			ASSERT_FATAL(e.msg << "\nERROR ENCOUNTERED WHILE RUNNING COMMAND GENERATED BY THIS EXPRESSION:\n" << expr_->debugPinpointLocation());
//		}
	} else {
		execute(lvl, obj);
	}
}

void CustomObjectCommandCallable::runCommand(Level& lvl, CustomObject& obj) const
{
	if(expr_) {
		try {
			fatal_assert_scope scope;
			execute(lvl, obj);
		} catch(const fatal_assert_failure_exception& e) {
			ASSERT_FATAL(e.msg << "\nERROR ENCOUNTERED WHILE RUNNING COMMAND GENERATED BY THIS EXPRESSION:\n" << expr_->debugPinpointLocation());
		}
	} else {
		execute(lvl, obj);
	}
}

using namespace game_logic;

namespace
{
	const std::string FunctionModule = "custom_object";

	FUNCTION_DEF(set_language, 1, 1, "set_language(str): set the language using a new locale code")
		std::string locale = EVAL_ARG(0).as_string();
		preferences::setLocale(locale);
		i18n::init();
		GraphicalFont::initForLocale(i18n::get_locale());
		return variant(0);
	RETURN_TYPE("int")
	END_FUNCTION_DEF(set_language)

	FUNCTION_DEF(translate, 1, 1, "translate(str): returns the translated version of the given string")
		return variant(i18n::tr(EVAL_ARG(0).as_string()));

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("string")
	END_FUNCTION_DEF(translate)

	ffl::IntrusivePtr<TextureObject> render_fbo(const rect& area, const std::vector<EntityPtr> objects)
	{
		const controls::control_backup_scope ctrl_backup;

		LevelPtr lvl(new Level("empty.cfg"));
		lvl->setRenderToTexture(area.w(), area.h());

		lvl->set_boundaries(area);
		screen_position pos;
		pos.x = area.x()*100;
		pos.y = area.y()*100;
		//KRE::WindowManager::getMainWindow()->setWindowSize(area.w(), area.h());
		{
			for(EntityPtr e : objects) {
				lvl->add_character(e);
				e->createObject();
			}
			lvl->process();
			lvl->process_draw();

			for(const EntityPtr& e : objects) {
				lvl->add_draw_character(e);
			}

			render_scene(*lvl, pos);
		}

		lvl->getRenderTarget()->unapply();

		return ffl::IntrusivePtr<TextureObject>(new TextureObject(lvl->getRenderTarget()->getTexture()));
	}

	FUNCTION_DEF(get_texture, 1, 1, "get_texture(string|map): loads a texture")
		game_logic::Formula::failIfStaticContext();
		variant arg = EVAL_ARG(0);
		PROFILE_INSTRUMENT(get_texture, (arg.is_map() ? arg["image"].write_json() : arg.write_json()));
		KRE::TexturePtr t = KRE::Texture::createTexture(arg);
		t->clearSurfaces();
		return variant(new TextureObject(t));

	FUNCTION_ARGS_DEF
		ARG_TYPE("string|map")
	RETURN_TYPE("builtin texture_object")

	END_FUNCTION_DEF(get_texture)


	FUNCTION_DEF(texture, 2, 3, "texture(objects, rect, bool half_size=false): render a texture")
		// XXX FIX half_size implementation.
		variant objects = EVAL_ARG(0);
		variant area = EVAL_ARG(1);

		ASSERT_LOG(objects.is_list(), "MUST PROVIDE A LIST OF OBJECTS TO RENDER");
		ASSERT_LOG(area.is_list() && area.num_elements() == 4, "MUST PROVIDE AN AREA TO texture");

		std::vector<EntityPtr> obj;
		for(int n = 0; n != objects.num_elements(); ++n) {
			obj.push_back(objects[n].convert_to<Entity>());
		}

		const rect r(area);
		auto t = render_fbo(r, obj);
		return variant(t.get());
	/*
		variant objects = EVAL_ARG(0);
		variant area = EVAL_ARG(1);

		ASSERT_LOG(objects.is_list(), "MUST PROVIDE A LIST OF OBJECTS TO RENDER");
		ASSERT_LOG(area.is_list() && area.num_elements() == 4, "MUST PROVIDE AN AREA TO texture");

		std::vector<EntityPtr> obj;
		for(int n = 0; n != objects.num_elements(); ++n) {
			obj.push_back(objects[n].convert_to<Entity>());
		}

		const rect r(area);

		graphics::texture t = render_fbo(r, obj);

		if(NUM_ARGS > 2 && EVAL_ARG(2).as_bool()) {
			using namespace graphics;
			surface src = t.get_surface();
			surface dst(SDL_CreateRGBSurface(0, src->w/2, src->h/2, 32, SURFACE_MASK));

			SDL_Rect src_rect = {0,0,src->w,src->h};
			SDL_Rect dst_rect = {0,0,dst->w,dst->h};

			SDL_SetSurfaceBlendMode(src.get(), SDL_BLENDMODE_NONE);

			SDL_BlitScaled(src.get(), &src_rect, dst.get(), &dst_rect);
			t = texture::get_no_cache(dst);
		}

		return variant(new texture_object(t));
	*/
		return variant();
	FUNCTION_ARGS_DEF
		ARG_TYPE("[object]")
		ARG_TYPE("[int]")
		ARG_TYPE("bool")
	RETURN_TYPE("builtin texture_object")

	END_FUNCTION_DEF(texture)

	namespace {
		void sanitizeNotificationStr(std::string& s) {
			for(char& c : s) {
				if(util::c_isalnum(c) == false && strchr(".,'-_", c) == nullptr) {
					c = ' ';
				}
			}
		}
	}

	FUNCTION_DEF(desktop_notification, 1, 3, "desktop_notification(message, title, icon)")
		std::string message = EVAL_ARG(0).as_string();
		std::string title;
		if(NUM_ARGS >= 2) {
			title = EVAL_ARG(1).as_string();
		}

		std::string icon;
		if(NUM_ARGS >= 3) {
			icon = EVAL_ARG(2).as_string();
		}

		sanitizeNotificationStr(message);
		sanitizeNotificationStr(title);
		sanitizeNotificationStr(icon);

		return variant(new FnCommandCallable("open_url", [=]() {
#ifdef __linux__
				std::ostringstream s;
				s << "/usr/bin/notify-send \"" << message << "\"";
				if(title.empty() == false) {
					s << " \"" << title << "\"";
				}

				if(icon.empty() == false) {
					s << " --icon=" << icon;
				}

				std::string cmd = s.str();
				const int result = system(cmd.c_str());
				if(result == -1) {
					LOG_ERROR("Could not spawn system process: " << errno);
				}

#elif defined(WIN32) || defined(WIN64)
			//Windows implementation here
#elif defined(__APPLE__) && TARGET_OS_MAC
			//OSX implementation here
#endif
		}));

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("string")
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(desktop_notification)

	FUNCTION_DEF(open_url, 1, 1, "open_url(string url): opens a given url on the platform's web browser")
		std::string url = EVAL_ARG(0).as_string();

		for(char& c : url) {
			if(!isprint(c) || c < 0 || isspace(c)) {
				c = 0;
			}
		}

		url.erase(std::remove(url.begin(), url.end(), 0), url.end());

		static const char* prefix = "http://";

		if(url.size() < strlen(prefix) || !std::equal(prefix, prefix+strlen(prefix), url.begin())) {
			return variant();
		}

		return variant(new FnCommandCallable("open_url", [=]() {
#if defined(WIN32) || defined(WIN64)
			const std::string open_str = "start";
#else
			const std::string open_str = "open";
#endif

			std::string cmd = open_str + " " + url;
#ifdef __linux__
			cmd = "xdg-" + cmd;
#endif
			const int result = system(cmd.c_str());
			if(result == -1) {
				LOG_ERROR("Could not execute command");
			}
		}));

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(open_url)

	FUNCTION_DEF(get_clipboard_text, 0, 0, "get_clipboard_text(): returns the text currentl in the windowing clipboard")
		Formula::failIfStaticContext();
		return variant(copy_from_clipboard(false));
	RETURN_TYPE("string")
	END_FUNCTION_DEF(get_clipboard_text)

	class set_clipboard_text_command : public game_logic::CommandCallable
	{
		std::string str_;
	public:
		explicit set_clipboard_text_command(const std::string& str) : str_(str)
		{}

		virtual void execute(game_logic::FormulaCallable& ob) const override {
			copy_to_clipboard(str_);
		}
	};

	FUNCTION_DEF(set_clipboard_text, 1, 1, "set_clipboard_text(str): sets the clipboard text to the given string")
		set_clipboard_text_command* cmd = new set_clipboard_text_command(EVAL_ARG(0).as_string());
		cmd->setExpression(this);
		return variant(cmd);

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_clipboard_text)

	#if !defined(__native_client__)

	FUNCTION_DEF(tbs_internal_client, 0, 1, "tbs_internal_client(session=-1): creates a client object to the local in-memory tbs server")
		Formula::failIfStaticContext();

		const int session = NUM_ARGS >= 1 ? EVAL_ARG(0).as_int() : -1;

		SharedMemoryPipePtr pipe;
		SharedMemoryPipePtr* pipe_ptr = nullptr;

		if(g_tbs_use_shared_mem) {
			pipe_ptr = &pipe;
		}

		const int port = session == -1 ? tbs::spawn_server_on_localhost(pipe_ptr) : tbs::get_server_on_localhost(pipe_ptr);

		if(pipe_ptr) {
			ASSERT_LOG(pipe.get() != nullptr, "Could not initialize ipc pipe");
			tbs::ipc_client* result = new tbs::ipc_client(pipe);
			return variant(result);
		} else {
			tbs::client* result = new tbs::client("127.0.0.1", formatter() << port, session);
			return variant(result);
		}

		//return variant(new tbs::internal_client(session));

	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("object")
	END_FUNCTION_DEF(tbs_internal_client)

	FUNCTION_DEF(tbs_client, 2, 3, "tbs_client(host, port, session=-1): creates a client object to the tbs server")
		Formula::failIfStaticContext();

		const std::string host = EVAL_ARG(0).as_string();
		variant port_var = EVAL_ARG(1);
		std::string port;
		if(port_var.is_string()) {
			port = port_var.as_string();
		} else {
			port = formatter() << port_var.as_int();
		}

		bool retry_on_timeout = false;

		std::string id;

		int session = -1;
		if(NUM_ARGS >= 3) {
			variant options = EVAL_ARG(2);
			if(options.is_int()) {
				session = options.as_int();
			} else {
				session = options["session"].as_int(-1);
				id = options["id"].as_string_default("");
				retry_on_timeout = options["retry_on_timeout"].as_bool(false);
			}
		}
		tbs::client* result = new tbs::client(host, port, session);
		if(retry_on_timeout) {
			result->set_timeout_and_retry();
		}
		result->setId(id);
		return variant(result);

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("string|int")
		ARG_TYPE("int|{session: int|null, id: string|null, retry_on_timeout: bool|null}")
	RETURN_TYPE("object")
	END_FUNCTION_DEF(tbs_client)


	void tbs_send_event(ffl::weak_ptr<Entity> e, game_logic::MapFormulaCallablePtr callable, const std::string& ev)
	{
		auto entity = e.get();
		if(entity) {
			entity->handleEvent(ev, callable.get());
		}
	}

	class tbs_send_command : public EntityCommandCallable
	{
		variant client_, msg_;
		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&client_);
			collector->surrenderVariant(&msg_);
		}
	public:
		tbs_send_command(variant client, variant msg) : client_(client), msg_(msg)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			using std::placeholders::_1;
			game_logic::MapFormulaCallablePtr callable(new game_logic::MapFormulaCallable);
			tbs::ipc_client* ipc_client = client_.try_convert<tbs::ipc_client>();
			if(ipc_client != nullptr) {
				ipc_client->set_handler(std::bind(tbs_send_event, ffl::weak_ptr<Entity>(&ob), callable, _1));
				ipc_client->set_callable(callable);
				ipc_client->send_request(msg_);
				return;
			}

			tbs::client* tbs_client = client_.try_convert<tbs::client>();
			if(tbs_client == nullptr) {
				tbs::internal_client* tbs_iclient = client_.try_convert<tbs::internal_client>();
				LOG_DEBUG("XX tbs_send: " << tbs_iclient->session_id());
				ASSERT_LOG(tbs_iclient != nullptr, "tbs_client object isn't valid.");
				tbs_iclient->send_request(msg_, tbs_iclient->session_id(), callable, std::bind(tbs_send_event, ffl::weak_ptr<Entity>(&ob), callable, _1));
			} else {
				tbs_client->send_request(msg_, callable, std::bind(tbs_send_event, ffl::weak_ptr<Entity>(&ob), callable, _1));
			}
		}

	};

	FUNCTION_DEF(tbs_send, 2, 2, "tbs_send(tbs_client, msg): sends a message through the given tbs_client connection")
		variant client = EVAL_ARG(0);
		variant msg = EVAL_ARG(1);

		tbs_send_command* cmd = new tbs_send_command(client, msg);
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("map|object|list")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(tbs_send)

	class tbs_process_command : public EntityCommandCallable
	{
		variant client_;
		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&client_);
		}
	public:
		explicit tbs_process_command(variant client) : client_(client)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override{
			tbs::ipc_client* ipc_client = client_.try_convert<tbs::ipc_client>();
			if(ipc_client != nullptr) {
				ipc_client->process();
				return;
			}

			tbs::client* tbs_client = client_.try_convert<tbs::client>();
			if(tbs_client == nullptr) {
				tbs::internal_client* iclient = client_.try_convert<tbs::internal_client>();
				ASSERT_LOG(iclient != nullptr, "tbs_client object isn't valid.");
				iclient->process();
			} else {
				tbs_client->process();
			}
		}
	};

	FUNCTION_DEF(tbs_process, 1, 1, "tbs_process(tbs_client): processes events for the tbs client")
		variant client = EVAL_ARG(0);
		tbs_process_command* cmd = (new tbs_process_command(client));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(tbs_process)

	class tbs_blocking_process_command : public EntityCommandCallable
	{
	public:
		explicit tbs_blocking_process_command(variant request) : request_(request)
		{}
	private:
		virtual void execute(Level& lvl, Entity& ob) const override;
		variant request_;

		mutable variant deferred_pipeline_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&request_);
			collector->surrenderVariant(&deferred_pipeline_);
		}
	};

	class tbs_blocking_request : public game_logic::FormulaCallable
	{
	public:
		tbs_blocking_request(variant client, variant request)
		  : client_(client),
		    ipc_client_(client.try_convert<tbs::ipc_client>()),
			tbs_client_(client.try_convert<tbs::client>()),
			callable_(new game_logic::MapFormulaCallable),
			got_response_(false)
		{
			if(ipc_client_ != nullptr) {
				ipc_client_->set_handler(std::bind(&tbs_blocking_request::handle, this, std::placeholders::_1));
				ipc_client_->set_callable(callable_);
				ipc_client_->send_request(request);
			} else {
				ASSERT_LOG(tbs_client_ != nullptr, "Illegal object given to tbs_blocking_request");
				tbs_client_->send_request(request, callable_, std::bind(&tbs_blocking_request::handle, this, std::placeholders::_1));
			}
		}

		void process() {
			if(ipc_client_) {
				ipc_client_->process();
			}

			if(tbs_client_) {
				tbs_client_->process();
			}
		}

		bool got_response() const { return got_response_; }
	private:
		DECLARE_CALLABLE(tbs_blocking_request);

		void handle(std::string type) {
			variant msg = callable_->queryValue("message");
			response_ = msg;
			got_response_ = true;
		}

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&client_);
			collector->surrenderPtr(&callable_);
			collector->surrenderVariant(&response_);
		}

		variant client_;
		tbs::ipc_client* ipc_client_;
		tbs::client* tbs_client_;
		game_logic::MapFormulaCallablePtr callable_;
		variant response_;
		bool got_response_;
	};

	void tbs_blocking_process_command::execute(Level& lvl, Entity& ob) const
	{
		tbs_blocking_request* req = request_.convert_to<tbs_blocking_request>();
		ASSERT_LOG(req, "Illegal object given to tbs blocking process");
		req->process();
		if(!req->got_response()) {
			if(deferred_pipeline_.is_null()) {
				deferred_pipeline_ = deferCurrentCommandSequence();
			}

			ob.addScheduledCommand(1, variant(this));
		} else {
			ob.executeCommand(deferred_pipeline_);
		}
	}

	BEGIN_DEFINE_CALLABLE_NOBASE(tbs_blocking_request)
		BEGIN_DEFINE_FN(block, "()->commands")
			auto cmd = new tbs_blocking_process_command(variant(&obj));
			return variant(cmd);
		END_DEFINE_FN

		DEFINE_FIELD(response, "any")
			return obj.response_;
	END_DEFINE_CALLABLE(tbs_blocking_request)

	FUNCTION_DEF(tbs_blocking_request, 2, 2, "tbs_blocking_request(tbs_client, request)")
		variant client = EVAL_ARG(0);
		variant request = EVAL_ARG(1);

		return variant(new tbs_blocking_request(client, request));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("map")
	RETURN_TYPE("builtin tbs_blocking_request")
	END_FUNCTION_DEF(tbs_blocking_request)


	#endif // __native_client__

	class report_command : public EntityCommandCallable
	{
		variant v_;
		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&v_);
		}
	public:
		explicit report_command(variant v) : v_(v)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			stats::record(v_);
		}
	};

	FUNCTION_DEF(report, 1, 1, "report(): Write a key and a value into [custom] in the stats.")
		report_command* cmd = (new report_command(EVAL_ARG(0)));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("any")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(report)


	class engine_performance_info : public game_logic::FormulaCallable
	{
	public:

		engine_performance_info()
		{
		}

	private:
		DECLARE_CALLABLE(engine_performance_info);
	};

	BEGIN_DEFINE_CALLABLE_NOBASE(engine_performance_info)
		DEFINE_FIELD(platform, "string")
#ifdef __linux__
			return variant("linux");
#elif defined(__APPLE__)
			return variant("apple");
#elif defined(_WIN32)
			return variant("win32");
#else
			return variant("unknown");
#endif

		DEFINE_FIELD(fps, "int") return variant(performance_data::current()->fps);
		DEFINE_FIELD(max_frame_time, "int") return variant(performance_data::current()->max_frame_time);
		DEFINE_FIELD(cycles_per_second, "int") return variant(performance_data::current()->cycles_per_second);
		DEFINE_FIELD(delay, "int") return variant(performance_data::current()->delay);
		DEFINE_FIELD(draw, "int") return variant(performance_data::current()->draw);
		DEFINE_FIELD(process, "int") return variant(performance_data::current()->process);
		DEFINE_FIELD(flip, "int") return variant(performance_data::current()->flip);
		DEFINE_FIELD(cycle, "int") return variant(performance_data::current()->cycle);
		DEFINE_FIELD(nevents, "int") return variant(performance_data::current()->nevents);
		DEFINE_FIELD(ticks, "int") return variant(SDL_GetTicks());
	END_DEFINE_CALLABLE(engine_performance_info)

	FUNCTION_DEF(get_perf_info, 0, 0, "get_perf_info(): return performance info")
		return variant(new engine_performance_info);

	FUNCTION_ARGS_DEF
	RETURN_TYPE("builtin engine_performance_info")
	END_FUNCTION_DEF(get_perf_info)


	namespace
	{
		int show_simple_option_dialog(Level& lvl, const std::string& text, const std::vector<std::string>& options)
		{
			std::vector<std::string> txt;
			txt.push_back(text);
			std::shared_ptr<SpeechDialog> d(new SpeechDialog);
			d->setText(txt);
			d->setOptions(options);

			lvl.add_speech_dialog(d);

			bool done = false;
			while(!done) {
				SDL_Event event;
				while(input::sdl_poll_event(&event)) {
					if(d->keyPress(event)) {
						done = true;
					}
				}

				if(d->process() || d->detectJoystickPress()) {
					done = true;
				}

				draw_scene(lvl, last_draw_position(), &lvl.player()->getEntity());
				KRE::WindowManager::getMainWindow()->swap();
				profile::delay(preferences::frame_time_millis());
			}

			lvl.remove_speech_dialog();

			return d->getOptionSelected();
		}
	}

	class set_save_slot_command : public EntityCommandCallable
	{
	public:
		explicit set_save_slot_command(int slot) : slot_(slot)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			bool has_options = false;
			std::vector<std::string> options;
			std::string SaveFiles[] = {"save.cfg", "save2.cfg", "save3.cfg", "save4.cfg"};
			int nslot = 0;
			for(const std::string& fname : SaveFiles) {
				options.push_back(formatter() << "Slot " << (nslot+1) << ": (empty)");
				const std::string path = std::string(preferences::user_data_path()) + "/" + fname;
				if(sys::file_exists(path)) {
					has_options = true;
					try {
						const variant doc = json::parse_from_file(path);
						if(doc.is_null() == false) {
							options.back() = formatter() << "Slot " << (nslot+1) << ": " << doc["title"].as_string();
						}
					} catch(const json::ParseError&) {
					}
				}

				++nslot;
			}

			if(slot_ >= 0) {
				LOG_INFO("setting save slot: " << slot_ << " -> " << SaveFiles[slot_]);
				preferences::set_save_slot(SaveFiles[slot_]);
			} else if(has_options) {
				const int noption = show_simple_option_dialog(lvl, _("Select save slot to use."), options);
				if(noption != -1) {
					LOG_INFO("setting save slot: " << noption << " -> " << SaveFiles[noption]);
					preferences::set_save_slot(SaveFiles[noption]);
				}
			}
		}
		int slot_;
	};

	FUNCTION_DEF(set_save_slot, 0, 1, "set_save_slot((optional) int slot): Allows the user to select the save slot, if no slot is specified a dialog is displayed.")
		const int slot = (NUM_ARGS > 0) ? EVAL_ARG(0).as_int() : -1;
		ASSERT_LOG(slot == -1 || (slot >= 1 && slot <= 4), "Invalid slot specified: " << slot);
		set_save_slot_command* cmd = (new set_save_slot_command(slot-1));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_save_slot)

	class save_game_command : public EntityCommandCallable
	{
		bool persistent_;
	public:
		explicit save_game_command(bool persistent) : persistent_(persistent)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			formula_profiler::Instrument instrument(persistent_ ? "SAVE_GAME" : "CHECKPOINT_GAME");
			lvl.player()->getEntity().saveGame();
			if(persistent_) {
				variant node = lvl.write();
				if(sound::current_music().empty() == false) {
					node = node.add_attr(variant("music"), variant(sound::current_music()));
				}

				sys::write_file(preferences::save_file_path(), node.write_json());
			}
		}
	};

	FUNCTION_DEF(checkpoint_game, 0, 0, "checkpoint_game(): saves a checkpoint of the game")
		save_game_command* cmd = (new save_game_command(false));
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(checkpoint_game)

	FUNCTION_DEF(get_save_document, 1, 1, "get_save_document(int slot): gets the FFL document for the save in the given slot")
		Formula::failIfStaticContext();

		const int slot = EVAL_ARG(0).as_int();
		std::string fname = "save.cfg";
		if(slot != 0) {
			fname = formatter() << "save" << (slot+1) << ".cfg";
		}
		const std::string path = std::string(preferences::user_data_path()) + "/" + fname;

		try {
			const variant v = json::parse_from_file(path, json::JSON_PARSE_OPTIONS::NO_PREPROCESSOR);
			return v;
		} catch(const json::ParseError&) {
			return variant();
		}
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("map|null")
	END_FUNCTION_DEF(get_save_document)

	FUNCTION_DEF(save_game, 0, 0, "saveGame(): saves the current game state")
		save_game_command* cmd = (new save_game_command(true));
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(save_game)

	class load_game_command : public EntityCommandCallable
	{
		std::string transition_;
		int slot_;
	public:
		load_game_command(const std::string& transition, int slot) : transition_(transition), slot_(slot) {}
		virtual void execute(Level& lvl, Entity& ob) const override {
			std::string dest_file;

			if(slot_ >= 0) {
				std::string fname = "save.cfg";
				if(slot_ != 0) {
					fname = formatter() << "save" << (slot_+1) << ".cfg";
				}

				if(sys::file_exists(std::string(preferences::user_data_path()) + "/" + fname)) {
					preferences::set_save_slot(fname);
					dest_file = fname;
				}
			}

			if(dest_file.empty()) {

				std::vector<std::string> save_options;
				save_options.push_back("save.cfg");

				for(int n = 2; n <= 4; ++n) {
					std::string fname = formatter() << "save" << n << ".cfg";
					if(sys::file_exists(std::string(preferences::user_data_path()) + "/" + fname)) {
						save_options.push_back(fname);
					}
				}

				int noption = 0;

				if(save_options.size() > 1) {
					int nslot = 1;
					std::vector<std::string> option_descriptions;
					for(const std::string& option : save_options) {
						const std::string fname = std::string(preferences::user_data_path()) + "/" + option;
						try {
							const variant doc = json::parse_from_file(fname);
							option_descriptions.push_back(formatter() << "Slot " << nslot << ": " << doc["title"].as_string());
						} catch(const json::ParseError&) {
							option_descriptions.push_back(formatter() << "Slot " << nslot << ": Frogatto");
						}

						++nslot;
					}

					noption = show_simple_option_dialog(lvl, _("Select save slot to load."), option_descriptions);

					if(noption == -1) {
						return;
					}

					preferences::set_save_slot(save_options[noption]);
				}
				dest_file = save_options[noption];
			}

			Level::portal p;
			p.level_dest = dest_file;
			p.dest_starting_pos = true;
			p.saved_game = true;
			p.transition = transition_;
			lvl.force_enter_portal(p);
		}
	};

	FUNCTION_DEF(load_game, 0, 2, "load_game(transition, slot): loads the saved game. If transition (a string) is given, it will use that type of transition.")
		std::string transition;
		int slot = -1;
		if(NUM_ARGS >= 1) {
			transition = EVAL_ARG(0).as_string();
		}

		if(NUM_ARGS >= 2) {
			slot = EVAL_ARG(1).as_int();
		}
		load_game_command* cmd = (new load_game_command(transition, slot));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(load_game)

	FUNCTION_DEF(can_load_game, 0, 0, "can_load_game(): returns true if there is a saved game available to load")
		return variant(sys::file_exists(preferences::save_file_path()));
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(can_load_game)

	FUNCTION_DEF(available_save_slots, 0, 0, "available_save_slots(): returns a list of numeric indexes of available save slots")
		std::vector<variant> result;
		for(int slot = 0; slot != 4; ++slot) {

			std::string fname = "save.cfg";
			if(slot != 0) {
				fname = formatter() << "save" << (slot+1) << ".cfg";
			}

			if(json::file_exists_and_is_valid(std::string(preferences::user_data_path()) + "/" + fname)) {
				result.emplace_back(slot);
			}
		}

		return variant(&result);

	RETURN_TYPE("commands")
	END_FUNCTION_DEF(available_save_slots)

	class move_to_standing_command : public EntityCommandCallable
	{
	public:
		virtual void execute(Level& lvl, Entity& ob) const override {
			ob.moveToStanding(lvl);
		}
	};

	FUNCTION_DEF(move_to_standing, 0, 0, "moveToStanding(): tries to move the object downwards if it's in the air, or upwards if it's in solid space, until it's standing on solid ground.");
		move_to_standing_command* cmd = (new move_to_standing_command());
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(move_to_standing)

	class music_command : public EntityCommandCallable
		{
		public:
			explicit music_command(const std::string& name, const bool loops, int fade_time)
			: name_(name), loops_(loops), queued_(false), fade_time_(fade_time)
			{}
			virtual void execute(Level& lvl, Entity& ob) const override {
				if(loops_){
					sound::play_music(name_, queued_, fade_time_);
				}else{
					sound::play_music_interrupt(name_);
				}
			}
			void setQueued() { queued_ = true; }
		private:
			std::string name_;
			bool loops_;
			bool queued_;
			int fade_time_;
		};

	FUNCTION_DEF(music, 1, 1, "music(string id): plays the music file given by 'id' in a loop")
		music_command* cmd = (new music_command(
										 EVAL_ARG(0).as_string(),
										 true, 500));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(music)

	FUNCTION_DEF(music_queue, 1, 2, "music_queue(string id, int fade_time=500): plays the music file given by 'id' in a loop after the current music is done.")
		int fade_time = 500;
		if(NUM_ARGS >= 2) {
			fade_time = EVAL_ARG(1).as_int();
		}
		music_command* cmd = (new music_command(
										 EVAL_ARG(0).as_string(),
										 true, fade_time));
		cmd->setQueued();
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(music_queue)

	FUNCTION_DEF(music_onetime, 1, 1, "music_onetime(string id): plays the music file given by 'id' once")
		music_command* cmd = (new music_command(
										 EVAL_ARG(0).as_string(),
										 false, 500));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(music_onetime)

	class sound_command : public EntityCommandCallable
	{
	public:
		explicit sound_command(const std::string& name, const bool loops, float volume, float fade_in_time, float stereo_left, float stereo_right)
		  : names_(util::split(name)), loops_(loops), volume_(volume), fade_in_time_(fade_in_time), stereo_left_(stereo_left), stereo_right_(stereo_right)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			sound::set_panning(stereo_left_, stereo_right_);
			if(loops_){
				if (names_.empty() == false){
					int randomNum = rand()%names_.size();  //like a 1d-size die
					if(names_[randomNum].empty() == false) {
						sound::play_looped(names_[randomNum], &ob, volume_, fade_in_time_);
					}
				}

			}else{
				if (names_.empty() == false){
					int randomNum = rand()%names_.size();  //like a 1d-size die
					if(names_[randomNum].empty() == false) {
						sound::play(names_[randomNum], &ob, volume_, fade_in_time_);
					}
				}
			}

			sound::set_panning(1.0, 1.0);
		}
	private:
		std::vector<std::string> names_;
		bool loops_;
		float volume_;
		float fade_in_time_;
		float stereo_left_, stereo_right_;
	};

	FUNCTION_DEF(sound, 1, 4, "sound(string id, decimal volume, decimal fade_in_time, [decimal,decimal] stereo): plays the sound file given by 'id', reaching full volume after fade_in_time seconds.")
		float stereo_left = 1.0, stereo_right = 1.0;
		if(NUM_ARGS > 3) {
			variant stereo_var = EVAL_ARG(3);
			stereo_left = stereo_var[0].as_float();
			stereo_right = stereo_var[1].as_float();
		}

		sound_command* cmd = new sound_command(
										 EVAL_ARG(0).as_string(),
										 false,
										 NUM_ARGS > 1 ? EVAL_ARG(1).as_float() : 1.0f,
										 NUM_ARGS > 2 ? EVAL_ARG(2).as_float() : 0.0f,
										 stereo_left, stereo_right);
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("decimal")
		ARG_TYPE("decimal")
		ARG_TYPE("[decimal,decimal]")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sound)

	FUNCTION_DEF(sound_loop, 1, 4, "sound_loop(string id, decimal volume, decimal fade_in_time, [decimal,decimal] stereo): plays the sound file given by 'id' in a loop, if fade_in_time is given it will reach full volume after this time.")
		float stereo_left = 1.0f, stereo_right = 1.0f;
		if(NUM_ARGS > 3) {
			variant stereo_var = EVAL_ARG(3);
			stereo_left = stereo_var[0].as_float();
			stereo_right = stereo_var[1].as_float();
		}

		sound_command* cmd = new sound_command(
										 EVAL_ARG(0).as_string(),
										 true,
										 NUM_ARGS > 1 ? EVAL_ARG(1).as_float() : 1.0f,
										 NUM_ARGS > 2 ? EVAL_ARG(2).as_float() : 0.0f,
										 stereo_left, stereo_right);
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("decimal")
		ARG_TYPE("decimal")
		ARG_TYPE("[decimal, decimal]")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sound_loop)

	class sound_pan_command : public EntityCommandCallable
	{
	public:
		explicit sound_pan_command(const std::string& name, float left, float right)
		  : name_(name), left_(left), right_(right)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			sound::update_panning(&ob, name_, left_, right_);
		}
	private:
		std::string name_;
		float left_, right_;
	};

	FUNCTION_DEF(sound_pan, 2, 2, "sound_pan(string id, [decimal,decimal] stereo): pans the sound being played with the given id by the specified stereo amount.")
		const std::string id = EVAL_ARG(0).as_string();

		variant stereo_var = EVAL_ARG(1);
		float stereo_left = stereo_var[0].as_float();
		float stereo_right = stereo_var[1].as_float();

		return variant(new sound_pan_command(id, stereo_left, stereo_right));

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("[decimal, decimal]")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sound_pan)


	class sound_volume_command : public EntityCommandCallable {
	public:
		sound_volume_command(float volume, float nseconds)
		: volume_(volume), nseconds_(nseconds)
		{}
		virtual void execute(Level& lvl,Entity& ob) const override {
			//sound::change_volume(&ob, volume_);
			ob.setSoundVolume(volume_, nseconds_);
		}
	private:
		std::string name_;
		float volume_, nseconds_;
	};

	FUNCTION_DEF(sound_volume, 1, 2, "sound_volume(decimal volume, decimal time_seconds): sets the volume of sound effects")
		float nseconds = 0.0f;
		if(NUM_ARGS > 1) {
			nseconds = EVAL_ARG(1).as_decimal().as_float();
		}
		sound_volume_command* cmd = (new sound_volume_command(
										 EVAL_ARG(0).as_decimal().as_float(), nseconds));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("decimal")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sound_volume)

	class stop_sound_command : public EntityCommandCallable
	{
	public:
		explicit stop_sound_command(const std::string& name, float fade_out_time)
		  : name_(name), fade_out_time_(fade_out_time)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			sound::stop_sound(name_, &ob, fade_out_time_);
		}
	private:
		std::string name_;
		float fade_out_time_;
	};

	FUNCTION_DEF(stop_sound, 1, 2, "stop_sound(string id, (opt) decimal fade_out_time): stops the sound that the current object is playing with the given id, if fade_out_time is given the sound fades out over this time before stopping")
		stop_sound_command* cmd = (new stop_sound_command(
						EVAL_ARG(0).as_string(),
						NUM_ARGS > 1 ? EVAL_ARG(1).as_float() : 0.0f));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("decimal")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(stop_sound)

	class preload_sound_command : public EntityCommandCallable
	{
	public:
		explicit preload_sound_command(const std::string& name)
		  : name_(name)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			sound::preload(name_);
		}
	private:
		std::string name_;
	};

	FUNCTION_DEF(preload_sound, 1, 1, "preload_sound(string id): preload the given sound so it'll be in the sound effects cache")
		preload_sound_command* cmd = (new preload_sound_command(
						EVAL_ARG(0).as_string()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(preload_sound)

	FUNCTION_DEF(create_haptic_effect, 2, 2, "create_haptic_effect(string id, map): Creates a haptic effect that can be played later.")
		const std::string s = EVAL_ARG(0).as_string();
		variant v = EVAL_ARG(1);
		return variant(new haptic::HapticEffectCallable(s,v));
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("map")
		RETURN_TYPE("HapticEffectCallable")
	END_FUNCTION_DEF(create_haptic_effect)

	class play_haptic_effect_command : public EntityCommandCallable
	{
	public:
		explicit play_haptic_effect_command(const std::string& name, int iterations)
		  : name_(name), iterations_(iterations)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			haptic::play(name_, iterations_);
		}
	private:
		std::string name_;
		int iterations_;
	};

	FUNCTION_DEF(play_haptic_effect, 1, 2, "play_haptic_effect(string id, (options) iterations): Plays the given haptic effect. Will default to 'rumble' if it couldn't be created.")
		int iterations = NUM_ARGS > 1 ? EVAL_ARG(1).as_int() : 1;
		play_haptic_effect_command* cmd = new play_haptic_effect_command(
			EVAL_ARG(0).as_string(),
			iterations);
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		RETURN_TYPE("commands")
	END_FUNCTION_DEF(play_haptic_effect)

	class stop_haptic_effect_command : public EntityCommandCallable
	{
	public:
		explicit stop_haptic_effect_command(const std::string& name)
		  : name_(name)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			haptic::stop(name_);
		}
	private:
		std::string name_;
	};

	FUNCTION_DEF(stop_haptic_effect_command, 1, 1, "stop_haptic_effect_command(string id): Stops the given haptic effect.")
		stop_haptic_effect_command* cmd = new stop_haptic_effect_command(EVAL_ARG(0).as_string());
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		RETURN_TYPE("commands")
	END_FUNCTION_DEF(stop_haptic_effect_command)

	class stop_all_haptic_effect_command : public EntityCommandCallable
	{
	public:
		stop_all_haptic_effect_command()
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			haptic::stop_all();
		}
	};

	FUNCTION_DEF(stop_all_haptic_effect_command, 0, 0, "stop_all_haptic_effect_command(string id): Stops the given haptic effect.")
		stop_all_haptic_effect_command* cmd = new stop_all_haptic_effect_command();
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		RETURN_TYPE("commands")
	END_FUNCTION_DEF(stop_all_haptic_effect_command)

	class screen_flash_command : public EntityCommandCallable
	{
	public:
		screen_flash_command(const KRE::ColorTransform& color,
							 const KRE::ColorTransform& delta, int duration)
		  : color_(color), delta_(delta), duration_(duration)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			screen_color_flash(color_, delta_, duration_);
		}
	private:
		KRE::ColorTransform color_, delta_;
		int duration_;
	};

	FUNCTION_DEF(screen_flash, 2, 3, "screen_flash(list int[4] color, (optional) list int[4] delta, int duration): flashes the screen the given color, and keeps the flash going for duration cycles. If delta is given, the color of the flash will be changed every cycle until the duration expires.")
		const variant color = EVAL_ARG(0);
		const variant delta = NUM_ARGS > 2 ? EVAL_ARG(1) : variant();
		const variant duration = EVAL_ARG(NUM_ARGS - 1);
		ASSERT_LOG(color.is_list() && color.num_elements() == 4 &&
				   (delta.is_null() || (delta.is_list() && delta.num_elements() == 4)) &&
				   duration.is_int(),
				   "BAD ARGUMENT TO screen_flash() FUNCTION: ARGUMENT FORMAT "
				   "IS screen_flash([r,g,b,a], (optional)[dr,dg,db,da], duration)");
		KRE::ColorTransform delta_color = KRE::ColorTransform(0,0,0,0);
		if(delta.is_null() == false) {
			delta_color = KRE::ColorTransform(
			  delta[0].as_int(), delta[1].as_int(),
			  delta[2].as_int(), delta[3].as_int());
		}

		screen_flash_command* cmd = (new screen_flash_command(
			KRE::ColorTransform(color[0].as_int(), color[1].as_int(),
								color[2].as_int(), color[3].as_int()),
			delta_color, duration.as_int()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("[int]")
		ARG_TYPE("[int]|int")
		ARG_TYPE("int|null")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(screen_flash)

	class title_command : public EntityCommandCallable
	{
	public:
		title_command(const std::string& title, int duration)
		  : title_(title), duration_(duration)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			set_scene_title(title_, duration_);
		}
	private:
		std::string title_;
		int duration_;
	};

	FUNCTION_DEF(title, 1, 2, "title(string text, int duration=50): shows Level title text on the screen for duration cycles")
		title_command* cmd = (new title_command(
		  EVAL_ARG(0).as_string(),
		  NUM_ARGS >= 2 ? EVAL_ARG(1).as_int() : 50));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(title)

	class shake_screen_command : public EntityCommandCallable
		{
		public:
			explicit shake_screen_command(int x_offset, int y_offset, int x_velocity, int y_velocity)
			: x_offset_(x_offset), y_offset_(y_offset), x_velocity_(x_velocity), y_velocity_(y_velocity)
			{}
			virtual void execute(Level& lvl, Entity& ob) const override {
				screen_position& screen_pos = last_draw_position();

				screen_pos.shake_x_offset = x_offset_;
				screen_pos.shake_y_offset = y_offset_;
				screen_pos.shake_x_vel = x_velocity_;
				screen_pos.shake_y_vel = y_velocity_;
			}
		private:
			int x_offset_,y_offset_,x_velocity_,y_velocity_;
		};

	FUNCTION_DEF(shake_screen, 4, 4, "shake_screen(int x_offset, int y_offset, int x_velocity, int y_velocity): makes the screen camera shake")
		shake_screen_command* cmd = (new shake_screen_command(
										EVAL_ARG(0).as_int(),
										EVAL_ARG(1).as_int(),
										EVAL_ARG(2).as_int(),
										EVAL_ARG(3).as_int() ));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(shake_screen)

	FUNCTION_DEF(radial_current, 2, 2, "radial_current(int intensity, int radius) -> current object: creates a current generator with the given intensity and radius")
		return variant(new RadialCurrentGenerator(EVAL_ARG(0).as_int(), EVAL_ARG(1).as_int()));
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("object")
	END_FUNCTION_DEF(radial_current)

	class execute_level_on_command : public CommandCallable
	{
		LevelPtr lvl_;
		variant cmd_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&lvl_);
			collector->surrenderVariant(&cmd_);
		}
	public:
		execute_level_on_command(LevelPtr lvl, variant cmd) : lvl_(lvl), cmd_(cmd)
		{}

		virtual void execute(game_logic::FormulaCallable& ob) const override {
			CurrentLevelScope scope(lvl_.get());
			ob.executeCommand(cmd_);
		}
	};

	FUNCTION_DEF(execute_on_level, 2, 2, "execute_on_level(Level lvl, command cmd): Executes the given commands, with the current level changed to the given level")
		LevelPtr lvl(EVAL_ARG(0).convert_to<Level>());
		variant command = EVAL_ARG(1);
		execute_level_on_command* cmd = (new execute_level_on_command(lvl, command));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("builtin level")
		ARG_TYPE("commands")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(execute_on_level)

	FUNCTION_DEF(create_level, 1, 1, "create_level")

		Formula::failIfStaticContext();

		variant node(EVAL_ARG(0));
		variant id = node["id"];
		std::string id_str;
		if(id.is_null()) {
			node.add_attr_mutation(variant("id"), variant("temp"));
			id_str = "temp.cfg";
		} else {
			id_str = id.as_string();
		}
		LevelPtr lvl(new Level(id_str, node));
		return variant(lvl.get());
	FUNCTION_ARGS_DEF
		ARG_TYPE("map")
	RETURN_TYPE("builtin level")
	END_FUNCTION_DEF(create_level)

	class execute_on_command : public CommandCallable
	{
		EntityPtr e_;
		variant cmd_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&e_);
			collector->surrenderVariant(&cmd_);
		}
	public:
		execute_on_command(EntityPtr e, variant cmd) : e_(e), cmd_(cmd)
		{}

		virtual void execute(game_logic::FormulaCallable& ob) const override {
			e_->executeCommand(cmd_);
		}
	};

	FUNCTION_DEF(execute, 2, 2, "execute(object context, command cmd): this function will execute the command or list of commands given by cmd on the object given by context. For instance, animation('foo') will set the current object to animation 'foo'. execute(obj, animation('foo')) can be used to set the object given by obj to the animation 'foo'.")
		EntityPtr e(EVAL_ARG(0).convert_to<Entity>());
		variant command = EVAL_ARG(1);
		execute_on_command* cmd = (new execute_on_command(e, command));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("commands")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(execute)

	class execute_on_command_instrumented : public EntityCommandCallable
	{
		EntityPtr e_;
		std::string id_;
		int time_;
		variant cmd_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&e_);
			collector->surrenderVariant(&cmd_);
		}
	public:
		execute_on_command_instrumented(EntityPtr e, std::string id, int t, variant cmd) : e_(e), id_(id), time_(t), cmd_(cmd)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			const int a = SDL_GetTicks();
			e_->executeCommand(cmd_);
			const int b = SDL_GetTicks();

			std::cerr << "Instrumented command: " << id_ << " ffl: " << time_ << "ms cmd: " << (b - a) << "ms\n";
		}
	};

	FUNCTION_DEF(execute_instrumented, 3, 3, "execute(object context, string id, command cmd): executes the given commands and instruments the time taken")
		EntityPtr e(EVAL_ARG(0).convert_to<Entity>());
		std::string id = EVAL_ARG(1).as_string();
		const int a = SDL_GetTicks();
		variant command = EVAL_ARG(2);
		const int b = SDL_GetTicks();
		execute_on_command_instrumented* cmd = (new execute_on_command_instrumented(e, id, b - a, command));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("string")
		ARG_TYPE("commands")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(execute_instrumented)

	class spawn_command : public EntityCommandCallable
	{
	public:
		spawn_command(ffl::IntrusivePtr<CustomObject> obj, variant instantiation_commands)
		  : obj_(obj), instantiation_commands_(instantiation_commands)
		{}
		virtual void execute(Level& lvl, Entity& ob) const override {
			obj_->setLevel(lvl);
			obj_->setSpawnedBy(ob.label());

			if(!place_entity_in_level_with_large_displacement(lvl, *obj_)) {
				return;
			}

			lvl.add_character(obj_);

			//send an event to the parent to let them know they've spawned a child,
			//and let them record the child's details.
			game_logic::MapFormulaCallable* spawn_callable(new game_logic::MapFormulaCallable);
			variant holder(spawn_callable);
			spawn_callable->add("spawner", variant(&ob));
			spawn_callable->add("child", variant(obj_.get()));
			ob.handleEvent(OBJECT_EVENT_CHILD_SPAWNED, spawn_callable);
			obj_->handleEvent(OBJECT_EVENT_SPAWNED, spawn_callable);

			obj_->executeCommand(instantiation_commands_);

			if(entity_collides(lvl, *obj_, MOVE_DIRECTION::NONE)) {
				lvl.remove_character(obj_);
			} else {
				//obj_->checkInitialized();
			}

			obj_->createObject();
		}
	private:
		ffl::IntrusivePtr<CustomObject> obj_;
		variant instantiation_commands_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&obj_);
			collector->surrenderVariant(&instantiation_commands_);
		}
	};

	FUNCTION_DEF_CTOR(spawn, 2, 6, "spawn(custom_obj|string type_id, int midpoint_x, int midpoint_y, (optional) properties map, (optional) list of commands cmd): will create a new object of type given by type_id with the given midpoint and facing. Immediately after creation the object will have any commands given by cmd executed on it. The child object will have the spawned event sent to it, and the parent object will have the child_spawned event sent to it.")
		base_slot_ = -1;
		const int ncommands_arg = args().size() <= 3 ? 2 : 4;
		if(args().size() > ncommands_arg) {
			base_slot_ = args()[ncommands_arg]->getDefinitionUsedByExpression()->getNumSlots()-1;
		}


		variant v;
		if(args().empty() == false && args()[0]->canReduceToVariant(v) && v.is_string()) {
			trackObjectSpawned(v.as_string());
		}

	FUNCTION_DEF_MEMBERS
		int base_slot_;
		bool optimizeArgNumToVM(int narg) const override {
			const int ncommands_arg = args().size() <= 3 ? 2 : 4;
			return narg != ncommands_arg;
		}
	FUNCTION_DEF_IMPL

		Formula::failIfStaticContext();

		variant obj_type = EVAL_ARG(0);

		ffl::IntrusivePtr<CustomObject> prototype;
		std::string type;
		if(obj_type.is_string()) {
			type = obj_type.as_string();
		} else {
			prototype.reset(obj_type.convert_to<CustomObject>());
		}

		int x = 0, y = 0;

		if(NUM_ARGS > 3) {
			x = EVAL_ARG(1).as_int();
			y = EVAL_ARG(2).as_int();
		}

		bool facing = true;

		variant arg3 = EVAL_ARG(NUM_ARGS > 3 ? 3 : 1);

		if(!arg3.is_map()) {
			facing = arg3.as_int() > 0;
		}

		ffl::IntrusivePtr<CustomObject> obj;
		if(prototype) {
			obj.reset(new CustomObject(*prototype));
			obj->setPos(x - obj->getCurrentFrame().width() / 2 , y - obj->getCurrentFrame().height() / 2);
		} else {
			obj.reset(new CustomObject(type, x, y, facing, true));
			obj->setPos(obj->x() - obj->getCurrentFrame().width() / 2 , obj->y() - obj->getCurrentFrame().height() / 2);
		}

		if(arg3.is_map()) {
			ConstCustomObjectTypePtr type_ptr = obj->getType();

			if(type_ptr->autoAnchor()) {
				static const variant XKey("x");
				static const variant YKey("y");
				static const variant X2Key("x2");
				static const variant Y2Key("y2");
				static const variant MidXKey("mid_x");
				static const variant MidYKey("mid_y");

				if(arg3.has_key(XKey)) {
					obj->setAnchorX(decimal::from_int(0));
				}

				if(arg3.has_key(X2Key)) {
					obj->setAnchorX(decimal::from_int(1));
				}

				if(arg3.has_key(MidXKey)) {
					obj->setAnchorX(decimal::from_int(1)/2);
				}

				if(arg3.has_key(YKey)) {
					obj->setAnchorY(decimal::from_int(0));
				}

				if(arg3.has_key(Y2Key)) {
					obj->setAnchorY(decimal::from_int(1));
				}

				if(arg3.has_key(MidYKey)) {
					obj->setAnchorY(decimal::from_int(1)/2);
				}
			}

			variant last_key;

			variant properties = arg3;
			variant keys = properties.getKeys();
			std::vector<std::pair<int,variant> > deferred_properties;
			const std::vector<int>& properties_with_setter = type_ptr->getPropertiesWithSetter();
			for(int n = 0; n != keys.num_elements(); ++n) {
				ASSERT_LOG(keys[n].is_string(), "Non-string key in spawn map: " << keys[n].write_json());
				const std::string& k = keys[n].as_string();
				if(type_ptr->getLastInitializationProperty().empty() == false && type_ptr->getLastInitializationProperty() == k) {
					last_key = keys[n];
					continue;
				}

				const int slot = type_ptr->callableDefinition()->getSlot(k);
				ASSERT_LOG(slot >= 0, "Could not look up key in object: " << k << " in " << type_ptr->id());

				variant value = properties[keys[n]];

				if(std::binary_search(properties_with_setter.begin(), properties_with_setter.end(), slot - type_ptr->getSlotPropertiesBase())) {
					deferred_properties.emplace_back(slot, value);
					continue;
				}

				obj->mutateValueBySlot(slot, value);
			}

			for(const std::pair<int,variant>& p : deferred_properties) {
				obj->mutateValueBySlot(p.first, p.second);
			}

			if(last_key.is_string()) {
				variant value = properties[last_key];
				obj->mutateValue(last_key.as_string(), value);
			}
		}

		obj->construct();

		const int ncommands_arg = NUM_ARGS <= 3 ? 2 : 4;

		variant commands;
		if(NUM_ARGS > ncommands_arg) {
			//Note that we insert the 'child' argument here, into the slot
			//formula callable. This relies on code in formula.cpp to look for
			//spawn() and give a callable definition with the child.
			ffl::IntrusivePtr<SlotFormulaCallable> callable = new SlotFormulaCallable;
			callable->setFallback(&variables);
			callable->setBaseSlot(base_slot_);

			callable->add(variant(obj.get()));
			commands = args()[ncommands_arg]->evaluate(*callable);
		}

		spawn_command* cmd = (new spawn_command(obj, commands));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_DYNAMIC_ARGUMENTS
	FUNCTION_ARGS_DEF
		//ASSERT_LOG(false, "spawn() not supported in strict mode " << debugPinpointLocation());
		if(NUM_ARGS <= 3) {
			ARG_TYPE("custom_obj|string")
			ARG_TYPE("map")
			ARG_TYPE("commands")
		} else {
			ARG_TYPE("custom_obj|string")
			ARG_TYPE("int|decimal")
			ARG_TYPE("int|decimal")
			ARG_TYPE("int|map")
			ARG_TYPE("commands")
		}

		std::string type_str;

		variant v;
		if(args()[0]->canReduceToVariant(v) && v.is_string()) {
			type_str = v.as_string();
		} else {
			variant_type_ptr type = args()[0]->queryVariantType();
			if(type && type->is_custom_object()) {
				type_str = *type->is_custom_object();
			}
		}

		if(type_str.empty() == false) {
			game_logic::FormulaCallableDefinitionPtr type_def = CustomObjectType::getDefinition(type_str);
			const CustomObjectCallable* type = dynamic_cast<const CustomObjectCallable*>(type_def.get());
			ASSERT_LOG(type, "Illegal object type: " << type_str << " " << debugPinpointLocation());

			const int nmap_arg = NUM_ARGS <= 3 ? 1 : 3;
			if(NUM_ARGS > nmap_arg) {
				variant_type_ptr map_type = args()[nmap_arg]->queryVariantType();
				assert(map_type);

				const std::map<variant, variant_type_ptr>* props = map_type->is_specific_map();
				if(props) {
					for(int slot : type->slots_requiring_initialization()) {
						const std::string& prop_id = type->getEntry(slot)->id;
						ASSERT_LOG(props->count(variant(prop_id)), "Must initialize " << type_str << "." << prop_id << " " << debugPinpointLocation());
					}

					for(std::map<variant,variant_type_ptr>::const_iterator itor = props->begin(); itor != props->end(); ++itor) {
						const int slot = type->getSlot(itor->first.as_string());
						ASSERT_LOG(slot >= 0, "Unknown property " << type_str << "." << itor->first.as_string() << " " << debugPinpointLocation());

						const FormulaCallableDefinition::Entry& entry = *type->getEntry(slot);
						ASSERT_LOG(variant_types_compatible(entry.getWriteType(), itor->second), "Initializing property " << type_str << "." << itor->first.as_string() << " with type " << itor->second->to_string() << " when " << entry.getWriteType()->to_string() << " is expected " << debugPinpointLocation());
					}
				}
			}

			ASSERT_LOG(type->slots_requiring_initialization().empty() || (NUM_ARGS > nmap_arg && args()[nmap_arg]->queryVariantType()->is_map_of().first), "Illegal spawn of " << type_str << " property " << type->getEntry(type->slots_requiring_initialization()[0])->id << " requires initialization " << debugPinpointLocation());
		}
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(spawn)

	FUNCTION_DEF_CTOR(spawn_player, 4, 5, "spawn_player(string type_id, int midpoint_x, int midpoint_y, int facing, (optional) list of commands cmd): identical to spawn except that the new object is playable.")
		base_slot_ = -1;
		if(args().size() > 4) {
			base_slot_ = args()[4]->getDefinitionUsedByExpression()->getNumSlots()-1;
		}
	FUNCTION_DEF_MEMBERS
		int base_slot_;
		bool optimizeArgNumToVM(int narg) const override {
			return narg != 4;
		}
	FUNCTION_DEF_IMPL

		Formula::failIfStaticContext();

		const std::string type = EVAL_ARG(0).as_string();
		const int x = EVAL_ARG(1).as_int();
		const int y = EVAL_ARG(2).as_int();
		const bool facing = EVAL_ARG(3).as_int() > 0;
		ffl::IntrusivePtr<CustomObject> obj(new PlayableCustomObject(type, x, y, facing, true));
		obj->setPos(obj->x() - obj->getCurrentFrame().width() / 2 , obj->y() - obj->getCurrentFrame().height() / 2);

		variant commands;
		if(NUM_ARGS > 4) {
			//Note that we insert the 'child' argument here, into the slot
			//formula callable. This relies on code in formula.cpp to look for
			//spawn() and give a callable definition with the child.
			ffl::IntrusivePtr<SlotFormulaCallable> callable = new SlotFormulaCallable;
			callable->setFallback(&variables);
			callable->setBaseSlot(base_slot_);

			callable->add(variant(obj.get()));
			commands = args()[4]->evaluate(*callable);
		}

		obj->construct();

		spawn_command* cmd = (new spawn_command(obj, commands));
		cmd->setExpression(this);
		return variant(cmd);

	FUNCTION_DYNAMIC_ARGUMENTS
	CAN_VM
		return false;
	FUNCTION_VM
		return ExpressionPtr();
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("commands")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(spawn_player)

	FUNCTION_DEF(object, 1, 5, "object(string type_id, int midpoint_x, int midpoint_y, (optional) map properties) -> object: constructs and returns a new object. Note that the difference between this and spawn is that spawn returns a command to actually place the object in the Level. object only creates the object and returns it. It may be stored for later use.")

		Formula::failIfStaticContext();

		variant obj_type = EVAL_ARG(0);

		ffl::IntrusivePtr<CustomObject> prototype;
		std::string type;
		if(obj_type.is_string()) {
			type = obj_type.as_string();
		} else {
			prototype.reset(obj_type.convert_to<CustomObject>());
		}

		ffl::IntrusivePtr<CustomObject> obj;

		variant properties;

		if(NUM_ARGS > 2) {
			const int x = EVAL_ARG(1).as_int();
			const int y = EVAL_ARG(2).as_int();

			if(NUM_ARGS >= 4) {
				properties = EVAL_ARG(3);
			}
			bool face_right = true;
			if(!properties.is_map()) {
				face_right = properties.as_int() > 0;
			}

			if(prototype) {
				obj.reset(new CustomObject(*prototype));
			} else {
				obj.reset(new CustomObject(type, x, y, face_right, true));
			}
		} else {
			const int x = 0;
			const int y = 0;
			const bool face_right = true;
			if(NUM_ARGS > 1) {
				properties = EVAL_ARG(1);
			}
			if(prototype) {
				obj.reset(new CustomObject(*prototype));
			} else {
				obj.reset(new CustomObject(type, x, y, face_right, true));
			}
		}

		//adjust so the object's x/y is its midpoint.
		obj->setPos(obj->x() - obj->getCurrentFrame().width() / 2 , obj->y() - obj->getCurrentFrame().height() / 2);

		if(NUM_ARGS > 4) {
			properties = EVAL_ARG(4);
		}

		if(properties.is_map()) {
			ConstCustomObjectTypePtr type_ptr = obj->getType();
			if(type_ptr->autoAnchor()) {
				static const variant XKey("x");
				static const variant YKey("y");
				static const variant X2Key("x2");
				static const variant Y2Key("y2");
				static const variant MidXKey("mid_x");
				static const variant MidYKey("mid_y");

				if(properties.has_key(XKey)) {
					obj->setAnchorX(decimal::from_int(0));
				}

				if(properties.has_key(X2Key)) {
					obj->setAnchorX(decimal::from_int(1));
				}

				if(properties.has_key(MidXKey)) {
					obj->setAnchorX(decimal::from_int(1)/2);
				}

				if(properties.has_key(YKey)) {
					obj->setAnchorY(decimal::from_int(0));
				}

				if(properties.has_key(Y2Key)) {
					obj->setAnchorY(decimal::from_int(1));
				}

				if(properties.has_key(MidYKey)) {
					obj->setAnchorY(decimal::from_int(1)/2);
				}
			}

			variant last_key;
			variant keys = properties.getKeys();
			std::vector<std::pair<int,variant> > deferred_properties;
			const std::vector<int>& properties_with_setter = type_ptr->getPropertiesWithSetter();
			for(int n = 0; n != keys.num_elements(); ++n) {
				if(type_ptr->getLastInitializationProperty().empty() == false && type_ptr->getLastInitializationProperty() == keys[n].as_string()) {
					last_key = keys[n];
					continue;
				}

				const std::string& k = keys[n].as_string();
				const int slot = type_ptr->callableDefinition()->getSlot(k);
				ASSERT_LOG(slot >= 0, "Could not look up key in object: " << k << " in " << type_ptr->id());

				variant value = properties[keys[n]];

				if(std::binary_search(properties_with_setter.begin(), properties_with_setter.end(), slot - type_ptr->getSlotPropertiesBase())) {
					deferred_properties.emplace_back(slot, value);
					continue;
				}

				obj->mutateValueBySlot(slot, value);
			}

			for(const std::pair<int,variant>& p : deferred_properties) {
				obj->mutateValueBySlot(p.first, p.second);
			}

			if(last_key.is_string()) {
				variant value = properties[last_key];
				obj->mutateValue(last_key.as_string(), value);
			}
		}

		obj->construct();

		return variant(obj.get());
	FUNCTION_ARGS_DEF
		if(NUM_ARGS == 2) {
			ARG_TYPE("custom_obj|string")
			ARG_TYPE("map")
		} else {
			ARG_TYPE("custom_obj|string")
			ARG_TYPE("int|decimal")
			ARG_TYPE("int|decimal")
			ARG_TYPE("int|map")
			ARG_TYPE("map")
		}

		std::string type_str;

		variant v;
		if(args()[0]->canReduceToVariant(v) && v.is_string()) {
			type_str = v.as_string();
		} else {
			variant_type_ptr type = args()[0]->queryVariantType();
			if(type && type->is_custom_object()) {
				type_str = *type->is_custom_object();
			}
		}

		if(type_str.empty() == false) {
			game_logic::FormulaCallableDefinitionPtr type_def = CustomObjectType::getDefinition(type_str);
			const CustomObjectCallable* type = dynamic_cast<const CustomObjectCallable*>(type_def.get());
			ASSERT_LOG(type, "Illegal object type: " << type_str << " " << debugPinpointLocation());

			const int nmap_arg = NUM_ARGS == 2 ? 1 : 4;
			if(NUM_ARGS > nmap_arg) {
				variant_type_ptr map_type = args()[nmap_arg]->queryVariantType();
				assert(map_type);

				const std::map<variant, variant_type_ptr>* props = map_type->is_specific_map();
				if(props) {
					for(int slot : type->slots_requiring_initialization()) {
						const std::string& prop_id = type->getEntry(slot)->id;
						ASSERT_LOG(props->count(variant(prop_id)), "Must initialize " << type_str << "." << prop_id << " " << debugPinpointLocation());
					}

					for(std::map<variant,variant_type_ptr>::const_iterator itor = props->begin(); itor != props->end(); ++itor) {
						const int slot = type->getSlot(itor->first.as_string());
						ASSERT_LOG(slot >= 0, "Unknown property " << type_str << "." << itor->first.as_string() << " " << debugPinpointLocation());

						const FormulaCallableDefinition::Entry& entry = *type->getEntry(slot);
						ASSERT_LOG(variant_types_compatible(entry.getWriteType(), itor->second), "Initializing property " << type_str << "." << itor->first.as_string() << " with type " << itor->second->to_string() << " when " << entry.getWriteType()->to_string() << " is expected " << debugPinpointLocation());
					}
				}
			}
		}

	//TODO: make this dynamically calculate the creature type
	FUNCTION_TYPE_DEF
		variant v;
		if(args()[0]->canReduceToVariant(v) && v.is_string()) {
			return variant_type::get_custom_object(v.as_string());
		} else {
			variant_type_ptr type = args()[0]->queryVariantType();
			if(type && type->is_custom_object()) {
				return variant_type::get_custom_object(*type->is_custom_object());
			}
		}

		return parse_variant_type(variant("custom_obj"));

	END_FUNCTION_DEF(object)

	FUNCTION_DEF(object_playable, 1, 5, "object_playable(string type_id, int midpoint_x, int midpoint_y, int facing, (optional) map properties) -> object: constructs and returns a new object. Note that the difference between this and spawn is that spawn returns a command to actually place the object in the Level. object_playable only creates the playble object and returns it. It may be stored for later use.")
		Formula::failIfStaticContext();
		const std::string type = EVAL_ARG(0).as_string();
		ffl::IntrusivePtr<CustomObject> obj;

		if(NUM_ARGS > 1) {
			const int x = EVAL_ARG(1).as_int();
			const int y = EVAL_ARG(2).as_int();
			const bool face_right = EVAL_ARG(3).as_int() > 0;
			obj = new PlayableCustomObject(CustomObject(type, x, y, face_right));
		} else {
			const int x = 0;
			const int y = 0;
			const bool face_right = true;
			obj = new PlayableCustomObject(CustomObject(type, x, y, face_right));
		}

		//adjust so the object's x/y is its midpoint.
		obj->setPos(obj->x() - obj->getCurrentFrame().width() / 2 , obj->y() - obj->getCurrentFrame().height() / 2);

		if(NUM_ARGS > 4) {
			ConstCustomObjectTypePtr type_ptr = CustomObjectType::getOrDie(type);
			variant last_key;
			variant properties = EVAL_ARG(4);
			variant keys = properties.getKeys();
			for(int n = 0; n != keys.num_elements(); ++n) {
				if(type_ptr->getLastInitializationProperty().empty() == false && type_ptr->getLastInitializationProperty() == keys[n].as_string()) {
					last_key = keys[n];
					continue;
				}
				variant value = properties[keys[n]];
				obj->mutateValue(keys[n].as_string(), value);
			}

			if(last_key.is_string()) {
				variant value = properties[last_key];
				obj->mutateValue(last_key.as_string(), value);
			}
		}

		obj->construct();

		return variant(obj.get());
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("map")

		variant v;
		if(args()[0]->canReduceToVariant(v) && v.is_string()) {
			game_logic::FormulaCallableDefinitionPtr type_def = CustomObjectType::getDefinition(v.as_string());
			const CustomObjectCallable* type = dynamic_cast<const CustomObjectCallable*>(type_def.get());
			ASSERT_LOG(type, "Illegal object type: " << v.as_string() << " " << debugPinpointLocation());

			if(NUM_ARGS > 4) {
				variant_type_ptr map_type = args()[4]->queryVariantType();
				assert(map_type);

				const std::map<variant, variant_type_ptr>* props = map_type->is_specific_map();
				if(props) {
					for(int slot : type->slots_requiring_initialization()) {
						const std::string& prop_id = type->getEntry(slot)->id;
						ASSERT_LOG(props->count(variant(prop_id)), "Must initialize " << v.as_string() << "." << prop_id << " " << debugPinpointLocation());
					}

					for(std::map<variant,variant_type_ptr>::const_iterator itor = props->begin(); itor != props->end(); ++itor) {
						const int slot = type->getSlot(itor->first.as_string());
						ASSERT_LOG(slot >= 0, "Unknown property " << v.as_string() << "." << itor->first.as_string() << " " << debugPinpointLocation());

						const FormulaCallableDefinition::Entry& entry = *type->getEntry(slot);
						ASSERT_LOG(variant_types_compatible(entry.getWriteType(), itor->second), "Initializing property " << v.as_string() << "." << itor->first.as_string() << " with type " << itor->second->to_string() << " when " << entry.getWriteType()->to_string() << " is expected " << debugPinpointLocation());
					}
				}
			}
		}

	//TODO: make this dynamically calculate the creature type
	FUNCTION_TYPE_DEF
		variant v;
		if(args()[0]->canReduceToVariant(v) && v.is_string()) {
			return variant_type::get_custom_object(v.as_string());
		}

		return parse_variant_type(variant("custom_obj"));

	END_FUNCTION_DEF(object_playable)


	FUNCTION_DEF(is_object, 1, 1, "is_object(obj_type): Returns true if passing the string to object(obj_type, ...) would work, false if a fatal assert would be thrown.")
		return variant::from_bool(CustomObjectType::hasDefinition(EVAL_ARG(0).as_string()));
	FUNCTION_ARGS_DEF
        ARG_TYPE("string")
	RETURN_TYPE("bool")
	END_FUNCTION_DEF(is_object)


    FUNCTION_DEF(get_object_type_animation, 2, 2, "get_object_type_animation(obj_type, animation_name): returns the animation object for the given object type")
        std::string objTypeName = EVAL_ARG(0).as_string();
        std::string anim_name = EVAL_ARG(1).as_string();
        auto objType = CustomObjectType::get(objTypeName);
        ASSERT_LOG(objType.get() != nullptr, "No object type: " << objTypeName);
        return variant(&objType->getFrame(anim_name));
    FUNCTION_ARGS_DEF
        ARG_TYPE("string")
        ARG_TYPE("string")
    RETURN_TYPE("builtin frame")
    END_FUNCTION_DEF(get_object_type_animation)

	class animation_command : public CustomObjectCommandCallable
	{
	public:
		animation_command(const std::string& anim)
		  : anim_(anim)
		{}

		virtual void execute(Level& lvl, CustomObject& ob) const override {
			ob.setFrame(anim_);
		}
	private:
		std::string anim_;
	};


	FUNCTION_DEF(animation, 1, 1, "animation(string id): changes the current object's animation to the given animation. time_in_animation is reset to 0.")
		animation_command* cmd = (new animation_command(EVAL_ARG(0).as_string()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(animation)

	class die_command : public CustomObjectCommandCallable
	{
	public:
		virtual void execute(Level& lvl, CustomObject& ob) const override {
			ob.die();
		}
	};

	FUNCTION_DEF(die, 0, 0, "die(): causes the current object to die. The object will receive the on_die signal and may even use it to resurrect itself. Use remove_object() to remove an object from play without it receiving on_die.")
		die_command* cmd = (new die_command());
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(die)

	class facing_command : public CustomObjectCommandCallable
	{
	public:
		explicit facing_command(int facing) : facing_(facing)
		{}
		virtual void execute(Level& lvl, CustomObject& ob) const override {
			ob.setFacingRight(facing_ > 0);
		}
	private:
		int facing_;
	};

	FUNCTION_DEF(facing, 1, 1, "facing(int new_facing): changes the current object's facing according to the value of new_facing (1 for right, otherwise left).")
		facing_command* cmd = (new facing_command(EVAL_ARG(0).as_int()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(facing)

	FUNCTION_DEF(debug_all_custom_objects, 0, 0, "debug_all_custom_objects(): gets access to all custom objects in memory")
		std::vector<variant> v;
		for(CustomObject* obj : CustomObject::getAll()) {
			v.emplace_back(obj);
		}

		return variant(&v);
	RETURN_TYPE("[object]")
	END_FUNCTION_DEF(debug_all_custom_objects)


	class add_debug_chart_command : public game_logic::CommandCallable {
		std::string id_;
		decimal value_;
	public:
		add_debug_chart_command(const std::string& id, decimal value)
		  : id_(id), value_(value)
		{}

		virtual void execute(game_logic::FormulaCallable& ob) const override {
		debug_console::add_graph_sample(id_, value_);
		}
	};

	FUNCTION_DEF(debug_chart, 2, 2, "debug_chart(string id, decimal value): plots a sample in a graph")
		add_debug_chart_command* cmd = (new add_debug_chart_command(EVAL_ARG(0).as_string(), EVAL_ARG(1).as_decimal()));
		cmd->setExpression(this);
		return variant(cmd);
	END_FUNCTION_DEF(debug_chart)

	class add_debug_rect_command : public game_logic::CommandCallable {
		rect r_;
	public:
		explicit add_debug_rect_command(const rect& r) : r_(r)
		{}

		virtual void execute(game_logic::FormulaCallable& ob) const override {
			add_debug_rect(r_);
		}
	};

	FUNCTION_DEF_CTOR(solid, 3, 6, "solid(Level, int x, int y, (optional)int w=1, (optional) int h=1, (optional) map options={}) -> boolean: returns true iff the Level contains solid space within the given (x,y,w,h) rectangle. If 'debug' is set, then the tested area will be displayed on-screen.")
		dynamic_options_ = true;
		static_dimensions_ = 0;

		variant v;
		if(args().size() >= 6 && args()[5]->canReduceToVariant(v)) {
			dynamic_options_ = false;
			if(v.is_map()) {
				variant dim_v = v["dimensions"];
				std::vector<std::string> flags = dim_v.as_list_string();
				for(auto f : flags) {
					static_dimensions_ |= (1 << get_solid_dimension_id(f));
				}
			}
		}
	FUNCTION_DEF_MEMBERS
		bool dynamic_options_;
		unsigned int static_dimensions_;
	FUNCTION_DEF_IMPL
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		const int x = EVAL_ARG(1).as_int();
		const int y = EVAL_ARG(2).as_int();
		int w = NUM_ARGS >= 4 ? EVAL_ARG(3).as_int() : 1;
		int h = NUM_ARGS >= 5 ? EVAL_ARG(4).as_int() : 1;
		rect r(x, y, w, h);
		bool level_collision = lvl->solid(r);

		if(level_collision) {
			return variant::from_bool(true);
		}

		unsigned int solid_dimensions = static_dimensions_;
		if(dynamic_options_ && NUM_ARGS >= 6) {
			variant v = EVAL_ARG(5);
			const std::map<variant,variant>& m = v.as_map();
			static const variant DimensionsStr("dimensions");
			auto dimensions_itor = m.find(DimensionsStr);
			if(dimensions_itor != m.end()) {
				const std::vector<std::string>& flags = dimensions_itor->second.as_list_string();

				for(auto f : flags) {
					solid_dimensions |= (1 << get_solid_dimension_id(f));
				}

			}
		}

		if(solid_dimensions == 0) {
			return variant::from_bool(false);
		} else {
			const std::vector<EntityPtr>& v = lvl->get_solid_chars();
			for(const EntityPtr& p : v) {
				if((p->getSolidDimensions()&solid_dimensions) == 0) {
					continue;
				}

				if(rects_intersect(r, p->solidRect())) {
					return variant::from_bool(true);
				}
			}
		}

		return variant::from_bool(false);

FUNCTION_ARGS_DEF
	ARG_TYPE("object")
	ARG_TYPE("int")
	ARG_TYPE("int")
	ARG_TYPE("int")
	ARG_TYPE("int")
	ARG_TYPE("{ dimensions: null|[string] }")
RETURN_TYPE("bool")
	END_FUNCTION_DEF(solid)

	FUNCTION_DEF(debug_rect, 2, 4, "debug_rect(int x, int y, (optional)int w=1, (optional) int h=1) -> Draws, for one frame, a rectangle on the Level")
		const int x = EVAL_ARG(0).as_int();
		const int y = EVAL_ARG(1).as_int();

		int w = NUM_ARGS >= 3 ? EVAL_ARG(2).as_int() : 100;
		int h = NUM_ARGS >= 4 ? EVAL_ARG(3).as_int() : 100;

		rect r(x, y, w, h);
		add_debug_rect_command* cmd = (new add_debug_rect_command(r));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(debug_rect)

	FUNCTION_DEF(plot_x, 1, 1, "plot_x(int x): plots a vertical debug line at the given position")
		const int x = EVAL_ARG(0).as_int();
		add_debug_rect_command* cmd = (new add_debug_rect_command(rect(x, -32000, 2, 64000)));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(plot_x)

	FUNCTION_DEF(plot_y, 1, 1, "plot_y(int x): plots a horizontal debug line at the given position")
		const int y = EVAL_ARG(0).as_int();
		add_debug_rect_command* cmd = (new add_debug_rect_command(rect(-32000, y, 64000, 2)));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(plot_y)

	FUNCTION_DEF(point_solid, 4, 4, "point_solid(Level, object, int x, int y) -> boolean: returns true iff the given point is solid for the given object")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		Entity* obj = EVAL_ARG(1).convert_to<Entity>();
		const int x = EVAL_ARG(2).as_int();
		const int y = EVAL_ARG(3).as_int();

		return variant(point_standable(*lvl, *obj, x, y, nullptr, SOLID_ONLY));
	FUNCTION_ARGS_DEF
		ARG_TYPE("builtin level")
		ARG_TYPE("custom_obj")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("bool")
	END_FUNCTION_DEF(point_solid)


	FUNCTION_DEF(find_first_solid_point_on_line, 7, 7, "find_first_solid_point_on_line(Level, object, int x1, int y1, int x2, int y2, decimal step_size)")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		Entity* obj = EVAL_ARG(1).convert_to<Entity>();
		const int x = EVAL_ARG(2).as_int();
		const int y = EVAL_ARG(3).as_int();
		const int x2 = EVAL_ARG(4).as_int();
		const int y2 = EVAL_ARG(5).as_int();
		float step_size = EVAL_ARG(6).as_decimal().as_float32();

		float line_length = sqrt((x - x2)*(x - x2) + (y - y2)*(y - y2));
		int num_steps = static_cast<int>(line_length/step_size);

		for(int i = 0; i < num_steps+1; ++i) {
			int denominator = num_steps;
			if(denominator <= 0) {
				denominator = 1;
			}
			int xpos = (i*x2 + (num_steps-i)*x)/denominator;
			int ypos = (i*y2 + (num_steps-i)*y)/denominator;

			if(point_standable(*lvl, *obj, xpos, ypos, nullptr, SOLID_ONLY)) {
				std::vector<variant> res;
				res.emplace_back(xpos);
				res.emplace_back(ypos);
				return variant(&res);
			}
		}

		return variant();

	FUNCTION_ARGS_DEF
		ARG_TYPE("builtin level")
		ARG_TYPE("custom_obj")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("decimal")
		RETURN_TYPE("null|[int,int]")
	END_FUNCTION_DEF(find_first_solid_point_on_line)


	FUNCTION_DEF(find_first_open_point_on_line, 7, 7, "find_first_solid_point_on_line(Level, object, int x1, int y1, int x2, int y2, decimal step_size)")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		Entity* obj = EVAL_ARG(1).convert_to<Entity>();
		const int x = EVAL_ARG(2).as_int();
		const int y = EVAL_ARG(3).as_int();
		const int x2 = EVAL_ARG(4).as_int();
		const int y2 = EVAL_ARG(5).as_int();
		float step_size = EVAL_ARG(6).as_decimal().as_float32();

		float line_length = sqrt((x - x2)*(x - x2) + (y - y2)*(y - y2));
		int num_steps = static_cast<int>(line_length/step_size);

		for(int i = 0; i < num_steps+1; ++i) {
			int denominator = num_steps;
			if(denominator <= 0) {
				denominator = 1;
			}
			int xpos = (i*x2 + (num_steps-i)*x)/denominator;
			int ypos = (i*y2 + (num_steps-i)*y)/denominator;

			if(!point_standable(*lvl, *obj, xpos, ypos, nullptr, SOLID_ONLY)) {
				std::vector<variant> res;
				res.emplace_back(xpos);
				res.emplace_back(ypos);
				return variant(&res);
			}
		}

		return variant();

	FUNCTION_ARGS_DEF
		ARG_TYPE("builtin level")
		ARG_TYPE("custom_obj")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("decimal")
		RETURN_TYPE("null|[int,int]")
	END_FUNCTION_DEF(find_first_open_point_on_line)


	FUNCTION_DEF(find_point_solid, 6, 7, "find_point_solid(Level, object, int x, int y, int dx, int dy, int max_search=1000) -> boolean: returns true iff the given point is solid for the given object")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		Entity* obj = EVAL_ARG(1).convert_to<Entity>();
		int x = EVAL_ARG(2).as_int();
		int y = EVAL_ARG(3).as_int();
		const int dx = EVAL_ARG(4).as_int();
		const int dy = EVAL_ARG(5).as_int();

		int max_search = 1000;
		if (NUM_ARGS > 6) {
			max_search = EVAL_ARG(6).as_int();
		}

		while (max_search > 0) {
			if (point_standable(*lvl, *obj, x, y, nullptr, SOLID_ONLY)) {
				break;
			}

			x += dx;
			y += dy;
		}

		std::vector<variant> res;
		res.emplace_back(x);
		res.emplace_back(y);
		return variant(&res);

	return variant(point_standable(*lvl, *obj, x, y, nullptr, SOLID_ONLY));
	FUNCTION_ARGS_DEF
		ARG_TYPE("builtin level")
		ARG_TYPE("custom_obj")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")

		RETURN_TYPE("[int,int]")
	END_FUNCTION_DEF(find_point_solid)


	FUNCTION_DEF(object_can_stand, 4, 4, "object_can_stand(Level, object, int x, int y) -> boolean: returns true iff the given point is standable")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		Entity* obj = EVAL_ARG(1).convert_to<Entity>();
		const int x = EVAL_ARG(2).as_int();
		const int y = EVAL_ARG(3).as_int();

		return variant(point_standable(*lvl, *obj, x, y));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("object")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("bool")
	END_FUNCTION_DEF(object_can_stand)

	FUNCTION_DEF(find_point_object_can_stand_on, 6, 7, "find_point_object_can_stand_on(Level, object, int x, int y, int dx, int dy, int max_search=1000) -> [int,int]|null: returns the first point that an object can stand on, starting at [x,y] and incrementing by [dx,dy] until the point is found")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		Entity* obj = EVAL_ARG(1).convert_to<Entity>();
		int x = EVAL_ARG(2).as_int();
		int y = EVAL_ARG(3).as_int();
		const int dx = EVAL_ARG(4).as_int();
		const int dy = EVAL_ARG(5).as_int();
		const int niterations = NUM_ARGS > 6 ? EVAL_ARG(6).as_int() : 1000;

		int rect_x = x;
		int rect_y = y;
		int rect_w = dx*niterations;
		int rect_h = dy*niterations;
		if(rect_w < 0) {
			rect_x += rect_w;
			rect_w = -rect_w;
		} else if(rect_w == 0) {
			rect_w = 1;
		}

		if(rect_h < 0) {
			rect_y += rect_h;
			rect_h = -rect_h;
		} else if(rect_h == 0) {
			rect_h = 1;
		}

		std::vector<EntityPtr> entities = get_potentially_standable_objects_in_area(*lvl, *obj, rect(rect_x, rect_y, rect_w, rect_h));

		for(int n = 0; n < niterations; ++n) {
			if(point_standable(*lvl, *obj, entities, x, y)) {
				std::vector<variant> result;
				result.reserve(2);
				result.emplace_back(x);
				result.emplace_back(y);
				return variant(&result);
			}

			x += dx;
			y += dy;
		}

		return variant();
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("object")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("[int]|null")
	END_FUNCTION_DEF(find_point_object_can_stand_on)

	FUNCTION_DEF(standable, 3, 5, "standable(Level, int x, int y, (optional)int w=1, (optional) int h=1) -> boolean: returns true iff the Level contains standable space within the given (x,y,w,h) rectangle")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		const int x = EVAL_ARG(1).as_int();
		const int y = EVAL_ARG(2).as_int();

		int w = NUM_ARGS >= 4 ? EVAL_ARG(3).as_int() : 1;
		int h = NUM_ARGS >= 5 ? EVAL_ARG(4).as_int() : 1;

		rect r(x, y, w, h);

		return variant::from_bool(lvl->standable(r));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("bool")
	END_FUNCTION_DEF(standable)

	class set_solid_command : public EntityCommandCallable {
		rect r_;
		bool is_solid_;
		bool platforms_;
	public:
		set_solid_command(const rect& r, bool is_solid, bool platforms) : r_(r), is_solid_(is_solid), platforms_(platforms)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.set_solid_area(r_, is_solid_, platforms_);
		}
	};

	FUNCTION_DEF(set_solid, 4, 5, "setSolid(x1, y1, x2, y2, boolean is_solid=false): modifies the solidity of the Level such that the rectangle given by (x1, y1, x2, y2) will have its solidity set to the value of is_solid")
		set_solid_command* cmd = (new set_solid_command(
		  rect::from_coordinates(
			EVAL_ARG(0).as_int(),
			EVAL_ARG(1).as_int(),
			EVAL_ARG(2).as_int(),
			EVAL_ARG(3).as_int()),
			NUM_ARGS > 4 ? EVAL_ARG(4).as_bool() : false, false));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("bool")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_solid)

	FUNCTION_DEF(set_standable, 4, 5, "setStandable(x1, y1, x2, y2, boolean is_standable=false): modifies the standability (platforms) of the Level such that the rectangle given by (x1, y1, x2, y2) will have its standable set to the value of is_standable")
		set_solid_command* cmd = (new set_solid_command(
		  rect::from_coordinates(
			EVAL_ARG(0).as_int(),
			EVAL_ARG(1).as_int(),
			EVAL_ARG(2).as_int(),
			EVAL_ARG(3).as_int()),
			NUM_ARGS > 4 ? EVAL_ARG(4).as_bool() : false, true));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("bool")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_standable)

	FUNCTION_DEF(group_size, 2, 2, "group_size(Level, int group_id) -> int: gives the number of objects in the object group given by group_id")
		Level* lvl = EVAL_ARG(0).convert_to<Level>();
		return variant(lvl->group_size(EVAL_ARG(1).as_int()));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("int")
	RETURN_TYPE("int")
	END_FUNCTION_DEF(group_size)

	class set_group_command : public EntityCommandCallable {
	public:
		explicit set_group_command(int group=-1) : group_(group)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			int group = group_;
			if(group < 0) {
				if(ob.group() >= 0) {
					return;
				}
				group = lvl.add_group();
			}

			lvl.set_character_group(&ob, group);
		}

	private:
		int group_;
	};

	FUNCTION_DEF(set_group, 0, 1, "setGroup((optional)int group_id): sets the current object to have the given group id, or to be in no group if group_id is not given")
		int group = -1;
		if(NUM_ARGS > 0) {
			group = EVAL_ARG(0).as_int();
		}

		set_group_command* cmd = (new set_group_command(group));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_group)

	class scroll_to_command : public CustomObjectCommandCallable
	{
	public:
		explicit scroll_to_command(EntityPtr focus) : focus_(focus)
		{}
		virtual void execute(Level& lvl, CustomObject& ob) const override {
			screen_position pos = last_draw_position();
			for(int n = 0; n != 50; ++n) {
				draw_scene(lvl, pos, focus_.get());
				KRE::WindowManager::getMainWindow()->swap();
				profile::delay(preferences::frame_time_millis());
			}
		}
	private:
		EntityPtr focus_;
		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&focus_);
		}
	};

	FUNCTION_DEF(tiles_at, 2, 2, "tiles_at(x, y): gives a list of the tiles at the given x, y position")
		Formula::failIfStaticContext();

		std::vector<variant> v;

		std::pair<Level::TileItor, Level::TileItor> range = Level::current().tiles_at_loc(EVAL_ARG(0).as_int(), EVAL_ARG(1).as_int());
		while(range.first != range.second) {
			v.emplace_back(range.first->object);
			++range.first;
		}

		return variant(&v);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("[object]")
	END_FUNCTION_DEF(tiles_at)

	FUNCTION_DEF(set_tiles, 3, 3, "set_tiles(zorder, [area], tile): modify the tilemap within a certain area.")
		int zorder = EVAL_ARG(0).as_int();
		std::vector<int> r = EVAL_ARG(1).as_list_int();
		std::string t = EVAL_ARG(2).as_string();
		return variant(new FnCommandCallable("set_tiles", [=]() {
			Level::current().add_tile_rect(zorder, r[0], r[1], r[2], r[3], t);

			std::vector<int> layers;
			layers.push_back(zorder);
			Level::current().start_rebuild_tiles_in_background(layers);
		}));
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("[int,int,int,int]")
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_tiles)

	FUNCTION_DEF(complete_rebuild_tiles, 0, 1, "complete_rebuild_tiles(bool): run to complete the rebuild of tiles started by a previous call to set_tiles")
		bool wait = false;
		if(NUM_ARGS > 0) {
			wait = EVAL_ARG(0).as_bool();
		}
		return variant(new FnCommandCallable("set_tiles", [=]() {
			bool result = false;
			do {
				result = Level::current().complete_rebuild_tiles_in_background();
			} while(wait && result == false);
		}));
	FUNCTION_ARGS_DEF
		ARG_TYPE("bool")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(complete_rebuild_tiles)

	FUNCTION_DEF(get_objects_at_point, 2, 2, "get_objects_at_point(x, y): Returns all objects which intersect the specified x,y point, in absolute Level-coordinates.")
		Level* lvl = &Level::current();
		std::vector<EntityPtr> v;
		v = lvl->get_characters_at_point(EVAL_ARG(0).as_int(),
														 EVAL_ARG(1).as_int(),
														 static_cast<int>(last_draw_position().x / 100 * lvl->zoom_level()),
														 static_cast<int>(last_draw_position().y / 100 * lvl->zoom_level()));
		if(!v.empty()){
			std::vector<variant> res;
			res.reserve(v.size());
			for(const EntityPtr& e : v){
				res.emplace_back(e.get());
			}
			//for(int n = 0; n != v.size(); ++n) {
			//	res.push_back(variant(v[n].get()));
			//}
			return variant(&res);
		} else {
			return variant();
		}
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("[object]")
	END_FUNCTION_DEF(get_objects_at_point)

	FUNCTION_DEF(toggle_pause, 0, 0, "toggle_pause()")
		Formula::failIfStaticContext();
		return variant(new FnCommandCallable("toggle_pause", [=]() {
			LevelRunner::getCurrent()->toggle_pause();
		}));
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(toggle_pause)

	FUNCTION_DEF(hide_window, 0, 0, "hide_window()")
		Formula::failIfStaticContext();
		return variant(new FnCommandCallable("hide_window", [=]() {
			KRE::WindowManager::getMainWindow()->setVisible(false);
		}));
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(hide_window)

	FUNCTION_DEF(quit_to_desktop, 0, 0, "quit_to_desktop()")
		Formula::failIfStaticContext();
		return variant(new FnCommandCallable("quit_to_desktop", [=]() {
			LevelRunner::getCurrent()->quit_game();
		}));
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(quit_to_desktop)

	FUNCTION_DEF(update_and_restart, 0, 0, "update_and_restart()")
		Formula::failIfStaticContext();
		return variant(new FnCommandCallable("update_and_restart", [=]() {
			run_auto_updater();
		}));
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(update_and_restart)



	FUNCTION_DEF(scroll_to, 1, 1, "scroll_to(object target): scrolls the screen to the target object")
		scroll_to_command* cmd = (new scroll_to_command(EVAL_ARG(0).try_convert<Entity>()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(scroll_to)

	namespace {
	static int g_in_speech_dialog = 0;
	class in_speech_dialog_tracker {
	public:
		in_speech_dialog_tracker() : cancelled_(false) {
			g_in_speech_dialog++;
		}

		~in_speech_dialog_tracker() {
			cancel();
		}

		void cancel() {
			if(!cancelled_) {
				g_in_speech_dialog--;
				cancelled_ = true;
			}
		}

	private:
		bool cancelled_;
	};
	}

	class transient_speech_dialog_command : public CustomObjectCommandCallable {
		EntityPtr speaker_;
		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&speaker_);
		}
		std::vector<std::string> text_;
		int duration_;
	public:
		transient_speech_dialog_command(EntityPtr speaker, const std::vector<std::string>& text, int duration) : speaker_(speaker), text_(text), duration_(duration)
		{}
		virtual void execute(Level& lvl, CustomObject& ob) const override {
			std::shared_ptr<SpeechDialog> d(new SpeechDialog);
			if(speaker_) {
				d->setSpeaker(speaker_);
			} else {
				d->setSpeaker(EntityPtr(&ob));
			}

			d->setText(text_);
			d->setExpiration(duration_);
			lvl.add_speech_dialog(d);
		}
	};

	FUNCTION_DEF(transient_speech_dialog, 1, -1, "transient_speech_dialog(...): schedules a sequence of speech dialogs to be shown. Arguments may include a list of strings, which contain text. An integer which sets the duration of the dialog. An object which sets the speaker.")
		EntityPtr speaker;
		int duration = 100;

		std::vector<variant> result;

		for(int n = 0; n != NUM_ARGS; ++n) {
			variant v = EVAL_ARG(n);
			Entity* e = v.try_convert<Entity>();
			if(e) {
				speaker = EntityPtr(e);
			} else if(v.is_int()) {
				duration = v.as_int();
			} else if(v.is_list()) {
				std::vector<std::string> str;
				for(int m = 0; m != v.num_elements(); ++m) {
					str.push_back(v[m].as_string());
				}
				transient_speech_dialog_command* cmd = new transient_speech_dialog_command(speaker, str, duration);
				cmd->setExpression(this);
				result.emplace_back(cmd);
			} else {
				ASSERT_LOG(false, "UNRECOGNIZED ARGUMENT to transient_speech_dialog: " << v.to_debug_string());
			}
		}

		//we add the dialogs in reverse order to the Level to make it get
		//said in the correct order.
		std::reverse(result.begin(), result.end());

		return variant(&result);
	FUNCTION_ARGS_DEF
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(transient_speech_dialog)

	namespace {
	//if this variable is set we are in a controlled dialog sequence skipping
	//scope and the end of dialog shouldn't automatically end skipping dialog.
	bool skipping_dialog_sequence = false;

	struct in_dialog_setter {
		bool was_in_dialog_;
		Level& lvl_;
		in_dialog_setter(Level& lvl) : lvl_(lvl) {
			was_in_dialog_ = lvl_.in_dialog();
			lvl_.set_in_dialog(true);
		}
		~in_dialog_setter() {
			lvl_.set_in_dialog(was_in_dialog_);
			if(!was_in_dialog_ && !skipping_dialog_sequence) {
				end_skipping_game();
			}
		}
	};

	struct speech_dialog_scope {
		Level& lvl_;
		std::shared_ptr<SpeechDialog> dialog_;

		speech_dialog_scope(Level& lvl, std::shared_ptr<SpeechDialog> dialog)
		  : lvl_(lvl), dialog_(dialog)
		{
			lvl_.add_speech_dialog(dialog_);
		}

		~speech_dialog_scope()
		{
			lvl_.remove_speech_dialog();
		}
	};

	}

	class skip_sequence_command : public CustomObjectCommandCallable {
		bool skip_on_;
	public:
		explicit skip_sequence_command(bool skip_on)
		  : skip_on_(skip_on)
		{}

		virtual void execute(Level& lvl, CustomObject& ob) const override {
			skipping_dialog_sequence = skip_on_;
			if(!skip_on_) {
				end_skipping_game();
			}
		}
	};

	FUNCTION_DEF(begin_skip_dialog_sequence, 0, 0, "begin_skip_dialog_sequence(): command that will cause everything up until the next time end_skip_dialog_sequence() is called to be considered a single storyline sequence. If the player selects to skip the sequence between now and then everything up until the call to end_skip_dialog_sequence() will be skipped.")
		skip_sequence_command* cmd = (new skip_sequence_command(true));
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(begin_skip_dialog_sequence)

	FUNCTION_DEF(end_skip_dialog_sequence, 0, 0, "end_skip_dialog_sequence(): ends the sequence begun with begin_skip_dialog_sequence().")
		skip_sequence_command* cmd = (new skip_sequence_command(false));
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(end_skip_dialog_sequence)

	class speech_dialog_command : public CustomObjectCommandCallable {
	public:
		explicit speech_dialog_command(const std::vector<variant>& args, bool paused=false)
		  : args_(args), paused_(paused)
		{}
		virtual void execute(Level& lvl, CustomObject& ob) const override {
	//		pause_scope pauser;

			if(!g_in_speech_dialog) {
				for(const EntityPtr& e : lvl.get_chars()) {
					e->handleEvent(OBJECT_EVENT_BEGIN_DIALOG);
				}
			}

			formula_profiler::SuspendScope profiler_suspend;
			in_dialog_setter dialog_setter(lvl);

			//make it so the player's controls become locked for the duration of the dialog.
			controls::local_controls_lock controller_lock;

			executeCommands(lvl, ob, args_);
		}

	private:
		void executeCommands(Level& lvl, CustomObject& ob, const std::vector<variant>& commands) const {
			in_speech_dialog_tracker dialog_tracker;

			std::shared_ptr<SpeechDialog> d(new SpeechDialog());
			speech_dialog_scope dialog_scope(lvl, d);

			for(variant var : commands) {
				if(var.is_callable()) {
					ConstEntityPtr e = var.try_convert<Entity>();
					if(e) {
						d->setSpeakerAndFlipSide(e);
					}

					const EntityCommandCallable* Entity_cmd = var.try_convert<const EntityCommandCallable>();
					if(Entity_cmd) {
						Entity_cmd->runCommand(lvl, ob);
					}

					const CustomObjectCommandCallable* obj_cmd = var.try_convert<const CustomObjectCommandCallable>();
					if(obj_cmd) {
						obj_cmd->runCommand(lvl, ob);
					}

					const game_logic::CommandCallable* callable_cmd = var.try_convert<const game_logic::CommandCallable>();
					if(callable_cmd) {
						callable_cmd->runCommand(ob);
					}

				}

				if(var.is_list()) {
					if(var.num_elements() > 0 && var[0].is_callable()) {
						std::vector<variant> cmd;
						for(int n = 0; n != var.num_elements(); ++n) {
							cmd.push_back(var[n]);
						}

						executeCommands(lvl, ob, cmd);
						continue;
					}

					bool is_default = false;
					int default_option = -1;

					std::vector<variant> option_commands;
					std::vector<std::string> options;
					std::vector<std::string> message;
					for(int n = 0; n != var.num_elements(); ++n) {
						if(var[n].is_string() && var[n].as_string() == "default_skip") {
							is_default = true;
						} else if(message.empty() == false && var[n].is_list()) {
							if(is_default) {
								default_option = n;
								is_default = false;
							}

							options.push_back(message.back());
							message.pop_back();
							option_commands.push_back(var[n]);
						} else {
							message.push_back(var[n].as_string());
						}
					}

					d->setText(message);
					d->setOptions(options);

					bool done = false;
					while(!done) {
						if(!paused_) {
							debug_console::process_graph();
							lvl.process();
							lvl.process_draw();
						}

						SDL_Event event;
						while(input::sdl_poll_event(&event)) {
							switch(event.type) {
							case SDL_QUIT:
								throw InterruptGameException();
							case SDL_USEREVENT:
							case SDL_KEYDOWN:
								if(event.key.keysym.sym == SDLK_ESCAPE || event.type == SDL_USEREVENT) {
									begin_skipping_game();
									if(default_option != -1) {
										d->setOptionSelected(default_option);
									}
									break;
								}

							case SDL_MOUSEBUTTONDOWN:
							case SDL_MOUSEBUTTONUP:
							case SDL_MOUSEMOTION:
								done = done || d->keyPress(event);
								break;
							}
						}

						done = done || d->detectJoystickPress();

						if(paused_)
							done = done || d->process();
						else if(!lvl.current_speech_dialog())
							done = true;
						draw(lvl);
					}

					if(options.empty() == false) {
						const int index = d->getOptionSelected();
						if(index >= 0 && static_cast<unsigned>(index) < option_commands.size()) {
							dialog_tracker.cancel();
							ob.executeCommand(option_commands[index]);
						}
					}

					d->setOptions(std::vector<std::string>());
				}
			}
		}

		void draw(const Level& lvl) const {
			draw_scene(lvl, last_draw_position(), &lvl.player()->getEntity());

			KRE::WindowManager::getMainWindow()->swap();

			profile::delay(preferences::frame_time_millis());
		}

		std::vector<variant> args_;

		bool paused_;
	};

	FUNCTION_DEF(speech_dialog, 1, -1, "speech_dialog(...): schedules a sequence of speech dialogs to be shown modally. Arguments may include a list of strings, which contain text. An integer which sets the duration of the dialog. An object which sets the speaker. A string by itself indicates an option that should be shown for the player to select from. A string should be followed by a list of commands that will be executed should the player choose that option.")
		std::vector<variant> v;
		for(int n = 0; n != NUM_ARGS; ++n) {
			v.push_back(EVAL_ARG(n));
		}

		speech_dialog_command* cmd = (new speech_dialog_command(v));
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(speech_dialog)

	FUNCTION_DEF(paused_speech_dialog, 1, -1, "paused_speech_dialog(...): like SpeechDialog(), except the game is paused while the dialog is displayed.")
		std::vector<variant> v;
		for(int n = 0; n != NUM_ARGS; ++n) {
			v.push_back(EVAL_ARG(n));
		}

		speech_dialog_command* cmd = (new speech_dialog_command(v, true));
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(paused_speech_dialog)

	class end_game_command : public CustomObjectCommandCallable
	{
	public:
		virtual void execute(Level& lvl, CustomObject& ob) const override {
			lvl.set_end_game();
		}
	};

	FUNCTION_DEF(end_game, 0, 0, "end_game(): exits the game")
		end_game_command* cmd = (new end_game_command());
		cmd->setExpression(this);
		return variant(cmd);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(end_game)

	class AchievementCommand : public EntityCommandCallable
	{
	public:
		explicit AchievementCommand(const std::string& str) : str_(str)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			if(Achievement::attain(str_)) {
				AchievementPtr a = Achievement::get(str_);
				if(a) {
					stats::Entry("achievement").addPlayerPos().set("achievement", variant(str_));
					sound::play("achievement-attained.wav");
					set_displayed_Achievement(a);
				}
			}
		}
	private:
		std::string str_;
	};

	FUNCTION_DEF(achievement, 1, 1, "achievement(id): unlocks the achievement with the given id")
		AchievementCommand* cmd = (new AchievementCommand(EVAL_ARG(0).as_string()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(achievement)

	static int event_depth = 0;
	struct event_depth_scope {
		event_depth_scope() { ++event_depth; }
		~event_depth_scope() { --event_depth; }
	};

	class fire_event_command : public EntityCommandCallable {
		const EntityPtr target_;
		const std::string event_;
		const ConstFormulaCallablePtr callable_;
	public:
		fire_event_command(EntityPtr target, const std::string& event, ConstFormulaCallablePtr callable)
		  : target_(target), event_(event), callable_(callable)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			ASSERT_LOG(event_depth < 1000, "INFINITE (or too deep?) RECURSION FOR EVENT " << event_);
			event_depth_scope scope;
			Entity* e = target_ ? target_.get() : &ob;
			e->handleEvent(event_, callable_.get());
		}

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&target_);
			collector->surrenderPtr(&callable_);
		}
	};

	FUNCTION_DEF(fire_event, 1, 3, "fire_event((optional) object target, string id, (optional)callable arg): fires the event with the given id. Targets the current object by default, or target if given. Sends arg as the event argument if given")
		EntityPtr target;
		std::string event;
		ConstFormulaCallablePtr callable;
		variant arg_value;

		if(NUM_ARGS == 3) {
			variant v1 = EVAL_ARG(0);
			if(v1.is_null()) {
				return variant();
			}

			target = v1.convert_to<Entity>();
			event = EVAL_ARG(1).as_string();

			arg_value = EVAL_ARG(2);
		} else if(NUM_ARGS == 2) {
			variant v1 = EVAL_ARG(0);
			if(v1.is_null()) {
				return variant();
			}

			variant v2 = EVAL_ARG(1);
			if(v1.is_string()) {
				event = v1.as_string();
				arg_value = v2;
			} else {
				target = v1.convert_to<Entity>();
				event = v2.as_string();
			}
		} else {
			event = EVAL_ARG(0).as_string();
		}

		variant_type_ptr arg_type = get_object_event_arg_type(get_object_event_id(event));
		if(arg_type) {
			ASSERT_LOG(arg_type->match(arg_value), "Calling fire_event('" << event << "'), arg type does not match. Expected " << arg_type->to_string() << " found " << arg_value.write_json() << " which is a " << get_variant_type_from_value(arg_value)->to_string());
		}

		if(arg_value.is_null() == false) {
			callable = map_into_callable(arg_value);
		}

		fire_event_command* cmd = (new fire_event_command(target, event, callable));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object|string")
		ARG_TYPE("any")
		ARG_TYPE("any")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(fire_event)

	FUNCTION_DEF(proto_event, 2, 3, "proto_event(prototype, event_name, (optional) arg): for the given prototype, fire the named event. e.g. proto_event('playable', 'process')")
		const std::string proto = EVAL_ARG(0).as_string();
		const std::string event_type = EVAL_ARG(1).as_string();
		const std::string event_name = proto + "_PROTO_" + event_type;
		ConstFormulaCallablePtr callable;
		if(NUM_ARGS >= 3) {
			callable.reset(EVAL_ARG(2).as_callable());
		} else {
			variant arg = variables.queryValue("arg");
			if(arg.is_callable()) {
				callable.reset(arg.as_callable());
			} else {
				callable.reset(&variables);
			}
		}
		ASSERT_LOG(event_depth < 100, "Infinite (or too deep?) recursion in proto_event(" << proto << ", " << event_type << ")");
		fire_event_command* cmd = (new fire_event_command(EntityPtr(), event_name, callable));
		cmd->setExpression(this);
		return variant(cmd);

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("string")
		ARG_TYPE("any")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(proto_event)

	FUNCTION_DEF(get_object, 2, 2, "get_object(Level, string label) -> custom_obj|null: returns the object that is present in the given Level that has the given label")

		Level* lvl = EVAL_ARG(0).try_convert<Level>();
		if(lvl) {
			return variant(lvl->get_entity_by_label(EVAL_ARG(1).as_string()).get());
		}

		return variant();
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("string")
	RETURN_TYPE("custom_obj|null")
	END_FUNCTION_DEF(get_object)

	FUNCTION_DEF(get_object_or_die, 2, 2, "get_object_or_die(Level, string label) -> custom_obj: returns the object that is present in the given Level that has the given label")

		Level* lvl = EVAL_ARG(0).try_convert<Level>();
		ASSERT_LOG(lvl, "Invalid Level argument to get_object_or_die");
		const std::string& label = EVAL_ARG(1).as_string();
		variant result(lvl->get_entity_by_label(label).get());

		ASSERT_LOG(result.is_null() == false, "Object " << label << " not found in Level");

		return result;

	FUNCTION_ARGS_DEF
		ARG_TYPE("builtin level")
		ARG_TYPE("string")
	RETURN_TYPE("custom_obj")
	END_FUNCTION_DEF(get_object_or_die)

	//a command which moves an object in a given direction enough to resolve
	//any solid conflicts.
	class resolve_solid_command : public EntityCommandCallable {
		EntityPtr e_;
		int xdir_, ydir_, max_cycles_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&e_);
		}
	public:
		resolve_solid_command(EntityPtr e, int xdir, int ydir, int max_cycles) : e_(e), xdir_(xdir), ydir_(ydir), max_cycles_(max_cycles)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			if(xdir_ == 0 && ydir_ == 0 && max_cycles_ == 0) {
				if(!place_entity_in_level_with_large_displacement(lvl, ob)) {
					CustomObject* custom_obj = dynamic_cast<CustomObject*>(&ob);
					if(custom_obj) {
						//we really couldn't place it despite our best efforts ...
						//killing it is our best choice.
						custom_obj->die();
					}
					return;
				}
			}

			const int start_x = e_->x();
			const int start_y = e_->y();

			int max_cycles = max_cycles_;
			while(entity_collides(lvl, *e_, MOVE_DIRECTION::NONE) && max_cycles > 0) {
				e_->setPos(e_->x() + xdir_, e_->y() + ydir_);
				--max_cycles;
			}

			if(max_cycles == 0) {
				e_->setPos(start_x, start_y);
			}
		}
	};

	FUNCTION_DEF(resolve_solid, 1, 4, "resolve_solid(object, int xdir, int ydir, int max_cycles=100): will attempt to move the given object in the direction indicated by xdir/ydir until the object no longer has a solid overlap. Gives up after max_cycles. If called with no arguments other than the object, will try desperately to place the object in the Level.")
		EntityPtr e(EVAL_ARG(0).try_convert<Entity>());
		if(NUM_ARGS == 1) {
			resolve_solid_command* cmd = (new resolve_solid_command(e, 0, 0, 0));
			cmd->setExpression(this);
			return variant(cmd);
		} else if(NUM_ARGS == 2) {
			LOG_ERROR("TWO ARGMENTS ISN'T A SUPPORTED OPTION FOR resolve_solid() CONTINUING AS IF ONE ARGUMENT SUPPLIED");
			resolve_solid_command* cmd = (new resolve_solid_command(e, 0, 0, 0));
			cmd->setExpression(this);
			return variant(cmd);
		}

		const int xdir = EVAL_ARG(1).as_int();
		const int ydir = EVAL_ARG(2).as_int();
		const int max_cycles = NUM_ARGS > 3 ? EVAL_ARG(3).as_int() : 100;
		if(e) {
			resolve_solid_command* cmd = (new resolve_solid_command(e, xdir, ydir, max_cycles));
			cmd->setExpression(this);
			return variant(cmd);
		} else {
			return variant();
		}
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(resolve_solid)

	class add_object_command : public EntityCommandCallable {
		EntityPtr e_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&e_);
		}
	public:
		explicit add_object_command(EntityPtr e) : e_(e)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			e_->setSpawnedBy(ob.label());
			if(place_entity_in_level_with_large_displacement(lvl, *e_)) {
				lvl.add_character(e_);
			} else {
				CollisionInfo collide_info;
				entity_collides(Level::current(), *e_, MOVE_DIRECTION::NONE, &collide_info);
				game_logic::MapFormulaCallable* callable(new game_logic::MapFormulaCallable(this));
				callable->add("collide_with", variant(collide_info.collide_with.get()));

				game_logic::FormulaCallablePtr callable_ptr(callable);
				e_->handleEvent(OBJECT_EVENT_ADD_OBJECT_FAIL, callable);

				if(!e_->destroyed()) {
					callable->add("object", variant(e_.get()));
					ob.handleEvent(OBJECT_EVENT_ADD_OBJECT_FAIL, callable);
				}
				return;
			}

			e_->createObject();
		}
	};

	FUNCTION_DEF(add_object, 1, 1, "add_object(object): inserts the given object into the Level. The object should not currently be present in the Level. The position of the object is tweaked to make sure there are no solid overlaps, however if it is not possible to reasonably place the object without a solid overlap, then the object will not be placed and the object and caller will both receive the event add_object_fail.")

		EntityPtr e(EVAL_ARG(0).try_convert<Entity>());

		if(e) {
			add_object_command* cmd = (new add_object_command(e));
			cmd->setExpression(this);
			return variant(cmd);
		} else {
			LOG_ERROR("NOT AN OBJECT!");
			return variant();
		}
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(add_object)

	class remove_object_command : public EntityCommandCallable {
		variant e_;

		void remove(Level& lvl, variant v) const {
			if(v.is_list()) {
				for(const variant& item : v.as_list()) {
					remove(lvl, item);
				}
			} else {
				EntityPtr e = v.try_convert<Entity>();
				if(e) {
					lvl.remove_character(e);
				}
			}
		}
	public:
		explicit remove_object_command(variant e) : e_(e)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			remove(lvl, e_);
		}

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&e_);
		}
	};

	FUNCTION_DEF(remove_object, 1, 1, "remove_object(object): removes the given object from the Level. If there are no references to the object stored, then the object will immediately be destroyed. However it is possible to keep a reference to the object and even insert it back into the Level later using add_object()")

		variant arg = EVAL_ARG(0);
		if(arg.is_null() == false) {
			remove_object_command* cmd = (new remove_object_command(arg));
			cmd->setExpression(this);
			return variant(cmd);
		} else {
			return variant();
		}

		return variant();
	FUNCTION_ARGS_DEF
		ARG_TYPE("null|object|[object]")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(remove_object)

	class teleport_command : public EntityCommandCallable
	{
	public:
		teleport_command(const std::string& level, const std::string& label, const std::string& transition, const EntityPtr& new_playable, bool no_move_to_standing, LevelPtr level_obj)
			: level_(level), label_(label), transition_(transition), new_playable_(new_playable), no_move_to_standing_(no_move_to_standing), level_obj_(level_obj)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			Level::portal p;
			p.level_dest = level_;
			p.level_dest_obj = level_obj_;
			p.dest_starting_pos = true;
			p.dest_label = label_;
			p.automatic = true;
			p.transition = transition_;
			p.new_playable = new_playable_;
			p.no_move_to_standing = no_move_to_standing_;
			lvl.force_enter_portal(p);
		}
	private:
		bool no_move_to_standing_;
		std::string level_, label_, transition_;
		EntityPtr new_playable_;
		LevelPtr level_obj_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&new_playable_);
		}
	};

	FUNCTION_DEF(suspend_level, 1, 1, "suspend_level(string dest_level)")
		std::string dst = EVAL_ARG(0).as_string();

		return variant(new FnCommandCallable("suspend_level", [=]() {
			ffl::IntrusivePtr<Level> old_level(&Level::current());
			std::string dst_str = dst;

			const controls::control_backup_scope ctrl_backup_scope(controls::CLEAR_LOCKS);

			ffl::IntrusivePtr<Level> pause_level(load_level(dst));
			pause_level->set_suspended_level(old_level);

			std::string return_id = Level::current().id();
			LevelRunner runner(pause_level, dst_str, return_id);
			const bool result = runner.play_level();

			old_level->setAsCurrentLevel();

			if(result) {
				LevelRunner::getCurrent()->force_return(true);
			}
		}));
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(suspend_level)

	FUNCTION_DEF(resume_level, 0, 0, "resume_level()")
		return variant(new FnCommandCallable("resume_level", [=]() {
			LevelRunner::getCurrent()->force_return();
		}));
	FUNCTION_ARGS_DEF
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(resume_level)

	FUNCTION_DEF(teleport, 1, 5, "teleport(string dest_level, (optional)string dest_label, (optional)string transition, (optional)playable): teleports the player to a new Level. The Level is given by dest_level, with null() for the current Level. If dest_label is given then the player will be teleported to the object in the destination Level with that label. If transition is given, it names a type of transition (such as 'flip' or 'fade') which indicates the kind of visual effect to use for the transition. If a playable is specified it is placed in the Level instead of the current one.  If no_move_to_standing is set to true, rather than auto-positioning the player on the ground under/above the target, the player will appear at precisely the position of the destination object - e.g. this is useful if they need to fall out of a pipe or hole coming out of the ceiling.")
		std::string label, transition;
		EntityPtr new_playable;
		bool no_move_to_standing = false;
		std::string dst_level_str;
		variant play;

		LevelPtr level_obj;

		if(!(NUM_ARGS == 1 && EVAL_ARG(0).is_map())) {
			variant dst_level = EVAL_ARG(0);
			dst_level_str = dst_level.is_null() ? "" : dst_level.as_string();
			if(NUM_ARGS > 1) {
				label = EVAL_ARG(1).as_string();
				if(NUM_ARGS > 2) {
					transition = EVAL_ARG(2).as_string();
					if(NUM_ARGS > 3) {
						play = EVAL_ARG(3);
						if(NUM_ARGS > 4) {
							no_move_to_standing = EVAL_ARG(4).as_bool();
						}
					}
				}
			}
		} else {
			variant argMap = EVAL_ARG(0);
			if(argMap.has_key("level_obj")) {
				level_obj = argMap["level_obj"].convert_to<Level>();
			}

			dst_level_str = argMap["level"].as_string_default("");
			label = argMap["label"].as_string_default("");
			if(argMap.has_key("player")) {
				play = argMap["player"];
			}
			transition = argMap["transition"].as_string_default("iris");
			no_move_to_standing = !argMap["stand"].as_bool(true);
		}

		if(play.is_null() == false) {
			if(play.is_string()) {
				new_playable = EntityPtr(new PlayableCustomObject(CustomObject(play.as_string(), 0, 0, 0)));
			} else {
				new_playable = play.try_convert<Entity>();
			}
		}

		teleport_command* cmd = new teleport_command(dst_level_str, label, transition, new_playable, no_move_to_standing, level_obj);
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("map|string|null")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(teleport)

	class schedule_command : public EntityCommandCallable {
	public:
		schedule_command(int cycles, variant cmd) : cycles_(cycles), cmd_(cmd)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			ob.addScheduledCommand(cycles_, cmd_);
		}

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderVariant(&cmd_);
		}
	private:
		int cycles_;
		variant cmd_;
	};

	FUNCTION_DEF(schedule, 2, 2, "schedule(int cycles_in_future, list of commands): schedules the given list of commands to be run on the current object the given number of cycles in the future. Note that the object must be valid (not destroyed) and still present in the Level for the commands to be run.")
		const int ncycles = EVAL_ARG(0).as_int();
		variant commands = EVAL_ARG(1);
		if(ncycles <= 0) {
			return commands;
		}

		schedule_command* cmd = (new schedule_command(ncycles, commands));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("commands")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(schedule)

	class sleep_command : public EntityCommandCallable {
	public:
		sleep_command(int ncycles) : ncycles_(ncycles)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			if(ncycles_ <= 0) {
				return;
			}

			variant cmd = deferCurrentCommandSequence();
			if(cmd.is_null() == false) {
				ob.addScheduledCommand(ncycles_, cmd);
			}
		}

		void surrenderReferences(GarbageCollector* collector) override {
		}
	private:
		int ncycles_;
	};

	FUNCTION_DEF(sleep, 1, 1, "sleep(nseconds)")
		const decimal nseconds = EVAL_ARG(0).as_decimal();
		const int ncycles = ((nseconds*1000) / preferences::frame_time_millis()).as_int();
		sleep_command* cmd = new sleep_command(ncycles);
		cmd->setExpression(this);
		return variant(cmd);

	FUNCTION_ARGS_DEF
		ARG_TYPE("decimal")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sleep)

	class sleep_until_command : public EntityCommandCallable {
		const ffl::IntrusivePtr<FormulaExpression> expr_;
		ConstFormulaCallablePtr variables_;
		mutable variant cmd_;
	public:
		sleep_until_command(ffl::IntrusivePtr<FormulaExpression> expr, const ConstFormulaCallablePtr& context) : expr_(expr), variables_(context)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			if(cmd_.is_null()) {
				cmd_ = deferCurrentCommandSequence();
				if(cmd_.is_null()) {
					return;
				}
			}

			if(expr_->evaluate(*variables_).as_bool()) {
				ob.executeCommand(cmd_);
			} else {
				ob.addScheduledCommand(1, variant(this));
			}
		}

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&variables_, "VARS");
			collector->surrenderVariant(&cmd_, "CMD");
		}
	};

	FUNCTION_DEF_CTOR(sleep_until, 1, 1, "sleep(expression)")
	FUNCTION_DEF_MEMBERS
		bool optimizeArgNumToVM(int narg) const override {
			return false;
		}
	FUNCTION_DEF_IMPL

		const bool value = EVAL_ARG(0).as_bool();
		if(value) {
			return variant();
		}

		const ConstFormulaCallablePtr vars(&variables);

		return variant(new sleep_until_command(args()[0], vars));


	FUNCTION_ARGS_DEF
		ARG_TYPE("bool")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sleep_until)

	class sleep_until_animation_finished_command : public EntityCommandCallable {
	public:
		sleep_until_animation_finished_command()
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			variant cmd = deferCurrentCommandSequence();
			ob.addEndAnimCommand(cmd);
		}

		void surrenderReferences(GarbageCollector* collector) override {
		}
	};

	FUNCTION_DEF(sleep_until_animation_finished, 0, 0, "sleeps until the current animation is finished.")
		return variant(new sleep_until_animation_finished_command);
	FUNCTION_ARGS_DEF
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(sleep_until_animation_finished)

	class add_water_command : public EntityCommandCallable
	{
		rect r_;
		KRE::Color color_;
	public:
		add_water_command(const rect& r, const KRE::Color& color) : r_(r), color_(color)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.get_or_create_water().addRect(r_, color_, variant(&ob));
		}
	};

	FUNCTION_DEF(add_water, 4, 5, "add_water(int x1, int y1, int x2, int y2, (optional)[r,g,b,a]=[70,0,0,50]): adds water of the given color in the given rectangle.")
		KRE::Color color(70, 0, 0, 50);

		if(NUM_ARGS > 4) {
			color = KRE::Color(EVAL_ARG(4));
		}

		add_water_command* cmd = (new add_water_command(
		  rect::from_coordinates(
			EVAL_ARG(0).as_int(),
			EVAL_ARG(1).as_int(),
			EVAL_ARG(2).as_int(),
			EVAL_ARG(3).as_int()), color));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("[int]")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(add_water)

	class remove_water_command : public EntityCommandCallable
	{
		rect r_;
	public:
		remove_water_command(const rect& r) : r_(r)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.get_or_create_water().deleteRect(r_);
		}
	};

	FUNCTION_DEF(remove_water, 4, 4, "remove_water(int x1, int y1, int x2, int y2): removes water that has the given rectangular area.")
		remove_water_command* cmd = (new remove_water_command(
		  rect::from_coordinates(
			EVAL_ARG(0).as_int(),
			EVAL_ARG(1).as_int(),
			EVAL_ARG(2).as_int(),
			EVAL_ARG(3).as_int())));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(remove_water)

	class add_wave_command : public EntityCommandCallable
	{
		int x_, y_, xvelocity_, height_, length_, delta_height_, delta_length_;
	public:
		add_wave_command(int x, int y, int xvelocity, int height, int length, int delta_height, int delta_length) : x_(x), y_(y), xvelocity_(xvelocity), height_(height), length_(length), delta_height_(delta_height), delta_length_(delta_length)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			Water* w = lvl.get_water();
			if(!w) {
				return;
			}

			point p(x_, y_);
			w->addWave(p, xvelocity_/1000.0, height_/1000.0, length_/1000.0, delta_height_/1000.0, delta_length_/1000.0);
		}
	};

	FUNCTION_DEF(add_wave, 7, 7, "addWave(int x, int y, int xvelocity, int height, int length, int delta_height, int delta_length): will add a wave with the given characteristics at the surface of the water above the (x,y) point. (x,y) must be within a body of water. Waves are a visual effect only and may not display at all on slower devices.")
		add_wave_command* cmd = (new add_wave_command(
			EVAL_ARG(0).as_int(),
			EVAL_ARG(1).as_int(),
			EVAL_ARG(2).as_int(),
			EVAL_ARG(3).as_int(),
			EVAL_ARG(4).as_int(),
			EVAL_ARG(5).as_int(),
			EVAL_ARG(6).as_int()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(add_wave)

	FUNCTION_DEF(rect_current, 7, 7, "rect_current(int x, int y, int w, int h, int xvelocity, int yvelocity, int strength) -> current generator object: creates a current generator object that has a current with the given parameters. Set the return value of this function to an object's rect_current to attach it to an object and thus place it in the Level.")
		return variant(new RectCurrentGenerator(rect(
						  EVAL_ARG(0).as_int(),
						  EVAL_ARG(1).as_int(),
						  EVAL_ARG(2).as_int(),
						  EVAL_ARG(3).as_int()),
						  EVAL_ARG(4).as_int(),
						  EVAL_ARG(5).as_int(),
						  EVAL_ARG(6).as_int()));
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("object")
	END_FUNCTION_DEF(rect_current)

	FUNCTION_DEF(circle_light, 2, 2, "circle_light(object, radius): creates a circle of light with the given radius")
		return variant(new CircleLight(
				  *EVAL_ARG(0).convert_to<CustomObject>(),
				  EVAL_ARG(1).as_int()));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("int")
	RETURN_TYPE("object")
	END_FUNCTION_DEF(circle_light)

	class add_particles_command : public CustomObjectCommandCallable {
	public:
		add_particles_command(const std::string& id, const std::string& type)
		  : id_(id), type_(type) {
		}

		virtual void execute(Level& lvl, CustomObject& ob) const override {
			ob.addParticleSystem(id_, type_);
		}
	private:
		std::string id_, type_;
	};

	FUNCTION_DEF(add_particles, 1, 2, "add_particles(string id): adds the particle system with the given id to the object")
		add_particles_command* cmd = new add_particles_command(
			EVAL_ARG(0).as_string(),
			EVAL_ARG(NUM_ARGS < 2 ? 0 : 1).as_string());
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(add_particles)

	FUNCTION_DEF(collides, 4, 4, "collides(object a, string area_a, object b, string area_b) -> boolean: returns true iff area_a within object a collides with area_b within object b.")
		return variant::from_bool(entity_user_collision_specific_areas(
				   *EVAL_ARG(0).convert_to<Entity>(),
				   EVAL_ARG(1).as_string(),
				   *EVAL_ARG(2).convert_to<Entity>(),
				   EVAL_ARG(3).as_string()));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("string")
		ARG_TYPE("object")
		ARG_TYPE("string")
	RETURN_TYPE("bool")
	END_FUNCTION_DEF(collides)

	FUNCTION_DEF(collides_with_level, 1, 1, "collides_with_level(object) -> boolean: returns true iff the given object collides with the Level.")
		return variant::from_bool(non_solid_entity_collides_with_level(
				   Level::current(),
				   *EVAL_ARG(0).convert_to<Entity>()));
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
	RETURN_TYPE("bool")
	END_FUNCTION_DEF(collides_with_level)

	FUNCTION_DEF(blur_object, 2, 2, "blur_object(properties, params)")
		Formula::failIfStaticContext();
		std::map<variant,variant> props = EVAL_ARG(0).as_map();
		std::map<std::string,variant> properties, end_properties;
		for(auto p : props) {
			properties[p.first.as_string()] = p.second;
		}

		std::map<variant,variant> options = EVAL_ARG(1).as_map();

		int duration = options[variant("duration")].as_int();
		variant easing = options[variant("easing")];
		variant anim = options[variant("animate")];
		if(anim.is_map()) {
			for(auto p : anim.as_map()) {
				end_properties[p.first.as_string()] = p.second;
			}
		}

		return variant(new BlurObject(properties, end_properties, duration, easing));

	FUNCTION_ARGS_DEF
		ARG_TYPE("{string -> any}")
		ARG_TYPE("{duration: int, easing: null|function(decimal)->decimal, animate: null|{string -> any} }")
	RETURN_TYPE("builtin BlurObject")
	END_FUNCTION_DEF(blur_object)

	class text_command : public CustomObjectCommandCallable {
	public:
		text_command(const std::string& text, const std::string& font, int size, int align)
		  : text_(text), font_(font), size_(size), align_(align) {
		}

		virtual void execute(Level& lvl, CustomObject& ob) const override {
			ob.setText(text_, font_, size_, align_);
		}
	private:
		std::string text_, font_;
		int size_;
		int align_;
	};

	FUNCTION_DEF(text, 1, 4, "text(string text, (optional)string font='default', (optional)int size=2, (optional)bool|string centered=false): adds text for the current object")
		const std::string text = EVAL_ARG(0).as_string();
		const std::string font = NUM_ARGS > 1 ? EVAL_ARG(1).as_string() : "default";
		const int size = NUM_ARGS > 2 ? EVAL_ARG(2).as_int() : 2;

		int align = -1;
		if(NUM_ARGS > 3) {
			variant align_var = EVAL_ARG(3);
			if(align_var.is_string()) {
				const std::string str = align_var.as_string();
				if(str == "left") {
					align = -1;
				} else if(str == "center") {
					align = 0;
				} else if(str == "right") {
					align = 1;
				}

			} else {
				align = align_var.as_bool() ? 0 : -1;
			}
		}

		text_command* cmd = (new text_command(text, font, size, align));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("string")
		ARG_TYPE("int")
		ARG_TYPE("bool|string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(text)

	FUNCTION_DEF(swallow_event, 0, 0, "swallow_event(): when used in an instance-specific event handler, this causes the event to be swallowed and not passed to the object's main event handler.")
		return variant(new SwallowObjectCommandCallable);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(swallow_event)

	FUNCTION_DEF(swallow_mouse_event, 0, 0, "swallow_mouse_event(): when used in an instance-specific event handler, this causes the mouse event to be swallowed and not passed to the next object in the z-order stack.")
		return variant(new FnCommandCallable("swallow_mouse_event", [=]() {
			g_mouse_event_swallowed = true;
		}));
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(swallow_mouse_event)

	class animate_command : public EntityCommandCallable {
		const ffl::IntrusivePtr<CustomObject> target_;
		variant attr_var_;
		variant options_;
		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&target_, "TARGET");
			collector->surrenderVariant(&attr_var_, "ATTR");
			collector->surrenderVariant(&options_, "OPTIONS");
		}
	public:
		animate_command(ffl::IntrusivePtr<CustomObject> target, variant attr_var, variant options) : target_(target), attr_var_(attr_var), options_(options)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			target_->addAnimatedMovement(attr_var_, options_);
		}
	};

	FUNCTION_DEF(animate, 3, 3, "animate(object, attributes, options)")
		const variant obj = EVAL_ARG(0);
		const variant attr_var = EVAL_ARG(1);
		const variant options = EVAL_ARG(2);

		auto cmd = new animate_command(obj.convert_to<CustomObject>(), attr_var, options);
		cmd->setExpression(this);
		return variant(cmd);

	FUNCTION_ARGS_DEF
		ARG_TYPE("custom_obj")
		ARG_TYPE("map")
		ARG_TYPE("{on_begin: null|commands|function()->commands, on_process: null|commands|function()->commands, on_complete: null|commands|function()->commands, name: null|string, easing: null|string|function(decimal)->decimal, duration: null|int, replace_existing: bool|null, sleep: bool|null}")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(animate)

	class set_widgets_command : public EntityCommandCallable {
		const EntityPtr target_;
		const std::vector<variant> widgets_;
		//const FormulaCallablePtr callable_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&target_);
			for(const variant& w : widgets_) {
				collector->surrenderVariant(&w);
			}
		}
	public:
		set_widgets_command(EntityPtr target, const std::vector<variant>& widgets/*, const FormulaCallablePtr callable*/)
		  : target_(target), widgets_(widgets)//, callable_(callable)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			Entity* e = target_ ? target_.get() : &ob;
			CustomObject* custom_obj = dynamic_cast<CustomObject*>(e);
			std::vector<gui::WidgetPtr> w;
			for(const variant& v : widgets_) {
				if(v.is_null()) {
					continue;
				}

				gui::DialogPtr dialog = ffl::IntrusivePtr<gui::Dialog>(v.try_convert<gui::Dialog>());
				if(dialog) {
					w.push_back(dialog);
				} else {
					w.push_back(widget_factory::create(v, custom_obj));
				}
			}
			custom_obj->addWidgets(&w);
		}
	};

	FUNCTION_DEF(set_widgets, 1, -1, "set_widgets((optional) obj, widget, ...): Adds a group of widgets to the current object, or the specified object")
		EntityPtr target = EVAL_ARG(0).try_convert<Entity>();
		int arg_start = (target == nullptr) ? 0 : 1;
		std::vector<variant> widgetsv;
		for(unsigned i = arg_start; i < NUM_ARGS; i++) {
			variant items = EVAL_ARG(i);
			if(items.is_list()) {
				for(unsigned n = 0; n != items.num_elements(); ++n) {
					widgetsv.push_back(items[n]);
				}
			} else {
				widgetsv.push_back(items);
			}
		}
		//ConstFormulaCallablePtr callable = map_into_callable(EVAL_ARG(2));
		set_widgets_command* cmd = (new set_widgets_command(target, widgetsv));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(set_widgets)

	class clear_widgets_command : public EntityCommandCallable {
		const EntityPtr target_;

		void surrenderReferences(GarbageCollector* collector) override {
			collector->surrenderPtr(&target_);
		}
	public:
		clear_widgets_command(EntityPtr target) : target_(target)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			Entity* e = target_ ? target_.get() : &ob;
			CustomObject* custom_obj = dynamic_cast<CustomObject*>(e);
			custom_obj->clearWidgets();
		}
	};

	FUNCTION_DEF(clear_widgets, 1, 1, "clear_widgets(obj): Clears all widgets from the object.")
		EntityPtr target = EVAL_ARG(0).try_convert<Entity>();
		clear_widgets_command* cmd = (new clear_widgets_command(target));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(clear_widgets)

	FUNCTION_DEF(get_widget, 2, 2, "get_widget(object obj, string id): returns the widget with the matching id for given object")
		ffl::IntrusivePtr<CustomObject> target = EVAL_ARG(0).try_convert<CustomObject>();
		std::string id = EVAL_ARG(1).as_string();
		return variant(target->getWidgetById(id).get());
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("string")
	RETURN_TYPE("object|null")
	END_FUNCTION_DEF(get_widget)

	FUNCTION_DEF(widget, 2, 2, "widget(callable, map w): Constructs a widget defined by w and returns it for later use")
		Formula::failIfStaticContext();
		game_logic::FormulaCallablePtr callable = map_into_callable(EVAL_ARG(0));
		gui::WidgetPtr w = widget_factory::create(EVAL_ARG(1), callable.get());
		return variant(w.get());
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("map")
	FUNCTION_TYPE_DEF
		variant v;
		if(args()[1]->canReduceToVariant(v) && v.is_map()) {
			return parse_variant_type(variant(widget_factory::convert_type_to_variant_type_name(v["type"].as_string())));
		}
		return parse_variant_type(variant("builtin widget"));
	END_FUNCTION_DEF(widget)

	class add_level_module_command : public EntityCommandCallable {
		std::string lvl_;
		int x_, y_;
	public:
		add_level_module_command(const std::string& lvl, int x, int y)
		  : lvl_(lvl), x_(x), y_(y)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.add_sub_level(lvl_, x_, y_);
		}
	};

	FUNCTION_DEF(add_level_module, 3, 3, "add_level_module(string lvl, int xoffset, int yoffset): adds the Level module with the given Level id at the given offset")
		add_level_module_command* cmd = (new add_level_module_command(EVAL_ARG(0).string_cast(), EVAL_ARG(1).as_int(), EVAL_ARG(2).as_int()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(add_level_module)

	class remove_level_module_command : public EntityCommandCallable {
		std::string lvl_;
	public:
		explicit remove_level_module_command(const std::string& lvl) : lvl_(lvl)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.remove_sub_level(lvl_);
		}
	};

	FUNCTION_DEF(remove_level_module, 1, 1, "remove_level_module(string lvl): removes the given Level module")
		remove_level_module_command* cmd = (new remove_level_module_command(EVAL_ARG(0).string_cast()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(remove_level_module)

	class shift_level_position_command : public EntityCommandCallable {
		int xoffset_, yoffset_;
	public:

		shift_level_position_command(int xoffset, int yoffset)
		  : xoffset_(xoffset), yoffset_(yoffset) {}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.adjust_level_offset(xoffset_, yoffset_);
		}
	};

	FUNCTION_DEF(cosmic_shift, 2, 2, "cosmic_shift(int xoffset, int yoffet): adjust position of all objects and tiles in the Level by the given offset")
		shift_level_position_command* cmd = (new shift_level_position_command(EVAL_ARG(0).as_int(), EVAL_ARG(1).as_int()));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("int")
		ARG_TYPE("int")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(cosmic_shift)

	FUNCTION_DEF(create_animation, 1, 1, "create_animation(map): creates an animation object from the given data")
		return variant(new Frame(EVAL_ARG(0)));
	FUNCTION_ARGS_DEF
		ARG_TYPE("map")
	RETURN_TYPE("builtin frame")
	END_FUNCTION_DEF(create_animation)

	#if !defined(NO_MODULES)

	class module_pump_command : public EntityCommandCallable {
		module::client* client_;
	public:
		explicit module_pump_command(module::client* cl) : client_(cl)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			client_->process();
		}
	};

	class module_install_command : public EntityCommandCallable {
		module::client* client_;
		std::string id_;
	public:
		module_install_command(module::client* cl, const std::string& id) : client_(cl), id_(id)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			client_->install_module(id_);
		}
	};

	class module_uninstall_command : public EntityCommandCallable {
		std::string id_;
	public:
		explicit module_uninstall_command(const std::string& id) : id_(id)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			module::uninstall_downloaded_module(id_);
		}
	};

	FUNCTION_DEF(module_client, 0, 0, "module_client(): creates a module client object. The object will immediately start retrieving basic module info from the server. module_pump() should be called on it every frame. Has the following fields:\n  is_complete: true iff the current operation is complete and a new operation can be started. When the module_client is first created it automatically starts an operation to get the summary of modules.\n  downloaded_modules: a list of downloaded modules that are currently installed.\n  module_info: info about the modules available on the server.\n  error: contains an error string if the operation resulted in an error, null otherwise.\n  kbytes_transferred: number of kbytes transferred in the current operation\n  kbytes_total: total number of kbytes to transfer to complete the operation.")
		return variant(new module::client);
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(module_client)

	FUNCTION_DEF(module_pump, 1, 1, "module_pump(module_client): pumps module client events. Should be called every cycle.")
		module::client* cl = EVAL_ARG(0).try_convert<module::client>();
		ASSERT_LOG(cl, "BAD ARGUMENT GIVEN TO module_pump");
		module_pump_command* cmd = (new module_pump_command(cl));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(module_pump)

	FUNCTION_DEF(module_install, 2, 2, "module_install(module_client, string module_id): begins downloading the given module and installing it. This should only be called when module_client.is_complete = true (i.e. there is no operation currently underway)")
		module::client* cl = EVAL_ARG(0).try_convert<module::client>();
		ASSERT_LOG(cl, "BAD ARGUMENT GIVEN TO module_pump");
		const std::string module_id = EVAL_ARG(1).as_string();
		module_install_command* cmd = (new module_install_command(cl, module_id));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(module_install)

	FUNCTION_DEF(module_uninstall, 1, 1, "module_uninstall(string module_id): uninstalls the given module")
		const std::string module_id = EVAL_ARG(0).as_string();
		module_uninstall_command* cmd = (new module_uninstall_command(module_id));
		cmd->setExpression(this);
		return variant(cmd);

	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
	RETURN_TYPE("commands")

	END_FUNCTION_DEF(module_uninstall)

	class module_rate_command : public EntityCommandCallable {
		module::client* client_;
		std::string id_;
		int rating_;
		std::string review_;
	public:
		module_rate_command(module::client* cl, const std::string& id, int rating, const std::string& review) : client_(cl), id_(id), rating_(rating), review_(review)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			client_->rate_module(id_, rating_, review_);
		}
	};

	FUNCTION_DEF(module_rate, 3, 4, "module_rate(module_client, string module_id, int num_stars (1-5), (optional) string review): begins a request to rate the given module with the given number of stars, optionally with a review.")
		module::client* cl = EVAL_ARG(0).try_convert<module::client>();
		ASSERT_LOG(cl, "BAD ARGUMENT GIVEN TO module_pump");
		const std::string module_id = EVAL_ARG(1).as_string();

		const int num_stars = EVAL_ARG(2).as_int();
		ASSERT_LOG(num_stars >= 1 && num_stars <= 5, "INVALID RATING: " << num_stars);
		std::string review;
		if(NUM_ARGS > 3) {
			review = EVAL_ARG(3).as_string();
		}

		module_rate_command* cmd = (new module_rate_command(cl, module_id, num_stars, review));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("object")
		ARG_TYPE("string")
		ARG_TYPE("int")
		ARG_TYPE("string")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(module_rate)

	class module_launch_command : public EntityCommandCallable {
		std::string id_;
		ConstFormulaCallablePtr callable_;
	public:
		explicit module_launch_command(const std::string& id, ConstFormulaCallablePtr callable)
			: id_(id), callable_(callable)
		{}

		virtual void execute(Level& lvl, Entity& ob) const override {
			lvl.launch_new_module(id_, callable_);
		}
	};

	FUNCTION_DEF(module_launch, 1, 2, "module_launch(string module_id, (optional) callable): launch the game using the given module.")
		const std::string module_id = EVAL_ARG(0).as_string();
		ConstFormulaCallablePtr callable;
		if(NUM_ARGS > 1) {
			callable = map_into_callable(EVAL_ARG(1));
		}
		module_launch_command* cmd = (new module_launch_command(module_id, callable));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("object")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(module_launch)

	FUNCTION_DEF(eval, 1, 2, "eval(str, [arg map]): evaluate the given string as FFL")
		variant s = EVAL_ARG(0);
		try {
			ConstFormulaCallablePtr callable(&variables);

			if(NUM_ARGS > 1) {
				variant v = EVAL_ARG(1);
				callable = map_into_callable(v);
			}

			const assert_recover_scope recovery_scope;
			ConstFormulaPtr f(Formula::createOptionalFormula(s, &get_custom_object_functions_symbol_table()));
			if(!f) {
				return variant();
			}

			return f->execute(*callable);
		} catch(const type_error&) {
		} catch(const validation_failure_exception&) {
		}
		LOG_ERROR("ERROR IN EVAL");
		return variant();
	FUNCTION_ARGS_DEF
		ARG_TYPE("string")
		ARG_TYPE("map")
	RETURN_TYPE("any")
	END_FUNCTION_DEF(eval)

	#endif // NO_MODULES

	#if 0 // XXX needs re-work
	class spawn_voxel_object_command : public EntityCommandCallable
	{
	public:
		spawn_voxel_object_command(voxel::UserVoxelObjectPtr obj, variant instantiation_commands)
		  : obj_(obj), instantiation_commands_(instantiation_commands)
		{}
		virtual void execute(Level& lvl, Entity& ob) const {

			ASSERT_LOG(lvl.iso_world() != nullptr, "No 'isoworld' to insert voxel_object into.");
			lvl.iso_world()->addObject(obj_);

			obj_->executeCommand(instantiation_commands_);

			/*
			obj_->setLevel(lvl);
			obj_->setSpawnedBy(ob.label());

			if(!place_Entity_in_level_with_large_displacement(lvl, *obj_)) {
				return;
			}

			lvl.add_character(obj_);

			//send an event to the parent to let them know they've spawned a child,
			//and let them record the child's details.
			game_logic::MapFormulaCallable* spawn_callable(new game_logic::MapFormulaCallable);
			variant holder(spawn_callable);
			spawn_callable->add("spawner", variant(&ob));
			spawn_callable->add("child", variant(obj_.get()));
			ob.handleEvent("child_spawned", spawn_callable);
			obj_->handleEvent("spawned", spawn_callable);

			obj_->executeCommand(instantiation_commands_);

			if(Entity_collides(lvl, *obj_, MOVE_DIRECTION::NONE)) {
				lvl.remove_character(obj_);
			} else {
				obj_->checkInitialized();
			}

			obj_->createObject();
			*/
		}
	private:
		voxel::UserVoxelObjectPtr obj_;
		variant instantiation_commands_;
	};

	FUNCTION_DEF(spawn_voxel_object, 1, 2, "spawn_voxel(properties, (optional) list of commands cmd): will create a new object of type given by type_id with the given midpoint and facing. Immediately after creation the object will have any commands given by cmd executed on it. The child object will have the spawned event sent to it, and the parent object will have the child_spawned event sent to it.")

		Formula::failIfStaticContext();

		variant doc = EVAL_ARG(0);

		voxel::UserVoxelObjectPtr obj(new voxel::user_voxel_object(doc));

		variant commands;
		if(NUM_ARGS > 1) {
			commands = EVAL_ARG(1);
		}
		spawn_voxel_object_command* cmd = (new spawn_voxel_object_command(obj, commands));
		cmd->setExpression(this);
		return variant(cmd);
	FUNCTION_ARGS_DEF
		ARG_TYPE("map")
		ARG_TYPE("commands")
	RETURN_TYPE("commands")
	END_FUNCTION_DEF(spawn_voxel_object)
	#endif


	class CustomObjectFunctionSymbolTable : public FunctionSymbolTable
	{
	public:
		virtual ExpressionPtr createFunction(
								   const std::string& fn,
								   const std::vector<ExpressionPtr>& args,
								   ConstFormulaCallableDefinitionPtr callable_def) const override;
	};

	ExpressionPtr CustomObjectFunctionSymbolTable::createFunction(
							   const std::string& fn,
							   const std::vector<ExpressionPtr>& args,
							   ConstFormulaCallableDefinitionPtr callable_def) const
	{
		const std::map<std::string, FunctionCreator*>& creators = get_function_creators(FunctionModule);
		std::map<std::string, FunctionCreator*>::const_iterator i = creators.find(fn);
		if(i != creators.end()) {
			return ExpressionPtr(i->second->create(args));
		}

		return FunctionSymbolTable::createFunction(fn, args, callable_def);
	}
} //namespace

bool in_speech_dialog ()
{
	return g_in_speech_dialog > 0;
}

FunctionSymbolTable& get_custom_object_functions_symbol_table()
{
	static CustomObjectFunctionSymbolTable table;
	return table;
}
