// Written in the D programming language

/**
 * Contains listing of named entities for the D lexer.
 *
 * Copyright: Brian Schott 2013
 * License: <a href="http://www.boost.org/LICENSE_1_0.txt">Boost License 1.0</a>.
 * Authors: Brian Schott
 */

module dparse.entities;

///
struct HtmlEntity
{
    ///
    string name;
    ///
    string value;
}

// the deprecated hexstrings replaced with this simple thing
private string x(string a) {
	string r = "\"";
	bool odd = true;
	foreach(ch; a) {
		if(odd) r ~= "\\x";
		r ~= ch;
		odd = !odd;
	}
	r ~= "\"";
	return r;
}

/**
 * Generated from $(LINK http://www.w3.org/TR/html5/entities.json)
 * Standards: $(LINK http://www.w3.org/TR/html5/syntax.html#named-character-references)
 */
immutable HtmlEntity[] characterEntities = [
    HtmlEntity("AElig", "\u00C6"),
    HtmlEntity("AMP", "\u0026"),
    HtmlEntity("Aacute", "\u00C1"),
    HtmlEntity("Abreve", "\u0102"),
    HtmlEntity("Acirc", "\u00C2"),
    HtmlEntity("Acy", "\u0410"),
    HtmlEntity("Afr", x("D835DD04")),
    HtmlEntity("Agrave", "\u00C0"),
    HtmlEntity("Alpha", "\u0391"),
    HtmlEntity("Amacr", "\u0100"),
    HtmlEntity("And", "\u2A53"),
    HtmlEntity("Aogon", "\u0104"),
    HtmlEntity("Aopf", x("D835DD38")),
    HtmlEntity("ApplyFunction", "\u2061"),
    HtmlEntity("Aring", "\u00C5"),
    HtmlEntity("Ascr", x("D835DC9C")),
    HtmlEntity("Assign", "\u2254"),
    HtmlEntity("Atilde", "\u00C3"),
    HtmlEntity("Auml", "\u00C4"),
    HtmlEntity("Backslash", "\u2216"),
    HtmlEntity("Barv", "\u2AE7"),
    HtmlEntity("Barwed", "\u2306"),
    HtmlEntity("Bcy", "\u0411"),
    HtmlEntity("Because", "\u2235"),
    HtmlEntity("Bernoullis", "\u212C"),
    HtmlEntity("Beta", "\u0392"),
    HtmlEntity("Bfr", x("D835DD05")),
    HtmlEntity("Bopf", x("D835DD39")),
    HtmlEntity("Breve", "\u02D8"),
    HtmlEntity("Bscr", "\u212C"),
    HtmlEntity("Bumpeq", "\u224E"),
    HtmlEntity("CHcy", "\u0427"),
    HtmlEntity("COPY", "\u00A9"),
    HtmlEntity("Cacute", "\u0106"),
    HtmlEntity("Cap", "\u22D2"),
    HtmlEntity("CapitalDifferentialD", "\u2145"),
    HtmlEntity("Cayleys", "\u212D"),
    HtmlEntity("Ccaron", "\u010C"),
    HtmlEntity("Ccedil", "\u00C7"),
    HtmlEntity("Ccirc", "\u0108"),
    HtmlEntity("Cconint", "\u2230"),
    HtmlEntity("Cdot", "\u010A"),
    HtmlEntity("Cedilla", "\u00B8"),
    HtmlEntity("CenterDot", "\u00B7"),
    HtmlEntity("Cfr", "\u212D"),
    HtmlEntity("Chi", "\u03A7"),
    HtmlEntity("CircleDot", "\u2299"),
    HtmlEntity("CircleMinus", "\u2296"),
    HtmlEntity("CirclePlus", "\u2295"),
    HtmlEntity("CircleTimes", "\u2297"),
    HtmlEntity("ClockwiseContourIntegral", "\u2232"),
    HtmlEntity("CloseCurlyDoubleQuote", "\u201D"),
    HtmlEntity("CloseCurlyQuote", "\u2019"),
    HtmlEntity("Colon", "\u2237"),
    HtmlEntity("Colone", "\u2A74"),
    HtmlEntity("Congruent", "\u2261"),
    HtmlEntity("Conint", "\u222F"),
    HtmlEntity("ContourIntegral", "\u222E"),
    HtmlEntity("Copf", "\u2102"),
    HtmlEntity("Coproduct", "\u2210"),
    HtmlEntity("CounterClockwiseContourIntegral", "\u2233"),
    HtmlEntity("Cross", "\u2A2F"),
    HtmlEntity("Cscr", x("D835DC9E")),
    HtmlEntity("Cup", "\u22D3"),
    HtmlEntity("CupCap", "\u224D"),
    HtmlEntity("DD", "\u2145"),
    HtmlEntity("DDotrahd", "\u2911"),
    HtmlEntity("DJcy", "\u0402"),
    HtmlEntity("DScy", "\u0405"),
    HtmlEntity("DZcy", "\u040F"),
    HtmlEntity("Dagger", "\u2021"),
    HtmlEntity("Darr", "\u21A1"),
    HtmlEntity("Dashv", "\u2AE4"),
    HtmlEntity("Dcaron", "\u010E"),
    HtmlEntity("Dcy", "\u0414"),
    HtmlEntity("Del", "\u2207"),
    HtmlEntity("Delta", "\u0394"),
    HtmlEntity("Dfr", x("D835DD07")),
    HtmlEntity("DiacriticalAcute", "\u00B4"),
    HtmlEntity("DiacriticalDot", "\u02D9"),
    HtmlEntity("DiacriticalDoubleAcute", "\u02DD"),
    HtmlEntity("DiacriticalGrave", "\u0060"),
    HtmlEntity("DiacriticalTilde", "\u02DC"),
    HtmlEntity("Diamond", "\u22C4"),
    HtmlEntity("DifferentialD", "\u2146"),
    HtmlEntity("Dopf", x("D835DD3B")),
    HtmlEntity("Dot", "\u00A8"),
    HtmlEntity("DotDot", "\u20DC"),
    HtmlEntity("DotEqual", "\u2250"),
    HtmlEntity("DoubleContourIntegral", "\u222F"),
    HtmlEntity("DoubleDot", "\u00A8"),
    HtmlEntity("DoubleDownArrow", "\u21D3"),
    HtmlEntity("DoubleLeftArrow", "\u21D0"),
    HtmlEntity("DoubleLeftRightArrow", "\u21D4"),
    HtmlEntity("DoubleLeftTee", "\u2AE4"),
    HtmlEntity("DoubleLongLeftArrow", "\u27F8"),
    HtmlEntity("DoubleLongLeftRightArrow", "\u27FA"),
    HtmlEntity("DoubleLongRightArrow", "\u27F9"),
    HtmlEntity("DoubleRightArrow", "\u21D2"),
    HtmlEntity("DoubleRightTee", "\u22A8"),
    HtmlEntity("DoubleUpArrow", "\u21D1"),
    HtmlEntity("DoubleUpDownArrow", "\u21D5"),
    HtmlEntity("DoubleVerticalBar", "\u2225"),
    HtmlEntity("DownArrow", "\u2193"),
    HtmlEntity("DownArrowBar", "\u2913"),
    HtmlEntity("DownArrowUpArrow", "\u21F5"),
    HtmlEntity("DownBreve", "\u0311"),
    HtmlEntity("DownLeftRightVector", "\u2950"),
    HtmlEntity("DownLeftTeeVector", "\u295E"),
    HtmlEntity("DownLeftVector", "\u21BD"),
    HtmlEntity("DownLeftVectorBar", "\u2956"),
    HtmlEntity("DownRightTeeVector", "\u295F"),
    HtmlEntity("DownRightVector", "\u21C1"),
    HtmlEntity("DownRightVectorBar", "\u2957"),
    HtmlEntity("DownTee", "\u22A4"),
    HtmlEntity("DownTeeArrow", "\u21A7"),
    HtmlEntity("Downarrow", "\u21D3"),
    HtmlEntity("Dscr", x("D835DC9F")),
    HtmlEntity("Dstrok", "\u0110"),
    HtmlEntity("ENG", "\u014A"),
    HtmlEntity("ETH", "\u00D0"),
    HtmlEntity("Eacute", "\u00C9"),
    HtmlEntity("Ecaron", "\u011A"),
    HtmlEntity("Ecirc", "\u00CA"),
    HtmlEntity("Ecy", "\u042D"),
    HtmlEntity("Edot", "\u0116"),
    HtmlEntity("Efr", x("D835DD08")),
    HtmlEntity("Egrave", "\u00C8"),
    HtmlEntity("Element", "\u2208"),
    HtmlEntity("Emacr", "\u0112"),
    HtmlEntity("EmptySmallSquare", "\u25FB"),
    HtmlEntity("EmptyVerySmallSquare", "\u25AB"),
    HtmlEntity("Eogon", "\u0118"),
    HtmlEntity("Eopf", x("D835DD3C")),
    HtmlEntity("Epsilon", "\u0395"),
    HtmlEntity("Equal", "\u2A75"),
    HtmlEntity("EqualTilde", "\u2242"),
    HtmlEntity("Equilibrium", "\u21CC"),
    HtmlEntity("Escr", "\u2130"),
    HtmlEntity("Esim", "\u2A73"),
    HtmlEntity("Eta", "\u0397"),
    HtmlEntity("Euml", "\u00CB"),
    HtmlEntity("Exists", "\u2203"),
    HtmlEntity("ExponentialE", "\u2147"),
    HtmlEntity("Fcy", "\u0424"),
    HtmlEntity("Ffr", x("D835DD09")),
    HtmlEntity("FilledSmallSquare", "\u25FC"),
    HtmlEntity("FilledVerySmallSquare", "\u25AA"),
    HtmlEntity("Fopf", x("D835DD3D")),
    HtmlEntity("ForAll", "\u2200"),
    HtmlEntity("Fouriertrf", "\u2131"),
    HtmlEntity("Fscr", "\u2131"),
    HtmlEntity("GJcy", "\u0403"),
    HtmlEntity("GT", "\u003E"),
    HtmlEntity("Gamma", "\u0393"),
    HtmlEntity("Gammad", "\u03DC"),
    HtmlEntity("Gbreve", "\u011E"),
    HtmlEntity("Gcedil", "\u0122"),
    HtmlEntity("Gcirc", "\u011C"),
    HtmlEntity("Gcy", "\u0413"),
    HtmlEntity("Gdot", "\u0120"),
    HtmlEntity("Gfr", x("D835DD0A")),
    HtmlEntity("Gg", "\u22D9"),
    HtmlEntity("Gopf", x("D835DD3E")),
    HtmlEntity("GreaterEqual", "\u2265"),
    HtmlEntity("GreaterEqualLess", "\u22DB"),
    HtmlEntity("GreaterFullEqual", "\u2267"),
    HtmlEntity("GreaterGreater", "\u2AA2"),
    HtmlEntity("GreaterLess", "\u2277"),
    HtmlEntity("GreaterSlantEqual", "\u2A7E"),
    HtmlEntity("GreaterTilde", "\u2273"),
    HtmlEntity("Gscr", x("D835DCA2")),
    HtmlEntity("Gt", "\u226B"),
    HtmlEntity("HARDcy", "\u042A"),
    HtmlEntity("Hacek", "\u02C7"),
    HtmlEntity("Hat", "\u005E"),
    HtmlEntity("Hcirc", "\u0124"),
    HtmlEntity("Hfr", "\u210C"),
    HtmlEntity("HilbertSpace", "\u210B"),
    HtmlEntity("Hopf", "\u210D"),
    HtmlEntity("HorizontalLine", "\u2500"),
    HtmlEntity("Hscr", "\u210B"),
    HtmlEntity("Hstrok", "\u0126"),
    HtmlEntity("HumpDownHump", "\u224E"),
    HtmlEntity("HumpEqual", "\u224F"),
    HtmlEntity("IEcy", "\u0415"),
    HtmlEntity("IJlig", "\u0132"),
    HtmlEntity("IOcy", "\u0401"),
    HtmlEntity("Iacute", "\u00CD"),
    HtmlEntity("Icirc", "\u00CE"),
    HtmlEntity("Icy", "\u0418"),
    HtmlEntity("Idot", "\u0130"),
    HtmlEntity("Ifr", "\u2111"),
    HtmlEntity("Igrave", "\u00CC"),
    HtmlEntity("Im", "\u2111"),
    HtmlEntity("Imacr", "\u012A"),
    HtmlEntity("ImaginaryI", "\u2148"),
    HtmlEntity("Implies", "\u21D2"),
    HtmlEntity("Int", "\u222C"),
    HtmlEntity("Integral", "\u222B"),
    HtmlEntity("Intersection", "\u22C2"),
    HtmlEntity("InvisibleComma", "\u2063"),
    HtmlEntity("InvisibleTimes", "\u2062"),
    HtmlEntity("Iogon", "\u012E"),
    HtmlEntity("Iopf", x("D835DD40")),
    HtmlEntity("Iota", "\u0399"),
    HtmlEntity("Iscr", "\u2110"),
    HtmlEntity("Itilde", "\u0128"),
    HtmlEntity("Iukcy", "\u0406"),
    HtmlEntity("Iuml", "\u00CF"),
    HtmlEntity("Jcirc", "\u0134"),
    HtmlEntity("Jcy", "\u0419"),
    HtmlEntity("Jfr", x("D835DD0D")),
    HtmlEntity("Jopf", x("D835DD41")),
    HtmlEntity("Jscr", x("D835DCA5")),
    HtmlEntity("Jsercy", "\u0408"),
    HtmlEntity("Jukcy", "\u0404"),
    HtmlEntity("KHcy", "\u0425"),
    HtmlEntity("KJcy", "\u040C"),
    HtmlEntity("Kappa", "\u039A"),
    HtmlEntity("Kcedil", "\u0136"),
    HtmlEntity("Kcy", "\u041A"),
    HtmlEntity("Kfr", x("D835DD0E")),
    HtmlEntity("Kopf", x("D835DD42")),
    HtmlEntity("Kscr", x("D835DCA6")),
    HtmlEntity("LJcy", "\u0409"),
    HtmlEntity("LT", "\u003C"),
    HtmlEntity("Lacute", "\u0139"),
    HtmlEntity("Lambda", "\u039B"),
    HtmlEntity("Lang", "\u27EA"),
    HtmlEntity("Laplacetrf", "\u2112"),
    HtmlEntity("Larr", "\u219E"),
    HtmlEntity("Lcaron", "\u013D"),
    HtmlEntity("Lcedil", "\u013B"),
    HtmlEntity("Lcy", "\u041B"),
    HtmlEntity("LeftAngleBracket", "\u27E8"),
    HtmlEntity("LeftArrow", "\u2190"),
    HtmlEntity("LeftArrowBar", "\u21E4"),
    HtmlEntity("LeftArrowRightArrow", "\u21C6"),
    HtmlEntity("LeftCeiling", "\u2308"),
    HtmlEntity("LeftDoubleBracket", "\u27E6"),
    HtmlEntity("LeftDownTeeVector", "\u2961"),
    HtmlEntity("LeftDownVector", "\u21C3"),
    HtmlEntity("LeftDownVectorBar", "\u2959"),
    HtmlEntity("LeftFloor", "\u230A"),
    HtmlEntity("LeftRightArrow", "\u2194"),
    HtmlEntity("LeftRightVector", "\u294E"),
    HtmlEntity("LeftTee", "\u22A3"),
    HtmlEntity("LeftTeeArrow", "\u21A4"),
    HtmlEntity("LeftTeeVector", "\u295A"),
    HtmlEntity("LeftTriangle", "\u22B2"),
    HtmlEntity("LeftTriangleBar", "\u29CF"),
    HtmlEntity("LeftTriangleEqual", "\u22B4"),
    HtmlEntity("LeftUpDownVector", "\u2951"),
    HtmlEntity("LeftUpTeeVector", "\u2960"),
    HtmlEntity("LeftUpVector", "\u21BF"),
    HtmlEntity("LeftUpVectorBar", "\u2958"),
    HtmlEntity("LeftVector", "\u21BC"),
    HtmlEntity("LeftVectorBar", "\u2952"),
    HtmlEntity("Leftarrow", "\u21D0"),
    HtmlEntity("Leftrightarrow", "\u21D4"),
    HtmlEntity("LessEqualGreater", "\u22DA"),
    HtmlEntity("LessFullEqual", "\u2266"),
    HtmlEntity("LessGreater", "\u2276"),
    HtmlEntity("LessLess", "\u2AA1"),
    HtmlEntity("LessSlantEqual", "\u2A7D"),
    HtmlEntity("LessTilde", "\u2272"),
    HtmlEntity("Lfr", x("D835DD0F")),
    HtmlEntity("Ll", "\u22D8"),
    HtmlEntity("Lleftarrow", "\u21DA"),
    HtmlEntity("Lmidot", "\u013F"),
    HtmlEntity("LongLeftArrow", "\u27F5"),
    HtmlEntity("LongLeftRightArrow", "\u27F7"),
    HtmlEntity("LongRightArrow", "\u27F6"),
    HtmlEntity("Longleftarrow", "\u27F8"),
    HtmlEntity("Longleftrightarrow", "\u27FA"),
    HtmlEntity("Longrightarrow", "\u27F9"),
    HtmlEntity("Lopf", x("D835DD43")),
    HtmlEntity("LowerLeftArrow", "\u2199"),
    HtmlEntity("LowerRightArrow", "\u2198"),
    HtmlEntity("Lscr", "\u2112"),
    HtmlEntity("Lsh", "\u21B0"),
    HtmlEntity("Lstrok", "\u0141"),
    HtmlEntity("Lt", "\u226A"),
    HtmlEntity("Map", "\u2905"),
    HtmlEntity("Mcy", "\u041C"),
    HtmlEntity("MediumSpace", "\u205F"),
    HtmlEntity("Mellintrf", "\u2133"),
    HtmlEntity("Mfr", x("D835DD10")),
    HtmlEntity("MinusPlus", "\u2213"),
    HtmlEntity("Mopf", x("D835DD44")),
    HtmlEntity("Mscr", "\u2133"),
    HtmlEntity("Mu", "\u039C"),
    HtmlEntity("NJcy", "\u040A"),
    HtmlEntity("Nacute", "\u0143"),
    HtmlEntity("Ncaron", "\u0147"),
    HtmlEntity("Ncedil", "\u0145"),
    HtmlEntity("Ncy", "\u041D"),
    HtmlEntity("NegativeMediumSpace", "\u200B"),
    HtmlEntity("NegativeThickSpace", "\u200B"),
    HtmlEntity("NegativeThinSpace", "\u200B"),
    HtmlEntity("NegativeVeryThinSpace", "\u200B"),
    HtmlEntity("NestedGreaterGreater", "\u226B"),
    HtmlEntity("NestedLessLess", "\u226A"),
    HtmlEntity("NewLine", "\u000A"),
    HtmlEntity("Nfr", x("D835DD11")),
    HtmlEntity("NoBreak", "\u2060"),
    HtmlEntity("NonBreakingSpace", "\u00A0"),
    HtmlEntity("Nopf", "\u2115"),
    HtmlEntity("Not", "\u2AEC"),
    HtmlEntity("NotCongruent", "\u2262"),
    HtmlEntity("NotCupCap", "\u226D"),
    HtmlEntity("NotDoubleVerticalBar", "\u2226"),
    HtmlEntity("NotElement", "\u2209"),
    HtmlEntity("NotEqual", "\u2260"),
    HtmlEntity("NotEqualTilde", x("22420338")),
    HtmlEntity("NotExists", "\u2204"),
    HtmlEntity("NotGreater", "\u226F"),
    HtmlEntity("NotGreaterEqual", "\u2271"),
    HtmlEntity("NotGreaterFullEqual", x("22670338")),
    HtmlEntity("NotGreaterGreater", x("226B0338")),
    HtmlEntity("NotGreaterLess", "\u2279"),
    HtmlEntity("NotGreaterSlantEqual", x("2A7E0338")),
    HtmlEntity("NotGreaterTilde", "\u2275"),
    HtmlEntity("NotHumpDownHump", x("224E0338")),
    HtmlEntity("NotHumpEqual", x("224F0338")),
    HtmlEntity("NotLeftTriangle", "\u22EA"),
    HtmlEntity("NotLeftTriangleBar", x("29CF0338")),
    HtmlEntity("NotLeftTriangleEqual", "\u22EC"),
    HtmlEntity("NotLess", "\u226E"),
    HtmlEntity("NotLessEqual", "\u2270"),
    HtmlEntity("NotLessGreater", "\u2278"),
    HtmlEntity("NotLessLess", x("226A0338")),
    HtmlEntity("NotLessSlantEqual", x("2A7D0338")),
    HtmlEntity("NotLessTilde", "\u2274"),
    HtmlEntity("NotNestedGreaterGreater", x("2AA20338")),
    HtmlEntity("NotNestedLessLess", x("2AA10338")),
    HtmlEntity("NotPrecedes", "\u2280"),
    HtmlEntity("NotPrecedesEqual", x("2AAF0338")),
    HtmlEntity("NotPrecedesSlantEqual", "\u22E0"),
    HtmlEntity("NotReverseElement", "\u220C"),
    HtmlEntity("NotRightTriangle", "\u22EB"),
    HtmlEntity("NotRightTriangleBar", x("29D00338")),
    HtmlEntity("NotRightTriangleEqual", "\u22ED"),
    HtmlEntity("NotSquareSubset", x("228F0338")),
    HtmlEntity("NotSquareSubsetEqual", "\u22E2"),
    HtmlEntity("NotSquareSuperset", x("22900338")),
    HtmlEntity("NotSquareSupersetEqual", "\u22E3"),
    HtmlEntity("NotSubset", x("228220D2")),
    HtmlEntity("NotSubsetEqual", "\u2288"),
    HtmlEntity("NotSucceeds", "\u2281"),
    HtmlEntity("NotSucceedsEqual", x("2AB00338")),
    HtmlEntity("NotSucceedsSlantEqual", "\u22E1"),
    HtmlEntity("NotSucceedsTilde", x("227F0338")),
    HtmlEntity("NotSuperset", x("228320D2")),
    HtmlEntity("NotSupersetEqual", "\u2289"),
    HtmlEntity("NotTilde", "\u2241"),
    HtmlEntity("NotTildeEqual", "\u2244"),
    HtmlEntity("NotTildeFullEqual", "\u2247"),
    HtmlEntity("NotTildeTilde", "\u2249"),
    HtmlEntity("NotVerticalBar", "\u2224"),
    HtmlEntity("Nscr", x("D835DCA9")),
    HtmlEntity("Ntilde", "\u00D1"),
    HtmlEntity("Nu", "\u039D"),
    HtmlEntity("OElig", "\u0152"),
    HtmlEntity("Oacute", "\u00D3"),
    HtmlEntity("Ocirc", "\u00D4"),
    HtmlEntity("Ocy", "\u041E"),
    HtmlEntity("Odblac", "\u0150"),
    HtmlEntity("Ofr", x("D835DD12")),
    HtmlEntity("Ograve", "\u00D2"),
    HtmlEntity("Omacr", "\u014C"),
    HtmlEntity("Omega", "\u03A9"),
    HtmlEntity("Omicron", "\u039F"),
    HtmlEntity("Oopf", x("D835DD46")),
    HtmlEntity("OpenCurlyDoubleQuote", "\u201C"),
    HtmlEntity("OpenCurlyQuote", "\u2018"),
    HtmlEntity("Or", "\u2A54"),
    HtmlEntity("Oscr", x("D835DCAA")),
    HtmlEntity("Oslash", "\u00D8"),
    HtmlEntity("Otilde", "\u00D5"),
    HtmlEntity("Otimes", "\u2A37"),
    HtmlEntity("Ouml", "\u00D6"),
    HtmlEntity("OverBar", "\u203E"),
    HtmlEntity("OverBrace", "\u23DE"),
    HtmlEntity("OverBracket", "\u23B4"),
    HtmlEntity("OverParenthesis", "\u23DC"),
    HtmlEntity("PartialD", "\u2202"),
    HtmlEntity("Pcy", "\u041F"),
    HtmlEntity("Pfr", x("D835DD13")),
    HtmlEntity("Phi", "\u03A6"),
    HtmlEntity("Pi", "\u03A0"),
    HtmlEntity("PlusMinus", "\u00B1"),
    HtmlEntity("Poincareplane", "\u210C"),
    HtmlEntity("Popf", "\u2119"),
    HtmlEntity("Pr", "\u2ABB"),
    HtmlEntity("Precedes", "\u227A"),
    HtmlEntity("PrecedesEqual", "\u2AAF"),
    HtmlEntity("PrecedesSlantEqual", "\u227C"),
    HtmlEntity("PrecedesTilde", "\u227E"),
    HtmlEntity("Prime", "\u2033"),
    HtmlEntity("Product", "\u220F"),
    HtmlEntity("Proportion", "\u2237"),
    HtmlEntity("Proportional", "\u221D"),
    HtmlEntity("Pscr", x("D835DCAB")),
    HtmlEntity("Psi", "\u03A8"),
    HtmlEntity("QUOT", "\u0022"),
    HtmlEntity("Qfr", x("D835DD14")),
    HtmlEntity("Qopf", "\u211A"),
    HtmlEntity("Qscr", x("D835DCAC")),
    HtmlEntity("RBarr", "\u2910"),
    HtmlEntity("REG", "\u00AE"),
    HtmlEntity("Racute", "\u0154"),
    HtmlEntity("Rang", "\u27EB"),
    HtmlEntity("Rarr", "\u21A0"),
    HtmlEntity("Rarrtl", "\u2916"),
    HtmlEntity("Rcaron", "\u0158"),
    HtmlEntity("Rcedil", "\u0156"),
    HtmlEntity("Rcy", "\u0420"),
    HtmlEntity("Re", "\u211C"),
    HtmlEntity("ReverseElement", "\u220B"),
    HtmlEntity("ReverseEquilibrium", "\u21CB"),
    HtmlEntity("ReverseUpEquilibrium", "\u296F"),
    HtmlEntity("Rfr", "\u211C"),
    HtmlEntity("Rho", "\u03A1"),
    HtmlEntity("RightAngleBracket", "\u27E9"),
    HtmlEntity("RightArrow", "\u2192"),
    HtmlEntity("RightArrowBar", "\u21E5"),
    HtmlEntity("RightArrowLeftArrow", "\u21C4"),
    HtmlEntity("RightCeiling", "\u2309"),
    HtmlEntity("RightDoubleBracket", "\u27E7"),
    HtmlEntity("RightDownTeeVector", "\u295D"),
    HtmlEntity("RightDownVector", "\u21C2"),
    HtmlEntity("RightDownVectorBar", "\u2955"),
    HtmlEntity("RightFloor", "\u230B"),
    HtmlEntity("RightTee", "\u22A2"),
    HtmlEntity("RightTeeArrow", "\u21A6"),
    HtmlEntity("RightTeeVector", "\u295B"),
    HtmlEntity("RightTriangle", "\u22B3"),
    HtmlEntity("RightTriangleBar", "\u29D0"),
    HtmlEntity("RightTriangleEqual", "\u22B5"),
    HtmlEntity("RightUpDownVector", "\u294F"),
    HtmlEntity("RightUpTeeVector", "\u295C"),
    HtmlEntity("RightUpVector", "\u21BE"),
    HtmlEntity("RightUpVectorBar", "\u2954"),
    HtmlEntity("RightVector", "\u21C0"),
    HtmlEntity("RightVectorBar", "\u2953"),
    HtmlEntity("Rightarrow", "\u21D2"),
    HtmlEntity("Ropf", "\u211D"),
    HtmlEntity("RoundImplies", "\u2970"),
    HtmlEntity("Rrightarrow", "\u21DB"),
    HtmlEntity("Rscr", "\u211B"),
    HtmlEntity("Rsh", "\u21B1"),
    HtmlEntity("RuleDelayed", "\u29F4"),
    HtmlEntity("SHCHcy", "\u0429"),
    HtmlEntity("SHcy", "\u0428"),
    HtmlEntity("SOFTcy", "\u042C"),
    HtmlEntity("Sacute", "\u015A"),
    HtmlEntity("Sc", "\u2ABC"),
    HtmlEntity("Scaron", "\u0160"),
    HtmlEntity("Scedil", "\u015E"),
    HtmlEntity("Scirc", "\u015C"),
    HtmlEntity("Scy", "\u0421"),
    HtmlEntity("Sfr", x("D835DD16")),
    HtmlEntity("ShortDownArrow", "\u2193"),
    HtmlEntity("ShortLeftArrow", "\u2190"),
    HtmlEntity("ShortRightArrow", "\u2192"),
    HtmlEntity("ShortUpArrow", "\u2191"),
    HtmlEntity("Sigma", "\u03A3"),
    HtmlEntity("SmallCircle", "\u2218"),
    HtmlEntity("Sopf", x("D835DD4A")),
    HtmlEntity("Sqrt", "\u221A"),
    HtmlEntity("Square", "\u25A1"),
    HtmlEntity("SquareIntersection", "\u2293"),
    HtmlEntity("SquareSubset", "\u228F"),
    HtmlEntity("SquareSubsetEqual", "\u2291"),
    HtmlEntity("SquareSuperset", "\u2290"),
    HtmlEntity("SquareSupersetEqual", "\u2292"),
    HtmlEntity("SquareUnion", "\u2294"),
    HtmlEntity("Sscr", x("D835DCAE")),
    HtmlEntity("Star", "\u22C6"),
    HtmlEntity("Sub", "\u22D0"),
    HtmlEntity("Subset", "\u22D0"),
    HtmlEntity("SubsetEqual", "\u2286"),
    HtmlEntity("Succeeds", "\u227B"),
    HtmlEntity("SucceedsEqual", "\u2AB0"),
    HtmlEntity("SucceedsSlantEqual", "\u227D"),
    HtmlEntity("SucceedsTilde", "\u227F"),
    HtmlEntity("SuchThat", "\u220B"),
    HtmlEntity("Sum", "\u2211"),
    HtmlEntity("Sup", "\u22D1"),
    HtmlEntity("Superset", "\u2283"),
    HtmlEntity("SupersetEqual", "\u2287"),
    HtmlEntity("Supset", "\u22D1"),
    HtmlEntity("THORN", "\u00DE"),
    HtmlEntity("TRADE", "\u2122"),
    HtmlEntity("TSHcy", "\u040B"),
    HtmlEntity("TScy", "\u0426"),
    HtmlEntity("Tab", "\u0009"),
    HtmlEntity("Tau", "\u03A4"),
    HtmlEntity("Tcaron", "\u0164"),
    HtmlEntity("Tcedil", "\u0162"),
    HtmlEntity("Tcy", "\u0422"),
    HtmlEntity("Tfr", x("D835DD17")),
    HtmlEntity("Therefore", "\u2234"),
    HtmlEntity("Theta", "\u0398"),
    HtmlEntity("ThickSpace", x("205F200A")),
    HtmlEntity("ThinSpace", "\u2009"),
    HtmlEntity("Tilde", "\u223C"),
    HtmlEntity("TildeEqual", "\u2243"),
    HtmlEntity("TildeFullEqual", "\u2245"),
    HtmlEntity("TildeTilde", "\u2248"),
    HtmlEntity("Topf", x("D835DD4B")),
    HtmlEntity("TripleDot", "\u20DB"),
    HtmlEntity("Tscr", x("D835DCAF")),
    HtmlEntity("Tstrok", "\u0166"),
    HtmlEntity("Uacute", "\u00DA"),
    HtmlEntity("Uarr", "\u219F"),
    HtmlEntity("Uarrocir", "\u2949"),
    HtmlEntity("Ubrcy", "\u040E"),
    HtmlEntity("Ubreve", "\u016C"),
    HtmlEntity("Ucirc", "\u00DB"),
    HtmlEntity("Ucy", "\u0423"),
    HtmlEntity("Udblac", "\u0170"),
    HtmlEntity("Ufr", x("D835DD18")),
    HtmlEntity("Ugrave", "\u00D9"),
    HtmlEntity("Umacr", "\u016A"),
    HtmlEntity("UnderBar", "\u005F"),
    HtmlEntity("UnderBrace", "\u23DF"),
    HtmlEntity("UnderBracket", "\u23B5"),
    HtmlEntity("UnderParenthesis", "\u23DD"),
    HtmlEntity("Union", "\u22C3"),
    HtmlEntity("UnionPlus", "\u228E"),
    HtmlEntity("Uogon", "\u0172"),
    HtmlEntity("Uopf", x("D835DD4C")),
    HtmlEntity("UpArrow", "\u2191"),
    HtmlEntity("UpArrowBar", "\u2912"),
    HtmlEntity("UpArrowDownArrow", "\u21C5"),
    HtmlEntity("UpDownArrow", "\u2195"),
    HtmlEntity("UpEquilibrium", "\u296E"),
    HtmlEntity("UpTee", "\u22A5"),
    HtmlEntity("UpTeeArrow", "\u21A5"),
    HtmlEntity("Uparrow", "\u21D1"),
    HtmlEntity("Updownarrow", "\u21D5"),
    HtmlEntity("UpperLeftArrow", "\u2196"),
    HtmlEntity("UpperRightArrow", "\u2197"),
    HtmlEntity("Upsi", "\u03D2"),
    HtmlEntity("Upsilon", "\u03A5"),
    HtmlEntity("Uring", "\u016E"),
    HtmlEntity("Uscr", x("D835DCB0")),
    HtmlEntity("Utilde", "\u0168"),
    HtmlEntity("Uuml", "\u00DC"),
    HtmlEntity("VDash", "\u22AB"),
    HtmlEntity("Vbar", "\u2AEB"),
    HtmlEntity("Vcy", "\u0412"),
    HtmlEntity("Vdash", "\u22A9"),
    HtmlEntity("Vdashl", "\u2AE6"),
    HtmlEntity("Vee", "\u22C1"),
    HtmlEntity("Verbar", "\u2016"),
    HtmlEntity("Vert", "\u2016"),
    HtmlEntity("VerticalBar", "\u2223"),
    HtmlEntity("VerticalLine", "\u007C"),
    HtmlEntity("VerticalSeparator", "\u2758"),
    HtmlEntity("VerticalTilde", "\u2240"),
    HtmlEntity("VeryThinSpace", "\u200A"),
    HtmlEntity("Vfr", x("D835DD19")),
    HtmlEntity("Vopf", x("D835DD4D")),
    HtmlEntity("Vscr", x("D835DCB1")),
    HtmlEntity("Vvdash", "\u22AA"),
    HtmlEntity("Wcirc", "\u0174"),
    HtmlEntity("Wedge", "\u22C0"),
    HtmlEntity("Wfr", x("D835DD1A")),
    HtmlEntity("Wopf", x("D835DD4E")),
    HtmlEntity("Wscr", x("D835DCB2")),
    HtmlEntity("Xfr", x("D835DD1B")),
    HtmlEntity("Xi", "\u039E"),
    HtmlEntity("Xopf", x("D835DD4F")),
    HtmlEntity("Xscr", x("D835DCB3")),
    HtmlEntity("YAcy", "\u042F"),
    HtmlEntity("YIcy", "\u0407"),
    HtmlEntity("YUcy", "\u042E"),
    HtmlEntity("Yacute", "\u00DD"),
    HtmlEntity("Ycirc", "\u0176"),
    HtmlEntity("Ycy", "\u042B"),
    HtmlEntity("Yfr", x("D835DD1C")),
    HtmlEntity("Yopf", x("D835DD50")),
    HtmlEntity("Yscr", x("D835DCB4")),
    HtmlEntity("Yuml", "\u0178"),
    HtmlEntity("ZHcy", "\u0416"),
    HtmlEntity("Zacute", "\u0179"),
    HtmlEntity("Zcaron", "\u017D"),
    HtmlEntity("Zcy", "\u0417"),
    HtmlEntity("Zdot", "\u017B"),
    HtmlEntity("ZeroWidthSpace", "\u200B"),
    HtmlEntity("Zeta", "\u0396"),
    HtmlEntity("Zfr", "\u2128"),
    HtmlEntity("Zopf", "\u2124"),
    HtmlEntity("Zscr", x("D835DCB5")),
    HtmlEntity("aacute", "\u00E1"),
    HtmlEntity("abreve", "\u0103"),
    HtmlEntity("ac", "\u223E"),
    HtmlEntity("acE", x("223E0333")),
    HtmlEntity("acd", "\u223F"),
    HtmlEntity("acirc", "\u00E2"),
    HtmlEntity("acute", "\u00B4"),
    HtmlEntity("acy", "\u0430"),
    HtmlEntity("aelig", "\u00E6"),
    HtmlEntity("af", "\u2061"),
    HtmlEntity("afr", x("D835DD1E")),
    HtmlEntity("agrave", "\u00E0"),
    HtmlEntity("alefsym", "\u2135"),
    HtmlEntity("aleph", "\u2135"),
    HtmlEntity("alpha", "\u03B1"),
    HtmlEntity("amacr", "\u0101"),
    HtmlEntity("amalg", "\u2A3F"),
    HtmlEntity("amp", "\u0026"),
    HtmlEntity("and", "\u2227"),
    HtmlEntity("andand", "\u2A55"),
    HtmlEntity("andd", "\u2A5C"),
    HtmlEntity("andslope", "\u2A58"),
    HtmlEntity("andv", "\u2A5A"),
    HtmlEntity("ang", "\u2220"),
    HtmlEntity("ange", "\u29A4"),
    HtmlEntity("angle", "\u2220"),
    HtmlEntity("angmsd", "\u2221"),
    HtmlEntity("angmsdaa", "\u29A8"),
    HtmlEntity("angmsdab", "\u29A9"),
    HtmlEntity("angmsdac", "\u29AA"),
    HtmlEntity("angmsdad", "\u29AB"),
    HtmlEntity("angmsdae", "\u29AC"),
    HtmlEntity("angmsdaf", "\u29AD"),
    HtmlEntity("angmsdag", "\u29AE"),
    HtmlEntity("angmsdah", "\u29AF"),
    HtmlEntity("angrt", "\u221F"),
    HtmlEntity("angrtvb", "\u22BE"),
    HtmlEntity("angrtvbd", "\u299D"),
    HtmlEntity("angsph", "\u2222"),
    HtmlEntity("angst", "\u00C5"),
    HtmlEntity("angzarr", "\u237C"),
    HtmlEntity("aogon", "\u0105"),
    HtmlEntity("aopf", x("D835DD52")),
    HtmlEntity("ap", "\u2248"),
    HtmlEntity("apE", "\u2A70"),
    HtmlEntity("apacir", "\u2A6F"),
    HtmlEntity("ape", "\u224A"),
    HtmlEntity("apid", "\u224B"),
    HtmlEntity("apos", "\u0027"),
    HtmlEntity("approx", "\u2248"),
    HtmlEntity("approxeq", "\u224A"),
    HtmlEntity("aring", "\u00E5"),
    HtmlEntity("ascr", x("D835DCB6")),
    HtmlEntity("ast", "\u002A"),
    HtmlEntity("asymp", "\u2248"),
    HtmlEntity("asympeq", "\u224D"),
    HtmlEntity("atilde", "\u00E3"),
    HtmlEntity("auml", "\u00E4"),
    HtmlEntity("awconint", "\u2233"),
    HtmlEntity("awint", "\u2A11"),
    HtmlEntity("bNot", "\u2AED"),
    HtmlEntity("backcong", "\u224C"),
    HtmlEntity("backepsilon", "\u03F6"),
    HtmlEntity("backprime", "\u2035"),
    HtmlEntity("backsim", "\u223D"),
    HtmlEntity("backsimeq", "\u22CD"),
    HtmlEntity("barvee", "\u22BD"),
    HtmlEntity("barwed", "\u2305"),
    HtmlEntity("barwedge", "\u2305"),
    HtmlEntity("bbrk", "\u23B5"),
    HtmlEntity("bbrktbrk", "\u23B6"),
    HtmlEntity("bcong", "\u224C"),
    HtmlEntity("bcy", "\u0431"),
    HtmlEntity("bdquo", "\u201E"),
    HtmlEntity("becaus", "\u2235"),
    HtmlEntity("because", "\u2235"),
    HtmlEntity("bemptyv", "\u29B0"),
    HtmlEntity("bepsi", "\u03F6"),
    HtmlEntity("bernou", "\u212C"),
    HtmlEntity("beta", "\u03B2"),
    HtmlEntity("beth", "\u2136"),
    HtmlEntity("between", "\u226C"),
    HtmlEntity("bfr", x("D835DD1F")),
    HtmlEntity("bigcap", "\u22C2"),
    HtmlEntity("bigcirc", "\u25EF"),
    HtmlEntity("bigcup", "\u22C3"),
    HtmlEntity("bigodot", "\u2A00"),
    HtmlEntity("bigoplus", "\u2A01"),
    HtmlEntity("bigotimes", "\u2A02"),
    HtmlEntity("bigsqcup", "\u2A06"),
    HtmlEntity("bigstar", "\u2605"),
    HtmlEntity("bigtriangledown", "\u25BD"),
    HtmlEntity("bigtriangleup", "\u25B3"),
    HtmlEntity("biguplus", "\u2A04"),
    HtmlEntity("bigvee", "\u22C1"),
    HtmlEntity("bigwedge", "\u22C0"),
    HtmlEntity("bkarow", "\u290D"),
    HtmlEntity("blacklozenge", "\u29EB"),
    HtmlEntity("blacksquare", "\u25AA"),
    HtmlEntity("blacktriangle", "\u25B4"),
    HtmlEntity("blacktriangledown", "\u25BE"),
    HtmlEntity("blacktriangleleft", "\u25C2"),
    HtmlEntity("blacktriangleright", "\u25B8"),
    HtmlEntity("blank", "\u2423"),
    HtmlEntity("blk12", "\u2592"),
    HtmlEntity("blk14", "\u2591"),
    HtmlEntity("blk34", "\u2593"),
    HtmlEntity("block", "\u2588"),
    HtmlEntity("bne", x("003D20E5")),
    HtmlEntity("bnequiv", x("226120E5")),
    HtmlEntity("bnot", "\u2310"),
    HtmlEntity("bopf", x("D835DD53")),
    HtmlEntity("bot", "\u22A5"),
    HtmlEntity("bottom", "\u22A5"),
    HtmlEntity("bowtie", "\u22C8"),
    HtmlEntity("boxDL", "\u2557"),
    HtmlEntity("boxDR", "\u2554"),
    HtmlEntity("boxDl", "\u2556"),
    HtmlEntity("boxDr", "\u2553"),
    HtmlEntity("boxH", "\u2550"),
    HtmlEntity("boxHD", "\u2566"),
    HtmlEntity("boxHU", "\u2569"),
    HtmlEntity("boxHd", "\u2564"),
    HtmlEntity("boxHu", "\u2567"),
    HtmlEntity("boxUL", "\u255D"),
    HtmlEntity("boxUR", "\u255A"),
    HtmlEntity("boxUl", "\u255C"),
    HtmlEntity("boxUr", "\u2559"),
    HtmlEntity("boxV", "\u2551"),
    HtmlEntity("boxVH", "\u256C"),
    HtmlEntity("boxVL", "\u2563"),
    HtmlEntity("boxVR", "\u2560"),
    HtmlEntity("boxVh", "\u256B"),
    HtmlEntity("boxVl", "\u2562"),
    HtmlEntity("boxVr", "\u255F"),
    HtmlEntity("boxbox!", "\u29C9"),
    HtmlEntity("boxdL", "\u2555"),
    HtmlEntity("boxdR", "\u2552"),
    HtmlEntity("boxdl", "\u2510"),
    HtmlEntity("boxdr", "\u250C"),
    HtmlEntity("boxh", "\u2500"),
    HtmlEntity("boxhD", "\u2565"),
    HtmlEntity("boxhU", "\u2568"),
    HtmlEntity("boxhd", "\u252C"),
    HtmlEntity("boxhu", "\u2534"),
    HtmlEntity("boxminus", "\u229F"),
    HtmlEntity("boxplus", "\u229E"),
    HtmlEntity("boxtimes", "\u22A0"),
    HtmlEntity("boxuL", "\u255B"),
    HtmlEntity("boxuR", "\u2558"),
    HtmlEntity("boxul", "\u2518"),
    HtmlEntity("boxur", "\u2514"),
    HtmlEntity("boxv", "\u2502"),
    HtmlEntity("boxvH", "\u256A"),
    HtmlEntity("boxvL", "\u2561"),
    HtmlEntity("boxvR", "\u255E"),
    HtmlEntity("boxvh", "\u253C"),
    HtmlEntity("boxvl", "\u2524"),
    HtmlEntity("boxvr", "\u251C"),
    HtmlEntity("bprime", "\u2035"),
    HtmlEntity("breve", "\u02D8"),
    HtmlEntity("brvbar", "\u00A6"),
    HtmlEntity("bscr", x("D835DCB7")),
    HtmlEntity("bsemi", "\u204F"),
    HtmlEntity("bsim", "\u223D"),
    HtmlEntity("bsime", "\u22CD"),
    HtmlEntity("bsol", "\u005C"),
    HtmlEntity("bsolb", "\u29C5"),
    HtmlEntity("bsolhsub", "\u27C8"),
    HtmlEntity("bull", "\u2022"),
    HtmlEntity("bullet", "\u2022"),
    HtmlEntity("bump", "\u224E"),
    HtmlEntity("bumpE", "\u2AAE"),
    HtmlEntity("bumpe", "\u224F"),
    HtmlEntity("bumpeq", "\u224F"),
    HtmlEntity("cacute", "\u0107"),
    HtmlEntity("cap", "\u2229"),
    HtmlEntity("capand", "\u2A44"),
    HtmlEntity("capbrcup", "\u2A49"),
    HtmlEntity("capcap", "\u2A4B"),
    HtmlEntity("capcup", "\u2A47"),
    HtmlEntity("capdot", "\u2A40"),
    HtmlEntity("caps", x("2229FE00")),
    HtmlEntity("caret", "\u2041"),
    HtmlEntity("caron", "\u02C7"),
    HtmlEntity("ccaps", "\u2A4D"),
    HtmlEntity("ccaron", "\u010D"),
    HtmlEntity("ccedil", "\u00E7"),
    HtmlEntity("ccirc", "\u0109"),
    HtmlEntity("ccups", "\u2A4C"),
    HtmlEntity("ccupssm", "\u2A50"),
    HtmlEntity("cdot", "\u010B"),
    HtmlEntity("cedil", "\u00B8"),
    HtmlEntity("cemptyv", "\u29B2"),
    HtmlEntity("cent", "\u00A2"),
    HtmlEntity("centerdot", "\u00B7"),
    HtmlEntity("cfr", x("D835DD20")),
    HtmlEntity("chcy", "\u0447"),
    HtmlEntity("check", "\u2713"),
    HtmlEntity("checkmark", "\u2713"),
    HtmlEntity("chi", "\u03C7"),
    HtmlEntity("cir", "\u25CB"),
    HtmlEntity("cirE", "\u29C3"),
    HtmlEntity("circ", "\u02C6"),
    HtmlEntity("circeq", "\u2257"),
    HtmlEntity("circlearrowleft", "\u21BA"),
    HtmlEntity("circlearrowright", "\u21BB"),
    HtmlEntity("circledR", "\u00AE"),
    HtmlEntity("circledS", "\u24C8"),
    HtmlEntity("circledast", "\u229B"),
    HtmlEntity("circledcirc", "\u229A"),
    HtmlEntity("circleddash", "\u229D"),
    HtmlEntity("cire", "\u2257"),
    HtmlEntity("cirfnint", "\u2A10"),
    HtmlEntity("cirmid", "\u2AEF"),
    HtmlEntity("cirscir", "\u29C2"),
    HtmlEntity("clubs", "\u2663"),
    HtmlEntity("clubsuit", "\u2663"),
    HtmlEntity("colon", "\u003A"),
    HtmlEntity("colone", "\u2254"),
    HtmlEntity("coloneq", "\u2254"),
    HtmlEntity("comma", "\u002C"),
    HtmlEntity("commat", "\u0040"),
    HtmlEntity("comp", "\u2201"),
    HtmlEntity("compfn", "\u2218"),
    HtmlEntity("complement", "\u2201"),
    HtmlEntity("complexes", "\u2102"),
    HtmlEntity("cong", "\u2245"),
    HtmlEntity("congdot", "\u2A6D"),
    HtmlEntity("conint", "\u222E"),
    HtmlEntity("copf", x("D835DD54")),
    HtmlEntity("coprod", "\u2210"),
    HtmlEntity("copy", "\u00A9"),
    HtmlEntity("copysr", "\u2117"),
    HtmlEntity("crarr", "\u21B5"),
    HtmlEntity("cross", "\u2717"),
    HtmlEntity("cscr", x("D835DCB8")),
    HtmlEntity("csub", "\u2ACF"),
    HtmlEntity("csube", "\u2AD1"),
    HtmlEntity("csup", "\u2AD0"),
    HtmlEntity("csupe", "\u2AD2"),
    HtmlEntity("ctdot", "\u22EF"),
    HtmlEntity("cudarrl", "\u2938"),
    HtmlEntity("cudarrr", "\u2935"),
    HtmlEntity("cuepr", "\u22DE"),
    HtmlEntity("cuesc", "\u22DF"),
    HtmlEntity("cularr", "\u21B6"),
    HtmlEntity("cularrp", "\u293D"),
    HtmlEntity("cup", "\u222A"),
    HtmlEntity("cupbrcap", "\u2A48"),
    HtmlEntity("cupcap", "\u2A46"),
    HtmlEntity("cupcup", "\u2A4A"),
    HtmlEntity("cupdot", "\u228D"),
    HtmlEntity("cupor", "\u2A45"),
    HtmlEntity("cups", x("222AFE00")),
    HtmlEntity("curarr", "\u21B7"),
    HtmlEntity("curarrm", "\u293C"),
    HtmlEntity("curlyeqprec", "\u22DE"),
    HtmlEntity("curlyeqsucc", "\u22DF"),
    HtmlEntity("curlyvee", "\u22CE"),
    HtmlEntity("curlywedge", "\u22CF"),
    HtmlEntity("curren", "\u00A4"),
    HtmlEntity("curvearrowleft", "\u21B6"),
    HtmlEntity("curvearrowright", "\u21B7"),
    HtmlEntity("cuvee", "\u22CE"),
    HtmlEntity("cuwed", "\u22CF"),
    HtmlEntity("cwconint", "\u2232"),
    HtmlEntity("cwint", "\u2231"),
    HtmlEntity("cylcty", "\u232D"),
    HtmlEntity("dArr", "\u21D3"),
    HtmlEntity("dHar", "\u2965"),
    HtmlEntity("dagger", "\u2020"),
    HtmlEntity("daleth", "\u2138"),
    HtmlEntity("darr", "\u2193"),
    HtmlEntity("dash", "\u2010"),
    HtmlEntity("dashv", "\u22A3"),
    HtmlEntity("dbkarow", "\u290F"),
    HtmlEntity("dblac", "\u02DD"),
    HtmlEntity("dcaron", "\u010F"),
    HtmlEntity("dcy", "\u0434"),
    HtmlEntity("dd", "\u2146"),
    HtmlEntity("ddagger", "\u2021"),
    HtmlEntity("ddarr", "\u21CA"),
    HtmlEntity("ddotseq", "\u2A77"),
    HtmlEntity("deg", "\u00B0"),
    HtmlEntity("delta", "\u03B4"),
    HtmlEntity("demptyv", "\u29B1"),
    HtmlEntity("dfisht", "\u297F"),
    HtmlEntity("dfr", x("D835DD21")),
    HtmlEntity("dharl", "\u21C3"),
    HtmlEntity("dharr", "\u21C2"),
    HtmlEntity("diam", "\u22C4"),
    HtmlEntity("diamond", "\u22C4"),
    HtmlEntity("diamondsuit", "\u2666"),
    HtmlEntity("diams", "\u2666"),
    HtmlEntity("die", "\u00A8"),
    HtmlEntity("digamma", "\u03DD"),
    HtmlEntity("disin", "\u22F2"),
    HtmlEntity("div", "\u00F7"),
    HtmlEntity("divide", "\u00F7"),
    HtmlEntity("divideontimes", "\u22C7"),
    HtmlEntity("divonx!", "\u22C7"),
    HtmlEntity("djcy", "\u0452"),
    HtmlEntity("dlcorn", "\u231E"),
    HtmlEntity("dlcrop", "\u230D"),
    HtmlEntity("dollar", "\u0024"),
    HtmlEntity("dopf", x("D835DD55")),
    HtmlEntity("dot", "\u02D9"),
    HtmlEntity("doteq", "\u2250"),
    HtmlEntity("doteqdot", "\u2251"),
    HtmlEntity("dotminus", "\u2238"),
    HtmlEntity("dotplus", "\u2214"),
    HtmlEntity("dotsquare", "\u22A1"),
    HtmlEntity("doublebarwedge", "\u2306"),
    HtmlEntity("downarrow", "\u2193"),
    HtmlEntity("downdownarrows", "\u21CA"),
    HtmlEntity("downharpoonleft", "\u21C3"),
    HtmlEntity("downharpoonright", "\u21C2"),
    HtmlEntity("drbkarow", "\u2910"),
    HtmlEntity("drcorn", "\u231F"),
    HtmlEntity("drcrop", "\u230C"),
    HtmlEntity("dscr", x("D835DCB9")),
    HtmlEntity("dscy", "\u0455"),
    HtmlEntity("dsol", "\u29F6"),
    HtmlEntity("dstrok", "\u0111"),
    HtmlEntity("dtdot", "\u22F1"),
    HtmlEntity("dtri", "\u25BF"),
    HtmlEntity("dtrif", "\u25BE"),
    HtmlEntity("duarr", "\u21F5"),
    HtmlEntity("duhar", "\u296F"),
    HtmlEntity("dwangle", "\u29A6"),
    HtmlEntity("dzcy", "\u045F"),
    HtmlEntity("dzigrarr", "\u27FF"),
    HtmlEntity("eDDot", "\u2A77"),
    HtmlEntity("eDot", "\u2251"),
    HtmlEntity("eacute", "\u00E9"),
    HtmlEntity("easter", "\u2A6E"),
    HtmlEntity("ecaron", "\u011B"),
    HtmlEntity("ecir", "\u2256"),
    HtmlEntity("ecirc", "\u00EA"),
    HtmlEntity("ecolon", "\u2255"),
    HtmlEntity("ecy", "\u044D"),
    HtmlEntity("edot", "\u0117"),
    HtmlEntity("ee", "\u2147"),
    HtmlEntity("efDot", "\u2252"),
    HtmlEntity("efr", x("D835DD22")),
    HtmlEntity("eg", "\u2A9A"),
    HtmlEntity("egrave", "\u00E8"),
    HtmlEntity("egs", "\u2A96"),
    HtmlEntity("egsdot", "\u2A98"),
    HtmlEntity("el", "\u2A99"),
    HtmlEntity("elinters", "\u23E7"),
    HtmlEntity("ell", "\u2113"),
    HtmlEntity("els", "\u2A95"),
    HtmlEntity("elsdot", "\u2A97"),
    HtmlEntity("emacr", "\u0113"),
    HtmlEntity("empty", "\u2205"),
    HtmlEntity("emptyset", "\u2205"),
    HtmlEntity("emptyv", "\u2205"),
    HtmlEntity("emsp", "\u2003"),
    HtmlEntity("emsp13", "\u2004"),
    HtmlEntity("emsp14", "\u2005"),
    HtmlEntity("eng", "\u014B"),
    HtmlEntity("ensp", "\u2002"),
    HtmlEntity("eogon", "\u0119"),
    HtmlEntity("eopf", x("D835DD56")),
    HtmlEntity("epar", "\u22D5"),
    HtmlEntity("eparsl", "\u29E3"),
    HtmlEntity("eplus", "\u2A71"),
    HtmlEntity("epsi", "\u03B5"),
    HtmlEntity("epsilon", "\u03B5"),
    HtmlEntity("epsiv", "\u03F5"),
    HtmlEntity("eqcirc", "\u2256"),
    HtmlEntity("eqcolon", "\u2255"),
    HtmlEntity("eqsim", "\u2242"),
    HtmlEntity("eqslantgtr", "\u2A96"),
    HtmlEntity("eqslantless", "\u2A95"),
    HtmlEntity("equals", "\u003D"),
    HtmlEntity("equest", "\u225F"),
    HtmlEntity("equiv", "\u2261"),
    HtmlEntity("equivDD", "\u2A78"),
    HtmlEntity("eqvparsl", "\u29E5"),
    HtmlEntity("erDot", "\u2253"),
    HtmlEntity("erarr", "\u2971"),
    HtmlEntity("escr", "\u212F"),
    HtmlEntity("esdot", "\u2250"),
    HtmlEntity("esim", "\u2242"),
    HtmlEntity("eta", "\u03B7"),
    HtmlEntity("eth", "\u00F0"),
    HtmlEntity("euml", "\u00EB"),
    HtmlEntity("euro", "\u20AC"),
    HtmlEntity("excl", "\u0021"),
    HtmlEntity("exist", "\u2203"),
    HtmlEntity("expectation", "\u2130"),
    HtmlEntity("exponentiale", "\u2147"),
    HtmlEntity("fallingdotseq", "\u2252"),
    HtmlEntity("fcy", "\u0444"),
    HtmlEntity("female", "\u2640"),
    HtmlEntity("ffilig", "\uFB03"),
    HtmlEntity("fflig", "\uFB00"),
    HtmlEntity("ffllig", "\uFB04"),
    HtmlEntity("ffr", x("D835DD23")),
    HtmlEntity("filig", "\uFB01"),
    HtmlEntity("fjlig", x("0066006A")),
    HtmlEntity("flat", "\u266D"),
    HtmlEntity("fllig", "\uFB02"),
    HtmlEntity("fltns", "\u25B1"),
    HtmlEntity("fnof", "\u0192"),
    HtmlEntity("fopf", x("D835DD57")),
    HtmlEntity("forall", "\u2200"),
    HtmlEntity("fork", "\u22D4"),
    HtmlEntity("forkv", "\u2AD9"),
    HtmlEntity("fpartint", "\u2A0D"),
    HtmlEntity("frac12", "\u00BD"),
    HtmlEntity("frac13", "\u2153"),
    HtmlEntity("frac14", "\u00BC"),
    HtmlEntity("frac15", "\u2155"),
    HtmlEntity("frac16", "\u2159"),
    HtmlEntity("frac18", "\u215B"),
    HtmlEntity("frac23", "\u2154"),
    HtmlEntity("frac25", "\u2156"),
    HtmlEntity("frac34", "\u00BE"),
    HtmlEntity("frac35", "\u2157"),
    HtmlEntity("frac38", "\u215C"),
    HtmlEntity("frac45", "\u2158"),
    HtmlEntity("frac56", "\u215A"),
    HtmlEntity("frac58", "\u215D"),
    HtmlEntity("frac78", "\u215E"),
    HtmlEntity("frasl", "\u2044"),
    HtmlEntity("frown", "\u2322"),
    HtmlEntity("fscr", x("D835DCBB")),
    HtmlEntity("gE", "\u2267"),
    HtmlEntity("gEl", "\u2A8C"),
    HtmlEntity("gacute", "\u01F5"),
    HtmlEntity("gamma", "\u03B3"),
    HtmlEntity("gammad", "\u03DD"),
    HtmlEntity("gap", "\u2A86"),
    HtmlEntity("gbreve", "\u011F"),
    HtmlEntity("gcirc", "\u011D"),
    HtmlEntity("gcy", "\u0433"),
    HtmlEntity("gdot", "\u0121"),
    HtmlEntity("ge", "\u2265"),
    HtmlEntity("gel", "\u22DB"),
    HtmlEntity("geq", "\u2265"),
    HtmlEntity("geqq", "\u2267"),
    HtmlEntity("geqslant", "\u2A7E"),
    HtmlEntity("ges", "\u2A7E"),
    HtmlEntity("gescc", "\u2AA9"),
    HtmlEntity("gesdot", "\u2A80"),
    HtmlEntity("gesdoto", "\u2A82"),
    HtmlEntity("gesdotol", "\u2A84"),
    HtmlEntity("gesl", x("22DBFE00")),
    HtmlEntity("gesles", "\u2A94"),
    HtmlEntity("gfr", x("D835DD24")),
    HtmlEntity("gg", "\u226B"),
    HtmlEntity("ggg", "\u22D9"),
    HtmlEntity("gimel", "\u2137"),
    HtmlEntity("gjcy", "\u0453"),
    HtmlEntity("gl", "\u2277"),
    HtmlEntity("glE", "\u2A92"),
    HtmlEntity("gla", "\u2AA5"),
    HtmlEntity("glj", "\u2AA4"),
    HtmlEntity("gnE", "\u2269"),
    HtmlEntity("gnap", "\u2A8A"),
    HtmlEntity("gnapprox!", "\u2A8A"),
    HtmlEntity("gne", "\u2A88"),
    HtmlEntity("gneq", "\u2A88"),
    HtmlEntity("gneqq", "\u2269"),
    HtmlEntity("gnsim", "\u22E7"),
    HtmlEntity("gopf", x("D835DD58")),
    HtmlEntity("grave", "\u0060"),
    HtmlEntity("gscr", "\u210A"),
    HtmlEntity("gsim", "\u2273"),
    HtmlEntity("gsime", "\u2A8E"),
    HtmlEntity("gsiml", "\u2A90"),
    HtmlEntity("gt", "\u003E"),
    HtmlEntity("gtcc", "\u2AA7"),
    HtmlEntity("gtcir", "\u2A7A"),
    HtmlEntity("gtdot", "\u22D7"),
    HtmlEntity("gtlPar", "\u2995"),
    HtmlEntity("gtquest", "\u2A7C"),
    HtmlEntity("gtrapprox!", "\u2A86"),
    HtmlEntity("gtrarr", "\u2978"),
    HtmlEntity("gtrdot", "\u22D7"),
    HtmlEntity("gtreqless", "\u22DB"),
    HtmlEntity("gtreqqless", "\u2A8C"),
    HtmlEntity("gtrless", "\u2277"),
    HtmlEntity("gtrsim", "\u2273"),
    HtmlEntity("gvertneqq", x("2269FE00")),
    HtmlEntity("gvnE", x("2269FE00")),
    HtmlEntity("hArr", "\u21D4"),
    HtmlEntity("hairsp", "\u200A"),
    HtmlEntity("half", "\u00BD"),
    HtmlEntity("hamilt", "\u210B"),
    HtmlEntity("hardcy", "\u044A"),
    HtmlEntity("harr", "\u2194"),
    HtmlEntity("harrcir", "\u2948"),
    HtmlEntity("harrw", "\u21AD"),
    HtmlEntity("hbar", "\u210F"),
    HtmlEntity("hcirc", "\u0125"),
    HtmlEntity("hearts", "\u2665"),
    HtmlEntity("heartsuit", "\u2665"),
    HtmlEntity("hellip", "\u2026"),
    HtmlEntity("hercon", "\u22B9"),
    HtmlEntity("hfr", x("D835DD25")),
    HtmlEntity("hksearow", "\u2925"),
    HtmlEntity("hkswarow", "\u2926"),
    HtmlEntity("hoarr", "\u21FF"),
    HtmlEntity("homtht", "\u223B"),
    HtmlEntity("hookleftarrow", "\u21A9"),
    HtmlEntity("hookrightarrow", "\u21AA"),
    HtmlEntity("hopf", x("D835DD59")),
    HtmlEntity("horbar", "\u2015"),
    HtmlEntity("hscr", x("D835DCBD")),
    HtmlEntity("hslash", "\u210F"),
    HtmlEntity("hstrok", "\u0127"),
    HtmlEntity("hybull", "\u2043"),
    HtmlEntity("hyphen", "\u2010"),
    HtmlEntity("iacute", "\u00ED"),
    HtmlEntity("ic", "\u2063"),
    HtmlEntity("icirc", "\u00EE"),
    HtmlEntity("icy", "\u0438"),
    HtmlEntity("iecy", "\u0435"),
    HtmlEntity("iexcl", "\u00A1"),
    HtmlEntity("iff", "\u21D4"),
    HtmlEntity("ifr", x("D835DD26")),
    HtmlEntity("igrave", "\u00EC"),
    HtmlEntity("ii", "\u2148"),
    HtmlEntity("iiiint", "\u2A0C"),
    HtmlEntity("iiint", "\u222D"),
    HtmlEntity("iinfin", "\u29DC"),
    HtmlEntity("iiota", "\u2129"),
    HtmlEntity("ijlig", "\u0133"),
    HtmlEntity("imacr", "\u012B"),
    HtmlEntity("image", "\u2111"),
    HtmlEntity("imagline", "\u2110"),
    HtmlEntity("imagpart", "\u2111"),
    HtmlEntity("imath", "\u0131"),
    HtmlEntity("imof", "\u22B7"),
    HtmlEntity("imped", "\u01B5"),
    HtmlEntity("in", "\u2208"),
    HtmlEntity("incare", "\u2105"),
    HtmlEntity("infin", "\u221E"),
    HtmlEntity("infintie", "\u29DD"),
    HtmlEntity("inodot", "\u0131"),
    HtmlEntity("int", "\u222B"),
    HtmlEntity("intcal", "\u22BA"),
    HtmlEntity("integers", "\u2124"),
    HtmlEntity("intercal", "\u22BA"),
    HtmlEntity("intlarhk", "\u2A17"),
    HtmlEntity("intprod", "\u2A3C"),
    HtmlEntity("iocy", "\u0451"),
    HtmlEntity("iogon", "\u012F"),
    HtmlEntity("iopf", x("D835DD5A")),
    HtmlEntity("iota", "\u03B9"),
    HtmlEntity("iprod", "\u2A3C"),
    HtmlEntity("iquest", "\u00BF"),
    HtmlEntity("iscr", x("D835DCBE")),
    HtmlEntity("isin", "\u2208"),
    HtmlEntity("isinE", "\u22F9"),
    HtmlEntity("isindot", "\u22F5"),
    HtmlEntity("isins", "\u22F4"),
    HtmlEntity("isinsv", "\u22F3"),
    HtmlEntity("isinv", "\u2208"),
    HtmlEntity("it", "\u2062"),
    HtmlEntity("itilde", "\u0129"),
    HtmlEntity("iukcy", "\u0456"),
    HtmlEntity("iuml", "\u00EF"),
    HtmlEntity("jcirc", "\u0135"),
    HtmlEntity("jcy", "\u0439"),
    HtmlEntity("jfr", x("D835DD27")),
    HtmlEntity("jmath", "\u0237"),
    HtmlEntity("jopf", x("D835DD5B")),
    HtmlEntity("jscr", x("D835DCBF")),
    HtmlEntity("jsercy", "\u0458"),
    HtmlEntity("jukcy", "\u0454"),
    HtmlEntity("kappa", "\u03BA"),
    HtmlEntity("kappav", "\u03F0"),
    HtmlEntity("kcedil", "\u0137"),
    HtmlEntity("kcy", "\u043A"),
    HtmlEntity("kfr", x("D835DD28")),
    HtmlEntity("kgreen", "\u0138"),
    HtmlEntity("khcy", "\u0445"),
    HtmlEntity("kjcy", "\u045C"),
    HtmlEntity("kopf", x("D835DD5C")),
    HtmlEntity("kscr", x("D835DCC0")),
    HtmlEntity("lAarr", "\u21DA"),
    HtmlEntity("lArr", "\u21D0"),
    HtmlEntity("lAtail", "\u291B"),
    HtmlEntity("lBarr", "\u290E"),
    HtmlEntity("lE", "\u2266"),
    HtmlEntity("lEg", "\u2A8B"),
    HtmlEntity("lHar", "\u2962"),
    HtmlEntity("lacute", "\u013A"),
    HtmlEntity("laemptyv", "\u29B4"),
    HtmlEntity("lagran", "\u2112"),
    HtmlEntity("lambda", "\u03BB"),
    HtmlEntity("lang", "\u27E8"),
    HtmlEntity("langd", "\u2991"),
    HtmlEntity("langle", "\u27E8"),
    HtmlEntity("lap", "\u2A85"),
    HtmlEntity("laquo", "\u00AB"),
    HtmlEntity("larr", "\u2190"),
    HtmlEntity("larrb", "\u21E4"),
    HtmlEntity("larrbfs", "\u291F"),
    HtmlEntity("larrfs", "\u291D"),
    HtmlEntity("larrhk", "\u21A9"),
    HtmlEntity("larrlp", "\u21AB"),
    HtmlEntity("larrpl", "\u2939"),
    HtmlEntity("larrsim", "\u2973"),
    HtmlEntity("larrtl", "\u21A2"),
    HtmlEntity("lat", "\u2AAB"),
    HtmlEntity("latail", "\u2919"),
    HtmlEntity("late", "\u2AAD"),
    HtmlEntity("lates", x("2AADFE00")),
    HtmlEntity("lbarr", "\u290C"),
    HtmlEntity("lbbrk", "\u2772"),
    HtmlEntity("lbrace", "\u007B"),
    HtmlEntity("lbrack", "\u005B"),
    HtmlEntity("lbrke", "\u298B"),
    HtmlEntity("lbrksld", "\u298F"),
    HtmlEntity("lbrkslu", "\u298D"),
    HtmlEntity("lcaron", "\u013E"),
    HtmlEntity("lcedil", "\u013C"),
    HtmlEntity("lceil", "\u2308"),
    HtmlEntity("lcub", "\u007B"),
    HtmlEntity("lcy", "\u043B"),
    HtmlEntity("ldca", "\u2936"),
    HtmlEntity("ldquo", "\u201C"),
    HtmlEntity("ldquor", "\u201E"),
    HtmlEntity("ldrdhar", "\u2967"),
    HtmlEntity("ldrushar", "\u294B"),
    HtmlEntity("ldsh", "\u21B2"),
    HtmlEntity("le", "\u2264"),
    HtmlEntity("leftarrow", "\u2190"),
    HtmlEntity("leftarrowtail", "\u21A2"),
    HtmlEntity("leftharpoondown", "\u21BD"),
    HtmlEntity("leftharpoonup", "\u21BC"),
    HtmlEntity("leftleftarrows", "\u21C7"),
    HtmlEntity("leftrightarrow", "\u2194"),
    HtmlEntity("leftrightarrows", "\u21C6"),
    HtmlEntity("leftrightharpoons", "\u21CB"),
    HtmlEntity("leftrightsquigarrow", "\u21AD"),
    HtmlEntity("leftthreetimes", "\u22CB"),
    HtmlEntity("leg", "\u22DA"),
    HtmlEntity("leq", "\u2264"),
    HtmlEntity("leqq", "\u2266"),
    HtmlEntity("leqslant", "\u2A7D"),
    HtmlEntity("les", "\u2A7D"),
    HtmlEntity("lescc", "\u2AA8"),
    HtmlEntity("lesdot", "\u2A7F"),
    HtmlEntity("lesdoto", "\u2A81"),
    HtmlEntity("lesdotor", "\u2A83"),
    HtmlEntity("lesg", x("22DAFE00")),
    HtmlEntity("lesges", "\u2A93"),
    HtmlEntity("lessapprox!", "\u2A85"),
    HtmlEntity("lessdot", "\u22D6"),
    HtmlEntity("lesseqgtr", "\u22DA"),
    HtmlEntity("lesseqqgtr", "\u2A8B"),
    HtmlEntity("lessgtr", "\u2276"),
    HtmlEntity("lesssim", "\u2272"),
    HtmlEntity("lfisht", "\u297C"),
    HtmlEntity("lfloor", "\u230A"),
    HtmlEntity("lfr", x("D835DD29")),
    HtmlEntity("lg", "\u2276"),
    HtmlEntity("lgE", "\u2A91"),
    HtmlEntity("lhard", "\u21BD"),
    HtmlEntity("lharu", "\u21BC"),
    HtmlEntity("lharul", "\u296A"),
    HtmlEntity("lhblk", "\u2584"),
    HtmlEntity("ljcy", "\u0459"),
    HtmlEntity("ll", "\u226A"),
    HtmlEntity("llarr", "\u21C7"),
    HtmlEntity("llcorner", "\u231E"),
    HtmlEntity("llhard", "\u296B"),
    HtmlEntity("lltri", "\u25FA"),
    HtmlEntity("lmidot", "\u0140"),
    HtmlEntity("lmoust", "\u23B0"),
    HtmlEntity("lmoustache", "\u23B0"),
    HtmlEntity("lnE", "\u2268"),
    HtmlEntity("lnap", "\u2A89"),
    HtmlEntity("lnapprox!", "\u2A89"),
    HtmlEntity("lne", "\u2A87"),
    HtmlEntity("lneq", "\u2A87"),
    HtmlEntity("lneqq", "\u2268"),
    HtmlEntity("lnsim", "\u22E6"),
    HtmlEntity("loang", "\u27EC"),
    HtmlEntity("loarr", "\u21FD"),
    HtmlEntity("lobrk", "\u27E6"),
    HtmlEntity("longleftarrow", "\u27F5"),
    HtmlEntity("longleftrightarrow", "\u27F7"),
    HtmlEntity("longmapsto", "\u27FC"),
    HtmlEntity("longrightarrow", "\u27F6"),
    HtmlEntity("looparrowleft", "\u21AB"),
    HtmlEntity("looparrowright", "\u21AC"),
    HtmlEntity("lopar", "\u2985"),
    HtmlEntity("lopf", x("D835DD5D")),
    HtmlEntity("loplus", "\u2A2D"),
    HtmlEntity("lotimes", "\u2A34"),
    HtmlEntity("lowast", "\u2217"),
    HtmlEntity("lowbar", "\u005F"),
    HtmlEntity("loz", "\u25CA"),
    HtmlEntity("lozenge", "\u25CA"),
    HtmlEntity("lozf", "\u29EB"),
    HtmlEntity("lpar", "\u0028"),
    HtmlEntity("lparlt", "\u2993"),
    HtmlEntity("lrarr", "\u21C6"),
    HtmlEntity("lrcorner", "\u231F"),
    HtmlEntity("lrhar", "\u21CB"),
    HtmlEntity("lrhard", "\u296D"),
    HtmlEntity("lrm", "\u200E"),
    HtmlEntity("lrtri", "\u22BF"),
    HtmlEntity("lsaquo", "\u2039"),
    HtmlEntity("lscr", x("D835DCC1")),
    HtmlEntity("lsh", "\u21B0"),
    HtmlEntity("lsim", "\u2272"),
    HtmlEntity("lsime", "\u2A8D"),
    HtmlEntity("lsimg", "\u2A8F"),
    HtmlEntity("lsqb", "\u005B"),
    HtmlEntity("lsquo", "\u2018"),
    HtmlEntity("lsquor", "\u201A"),
    HtmlEntity("lstrok", "\u0142"),
    HtmlEntity("lt", "\u003C"),
    HtmlEntity("ltcc", "\u2AA6"),
    HtmlEntity("ltcir", "\u2A79"),
    HtmlEntity("ltdot", "\u22D6"),
    HtmlEntity("lthree", "\u22CB"),
    HtmlEntity("ltimes", "\u22C9"),
    HtmlEntity("ltlarr", "\u2976"),
    HtmlEntity("ltquest", "\u2A7B"),
    HtmlEntity("ltrPar", "\u2996"),
    HtmlEntity("ltri", "\u25C3"),
    HtmlEntity("ltrie", "\u22B4"),
    HtmlEntity("ltrif", "\u25C2"),
    HtmlEntity("lurdshar", "\u294A"),
    HtmlEntity("luruhar", "\u2966"),
    HtmlEntity("lvertneqq", x("2268FE00")),
    HtmlEntity("lvnE", x("2268FE00")),
    HtmlEntity("mDDot", "\u223A"),
    HtmlEntity("macr", "\u00AF"),
    HtmlEntity("male", "\u2642"),
    HtmlEntity("malt", "\u2720"),
    HtmlEntity("maltese", "\u2720"),
    HtmlEntity("map", "\u21A6"),
    HtmlEntity("mapsto", "\u21A6"),
    HtmlEntity("mapstodown", "\u21A7"),
    HtmlEntity("mapstoleft", "\u21A4"),
    HtmlEntity("mapstoup", "\u21A5"),
    HtmlEntity("marker", "\u25AE"),
    HtmlEntity("mcomma", "\u2A29"),
    HtmlEntity("mcy", "\u043C"),
    HtmlEntity("mdash", "\u2014"),
    HtmlEntity("measuredangle", "\u2221"),
    HtmlEntity("mfr", x("D835DD2A")),
    HtmlEntity("mho", "\u2127"),
    HtmlEntity("micro", "\u00B5"),
    HtmlEntity("mid", "\u2223"),
    HtmlEntity("midast", "\u002A"),
    HtmlEntity("midcir", "\u2AF0"),
    HtmlEntity("middot", "\u00B7"),
    HtmlEntity("minus", "\u2212"),
    HtmlEntity("minusb", "\u229F"),
    HtmlEntity("minusd", "\u2238"),
    HtmlEntity("minusdu", "\u2A2A"),
    HtmlEntity("mlcp", "\u2ADB"),
    HtmlEntity("mldr", "\u2026"),
    HtmlEntity("mnplus", "\u2213"),
    HtmlEntity("models", "\u22A7"),
    HtmlEntity("mopf", x("D835DD5E")),
    HtmlEntity("mp", "\u2213"),
    HtmlEntity("mscr", x("D835DCC2")),
    HtmlEntity("mstpos", "\u223E"),
    HtmlEntity("mu", "\u03BC"),
    HtmlEntity("multimap", "\u22B8"),
    HtmlEntity("mumap", "\u22B8"),
    HtmlEntity("nGg", x("22D90338")),
    HtmlEntity("nGt", x("226B20D2")),
    HtmlEntity("nGtv", x("226B0338")),
    HtmlEntity("nLeftarrow", "\u21CD"),
    HtmlEntity("nLeftrightarrow", "\u21CE"),
    HtmlEntity("nLl", x("22D80338")),
    HtmlEntity("nLt", x("226A20D2")),
    HtmlEntity("nLtv", x("226A0338")),
    HtmlEntity("nRightarrow", "\u21CF"),
    HtmlEntity("nVDash", "\u22AF"),
    HtmlEntity("nVdash", "\u22AE"),
    HtmlEntity("nabla", "\u2207"),
    HtmlEntity("nacute", "\u0144"),
    HtmlEntity("nang", x("222020D2")),
    HtmlEntity("nap", "\u2249"),
    HtmlEntity("napE", x("2A700338")),
    HtmlEntity("napid", x("224B0338")),
    HtmlEntity("napos", "\u0149"),
    HtmlEntity("napprox!", "\u2249"),
    HtmlEntity("natur", "\u266E"),
    HtmlEntity("natural", "\u266E"),
    HtmlEntity("naturals", "\u2115"),
    HtmlEntity("nbsp", "\u00A0"),
    HtmlEntity("nbump", x("224E0338")),
    HtmlEntity("nbumpe", x("224F0338")),
    HtmlEntity("ncap", "\u2A43"),
    HtmlEntity("ncaron", "\u0148"),
    HtmlEntity("ncedil", "\u0146"),
    HtmlEntity("ncong", "\u2247"),
    HtmlEntity("ncongdot", x("2A6D0338")),
    HtmlEntity("ncup", "\u2A42"),
    HtmlEntity("ncy", "\u043D"),
    HtmlEntity("ndash", "\u2013"),
    HtmlEntity("ne", "\u2260"),
    HtmlEntity("neArr", "\u21D7"),
    HtmlEntity("nearhk", "\u2924"),
    HtmlEntity("nearr", "\u2197"),
    HtmlEntity("nearrow", "\u2197"),
    HtmlEntity("nedot", x("22500338")),
    HtmlEntity("nequiv", "\u2262"),
    HtmlEntity("nesear", "\u2928"),
    HtmlEntity("nesim", x("22420338")),
    HtmlEntity("nexist", "\u2204"),
    HtmlEntity("nexists", "\u2204"),
    HtmlEntity("nfr", x("D835DD2B")),
    HtmlEntity("ngE", x("22670338")),
    HtmlEntity("nge", "\u2271"),
    HtmlEntity("ngeq", "\u2271"),
    HtmlEntity("ngeqq", x("22670338")),
    HtmlEntity("ngeqslant", x("2A7E0338")),
    HtmlEntity("nges", x("2A7E0338")),
    HtmlEntity("ngsim", "\u2275"),
    HtmlEntity("ngt", "\u226F"),
    HtmlEntity("ngtr", "\u226F"),
    HtmlEntity("nhArr", "\u21CE"),
    HtmlEntity("nharr", "\u21AE"),
    HtmlEntity("nhpar", "\u2AF2"),
    HtmlEntity("ni", "\u220B"),
    HtmlEntity("nis", "\u22FC"),
    HtmlEntity("nisd", "\u22FA"),
    HtmlEntity("niv", "\u220B"),
    HtmlEntity("njcy", "\u045A"),
    HtmlEntity("nlArr", "\u21CD"),
    HtmlEntity("nlE", x("22660338")),
    HtmlEntity("nlarr", "\u219A"),
    HtmlEntity("nldr", "\u2025"),
    HtmlEntity("nle", "\u2270"),
    HtmlEntity("nleftarrow", "\u219A"),
    HtmlEntity("nleftrightarrow", "\u21AE"),
    HtmlEntity("nleq", "\u2270"),
    HtmlEntity("nleqq", x("22660338")),
    HtmlEntity("nleqslant", x("2A7D0338")),
    HtmlEntity("nles", x("2A7D0338")),
    HtmlEntity("nless", "\u226E"),
    HtmlEntity("nlsim", "\u2274"),
    HtmlEntity("nlt", "\u226E"),
    HtmlEntity("nltri", "\u22EA"),
    HtmlEntity("nltrie", "\u22EC"),
    HtmlEntity("nmid", "\u2224"),
    HtmlEntity("nopf", x("D835DD5F")),
    HtmlEntity("not", "\u00AC"),
    HtmlEntity("notin", "\u2209"),
    HtmlEntity("notinE", x("22F90338")),
    HtmlEntity("notindot", x("22F50338")),
    HtmlEntity("notinva", "\u2209"),
    HtmlEntity("notinvb", "\u22F7"),
    HtmlEntity("notinvc", "\u22F6"),
    HtmlEntity("notni", "\u220C"),
    HtmlEntity("notniva", "\u220C"),
    HtmlEntity("notnivb", "\u22FE"),
    HtmlEntity("notnivc", "\u22FD"),
    HtmlEntity("npar", "\u2226"),
    HtmlEntity("nparallel", "\u2226"),
    HtmlEntity("nparsl", x("2AFD20E5")),
    HtmlEntity("npart", x("22020338")),
    HtmlEntity("npolint", "\u2A14"),
    HtmlEntity("npr", "\u2280"),
    HtmlEntity("nprcue", "\u22E0"),
    HtmlEntity("npre", x("2AAF0338")),
    HtmlEntity("nprec", "\u2280"),
    HtmlEntity("npreceq", x("2AAF0338")),
    HtmlEntity("nrArr", "\u21CF"),
    HtmlEntity("nrarr", "\u219B"),
    HtmlEntity("nrarrc", x("29330338")),
    HtmlEntity("nrarrw", x("219D0338")),
    HtmlEntity("nrightarrow", "\u219B"),
    HtmlEntity("nrtri", "\u22EB"),
    HtmlEntity("nrtrie", "\u22ED"),
    HtmlEntity("nsc", "\u2281"),
    HtmlEntity("nsccue", "\u22E1"),
    HtmlEntity("nsce", x("2AB00338")),
    HtmlEntity("nscr", x("D835DCC3")),
    HtmlEntity("nshortmid", "\u2224"),
    HtmlEntity("nshortparallel", "\u2226"),
    HtmlEntity("nsim", "\u2241"),
    HtmlEntity("nsime", "\u2244"),
    HtmlEntity("nsimeq", "\u2244"),
    HtmlEntity("nsmid", "\u2224"),
    HtmlEntity("nspar", "\u2226"),
    HtmlEntity("nsqsube", "\u22E2"),
    HtmlEntity("nsqsupe", "\u22E3"),
    HtmlEntity("nsub", "\u2284"),
    HtmlEntity("nsubE", x("2AC50338")),
    HtmlEntity("nsube", "\u2288"),
    HtmlEntity("nsubset", x("228220D2")),
    HtmlEntity("nsubseteq", "\u2288"),
    HtmlEntity("nsubseteqq", x("2AC50338")),
    HtmlEntity("nsucc", "\u2281"),
    HtmlEntity("nsucceq", x("2AB00338")),
    HtmlEntity("nsup", "\u2285"),
    HtmlEntity("nsupE", x("2AC60338")),
    HtmlEntity("nsupe", "\u2289"),
    HtmlEntity("nsupset", x("228320D2")),
    HtmlEntity("nsupseteq", "\u2289"),
    HtmlEntity("nsupseteqq", x("2AC60338")),
    HtmlEntity("ntgl", "\u2279"),
    HtmlEntity("ntilde", "\u00F1"),
    HtmlEntity("ntlg", "\u2278"),
    HtmlEntity("ntriangleleft", "\u22EA"),
    HtmlEntity("ntrianglelefteq", "\u22EC"),
    HtmlEntity("ntriangleright", "\u22EB"),
    HtmlEntity("ntrianglerighteq", "\u22ED"),
    HtmlEntity("nu", "\u03BD"),
    HtmlEntity("num", "\u0023"),
    HtmlEntity("numero", "\u2116"),
    HtmlEntity("numsp", "\u2007"),
    HtmlEntity("nvDash", "\u22AD"),
    HtmlEntity("nvHarr", "\u2904"),
    HtmlEntity("nvap", x("224D20D2")),
    HtmlEntity("nvdash", "\u22AC"),
    HtmlEntity("nvge", x("226520D2")),
    HtmlEntity("nvgt", x("003E20D2")),
    HtmlEntity("nvinfin", "\u29DE"),
    HtmlEntity("nvlArr", "\u2902"),
    HtmlEntity("nvle", x("226420D2")),
    HtmlEntity("nvlt", x("003C20D2")),
    HtmlEntity("nvltrie", x("22B420D2")),
    HtmlEntity("nvrArr", "\u2903"),
    HtmlEntity("nvrtrie", x("22B520D2")),
    HtmlEntity("nvsim", x("223C20D2")),
    HtmlEntity("nwArr", "\u21D6"),
    HtmlEntity("nwarhk", "\u2923"),
    HtmlEntity("nwarr", "\u2196"),
    HtmlEntity("nwarrow", "\u2196"),
    HtmlEntity("nwnear", "\u2927"),
    HtmlEntity("oS", "\u24C8"),
    HtmlEntity("oacute", "\u00F3"),
    HtmlEntity("oast", "\u229B"),
    HtmlEntity("ocir", "\u229A"),
    HtmlEntity("ocirc", "\u00F4"),
    HtmlEntity("ocy", "\u043E"),
    HtmlEntity("odash", "\u229D"),
    HtmlEntity("odblac", "\u0151"),
    HtmlEntity("odiv", "\u2A38"),
    HtmlEntity("odot", "\u2299"),
    HtmlEntity("odsold", "\u29BC"),
    HtmlEntity("oelig", "\u0153"),
    HtmlEntity("ofcir", "\u29BF"),
    HtmlEntity("ofr", x("D835DD2C")),
    HtmlEntity("ogon", "\u02DB"),
    HtmlEntity("ograve", "\u00F2"),
    HtmlEntity("ogt", "\u29C1"),
    HtmlEntity("ohbar", "\u29B5"),
    HtmlEntity("ohm", "\u03A9"),
    HtmlEntity("oint", "\u222E"),
    HtmlEntity("olarr", "\u21BA"),
    HtmlEntity("olcir", "\u29BE"),
    HtmlEntity("olcross", "\u29BB"),
    HtmlEntity("oline", "\u203E"),
    HtmlEntity("olt", "\u29C0"),
    HtmlEntity("omacr", "\u014D"),
    HtmlEntity("omega", "\u03C9"),
    HtmlEntity("omicron", "\u03BF"),
    HtmlEntity("omid", "\u29B6"),
    HtmlEntity("ominus", "\u2296"),
    HtmlEntity("oopf", x("D835DD60")),
    HtmlEntity("opar", "\u29B7"),
    HtmlEntity("operp", "\u29B9"),
    HtmlEntity("oplus", "\u2295"),
    HtmlEntity("or", "\u2228"),
    HtmlEntity("orarr", "\u21BB"),
    HtmlEntity("ord", "\u2A5D"),
    HtmlEntity("order", "\u2134"),
    HtmlEntity("orderof", "\u2134"),
    HtmlEntity("ordf", "\u00AA"),
    HtmlEntity("ordm", "\u00BA"),
    HtmlEntity("origof", "\u22B6"),
    HtmlEntity("oror", "\u2A56"),
    HtmlEntity("orslope", "\u2A57"),
    HtmlEntity("orv", "\u2A5B"),
    HtmlEntity("oscr", "\u2134"),
    HtmlEntity("oslash", "\u00F8"),
    HtmlEntity("osol", "\u2298"),
    HtmlEntity("otilde", "\u00F5"),
    HtmlEntity("otimes", "\u2297"),
    HtmlEntity("otimesas", "\u2A36"),
    HtmlEntity("ouml", "\u00F6"),
    HtmlEntity("ovbar", "\u233D"),
    HtmlEntity("par", "\u2225"),
    HtmlEntity("para", "\u00B6"),
    HtmlEntity("parallel", "\u2225"),
    HtmlEntity("parsim", "\u2AF3"),
    HtmlEntity("parsl", "\u2AFD"),
    HtmlEntity("part", "\u2202"),
    HtmlEntity("pcy", "\u043F"),
    HtmlEntity("percnt", "\u0025"),
    HtmlEntity("period", "\u002E"),
    HtmlEntity("permil", "\u2030"),
    HtmlEntity("perp", "\u22A5"),
    HtmlEntity("pertenk", "\u2031"),
    HtmlEntity("pfr", x("D835DD2D")),
    HtmlEntity("phi", "\u03C6"),
    HtmlEntity("phiv", "\u03D5"),
    HtmlEntity("phmmat", "\u2133"),
    HtmlEntity("phone", "\u260E"),
    HtmlEntity("pi", "\u03C0"),
    HtmlEntity("pitchfork", "\u22D4"),
    HtmlEntity("piv", "\u03D6"),
    HtmlEntity("planck", "\u210F"),
    HtmlEntity("planckh", "\u210E"),
    HtmlEntity("plankv", "\u210F"),
    HtmlEntity("plus", "\u002B"),
    HtmlEntity("plusacir", "\u2A23"),
    HtmlEntity("plusb", "\u229E"),
    HtmlEntity("pluscir", "\u2A22"),
    HtmlEntity("plusdo", "\u2214"),
    HtmlEntity("plusdu", "\u2A25"),
    HtmlEntity("pluse", "\u2A72"),
    HtmlEntity("plusmn", "\u00B1"),
    HtmlEntity("plussim", "\u2A26"),
    HtmlEntity("plustwo", "\u2A27"),
    HtmlEntity("pm", "\u00B1"),
    HtmlEntity("pointint", "\u2A15"),
    HtmlEntity("popf", x("D835DD61")),
    HtmlEntity("pound", "\u00A3"),
    HtmlEntity("pr", "\u227A"),
    HtmlEntity("prE", "\u2AB3"),
    HtmlEntity("prap", "\u2AB7"),
    HtmlEntity("prcue", "\u227C"),
    HtmlEntity("pre", "\u2AAF"),
    HtmlEntity("prec", "\u227A"),
    HtmlEntity("precapprox!", "\u2AB7"),
    HtmlEntity("preccurlyeq", "\u227C"),
    HtmlEntity("preceq", "\u2AAF"),
    HtmlEntity("precnapprox!", "\u2AB9"),
    HtmlEntity("precneqq", "\u2AB5"),
    HtmlEntity("precnsim", "\u22E8"),
    HtmlEntity("precsim", "\u227E"),
    HtmlEntity("prime", "\u2032"),
    HtmlEntity("primes", "\u2119"),
    HtmlEntity("prnE", "\u2AB5"),
    HtmlEntity("prnap", "\u2AB9"),
    HtmlEntity("prnsim", "\u22E8"),
    HtmlEntity("prod", "\u220F"),
    HtmlEntity("profalar", "\u232E"),
    HtmlEntity("profline", "\u2312"),
    HtmlEntity("profsurf", "\u2313"),
    HtmlEntity("prop", "\u221D"),
    HtmlEntity("propto", "\u221D"),
    HtmlEntity("prsim", "\u227E"),
    HtmlEntity("prurel", "\u22B0"),
    HtmlEntity("pscr", x("D835DCC5")),
    HtmlEntity("psi", "\u03C8"),
    HtmlEntity("puncsp", "\u2008"),
    HtmlEntity("qfr", x("D835DD2E")),
    HtmlEntity("qint", "\u2A0C"),
    HtmlEntity("qopf", x("D835DD62")),
    HtmlEntity("qprime", "\u2057"),
    HtmlEntity("qscr", x("D835DCC6")),
    HtmlEntity("quaternions", "\u210D"),
    HtmlEntity("quatint", "\u2A16"),
    HtmlEntity("quest", "\u003F"),
    HtmlEntity("questeq", "\u225F"),
    HtmlEntity("quot", "\u0022"),
    HtmlEntity("rAarr", "\u21DB"),
    HtmlEntity("rArr", "\u21D2"),
    HtmlEntity("rAtail", "\u291C"),
    HtmlEntity("rBarr", "\u290F"),
    HtmlEntity("rHar", "\u2964"),
    HtmlEntity("race", x("223D0331")),
    HtmlEntity("racute", "\u0155"),
    HtmlEntity("radic", "\u221A"),
    HtmlEntity("raemptyv", "\u29B3"),
    HtmlEntity("rang", "\u27E9"),
    HtmlEntity("rangd", "\u2992"),
    HtmlEntity("range", "\u29A5"),
    HtmlEntity("rangle", "\u27E9"),
    HtmlEntity("raquo", "\u00BB"),
    HtmlEntity("rarr", "\u2192"),
    HtmlEntity("rarrap", "\u2975"),
    HtmlEntity("rarrb", "\u21E5"),
    HtmlEntity("rarrbfs", "\u2920"),
    HtmlEntity("rarrc", "\u2933"),
    HtmlEntity("rarrfs", "\u291E"),
    HtmlEntity("rarrhk", "\u21AA"),
    HtmlEntity("rarrlp", "\u21AC"),
    HtmlEntity("rarrpl", "\u2945"),
    HtmlEntity("rarrsim", "\u2974"),
    HtmlEntity("rarrtl", "\u21A3"),
    HtmlEntity("rarrw", "\u219D"),
    HtmlEntity("ratail", "\u291A"),
    HtmlEntity("ratio", "\u2236"),
    HtmlEntity("rationals", "\u211A"),
    HtmlEntity("rbarr", "\u290D"),
    HtmlEntity("rbbrk", "\u2773"),
    HtmlEntity("rbrace", "\u007D"),
    HtmlEntity("rbrack", "\u005D"),
    HtmlEntity("rbrke", "\u298C"),
    HtmlEntity("rbrksld", "\u298E"),
    HtmlEntity("rbrkslu", "\u2990"),
    HtmlEntity("rcaron", "\u0159"),
    HtmlEntity("rcedil", "\u0157"),
    HtmlEntity("rceil", "\u2309"),
    HtmlEntity("rcub", "\u007D"),
    HtmlEntity("rcy", "\u0440"),
    HtmlEntity("rdca", "\u2937"),
    HtmlEntity("rdldhar", "\u2969"),
    HtmlEntity("rdquo", "\u201D"),
    HtmlEntity("rdquor", "\u201D"),
    HtmlEntity("rdsh", "\u21B3"),
    HtmlEntity("real", "\u211C"),
    HtmlEntity("realine", "\u211B"),
    HtmlEntity("realpart", "\u211C"),
    HtmlEntity("reals", "\u211D"),
    HtmlEntity("rect", "\u25AD"),
    HtmlEntity("reg", "\u00AE"),
    HtmlEntity("rfisht", "\u297D"),
    HtmlEntity("rfloor", "\u230B"),
    HtmlEntity("rfr", x("D835DD2F")),
    HtmlEntity("rhard", "\u21C1"),
    HtmlEntity("rharu", "\u21C0"),
    HtmlEntity("rharul", "\u296C"),
    HtmlEntity("rho", "\u03C1"),
    HtmlEntity("rhov", "\u03F1"),
    HtmlEntity("rightarrow", "\u2192"),
    HtmlEntity("rightarrowtail", "\u21A3"),
    HtmlEntity("rightharpoondown", "\u21C1"),
    HtmlEntity("rightharpoonup", "\u21C0"),
    HtmlEntity("rightleftarrows", "\u21C4"),
    HtmlEntity("rightleftharpoons", "\u21CC"),
    HtmlEntity("rightrightarrows", "\u21C9"),
    HtmlEntity("rightsquigarrow", "\u219D"),
    HtmlEntity("rightthreetimes", "\u22CC"),
    HtmlEntity("ring", "\u02DA"),
    HtmlEntity("risingdotseq", "\u2253"),
    HtmlEntity("rlarr", "\u21C4"),
    HtmlEntity("rlhar", "\u21CC"),
    HtmlEntity("rlm", "\u200F"),
    HtmlEntity("rmoust", "\u23B1"),
    HtmlEntity("rmoustache", "\u23B1"),
    HtmlEntity("rnmid", "\u2AEE"),
    HtmlEntity("roang", "\u27ED"),
    HtmlEntity("roarr", "\u21FE"),
    HtmlEntity("robrk", "\u27E7"),
    HtmlEntity("ropar", "\u2986"),
    HtmlEntity("ropf", x("D835DD63")),
    HtmlEntity("roplus", "\u2A2E"),
    HtmlEntity("rotimes", "\u2A35"),
    HtmlEntity("rpar", "\u0029"),
    HtmlEntity("rpargt", "\u2994"),
    HtmlEntity("rppolint", "\u2A12"),
    HtmlEntity("rrarr", "\u21C9"),
    HtmlEntity("rsaquo", "\u203A"),
    HtmlEntity("rscr", x("D835DCC7")),
    HtmlEntity("rsh", "\u21B1"),
    HtmlEntity("rsqb", "\u005D"),
    HtmlEntity("rsquo", "\u2019"),
    HtmlEntity("rsquor", "\u2019"),
    HtmlEntity("rthree", "\u22CC"),
    HtmlEntity("rtimes", "\u22CA"),
    HtmlEntity("rtri", "\u25B9"),
    HtmlEntity("rtrie", "\u22B5"),
    HtmlEntity("rtrif", "\u25B8"),
    HtmlEntity("rtriltri", "\u29CE"),
    HtmlEntity("ruluhar", "\u2968"),
    HtmlEntity("rx!", "\u211E"),
    HtmlEntity("sacute", "\u015B"),
    HtmlEntity("sbquo", "\u201A"),
    HtmlEntity("sc", "\u227B"),
    HtmlEntity("scE", "\u2AB4"),
    HtmlEntity("scap", "\u2AB8"),
    HtmlEntity("scaron", "\u0161"),
    HtmlEntity("sccue", "\u227D"),
    HtmlEntity("sce", "\u2AB0"),
    HtmlEntity("scedil", "\u015F"),
    HtmlEntity("scirc", "\u015D"),
    HtmlEntity("scnE", "\u2AB6"),
    HtmlEntity("scnap", "\u2ABA"),
    HtmlEntity("scnsim", "\u22E9"),
    HtmlEntity("scpolint", "\u2A13"),
    HtmlEntity("scsim", "\u227F"),
    HtmlEntity("scy", "\u0441"),
    HtmlEntity("sdot", "\u22C5"),
    HtmlEntity("sdotb", "\u22A1"),
    HtmlEntity("sdote", "\u2A66"),
    HtmlEntity("seArr", "\u21D8"),
    HtmlEntity("searhk", "\u2925"),
    HtmlEntity("searr", "\u2198"),
    HtmlEntity("searrow", "\u2198"),
    HtmlEntity("sect", "\u00A7"),
    HtmlEntity("semi", "\u003B"),
    HtmlEntity("seswar", "\u2929"),
    HtmlEntity("setminus", "\u2216"),
    HtmlEntity("setmn", "\u2216"),
    HtmlEntity("sext", "\u2736"),
    HtmlEntity("sfr", x("D835DD30")),
    HtmlEntity("sfrown", "\u2322"),
    HtmlEntity("sharp", "\u266F"),
    HtmlEntity("shchcy", "\u0449"),
    HtmlEntity("shcy", "\u0448"),
    HtmlEntity("shortmid", "\u2223"),
    HtmlEntity("shortparallel", "\u2225"),
    HtmlEntity("shy", "\u00AD"),
    HtmlEntity("sigma", "\u03C3"),
    HtmlEntity("sigmaf", "\u03C2"),
    HtmlEntity("sigmav", "\u03C2"),
    HtmlEntity("sim", "\u223C"),
    HtmlEntity("simdot", "\u2A6A"),
    HtmlEntity("sime", "\u2243"),
    HtmlEntity("simeq", "\u2243"),
    HtmlEntity("simg", "\u2A9E"),
    HtmlEntity("simgE", "\u2AA0"),
    HtmlEntity("siml", "\u2A9D"),
    HtmlEntity("simlE", "\u2A9F"),
    HtmlEntity("simne", "\u2246"),
    HtmlEntity("simplus", "\u2A24"),
    HtmlEntity("simrarr", "\u2972"),
    HtmlEntity("slarr", "\u2190"),
    HtmlEntity("smallsetminus", "\u2216"),
    HtmlEntity("smashp", "\u2A33"),
    HtmlEntity("smeparsl", "\u29E4"),
    HtmlEntity("smid", "\u2223"),
    HtmlEntity("smile", "\u2323"),
    HtmlEntity("smt", "\u2AAA"),
    HtmlEntity("smte", "\u2AAC"),
    HtmlEntity("smtes", x("2AACFE00")),
    HtmlEntity("softcy", "\u044C"),
    HtmlEntity("sol", "\u002F"),
    HtmlEntity("solb", "\u29C4"),
    HtmlEntity("solbar", "\u233F"),
    HtmlEntity("sopf", x("D835DD64")),
    HtmlEntity("spades", "\u2660"),
    HtmlEntity("spadesuit", "\u2660"),
    HtmlEntity("spar", "\u2225"),
    HtmlEntity("sqcap", "\u2293"),
    HtmlEntity("sqcaps", x("2293FE00")),
    HtmlEntity("sqcup", "\u2294"),
    HtmlEntity("sqcups", x("2294FE00")),
    HtmlEntity("sqsub", "\u228F"),
    HtmlEntity("sqsube", "\u2291"),
    HtmlEntity("sqsubset", "\u228F"),
    HtmlEntity("sqsubseteq", "\u2291"),
    HtmlEntity("sqsup", "\u2290"),
    HtmlEntity("sqsupe", "\u2292"),
    HtmlEntity("sqsupset", "\u2290"),
    HtmlEntity("sqsupseteq", "\u2292"),
    HtmlEntity("squ", "\u25A1"),
    HtmlEntity("square", "\u25A1"),
    HtmlEntity("squarf", "\u25AA"),
    HtmlEntity("squf", "\u25AA"),
    HtmlEntity("srarr", "\u2192"),
    HtmlEntity("sscr", x("D835DCC8")),
    HtmlEntity("ssetmn", "\u2216"),
    HtmlEntity("ssmile", "\u2323"),
    HtmlEntity("sstarf", "\u22C6"),
    HtmlEntity("star", "\u2606"),
    HtmlEntity("starf", "\u2605"),
    HtmlEntity("straightepsilon", "\u03F5"),
    HtmlEntity("straightphi", "\u03D5"),
    HtmlEntity("strns", "\u00AF"),
    HtmlEntity("sub", "\u2282"),
    HtmlEntity("subE", "\u2AC5"),
    HtmlEntity("subdot", "\u2ABD"),
    HtmlEntity("sube", "\u2286"),
    HtmlEntity("subedot", "\u2AC3"),
    HtmlEntity("submult", "\u2AC1"),
    HtmlEntity("subnE", "\u2ACB"),
    HtmlEntity("subne", "\u228A"),
    HtmlEntity("subplus", "\u2ABF"),
    HtmlEntity("subrarr", "\u2979"),
    HtmlEntity("subset", "\u2282"),
    HtmlEntity("subseteq", "\u2286"),
    HtmlEntity("subseteqq", "\u2AC5"),
    HtmlEntity("subsetneq", "\u228A"),
    HtmlEntity("subsetneqq", "\u2ACB"),
    HtmlEntity("subsim", "\u2AC7"),
    HtmlEntity("subsub", "\u2AD5"),
    HtmlEntity("subsup", "\u2AD3"),
    HtmlEntity("succ", "\u227B"),
    HtmlEntity("succapprox!", "\u2AB8"),
    HtmlEntity("succcurlyeq", "\u227D"),
    HtmlEntity("succeq", "\u2AB0"),
    HtmlEntity("succnapprox!", "\u2ABA"),
    HtmlEntity("succneqq", "\u2AB6"),
    HtmlEntity("succnsim", "\u22E9"),
    HtmlEntity("succsim", "\u227F"),
    HtmlEntity("sum", "\u2211"),
    HtmlEntity("sung", "\u266A"),
    HtmlEntity("sup", "\u2283"),
    HtmlEntity("sup1", "\u00B9"),
    HtmlEntity("sup2", "\u00B2"),
    HtmlEntity("sup3", "\u00B3"),
    HtmlEntity("supE", "\u2AC6"),
    HtmlEntity("supdot", "\u2ABE"),
    HtmlEntity("supdsub", "\u2AD8"),
    HtmlEntity("supe", "\u2287"),
    HtmlEntity("supedot", "\u2AC4"),
    HtmlEntity("suphsol", "\u27C9"),
    HtmlEntity("suphsub", "\u2AD7"),
    HtmlEntity("suplarr", "\u297B"),
    HtmlEntity("supmult", "\u2AC2"),
    HtmlEntity("supnE", "\u2ACC"),
    HtmlEntity("supne", "\u228B"),
    HtmlEntity("supplus", "\u2AC0"),
    HtmlEntity("supset", "\u2283"),
    HtmlEntity("supseteq", "\u2287"),
    HtmlEntity("supseteqq", "\u2AC6"),
    HtmlEntity("supsetneq", "\u228B"),
    HtmlEntity("supsetneqq", "\u2ACC"),
    HtmlEntity("supsim", "\u2AC8"),
    HtmlEntity("supsub", "\u2AD4"),
    HtmlEntity("supsup", "\u2AD6"),
    HtmlEntity("swArr", "\u21D9"),
    HtmlEntity("swarhk", "\u2926"),
    HtmlEntity("swarr", "\u2199"),
    HtmlEntity("swarrow", "\u2199"),
    HtmlEntity("swnwar", "\u292A"),
    HtmlEntity("szlig", "\u00DF"),
    HtmlEntity("target", "\u2316"),
    HtmlEntity("tau", "\u03C4"),
    HtmlEntity("tbrk", "\u23B4"),
    HtmlEntity("tcaron", "\u0165"),
    HtmlEntity("tcedil", "\u0163"),
    HtmlEntity("tcy", "\u0442"),
    HtmlEntity("tdot", "\u20DB"),
    HtmlEntity("telrec", "\u2315"),
    HtmlEntity("tfr", x("D835DD31")),
    HtmlEntity("there4", "\u2234"),
    HtmlEntity("therefore", "\u2234"),
    HtmlEntity("theta", "\u03B8"),
    HtmlEntity("thetasym", "\u03D1"),
    HtmlEntity("thetav", "\u03D1"),
    HtmlEntity("thickapprox!", "\u2248"),
    HtmlEntity("thicksim", "\u223C"),
    HtmlEntity("thinsp", "\u2009"),
    HtmlEntity("thkap", "\u2248"),
    HtmlEntity("thksim", "\u223C"),
    HtmlEntity("thorn", "\u00FE"),
    HtmlEntity("tilde", "\u02DC"),
    HtmlEntity("times", "\u00D7"),
    HtmlEntity("timesb", "\u22A0"),
    HtmlEntity("timesbar", "\u2A31"),
    HtmlEntity("timesd", "\u2A30"),
    HtmlEntity("tint", "\u222D"),
    HtmlEntity("toea", "\u2928"),
    HtmlEntity("top", "\u22A4"),
    HtmlEntity("topbot", "\u2336"),
    HtmlEntity("topcir", "\u2AF1"),
    HtmlEntity("topf", x("D835DD65")),
    HtmlEntity("topfork", "\u2ADA"),
    HtmlEntity("tosa", "\u2929"),
    HtmlEntity("tprime", "\u2034"),
    HtmlEntity("trade", "\u2122"),
    HtmlEntity("triangle", "\u25B5"),
    HtmlEntity("triangledown", "\u25BF"),
    HtmlEntity("triangleleft", "\u25C3"),
    HtmlEntity("trianglelefteq", "\u22B4"),
    HtmlEntity("triangleq", "\u225C"),
    HtmlEntity("triangleright", "\u25B9"),
    HtmlEntity("trianglerighteq", "\u22B5"),
    HtmlEntity("tridot", "\u25EC"),
    HtmlEntity("trie", "\u225C"),
    HtmlEntity("triminus", "\u2A3A"),
    HtmlEntity("triplus", "\u2A39"),
    HtmlEntity("trisb", "\u29CD"),
    HtmlEntity("tritime", "\u2A3B"),
    HtmlEntity("trpezium", "\u23E2"),
    HtmlEntity("tscr", x("D835DCC9")),
    HtmlEntity("tscy", "\u0446"),
    HtmlEntity("tshcy", "\u045B"),
    HtmlEntity("tstrok", "\u0167"),
    HtmlEntity("twixt", "\u226C"),
    HtmlEntity("twoheadleftarrow", "\u219E"),
    HtmlEntity("twoheadrightarrow", "\u21A0"),
    HtmlEntity("uArr", "\u21D1"),
    HtmlEntity("uHar", "\u2963"),
    HtmlEntity("uacute", "\u00FA"),
    HtmlEntity("uarr", "\u2191"),
    HtmlEntity("ubrcy", "\u045E"),
    HtmlEntity("ubreve", "\u016D"),
    HtmlEntity("ucirc", "\u00FB"),
    HtmlEntity("ucy", "\u0443"),
    HtmlEntity("udarr", "\u21C5"),
    HtmlEntity("udblac", "\u0171"),
    HtmlEntity("udhar", "\u296E"),
    HtmlEntity("ufisht", "\u297E"),
    HtmlEntity("ufr", x("D835DD32")),
    HtmlEntity("ugrave", "\u00F9"),
    HtmlEntity("uharl", "\u21BF"),
    HtmlEntity("uharr", "\u21BE"),
    HtmlEntity("uhblk", "\u2580"),
    HtmlEntity("ulcorn", "\u231C"),
    HtmlEntity("ulcorner", "\u231C"),
    HtmlEntity("ulcrop", "\u230F"),
    HtmlEntity("ultri", "\u25F8"),
    HtmlEntity("umacr", "\u016B"),
    HtmlEntity("uml", "\u00A8"),
    HtmlEntity("uogon", "\u0173"),
    HtmlEntity("uopf", x("D835DD66")),
    HtmlEntity("uparrow", "\u2191"),
    HtmlEntity("updownarrow", "\u2195"),
    HtmlEntity("upharpoonleft", "\u21BF"),
    HtmlEntity("upharpoonright", "\u21BE"),
    HtmlEntity("uplus", "\u228E"),
    HtmlEntity("upsi", "\u03C5"),
    HtmlEntity("upsih", "\u03D2"),
    HtmlEntity("upsilon", "\u03C5"),
    HtmlEntity("upuparrows", "\u21C8"),
    HtmlEntity("urcorn", "\u231D"),
    HtmlEntity("urcorner", "\u231D"),
    HtmlEntity("urcrop", "\u230E"),
    HtmlEntity("uring", "\u016F"),
    HtmlEntity("urtri", "\u25F9"),
    HtmlEntity("uscr", x("D835DCCA")),
    HtmlEntity("utdot", "\u22F0"),
    HtmlEntity("utilde", "\u0169"),
    HtmlEntity("utri", "\u25B5"),
    HtmlEntity("utrif", "\u25B4"),
    HtmlEntity("uuarr", "\u21C8"),
    HtmlEntity("uuml", "\u00FC"),
    HtmlEntity("uwangle", "\u29A7"),
    HtmlEntity("vArr", "\u21D5"),
    HtmlEntity("vBar", "\u2AE8"),
    HtmlEntity("vBarv", "\u2AE9"),
    HtmlEntity("vDash", "\u22A8"),
    HtmlEntity("vangrt", "\u299C"),
    HtmlEntity("varepsilon", "\u03F5"),
    HtmlEntity("varkappa", "\u03F0"),
    HtmlEntity("varnothing", "\u2205"),
    HtmlEntity("varphi", "\u03D5"),
    HtmlEntity("varpi", "\u03D6"),
    HtmlEntity("varpropto", "\u221D"),
    HtmlEntity("varr", "\u2195"),
    HtmlEntity("varrho", "\u03F1"),
    HtmlEntity("varsigma", "\u03C2"),
    HtmlEntity("varsubsetneq", x("228AFE00")),
    HtmlEntity("varsubsetneqq", x("2ACBFE00")),
    HtmlEntity("varsupsetneq", x("228BFE00")),
    HtmlEntity("varsupsetneqq", x("2ACCFE00")),
    HtmlEntity("vartheta", "\u03D1"),
    HtmlEntity("vartriangleleft", "\u22B2"),
    HtmlEntity("vartriangleright", "\u22B3"),
    HtmlEntity("vcy", "\u0432"),
    HtmlEntity("vdash", "\u22A2"),
    HtmlEntity("vee", "\u2228"),
    HtmlEntity("veebar", "\u22BB"),
    HtmlEntity("veeeq", "\u225A"),
    HtmlEntity("vellip", "\u22EE"),
    HtmlEntity("verbar", "\u007C"),
    HtmlEntity("vert", "\u007C"),
    HtmlEntity("vfr", x("D835DD33")),
    HtmlEntity("vltri", "\u22B2"),
    HtmlEntity("vnsub", x("228220D2")),
    HtmlEntity("vnsup", x("228320D2")),
    HtmlEntity("vopf", x("D835DD67")),
    HtmlEntity("vprop", "\u221D"),
    HtmlEntity("vrtri", "\u22B3"),
    HtmlEntity("vscr", x("D835DCCB")),
    HtmlEntity("vsubnE", x("2ACBFE00")),
    HtmlEntity("vsubne", x("228AFE00")),
    HtmlEntity("vsupnE", x("2ACCFE00")),
    HtmlEntity("vsupne", x("228BFE00")),
    HtmlEntity("vzigzag", "\u299A"),
    HtmlEntity("wcirc", "\u0175"),
    HtmlEntity("wedbar", "\u2A5F"),
    HtmlEntity("wedge", "\u2227"),
    HtmlEntity("wedgeq", "\u2259"),
    HtmlEntity("weierp", "\u2118"),
    HtmlEntity("wfr", x("D835DD34")),
    HtmlEntity("wopf", x("D835DD68")),
    HtmlEntity("wp", "\u2118"),
    HtmlEntity("wr", "\u2240"),
    HtmlEntity("wreath", "\u2240"),
    HtmlEntity("wscr", x("D835DCCC")),
    HtmlEntity("xcap", "\u22C2"),
    HtmlEntity("xcirc", "\u25EF"),
    HtmlEntity("xcup", "\u22C3"),
    HtmlEntity("xdtri", "\u25BD"),
    HtmlEntity("xfr", x("D835DD35")),
    HtmlEntity("xhArr", "\u27FA"),
    HtmlEntity("xharr", "\u27F7"),
    HtmlEntity("xi", "\u03BE"),
    HtmlEntity("xlArr", "\u27F8"),
    HtmlEntity("xlarr", "\u27F5"),
    HtmlEntity("xmap", "\u27FC"),
    HtmlEntity("xnis", "\u22FB"),
    HtmlEntity("xodot", "\u2A00"),
    HtmlEntity("xopf", x("D835DD69")),
    HtmlEntity("xoplus", "\u2A01"),
    HtmlEntity("xotime", "\u2A02"),
    HtmlEntity("xrArr", "\u27F9"),
    HtmlEntity("xrarr", "\u27F6"),
    HtmlEntity("xscr", x("D835DCCD")),
    HtmlEntity("xsqcup", "\u2A06"),
    HtmlEntity("xuplus", "\u2A04"),
    HtmlEntity("xutri", "\u25B3"),
    HtmlEntity("xvee", "\u22C1"),
    HtmlEntity("xwedge", "\u22C0"),
    HtmlEntity("yacute", "\u00FD"),
    HtmlEntity("yacy", "\u044F"),
    HtmlEntity("ycirc", "\u0177"),
    HtmlEntity("ycy", "\u044B"),
    HtmlEntity("yen", "\u00A5"),
    HtmlEntity("yfr", x("D835DD36")),
    HtmlEntity("yicy", "\u0457"),
    HtmlEntity("yopf", x("D835DD6A")),
    HtmlEntity("yscr", x("D835DCCE")),
    HtmlEntity("yucy", "\u044E"),
    HtmlEntity("yuml", "\u00FF"),
    HtmlEntity("zacute", "\u017A"),
    HtmlEntity("zcaron", "\u017E"),
    HtmlEntity("zcy", "\u0437"),
    HtmlEntity("zdot", "\u017C"),
    HtmlEntity("zeetrf", "\u2128"),
    HtmlEntity("zeta", "\u03B6"),
    HtmlEntity("zfr", x("D835DD37")),
    HtmlEntity("zhcy", "\u0436"),
    HtmlEntity("zigrarr", "\u21DD"),
    HtmlEntity("zopf", x("D835DD6B")),
    HtmlEntity("zscr", x("D835DCCF")),
    HtmlEntity("zwj", "\u200D"),
    HtmlEntity("zwnj", "\u200C")
];

