/*
 * Copyright (c) 2003 Rich Felker
 * Copyright (c) 2012 Stefano Sabatini
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * field filter, based on libmpcodecs/vf_field.c by Rich Felker
 */

#include "libavutil/opt.h"
#include "libavutil/pixdesc.h"
#include "avfilter.h"
#include "internal.h"

enum FieldType { FIELD_TYPE_TOP = 0, FIELD_TYPE_BOTTOM };

typedef struct {
    const AVClass *class;
    enum FieldType type;
    int nb_planes;              ///< number of planes of the current format
} FieldContext;

#define OFFSET(x) offsetof(FieldContext, x)
#define FLAGS AV_OPT_FLAG_FILTERING_PARAM|AV_OPT_FLAG_VIDEO_PARAM

static const AVOption field_options[] = {
    {"type", "set field type (top or bottom)", OFFSET(type), AV_OPT_TYPE_INT, {.i64=FIELD_TYPE_TOP}, 0, 1, FLAGS, "field_type" },
    {"top",    "select top field",    0, AV_OPT_TYPE_CONST, {.i64=FIELD_TYPE_TOP},    INT_MIN, INT_MAX, FLAGS, "field_type"},
    {"bottom", "select bottom field", 0, AV_OPT_TYPE_CONST, {.i64=FIELD_TYPE_BOTTOM}, INT_MIN, INT_MAX, FLAGS, "field_type"},

    {NULL}
};

AVFILTER_DEFINE_CLASS(field);

static av_cold int init(AVFilterContext *ctx, const char *args)
{
    FieldContext *field = ctx->priv;
    static const char *shorthand[] = { "type", NULL };

    field->class = &field_class;
    av_opt_set_defaults(field);

    return av_opt_set_from_string(field, args, shorthand, "=", ":");
}

static int config_props_output(AVFilterLink *outlink)
{
    AVFilterContext *ctx = outlink->src;
    FieldContext *field = ctx->priv;
    AVFilterLink *inlink = ctx->inputs[0];
    const AVPixFmtDescriptor *desc = av_pix_fmt_desc_get(inlink->format);
    int i;

    for (i = 0; i < desc->nb_components; i++)
        field->nb_planes = FFMAX(field->nb_planes, desc->comp[i].plane);
    field->nb_planes++;

    outlink->w = inlink->w;
    outlink->h = (inlink->h + (field->type == FIELD_TYPE_TOP)) / 2;

    av_log(ctx, AV_LOG_VERBOSE, "w:%d h:%d type:%s -> w:%d h:%d\n",
           inlink->w, inlink->h, field->type == FIELD_TYPE_BOTTOM ? "bottom" : "top",
           outlink->w, outlink->h);
    return 0;
}

static int start_frame(AVFilterLink *inlink, AVFilterBufferRef *inpicref)
{
    FieldContext *field = inlink->dst->priv;
    AVFilterBufferRef *outpicref = avfilter_ref_buffer(inpicref, ~0);
    AVFilterLink *outlink = inlink->dst->outputs[0];
    int i;

    if (!outpicref)
        return AVERROR(ENOMEM);

    outpicref->video->h = outlink->h;
    outpicref->video->interlaced = 0;

    for (i = 0; i < field->nb_planes; i++) {
        if (field->type == FIELD_TYPE_BOTTOM)
            outpicref->data[i] = inpicref->data[i] + inpicref->linesize[i];
        outpicref->linesize[i] = 2 * inpicref->linesize[i];
    }
    return ff_start_frame(outlink, outpicref);
}

static int draw_slice(AVFilterLink *inlink, int y, int h, int slice_dir)
{
    FieldContext *field = inlink->dst->priv;
    int y1 = (y + (field->type == FIELD_TYPE_TOP)) / 2;
    int h1 = (h + (field->type == FIELD_TYPE_TOP)) / 2;
    return ff_draw_slice(inlink->dst->outputs[0], y1, h1, slice_dir);
}

static const AVFilterPad field_inputs[] = {
    {
        .name             = "default",
        .type             = AVMEDIA_TYPE_VIDEO,
        .get_video_buffer = ff_null_get_video_buffer,
        .start_frame      = start_frame,
        .draw_slice       = draw_slice,
        .end_frame        = ff_null_end_frame,
    },
    { NULL }
};

static const AVFilterPad field_outputs[] = {
    {
        .name          = "default",
        .type          = AVMEDIA_TYPE_VIDEO,
        .config_props  = config_props_output,
    },
    { NULL }
};

AVFilter avfilter_vf_field = {
    .name          = "field",
    .description   = NULL_IF_CONFIG_SMALL("Extract a field from the input video."),

    .priv_size     = sizeof(FieldContext),
    .init          = init,

    .inputs        = field_inputs,
    .outputs       = field_outputs,
    .priv_class    = &field_class,
};
