// Copyright (c) 2024 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/common/extensions/api/generated_schemas.h"

#include <algorithm>
#include <iterator>

#include "base/stl_util.h"

namespace {
constexpr char kEnterpriseHardwarePlatform[] = R"R({"namespace":"enterprise.hardwarePlatform","types":[{"id":"enterprise.hardwarePlatform.HardwarePlatformInfo","properties":{"model":{"name":"model","type":"string"},"manufacturer":{"name":"manufacturer","type":"string"}},"type":"object"}],"functions":[{"parameters":[{"name":"callback","parameters":[{"name":"info","$ref":"enterprise.hardwarePlatform.HardwarePlatformInfo"}],"type":"function"}],"name":"getHardwarePlatformInfo","type":"function"}],"properties":{},"internal":false,"events":[],"platforms":null,"deprecated":null,"documentation_options":{}})R";
constexpr char kWebrtcLoggingPrivate[] = R"R({"namespace":"webrtcLoggingPrivate","types":[{"id":"webrtcLoggingPrivate.MetaDataEntry","properties":{"key":{"name":"key","type":"string"},"value":{"name":"value","type":"string"}},"type":"object"},{"id":"webrtcLoggingPrivate.UploadResult","properties":{"reportId":{"name":"reportId","type":"string"}},"type":"object"},{"id":"webrtcLoggingPrivate.RequestInfo","properties":{"tabId":{"optional":true,"name":"tabId","type":"integer"},"guestProcessId":{"optional":true,"name":"guestProcessId","type":"integer"},"targetWebview":{"optional":true,"name":"targetWebview","type":"boolean"}},"type":"object"},{"id":"webrtcLoggingPrivate.RecordingInfo","properties":{"prefixPath":{"name":"prefixPath","type":"string"},"didStop":{"name":"didStop","type":"boolean"},"didManualStop":{"name":"didManualStop","type":"boolean"}},"type":"object"},{"id":"webrtcLoggingPrivate.StartEventLoggingResult","properties":{"logId":{"name":"logId","type":"string"}},"type":"object"}],"functions":[{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"metaData","type":"array","items":{"$ref":"webrtcLoggingPrivate.MetaDataEntry"}},{"name":"callback","parameters":[],"type":"function"}],"name":"setMetaData","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"callback","parameters":[],"type":"function"}],"name":"start","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"shouldUpload","type":"boolean"}],"name":"setUploadOnRenderClose","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"callback","parameters":[],"type":"function"}],"name":"stop","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"logId","type":"string"},{"name":"callback","parameters":[],"type":"function"}],"name":"store","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"logId","type":"string"},{"name":"callback","parameters":[{"name":"result","$ref":"webrtcLoggingPrivate.UploadResult"}],"type":"function"}],"name":"uploadStored","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"callback","parameters":[{"name":"result","$ref":"webrtcLoggingPrivate.UploadResult"}],"type":"function"}],"name":"upload","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"callback","parameters":[],"type":"function"}],"name":"discard","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"incoming","type":"boolean"},{"name":"outgoing","type":"boolean"},{"name":"callback","parameters":[],"type":"function"}],"name":"startRtpDump","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"incoming","type":"boolean"},{"name":"outgoing","type":"boolean"},{"name":"callback","parameters":[],"type":"function"}],"name":"stopRtpDump","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"seconds","type":"integer"},{"name":"callback","parameters":[{"name":"info","$ref":"webrtcLoggingPrivate.RecordingInfo"}],"type":"function"}],"name":"startAudioDebugRecordings","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"callback","parameters":[{"name":"info","$ref":"webrtcLoggingPrivate.RecordingInfo"}],"type":"function"}],"name":"stopAudioDebugRecordings","type":"function"},{"parameters":[{"name":"request","$ref":"webrtcLoggingPrivate.RequestInfo"},{"name":"securityOrigin","type":"string"},{"name":"sessionId","type":"string"},{"name":"maxLogSizeBytes","type":"integer"},{"name":"outputPeriodMs","type":"integer"},{"name":"webAppId","type":"integer"},{"name":"callback","parameters":[{"name":"result","$ref":"webrtcLoggingPrivate.StartEventLoggingResult"}],"type":"function"}],"name":"startEventLogging","type":"function"},{"parameters":[{"name":"callback","parameters":[{"name":"entry","type":"object","additionalProperties":{"type":"any"},"isInstanceOf":"DirectoryEntry"}],"type":"function"}],"name":"getLogsDirectory","type":"function"}],"properties":{},"internal":false,"events":[],"platforms":null,"deprecated":null,"documentation_options":{}})R";
constexpr char kApp[] = R"R({"namespace":"app","types":[{"id":"app.Details","type":"object","properties":{},"additionalProperties":{"type":"any"}},{"id":"app.DOMWindow","type":"object","properties":{},"additionalProperties":{"type":"any"}},{"id":"app.InstallState","type":"string","enum":["not_installed","installed","disabled"]},{"id":"app.RunningState","type":"string","enum":["running","cannot_run","ready_to_run"]}],"functions":[{"name":"getIsInstalled","type":"function","parameters":[],"returns":{"name":"isInstalled","type":"boolean"}},{"name":"installState","type":"function","parameters":[{"type":"function","name":"callback","parameters":[{"$ref":"app.InstallState","name":"state"}]}]},{"name":"runningState","type":"function","parameters":[],"returns":{"$ref":"app.RunningState"}},{"name":"getDetails","type":"function","parameters":[],"returns":{"$ref":"app.Details","optional":true}}]})R";
constexpr char kCommands[] = R"R({"namespace":"commands","types":[{"id":"commands.Command","type":"object","properties":{"name":{"type":"string","optional":true},"description":{"type":"string","optional":true},"shortcut":{"type":"string","optional":true}}}],"events":[{"name":"onCommand","type":"function","parameters":[{"name":"command","type":"string"},{"$ref":"tabs.Tab","name":"tab","optional":true}]}],"functions":[{"name":"getAll","type":"function","parameters":[{"type":"function","name":"callback","optional":true,"parameters":[{"name":"commands","type":"array","items":{"$ref":"commands.Command"}}]}]}]})R";
constexpr char kDeclarativeContent[] = R"R({"namespace":"declarativeContent","types":[{"id":"declarativeContent.ImageDataType","type":"binary","isInstanceOf":"ImageData","additionalProperties":{"type":"any"},"inline_doc":true},{"id":"declarativeContent.PageStateMatcherInstanceType","type":"string","enum":["declarativeContent.PageStateMatcher"]},{"id":"declarativeContent.ShowPageActionInstanceType","type":"string","enum":["declarativeContent.ShowPageAction"]},{"id":"declarativeContent.ShowActionInstanceType","type":"string","enum":["declarativeContent.ShowAction"]},{"id":"declarativeContent.SetIconInstanceType","type":"string","enum":["declarativeContent.SetIcon"]},{"id":"declarativeContent.RequestContentScriptInstanceType","type":"string","enum":["declarativeContent.RequestContentScript"]},{"id":"declarativeContent.PageStateMatcher","type":"object","properties":{"pageUrl":{"$ref":"events.UrlFilter","optional":true},"css":{"type":"array","optional":true,"items":{"type":"string"}},"isBookmarked":{"type":"boolean","optional":true},"instanceType":{"$ref":"declarativeContent.PageStateMatcherInstanceType"}}},{"id":"declarativeContent.ShowPageAction","type":"object","properties":{"instanceType":{"$ref":"declarativeContent.ShowPageActionInstanceType"}}},{"id":"declarativeContent.ShowAction","type":"object","properties":{"instanceType":{"$ref":"declarativeContent.ShowActionInstanceType"}}},{"id":"declarativeContent.SetIcon","type":"object","properties":{"instanceType":{"$ref":"declarativeContent.SetIconInstanceType"},"imageData":{"choices":[{"$ref":"declarativeContent.ImageDataType"},{"type":"object","additionalProperties":{"type":"any"}}],"optional":true}}},{"id":"declarativeContent.RequestContentScript","type":"object","properties":{"css":{"type":"array","optional":true,"items":{"type":"string"}},"js":{"type":"array","optional":true,"items":{"type":"string"}},"allFrames":{"type":"boolean","optional":true},"matchAboutBlank":{"type":"boolean","optional":true},"instanceType":{"$ref":"declarativeContent.RequestContentScriptInstanceType"}}}],"functions":[],"events":[{"name":"onPageChanged","options":{"supportsListeners":false,"supportsRules":true,"conditions":["declarativeContent.PageStateMatcher"],"actions":["declarativeContent.RequestContentScript","declarativeContent.SetIcon","declarativeContent.ShowPageAction","declarativeContent.ShowAction"]}}]})R";
constexpr char kDom[] = R"R({"namespace":"dom","functions":[{"name":"openOrClosedShadowRoot","type":"function","parameters":[{"name":"element","type":"object","isInstanceOf":"HTMLElement","additionalProperties":{"type":"any"}}],"returns":{"name":"shadowRoot","type":"object","additionalProperties":{"type":"any"}}}]})R";
constexpr char kPrivacy[] = R"R({"namespace":"privacy","types":[{"id":"privacy.IPHandlingPolicy","type":"string","enum":["default","default_public_and_private_interfaces","default_public_interface_only","disable_non_proxied_udp"]}],"properties":{"network":{"type":"object","value":{},"properties":{"networkPredictionEnabled":{"$ref":"types.ChromeSetting","value":["networkPredictionEnabled",{"type":"boolean"}]},"webRTCMultipleRoutesEnabled":{"$ref":"types.ChromeSetting","value":["webRTCMultipleRoutesEnabled",{"type":"boolean"}],"deprecated":"Please use privacy.network.webRTCIPHandlingPolicy. This remains for backward compatibility in this release and will be removed in the future."},"webRTCNonProxiedUdpEnabled":{"$ref":"types.ChromeSetting","value":["webRTCNonProxiedUdpEnabled",{"type":"boolean"}],"deprecated":"Please use privacy.network.webRTCIPHandlingPolicy. This remains for backward compatibility in this release and will be removed in the future."},"webRTCIPHandlingPolicy":{"$ref":"types.ChromeSetting","value":["webRTCIPHandlingPolicy",{"$ref":"privacy.IPHandlingPolicy"}]}}},"services":{"type":"object","value":{},"properties":{"alternateErrorPagesEnabled":{"$ref":"types.ChromeSetting","value":["alternateErrorPagesEnabled",{"type":"boolean"}]},"autofillEnabled":{"$ref":"types.ChromeSetting","value":["autofillEnabled",{"type":"boolean"}],"deprecated":"Please use privacy.services.autofillAddressEnabled and privacy.services.autofillCreditCardEnabled. This remains for backward compatibility in this release and will be removed in the future."},"autofillAddressEnabled":{"$ref":"types.ChromeSetting","value":["autofillAddressEnabled",{"type":"boolean"}]},"autofillCreditCardEnabled":{"$ref":"types.ChromeSetting","value":["autofillCreditCardEnabled",{"type":"boolean"}]},"passwordSavingEnabled":{"$ref":"types.ChromeSetting","value":["passwordSavingEnabled",{"type":"boolean"}]},"safeBrowsingEnabled":{"$ref":"types.ChromeSetting","value":["safeBrowsingEnabled",{"type":"boolean"}]},"safeBrowsingExtendedReportingEnabled":{"$ref":"types.ChromeSetting","value":["safeBrowsingExtendedReportingEnabled",{"type":"boolean"}]},"searchSuggestEnabled":{"$ref":"types.ChromeSetting","value":["searchSuggestEnabled",{"type":"boolean"}]},"spellingServiceEnabled":{"$ref":"types.ChromeSetting","value":["spellingServiceEnabled",{"type":"boolean"}]},"translationServiceEnabled":{"$ref":"types.ChromeSetting","value":["translationServiceEnabled",{"type":"boolean"}]}}},"websites":{"type":"object","value":{},"properties":{"thirdPartyCookiesAllowed":{"$ref":"types.ChromeSetting","value":["thirdPartyCookiesAllowed",{"type":"boolean"}]},"hyperlinkAuditingEnabled":{"$ref":"types.ChromeSetting","value":["hyperlinkAuditingEnabled",{"type":"boolean"}]},"referrersEnabled":{"$ref":"types.ChromeSetting","value":["referrersEnabled",{"type":"boolean"}]},"doNotTrackEnabled":{"$ref":"types.ChromeSetting","value":["doNotTrackEnabled",{"type":"boolean"}]},"protectedContentEnabled":{"$ref":"types.ChromeSetting","value":["protectedContentEnabled",{"type":"boolean"}],"platforms":["win","chromeos"]}}}}})R";
constexpr char kProxy[] = R"R({"namespace":"proxy","types":[{"id":"proxy.Scheme","type":"string","enum":["http","https","quic","socks4","socks5"]},{"id":"proxy.Mode","type":"string","enum":["direct","auto_detect","pac_script","fixed_servers","system"]},{"id":"proxy.ProxyServer","type":"object","properties":{"scheme":{"$ref":"proxy.Scheme","optional":true},"host":{"type":"string"},"port":{"type":"integer","optional":true}}},{"id":"proxy.ProxyRules","type":"object","properties":{"singleProxy":{"$ref":"proxy.ProxyServer","optional":true},"proxyForHttp":{"$ref":"proxy.ProxyServer","optional":true},"proxyForHttps":{"$ref":"proxy.ProxyServer","optional":true},"proxyForFtp":{"$ref":"proxy.ProxyServer","optional":true},"fallbackProxy":{"$ref":"proxy.ProxyServer","optional":true},"bypassList":{"type":"array","items":{"type":"string"},"optional":true}}},{"id":"proxy.PacScript","type":"object","properties":{"url":{"type":"string","optional":true},"data":{"type":"string","optional":true},"mandatory":{"type":"boolean","optional":true}}},{"id":"proxy.ProxyConfig","type":"object","properties":{"rules":{"$ref":"proxy.ProxyRules","optional":true},"pacScript":{"$ref":"proxy.PacScript","optional":true},"mode":{"$ref":"proxy.Mode"}}}],"properties":{"settings":{"$ref":"types.ChromeSetting","value":["proxy",{"$ref":"proxy.ProxyConfig"}]}},"events":[{"name":"onProxyError","type":"function","parameters":[{"type":"object","name":"details","properties":{"fatal":{"type":"boolean"},"error":{"type":"string"},"details":{"type":"string"}}}]}]})R";
constexpr char kTts[] = R"R({"namespace":"tts","types":[{"id":"tts.EventType","type":"string","enum":["start","end","word","sentence","marker","interrupted","cancelled","error","pause","resume"]},{"id":"tts.VoiceGender","type":"string","deprecated":"Gender is deprecated and is ignored.","enum":["male","female"]},{"id":"tts.TtsOptions","type":"object","properties":{"enqueue":{"type":"boolean","optional":true},"voiceName":{"type":"string","optional":true},"extensionId":{"type":"string","optional":true},"lang":{"type":"string","optional":true},"gender":{"$ref":"tts.VoiceGender","optional":true,"deprecated":"Gender is deprecated and will be ignored."},"rate":{"type":"number","optional":true,"minimum":0.1,"maximum":10},"pitch":{"type":"number","optional":true,"minimum":0,"maximum":2},"volume":{"type":"number","optional":true,"minimum":0,"maximum":1},"requiredEventTypes":{"type":"array","items":{"type":"string"},"optional":true},"desiredEventTypes":{"type":"array","items":{"type":"string"},"optional":true},"onEvent":{"type":"function","optional":true,"parameters":[{"name":"event","$ref":"tts.TtsEvent"}]}}},{"id":"tts.TtsEvent","type":"object","properties":{"type":{"$ref":"tts.EventType"},"charIndex":{"type":"integer","optional":true},"errorMessage":{"type":"string","optional":true},"srcId":{"type":"number","optional":true},"isFinalEvent":{"type":"boolean","optional":true},"length":{"type":"integer","optional":true}}},{"id":"tts.TtsVoice","type":"object","properties":{"voiceName":{"type":"string","optional":true},"lang":{"type":"string","optional":true},"gender":{"$ref":"tts.VoiceGender","optional":true,"deprecated":"Gender is deprecated and will be ignored."},"remote":{"type":"boolean","optional":true},"extensionId":{"type":"string","optional":true},"eventTypes":{"type":"array","items":{"$ref":"tts.EventType"},"optional":true}}}],"functions":[{"name":"speak","type":"function","parameters":[{"type":"string","name":"utterance"},{"$ref":"tts.TtsOptions","name":"options","optional":true},{"type":"function","name":"callback","optional":true,"parameters":[]}]},{"name":"stop","type":"function","parameters":[]},{"name":"pause","type":"function","parameters":[]},{"name":"resume","type":"function","parameters":[]},{"name":"isSpeaking","type":"function","parameters":[{"type":"function","name":"callback","optional":true,"parameters":[{"name":"speaking","type":"boolean"}]}]},{"name":"getVoices","type":"function","parameters":[{"type":"function","name":"callback","optional":true,"parameters":[{"type":"array","name":"voices","items":{"$ref":"tts.TtsVoice"}}]}]}],"events":[{"name":"onEvent","type":"function","parameters":[{"name":"event","$ref":"tts.TtsEvent"}]}]})R";
constexpr char kTtsEngine[] = R"R({"namespace":"ttsEngine","types":[{"id":"ttsEngine.VoiceGender","type":"string","deprecated":"Gender is deprecated and will be ignored.","enum":["male","female"]}],"functions":[{"name":"updateVoices","type":"function","parameters":[{"type":"array","name":"voices","items":{"$ref":"tts.TtsVoice"}}]},{"name":"sendTtsEvent","type":"function","parameters":[{"type":"integer","name":"requestId"},{"name":"event","$ref":"tts.TtsEvent"}]}],"events":[{"name":"onSpeak","type":"function","parameters":[{"type":"string","name":"utterance"},{"type":"object","name":"options","properties":{"voiceName":{"type":"string","optional":true},"lang":{"type":"string","optional":true},"gender":{"$ref":"ttsEngine.VoiceGender","optional":true,"deprecated":"Gender is deprecated and will be ignored."},"rate":{"type":"number","optional":true,"minimum":0.1,"maximum":10.0},"pitch":{"type":"number","optional":true,"minimum":0,"maximum":2},"volume":{"type":"number","optional":true,"minimum":0,"maximum":1}}},{"name":"sendTtsEvent","type":"function","parameters":[{"name":"event","$ref":"tts.TtsEvent"}]}]},{"name":"onStop","type":"function"},{"name":"onPause","type":"function"},{"name":"onResume","type":"function"}]})R";
constexpr char kInputMethodPrivate[] = R"R({"namespace":"inputMethodPrivate","types":[{"id":"inputMethodPrivate.MenuItemStyle","type":"string","enum":["check","radio","separator"]},{"id":"inputMethodPrivate.MenuItem","type":"object","properties":{"id":{"type":"string"},"label":{"type":"string","optional":true},"style":{"$ref":"inputMethodPrivate.MenuItemStyle","optional":true},"visible":{"type":"boolean","optional":true},"checked":{"type":"boolean","optional":true},"enabled":{"type":"boolean","optional":true}}},{"id":"inputMethodPrivate.UnderlineStyle","type":"string","enum":["underline","doubleUnderline","noUnderline"]},{"id":"inputMethodPrivate.FocusReason","type":"string","enum":["mouse","touch","pen","other"]},{"id":"inputMethodPrivate.InputModeType","type":"string","enum":["noKeyboard","text","tel","url","email","numeric","decimal","search"]},{"id":"inputMethodPrivate.InputContextType","type":"string","enum":["text","search","tel","url","email","number","password","null"]},{"id":"inputMethodPrivate.AutoCapitalizeType","type":"string","enum":["off","characters","words","sentences"]},{"id":"inputMethodPrivate.InputContext","type":"object","properties":{"contextID":{"type":"integer"},"type":{"$ref":"inputMethodPrivate.InputContextType"},"mode":{"$ref":"inputMethodPrivate.InputModeType"},"autoCorrect":{"type":"boolean"},"autoComplete":{"type":"boolean"},"autoCapitalize":{"$ref":"inputMethodPrivate.AutoCapitalizeType"},"spellCheck":{"type":"boolean"},"shouldDoLearning":{"type":"boolean"},"focusReason":{"$ref":"inputMethodPrivate.FocusReason"},"hasBeenPassword":{"type":"boolean"},"appKey":{"type":"string","optional":true}}},{"id":"inputMethodPrivate.InputMethodSettings","type":"object","properties":{"enableCompletion":{"type":"boolean","optional":true},"enableDoubleSpacePeriod":{"type":"boolean","optional":true},"enableGestureTyping":{"type":"boolean","optional":true},"enablePrediction":{"type":"boolean","optional":true},"enableSoundOnKeypress":{"type":"boolean","optional":true},"physicalKeyboardAutoCorrectionLevel":{"type":"integer","optional":true},"physicalKeyboardEnableCapitalization":{"type":"boolean","optional":true},"virtualKeyboardAutoCorrectionLevel":{"type":"integer","optional":true},"virtualKeyboardEnableCapitalization":{"type":"boolean","optional":true},"xkbLayout":{"type":"string","optional":true},"koreanEnableSyllableInput":{"type":"boolean","optional":true},"koreanKeyboardLayout":{"type":"string","optional":true},"koreanShowHangulCandidate":{"type":"boolean","optional":true},"pinyinChinesePunctuation":{"type":"boolean","optional":true},"pinyinDefaultChinese":{"type":"boolean","optional":true},"pinyinEnableFuzzy":{"type":"boolean","optional":true},"pinyinEnableLowerPaging":{"type":"boolean","optional":true},"pinyinEnableUpperPaging":{"type":"boolean","optional":true},"pinyinFullWidthCharacter":{"type":"boolean","optional":true},"pinyinFuzzyConfig":{"type":"object","optional":true,"properties":{"an_ang":{"type":"boolean","optional":true},"c_ch":{"type":"boolean","optional":true},"en_eng":{"type":"boolean","optional":true},"f_h":{"type":"boolean","optional":true},"ian_iang":{"type":"boolean","optional":true},"in_ing":{"type":"boolean","optional":true},"k_g":{"type":"boolean","optional":true},"l_n":{"type":"boolean","optional":true},"r_l":{"type":"boolean","optional":true},"s_sh":{"type":"boolean","optional":true},"uan_uang":{"type":"boolean","optional":true},"z_zh":{"type":"boolean","optional":true}}},"zhuyinKeyboardLayout":{"type":"string","optional":true},"zhuyinPageSize":{"type":"integer","optional":true},"zhuyinSelectKeys":{"type":"string","optional":true}}}],"functions":[{"name":"getInputMethodConfig","type":"function","parameters":[{"name":"callback","type":"function","optional":false,"parameters":[{"name":"config","type":"object","properties":{"isPhysicalKeyboardAutocorrectEnabled":{"type":"boolean"},"isImeMenuActivated":{"type":"boolean"}}}]}]},{"name":"getInputMethods","type":"function","parameters":[{"name":"callback","type":"function","optional":false,"parameters":[{"name":"inputMethods","type":"array","items":{"type":"object","properties":{"id":{"type":"string"},"name":{"type":"string"},"indicator":{"type":"string"}}}}]}]},{"name":"getCurrentInputMethod","type":"function","parameters":[{"name":"callback","type":"function","optional":false,"parameters":[{"name":"inputMethodId","type":"string"}]}]},{"name":"setCurrentInputMethod","type":"function","parameters":[{"name":"inputMethodId","type":"string","optional":false},{"name":"callback","type":"function","optional":true,"parameters":[]}]},{"name":"fetchAllDictionaryWords","type":"function","parameters":[{"name":"callback","type":"function","optional":false,"parameters":[{"name":"words","type":"array","items":{"type":"string"}}]}]},{"name":"addWordToDictionary","type":"function","parameters":[{"name":"word","type":"string","optional":false},{"name":"callback","type":"function","optional":true,"parameters":[]}]},{"name":"getEncryptSyncEnabled","type":"function","parameters":[{"name":"callback","type":"function","optional":true,"parameters":[{"name":"enabled","type":"boolean","optional":false}]}]},{"name":"setXkbLayout","type":"function","parameters":[{"name":"xkb_name","type":"string"},{"name":"callback","type":"function","optional":true,"parameters":[]}]},{"name":"finishComposingText","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"}}},{"type":"function","name":"callback","optional":true,"parameters":[]}]},{"name":"setSelectionRange","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"},"selectionStart":{"optional":true,"type":"integer"},"selectionEnd":{"optional":true,"type":"integer"}}},{"type":"function","name":"callback","optional":true,"parameters":[{"name":"success","type":"boolean"}]}]},{"name":"notifyImeMenuItemActivated","type":"function","parameters":[{"name":"engineID","type":"string"},{"name":"name","type":"string"}]},{"name":"showInputView","type":"function","parameters":[{"type":"function","name":"callback","optional":true,"parameters":[]}]},{"name":"hideInputView","type":"function","parameters":[{"type":"function","name":"callback","optional":true,"parameters":[]}]},{"name":"openOptionsPage","type":"function","parameters":[{"name":"inputMethodId","type":"string"}]},{"name":"getCompositionBounds","type":"function","parameters":[{"type":"function","name":"callback","parameters":[{"name":"boundsList","type":"array","items":{"type":"object","properties":{"x":{"type":"integer"},"y":{"type":"integer"},"w":{"type":"integer"},"h":{"type":"integer"}}}}]}]},{"name":"getSurroundingText","type":"function","parameters":[{"name":"beforeLength","type":"integer"},{"name":"afterLength","type":"integer"},{"type":"function","name":"callback","parameters":[{"name":"surroundingInfo","type":"object","properties":{"before":{"type":"string"},"selected":{"type":"string"},"after":{"type":"string"}}}]}]},{"name":"getSettings","type":"function","parameters":[{"name":"engineID","type":"string"},{"type":"function","name":"callback","parameters":[{"name":"settings","$ref":"inputMethodPrivate.InputMethodSettings","optional":true}]}]},{"name":"setSettings","type":"function","parameters":[{"name":"engineID","type":"string"},{"name":"settings","$ref":"inputMethodPrivate.InputMethodSettings"},{"type":"function","name":"callback","optional":true,"parameters":[]}]},{"name":"setCompositionRange","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"},"selectionBefore":{"type":"integer"},"selectionAfter":{"type":"integer"},"segments":{"type":"array","optional":true,"items":{"type":"object","properties":{"start":{"type":"integer"},"end":{"type":"integer"},"style":{"$ref":"inputMethodPrivate.UnderlineStyle"}}}}}},{"type":"function","name":"callback","optional":true,"parameters":[{"name":"success","type":"boolean"}]}]},{"name":"setComposingRange","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"},"start":{"type":"integer"},"end":{"type":"integer"},"segments":{"type":"array","optional":true,"items)R" R"R(":{"type":"object","properties":{"start":{"type":"integer"},"end":{"type":"integer"},"style":{"$ref":"inputMethodPrivate.UnderlineStyle"}}}}}},{"type":"function","name":"callback","optional":true}]},{"name":"getAutocorrectRange","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"}}},{"type":"function","name":"callback","parameters":[{"name":"autocorrectCharacterBounds","type":"object","properties":{"start":{"type":"integer"},"end":{"type":"integer"}}}]}]},{"name":"getAutocorrectCharacterBounds","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"}}},{"type":"function","name":"callback","parameters":[{"name":"autocorrectCharacterBounds","type":"object","properties":{"x":{"type":"integer"},"y":{"type":"integer"},"width":{"type":"integer"},"height":{"type":"integer"}}}]}]},{"name":"setAutocorrectRange","type":"function","parameters":[{"name":"parameters","type":"object","properties":{"contextID":{"type":"integer"},"autocorrectString":{"type":"string"},"selectionStart":{"type":"integer"},"selectionEnd":{"type":"integer"}}},{"type":"function","name":"callback","optional":true,"parameters":[]}]},{"name":"reset","type":"function","parameters":[]}],"events":[{"name":"onChanged","type":"function","parameters":[{"name":"newInputMethodId","type":"string"}]},{"name":"onCompositionBoundsChanged","type":"function","parameters":[{"name":"firstBounds","type":"object","properties":{"x":{"type":"integer"},"y":{"type":"integer"},"w":{"type":"integer"},"h":{"type":"integer"}}},{"name":"boundsList","type":"array","items":{"type":"object","properties":{"x":{"type":"integer"},"y":{"type":"integer"},"w":{"type":"integer"},"h":{"type":"integer"}}}}]},{"name":"onDictionaryLoaded","type":"function","parameters":[]},{"name":"onDictionaryChanged","type":"function","parameters":[{"name":"added","type":"array","items":{"type":"string"}},{"name":"removed","type":"array","items":{"type":"string"}}]},{"name":"onImeMenuActivationChanged","type":"function","parameters":[{"name":"activation","type":"boolean"}]},{"name":"onImeMenuListChanged","type":"function","parameters":[]},{"name":"onImeMenuItemsChanged","type":"function","parameters":[{"name":"engineID","type":"string"},{"name":"items","type":"array","items":{"$ref":"inputMethodPrivate.MenuItem"}}]},{"name":"onFocus","type":"function","parameters":[{"$ref":"inputMethodPrivate.InputContext","name":"context"}]},{"name":"onSettingsChanged","type":"function","parameters":[{"name":"engineID","type":"string"},{"name":"settings","$ref":"inputMethodPrivate.InputMethodSettings"}]},{"name":"onScreenProjectionChanged","type":"function","parameters":[{"name":"isProjected","type":"boolean"}]},{"name":"onSuggestionsChanged","type":"function","parameters":[{"name":"suggestions","type":"array","items":{"type":"string"}}]},{"name":"onInputMethodOptionsChanged","type":"function","parameters":[{"name":"engineID","type":"string"}]}]})R";
}  // namespace

namespace extensions {
namespace api {

// static
bool ChromeGeneratedSchemas::IsGenerated(base::StringPiece name) {
  return !Get(name).empty();
}

// static
base::StringPiece ChromeGeneratedSchemas::Get(base::StringPiece name) {
  static const struct kSchemaMapping {
    const base::StringPiece name;
    const base::StringPiece schema;
    constexpr bool operator<(const kSchemaMapping& that) const {
      return name < that.name;
    }
  } kSchemas[] = {
    {"app", kApp},
    {"commands", kCommands},
    {"declarativeContent", kDeclarativeContent},
    {"dom", kDom},
    {"enterprise.hardwarePlatform", kEnterpriseHardwarePlatform},
    {"inputMethodPrivate", kInputMethodPrivate},
    {"privacy", kPrivacy},
    {"proxy", kProxy},
    {"tts", kTts},
    {"ttsEngine", kTtsEngine},
    {"webrtcLoggingPrivate", kWebrtcLoggingPrivate},
  };
  auto it = std::lower_bound(std::begin(kSchemas), std::end(kSchemas),
    kSchemaMapping{name, base::StringPiece()});
  if (it != std::end(kSchemas) && it->name == name)
    return it->schema;
  return base::StringPiece();
}

}  // namespace api
}  // namespace extensions
